"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderInstallCommand = exports.installPackage = void 0;
const path = require("path");
const fs = require("fs-extra");
const logging = require("../logging");
const util_1 = require("../util");
/**
 * Installs the npm module (through `npm install`) to node_modules under `projectDir`.
 * @param spec The npm package spec (e.g. `foo@^1.2` or `foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz`)
 * @returns The installed package name (e.g. `@foo/bar`)
 */
function installPackage(baseDir, spec) {
    const packageJsonPath = path.join(baseDir, "package.json");
    const packageJsonExisted = fs.existsSync(packageJsonPath);
    if (!packageJsonExisted) {
        // Make sure we have a package.json to read from later
        util_1.exec("npm init --yes", { cwd: baseDir });
    }
    logging.info(`installing external module ${spec}...`);
    util_1.exec(renderInstallCommand(baseDir, spec), { cwd: baseDir });
    // Get the true installed package name
    const packageJson = fs.readJsonSync(packageJsonPath);
    const packageName = Object.keys(packageJson.devDependencies).find((name) => name !== "projen");
    if (!packageName) {
        throw new Error(`Unable to resolve package name from spec ${spec}`);
    }
    // if package.json did not exist before calling `npm install`, we should remove it
    // so we can start off clean.
    if (!packageJsonExisted) {
        fs.removeSync(packageJsonPath);
    }
    return packageName;
}
exports.installPackage = installPackage;
/**
 * Render a command to install an npm package.
 *
 * Engine checks are ignorred at this point so that the module can be installed
 * regardless of the environment. This was needed to unblock the upgrade of the
 * minimum node version of projen, but also okay generally because engine checks
 * will be performed later and for all eternety.
 *
 * @param dir Base directory
 * @param module The module to install (e.g. foo@^1.2)
 * @returns The string that includes the install command ("npm install ...")
 */
function renderInstallCommand(dir, module) {
    return `npm install --save-dev -f --no-package-lock --prefix="${dir}" ${module}`;
}
exports.renderInstallCommand = renderInstallCommand;
//# sourceMappingURL=data:application/json;base64,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