// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { useCallback, useRef } from 'react';
export function isScrollable(ele) {
    const overflowXStyle = window.getComputedStyle(ele).overflowX;
    const isOverflowHidden = overflowXStyle.indexOf('hidden') !== -1;
    return ele.scrollWidth > ele.clientWidth && !isOverflowHidden;
}
export function getScrollableParent(ele) {
    return !ele || ele === document.body
        ? document.body
        : isScrollable(ele)
            ? ele
            : getScrollableParent(ele.parentElement);
}
const shouldScroll = ([cx, cy], [px, py]) => {
    return cx - px > 5 || cy - py > 5;
};
/**
 * This hook stores the scroll position of the nearest scrollable parent of the
 * `activeElementRef` when `storeScrollPosition` is called, and restores it when
 * `restoreScrollPosition` is called.
 * @param activeElementRef Ref to an active element in the table. This is used to find the nearest scrollable parent.
 */
export function useStableScrollPosition(activeElementRef) {
    const scrollRef = useRef();
    const storeScrollPosition = useCallback(() => {
        var _a;
        const scrollableParent = getScrollableParent((_a = activeElementRef.current) !== null && _a !== void 0 ? _a : document.body);
        if (scrollableParent) {
            scrollRef.current = [scrollableParent.scrollLeft, scrollableParent.scrollTop];
        }
    }, [activeElementRef]);
    const restoreScrollPosition = useCallback(() => {
        var _a;
        const scrollableParent = getScrollableParent((_a = activeElementRef.current) !== null && _a !== void 0 ? _a : document.body);
        if (scrollRef.current &&
            scrollRef.current.toString() !== '0,0' &&
            shouldScroll(scrollRef.current, [scrollableParent.scrollLeft, scrollableParent.scrollTop])) {
            [scrollableParent.scrollLeft, scrollableParent.scrollTop] = scrollRef.current;
        }
    }, [activeElementRef]);
    return { storeScrollPosition, restoreScrollPosition };
}
//# sourceMappingURL=use-stable-scroll-position.js.map