// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import React, { useContext } from 'react';
import IntlMessageFormat from 'intl-messageformat';
import { InternalI18nContext } from './context';
/**
 * Context to send parent messages down to child I18nProviders. This isn't
 * included in the InternalI18nContext to avoid components from depending on
 * MessageFormatElement types.
 */
const I18nMessagesContext = React.createContext({});
export function I18nProvider({ messages: messagesArray, locale: providedLocale, children }) {
    // The provider accepts an array of configs. We merge parent messages and
    // flatten the tree early on so that accesses by key are simpler and faster.
    const parentMessages = useContext(I18nMessagesContext);
    const messages = mergeMessages([parentMessages, ...messagesArray]);
    // If a locale isn't provided, we can try and guess from the html lang,
    // and lastly default to English. Locales have a recommended case, but are
    // matched case-insensitively, so we lowercase it internally.
    const locale = (providedLocale || (document === null || document === void 0 ? void 0 : document.documentElement.lang) || 'en').toLowerCase();
    const format = (namespace, component, key, provided, customHandler) => {
        var _a, _b, _c;
        // A general rule in the library is that undefined is basically
        // treated as "not provided". So even if a user explicitly provides an
        // undefined value, it will default to i18n values.
        if (provided !== undefined) {
            return provided;
        }
        // Widen the locale string (e.g. en-GB -> en) until we find a locale
        // that contains the message we need.
        let message;
        const matchableLocales = getMatchableLocales(locale);
        for (const matchableLocale of matchableLocales) {
            const localeMessage = (_c = (_b = (_a = messages === null || messages === void 0 ? void 0 : messages[namespace]) === null || _a === void 0 ? void 0 : _a[matchableLocale]) === null || _b === void 0 ? void 0 : _b[component]) === null || _c === void 0 ? void 0 : _c[key];
            if (localeMessage !== undefined) {
                message = localeMessage;
                break;
            }
        }
        // If a message wasn't found, exit early.
        if (message === undefined) {
            return provided;
        }
        const intlMessageFormat = new IntlMessageFormat(message, locale);
        if (customHandler) {
            return customHandler(args => intlMessageFormat.format(args));
        }
        // Assuming `T extends string` since a customHandler wasn't provided.
        return intlMessageFormat.format();
    };
    return (React.createElement(InternalI18nContext.Provider, { value: format },
        React.createElement(I18nMessagesContext.Provider, { value: messages }, children)));
}
function mergeMessages(sources) {
    const result = {};
    for (const messages of sources) {
        for (const namespace in messages) {
            if (!(namespace in result)) {
                result[namespace] = {};
            }
            for (const casedLocale in messages[namespace]) {
                const locale = casedLocale.toLowerCase();
                if (!(locale in result[namespace])) {
                    result[namespace][locale] = {};
                }
                for (const component in messages[namespace][casedLocale]) {
                    if (!(component in result[namespace][locale])) {
                        result[namespace][locale][component] = {};
                    }
                    for (const key in messages[namespace][casedLocale][component]) {
                        result[namespace][locale][component][key] = messages[namespace][casedLocale][component][key];
                    }
                }
            }
        }
    }
    return result;
}
function getMatchableLocales(ietfLanguageTag) {
    const parts = ietfLanguageTag.split('-');
    if (parts.length === 1) {
        return [ietfLanguageTag];
    }
    const localeStrings = [];
    for (let i = parts.length; i > 0; i--) {
        localeStrings.push(parts.slice(0, i).join('-'));
    }
    return localeStrings;
}
//# sourceMappingURL=provider.js.map