import { AuthOptions, FederatedResponse, SignUpParams, FederatedUser, ConfirmSignUpOptions, SignOutOpts, CurrentUserOpts, GetPreferredMFAOpts, SignInOpts, FederatedSignInOptionsCustom, LegacyProvider, FederatedSignInOptions, ClientMetaData } from './types';
import { ICredentials } from '@aws-amplify/core';
import { ISignUpResult, CognitoUser, MFAOption, CognitoUserSession, CognitoUserAttribute, NodeCallback } from 'amazon-cognito-identity-js';
import { AuthError } from './Errors';
import { IAuthDevice } from './types/Auth';
/**
 * Provide authentication steps
 */
export declare class AuthClass {
    private _config;
    private userPool;
    private user;
    private _oAuthHandler;
    private _storage;
    private _storageSync;
    private oAuthFlowInProgress;
    private pendingSignIn;
    private autoSignInInitiated;
    private inflightSessionPromise;
    private inflightSessionPromiseCounter;
    Credentials: import("@aws-amplify/core").CredentialsClass;
    /**
     * Initialize Auth with AWS configurations
     * @param {Object} config - Configuration of the Auth
     */
    constructor(config: AuthOptions);
    getModuleName(): string;
    configure(config?: any): AuthOptions;
    wrapRefreshSessionCallback: (callback: NodeCallback<Error, any>) => NodeCallback<Error, any>;
    /**
     * Sign up with username, password and other attributes like phone, email
     * @param {String | object} params - The user attributes used for signin
     * @param {String[]} restOfAttrs - for the backward compatability
     * @return - A promise resolves callback data if success
     */
    signUp(params: string | SignUpParams, ...restOfAttrs: string[]): Promise<ISignUpResult>;
    private handleAutoSignIn;
    private handleCodeAutoSignIn;
    private handleLinkAutoSignIn;
    private signInAfterUserConfirmed;
    /**
     * Send the verification code to confirm sign up
     * @param {String} username - The username to be confirmed
     * @param {String} code - The verification code
     * @param {ConfirmSignUpOptions} options - other options for confirm signup
     * @return - A promise resolves callback data if success
     */
    confirmSignUp(username: string, code: string, options?: ConfirmSignUpOptions): Promise<any>;
    private isTrueStorageValue;
    /**
     * Resend the verification code
     * @param {String} username - The username to be confirmed
     * @param {ClientMetadata} clientMetadata - Metadata to be passed to Cognito Lambda triggers
     * @return - A promise resolves code delivery details if successful
     */
    resendSignUp(username: string, clientMetadata?: ClientMetaData): Promise<any>;
    /**
     * Sign in
     * @param {String | SignInOpts} usernameOrSignInOpts - The username to be signed in or the sign in options
     * @param {String} pw - The password of the username
     * @param {ClientMetaData} clientMetadata - Client metadata for custom workflows
     * @return - A promise resolves the CognitoUser
     */
    signIn(usernameOrSignInOpts: string | SignInOpts, pw?: string, clientMetadata?: ClientMetaData): Promise<CognitoUser | any>;
    /**
     * Return an object with the authentication callbacks
     * @param {CognitoUser} user - the cognito user object
     * @param {} resolve - function called when resolving the current step
     * @param {} reject - function called when rejecting the current step
     * @return - an object with the callback methods for user authentication
     */
    private authCallbacks;
    /**
     * Sign in with a password
     * @private
     * @param {AuthenticationDetails} authDetails - the user sign in data
     * @return - A promise resolves the CognitoUser object if success or mfa required
     */
    private signInWithPassword;
    /**
     * Sign in without a password
     * @private
     * @param {AuthenticationDetails} authDetails - the user sign in data
     * @return - A promise resolves the CognitoUser object if success or mfa required
     */
    private signInWithoutPassword;
    /**
     * This was previously used by an authenticated user to get MFAOptions,
     * but no longer returns a meaningful response. Refer to the documentation for
     * how to setup and use MFA: https://docs.amplify.aws/lib/auth/mfa/q/platform/js
     * @deprecated
     * @param {CognitoUser} user - the current user
     * @return - A promise resolves the current preferred mfa option if success
     */
    getMFAOptions(user: CognitoUser | any): Promise<MFAOption[]>;
    /**
     * get preferred mfa method
     * @param {CognitoUser} user - the current cognito user
     * @param {GetPreferredMFAOpts} params - options for getting the current user preferred MFA
     */
    getPreferredMFA(user: CognitoUser | any, params?: GetPreferredMFAOpts): Promise<string>;
    private _getMfaTypeFromUserData;
    private _getUserData;
    /**
     * set preferred MFA method
     * @param {CognitoUser} user - the current Cognito user
     * @param {string} mfaMethod - preferred mfa method
     * @return - A promise resolve if success
     */
    setPreferredMFA(user: CognitoUser | any, mfaMethod: 'TOTP' | 'SMS' | 'NOMFA' | 'SMS_MFA' | 'SOFTWARE_TOKEN_MFA'): Promise<string>;
    /**
     * disable SMS
     * @deprecated
     * @param {CognitoUser} user - the current user
     * @return - A promise resolves is success
     */
    disableSMS(user: CognitoUser): Promise<string>;
    /**
     * enable SMS
     * @deprecated
     * @param {CognitoUser} user - the current user
     * @return - A promise resolves is success
     */
    enableSMS(user: CognitoUser): Promise<string>;
    /**
     * Setup TOTP
     * @param {CognitoUser} user - the current user
     * @return - A promise resolves with the secret code if success
     */
    setupTOTP(user: CognitoUser | any): Promise<string>;
    /**
     * verify TOTP setup
     * @param {CognitoUser} user - the current user
     * @param {string} challengeAnswer - challenge answer
     * @return - A promise resolves is success
     */
    verifyTotpToken(user: CognitoUser | any, challengeAnswer: string): Promise<CognitoUserSession>;
    /**
     * Send MFA code to confirm sign in
     * @param {Object} user - The CognitoUser object
     * @param {String} code - The confirmation code
     */
    confirmSignIn(user: CognitoUser | any, code: string, mfaType?: 'SMS_MFA' | 'SOFTWARE_TOKEN_MFA' | null, clientMetadata?: ClientMetaData): Promise<CognitoUser | any>;
    completeNewPassword(user: CognitoUser | any, password: string, requiredAttributes?: any, clientMetadata?: ClientMetaData): Promise<CognitoUser | any>;
    /**
     * Send the answer to a custom challenge
     * @param {CognitoUser} user - The CognitoUser object
     * @param {String} challengeResponses - The confirmation code
     */
    sendCustomChallengeAnswer(user: CognitoUser | any, challengeResponses: string, clientMetadata?: ClientMetaData): Promise<CognitoUser | any>;
    /**
     * Delete an authenticated users' attributes
     * @param {CognitoUser} - The currently logged in user object
     * @return {Promise}
     **/
    deleteUserAttributes(user: CognitoUser | any, attributeNames: string[]): Promise<unknown>;
    /**
     * Delete the current authenticated user
     * @return {Promise}
     **/
    deleteUser(): Promise<string | void>;
    /**
     * Update an authenticated users' attributes
     * @param {CognitoUser} - The currently logged in user object
     * @return {Promise}
     **/
    updateUserAttributes(user: CognitoUser | any, attributes: object, clientMetadata?: ClientMetaData): Promise<string>;
    private createUpdateAttributesResultList;
    /**
     * Return user attributes
     * @param {Object} user - The CognitoUser object
     * @return - A promise resolves to user attributes if success
     */
    userAttributes(user: CognitoUser | any): Promise<CognitoUserAttribute[]>;
    verifiedContact(user: CognitoUser | any): Promise<{
        verified: {};
        unverified: {};
    }>;
    private isErrorWithMessage;
    private isTokenRevokedError;
    private isRefreshTokenRevokedError;
    private isUserDisabledError;
    private isUserDoesNotExistError;
    private isRefreshTokenExpiredError;
    private isSignedInHostedUI;
    private isSessionInvalid;
    private cleanUpInvalidSession;
    /**
     * Get current authenticated user
     * @return - A promise resolves to current authenticated CognitoUser if success
     */
    currentUserPoolUser(params?: CurrentUserOpts): Promise<CognitoUser | any>;
    private isOAuthInProgress;
    /**
     * Get current authenticated user
     * @param {CurrentUserOpts} - options for getting the current user
     * @return - A promise resolves to current authenticated CognitoUser if success
     */
    currentAuthenticatedUser(params?: CurrentUserOpts): Promise<CognitoUser | any>;
    /**
     * Get current user's session
     * @return - A promise resolves to session object if success
     */
    currentSession(): Promise<CognitoUserSession>;
    private _userSession;
    /**
     * Get the corresponding user session
     * @param {Object} user - The CognitoUser object
     * @return - A promise resolves to the session
     */
    userSession(user: any): Promise<CognitoUserSession>;
    /**
     * Get authenticated credentials of current user.
     * @return - A promise resolves to be current user's credentials
     */
    currentUserCredentials(): Promise<ICredentials>;
    currentCredentials(): Promise<ICredentials>;
    /**
     * Initiate an attribute confirmation request
     * @param {Object} user - The CognitoUser
     * @param {Object} attr - The attributes to be verified
     * @return - A promise resolves to callback data if success
     */
    verifyUserAttribute(user: CognitoUser | any, attr: string, clientMetadata?: ClientMetaData): Promise<void>;
    /**
     * Confirm an attribute using a confirmation code
     * @param {Object} user - The CognitoUser
     * @param {Object} attr - The attribute to be verified
     * @param {String} code - The confirmation code
     * @return - A promise resolves to callback data if success
     */
    verifyUserAttributeSubmit(user: CognitoUser | any, attr: string, code: string): Promise<string>;
    verifyCurrentUserAttribute(attr: string): Promise<void>;
    /**
     * Confirm current user's attribute using a confirmation code
     * @param {Object} attr - The attribute to be verified
     * @param {String} code - The confirmation code
     * @return - A promise resolves to callback data if success
     */
    verifyCurrentUserAttributeSubmit(attr: string, code: string): Promise<string>;
    private cognitoIdentitySignOut;
    private oAuthSignOutRedirect;
    private oAuthSignOutAndResolve;
    private oAuthSignOutRedirectOrReject;
    /**
     * Sign out method
     * @
     * @return - A promise resolved if success
     */
    signOut(opts?: SignOutOpts): Promise<any>;
    private cleanCachedItems;
    /**
     * Change a password for an authenticated user
     * @param {Object} user - The CognitoUser object
     * @param {String} oldPassword - the current password
     * @param {String} newPassword - the requested new password
     * @return - A promise resolves if success
     */
    changePassword(user: CognitoUser | any, oldPassword: string, newPassword: string, clientMetadata?: ClientMetaData): Promise<'SUCCESS'>;
    /**
     * Initiate a forgot password request
     * @param {String} username - the username to change password
     * @return - A promise resolves if success
     */
    forgotPassword(username: string, clientMetadata?: ClientMetaData): Promise<any>;
    /**
     * Confirm a new password using a confirmation Code
     * @param {String} username - The username
     * @param {String} code - The confirmation code
     * @param {String} password - The new password
     * @return - A promise that resolves if success
     */
    forgotPasswordSubmit(username: string, code: string, password: string, clientMetadata?: ClientMetaData): Promise<string>;
    /**
     * Get user information
     * @async
     * @return {Object }- current User's information
     */
    currentUserInfo(): Promise<any>;
    federatedSignIn(options?: FederatedSignInOptions): Promise<ICredentials>;
    federatedSignIn(provider: LegacyProvider, response: FederatedResponse, user: FederatedUser): Promise<ICredentials>;
    federatedSignIn(options?: FederatedSignInOptionsCustom): Promise<ICredentials>;
    /**
     * Used to complete the OAuth flow with or without the Cognito Hosted UI
     * @param {String} URL - optional parameter for customers to pass in the response URL
     */
    private _handleAuthResponse;
    /**
     * Compact version of credentials
     * @param {Object} credentials
     * @return {Object} - Credentials
     */
    essentialCredentials(credentials: any): ICredentials;
    private attributesToObject;
    private isTruthyString;
    private createCognitoUser;
    private _isValidAuthStorage;
    private noUserPoolErrorHandler;
    private rejectAuthError;
    private rejectNoUserPool;
    rememberDevice(): Promise<string | AuthError>;
    forgetDevice(): Promise<void>;
    fetchDevices(): Promise<IAuthDevice[]>;
}
export declare const Auth: AuthClass;
