# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['csv_reconcile', 'csv_reconcile_dice']

package_data = \
{'': ['*']}

install_requires = \
['cython>=0.29.21,<0.30.0',
 'flask-cors>=3.0.10,<4.0.0',
 'flask>=2.0,<3.0',
 'normality>=2.1.1,<3.0.0']

extras_require = \
{':python_version >= "3.7" and python_version < "3.8"': ['importlib_metadata>=3.7.3,<5.0.0']}

entry_points = \
{'console_scripts': ['csv-reconcile = csv_reconcile:main'],
 'csv_reconcile.scorers': ['dice = csv_reconcile_dice']}

setup_kwargs = {
    'name': 'csv-reconcile',
    'version': '0.2.4',
    'description': 'OpenRefine reconciliation service backed by csv resource',
    'long_description': '-   [CSV Reconcile](#csv-reconcile)\n    -   [Quick start](#quick-start)\n    -   [Poetry](#poetry)\n        -   [Building](#building)\n    -   [Description](#description)\n    -   [Usage](#usage)\n    -   [Common configuration](#common-configuration)\n    -   [Scoring plugins](#scoring-plugins)\n        -   [Implementing](#implementing)\n        -   [Installing](#installing)\n        -   [Using](#using)\n    -   [Future enhancements](#future-enhancements)\n\n# CSV Reconcile\n\nA [reconciliation service](https://github.com/reconciliation-api/specs)\nfor [OpenRefine](https://openrefine.org/) based on a CSV file similar to\n[reconcile-csv](http://okfnlabs.org/reconcile-csv/). This one is written\nin Python and has some more configurability.\n\n## Quick start\n\n-   Clone this repository\n\n-   Run the service\n\n        $ python -m venv venv                                             # create virtualenv\n        $ venv/bin/pip install csv-reconcile                              # install package\n        $ source venv/bin/activate                                        # activate virtual environment\n        (venv) $ csv-reconcile --init-db sample/reps.tsv item itemLabel   # start the service\n        (venv) $ deactivate                                               # remove virtual environment\n\nThe service is run at <http://127.0.0.1:5000/reconcile>. You can point\nat a different host:port by adding\n[SERVER\\_NAME](https://flask.palletsprojects.com/en/0.12.x/config/) to\nthe sample.cfg. Since this is running from a virtualenv, you can simply\ndelete the whole lot to clean up.\n\nIf you have a C compiler installed you may prefer to install the sdist\n`dist/csv-reconcile-0.1.0.tar.gz` which will build a\n[Cython](https://cython.readthedocs.io/en/latest/) version of the\ncomputationally intensive fuzzy match routine for speed. With `pip` add\nthe option `--no-binary csv-reconcile`.\n\n## Poetry\n\nThis is packaged with [poetry](https://python-poetry.org/docs/), so you\ncan use those commands if you have it installed.\n\n    $ poetry run poe install\n    $ poetry run csv-reconcile --init-db sample/reps.tsv item itemLabel\n\n### Building\n\nBecause this package uses a `README.org` file and `pip` requires a\n`README.md`, there are extra build steps beyond what `poetry` supplies.\nThese are managed using\n[poethepoet](https://pypi.org/project/poethepoet/0.0.3/). Thus building\nis done as follows:\n\n    $ poetry run poe build\n\nIf you want to build a platform agnostic wheel, you\'ll have to comment\nout the `build =\n    "build.py"` line from `pyproject.toml` until `poetry` supports\n[selecting build\nplatform](https://github.com/python-poetry/poetry/issues/3594).\n\n## Description\n\nThis reconciliation service uses [Dice coefficient\nscoring](https://en.wikipedia.org/wiki/S%C3%B8rensen%E2%80%93Dice_coefficient)\nto reconcile values against a given column in a\n[CSV](https://en.wikipedia.org/wiki/Comma-separated_values) file. The\nCSV file must contain a column containing distinct values to reconcile\nto. We\'ll call this the *id column*. We\'ll call the column being\nreconciled against the *name column*.\n\nFor performance reasons, the *name column* is preprocessed to normalized\nvalues which are stored in an\n[sqlite](https://www.sqlite.org/index.html) database. This database must\nbe initialized at least once by passing the `--init-db` on the command\nline. Once initialized this option can be removed from subsequent runs.\n\nNote that the service supplies all its data with a dummy *type* so there\nis no reason to reconcile against any particular *type*.\n\nIn addition to reconciling against the *name column*, the service also\nfunctions as a [data extension\nservice](https://reconciliation-api.github.io/specs/latest/#data-extension-service),\nwhich offers any of the other columns of the CSV file.\n\nNote that Dice coefficient scoring is agnostic to word ordering.\n\n## Usage\n\nBasic usage requires passing the name of the CSV file, the *id column*\nand the *name column*.\n\n    $ poetry run csv-reconcile --help\n    Usage: csv-reconcile [OPTIONS] CSVFILE IDCOL NAMECOL\n\n    Options:\n      --config TEXT  config file\n      --scorer TEXT  scoring plugin to use\n      --init-db      initialize the db\n      --help         Show this message and exit.\n    $\n\nIn addition to the `--init-db` switch mentioned above you may use the\n`--config` option to point to a configuration file. The file is a [Flask\nconfiguration](https://flask.palletsprojects.com/en/1.1.x/config/) and\nhence is Python code though most configuration is simply setting\nvariables to constant values.\n\n## Common configuration\n\n-   `SERVER_NAME` - The host and port the service is bound to. e.g.\n    `SERVER_NAME=localhost:5555`. ( Default localhost:5000 )\n-   `CSVKWARGS` - Arguments to pass to\n    [csv.reader](https://docs.python.org/3/library/csv.html). e.g.\n    `CSVKWARGS={\'delimiter\': \',\', \'quotechar\': \'"\'}` for comma delimited\n    files using `"` as quote character.\n-   `CSVENCODING` - Encoding of the CSV file. e.g.\n    `CSVECODING=\'utf-8-sig\'` is the encoding used for data downloaded\n    from\n    [GNIS](https://www.usgs.gov/core-science-systems/ngp/board-on-geographic-names/download-gnis-data).\n-   `SCOREOPTIONS` - Options passed to scoring plugin during\n    normalization. e.g.\n    `SCOREOPTIONS={\'stopwords\':[\'lake\',\'reservoir\']}`\n-   `LIMIT` - The maximum number of reonciliation candidates returned\n    per entry. ( Default 10 ) e.g. `LIMIT=10`\n-   `THRESHOLD` - The minimum score for returned reconciliation\n    candidates. ( Default 30.0 ) e.g. `THRESHOLD=80.5`\n-   `DATABASE` - The name of the generated sqlite database containing\n    pre-processed values. (Default `csvreconcile.db`) e.g.\n    `DATABASE=\'lakes.db\'` You may want to change the name of the\n    database if you regularly switch between databases being used.\n-   `MANIFEST` - Overrides for the service manifest. e.g.\n    `MANIFEST={"name": "My service"}` sets the name of the service to\n    "My service".\n\nThis last is most interesting. If your data is coming from\n[Wikidata](https://www.wikidata.org) and your *id column* contains [Q\nvalues](https://www.wikidata.org/wiki/Help:Items), then a manifest like\nthe following will allow your links to be clickable inside OpenRefine.\n\n    MANIFEST = {\n      "identifierSpace": "http://www.wikidata.org/entity/",\n      "schemaSpace": "http://www.wikidata.org/prop/direct/",\n      "view": {"url":"https://www.wikidata.org/wiki/{{id}}"},\n      "name": "My reconciliation service"\n    }\n\nIf your CSV is made up of data taken from another [reconciliation\nservice](https://reconciliation-api.github.io/testbench/), you may\nsimiliarly copy parts of their manifest to make use of their features,\nsuch as the [preview\nservice](https://reconciliation-api.github.io/specs/latest/#preview-service).\nSee the reconciliation spec for details.\n\n## Scoring plugins\n\nAs mentioned above the default scoring method is to use [Dice\ncoefficient\nscoring](https://en.wikipedia.org/wiki/S%C3%B8rensen%E2%80%93Dice_coefficient),\nbut this method can be overridden by implementing a\n`cvs_reconcile.scorers` plugin.\n\n### Implementing\n\nA plugin module may override any of the methods in the\n`csv_reconcile.scorers` module by simply implementing a method of the\nsame name with the decorator `@cvs_reconcile.scorer.register`.\n\nSee `csv_reconcile_dice` for how Dice coefficient scoring is\nimplemented.\n\nThe basic hooks are as follows:\n\n-   `normalizedWord(word, **scoreOptions)` preprocesses values to be\n    reconciled to produce a tuple used in fuzzy match scoring. The value\n    of `SCOREOPTIONS` in the configuration will be passed in to allow\n    configuration of this preprocessing. This hook is required.\n-   `normalizedRow(word, row, **scoreOptions)` preprocesses values to be\n    reconciled against to produce a tuple used in fuzzy match scoring.\n    Note that both the reconciled column and the entire row is available\n    for calculating the normalized value and that the column reconciled\n    against is required even when not used. The value of `SCOREOPTIONS`\n    in the configuration will be passed in to allow configuration of\n    this preprocessing. This defaults to calling\n    normalizeWord(word,\\*\\*scoreOptions).\n-   `getNormalizedFields()` returns a tuple of names for the columns\n    produced by `normalizeWord()`. The length of the return value from\n    both functions must match. This defaults to calling\n    normalizeWord(word,\\*\\*scoreOptions). This hook is required.\n-   `processScoreOptions(options)` is passed the value of `SCOREOPTIONS`\n    to allow it to be adjusted prior to being used. This can be used for\n    adding defaults and/or validating the configuration. This hook is\n    optional\n-   `scoreMatch(left, right, **scoreOptions)` gets passed two tuples as\n    returned by `normalizedWord()`. The `left` value is the value being\n    reconciled and the `right` value is the value being reconciled\n    against. The value of `SCOREOPTIONS` in the configuration will be\n    passed in to allow configuration of this preprocessing. Returning a\n    score of `None` will not add tested value as a candidate. This hook\n    is required.\n-   `valid(normalizedFields)` is passed the normalized tuple prior to\n    being scored to make sure it\'s appropriate for the calculation. This\n    hook is optional.\n-   `features(word, row, **scoreOptions)` calculates\n    [features](https://reconciliation-api.github.io/specs/latest/#reconciliation-query-responses)\n    using the query string and the normalized row. By default\n    calculating features is disabled. Implementions of this hook are\n    automatically enabled. This hook is optional.\n\n### Installing\n\nHooks are automatically discovered as long as they provide a\n`csv_reconcile.scorers` [setuptools entry\npoint](https://setuptools.readthedocs.io/en/latest/userguide/entry_point.html).\nPoetry supplies a\n[plugins](https://python-poetry.org/docs/pyproject/#plugins)\nconfiguration which wraps the setuptools funtionality.\n\nThe default Dice coefficent scoring is supplied via the following\nsnippet from `pyproject.toml` file.\n\n    [tool.poetry.plugins."csv_reconcile.scorers"]\n    "dice" = "csv_reconcile_dice"\n\nHere `dice` becomes the name of the scoring option and\n`csv_reconcile_dice` is the package implementing the plugin.\n\n### Using\n\nIf there is only one scoring plugin available, that plugin is used. If\nthere are more than one available, you will be prompted to pass the\n`--scorer` option to select among the scoring options.\n\n## Future enhancements\n\nIt would be nice to add support for using\n[properties](https://reconciliation-api.github.io/specs/latest/#structure-of-a-reconciliation-query)\nas part of the scoring, so that more than one column of the csv could be\ntaken into consideration.\n',
    'author': 'Douglas Mennella',
    'author_email': 'trx2358-pypi@yahoo.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/gitonthescene/csv-reconcile',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}
from build import *
build(setup_kwargs)

setup(**setup_kwargs)
