# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['chemxor',
 'chemxor.cli',
 'chemxor.config',
 'chemxor.data',
 'chemxor.model',
 'chemxor.pipelines',
 'chemxor.routes',
 'chemxor.schema',
 'chemxor.service']

package_data = \
{'': ['*'],
 'chemxor.config': ['base/*', 'base/catalog/*', 'base/parameters/*', 'local/*']}

install_requires = \
['Flask>=2.1.2,<3.0.0',
 'ase>=3.22.1,<4.0.0',
 'feather-format>=0.4.1,<0.5.0',
 'gensim>=4.1.2,<5.0.0',
 'logzero>=1.7.0,<2.0.0',
 'matplotlib>=3.5.1,<4.0.0',
 'nltk>=3.6.7,<4.0.0',
 'onnx>=1.11.0,<2.0.0',
 'pandas>=1.4,<2.0',
 'protobuf==3.19.1',
 'pydantic[dotenv]>=1.8.2,<2.0.0',
 'pytorch-lightning>=1.6.1,<2.0.0',
 'rdkit-pypi>=2022.3.1,<2023.0.0',
 'regex==2022.7.9',
 'scikit-learn>=1.0.2,<2.0.0',
 'scipy==1.7.3',
 'tenseal>=0.3.11,<0.4.0',
 'torch[cpu]>=1.10.0,<2.0.0',
 'torchani>=2.2,<3.0',
 'torchvision>=0.12.0,<0.13.0']

entry_points = \
{'console_scripts': ['chemxor = chemxor.cli.main:main']}

setup_kwargs = {
    'name': 'chemxor',
    'version': '0.3.0',
    'description': 'Privacy Preserving AI/ML for Drug Discovery',
    'long_description': '# ChemXor\n\nPrivacy Preserving AI/ML for Drug Discovery\n\n---\n\n## Overview\n\nChemXor is an open source library thta provides a set of pre-tuned model architectures for evaluating FHE(Fully homomorphic encryption) encrypted inputs. These models can be trained as normal Pytorch models. It also provides convenient functions to quickly query and host these models as a service with strong privacy guarantees for the end-user. It is built on top of TenSEAL and Pytorch.\n\n### What is Fully Homomorphic Encryption (FHE)?\n\n> A cryptosystem that supports arbitrary computation on ciphertexts is known as fully homomorphic encryption (FHE). Such a scheme enables the construction of programs for any desirable functionality, which can be run on encrypted inputs to produce an encryption of the result. Since such a program need never decrypt its inputs, it can be run by an untrusted party without revealing its inputs and internal state. Fully homomorphic cryptosystems have great practical implications in the outsourcing of private computations. (Wikipedia)\n\n### Why do you need FHE for Machine Learning?\n\nUsing FHE, one can compute on encrypted data, without learning anything about the data. This enables novel privacy preserving interactions between actors in the context of machine learning.\n\n## Getting Started\n\nChemxor is available on PyPi and can be installed using pip.\n\n```bash\npip install chemxor\n```\n\n### Model selection\n\nAt the moment, one can choose from 3 pre-tuned models.\n\n- OlindaNetZero : Slimmest model with one convolution and 3 linear layers\n- OlindaNet: Model with two convolutions and 4 linear layers\n- OlindaOneNet: Model with four convolutions and 4 linear layers\n\nThese models accept a 32 x 32 input and can be configured to produce a signle or multiple outputs.\n\n```python\nfrom chemxor.models import OlindaNetZero, OlindaNetOne, OlindaNet\n\n# model for regression\nmodel = OlindaNetZero(output = 1)\n```\n\nThe model is a normal Pytorch Lightning module which is compatible with Pytorch NN module.\n\n### Dataset Preparation\n\nChemXor provides two generic Pytorch Lightning Datamodules (Regression, Classification) that can be used to train and evaluate the models. These Datamodules expects raw data as CSV files with two columns (SMILES, target).\n\n```python\nfrom chemxor.data import OlindaCDataModule, OlindaRDataModule\n\ndm_regression = OlindaRDataModule(csv_path="path/to/csv")\n\n# Use the threshold value to automatically create categorical \n# classes from the target column of the CSV\ndm_classification = OlindaCDataModule(csv_path="path/to/csv", threshold=[0.5])\n```\n\nThe DataModules will take care of converting the smiles input to 32 x 32 images.\n\n### Model Training\n\nIt is recommended to use a Pytorch Lightning trainer to train the models. Although a normal Pytorch training loop can also be used.\n\n```python\nimport pytorch_lightning as pl\n\n# Save the best 3 checkpoints based on validation loss\ncheckpoint_callback = pl.callbacks.ModelCheckpoint(\n        dirpath="path/to/save/checkpoints",\n        save_top_k=3,\n        monitor="VAL_Loss",\n    )\ntrainer = pl.Trainer(callbacks=[checkpoint_callback], accelerator="auto")\ntrainer.fit(model=model, datamodule=data_module)\n```\n\n### FHE models\n\nAfter training, the models can be wrapped using their specific FHE wrappers to process FHE inputs. FHE wrappers will take care of Tenseal context parameters and keys management.\n\n```python\nfrom chemxor.models import OlindaNetZero, OlindaNetOne, OlindaNet\nfrom chemxor.models import FHEOlindaNetZero, FHEOlindaNetOne, FHEOlindaNet\n\nmodel = OlindaNetZero(output = 1)\nmodel.load("path/to/checkpoint")\nfhe_model = FHEOlindaNetZero(model=model)\n```\n\n### FHE inputs evaluation\n\nThe Datamodules can generate Pytorch dataloaders that produce encrypted inputs for the model.\n\n```python\nfrom chemxor.data import OlindaCDataModule, OlindaRDataModule\n\ndm_regression = OlindaRDataModule(csv_path="path/to/csv")\ndm_regression.setup("test")\nenc_data_loader = dm_classification.enc_dataloader(context=fhe_model.enc_context)\nenc_sample = next(iter(enc_data_loader))\n```\n\nAlso, the FHE models are partitioned to control multiplicative depth. So, the forward function is modified to accept a step parameter. For testing, The FHE model can be evaluated locally as follows:\n\n```python\nfrom chemxor.utils import prepare_fhe_input\n\noutput = enc_sample\nfor step in range(fhe_model.steps):\n    output = fhe_model(output, step)\n    dec_out = output.decrypt().tolist()\n    output = prepare_fhe_input(\n                    dec_out,\n                    fhe_model.pre_process[step],\n                    fhe_model.enc_context\n                )\n\n# final decryted output\ndecrypted_output = output.decrypt().tolist()\n```\n\nThis process can automated using a utility function provided by ChemXor\n\n```python\nfrom chemxor.utils import evaluate_fhe_model\n\ndecrypted_output = evaluate_fhe_model(fhe_model, enc_sample, decrypt = True)\n```\n\n### Serve models\n\nFHE Models can be served in the form of a Flask app as follows:\n\n```python\nfrom chemxor.service import PartitionNetServer\n\nfhe_model_server = PartitionNetServer(fhe_model)\n\nif __name__ == "__main__":\n    fhe_model_server.run()\n```\n\n### Query models\n\n```bash\nchemxor query -i [input file path] [model url]\n```\n\n## Developing\n\nWe use poetry to manage project dependecies. Use poetry to install project in editable mode.\n\n```bash\npoetry install\n```\n\n## License\n\nThis project is licensed under GNU AFFERO GENERAL PUBLIC LICENSE Version 3.\nThe direct and indirect dependecies of the project are licensed as follows:\n\n| Name                      | Version     | License                                                                                             | Author                                                                                                           |\n|---------------------------|-------------|-----------------------------------------------------------------------------------------------------|------------------------------------------------------------------------------------------------------------------|\n| Babel                     | 2.10.1      | BSD License                                                                                         | Armin Ronacher                                                                                                   |\n| CacheControl              | 0.12.11     | Apache Software License                                                                             | Eric Larson                                                                                                      |\n| Faker                     | 13.12.0     | MIT License                                                                                         | joke2k                                                                                                           |\n| Flask                     | 2.1.2       | BSD License                                                                                         | Armin Ronacher                                                                                                   |\n| GitPython                 | 3.1.27      | BSD License                                                                                         | Sebastian Thiel, Michael Trier                                                                                   |\n| Jinja2                    | 3.1.2       | BSD License                                                                                         | Armin Ronacher                                                                                                   |\n| Markdown                  | 3.3.7       | BSD License                                                                                         | Manfred Stienstra, Yuri takhteyev and Waylan limberg                                                             |\n| MarkupSafe                | 2.1.1       | BSD License                                                                                         | Armin Ronacher                                                                                                   |\n| Pillow                    | 9.1.1       | Historical Permission Notice and Disclaimer (HPND)                                                  | Alex Clark (PIL Fork Author)                                                                                     |\n| PyJWT                     | 2.3.0       | MIT License                                                                                         | Jose Padilla                                                                                                     |\n| PyNaCl                    | 1.4.0       | Apache License 2.0                                                                                  | The PyNaCl developers                                                                                            |\n| PyYAML                    | 5.4.1       | MIT License                                                                                         | Kirill Simonov                                                                                                   |\n| Pygments                  | 2.12.0      | BSD License                                                                                         | Georg Brandl                                                                                                     |\n| SQLAlchemy                | 1.4.36      | MIT License                                                                                         | Mike Bayer                                                                                                       |\n| SecretStorage             | 3.3.2       | BSD License                                                                                         | Dmitry Shachnev                                                                                                  |\n| Send2Trash                | 1.8.0       | BSD License                                                                                         | Andrew Senetar                                                                                                   |\n| Werkzeug                  | 2.1.2       | BSD License                                                                                         | Armin Ronacher                                                                                                   |\n| absl-py                   | 1.0.0       | Apache Software License                                                                             | The Abseil Authors                                                                                               |\n| aiofiles                  | 0.6.0       | Apache Software License                                                                             | Tin Tvrtkovic                                                                                                    |\n| aiohttp                   | 3.8.1       | Apache Software License                                                                             | UNKNOWN                                                                                                          |\n| aiosignal                 | 1.2.0       | Apache Software License                                                                             | Nikolay Kim                                                                                                      |\n| anyconfig                 | 0.10.1      | MIT License                                                                                         | Satoru SATOH                                                                                                     |\n| anyio                     | 3.6.1       | MIT License                                                                                         | Alex Grönholm                                                                                                    |\n| argcomplete               | 1.12.3      | Apache Software License                                                                             | Andrey Kislyuk                                                                                                   |\n| argon2-cffi               | 21.3.0      | MIT License                                                                                         | UNKNOWN                                                                                                          |\n| argon2-cffi-bindings      | 21.2.0      | MIT License                                                                                         | Hynek Schlawack                                                                                                  |\n| arrow                     | 1.2.2       | Apache Software License                                                                             | Chris Smith                                                                                                      |\n| ascii-magic               | 1.6         | MIT License                                                                                         | Leandro Barone                                                                                                   |\n| ase                       | 3.22.1      | GNU Lesser General Public License v2 or later (LGPLv2+)                                             | UNKNOWN                                                                                                          |\n| asgiref                   | 3.5.2       | BSD License                                                                                         | Django Software Foundation                                                                                       |\n| async-timeout             | 4.0.2       | Apache Software License                                                                             | Andrew Svetlov <andrew.svetlov@gmail.com>                                                                        |\n| attrs                     | 21.4.0      | MIT License                                                                                         | Hynek Schlawack                                                                                                  |\n| autodp                    | 0.2         | Apache Software License                                                                             | Yu-Xiang Wang                                                                                                    |\n| backcall                  | 0.2.0       | BSD License                                                                                         | Thomas Kluyver                                                                                                   |\n| backports.cached-property | 1.0.1       | MIT License                                                                                         | Aleksei Stepanov                                                                                                 |\n| bandit                    | 1.7.4       | Apache Software License                                                                             | PyCQA                                                                                                            |\n| bcrypt                    | 3.2.0       | Apache Software License                                                                             | The Python Cryptographic Authority developers                                                                    |\n| beautifulsoup4            | 4.11.1      | MIT License                                                                                         | Leonard Richardson                                                                                               |\n| binaryornot               | 0.4.4       | BSD License                                                                                         | Audrey Roy Greenfeld                                                                                             |\n| black                     | 22.3.0      | MIT License                                                                                         | Łukasz Langa                                                                                                     |\n| bleach                    | 5.0.0       | Apache Software License                                                                             | UNKNOWN                                                                                                          |\n| cachetools                | 4.2.4       | MIT License                                                                                         | Thomas Kemmer                                                                                                    |\n| cachy                     | 0.3.0       | MIT License                                                                                         | Sébastien Eustace                                                                                                |\n| certifi                   | 2022.5.18.1 | Mozilla Public License 2.0 (MPL 2.0)                                                                | Kenneth Reitz                                                                                                    |\n| cffi                      | 1.15.0      | MIT License                                                                                         | Armin Rigo, Maciej Fijalkowski                                                                                   |\n| cfgv                      | 3.3.1       | MIT License                                                                                         | Anthony Sottile                                                                                                  |\n| chardet                   | 4.0.0       | GNU Library or Lesser General Public License (LGPL)                                                 | Mark Pilgrim                                                                                                     |\n| charset-normalizer        | 2.0.12      | MIT License                                                                                         | Ahmed TAHRI @Ousret                                                                                              |\n| chemxor                   | 0.1.0       | GPLv3                                                                     | Ersilia Open Source Initiative                                                                                               |\n| cleo                      | 0.8.1       | MIT License                                                                                         | Sébastien Eustace                                                                                                |\n| click                     | 8.1.3       | BSD License                                                                                         | Armin Ronacher                                                                                                   |\n| clikit                    | 0.6.2       | MIT License                                                                                         | Sébastien Eustace                                                                                                |\n| colorlog                  | 6.6.0       | MIT License                                                                                         | Sam Clements                                                                                                     |\n| cookiecutter              | 1.7.3       | BSD License                                                                                         | Audrey Roy                                                                                                       |\n| coverage                  | 6.4         | Apache Software License                                                                             | Ned Batchelder and 152 others                                                                                    |\n| crashtest                 | 0.3.1       | MIT License                                                                                         | Sébastien Eustace                                                                                                |\n| cryptography              | 37.0.2      | Apache Software License; BSD License                                                                | The Python Cryptographic Authority and individual contributors                                                   |\n| cycler                    | 0.11.0      | BSD License                                                                                         | Thomas A Caswell                                                                                                 |\n| darglint                  | 1.8.1       | MIT License                                                                                         | terrencepreilly                                                                                                  |\n| debugpy                   | 1.6.0       | Eclipse Public License 2.0 (EPL-2.0); MIT License                                                   | Microsoft Corporation                                                                                            |\n| decorator                 | 5.1.1       | BSD License                                                                                         | Michele Simionato                                                                                                |\n| defusedxml                | 0.7.1       | Python Software Foundation License                                                                  | Christian Heimes                                                                                                 |\n| distlib                   | 0.3.4       | Python Software Foundation License                                                                  | Vinay Sajip                                                                                                      |\n| dnspython                 | 2.2.1       | ISC                                                                                                 | Bob Halley                                                                                                       |\n| dparse                    | 0.5.1       | MIT License                                                                                         | Jannis Gebauer                                                                                                   |\n| dynaconf                  | 3.1.8       | MIT License                                                                                         | Bruno Rocha                                                                                                      |\n| email-validator           | 1.2.1       | CC0 1.0 Universal (CC0 1.0) Public Domain Dedication                                                | Joshua Tauberer                                                                                                  |\n| entrypoints               | 0.4         | MIT License                                                                                         | Thomas Kluyver                                                                                                   |\n| eva                       | 1.0.1       | UNKNOWN                                                                                             | Microsoft Research EVA compiler team                                                                             |\n| factory-boy               | 3.2.1       | MIT License                                                                                         | Mark Sandstrom                                                                                                   |\n| fastapi                   | 0.66.1      | MIT License                                                                                         | Sebastián Ramírez                                                                                                |\n| fastjsonschema            | 2.15.3      | BSD License                                                                                         | Michal Horejsek                                                                                                  |\n| filelock                  | 3.7.0       | Public Domain                                                                                       | Benedikt Schmitt                                                                                                 |\n| flake8                    | 4.0.1       | MIT License                                                                                         | Tarek Ziade                                                                                                      |\n| flake8-annotations        | 2.9.0       | MIT License                                                                                         | S Co1                                                                                                            |\n| flake8-bandit             | 3.0.0       | MIT License                                                                                         | Tyler Wince                                                                                                      |\n| flake8-black              | 0.3.3       | MIT License                                                                                         | Peter J. A. Cock                                                                                                 |\n| flake8-bugbear            | 22.4.25     | MIT License                                                                                         | Łukasz Langa                                                                                                     |\n| flake8-docstrings         | 1.6.0       | MIT License                                                                                         | Simon ANDRÉ                                                                                                      |\n| flake8-import-order       | 0.18.1      | GNU Lesser General Public License v3 (LGPLv3); MIT License                                          | Alex Stapleton                                                                                                   |\n| flake8-polyfill           | 1.0.2       | MIT License                                                                                         | Ian Cordasco                                                                                                     |\n| fonttools                 | 4.33.3      | MIT License                                                                                         | Just van Rossum                                                                                                  |\n| forbiddenfruit            | 0.1.4       | GNU General Public License v3 or later (GPLv3+); MIT License                                        | Lincoln de Sousa                                                                                                 |\n| frozenlist                | 1.3.0       | Apache Software License                                                                             | UNKNOWN                                                                                                          |\n| fsspec                    | 2022.1.0    | BSD License                                                                                         | UNKNOWN                                                                                                          |\n| gensim                    | 4.2.0       | LGPL-2.1-only                                                                                       | Radim Rehurek                                                                                                    |\n| gevent                    | 21.8.0      | MIT License                                                                                         | Denis Bilenko                                                                                                    |\n| gitdb                     | 4.0.9       | BSD License                                                                                         | Sebastian Thiel                                                                                                  |\n| google-auth               | 2.6.6       | Apache Software License                                                                             | Google Cloud Platform                                                                                            |\n| google-auth-oauthlib      | 0.4.6       | Apache Software License                                                                             | Google Cloud Platform                                                                                            |\n| graphql-core              | 3.2.1       | MIT License                                                                                         | Christoph Zwerschke                                                                                              |\n| greenlet                  | 1.1.2       | MIT License                                                                                         | Alexey Borzenkov                                                                                                 |\n| griddify                  | 0.0.1       | MIT License                                                                                         | Miquel Duran-Frigola                                                                                             |\n| grpcio                    | 1.46.3      | Apache Software License                                                                             | The gRPC Authors                                                                                                 |\n| h11                       | 0.13.0      | MIT License                                                                                         | Nathaniel J. Smith                                                                                               |\n| html5lib                  | 1.1         | MIT License                                                                                         | UNKNOWN                                                                                                          |\n| httptools                 | 0.4.0       | MIT License                                                                                         | Yury Selivanov                                                                                                   |\n| identify                  | 2.5.1       | MIT License                                                                                         | Chris Kuehl                                                                                                      |\n| idna                      | 3.3         | BSD License                                                                                         | Kim Davies                                                                                                       |\n| importlib-metadata        | 4.11.4      | Apache Software License                                                                             | Jason R. Coombs                                                                                                  |\n| importlib-resources       | 5.7.1       | Apache Software License                                                                             | Barry Warsaw                                                                                                     |\n| iniconfig                 | 1.1.1       | MIT License                                                                                         | Ronny Pfannschmidt, Holger Krekel                                                                                |\n| ipykernel                 | 6.13.0      | BSD License                                                                                         | IPython Development Team                                                                                         |\n| ipython                   | 7.34.0      | BSD License                                                                                         | The IPython Development Team                                                                                     |\n| ipython-genutils          | 0.2.0       | BSD License                                                                                         | IPython Development Team                                                                                         |\n| ipywidgets                | 7.7.0       | BSD License                                                                                         | IPython Development Team                                                                                         |\n| itsdangerous              | 2.1.2       | BSD License                                                                                         | Armin Ronacher                                                                                                   |\n| jedi                      | 0.18.1      | MIT License                                                                                         | David Halter                                                                                                     |\n| jeepney                   | 0.8.0       | MIT License                                                                                         | Thomas Kluyver                                                                                                   |\n| jinja2-time               | 0.2.0       | MIT License                                                                                         | Raphael Pierzina                                                                                                 |\n| jmespath                  | 0.10.0      | MIT License                                                                                         | James Saryerwinnie                                                                                               |\n| joblib                    | 1.1.0       | BSD License                                                                                         | Gael Varoquaux                                                                                                   |\n| json5                     | 0.9.8       | Apache Software License                                                                             | Dirk Pranke                                                                                                      |\n| jsonschema                | 4.5.1       | MIT License                                                                                         | Julian Berman                                                                                                    |\n| jupyter-client            | 7.3.1       | BSD License                                                                                         | Jupyter Development Team                                                                                         |\n| jupyter-core              | 4.10.0      | BSD License                                                                                         | Jupyter Development Team                                                                                         |\n| jupyter-server            | 1.17.0      | BSD License                                                                                         | Jupyter Development Team                                                                                         |\n| jupyterlab                | 3.4.2       | BSD License                                                                                         | Jupyter Development Team                                                                                         |\n| jupyterlab-pygments       | 0.2.2       | BSD                                                                                                 | Jupyter Development Team                                                                                         |\n| jupyterlab-server         | 2.14.0      | BSD License                                                                                         | UNKNOWN                                                                                                          |\n| jupyterlab-widgets        | 1.1.0       | BSD License                                                                                         | Jupyter Development Team                                                                                         |\n| kedro                     | 0.18.1      | Apache Software License (Apache 2.0)                                                                | Kedro                                                                                                            |\n| kedro-viz                 | 4.6.0       | Apache Software License (Apache 2.0)                                                                | Kedro                                                                                                            |\n| keyring                   | 23.5.0      | MIT License; Python Software Foundation License                                                     | Kang Zhang                                                                                                       |\n| kiwisolver                | 1.4.2       | BSD License                                                                                         | UNKNOWN                                                                                                          |\n| lap                       | 0.4.0       | BSD (2-clause)                                                                                      | UNKNOWN                                                                                                          |\n| lark-parser               | 0.12.0      | MIT License                                                                                         | Erez Shinan                                                                                                      |\n| llvmlite                  | 0.38.1      | BSD                                                                                                 | UNKNOWN                                                                                                          |\n| lockfile                  | 0.12.2      | MIT License                                                                                         | OpenStack                                                                                                        |\n| loguru                    | 0.5.3       | MIT License                                                                                         | Delgan                                                                                                           |\n| logzero                   | 1.7.0       | MIT License                                                                                         | Chris Hager                                                                                                      |\n| matplotlib                | 3.5.2       | Python Software Foundation License                                                                  | John D. Hunter, Michael Droettboom                                                                               |\n| matplotlib-inline         | 0.1.3       | BSD 3-Clause                                                                                        | IPython Development Team                                                                                         |\n| mccabe                    | 0.6.1       | MIT License                                                                                         | Ian Cordasco                                                                                                     |\n| mistune                   | 0.8.4       | BSD License                                                                                         | Hsiaoming Yang                                                                                                   |\n| mpmath                    | 1.2.1       | BSD License                                                                                         | Fredrik Johansson                                                                                                |\n| msgpack                   | 1.0.3       | Apache Software License                                                                             | Inada Naoki                                                                                                      |\n| multidict                 | 6.0.2       | Apache Software License                                                                             | Andrew Svetlov                                                                                                   |\n| mypy                      | 0.942       | MIT License                                                                                         | Jukka Lehtosalo                                                                                                  |\n| mypy-extensions           | 0.4.3       | MIT License                                                                                         | The mypy developers                                                                                              |\n| names                     | 0.3.0       | MIT License                                                                                         | Trey Hunner                                                                                                      |\n| nbclassic                 | 0.3.7       | BSD License                                                                                         | Jupyter Development Team                                                                                         |\n| nbclient                  | 0.6.3       | BSD License                                                                                         | Jupyter Development Team                                                                                         |\n| nbconvert                 | 6.5.0       | BSD License                                                                                         | Jupyter Development Team                                                                                         |\n| nbformat                  | 5.4.0       | BSD License                                                                                         | Jupyter Development Team                                                                                         |\n| nest-asyncio              | 1.5.5       | BSD License                                                                                         | Ewald R. de Wit                                                                                                  |\n| networkx                  | 2.8.2       | BSD License                                                                                         | Aric Hagberg                                                                                                     |\n| nglview                   | 3.0.3       | MIT License                                                                                         | Alexander S. Rose, Hai Nguyen                                                                                    |\n| nltk                      | 3.7         | Apache Software License                                                                             | NLTK Team                                                                                                        |\n| nodeenv                   | 1.6.0       | BSD License                                                                                         | Eugene Kalinin                                                                                                   |\n| notebook                  | 6.4.11      | BSD License                                                                                         | Jupyter Development Team                                                                                         |\n| notebook-shim             | 0.1.0       | BSD License                                                                                         | Jupyter Development Team                                                                                         |\n| nox                       | 2022.1.7    | Apache Software License                                                                             | Alethea Katherine Flowers                                                                                        |\n| numba                     | 0.55.2      | BSD License                                                                                         | UNKNOWN                                                                                                          |\n| numpy                     | 1.22.4      | BSD License                                                                                         | Travis E. Oliphant et al.                                                                                        |\n| oauthlib                  | 3.2.0       | BSD License                                                                                         | The OAuthlib Community                                                                                           |\n| onnx                      | 1.11.0      | Apache License v2.0                                                                                 | ONNX                                                                                                             |\n| packaging                 | 21.3        | Apache Software License; BSD License                                                                | Donald Stufft and individual contributors                                                                        |\n| pandas                    | 1.3.4       | BSD License                                                                                         | The Pandas Development Team                                                                                      |\n| pandocfilters             | 1.5.0       | BSD License                                                                                         | John MacFarlane                                                                                                  |\n| parso                     | 0.8.3       | MIT License                                                                                         | David Halter                                                                                                     |\n| pastel                    | 0.2.1       | MIT License                                                                                         | Sébastien Eustace                                                                                                |\n| pathspec                  | 0.9.0       | Mozilla Public License 2.0 (MPL 2.0)                                                                | Caleb P. Burns                                                                                                   |\n| pbr                       | 5.9.0       | Apache Software License                                                                             | OpenStack                                                                                                        |\n| pep517                    | 0.12.0      | MIT License                                                                                         | Thomas Kluyver                                                                                                   |\n| pexpect                   | 4.8.0       | ISC License (ISCL)                                                                                  | Noah Spurrier; Thomas Kluyver; Jeff Quast                                                                        |\n| pickleshare               | 0.7.5       | MIT License                                                                                         | Ville Vainio                                                                                                     |\n| pip-tools                 | 6.6.2       | BSD License                                                                                         | Vincent Driessen                                                                                                 |\n| pkginfo                   | 1.8.2       | MIT License                                                                                         | Tres Seaver, Agendaless Consulting                                                                               |\n| platformdirs              | 2.5.2       | MIT License                                                                                         | UNKNOWN                                                                                                          |\n| plotly                    | 5.8.0       | MIT                                                                                                 | Chris P                                                                                                          |\n| pluggy                    | 1.0.0       | MIT License                                                                                         | Holger Krekel                                                                                                    |\n| poetry                    | 1.1.13      | MIT License                                                                                         | Sébastien Eustace                                                                                                |\n| poetry-core               | 1.0.8       | MIT License                                                                                         | Sébastien Eustace                                                                                                |\n| poyo                      | 0.5.0       | MIT License                                                                                         | Raphael Pierzina                                                                                                 |\n| pre-commit                | 2.19.0      | MIT License                                                                                         | Anthony Sottile                                                                                                  |\n| prometheus-client         | 0.14.1      | Apache Software License                                                                             | Brian Brazil                                                                                                     |\n| prompt-toolkit            | 3.0.29      | BSD License                                                                                         | Jonathan Slenders                                                                                                |\n| protobuf                  | 3.19.1      | 3-Clause BSD License                                                                                | UNKNOWN                                                                                                          |\n| psutil                    | 5.9.1       | BSD License                                                                                         | Giampaolo Rodola                                                                                                 |\n| ptyprocess                | 0.7.0       | ISC License (ISCL)                                                                                  | Thomas Kluyver                                                                                                   |\n| py                        | 1.11.0      | MIT License                                                                                         | holger krekel, Ronny Pfannschmidt, Benjamin Peterson and others                                                  |\n| pyDeprecate               | 0.3.2       | MIT                                                                                                 | Jiri Borovec                                                                                                     |\n| pyarrow                   | 6.0.0       | Apache Software License                                                                             | UNKNOWN                                                                                                          |\n| pyasn1                    | 0.4.8       | BSD License                                                                                         | Ilya Etingof                                                                                                     |\n| pyasn1-modules            | 0.2.8       | BSD License                                                                                         | Ilya Etingof                                                                                                     |\n| pycodestyle               | 2.8.0       | MIT License                                                                                         | Johann C. Rocholl                                                                                                |\n| pycparser                 | 2.21        | BSD License                                                                                         | Eli Bendersky                                                                                                    |\n| pydantic                  | 1.9.1       | MIT License                                                                                         | Samuel Colvin                                                                                                    |\n| pydocstyle                | 6.1.1       | MIT License                                                                                         | Amir Rachum                                                                                                      |\n| pyflakes                  | 2.4.0       | MIT License                                                                                         | A lot of people                                                                                                  |\n| pylev                     | 1.4.0       | BSD License                                                                                         | Daniel Lindsley                                                                                                  |\n| pymbolic                  | 2021.1      | MIT License                                                                                         | Andreas Kloeckner                                                                                                |\n| pynndescent               | 0.5.7       | BSD                                                                                                 | Leland McInnes                                                                                                   |\n| pyparsing                 | 3.0.9       | MIT License                                                                                         | UNKNOWN                                                                                                          |\n| pyrsistent                | 0.18.1      | MIT License                                                                                         | Tobias Gustafsson                                                                                                |\n| pytest                    | 7.1.2       | MIT License                                                                                         | Holger Krekel, Bruno Oliveira, Ronny Pfannschmidt, Floris Bruynooghe, Brianna Laugher, Florian Bruhin and others |\n| pytest-cov                | 3.0.0       | MIT License                                                                                         | Marc Schlaich                                                                                                    |\n| pytest-mock               | 3.7.0       | MIT License                                                                                         | Bruno Oliveira                                                                                                   |\n| python-dateutil           | 2.8.2       | Apache Software License; BSD License                                                                | Gustavo Niemeyer                                                                                                 |\n| python-dotenv             | 0.20.0      | BSD License                                                                                         | Saurabh Kumar                                                                                                    |\n| python-json-logger        | 2.0.2       | BSD License                                                                                         | Zakaria Zajac                                                                                                    |\n| python-multipart          | 0.0.5       | Apache Software License                                                                             | Andrew Dunham                                                                                                    |\n| python-slugify            | 6.1.2       | MIT License                                                                                         | Val Neekman                                                                                                      |\n| pytools                   | 2022.1.7    | MIT License                                                                                         | Andreas Kloeckner                                                                                                |\n| pytorch-lightning         | 1.6.3       | Apache Software License                                                                             | William Falcon et al.                                                                                            |\n| pytz                      | 2022.1      | MIT License                                                                                         | Stuart Bishop                                                                                                    |\n| pyzmq                     | 23.0.0      | BSD License; GNU Library or Lesser General Public License (LGPL)                                    | Brian E. Granger, Min Ragan-Kelley                                                                               |\n| rdkit-pypi                | 2022.3.2.1  | BSD-3-Clause                                                                                        | Christopher Kuenneth                                                                                             |\n| regex                     | 2022.4.24   | Apache Software License                                                                             | Matthew Barnett                                                                                                  |\n| requests                  | 2.27.1      | Apache Software License                                                                             | Kenneth Reitz                                                                                                    |\n| requests-oauthlib         | 1.3.1       | BSD License                                                                                         | Kenneth Reitz                                                                                                    |\n| requests-toolbelt         | 0.9.1       | Apache Software License                                                                             | Ian Cordasco, Cory Benfield                                                                                      |\n| rope                      | 0.21.1      | GNU Lesser General Public License v3 or later (LGPLv3+)                                             | Ali Gholami Rudi                                                                                                 |\n| rsa                       | 4.8         | Apache Software License                                                                             | Sybren A. Stüvel                                                                                                 |\n| safety                    | 1.10.3      | MIT License                                                                                         | pyup.io                                                                                                          |\n| scikit-learn              | 1.1.1       | new BSD                                                                                             | UNKNOWN                                                                                                          |\n| scipy                     | 1.7.3       | BSD License                                                                                         | UNKNOWN                                                                                                          |\n| semver                    | 2.13.0      | BSD License                                                                                         | Kostiantyn Rybnikov                                                                                              |\n| setuptools-scm            | 6.4.2       | MIT License                                                                                         | Ronny Pfannschmidt                                                                                               |\n| shellingham               | 1.4.0       | ISC License (ISCL)                                                                                  | Tzu-ping Chung                                                                                                   |\n| six                       | 1.16.0      | MIT License                                                                                         | Benjamin Peterson                                                                                                |\n| smart-open                | 6.0.0       | MIT License                                                                                         | Radim Rehurek                                                                                                    |\n| smmap                     | 5.0.0       | BSD License                                                                                         | Sebastian Thiel                                                                                                  |\n| sniffio                   | 1.2.0       | Apache Software License; MIT License                                                                | Nathaniel J. Smith                                                                                               |\n| snowballstemmer           | 2.2.0       | BSD License                                                                                         | Snowball Developers                                                                                              |\n| soupsieve                 | 2.3.2.post1 | MIT License                                                                                         | UNKNOWN                                                                                                          |\n| starlette                 | 0.14.2      | BSD License                                                                                         | Tom Christie                                                                                                     |\n| stevedore                 | 3.5.0       | Apache Software License                                                                             | OpenStack                                                                                                        |\n| strawberry-graphql        | 0.114.0     | MIT License                                                                                         | Patrick Arminio                                                                                                  |\n| syft                      | 0.6.0       | Apache-2.0                                                                                          | OpenMined                                                                                                        |\n| sympy                     | 1.9         | BSD License                                                                                         | SymPy development team                                                                                           |\n| tenacity                  | 8.0.1       | Apache Software License                                                                             | Julien Danjou                                                                                                    |\n| tenseal                   | 0.3.11      | Apache-2.0                                                                                          | OpenMined                                                                                                        |\n| tensorboard               | 2.9.0       | Apache Software License                                                                             | Google Inc.                                                                                                      |\n| tensorboard-data-server   | 0.6.1       | Apache Software License                                                                             | Google Inc.                                                                                                      |\n| tensorboard-plugin-wit    | 1.8.1       | Apache 2.0                                                                                          | Google Inc.                                                                                                      |\n| terminado                 | 0.15.0      | BSD License                                                                                         | UNKNOWN                                                                                                          |\n| text-unidecode            | 1.3         | Artistic License; GNU General Public License (GPL); GNU General Public License v2 or later (GPLv2+) | Mikhail Korobov                                                                                                  |\n| threadpoolctl             | 3.1.0       | BSD License                                                                                         | Thomas Moreau                                                                                                    |\n| tinycss2                  | 1.1.1       | BSD License                                                                                         | UNKNOWN                                                                                                          |\n| toml                      | 0.10.2      | MIT License                                                                                         | William Pearson                                                                                                  |\n| tomli                     | 2.0.1       | MIT License                                                                                         | UNKNOWN                                                                                                          |\n| tomlkit                   | 0.10.2      | MIT License                                                                                         | Sébastien Eustace                                                                                                |\n| toposort                  | 1.7         | Apache Software License                                                                             | "Eric V. Smith"                                                                                                  |\n| torch                     | 1.11.0      | BSD License                                                                                         | PyTorch Team                                                                                                     |\n| torch-tb-profiler         | 0.4.0       | BSD License                                                                                         | PyTorch Team                                                                                                     |\n| torchani                  | 2.2         | MIT                                                                                                 | Xiang Gao                                                                                                        |\n| torchmetrics              | 0.8.2       | Apache Software License                                                                             | PyTorchLightning et al.                                                                                          |\n| torchvision               | 0.12.0      | BSD                                                                                                 | PyTorch Core Team                                                                                                |\n| tornado                   | 6.1         | Apache Software License                                                                             | Facebook                                                                                                         |\n| tqdm                      | 4.64.0      | MIT License; Mozilla Public License 2.0 (MPL 2.0)                                                   | UNKNOWN                                                                                                          |\n| traitlets                 | 5.2.1.post0 | BSD License                                                                                         | UNKNOWN                                                                                                          |\n| typeguard                 | 2.13.3      | MIT License                                                                                         | Alex Grönholm                                                                                                    |\n| typing-extensions         | 4.2.0       | Python Software Foundation License                                                                  | UNKNOWN                                                                                                          |\n| umap-learn                | 0.5.3       | BSD                                                                                                 | UNKNOWN                                                                                                          |\n| urllib3                   | 1.26.9      | MIT License                                                                                         | Andrey Petrov                                                                                                    |\n| uvicorn                   | 0.17.6      | BSD License                                                                                         | Tom Christie                                                                                                     |\n| uvloop                    | 0.16.0      | Apache Software License; MIT License                                                                | Yury Selivanov                                                                                                   |\n| virtualenv                | 20.14.1     | MIT License                                                                                         | Bernat Gabor                                                                                                     |\n| watchgod                  | 0.8.2       | MIT License                                                                                         | Samuel Colvin                                                                                                    |\n| wcwidth                   | 0.2.5       | MIT License                                                                                         | Jeff Quast                                                                                                       |\n| webencodings              | 0.5.1       | BSD License                                                                                         | Geoffrey Sneddon                                                                                                 |\n| websocket-client          | 1.3.2       | Apache Software License                                                                             | liris                                                                                                            |\n| websockets                | 10.3        | BSD License                                                                                         | Aymeric Augustin                                                                                                 |\n| widgetsnbextension        | 3.6.0       | BSD License                                                                                         | IPython Development Team                                                                                         |\n| yarl                      | 1.7.2       | Apache Software License                                                                             | Andrew Svetlov                                                                                                   |\n| zipp                      | 3.8.0       | MIT License                                                                                         | Jason R. Coombs                                                                                                  |\n| zope.event                | 4.5.0       | Zope Public License                                                                                 | Zope Foundation and Contributors                                                                                 |\n| zope.interface            | 5.4.0       | Zope Public License                                                                                 | Zope Foundation and Contributors                                                                                 |\n',
    'author': 'Ankur Kumar',
    'author_email': 'ank@leoank.me',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<3.9',
}


setup(**setup_kwargs)
