"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const events = require("@aws-cdk/aws-events");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const targets = require("../../lib");
test('sqs queue as an event rule target', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const queue = new sqs.Queue(stack, 'MyQueue');
    const rule = new events.Rule(stack, 'MyRule', {
        schedule: events.Schedule.rate(core_1.Duration.hours(1)),
    });
    // WHEN
    rule.addTarget(new targets.SqsQueue(queue));
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::QueuePolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'sqs:SendMessage',
                        'sqs:GetQueueAttributes',
                        'sqs:GetQueueUrl',
                    ],
                    Condition: {
                        ArnEquals: {
                            'aws:SourceArn': {
                                'Fn::GetAtt': [
                                    'MyRuleA44AB831',
                                    'Arn',
                                ],
                            },
                        },
                    },
                    Effect: 'Allow',
                    Principal: { Service: 'events.amazonaws.com' },
                    Resource: {
                        'Fn::GetAtt': [
                            'MyQueueE6CA6235',
                            'Arn',
                        ],
                    },
                },
            ],
            Version: '2012-10-17',
        },
        Queues: [{ Ref: 'MyQueueE6CA6235' }],
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
        ScheduleExpression: 'rate(1 hour)',
        State: 'ENABLED',
        Targets: [
            {
                Arn: {
                    'Fn::GetAtt': [
                        'MyQueueE6CA6235',
                        'Arn',
                    ],
                },
                Id: 'Target0',
            },
        ],
    }));
});
test('multiple uses of a queue as a target results in multi policy statement because of condition', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const queue = new sqs.Queue(stack, 'MyQueue');
    // WHEN
    for (let i = 0; i < 2; ++i) {
        const rule = new events.Rule(stack, `Rule${i}`, {
            schedule: events.Schedule.rate(core_1.Duration.hours(1)),
        });
        rule.addTarget(new targets.SqsQueue(queue));
    }
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::QueuePolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'sqs:SendMessage',
                        'sqs:GetQueueAttributes',
                        'sqs:GetQueueUrl',
                    ],
                    Condition: {
                        ArnEquals: {
                            'aws:SourceArn': {
                                'Fn::GetAtt': [
                                    'Rule071281D88',
                                    'Arn',
                                ],
                            },
                        },
                    },
                    Effect: 'Allow',
                    Principal: { Service: 'events.amazonaws.com' },
                    Resource: {
                        'Fn::GetAtt': [
                            'MyQueueE6CA6235',
                            'Arn',
                        ],
                    },
                },
                {
                    Action: [
                        'sqs:SendMessage',
                        'sqs:GetQueueAttributes',
                        'sqs:GetQueueUrl',
                    ],
                    Condition: {
                        ArnEquals: {
                            'aws:SourceArn': {
                                'Fn::GetAtt': [
                                    'Rule136483A30',
                                    'Arn',
                                ],
                            },
                        },
                    },
                    Effect: 'Allow',
                    Principal: { Service: 'events.amazonaws.com' },
                    Resource: {
                        'Fn::GetAtt': [
                            'MyQueueE6CA6235',
                            'Arn',
                        ],
                    },
                },
            ],
            Version: '2012-10-17',
        },
        Queues: [{ Ref: 'MyQueueE6CA6235' }],
    }));
});
test('fail if messageGroupId is specified on non-fifo queues', () => {
    const stack = new core_1.Stack();
    const queue = new sqs.Queue(stack, 'MyQueue');
    expect(() => new targets.SqsQueue(queue, { messageGroupId: 'MyMessageGroupId' }))
        .toThrow(/messageGroupId cannot be specified/);
});
test('fifo queues are synthesized correctly', () => {
    const stack = new core_1.Stack();
    const queue = new sqs.Queue(stack, 'MyQueue', { fifo: true });
    const rule = new events.Rule(stack, 'MyRule', {
        schedule: events.Schedule.rate(core_1.Duration.hours(1)),
    });
    // WHEN
    rule.addTarget(new targets.SqsQueue(queue, {
        messageGroupId: 'MyMessageGroupId',
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
        ScheduleExpression: 'rate(1 hour)',
        State: 'ENABLED',
        Targets: [
            {
                Arn: {
                    'Fn::GetAtt': [
                        'MyQueueE6CA6235',
                        'Arn',
                    ],
                },
                Id: 'Target0',
                SqsParameters: {
                    MessageGroupId: 'MyMessageGroupId',
                },
            },
        ],
    }));
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3FzLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzcXMudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLDRDQUFvRTtBQUNwRSw4Q0FBOEM7QUFDOUMsd0NBQXdDO0FBQ3hDLHdDQUFnRDtBQUNoRCxxQ0FBcUM7QUFFckMsSUFBSSxDQUFDLG1DQUFtQyxFQUFFLEdBQUcsRUFBRTtJQUM3QyxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQzlDLE1BQU0sSUFBSSxHQUFHLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1FBQzVDLFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxlQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO0tBQ2xELENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksT0FBTyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO0lBRTVDLE9BQU87SUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUJBQXVCLEVBQUU7UUFDeEQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFO2dCQUNUO29CQUNFLE1BQU0sRUFBRTt3QkFDTixpQkFBaUI7d0JBQ2pCLHdCQUF3Qjt3QkFDeEIsaUJBQWlCO3FCQUNsQjtvQkFDRCxTQUFTLEVBQUU7d0JBQ1QsU0FBUyxFQUFFOzRCQUNULGVBQWUsRUFBRTtnQ0FDZixZQUFZLEVBQUU7b0NBQ1osZ0JBQWdCO29DQUNoQixLQUFLO2lDQUNOOzZCQUNGO3lCQUNGO3FCQUNGO29CQUNELE1BQU0sRUFBRSxPQUFPO29CQUNmLFNBQVMsRUFBRSxFQUFFLE9BQU8sRUFBRSxzQkFBc0IsRUFBRTtvQkFDOUMsUUFBUSxFQUFFO3dCQUNSLFlBQVksRUFBRTs0QkFDWixpQkFBaUI7NEJBQ2pCLEtBQUs7eUJBQ047cUJBQ0Y7aUJBQ0Y7YUFDRjtZQUNELE9BQU8sRUFBRSxZQUFZO1NBQ3RCO1FBQ0QsTUFBTSxFQUFFLENBQUMsRUFBRSxHQUFHLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQztLQUNyQyxDQUFDLENBQUMsQ0FBQztJQUVKLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxtQkFBbUIsRUFBRTtRQUNwRCxrQkFBa0IsRUFBRSxjQUFjO1FBQ2xDLEtBQUssRUFBRSxTQUFTO1FBQ2hCLE9BQU8sRUFBRTtZQUNQO2dCQUNFLEdBQUcsRUFBRTtvQkFDSCxZQUFZLEVBQUU7d0JBQ1osaUJBQWlCO3dCQUNqQixLQUFLO3FCQUNOO2lCQUNGO2dCQUNELEVBQUUsRUFBRSxTQUFTO2FBQ2Q7U0FDRjtLQUNGLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNkZBQTZGLEVBQUUsR0FBRyxFQUFFO0lBQ3ZHLFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFFOUMsT0FBTztJQUNQLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLEVBQUU7UUFDMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsRUFBRSxFQUFFO1lBQzlDLFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxlQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ2xELENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxPQUFPLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7S0FDN0M7SUFFRCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixFQUFFO1FBQ3hELGNBQWMsRUFBRTtZQUNkLFNBQVMsRUFBRTtnQkFDVDtvQkFDRSxNQUFNLEVBQUU7d0JBQ04saUJBQWlCO3dCQUNqQix3QkFBd0I7d0JBQ3hCLGlCQUFpQjtxQkFDbEI7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULFNBQVMsRUFBRTs0QkFDVCxlQUFlLEVBQUU7Z0NBQ2YsWUFBWSxFQUFFO29DQUNaLGVBQWU7b0NBQ2YsS0FBSztpQ0FDTjs2QkFDRjt5QkFDRjtxQkFDRjtvQkFDRCxNQUFNLEVBQUUsT0FBTztvQkFDZixTQUFTLEVBQUUsRUFBRSxPQUFPLEVBQUUsc0JBQXNCLEVBQUU7b0JBQzlDLFFBQVEsRUFBRTt3QkFDUixZQUFZLEVBQUU7NEJBQ1osaUJBQWlCOzRCQUNqQixLQUFLO3lCQUNOO3FCQUNGO2lCQUNGO2dCQUNEO29CQUNFLE1BQU0sRUFBRTt3QkFDTixpQkFBaUI7d0JBQ2pCLHdCQUF3Qjt3QkFDeEIsaUJBQWlCO3FCQUNsQjtvQkFDRCxTQUFTLEVBQUU7d0JBQ1QsU0FBUyxFQUFFOzRCQUNULGVBQWUsRUFBRTtnQ0FDZixZQUFZLEVBQUU7b0NBQ1osZUFBZTtvQ0FDZixLQUFLO2lDQUNOOzZCQUNGO3lCQUNGO3FCQUNGO29CQUNELE1BQU0sRUFBRSxPQUFPO29CQUNmLFNBQVMsRUFBRSxFQUFFLE9BQU8sRUFBRSxzQkFBc0IsRUFBRTtvQkFDOUMsUUFBUSxFQUFFO3dCQUNSLFlBQVksRUFBRTs0QkFDWixpQkFBaUI7NEJBQ2pCLEtBQUs7eUJBQ047cUJBQ0Y7aUJBQ0Y7YUFDRjtZQUNELE9BQU8sRUFBRSxZQUFZO1NBQ3RCO1FBQ0QsTUFBTSxFQUFFLENBQUMsRUFBRSxHQUFHLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQztLQUNyQyxDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHdEQUF3RCxFQUFFLEdBQUcsRUFBRTtJQUNsRSxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFFOUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksT0FBTyxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsRUFBRSxjQUFjLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxDQUFDO1NBQzlFLE9BQU8sQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO0FBQ25ELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHVDQUF1QyxFQUFFLEdBQUcsRUFBRTtJQUNqRCxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7SUFDOUQsTUFBTSxJQUFJLEdBQUcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7UUFDNUMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLGVBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7S0FDbEQsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxPQUFPLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRTtRQUN6QyxjQUFjLEVBQUUsa0JBQWtCO0tBQ25DLENBQUMsQ0FBQyxDQUFDO0lBRUosZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLG1CQUFtQixFQUFFO1FBQ3BELGtCQUFrQixFQUFFLGNBQWM7UUFDbEMsS0FBSyxFQUFFLFNBQVM7UUFDaEIsT0FBTyxFQUFFO1lBQ1A7Z0JBQ0UsR0FBRyxFQUFFO29CQUNILFlBQVksRUFBRTt3QkFDWixpQkFBaUI7d0JBQ2pCLEtBQUs7cUJBQ047aUJBQ0Y7Z0JBQ0QsRUFBRSxFQUFFLFNBQVM7Z0JBQ2IsYUFBYSxFQUFFO29CQUNiLGNBQWMsRUFBRSxrQkFBa0I7aUJBQ25DO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBleHBlY3QgYXMgY2RrRXhwZWN0LCBoYXZlUmVzb3VyY2UgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0ICogYXMgZXZlbnRzIGZyb20gJ0Bhd3MtY2RrL2F3cy1ldmVudHMnO1xuaW1wb3J0ICogYXMgc3FzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zcXMnO1xuaW1wb3J0IHsgRHVyYXRpb24sIFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyB0YXJnZXRzIGZyb20gJy4uLy4uL2xpYic7XG5cbnRlc3QoJ3NxcyBxdWV1ZSBhcyBhbiBldmVudCBydWxlIHRhcmdldCcsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgY29uc3QgcXVldWUgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAnTXlRdWV1ZScpO1xuICBjb25zdCBydWxlID0gbmV3IGV2ZW50cy5SdWxlKHN0YWNrLCAnTXlSdWxlJywge1xuICAgIHNjaGVkdWxlOiBldmVudHMuU2NoZWR1bGUucmF0ZShEdXJhdGlvbi5ob3VycygxKSksXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgcnVsZS5hZGRUYXJnZXQobmV3IHRhcmdldHMuU3FzUXVldWUocXVldWUpKTtcblxuICAvLyBUSEVOXG4gIGNka0V4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OlNRUzo6UXVldWVQb2xpY3knLCB7XG4gICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAnc3FzOlNlbmRNZXNzYWdlJyxcbiAgICAgICAgICAgICdzcXM6R2V0UXVldWVBdHRyaWJ1dGVzJyxcbiAgICAgICAgICAgICdzcXM6R2V0UXVldWVVcmwnLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgQ29uZGl0aW9uOiB7XG4gICAgICAgICAgICBBcm5FcXVhbHM6IHtcbiAgICAgICAgICAgICAgJ2F3czpTb3VyY2VBcm4nOiB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICAgICAnTXlSdWxlQTQ0QUI4MzEnLFxuICAgICAgICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgUHJpbmNpcGFsOiB7IFNlcnZpY2U6ICdldmVudHMuYW1hem9uYXdzLmNvbScgfSxcbiAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICdNeVF1ZXVlRTZDQTYyMzUnLFxuICAgICAgICAgICAgICAnQXJuJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgfSxcbiAgICBRdWV1ZXM6IFt7IFJlZjogJ015UXVldWVFNkNBNjIzNScgfV0sXG4gIH0pKTtcblxuICBjZGtFeHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFdmVudHM6OlJ1bGUnLCB7XG4gICAgU2NoZWR1bGVFeHByZXNzaW9uOiAncmF0ZSgxIGhvdXIpJyxcbiAgICBTdGF0ZTogJ0VOQUJMRUQnLFxuICAgIFRhcmdldHM6IFtcbiAgICAgIHtcbiAgICAgICAgQXJuOiB7XG4gICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAnTXlRdWV1ZUU2Q0E2MjM1JyxcbiAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICAgIElkOiAnVGFyZ2V0MCcsXG4gICAgICB9LFxuICAgIF0sXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdtdWx0aXBsZSB1c2VzIG9mIGEgcXVldWUgYXMgYSB0YXJnZXQgcmVzdWx0cyBpbiBtdWx0aSBwb2xpY3kgc3RhdGVtZW50IGJlY2F1c2Ugb2YgY29uZGl0aW9uJywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBjb25zdCBxdWV1ZSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdNeVF1ZXVlJyk7XG5cbiAgLy8gV0hFTlxuICBmb3IgKGxldCBpID0gMDsgaSA8IDI7ICsraSkge1xuICAgIGNvbnN0IHJ1bGUgPSBuZXcgZXZlbnRzLlJ1bGUoc3RhY2ssIGBSdWxlJHtpfWAsIHtcbiAgICAgIHNjaGVkdWxlOiBldmVudHMuU2NoZWR1bGUucmF0ZShEdXJhdGlvbi5ob3VycygxKSksXG4gICAgfSk7XG4gICAgcnVsZS5hZGRUYXJnZXQobmV3IHRhcmdldHMuU3FzUXVldWUocXVldWUpKTtcbiAgfVxuXG4gIC8vIFRIRU5cbiAgY2RrRXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6U1FTOjpRdWV1ZVBvbGljeScsIHtcbiAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICdzcXM6U2VuZE1lc3NhZ2UnLFxuICAgICAgICAgICAgJ3NxczpHZXRRdWV1ZUF0dHJpYnV0ZXMnLFxuICAgICAgICAgICAgJ3NxczpHZXRRdWV1ZVVybCcsXG4gICAgICAgICAgXSxcbiAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgIEFybkVxdWFsczoge1xuICAgICAgICAgICAgICAnYXdzOlNvdXJjZUFybic6IHtcbiAgICAgICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgICAgICdSdWxlMDcxMjgxRDg4JyxcbiAgICAgICAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgIFByaW5jaXBhbDogeyBTZXJ2aWNlOiAnZXZlbnRzLmFtYXpvbmF3cy5jb20nIH0sXG4gICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAnTXlRdWV1ZUU2Q0E2MjM1JyxcbiAgICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICdzcXM6U2VuZE1lc3NhZ2UnLFxuICAgICAgICAgICAgJ3NxczpHZXRRdWV1ZUF0dHJpYnV0ZXMnLFxuICAgICAgICAgICAgJ3NxczpHZXRRdWV1ZVVybCcsXG4gICAgICAgICAgXSxcbiAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgIEFybkVxdWFsczoge1xuICAgICAgICAgICAgICAnYXdzOlNvdXJjZUFybic6IHtcbiAgICAgICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgICAgICdSdWxlMTM2NDgzQTMwJyxcbiAgICAgICAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgIFByaW5jaXBhbDogeyBTZXJ2aWNlOiAnZXZlbnRzLmFtYXpvbmF3cy5jb20nIH0sXG4gICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAnTXlRdWV1ZUU2Q0E2MjM1JyxcbiAgICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnLFxuICAgIH0sXG4gICAgUXVldWVzOiBbeyBSZWY6ICdNeVF1ZXVlRTZDQTYyMzUnIH1dLFxuICB9KSk7XG59KTtcblxudGVzdCgnZmFpbCBpZiBtZXNzYWdlR3JvdXBJZCBpcyBzcGVjaWZpZWQgb24gbm9uLWZpZm8gcXVldWVzJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBjb25zdCBxdWV1ZSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdNeVF1ZXVlJyk7XG5cbiAgZXhwZWN0KCgpID0+IG5ldyB0YXJnZXRzLlNxc1F1ZXVlKHF1ZXVlLCB7IG1lc3NhZ2VHcm91cElkOiAnTXlNZXNzYWdlR3JvdXBJZCcgfSkpXG4gICAgLnRvVGhyb3coL21lc3NhZ2VHcm91cElkIGNhbm5vdCBiZSBzcGVjaWZpZWQvKTtcbn0pO1xuXG50ZXN0KCdmaWZvIHF1ZXVlcyBhcmUgc3ludGhlc2l6ZWQgY29ycmVjdGx5JywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBjb25zdCBxdWV1ZSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdNeVF1ZXVlJywgeyBmaWZvOiB0cnVlIH0pO1xuICBjb25zdCBydWxlID0gbmV3IGV2ZW50cy5SdWxlKHN0YWNrLCAnTXlSdWxlJywge1xuICAgIHNjaGVkdWxlOiBldmVudHMuU2NoZWR1bGUucmF0ZShEdXJhdGlvbi5ob3VycygxKSksXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgcnVsZS5hZGRUYXJnZXQobmV3IHRhcmdldHMuU3FzUXVldWUocXVldWUsIHtcbiAgICBtZXNzYWdlR3JvdXBJZDogJ015TWVzc2FnZUdyb3VwSWQnLFxuICB9KSk7XG5cbiAgY2RrRXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RXZlbnRzOjpSdWxlJywge1xuICAgIFNjaGVkdWxlRXhwcmVzc2lvbjogJ3JhdGUoMSBob3VyKScsXG4gICAgU3RhdGU6ICdFTkFCTEVEJyxcbiAgICBUYXJnZXRzOiBbXG4gICAgICB7XG4gICAgICAgIEFybjoge1xuICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgJ015UXVldWVFNkNBNjIzNScsXG4gICAgICAgICAgICAnQXJuJyxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgICBJZDogJ1RhcmdldDAnLFxuICAgICAgICBTcXNQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgTWVzc2FnZUdyb3VwSWQ6ICdNeU1lc3NhZ2VHcm91cElkJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgXSxcbiAgfSkpO1xufSk7XG4iXX0=