"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const events = require("@aws-cdk/aws-events");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const targets = require("../../lib");
test('use lambda as an event rule target', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const fn = newTestLambda(stack);
    const rule1 = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.rate(cdk.Duration.minutes(1)),
    });
    const rule2 = new events.Rule(stack, 'Rule2', {
        schedule: events.Schedule.rate(cdk.Duration.minutes(5)),
    });
    // WHEN
    rule1.addTarget(new targets.LambdaFunction(fn));
    rule2.addTarget(new targets.LambdaFunction(fn));
    // THEN
    const lambdaId = 'MyLambdaCCE802FB';
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
        Action: 'lambda:InvokeFunction',
        FunctionName: {
            'Fn::GetAtt': [
                lambdaId,
                'Arn',
            ],
        },
        Principal: 'events.amazonaws.com',
        SourceArn: { 'Fn::GetAtt': ['Rule4C995B7F', 'Arn'] },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
        Action: 'lambda:InvokeFunction',
        FunctionName: {
            'Fn::GetAtt': [
                lambdaId,
                'Arn',
            ],
        },
        Principal: 'events.amazonaws.com',
        SourceArn: { 'Fn::GetAtt': ['Rule270732244', 'Arn'] },
    }));
    assert_1.expect(stack).to(assert_1.countResources('AWS::Events::Rule', 2));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
        Targets: [
            {
                Arn: { 'Fn::GetAtt': [lambdaId, 'Arn'] },
                Id: 'Target0',
            },
        ],
    }));
});
test('adding same lambda function as target mutiple times creates permission only once', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const fn = newTestLambda(stack);
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.rate(cdk.Duration.minutes(1)),
    });
    // WHEN
    rule.addTarget(new targets.LambdaFunction(fn, {
        event: events.RuleTargetInput.fromObject({ key: 'value1' }),
    }));
    rule.addTarget(new targets.LambdaFunction(fn, {
        event: events.RuleTargetInput.fromObject({ key: 'value2' }),
    }));
    // THEN
    assert_1.expect(stack).to(assert_1.countResources('AWS::Lambda::Permission', 1));
});
test('adding same singleton lambda function as target mutiple times creates permission only once', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const fn = new lambda.SingletonFunction(stack, 'MyLambda', {
        code: new lambda.InlineCode('foo'),
        handler: 'bar',
        runtime: lambda.Runtime.PYTHON_2_7,
        uuid: 'uuid',
    });
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.rate(cdk.Duration.minutes(1)),
    });
    // WHEN
    rule.addTarget(new targets.LambdaFunction(fn, {
        event: events.RuleTargetInput.fromObject({ key: 'value1' }),
    }));
    rule.addTarget(new targets.LambdaFunction(fn, {
        event: events.RuleTargetInput.fromObject({ key: 'value2' }),
    }));
    // THEN
    assert_1.expect(stack).to(assert_1.countResources('AWS::Lambda::Permission', 1));
});
function newTestLambda(scope) {
    return new lambda.Function(scope, 'MyLambda', {
        code: new lambda.InlineCode('foo'),
        handler: 'bar',
        runtime: lambda.Runtime.PYTHON_2_7,
    });
}
//# sourceMappingURL=data:application/json;base64,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