"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const events = require("@aws-cdk/aws-events");
const kinesis = require("@aws-cdk/aws-kinesis");
const core_1 = require("@aws-cdk/core");
const targets = require("../../lib");
describe('KinesisStream event target', () => {
    let stack;
    let stream;
    let streamArn;
    beforeEach(() => {
        stack = new core_1.Stack();
        stream = new kinesis.Stream(stack, 'MyStream');
        streamArn = { 'Fn::GetAtt': ['MyStream5C050E93', 'Arn'] };
    });
    describe('when added to an event rule as a target', () => {
        let rule;
        beforeEach(() => {
            rule = new events.Rule(stack, 'rule', {
                schedule: events.Schedule.expression('rate(1 minute)'),
            });
        });
        describe('with default settings', () => {
            beforeEach(() => {
                rule.addTarget(new targets.KinesisStream(stream));
            });
            test("adds the stream's ARN and role to the targets of the rule", () => {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
                    Targets: [
                        {
                            Arn: streamArn,
                            Id: 'Target0',
                            RoleArn: { 'Fn::GetAtt': ['MyStreamEventsRole5B6CC6AF', 'Arn'] },
                        },
                    ],
                }));
            });
            test("creates a policy that has PutRecord and PutRecords permissions on the stream's ARN", () => {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: [
                            {
                                Action: ['kinesis:PutRecord', 'kinesis:PutRecords'],
                                Effect: 'Allow',
                                Resource: streamArn,
                            },
                        ],
                        Version: '2012-10-17',
                    },
                }));
            });
        });
        describe('with an explicit partition key path', () => {
            beforeEach(() => {
                rule.addTarget(new targets.KinesisStream(stream, {
                    partitionKeyPath: events.EventField.eventId,
                }));
            });
            test('sets the partition key path', () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                    Targets: [
                        {
                            Arn: streamArn,
                            Id: 'Target0',
                            RoleArn: { 'Fn::GetAtt': ['MyStreamEventsRole5B6CC6AF', 'Arn'] },
                            KinesisParameters: {
                                PartitionKeyPath: '$.id',
                            },
                        },
                    ],
                }));
            });
        });
        describe('with an explicit message', () => {
            beforeEach(() => {
                rule.addTarget(new targets.KinesisStream(stream, {
                    message: events.RuleTargetInput.fromText('fooBar'),
                }));
            });
            test('sets the input', () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                    Targets: [
                        {
                            Arn: streamArn,
                            Id: 'Target0',
                            Input: '"fooBar"',
                        },
                    ],
                }));
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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