"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const targets = require("../../lib");
test('Can use EC2 taskdef as EventRule target', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    cluster.addCapacity('DefaultAutoScalingGroup', {
        instanceType: new ec2.InstanceType('t2.micro'),
    });
    const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
        memoryLimitMiB: 256,
    });
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    // WHEN
    rule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
        taskCount: 1,
        containerOverrides: [{
                containerName: 'TheContainer',
                command: ['echo', events.EventField.fromPath('$.detail.event')],
            }],
    }));
    // THEN
    expect(stack).toHaveResourceLike('AWS::Events::Rule', {
        Targets: [
            {
                Arn: { 'Fn::GetAtt': ['EcsCluster97242B84', 'Arn'] },
                EcsParameters: {
                    TaskCount: 1,
                    TaskDefinitionArn: { Ref: 'TaskDef54694570' },
                },
                InputTransformer: {
                    InputPathsMap: {
                        'detail-event': '$.detail.event',
                    },
                    InputTemplate: '{"containerOverrides":[{"name":"TheContainer","command":["echo",<detail-event>]}]}',
                },
                RoleArn: { 'Fn::GetAtt': ['TaskDefEventsRoleFB3B67B8', 'Arn'] },
                Id: 'Target0',
            },
        ],
    });
});
test('Can use Fargate taskdef as EventRule target', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
    });
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    // WHEN
    const target = new targets.EcsTask({
        cluster,
        taskDefinition,
        taskCount: 1,
        containerOverrides: [{
                containerName: 'TheContainer',
                command: ['echo', events.EventField.fromPath('$.detail.event')],
            }],
    });
    rule.addTarget(target);
    // THEN
    expect(target.securityGroup).toBeDefined(); // Generated security groups should be accessible.
    expect(stack).toHaveResourceLike('AWS::Events::Rule', {
        Targets: [
            {
                Arn: { 'Fn::GetAtt': ['EcsCluster97242B84', 'Arn'] },
                EcsParameters: {
                    TaskCount: 1,
                    TaskDefinitionArn: { Ref: 'TaskDef54694570' },
                    LaunchType: 'FARGATE',
                    NetworkConfiguration: {
                        AwsVpcConfiguration: {
                            Subnets: [
                                {
                                    Ref: 'VpcPrivateSubnet1Subnet536B997A',
                                },
                            ],
                            AssignPublicIp: 'DISABLED',
                            SecurityGroups: [
                                {
                                    'Fn::GetAtt': [
                                        'TaskDefSecurityGroupD50E7CF0',
                                        'GroupId',
                                    ],
                                },
                            ],
                        },
                    },
                },
                InputTransformer: {
                    InputPathsMap: {
                        'detail-event': '$.detail.event',
                    },
                    InputTemplate: '{"containerOverrides":[{"name":"TheContainer","command":["echo",<detail-event>]}]}',
                },
                RoleArn: { 'Fn::GetAtt': ['TaskDefEventsRoleFB3B67B8', 'Arn'] },
                Id: 'Target0',
            },
        ],
    });
});
test('Can use same fargate taskdef with multiple rules', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
    });
    const scheduledRule = new events.Rule(stack, 'ScheduleRule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    const patternRule = new events.Rule(stack, 'PatternRule', {
        eventPattern: {
            detail: ['test'],
        },
    });
    scheduledRule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
    }));
    expect(() => patternRule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
    }))).not.toThrow();
});
test('Can use same fargate taskdef multiple times in a rule', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
    });
    const rule = new events.Rule(stack, 'ScheduleRule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    rule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
        containerOverrides: [{
                containerName: 'TheContainer',
                command: ['echo', events.EventField.fromPath('$.detail.a')],
            }],
    }));
    expect(() => rule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
        containerOverrides: [{
                containerName: 'TheContainer',
                command: ['echo', events.EventField.fromPath('$.detail.b')],
            }],
    }))).not.toThrow();
});
test('Isolated subnet does not have AssignPublicIp=true', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', {
        maxAzs: 1,
        subnetConfiguration: [{
                subnetType: ec2.SubnetType.ISOLATED,
                name: 'Isolated',
            }],
    });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
    });
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    // WHEN
    rule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
        taskCount: 1,
        subnetSelection: { subnetType: ec2.SubnetType.ISOLATED },
        containerOverrides: [{
                containerName: 'TheContainer',
                command: ['echo', 'yay'],
            }],
    }));
    // THEN
    expect(stack).toHaveResourceLike('AWS::Events::Rule', {
        Targets: [
            {
                Arn: { 'Fn::GetAtt': ['EcsCluster97242B84', 'Arn'] },
                EcsParameters: {
                    TaskCount: 1,
                    TaskDefinitionArn: { Ref: 'TaskDef54694570' },
                    LaunchType: 'FARGATE',
                    NetworkConfiguration: {
                        AwsVpcConfiguration: {
                            Subnets: [
                                {
                                    Ref: 'VpcIsolatedSubnet1SubnetE48C5737',
                                },
                            ],
                            AssignPublicIp: 'DISABLED',
                            SecurityGroups: [
                                {
                                    'Fn::GetAtt': [
                                        'TaskDefSecurityGroupD50E7CF0',
                                        'GroupId',
                                    ],
                                },
                            ],
                        },
                    },
                },
                Input: '{"containerOverrides":[{"name":"TheContainer","command":["echo","yay"]}]}',
                RoleArn: { 'Fn::GetAtt': ['TaskDefEventsRoleFB3B67B8', 'Arn'] },
                Id: 'Target0',
            },
        ],
    });
});
test('throws an error if both securityGroup and securityGroups is specified', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
    });
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    const securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc });
    // THEN
    expect(() => {
        rule.addTarget(new targets.EcsTask({
            cluster,
            taskDefinition,
            taskCount: 1,
            securityGroup,
            securityGroups: [securityGroup],
            containerOverrides: [{
                    containerName: 'TheContainer',
                    command: ['echo', 'yay'],
                }],
        }));
    }).toThrow(/Only one of SecurityGroup or SecurityGroups can be populated./);
});
test('uses multiple security groups', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
    });
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    const securityGroups = [
        new ec2.SecurityGroup(stack, 'SecurityGroupA', { vpc }),
        new ec2.SecurityGroup(stack, 'SecurityGroupB', { vpc }),
    ];
    // WHEN
    rule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
        taskCount: 1,
        securityGroups,
        containerOverrides: [{
                containerName: 'TheContainer',
                command: ['echo', 'yay'],
            }],
    }));
    // THEN
    expect(stack).toHaveResourceLike('AWS::Events::Rule', {
        Targets: [
            {
                Arn: { 'Fn::GetAtt': ['EcsCluster97242B84', 'Arn'] },
                EcsParameters: {
                    LaunchType: 'FARGATE',
                    NetworkConfiguration: {
                        AwsVpcConfiguration: {
                            AssignPublicIp: 'DISABLED',
                            SecurityGroups: [
                                { 'Fn::GetAtt': ['SecurityGroupAED40ADC5', 'GroupId'] },
                                { 'Fn::GetAtt': ['SecurityGroupB04591F90', 'GroupId'] },
                            ],
                            Subnets: [{ Ref: 'VpcPrivateSubnet1Subnet536B997A' }],
                        },
                    },
                    TaskCount: 1,
                    TaskDefinitionArn: {
                        Ref: 'TaskDef54694570',
                    },
                },
                Id: 'Target0',
                Input: '{"containerOverrides":[{"name":"TheContainer","command":["echo","yay"]}]}',
                RoleArn: { 'Fn::GetAtt': ['TaskDefEventsRoleFB3B67B8', 'Arn'] },
            },
        ],
    });
});
test('uses existing IAM role', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    const role = new iam.Role(stack, 'CustomIamRole', {
        assumedBy: new iam.ServicePrincipal('events.amazonaws.com'),
    });
    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
    });
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    // WHEN
    rule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
        taskCount: 1,
        containerOverrides: [{
                containerName: 'TheContainer',
                command: ['echo', events.EventField.fromPath('$.detail.event')],
            }],
        role,
    }));
    // THEN
    expect(stack).toHaveResourceLike('AWS::Events::Rule', {
        Targets: [
            {
                Arn: { 'Fn::GetAtt': ['EcsCluster97242B84', 'Arn'] },
                EcsParameters: {
                    LaunchType: 'FARGATE',
                    TaskCount: 1,
                    TaskDefinitionArn: {
                        Ref: 'TaskDef54694570',
                    },
                },
                RoleArn: { 'Fn::GetAtt': ['CustomIamRoleE653F2D1', 'Arn'] },
                Id: 'Target0',
            },
        ],
    });
});
test('uses the specific fargate platform version', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'Vpc', { maxAzs: 1 });
    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
    const platformVersion = ecs.FargatePlatformVersion.VERSION1_4;
    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef');
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('henk'),
    });
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    // WHEN
    rule.addTarget(new targets.EcsTask({
        cluster,
        taskDefinition,
        taskCount: 1,
        containerOverrides: [{
                containerName: 'TheContainer',
                command: ['echo', events.EventField.fromPath('$.detail.event')],
            }],
        platformVersion,
    }));
    // THEN
    expect(stack).toHaveResourceLike('AWS::Events::Rule', {
        Targets: [
            {
                Arn: { 'Fn::GetAtt': ['EcsCluster97242B84', 'Arn'] },
                EcsParameters: {
                    LaunchType: 'FARGATE',
                    TaskCount: 1,
                    TaskDefinitionArn: {
                        Ref: 'TaskDef54694570',
                    },
                    PlatformVersion: '1.4.0',
                },
                Id: 'Target0',
            },
        ],
    });
});
//# sourceMappingURL=data:application/json;base64,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