"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const targets = require("../../lib");
describe('CodePipeline event target', () => {
    let stack;
    let pipeline;
    let pipelineArn;
    beforeEach(() => {
        stack = new core_1.Stack();
        pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
        const srcArtifact = new codepipeline.Artifact('Src');
        const buildArtifact = new codepipeline.Artifact('Bld');
        pipeline.addStage({
            stageName: 'Source',
            actions: [new TestAction({
                    actionName: 'Hello',
                    category: codepipeline.ActionCategory.SOURCE,
                    provider: 'x',
                    artifactBounds: { minInputs: 0, maxInputs: 0, minOutputs: 1, maxOutputs: 1 },
                    outputs: [srcArtifact],
                })],
        });
        pipeline.addStage({
            stageName: 'Build',
            actions: [new TestAction({
                    actionName: 'Hello',
                    category: codepipeline.ActionCategory.BUILD,
                    provider: 'y',
                    inputs: [srcArtifact],
                    outputs: [buildArtifact],
                    artifactBounds: { minInputs: 1, maxInputs: 1, minOutputs: 1, maxOutputs: 1 },
                })],
        });
        pipelineArn = {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':codepipeline:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'PipelineC660917D' },
                ]],
        };
    });
    describe('when added to an event rule as a target', () => {
        let rule;
        beforeEach(() => {
            rule = new events.Rule(stack, 'rule', {
                schedule: events.Schedule.expression('rate(1 minute)'),
            });
        });
        describe('with default settings', () => {
            beforeEach(() => {
                rule.addTarget(new targets.CodePipeline(pipeline));
            });
            test("adds the pipeline's ARN and role to the targets of the rule", () => {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
                    Targets: [
                        {
                            Arn: pipelineArn,
                            Id: 'Target0',
                            RoleArn: { 'Fn::GetAtt': ['PipelineEventsRole46BEEA7C', 'Arn'] },
                        },
                    ],
                }));
            });
            test("creates a policy that has StartPipeline permissions on the pipeline's ARN", () => {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: [
                            {
                                Action: 'codepipeline:StartPipelineExecution',
                                Effect: 'Allow',
                                Resource: pipelineArn,
                            },
                        ],
                        Version: '2012-10-17',
                    },
                }));
            });
        });
        describe('with an explicit event role', () => {
            beforeEach(() => {
                const role = new iam.Role(stack, 'MyExampleRole', {
                    assumedBy: new iam.AnyPrincipal(),
                });
                const roleResource = role.node.defaultChild;
                roleResource.overrideLogicalId('MyRole'); // to make it deterministic in the assertion below
                rule.addTarget(new targets.CodePipeline(pipeline, {
                    eventRole: role,
                }));
            });
            test("points at the given event role in the rule's targets", () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                    Targets: [
                        {
                            Arn: pipelineArn,
                            RoleArn: { 'Fn::GetAtt': ['MyRole', 'Arn'] },
                        },
                    ],
                }));
            });
        });
    });
});
class TestAction {
    constructor(actionProperties) {
        this.actionProperties = actionProperties;
        // nothing to do
    }
    bind(_scope, _stage, _options) {
        return {};
    }
    onStateChange(_name, _target, _options) {
        throw new Error('onStateChange() is not available on MockAction');
    }
}
//# sourceMappingURL=data:application/json;base64,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