"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const targets = require("../../lib");
test('use AwsApi as an event rule target', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(15 minutes)'),
    });
    // WHEN
    rule.addTarget(new targets.AwsApi({
        service: 'ECS',
        action: 'updateService',
        parameters: {
            service: 'cool-service',
            forceNewDeployment: true,
        },
        catchErrorPattern: 'error',
        apiVersion: '2019-01-01',
    }));
    rule.addTarget(new targets.AwsApi({
        service: 'RDS',
        action: 'createDBSnapshot',
        parameters: {
            DBInstanceIdentifier: 'cool-instance',
        },
    }));
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
        Targets: [
            {
                Arn: {
                    'Fn::GetAtt': [
                        'AWSb4cf1abd4e4f4bc699441af7ccd9ec371511E620',
                        'Arn',
                    ],
                },
                Id: 'Target0',
                Input: JSON.stringify({
                    service: 'ECS',
                    action: 'updateService',
                    parameters: {
                        service: 'cool-service',
                        forceNewDeployment: true,
                    },
                    catchErrorPattern: 'error',
                    apiVersion: '2019-01-01',
                }),
            },
            {
                Arn: {
                    'Fn::GetAtt': [
                        'AWSb4cf1abd4e4f4bc699441af7ccd9ec371511E620',
                        'Arn',
                    ],
                },
                Id: 'Target1',
                Input: JSON.stringify({
                    service: 'RDS',
                    action: 'createDBSnapshot',
                    parameters: {
                        DBInstanceIdentifier: 'cool-instance',
                    },
                }),
            },
        ],
    }));
    // Uses a singleton function
    assert_1.expect(stack).to(assert_1.countResources('AWS::Lambda::Function', 1));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'ecs:UpdateService',
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'rds:CreateDBSnapshot',
                    Effect: 'Allow',
                    Resource: '*',
                },
            ],
            Version: '2012-10-17',
        },
    }));
});
test('with policy statement', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(15 minutes)'),
    });
    // WHEN
    rule.addTarget(new targets.AwsApi({
        service: 'service',
        action: 'action',
        policyStatement: new iam.PolicyStatement({
            actions: ['s3:GetObject'],
            resources: ['resource'],
        }),
    }));
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
        Targets: [
            {
                Arn: {
                    'Fn::GetAtt': [
                        'AWSb4cf1abd4e4f4bc699441af7ccd9ec371511E620',
                        'Arn',
                    ],
                },
                Id: 'Target0',
                Input: JSON.stringify({
                    service: 'service',
                    action: 'action',
                }),
            },
        ],
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Resource: 'resource',
                },
            ],
            Version: '2012-10-17',
        },
    }));
});
//# sourceMappingURL=data:application/json;base64,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