"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsQueue = void 0;
const iam = require("@aws-cdk/aws-iam");
/**
 * Use an SQS Queue as a target for Amazon EventBridge rules.
 *
 * @example
 *
 *    // publish to an SQS queue every time code is committed
 *    // to a CodeCommit repository
 *    repository.onCommit(new targets.SqsQueue(queue));
 */
class SqsQueue {
    /**
     *
     */
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        if (props.messageGroupId !== undefined && !queue.fifo) {
            throw new Error('messageGroupId cannot be specified for non-FIFO queues');
        }
    }
    /**
     * Returns a RuleTarget that can be used to trigger this SQS queue as a result from an EventBridge event.
     *
     * @see https://docs.aws.amazon.com/eventbridge/latest/userguide/resource-based-policies-eventbridge.html#sqs-permissions
     */
    bind(rule, _id) {
        // deduplicated automatically
        this.queue.grantSendMessages(new iam.ServicePrincipal('events.amazonaws.com', {
            conditions: {
                ArnEquals: { 'aws:SourceArn': rule.ruleArn },
            },
        }));
        return {
            id: '',
            arn: this.queue.queueArn,
            input: this.props.message,
            targetResource: this.queue,
            sqsParameters: this.props.messageGroupId ? { messageGroupId: this.props.messageGroupId } : undefined,
        };
    }
}
exports.SqsQueue = SqsQueue;
//# sourceMappingURL=data:application/json;base64,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