"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsApi = void 0;
const path = require("path");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const sdk_api_metadata_generated_1 = require("./sdk-api-metadata.generated");
const util_1 = require("./util");
const awsSdkMetadata = sdk_api_metadata_generated_1.metadata;
/**
 * Use an AWS Lambda function that makes API calls as an event rule target.
 */
class AwsApi {
    /**
     *
     */
    constructor(props) {
        this.props = props;
    }
    /**
     * Returns a RuleTarget that can be used to trigger this AwsApi as a result from an EventBridge event.
     */
    bind(rule, id) {
        const handler = new lambda.SingletonFunction(rule, `${rule.node.id}${id}Handler`, {
            code: lambda.Code.fromAsset(path.join(__dirname, 'aws-api-handler')),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            uuid: 'b4cf1abd-4e4f-4bc6-9944-1af7ccd9ec37',
            lambdaPurpose: 'AWS',
        });
        if (this.props.policyStatement) {
            handler.addToRolePolicy(this.props.policyStatement);
        }
        else {
            handler.addToRolePolicy(new iam.PolicyStatement({
                actions: [awsSdkToIamAction(this.props.service, this.props.action)],
                resources: ['*'],
            }));
        }
        // Allow handler to be called from rule
        util_1.addLambdaPermission(rule, handler);
        const input = {
            service: this.props.service,
            action: this.props.action,
            parameters: this.props.parameters,
            catchErrorPattern: this.props.catchErrorPattern,
            apiVersion: this.props.apiVersion,
        };
        return {
            id: '',
            arn: handler.functionArn,
            input: events.RuleTargetInput.fromObject(input),
            targetResource: handler,
        };
    }
}
exports.AwsApi = AwsApi;
/**
 * Transform SDK service/action to IAM action using metadata from aws-sdk module.
 */
function awsSdkToIamAction(service, action) {
    const srv = service.toLowerCase();
    const iamService = awsSdkMetadata[srv].prefix || srv;
    const iamAction = action.charAt(0).toUpperCase() + action.slice(1);
    return `${iamService}:${iamAction}`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzLWFwaS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImF3cy1hcGkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkJBQTZCO0FBQzdCLDhDQUE4QztBQUM5Qyx3Q0FBd0M7QUFDeEMsOENBQThDO0FBQzlDLDZFQUF3RDtBQUN4RCxpQ0FBNkM7QUFPN0MsTUFBTSxjQUFjLEdBQW1CLHFDQUFRLENBQUM7Ozs7QUErRGhELE1BQWEsTUFBTTs7OztJQUNqQixZQUE2QixLQUFrQjtRQUFsQixVQUFLLEdBQUwsS0FBSyxDQUFhO0lBQUcsQ0FBQzs7OztJQU01QyxJQUFJLENBQUMsSUFBa0IsRUFBRSxFQUFXO1FBQ3pDLE1BQU0sT0FBTyxHQUFHLElBQUksTUFBTSxDQUFDLGlCQUFpQixDQUFDLElBQW1CLEVBQUUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsR0FBRyxFQUFFLFNBQVMsRUFBRTtZQUMvRixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztZQUNwRSxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLElBQUksRUFBRSxzQ0FBc0M7WUFDNUMsYUFBYSxFQUFFLEtBQUs7U0FDckIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsRUFBRTtZQUM5QixPQUFPLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUM7U0FDckQ7YUFBTTtZQUNMLE9BQU8sQ0FBQyxlQUFlLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUM5QyxPQUFPLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDO2dCQUNuRSxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7YUFDakIsQ0FBQyxDQUFDLENBQUM7U0FDTDtRQUVELHVDQUF1QztRQUN2QywwQkFBbUIsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFbkMsTUFBTSxLQUFLLEdBQWdCO1lBQ3pCLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU87WUFDM0IsTUFBTSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTTtZQUN6QixVQUFVLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVO1lBQ2pDLGlCQUFpQixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsaUJBQWlCO1lBQy9DLFVBQVUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVU7U0FDbEMsQ0FBQztRQUVGLE9BQU87WUFDTCxFQUFFLEVBQUUsRUFBRTtZQUNOLEdBQUcsRUFBRSxPQUFPLENBQUMsV0FBVztZQUN4QixLQUFLLEVBQUUsTUFBTSxDQUFDLGVBQWUsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1lBQy9DLGNBQWMsRUFBRSxPQUFPO1NBQ3hCLENBQUM7SUFDSixDQUFDO0NBQ0Y7QUEzQ0Qsd0JBMkNDO0FBRUQ7O0dBRUc7QUFDSCxTQUFTLGlCQUFpQixDQUFDLE9BQWUsRUFBRSxNQUFjO0lBQ3hELE1BQU0sR0FBRyxHQUFHLE9BQU8sQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUNsQyxNQUFNLFVBQVUsR0FBRyxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxJQUFJLEdBQUcsQ0FBQztJQUNyRCxNQUFNLFNBQVMsR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDbkUsT0FBTyxHQUFHLFVBQVUsSUFBSSxTQUFTLEVBQUUsQ0FBQztBQUN0QyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGV2ZW50cyBmcm9tICdAYXdzLWNkay9hd3MtZXZlbnRzJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCB7IG1ldGFkYXRhIH0gZnJvbSAnLi9zZGstYXBpLW1ldGFkYXRhLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBhZGRMYW1iZGFQZXJtaXNzaW9uIH0gZnJvbSAnLi91dGlsJztcblxuLyoqXG4gKiBBV1MgU0RLIHNlcnZpY2UgbWV0YWRhdGEuXG4gKi9cbmV4cG9ydCB0eXBlIEF3c1Nka01ldGFkYXRhID0ge1trZXk6IHN0cmluZ106IGFueX07XG5cbmNvbnN0IGF3c1Nka01ldGFkYXRhOiBBd3NTZGtNZXRhZGF0YSA9IG1ldGFkYXRhO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBBd3NBcGlJbnB1dCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2VydmljZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFjdGlvbjogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwYXJhbWV0ZXJzPzogYW55O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNhdGNoRXJyb3JQYXR0ZXJuPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYXBpVmVyc2lvbj86IHN0cmluZztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBBd3NBcGlQcm9wcyBleHRlbmRzIEF3c0FwaUlucHV0IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcG9saWN5U3RhdGVtZW50PzogaWFtLlBvbGljeVN0YXRlbWVudDtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBBd3NBcGkgaW1wbGVtZW50cyBldmVudHMuSVJ1bGVUYXJnZXQge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBBd3NBcGlQcm9wcykge31cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYmluZChydWxlOiBldmVudHMuSVJ1bGUsIGlkPzogc3RyaW5nKTogZXZlbnRzLlJ1bGVUYXJnZXRDb25maWcge1xuICAgIGNvbnN0IGhhbmRsZXIgPSBuZXcgbGFtYmRhLlNpbmdsZXRvbkZ1bmN0aW9uKHJ1bGUgYXMgZXZlbnRzLlJ1bGUsIGAke3J1bGUubm9kZS5pZH0ke2lkfUhhbmRsZXJgLCB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ2F3cy1hcGktaGFuZGxlcicpKSxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xMl9YLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgdXVpZDogJ2I0Y2YxYWJkLTRlNGYtNGJjNi05OTQ0LTFhZjdjY2Q5ZWMzNycsXG4gICAgICBsYW1iZGFQdXJwb3NlOiAnQVdTJyxcbiAgICB9KTtcblxuICAgIGlmICh0aGlzLnByb3BzLnBvbGljeVN0YXRlbWVudCkge1xuICAgICAgaGFuZGxlci5hZGRUb1JvbGVQb2xpY3kodGhpcy5wcm9wcy5wb2xpY3lTdGF0ZW1lbnQpO1xuICAgIH0gZWxzZSB7XG4gICAgICBoYW5kbGVyLmFkZFRvUm9sZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFthd3NTZGtUb0lhbUFjdGlvbih0aGlzLnByb3BzLnNlcnZpY2UsIHRoaXMucHJvcHMuYWN0aW9uKV0sXG4gICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICB9KSk7XG4gICAgfVxuXG4gICAgLy8gQWxsb3cgaGFuZGxlciB0byBiZSBjYWxsZWQgZnJvbSBydWxlXG4gICAgYWRkTGFtYmRhUGVybWlzc2lvbihydWxlLCBoYW5kbGVyKTtcblxuICAgIGNvbnN0IGlucHV0OiBBd3NBcGlJbnB1dCA9IHtcbiAgICAgIHNlcnZpY2U6IHRoaXMucHJvcHMuc2VydmljZSxcbiAgICAgIGFjdGlvbjogdGhpcy5wcm9wcy5hY3Rpb24sXG4gICAgICBwYXJhbWV0ZXJzOiB0aGlzLnByb3BzLnBhcmFtZXRlcnMsXG4gICAgICBjYXRjaEVycm9yUGF0dGVybjogdGhpcy5wcm9wcy5jYXRjaEVycm9yUGF0dGVybixcbiAgICAgIGFwaVZlcnNpb246IHRoaXMucHJvcHMuYXBpVmVyc2lvbixcbiAgICB9O1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGlkOiAnJyxcbiAgICAgIGFybjogaGFuZGxlci5mdW5jdGlvbkFybixcbiAgICAgIGlucHV0OiBldmVudHMuUnVsZVRhcmdldElucHV0LmZyb21PYmplY3QoaW5wdXQpLFxuICAgICAgdGFyZ2V0UmVzb3VyY2U6IGhhbmRsZXIsXG4gICAgfTtcbiAgfVxufVxuXG4vKipcbiAqIFRyYW5zZm9ybSBTREsgc2VydmljZS9hY3Rpb24gdG8gSUFNIGFjdGlvbiB1c2luZyBtZXRhZGF0YSBmcm9tIGF3cy1zZGsgbW9kdWxlLlxuICovXG5mdW5jdGlvbiBhd3NTZGtUb0lhbUFjdGlvbihzZXJ2aWNlOiBzdHJpbmcsIGFjdGlvbjogc3RyaW5nKTogc3RyaW5nIHtcbiAgY29uc3Qgc3J2ID0gc2VydmljZS50b0xvd2VyQ2FzZSgpO1xuICBjb25zdCBpYW1TZXJ2aWNlID0gYXdzU2RrTWV0YWRhdGFbc3J2XS5wcmVmaXggfHwgc3J2O1xuICBjb25zdCBpYW1BY3Rpb24gPSBhY3Rpb24uY2hhckF0KDApLnRvVXBwZXJDYXNlKCkgKyBhY3Rpb24uc2xpY2UoMSk7XG4gIHJldHVybiBgJHtpYW1TZXJ2aWNlfToke2lhbUFjdGlvbn1gO1xufVxuIl19