/*! \file ProSHADE_precomputedValues.cpp
    \brief This header file contains the pre-computed value that would take too long to obtain in run-time.
 
    This header file currently contains an array listing the largest Gauss-Legendre integration sampling distance between absciccas for a given integration order. There
    may be more variables in the future, the main point of this file is that computing the required value in run-time would be time-consuming and provide no particular advantage.
 
    Copyright by Michal Tykac and individual contributors. All rights reserved.

    Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
    1) Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    2) Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    3) Neither the name of Michal Tykac nor the names of this code's contributors may be used to endorse or promote products derived from this software without specific prior written permission.

    This software is provided by the copyright holder and contributors "as is" and any express or implied warranties, including, but not limitted to, the implied warranties of merchantibility and fitness for a particular purpose are disclaimed. In no event shall the copyright owner or the contributors be liable for any direct, indirect, incidental, special, exemplary, or consequential damages (including, but not limitted to, procurement of substitute goods or services, loss of use, data or profits, or business interuption) however caused and on any theory of liability, whether in contract, strict liability or tort (including negligence or otherwise) arising in any way out of the use of this software, even if advised of the possibility of such damage.
 
    \author    Michal Tykac
    \author    Garib N. Murshudov
    \version   0.7.5.1
    \date      JAN 2021
 */

//==================================================== ProSHADE
#include "ProSHADE_typedefs.hpp"

//==================================================== Overinclusion protection
#ifndef __PROSHADE_PRECOMPUTED__
#define __PROSHADE_PRECOMPUTED__

//==================================================== ProSHADE_internal_precomputedVals Namespace
/*! \namespace ProSHADE_internal_precomputedVals
    \brief This namespace contains the pre-computed values for the ProSHADE run.
 
 
    The ProSHADE_internal_precomputedVals namespace contains the pre-computed numbers so that the slow functions required
    to obtain these would not have to be run every time ProSHADE is being used. This approach has its limits, but in some
    cases these do outweighht the benefits of computing everything afresh.
 */
namespace ProSHADE_internal_precomputedVals
{
    static const proshade_double glIntMaxDists[]      = { +999.9,
                                                          +999.9,
                                                          +0.57735026918962573106,
                                                          +0.38729833462074170214,
                                                          +0.33998104358485631282,
                                                          +0.26923465505284160937,
                                                          +0.23861918608319690471,
                                                          +0.20292257568869859208,
                                                          +0.18343464249564980784,
                                                          +0.16212671170190445791,
                                                          +0.14887433898163121571,
                                                          +0.13477157797617247970,
                                                          +0.12523340851146891328,
                                                          +0.11522915797756738687,
                                                          +0.10805494870734366764,
                                                          +0.10059704699871724332,
                                                          +0.09501250983763744051,
                                                          +0.08924209074792392726,
                                                          +0.08477501304173530594,
                                                          +0.08017932282011268341,
                                                          +0.07652652113349733831,
                                                          +0.07278092708044756054,
                                                          +0.06973927331972222532,
                                                          +0.06662841214923304056,
                                                          +0.06405689286260562998,
                                                          +0.06143234630535520518,
                                                          +0.05923009342931320753,
                                                          +0.05698629280476497777,
                                                          +0.05507928988403426596,
                                                          +0.05313911506633961501,
                                                          +0.05147184255531769836,
                                                          +0.04977765607617076066,
                                                          +0.04830766568773831732,
                                                          +0.04681553292736669036,
                                                          +0.04550982195310254025,
                                                          +0.04418567163782963214,
                                                          +0.04301819847370860761,
                                                          +0.04183520447738495185,
                                                          +0.04078514790457823946,
                                                          +0.03972190230437773473,
                                                          +0.03877241750605082277,
                                                          +0.03781162949458149319,
                                                          +0.03694894316535177886,
                                                          +0.03607649543729311847,
                                                          +0.03528923696413536343,
                                                          +0.03449349008157208407,
                                                          +0.03377219001605204218,
                                                          +0.03304346195817783882,
                                                          +0.03238017096286936042,
                                                          +0.03171034249134339206,
                                                          +0.03109833832718887983,
                                                          +0.03048055007528936017,
                                                          +0.02991410979733876915,
                                                          +0.02934252715012973869,
                                                          +0.02881674819934177539,
                                                          +0.02828637690916838490,
                                                          +0.02779703528727543727,
                                                          +0.02730357550082341406,
                                                          +0.02684701236594235624,
                                                          +0.02638674204415499339,
                                                          +0.02595977230124780002,
                                                          +0.02552945335398717347,
                                                          +0.02512929142182061468,
                                                          +0.02472609355807981260,
                                                          +0.02435029266342443252,
                                                          +0.02397173117658592603,
                                                          +0.02361813338592456957,
                                                          +0.02326201798800861184,
                                                          +0.02292871282542234893,
                                                          +0.02259310530755085852,
                                                          +0.02227839528614030881,
                                                          +0.02196157427906766241,
                                                          +0.02166394603542404554,
                                                          +0.02136437751920699721,
                                                          +0.02108247751674830264,
                                                          +0.02079879001453972809,
                                                          +0.02053140399399863703,
                                                          +0.02026236760455796260,
                                                          +0.02000840312451792474,
                                                          +0.01975291171798702894,
                                                          +0.01951138325679399946,
                                                          +0.01926843929075599052,
                                                          +0.01903845548200681220,
                                                          +0.01880715699879219327,
                                                          +0.01858790965233213244,
                                                          +0.01836743910062483762,
                                                          +0.01815819372438581478,
                                                          +0.01794780831340044708,
                                                          +0.01774789590211209647,
                                                          +0.01754691924872850070,
                                                          +0.01735572914629965324,
                                                          +0.01716354401741266786,
                                                          +0.01698051769272823294,
                                                          +0.01679655967888827803,
                                                          +0.01662118528152603525,
                                                          +0.01644493726539446310,
                                                          +0.01627674484960297055,
                                                          +0.01610773215513085221,
                                                          +0.01594628947830939794,
                                                          +0.01578407560488613234,
                                                          +0.01562898442154308024,
                                                          +0.01547316728244910178,
                                                          +0.01532406006679587485,
                                                          +0.01517426866375888583,
                                                          +0.01503080570420580776,
                                                          +0.01488669718019959912,
                                                          +0.01474856399744101272,
                                                          +0.01460982101847505162,
                                                          +0.01447672606589194173,
                                                          +0.01434305448973451080,
                                                          +0.01421472710075017201,
                                                          +0.01408585389656138477,
                                                          +0.01396204244855868241,
                                                          +0.01383771383648671992,
                                                          +0.01371818410509546664,
                                                          +0.01359816388917564969,
                                                          +0.01348269757029667976,
                                                          +0.01336676564162248698,
                                                          +0.01325515902157874647,
                                                          +0.01314311001179721376,
                                                          +0.01303517276857863588,
                                                          +0.01292681483638894596,
                                                          +0.01282236895716001683,
                                                          +0.01271752269273791758,
                                                          +0.01261640149466360453,
                                                          +0.01251489892885638069,
                                                          +0.01241694617192320819,
                                                          +0.01231862987871047178,
                                                          +0.01222369896061576387,
                                                          +0.01212842124275292054,
                                                          +0.01203637446714071572,
                                                          +0.01194399661612855455,
                                                          +0.01185470452649396711,
                                                          +0.01176509614907936090,
                                                          +0.01167843692156819539,
                                                          +0.01159147532590126990,
                                                          +0.01150733421501838258,
                                                          +0.01142290385039149841,
                                                          +0.01134117268231683677,
                                                          +0.01125916462710743922,
                                                          +0.01117974133591704070,
                                                          +0.01110005282896457147,
                                                          +0.01102284103157731912,
                                                          +0.01094537504275623456,
                                                          +0.01087028364888588915,
                                                          +0.01079494848510285064,
                                                          +0.01072189133889790702,
                                                          +0.01064860028215036669,
                                                          +0.01057749583255868965,
                                                          +0.01050616680705212212,
                                                          +0.01043693780425977193,
                                                          +0.01036749306989779434,
                                                          +0.01030006628546735990,
                                                          +0.01023243215530906559,
                                                          +0.01016673812388675657,
                                                          +0.01010084470341309631,
                                                          +0.01003681748408997813,
                                                          +0.00997259843034026608,
                                                          +0.00991017538594488392,
                                                          +0.00984756768477908218,
                                                          +0.00978668927754903183,
                                                          +0.00972563303746422705,
                                                          +0.00966624263969608107,
                                                          +0.00960668090077945389,
                                                          +0.00954872461919153981,
                                                          +0.00949060317592910485,
                                                          +0.00943402968859249728,
                                                          +0.00937729692537539808,
                                                          +0.00932205733017621788,
                                                          +0.00926666406845549559,
                                                          +0.00921271174214843662,
                                                          +0.00915861109828727989,
                                                          +0.00910590156528665214,
                                                          +0.00905304881824673530,
                                                          +0.00900153962837920221,
                                                          +0.00894989209645643928,
                                                          +0.00889954271096943597,
                                                          +0.00884905963686529080,
                                                          +0.00879983132204789804,
                                                          +0.00875047376562596013,
                                                          +0.00870232949345575886,
                                                          +0.00865406023159066666,
                                                          +0.00860696458687108291,
                                                          +0.00855974801984858355,
                                                          +0.00851366711334740518,
                                                          +0.00846746917732101034,
                                                          +0.00842237056446247870,
                                                          +0.00837715864951438238,
                                                          +0.00833301125421499062,
                                                          +0.00828875412760714941,
                                                          +0.00824552817087947573,
                                                          +0.00820219590511530282,
                                                          +0.00815986283809528624,
                                                          +0.00811742674344385906,
                                                          +0.00807595918452503475,
                                                          +0.00803439174568813287,
                                                          +0.00799376342147201623,
                                                          +0.00795303823810031806,
                                                          +0.00791322392789535478,
                                                          +0.00787331565868353879,
                                                          +0.00783429114230636998,
                                                          +0.00779517545241847302,
                                                          +0.00775691746107047900,
                                                          +0.00771857097266632521,
                                                          +0.00768105714267617017,
                                                          +0.00764345738832754220,
                                                          +0.00760666621756652013,
                                                          +0.00756979159636806710,
                                                          +0.00753370240315981519,
                                                          +0.00749753213935459416,
                                                          +0.00746212502371416779,
                                                          +0.00742663912766731125,
                                                          +0.00739189493471704637,
                                                          +0.00735707416608354006,
                                                          +0.00732297445150440797,
                                                          +0.00728880028444841427,
                                                          +0.00725532728183594499,
                                                          +0.00722178187216957965,
                                                          +0.00718891846217299159,
                                                          +0.00715598461629211608,
                                                          +0.00712371429742403219,
                                                          +0.00709137544292742523,
                                                          +0.00705968230393962635,
                                                          +0.00702792246182610495,
                                                          +0.00699679115555415509,
                                                          +0.00696559491389959606,
                                                          +0.00693501063248605672,
                                                          +0.00690436312150929238,
                                                          +0.00687431157292144566,
                                                          +0.00684419844135424494,
                                                          +0.00681466582711814060,
                                                          +0.00678507321980039661,
                                                          +0.00675604621387834205,
                                                          +0.00672696075050501982,
                                                          +0.00669842647924858044,
                                                          +0.00666983523419989715,
                                                          +0.00664178125731506273,
                                                          +0.00661367174050604949,
                                                          +0.00658608603297146118,
                                                          +0.00655844617166131960,
                                                          +0.00653131710654431159,
                                                          +0.00650413522804988656,
                                                          +0.00647745156016879889,
                                                          +0.00645071637543018304,
                                                          +0.00642446722581469301,
                                                          +0.00639816781376436380,
                                                          +0.00637234265486874996,
                                                          +0.00634646844755876269,
                                                          +0.00632105708918586067,
                                                          +0.00629559785763056639,
                                                          +0.00627059043352690311,
                                                          +0.00624553627422133346,
                                                          +0.00622092322930638524,
                                                          +0.00619626455138300083,
                                                          +0.00617203662957786000,
                                                          +0.00614776414256665987,
                                                          +0.00612391237518953007,
                                                          +0.00610001707734871002,
                                                          +0.00607653277204669908,
                                                          +0.00605300593923308993,
                                                          +0.00602988066942157074,
                                                          +0.00600671384447192154,
                                                          +0.00598393943925453527,
                                                          +0.00596112442185051192,
                                                          +0.00593869295639449200,
                                                          +0.00591622179338588926,
                                                          +0.00589412557972885005,
                                                          +0.00587199055589100304,
                                                          +0.00585022213415685097,
                                                          +0.00582841576335969141,
                                                          +0.00580696789336253988,
                                                          +0.00578548291013009520,
                                                          +0.00576434856334633039,
                                                          +0.00574317791478662874,
                                                          +0.00572235026667645540,
                                                          +0.00570148710476304707,
                                                          +0.00568095952742387385,
                                                          +0.00566039720161120967,
                                                          +0.00564016325674625806,
                                                          +0.00561989530690218658,
                                                          +0.00559994873908869172,
                                                          +0.00557996888872833077,
                                                          +0.00556030361897049893,
                                                          +0.00554060576877656724,
                                                          +0.00552121588832936005,
                                                          +0.00550179410994493332,
                                                          +0.00548267387439551300,
                                                          +0.00546352240447592168,
                                                          +0.00544466622807027265,
                                                          +0.00542577946258163222,
                                                          +0.00540718191278461054,
                                                          +0.00538855440153710591,
                                                          +0.00537021019381478219,
                                                          +0.00535183663521953019,
                                                          +0.00533374062803330506,
                                                          +0.00531561586407218196,
                                                          +0.00529776305407470959,
                                                          +0.00527988206547312829,
                                                          +0.00526226758289661742,
                                                          +0.00524462548448977398,
                                                          +0.00522724458871774884,
                                                          +0.00520983662500132567,
                                                          +0.00519268470031537203,
                                                          +0.00517550624117222875,
                                                          +0.00515857879266573040,
                                                          +0.00514162532926046474,
                                                          +0.00512491797891171004,
                                                          +0.00510818511974549163,
                                                          +0.00509169360264296982,
                                                          +0.00507517706976136004,
                                                          +0.00505889723047438841,
                                                          +0.00504259285582129776,
                                                          +0.00502652064490949693,
                                                          +0.00501042436682072254,
                                                          +0.00499455583747620679,
                                                          +0.00497866369730638400,
                                                          +0.00496299500212277461,
                                                          +0.00494730314099983345,
                                                          +0.00493183052886258209,
                                                          +0.00491633518456414519,
                                                          +0.00490105499765685314,
                                                          +0.00488575250160323479,
                                                          +0.00487066117252498397,
                                                          +0.00485554794688377363,
                                                          +0.00484064199587266614,
                                                          +0.00482571455077006502,
                                                          +0.00481099058302845734,
                                                          +0.00479624551386284136,
                                                          +0.00478170021697993212,
                                                          +0.00476713420183325250,
                                                          +0.00475276434330093762,
                                                          +0.00473837414044385973,
                                                          +0.00472417656526193282,
                                                          +0.00470995901074877221,
                                                          +0.00469593063911571702,
                                                          +0.00468188264446540731,
                                                          +0.00466802046955129384,
                                                          +0.00465413901951083290,
                                                          +0.00464044010530887015,
                                                          +0.00462672225569584102,
                                                          +0.00461318373494939808,
                                                          +0.00459962661057032249,
                                                          +0.00458624568277233441,
                                                          +0.00457284647541374850,
                                                          +0.00455962040487558565,
                                                          +0.00454637637136493140,
                                                          +0.00453330248535190571,
                                                          +0.00452021094568537483,
                                                          +0.00450728663261625983,
                                                          +0.00449434496815095258,
                                                          +0.00448156767585892058,
                                                          +0.00446877332756672428,
                                                          +0.00445614056161931088,
                                                          +0.00444349102840007129,
                                                          +0.00443100035047581334,
                                                          +0.00441849318752746568,
                                                          +0.00440614221384701046,
                                                          +0.00439377503109041230,
                                                          +0.00438156143089997625,
                                                          +0.00436933189145635639,
                                                          +0.00435725338556150130,
                                                          +0.00434515920428043666,
                                                          +0.00433321356362822985,
                                                          +0.00432125250566424771,
                                                          +0.00430943754997195096,
                                                          +0.00429760742940786620,
                                                          +0.00428592102583638754,
                                                          +0.00427421970435159712,
                                                          +0.00426265976622200362,
                                                          +0.00425108515180405103,
                                                          +0.00423964963735552584,
                                                          +0.00422819968305328427,
                                                          +0.00421688659424096588,
                                                          +0.00420555929695789511,
                                                          +0.00419436667828912771,
                                                          +0.00418316007761509751,
                                                          +0.00417208601502265432,
                                                          +0.00416099819210291426,
                                                          +0.00415004081185384004,
                                                          +0.00413906988829376103,
                                                          +0.00412822735593251864,
                                                          +0.00411737149273679135,
                                                          +0.00410664201206146617,
                                                          +0.00409589940860650287,
                                                          +0.00408528122067686791,
                                                          +0.00407465011371519695,
                                                          +0.00406414149589147660,
                                                          +0.00405362015858696904,
                                                          +0.00404321942359821968,
                                                          +0.00403280616459106116,
                                                          +0.00402251165963207497,
                                                          +0.00401220482213238919,
                                                          +0.00400201492798814805,
                                                          +0.00399181288889727520,
                                                          +0.00398172601909394918,
                                                          +0.00397162718815238357,
                                                          +0.00396164178813395145,
                                                          +0.00395164460709502077,
                                                          +0.00394175915342461717,
                                                          +0.00393186209525296822,
                                                          +0.00392207509483808760,
                                                          +0.00391227666293215287,
                                                          +0.00390258665227289020,
                                                          +0.00389288537971046924,
                                                          +0.00388329092416999852,
                                                          +0.00387368537297617770,
                                                          +0.00386418506607270331,
                                                          +0.00385467382650933220,
                                                          +0.00384526628922878541,
                                                          +0.00383584797910478106,
                                                          +0.00382653185923355170,
                                                          +0.00381720512323532272,
                                                          +0.00380797909471234427,
                                                          +0.00379874260375364725,
                                                          +0.00378960536604118926,
                                                          +0.00378045781663176751,
                                                          +0.00377140809410431155,
                                                          +0.00376234820773667985,
                                                          +0.00375338474908726467,
                                                          +0.00374441127164104473,
                                                          +0.00373553284930451144,
                                                          +0.00372664455046773117,
                                                          +0.00371784996006029695,
                                                          +0.00370904563276708158,
                                                          +0.00370033369254172502,
                                                          +0.00369161215242585529,
                                                          +0.00368298170274297849,
                                                          +0.00367434178760679426,
                                                          +0.00366579169041966182,
                                                          +0.00365723225971780050,
                                                          +0.00364876139807229815,
                                                          +0.00364028133240979759,
                                                          +0.00363188860995801964,
                                                          +0.00362348681060231399,
                                                          +0.00361517115112956235,
                                                          +0.00360684653953593191,
                                                          +0.00359860688650067177,
                                                          +0.00359035840385070687,
                                                          +0.00358219371993708782,
                                                          +0.00357402032668976142,
                                                          +0.00356592959337229121,
                                                          +0.00355783026882723011,
                                                          +0.00354981248594723325,
                                                          +0.00354178622781981586,
                                                          +0.00353384041317328108,
                                                          +0.00352588623718117378,
                                                          +0.00351801142611766907,
                                                          +0.00351012836557846111,
                                                          +0.00350232361061073288,
                                                          +0.00349451071605030991,
                                                          +0.00348677508647427361,
                                                          +0.00347903142524560050,
                                                          +0.00347136400677036791,
                                                          +0.00346368866268236680,
                                                          +0.00345608855707002938,
                                                          +0.00344848063002623027,
                                                          +0.00344094695474107858,
                                                          +0.00343340556038775318,
                                                          +0.00342593744825465951,
                                                          +0.00341846171763814880,
                                                          +0.00341105831650982726,
                                                          +0.00340364739574277285,
                                                          +0.00339630786817565321,
                                                          +0.00338896091811188111,
                                                          +0.00338168444105033323,
                                                          +0.00337440063696811112,
                                                          +0.00336718640143677736,
                                                          +0.00335996493273021425,
                                                          +0.00335281214353419293,
                                                          +0.00334565221341251570,
                                                          +0.00333856008884518812,
                                                          +0.00333146091403968342,
                                                          +0.00332442868559792775,
                                                          +0.00331738949607630635,
                                                          +0.00331041640818290680,
                                                          +0.00330343644687088218,
                                                          +0.00329652175660390774,
                                                          +0.00328960027911376829,
                                                          +0.00328274325594272087,
                                                          +0.00327587953030870124,
                                                          +0.00326907945583724191,
                                                          +0.00326227276225748335,
                                                          +0.00325552892997254054,
                                                          +0.00324877856055746494,
                                                          +0.00324209027558453955,
                                                          +0.00323539553411144112,
                                                          +0.00322876211297593203,
                                                          +0.00322212231464960846,
                                                          +0.00321554308504399568,
                                                          +0.00320895755626324653,
                                                          +0.00320243185681995087,
                                                          +0.00319589993494979194,
                                                          +0.00318942711501954933,
                                                          +0.00318294814816894511,
                                                          +0.00317652756760456291,
                                                          +0.00317010091440956447,
                                                          +0.00316373194335487851,
                                                          +0.00315735697276697336,
                                                          +0.00315103899145088200,
                                                          +0.00314471508253044794,
                                                          +0.00313844748106586590,
                                                          +0.00313217402278055538,
                                                          +0.00312595620096816360,
                                                          +0.00311973259199609760,
                                                          +0.00311356395913274953,
                                                          +0.00310738960767038786,
                                                          +0.00310126958236204078,
                                                          +0.00309514390593659209,
                                                          +0.00308907191591564724,
                                                          +0.00308299434120189739,
                                                          +0.00307696982314882572,
                                                          +0.00307093978579026658,
                                                          +0.00306496218515939613,
                                                          +0.00305897912959353716,
                                                          +0.00305304790044289833,
                                                          +0.00304711127973063874,
                                                          +0.00304122588455575486,
                                                          +0.00303533516021472138,
                                                          +0.00302949506978623289,
                                                          +0.00302364971162795947,
                                                          +0.00301785440483299128,
                                                          +0.00301205389080385364,
                                                          +0.00300630285449100939,
                                                          +0.00300054667051680027,
                                                          +0.00299483939934470141,
                                                          +0.00298912703917876164,
                                                          +0.00298346303546802782,
                                                          +0.00297779400054282639,
                                                          +0.00297217277413140820,
                                                          +0.00296654657341349206,
                                                          +0.00296096764151527721,
                                                          +0.00295538379136349319,
                                                          +0.00294984667843008067,
                                                          +0.00294430470245698756,
                                                          +0.00293880894004256744,
                                                          +0.00293330836897895538,
                                                          +0.00292785349560820184,
                                                          +0.00292239386717063664,
                                                          +0.00291697942820953477,
                                                          +0.00291156028697084780,
                                                          +0.00290618583450038756,
                                                          +0.00290080673176303745,
                                                          +0.00289547182445569441,
                                                          +0.00289013231812792386,
                                                          +0.00288483652112685564,
                                                          +0.00287953617560157389,
                                                          +0.00287427906040246166,
                                                          +0.00286901744643878369,
                                                          +0.00286379859077425707,
                                                          +0.00285857528538162580,
                                                          +0.00285339427310820010,
                                                          +0.00284820885943303653,
                                                          +0.00284306528042050028,
                                                          +0.00283791734763530522,
                                                          +0.00283281079765849726,
                                                          +0.00282769994085335218,
                                                          +0.00282263002148626764,
                                                          +0.00281755584156267308,
                                                          +0.00281252216007483969,
                                                          +0.00280748426364182870,
                                                          +0.00280248643289689485,
                                                          +0.00279748443216936565,
                                                          +0.00279252207052585423,
                                                          +0.00278755558322507039,
                                                          +0.00278262831443923039,
                                                          +0.00277769696369543342,
                                                          +0.00277280441682614979,
                                                          +0.00276790783108322915,
                                                          +0.00276304964039893576,
                                                          +0.00275818745332111030,
                                                          +0.00275336325820865107,
                                                          +0.00274853510858910849,
                                                          +0.00274374455346451749,
                                                          +0.00273895008513596466,
                                                          +0.00273419281935709246,
                                                          +0.00272943168110417614,
                                                          +0.00272470735888513844,
                                                          +0.00271997920435867969,
                                                          +0.00271528748468607929,
                                                          +0.00271059197231908770,
                                                          +0.00270593251886995772,
                                                          +0.00270126931179538125,
                                                          +0.00269664179285682148,
                                                          +0.00269201055882698241,
                                                          +0.00268741464721743845,
                                                          +0.00268281505852512600,
                                                          +0.00267825043151727988,
                                                          +0.00267368216491845100,
                                                          +0.00266914850416367835,
                                                          +0.00266461124080173391,
                                                          +0.00266010823225609370,
                                                          +0.00265560165758769529,
                                                          +0.00265112899143941363,
                                                          +0.00264665279516178905,
                                                          +0.00264221016576021228,
                                                          +0.00263776404173994040,
                                                          +0.00263335114752590201,
                                                          +0.00262893479372911703,
                                                          +0.00262455133716670832,
                                                          +0.00262016445559070960,
                                                          +0.00261581014310040489,
                                                          +0.00261145243970662246,
                                                          +0.00260712698159974070,
                                                          +0.00260279816624804303,
                                                          +0.00259850127666249979,
                                                          +0.00259420106304679605,
                                                          +0.00258993245988413086,
                                                          +0.00258566056546925219,
                                                          +0.00258141997033288972,
                                                          +0.00257717611629270510,
                                                          +0.00257296325442743631,
                                                          +0.00256874716558419536,
                                                          +0.00256456176581682871,
                                                          +0.00256037317058167803,
                                                          +0.00255621496526286119,
                                                          +0.00255205359557752597,
                                                          +0.00254792232052469163,
                                                          +0.00254378791180429444,
                                                          +0.00253968330624570896,
                                                          +0.00253557559732269129,
                                                          +0.00253149740384258715,
                                                          +0.00252741613691171655,
                                                          +0.00252336410139647905,
                                                          +0.00251930902196091135,
                                                          +0.00251528289354629878,
                                                          +0.00251125375036468073,
                                                          +0.00250725328138405069,
                                                          +0.00250324982641862964,
                                                          +0.00249927477235215544,
                                                          +0.00249529676071788119,
                                                          +0.00249134688014273145,
                                                          +0.00248739407005732233,
                                                          +0.00248346912459878385,
                                                          +0.00247954127733374281,
                                                          +0.00247564103161726761,
                                                          +0.00247173791144982176,
                                                          +0.00246786213305397271,
                                                          +0.00246398350721991352,
                                                          +0.00246013196663020075,
                                                          +0.00245627760527761859,
                                                          +0.00245245007584118805,
                                                          +0.00244861975198506360,
                                                          +0.00244481600986623960,
                                                          +0.00244100949934388488,
                                                          +0.00243722932348053615,
                                                          +0.00243344640490786605,
                                                          +0.00242968957696858018,
                                                          +0.00242593003169719280,
                                                          +0.00242219633603923913,
                                                          +0.00241845994811428801,
                                                          +0.00241474917174235937,
                                                          +0.00241103572786120594,
                                                          +0.00240734766038691291,
                                                          +0.00240365694985853308,
                                                          +0.00239999138346064624,
                                                          +0.00239632319816578338,
                                                          +0.00239267992755119296,
                                                          +0.00238903406190323813,
                                                          +0.00238541288426862876,
                                                          +0.00238178913517521221,
                                                          +0.00237818985016943459,
                                                          +0.00237458801699471419,
                                                          +0.00237101042668183114,
                                                          +0.00236743031120946294,
                                                          +0.00236387422003246509,
                                                          +0.00236031562642923963,
                                                          +0.00235678084117441717,
                                                          +0.00235324357595455671,
                                                          +0.00234972990571650055,
                                                          +0.00234621377770658597,
                                                          +0.00234272103385382196,
                                                          +0.00233922585415835592,
                                                          +0.00233575385029958756,
                                                          +0.00233227943226716394,
                                                          +0.00232882798421811749,
                                                          +0.00232537414340819749,
                                                          +0.00232194306915905124,
                                                          +0.00231850962330932371,
                                                          +0.00231509874299271230,
                                                          +0.00231168551198703606,
                                                          +0.00230829464784661990,
                                                          +0.00230490145368352547,
                                                          +0.00230153043004311135,
                                                          +0.00229815709680485526,
                                                          +0.00229480574003805950,
                                                          +0.00229145209386022583,
                                                          +0.00228812023236066005,
                                                          +0.00228478610140228806,
                                                          +0.00228147356555427226,
                                                          +0.00227815877996850533,
                                                          +0.00227486540211828360,
                                                          +0.00227156979402352768,
                                                          +0.00226829540845098271,
                                                          +0.00226501881190256443,
                                                          +0.00226176325479342244,
                                                          +0.00225850550575573510,
                                                          +0.00225526861517424569,
                                                          +0.00225202955149337485,
                                                          +0.00224881116735546139,
                                                          +0.00224559062873228151,
                                                          +0.00224239059277915034,
                                                          +0.00223918842074287980,
                                                          +0.00223600657651507788,
                                                          +0.00223282261439728758,
                                                          +0.00222965880720920511,
                                                          +0.00222649290011826297,
                                                          +0.00222334697703306617,
                                                          +0.00222019897182902572,
                                                          +0.00221707078163401036,
                                                          +0.00221394052690391391,
                                                          +0.00221082992008628113,
                                                          +0.00220771726611988751,
                                                          +0.00220462409484291876,
                                                          +0.00220152889360883180,
                                                          +0.00219845301168847349,
                                                          +0.00219537511681067686,
                                                          +0.00219231637969250718,
                                                          +0.00218925564642728863,
                                                          +0.00218621391116387330,
                                                          +0.00218317019637713668,
                                                          +0.00218014532160575990,
                                                          +0.00217711848375071139,
                                                          +0.00217411032967147538,
                                                          +0.00217110022876667967,
                                                          +0.00216810865712097067,
                                                          +0.00216511515472877321,
                                                          +0.00216214002877807704,
                                                          +0.00215916298798338223,
                                                          +0.00215620417248844681,
                                                          +0.00215324345787784198,
                                                          +0.00215030081907819325,
                                                          +0.00214735629671941554,
                                                          +0.00214442970231319349,
                                                          +0.00214150123973493858,
                                                          +0.00213859055885907328,
                                                          +0.00213567802503112194,
                                                          +0.00213278312824182739,
                                                          +0.00212988639355551250,
                                                          +0.00212700715280909414,
                                                          +0.00212412608905806566,
                                                          +0.00212126237769204697,
                                                          +0.00211839685805336299,
                                                          +0.00211554855076790792,
                                                          +0.00211269844978342241,
                                                          +0.00210986542262306224,
                                                          +0.00210703061618111847,
                                                          +0.00210421274651676794,
                                                          +0.00210139311183417908,
                                                          +0.00209859027834544609,
                                                          +0.00209578569394977023,
                                                          +0.00209299777660754225,
                                                          +0.00209020812231963974,
                                                          +0.00208743500236894652,
                                                          +0.00208466015928580927,
                                                          +0.00208190171922896950,
                                                          +0.00207914156970683255,
                                                          +0.00207639769328685003,
                                                          +0.00207365212092456034,
                                                          +0.00207092269310879670,
                                                          +0.00206819158273145601,
                                                          +0.00206547648969555202,
                                                          +0.00206275972733840733,
                                                          +0.00206005885645046093,
                                                          +0.00205735632934305130,
                                                          +0.00205466956914804712,
                                                          +0.00205198116569860504,
                                                          +0.00204930840590307516,
                                                          +0.00204663401568316340,
                                                          +0.00204397514714009998,
                                                          +0.00204131466086949175,
                                                          +0.00203866957556348913,
                                                          +0.00203602288509528457,
                                                          +0.00203339147612791081,
                                                          +0.00203075847443388120,
                                                          +0.00202814063600927985,
                                                          +0.00202552121716544379,
                                                          +0.00202291684457614994,
                                                          +0.00202031090374856425,
                                                          +0.00201771989336155082,
                                                          +0.00201512732679232911,
                                                          +0.00201254957603525615,
                                                          +0.00200997028102879640,
                                                          +0.00200740568837647635,
                                                          +0.00200483956328589884,
                                                          +0.00200228802824696950,
                                                          +0.00199973497246076770,
                                                          +0.00199719639556456430,
                                                          +0.00199465630949345347,
                                                          +0.00199213059227708366,
                                                          +0.00198960337734105331,
                                                          +0.00198709042233666516,
                                                          +0.00198457598095222064,
                                                          +0.00198207569167446881,
                                                          +0.00197957392724207835,
                                                          +0.00197708620817577032,
                                                          +0.00197459702506748913,
                                                          +0.00197212178165542601,
                                                          +0.00196964508520271501,
                                                          +0.00196718222383370856,
                                                          +0.00196471792031543014,
                                                          +0.00196226734831250358,
                                                          +0.00195981534494310159,
                                                          +0.00195737697055186309,
                                                          +0.00195493717546971766,
                                                          +0.00195251090784690938,
                                                          +0.00195008323010286279,
                                                          +0.00194766897930508044,
                                                          +0.00194525332885113760,
                                                          +0.00194285100582371520,
                                                          +0.00194044729350190907,
                                                          +0.00193805681006796807,
                                                          +0.00193566494759939384,
                                                          +0.00193328621644905355,
                                                          +0.00193090611642305590,
                                                          +0.00192853905110280835,
                                                          +0.00192617062696633654,
                                                          +0.00192381514186856740,
                                                          +0.00192145830791568407,
                                                          +0.00191911431826835544,
                                                          +0.00191676898962989616,
                                                          +0.00191443641148637497,
                                                          +0.00191210250411976150,
                                                          +0.00190978125434880085,
                                                          +0.00190745868502800319,
                                                          +0.00190514868130386267,
                                                          +0.00190283736760950609,
                                                          +0.00190053852840222173,
                                                          +0.00189823838871183790,
                                                          +0.00189595063327762802,
                                                          +0.00189366158675604484,
                                                          +0.00189138483512785931,
                                                          +0.00188910680171773320,
                                                          +0.00188684097469593451,
                                                          +0.00188457387510841657,
                                                          +0.00188231889425159553,
                                                          +0.00188006264995713222,
                                                          +0.00187781843757305663,
                                                          +0.00187557297079232375,
                                                          +0.00187333944992901539,
                                                          +0.00187110468362397834,
                                                          +0.00186888177806091797,
                                                          +0.00186665763592602091,
                                                          +0.00186444527016547848,
                                                          +0.00186223167661896420,
                                                          +0.00186002977587744868,
                                                          +0.00185782665605279057,
                                                          +0.00185563514625262764,
                                                          +0.00185344242599009396,
                                                          +0.00185126123375111643,
                                                          +0.00184907883958944689,
                                                          +0.00184690789222080403,
                                                          +0.00184473575138900756,
                                                          +0.00184257497688108973,
                                                          +0.00184041301729035529,
                                                          +0.00183826234430683436,
                                                          +0.00183611049454255422,
                                                          +0.00183396985241253137,
                                                          +0.00183182804172644020,
                                                          +0.00182969736043670558,
                                                          +0.00182756551873912649,
                                                          +0.00182544472892652468,
                                                          +0.00182332278677872295,
                                                          +0.00182121181972262776,
                                                          +0.00181909970832927004,
                                                          +0.00181699849594416226,
                                                          +0.00181489614714588454,
                                                          +0.00181280462197402946,
                                                          +0.00181071196824010463,
                                                          +0.00180863006344433451,
                                                          +0.00180654703786543944,
                                                          +0.00180447468722203528,
                                                          +0.00180240122350312174,
                                                          +0.00180033836139478830,
                                                          +0.00179827439384804976,
                                                          +0.00179622095525699250,
                                                          +0.00179416641879492651,
                                                          +0.00179212233929601961,
                                                          +0.00179007716942458498,
                                                          +0.00178804238517863497,
                                                          +0.00178600651799050675,
                                                          +0.00178398096573760328,
                                                          +0.00178195433790551182,
                                                          +0.00177993795495847611,
                                                          +0.00177792050372864462,
                                                          +0.00177591322796655888,
                                                          +0.00177390489115222187,
                                                          +0.00177190666101405313,
                                                          +0.00176990737698907172,
                                                          +0.00176791813146737523,
                                                          +0.00176592783915993190,
                                                          +0.00176394751779464318,
                                                          +0.00176196615668101886,
                                                          +0.00175999469955333146,
                                                          +0.00175802220965177785,
                                                          +0.00175605955737809520,
                                                          +0.00175409587924277690,
                                                          +0.00175214197296875355,
                                                          +0.00175018704768377864,
                                                          +0.00174824182907843393,
                                                          +0.00174629559825195853,
                                                          +0.00174435900950187673,
                                                          +0.00174242141526029606,
                                                          +0.00174049339906389333,
                                                          +0.00173856438404610021,
                                                          +0.00173664488360797766,
                                                          +0.00173472439095970579,
                                                          +0.00173281334998506842,
                                                          +0.00173090132335330439,
                                                          +0.00172899868604246072,
                                                          +0.00172709506956993347,
                                                          +0.00172520078061286306,
                                                          +0.00172330551893260485,
                                                          +0.00172141952350359797,
                                                          +0.00171953256173357346,
                                                          +0.00171765480548594475,
                                                          +0.00171577608922375731,
                                                          +0.00171390651828462210,
                                                          +0.00171203599360228247,
                                                          +0.00171017455456740747,
                                                          +0.00170831216800617649,
                                                          +0.00170645880793489162,
                                                          +0.00170460450650018551,
                                                          +0.00170275917291036775,
                                                          +0.00170091290406673180,
                                                          +0.00169907554492985079,
                                                          +0.00169723725659599715,
                                                          +0.00169540782033222480,
                                                          +0.00169357746087613840,
                                                          +0.00169175589634952116,
                                                          +0.00168993341458362222,
                                                          +0.00168811967109731907,
                                                          +0.00168630501627369282,
                                                          +0.00168449904356526956,
                                                          +0.00168269216537095517,
                                                          +0.00168089391360774386,
                                                          +0.00167909476216008559,
                                                          +0.00167730418193459837,
                                                          +0.00167551270777665530,
                                                          +0.00167372975010206151,
                                                          +0.00167194590419807295,
                                                          +0.00167017052050373138,
                                                          +0.00166839425423464713,
                                                          +0.00166662639636169351,
                                                          +0.00166485766152075381,
                                                          +0.00166309728171774927,
                                                          +0.00166133603050612282,
                                                          +0.00165958308142475416,
                                                          +0.00165782926644723557,
                                                          +0.00165608370113806669,
                                                          +0.00165433727539882384,
                                                          +0.00165259904730710771,
                                                          +0.00165085996420548321,
                                                          +0.00164912902716702215,
                                                          +0.00164739724049338724,
                                                          +0.00164567354873044634,
                                                          +0.00164394901266211139,
                                                          +0.00164223252077938162,
                                                          +0.00164051518987655279,
                                                          +0.00163880585285716674,
                                                          +0.00163709568205895432,
                                                          +0.00163539345526055144,
                                                          +0.00163369039988102926,
                                                          +0.00163199523903186810,
                                                          +0.00163029925475618120,
                                                          +0.00162861111595130246,
                                                          +0.00162692215883182448,
                                                          +0.00162524099852925930,
                                                          +0.00162355902498179431,
                                                          +0.00162188479999882226,
                                                          +0.00162020976679885841,
                                                          +0.00161854243430830801,
                                                          +0.00161687429858731169,
                                                          +0.00161521381611391115,
                                                          +0.00161355253535567264,
                                                          +0.00161189886077244037,
                                                          +0.00161024439280946056,
                                                          +0.00160859748433414378,
                                                          +0.00160694978734406612,
                                                          +0.00160530960353562281,
                                                          +0.00160366863603770963,
                                                          +0.00160203513579283067,
                                                          +0.00160040085664448200,
                                                          +0.00159877399919415943,
                                                          +0.00159714636758747546,
                                                          +0.00159552611249360872,
                                                          +0.00159390508795199102,
                                                          +0.00159229139510403978,
                                                          +0.00159067693747883812,
                                                          +0.00158906976709050790,
                                                          +0.00158746183655770708,
                                                          +0.00158586114916368123,
                                                          +0.00158425970622062123,
                                                          +0.00158266546267333283,
                                                          +0.00158107046813547804,
                                                          +0.00157948262960191650,
                                                          +0.00157789404459965886,
                                                          +0.00157631257255821597,
                                                          +0.00157473035853371779,
                                                          +0.00157315521477107410,
                                                          +0.00157157933347514748,
                                                          +0.00157001048008319384,
                                                          +0.00156844089357221854,
                                                          +0.00156687829294501553,
                                                          +0.00156531496357789623,
                                                          +0.00156375857840866704,
                                                          +0.00156220146884381971,
                                                          +0.00156065126212198506,
                                                          +0.00155910033531436731,
                                                          +0.00155755627032260944,
                                                          +0.00155601148952078171,
                                                          +0.00155447352983214518,
                                                          +0.00155293485857536787,
                                                          +0.00155140296805039517,
                                                          +0.00154987037016575987,
                                                          +0.00154834451294966180,
                                                          +0.00154681795254925716,
                                                          +0.00154529809306911390,
                                                          +0.00154377753454722298,
                                                          +0.00154226363750922151,
                                                          +0.00154074904553955531,
                                                          +0.00153924107592625530,
                                                          +0.00153773241545921628,
                                                          +0.00153623033852685127,
                                                          +0.00153472757478682800,
                                                          +0.00153323135606263849,
                                                          +0.00153173445454533809,
                                                          +0.00153024405982493000,
                                                          +0.00152875298629473206,
                                                          +0.00152726838163947569,
                                                          +0.00152578310212682497,
                                                          +0.00152430425386127530,
                                                          +0.00152282473466010260,
                                                          +0.00152135160936945420,
                                                          +0.00151987781703462506,
                                                          +0.00151841038156219480,
                                                          +0.00151694228290699066,
                                                          +0.00151548050435173077,
                                                          +0.00151401806644535682,
                                                          +0.00151256191215939548,
                                                          +0.00151110510232451887,
                                                          +0.00150965453991073052,
                                                          +0.00150820332572104742,
                                                          +0.00150675832303064821,
                                                          +0.00150531267230847600,
                                                          +0.00150387319743865134,
                                                          +0.00150243307825255207,
                                                          +0.00150099909954410607,
                                                          +0.00149956448020653523,
                                                          +0.00149813596624157183,
                                                          +0.00149670681530655251,
                                                          +0.00149528373490617978,
                                                          +0.00149386002116700745,
                                                          +0.00149244234338907007,
                                                          +0.00149102403587603942,
                                                          +0.00148961173001287465,
                                                          +0.00148819879799103462,
                                                          +0.00148679183356725700,
                                                          +0.00148538424653419168,
                                                          +0.00148398259330449886,
                                                          +0.00148258032098813023,
                                                          +0.00148118394893513756,
                                                          +0.00147978696129155663,
                                                          +0.00147839584062365377,
                                                          +0.00147700410783497518,
                                                          +0.00147561820898420705,
                                                          +0.00147423170145644794,
                                                          +0.00147285099507641921,
                                                          +0.00147146968343740019,
                                                          +0.00147009414040120533,
                                                          +0.00146871799549847665,
                                                          +0.00146734758689665164,
                                                          +0.00146597657979544312,
                                                          +0.00146461127693393910,
                                                          +0.00146324537891513208,
                                                          +0.00146188515331331279,
                                                          +0.00146052433587143412,
                                                          +0.00145916915926009661,
                                                          +0.00145781339410133541,
                                                          +0.00145646323842075088,
                                                          +0.00145511249746100007,
                                                          +0.00145376733485897639,
                                                          +0.00145242159022188956,
                                                          +0.00145108139305185810,
                                                          +0.00144974061706693480,
                                                          +0.00144840535788605504,
                                                          +0.00144706952308674432,
                                                          +0.00144573917465402944,
                                                          +0.00144440825377585179,
                                                          +0.00144308278905031905,
                                                          +0.00144175675502901347,
                                                          +0.00144043614716785106,
                                                          +0.00143911497313754040,
                                                          +0.00143779919549429352,
                                                          +0.00143648285478566976,
                                                          +0.00143517188090845063,
                                                          +0.00143386034704698091,
                                                          +0.00143255415067669459,
                                                          +0.00143124739738084423,
                                                          +0.00142994595244943764,
                                                          +0.00142864395362891829,
                                                          +0.00142734723425764262,
                                                          +0.00142604996401167233,
                                                          +0.00142475794450937113,
                                                          +0.00142346537712495871,
                                                          +0.00142217803198636957,
                                                          +0.00142089014193661563,
                                                          +0.00141960744584069079,
                                                          +0.00141832420778310680,
                                                          +0.00141704613559135600,
                                                          +0.00141576752436620157,
                                                          +0.00141449405112104715,
                                                          +0.00141322004174968766,
                                                          +0.00141195114267284017,
                                                          +0.00141068171035611768,
                                                          +0.00140941736084697106,
                                                          +0.00140815248096359791,
                                                          +0.00140689265659763637,
                                                          +0.00140563230470259993,
                                                          +0.00140437698122982887,
                                                          +0.00140312113305281986,
                                                          +0.00140187028639620639,
                                                          +0.00140061891784005999,
                                                          +0.00139937252409399657,
                                                          +0.00139812561123315053,
                                                          +0.00139688364666192910,
                                                          +0.00139564116574089950,
                                                          +0.00139440360677720769,
                                                          +0.00139316553420908065,
                                                          +0.00139193235745250902,
                                                          +0.00139069866981744662,
                                                          +0.00138946985203301505,
                                                          +0.00138824052607678067,
                                                          +0.00138701604419347833,
                                                          +0.00138579105682597376,
                                                          +0.00138457088793531523,
                                                          +0.00138335021622913680,
                                                          +0.00138213433758373346,
                                                          +0.00138091795877273976,
                                                          +0.00137970634778488668,
                                                          +0.00137849423926278457,
                                                          +0.00137728687350306147,
                                                          +0.00137607901282200506,
                                                          +0.00137487587001789194,
                                                          +0.00137367223488709636,
                                                          +0.00137247329292160443,
                                                          +0.00137127386120597622,
                                                          +0.00137007909811629239,
                                                          +0.00136888384783507080,
                                                          +0.00136769324181121635,
                                                          +0.00136650215113663133,
                                                          +0.00136531568052013523,
                                                          +0.00136412872777607859,
                                                          +0.00136294637105866478,
                                                          +0.00136176353471937532,
                                                          +0.00136058527054166165,
                                                          +0.00135940652923042453,
                                                          +0.00135823233638063669,
                                                          +0.00135705766886849407,
                                                          +0.00135588752628119447,
                                                          +0.00135471691148567248,
                                                          +0.00135355079824049825,
                                                          +0.00135238421522434517,
                                                          +0.00135122211054476303,
                                                          +0.00135005953851470161,
                                                          +0.00134890142176677277,
                                                          +0.00134774284007226400,
                                                          +0.00134658869076342315,
                                                          +0.00134543407889544373,
                                                          +0.00134428387667329162,
                                                          +0.00134313321426312421,
                                                          +0.00134198693891422966,
                                                          +0.00134084020573226460,
                                                          +0.00133969783718098159,
                                                          +0.00133855501313553168,
                                                          +0.00133741653144282629,
                                                          +0.00133627759657895385,
                                                          +0.00133514298194124478,
                                                          +0.00133400791643959881,
                                                          +0.00133287714918760964,
                                                          +0.00133174593336327757,
                                                          +0.00133061899396089882,
                                                          +0.00132949160826226773,
                                                          +0.00132836847730543359,
                                                          +0.00132724490231306389,
                                                          +0.00132612556052863636,
                                                          +0.00132500577695414791,
                                                          +0.00132389020519881593,
                                                          +0.00132277419388378238,
                                                          +0.00132166237314297022,
                                                          +0.00132055011505782783,
                                                          +0.00131944202644461417,
                                                          +0.00131833350268757959,
                                                          +0.00131722912744162870,
                                                          +0.00131612431923762798,
                                                          +0.00131502363872413133,
                                                          +0.00131392252742374110,
                                                          +0.00131282552313236739,
                                                          +0.00131172809021076362,
                                                          +0.00131063474375462362,
                                                          +0.00130954097081054533,
                                                          +0.00130845126392516273,
                                                          +0.00130736113267988091,
                                                          +0.00130627504722217558,
                                                          +0.00130518853951847600,
                                                          +0.00130410605746575847,
                                                          +0.00130302315526693423,
                                                          +0.00130194425871590740,
                                                          +0.00130086494410475997,
                                                          +0.00129978961527053155,
                                                          +0.00129871387044838368,
                                                          +0.00129764209166349049,
                                                          +0.00129656989894920568,
                                                          +0.00129550165266264562,
                                                          +0.00129443299449166186,
                                                          +0.00129336826326793483,
                                                          +0.00129230312219130424,
                                                          +0.00129124188870946522,
                                                          +0.00129018024739290405,
                                                          +0.00128912249444562265,
                                                          +0.00128806433566857194,
                                                          +0.00128701004616120257,
                                                          +0.00128595535281589616,
                                                          +0.00128490450976555736,
                                                          +0.00128385326485610032,
                                                          +0.00128280585139076315,
                                                          +0.00128175803803221868,
                                                          +0.00128071403738980226,
                                                          +0.00127966963880728831,
                                                          +0.00127862903433476798,
                                                          +0.00127758803386256083,
                                                          +0.00127655080901508105,
                                                          +0.00127551319009572956,
                                                          +0.00127447932843572720,
                                                          +0.00127344507461917607,
                                                          +0.00127241455981551245,
                                                          +0.00127138365475823012,
                                                          +0.00127035647058533138,
                                                          +0.00126932889804945198,
                                                          +0.00126830502838645655,
                                                          +0.00126728077223892577,
                                                          +0.00126626020106884187,
                                                          +0.00126523924528057388,
                                                          +0.00126422195668944321,
                                                          +0.00126320428533448193,
                                                          +0.00126219026351055545,
                                                          +0.00126117586076524924,
                                                          +0.00126016508999816424,
                                                          +0.00125915394014034497,
                                                          +0.00125814640482031499,
                                                          +0.00125713849222848484,
                                                          +0.00125613417684549778,
                                                          +0.00125512948599802569,
                                                          +0.00125412837514104537,
                                                          +0.00125312689061537054,
                                                          +0.00125212896897154984,
                                                          +0.00125113067544339093,
                                                          +0.00125013592779729178,
                                                          +0.00124914081003986801,
                                                          +0.00124814922127268555,
                                                          +0.00124715726415593892,
                                                          +0.00124616881924473908,
                                                          +0.00124518000773457104,
                                                          +0.00124419469175152814,
                                                          +0.00124320901090903854,
                                                          +0.00124222680902068630,
                                                          +0.00124124424400142049,
                                                          +0.00124026514146790743,
                                                          +0.00123928567752111226,
                                                          +0.00123830965969546247,
                                                          +0.00123733328216334691,
                                                          +0.00123636033449073300,
                                                          +0.00123538702880773778,
                                                          +0.00123441713682475359,
                                                          +0.00123344688851682741,
                                                          +0.00123248003785077481,
                                                          +0.00123151283253465464,
                                                          +0.00123054900890283100,
                                                          +0.00122958483228533476,
                                                          +0.00122862402149433018,
                                                          +0.00122766285937165025,
                                                          +0.00122670504731665171,
                                                          +0.00122574688557365694,
                                                          +0.00122479205823775787,
                                                          +0.00122383688284730466,
                                                          +0.00122288502630082084,
                                                          +0.00122193282332306728,
                                                          +0.00122098392372285897,
                                                          +0.00122003467930458681,
                                                          +0.00121908872289338915,
                                                          +0.00121814242326733282,
                                                          +0.00121719939637308949,
                                                          +0.00121625602785726877,
                                                          +0.00121531591689247429,
                                                          +0.00121437546588953720,
                                                          +0.00121343825735058339,
                                                          +0.00121250071034715158,
                                                          +0.00121156639081368184,
                                                          +0.00121063173437970309,
                                                          +0.00120970029051397157,
                                                          +0.00120876851130207862,
                                                          +0.00120783992984831580,
                                                          +0.00120691101459319250,
                                                          +0.00120598528237697501,
                                                          +0.00120505921789472491,
                                                          +0.00120413632182235475,
                                                          +0.00120321309500987742,
                                                          +0.00120229302206776309,
                                                          +0.00120137261990213790,
                                                          +0.00120045535715618246,
                                                          +0.00119953776669405490,
                                                          +0.00119862330128905122,
                                                          +0.00119770850966602533,
                                                          +0.00119679682882505457,
                                                          +0.00119588482325509423,
                                                          +0.00119497591427893081,
                                                          +0.00119406668205376170,
                                                          +0.00119316053232028417,
                                                          +0.00119225406080880484,
                                                          +0.00119135065777241107,
                                                          +0.00119044693442010765,
                                                          +0.00118954626561113561,
                                                          +0.00118864527793950182,
                                                          +0.00118774733096365715,
                                                          +0.00118684906656962071,
                                                          +0.00118595382910740687,
                                                          +0.00118505827566275901,
                                                          +0.00118416573546891433,
                                                          +0.00118327288071974819,
                                                          +0.00118238302562268774,
                                                          +0.00118149285738883649,
                                                          +0.00118060567529009873,
                                                          +0.00117971818146458463,
                                                          +0.00117883366033828254,
                                                          +0.00117794882888676549,
                                                          +0.00117706695677904363,
                                                          +0.00117618477573927928,
                                                          +0.00117530554076777558,
                                                          +0.00117442599824907454,
                                                          +0.00117354938860238667,
                                                          +0.00117267247278508152,
                                                          +0.00117179847672223714,
                                                          +0.00117092417585715289,
                                                          +0.00117005278170708577,
                                                          +0.00116918108411501382,
                                                          +0.00116831228027604534,
                                                          +0.00116744317434722524,
                                                          +0.00116657694928654770,
                                                          +0.00116571042348015039,
                                                          +0.00116484676573331918,
                                                          +0.00116398280857693613,
                                                          +0.00116312170674736119,
                                                          +0.00116226030683650025,
                                                          +0.00116140174959494607,
                                                          +0.00116054289559252666,
                                                          +0.00115968687167661614,
                                                          +0.00115883055231247333,
                                                          +0.00115797705052619426,
                                                          +0.00115712325459658614,
                                                          +0.00115627226380980350,
                                                          +0.00115542098017692215,
                                                          +0.00115457248932489417,
                                                          +0.00115372370691638041,
                                                          +0.00115287770499927951,
                                                          +0.00115203141280774413,
                                                          +0.00115118788889018121,
                                                          +0.00115034407597272939,
                                                          +0.00114950301918328155,
                                                          +0.00114866167466104040,
                                                          +0.00114782307419178582,
                                                          +0.00114698418724943642,
                                                          +0.00114614803235549051,
                                                          +0.00114531159224080564,
                                                          +0.00114447787223986158,
                                                          +0.00114364386826324642,
                                                          +0.00114281257253512109,
                                                          +0.00114198099406915524,
                                                          +0.00114115211205533884,
                                                          +0.00114032294853432816,
                                                          +0.00113949646973753537,
                                                          +0.00113866971065706415,
                                                          +0.00113784562464079196,
                                                          +0.00113702125955727627,
                                                          +0.00113619955594536554,
                                                          +0.00113537757447561738,
                                                          +0.00113455824295181570,
                                                          +0.00113373863477260429,
                                                          +0.00113292166508013534,
                                                          +0.00113210441992775673,
                                                          +0.00113128980186889177,
                                                          +0.00113047490953873927,
                                                          +0.00112966263297437325,
                                                          +0.00112885008332051285,
                                                          +0.00112804013816974395,
                                                          +0.00112722992110449367,
                                                          +0.00112642229734420500,
                                                          +0.00112561440283771722,
                                                          +0.00112480909050216549,
                                                          +0.00112400350858201281,
                                                          +0.00112320049776241590,
                                                          +0.00112239721851318131,
                                                          +0.00112159649935731406,
                                                          +0.00112079551292018392,
                                                          +0.00111999707563197395,
                                                          +0.00111919837220433491,
                                                          +0.00111840220704346278,
                                                          +0.00111760577687850080,
                                                          +0.00111681187416000599,
                                                          +0.00111601770756631171,
                                                          +0.00111522605766019712,
                                                          +0.00111443414500137080,
                                                          +0.00111364473833221590,
                                                          +0.00111285507002647924,
                                                          +0.00111206789707305213,
                                                          +0.00111128046359286058,
                                                          +0.00111049551488773659,
                                                          +0.00110971030675939769,
                                                          +0.00110892757288857794,
                                                          +0.00110814458069186930,
                                                          +0.00110736405229440709,
                                                          +0.00110658326666220161,
                                                          +0.00110580493442982708,
                                                          +0.00110502634604771831,
                                                          +0.00110425020072446914,
                                                          +0.00110347380033040112,
                                                          +0.00110269983271225544,
                                                          +0.00110192561109615601,
                                                          +0.00110115381203066895,
                                                          +0.00110038176003408438,
                                                          +0.00109961212042002684,
                                                          +0.00109884222893576189,
                                                          +0.00109807473972276318,
                                                          +0.00109730699969452197,
                                                          +0.00109654165188271526,
                                                          +0.00109577605430474768,
                                                          +0.00109501283894441657,
                                                          +0.00109424937486116386,
                                                          +0.00109348828305239672,
                                                          +0.00109272694355814484,
                                                          +0.00109196796645048464,
                                                          +0.00109120874268901617,
                                                          +0.00109045187148112174,
                                                          +0.00108969475464537266,
                                                          +0.00108893998058467523,
                                                          +0.00108818496191639535,
                                                          +0.00108743227629876332,
                                                          +0.00108667934708817754,
                                                          +0.00108592874125758017,
                                                          +0.00108517789284305502,
                                                          +0.00108442935819123127,
                                                          +0.00108368058195893987,
                                                          +0.00108293410992507081,
                                                          +0.00108218739730866695,
                                                          +0.00108144297937904656,
                                                          +0.00108069832185933579,
                                                          +0.00107995594956704919,
                                                          +0.00107921333867166546,
                                                          +0.00107847300359626796,
                                                          +0.00107773243089935274,
                                                          +0.00107699412466655026,
                                                          +0.00107625558178843267,
                                                          +0.00107551929606976727,
                                                          +0.00107478277467664969,
                                                          +0.00107404850118918561,
                                                          +0.00107331399299282816,
                                                          +0.00107258172349884175,
                                                          +0.00107184922025625393,
                                                          +0.00107111894656292464,
                                                          +0.00107038844007605440,
                                                          +0.00106966015403516077,
                                                          +0.00106893163615059007,
                                                          +0.00106820532965820527,
                                                          +0.00106747879226684714,
                                                          +0.00106675445726303912,
                                                          +0.00106602989229983625,
                                                          +0.00106530752076836824,
                                                          +0.00106458492021199312,
                                                          +0.00106386450418003158,
                                                          +0.00106314386005259285,
                                                          +0.00106242539159040949,
                                                          +0.00106170669595715751,
                                                          +0.00106099016717784095,
                                                          +0.00106027341214687590,
                                                          +0.00105955881520604324,
                                                          +0.00105884399292802698,
                                                          +0.00105813132002353602,
                                                          +0.00105741842269140560,
                                                          +0.00105670766606307236,
                                                          +0.00105599668591175493,
                                                          +0.00105528783784107128,
                                                          +0.00105457876714720278,
                                                          +0.00105387181995705739,
                                                          +0.00105316465103870351,
                                                          +0.00105245959709310386,
                                                          +0.00105175432230948142,
                                                          +0.00105105115401327989,
                                                          +0.00105034776576448273,
                                                          +0.00104964647556310322,
                                                          +0.00104894496628982811,
                                                          +0.00104824554666899583,
                                                          +0.00104754590885227229,
                                                          +0.00104684835233774575,
                                                          +0.00104615057849866784,
                                                          +0.00104545487765597147,
                                                          +0.00104475896035542996,
                                                          +0.00104406510778959089,
                                                          +0.00104337103962801069,
                                                          +0.00104267902798329658,
                                                          +0.00104198680160037349,
                                                          +0.00104129662356003232,
                                                          +0.00104060623163447215,
                                                          +0.00103991787992047482,
                                                          +0.00103922931516973712,
                                                          +0.00103854278254252030,
                                                          +0.00103785603772256008,
                                                          +0.00103717131698077510,
                                                          +0.00103648638488579108,
                                                          +0.00103580346886604914,
                                                          +0.00103512034232822940,
                                                          +0.00103443922390485435,
                                                          +0.00103375789579412806,
                                                          +0.00103307856787890600,
                                                          +0.00103239903110269467,
                                                          +0.00103172148664463039,
                                                          +0.00103104373414760224,
                                                          +0.00103036796613267334,
                                                          +0.00102969199089649800,
                                                          +0.00102901799234741487,
                                                          +0.00102834378739052271,
                                                          +0.00102767155136648607,
                                                          +0.00102699910974382645,
                                                          +0.00102632862934029121,
                                                          +0.00102565794414309413,
                                                          +0.00102498921249153199,
                                                          +0.00102432027684707283,
                                                          +0.00102365328711473782,
                                                          +0.00102298609418610101,
                                                          +0.00102232083957579643,
                                                          +0.00102165538256164382,
                                                          +0.00102099185631149194,
                                                          +0.00102032812844582996,
                                                          +0.00101966632382904346,
                                                          +0.00101900431838099562,
                                                          +0.00101834422870564841,
                                                          +0.00101768393897922740,
                                                          +0.00101702555758803040,
                                                          +0.00101636697692191066,
                                                          +0.00101571029719198886,
                                                          +0.00101505341895928515,
                                                          +0.00101439843430195325,
                                                          +0.00101374325190999623,
                                                          +0.00101308995577053997,
                                                          +0.00101243646266065742,
                                                          +0.00101178484851811423,
                                                          +0.00101113303816541072,
                                                          +0.00101048309953235258,
                                                          +0.00100983296544549309,
                                                          +0.00100918469586781258,
                                                          +0.00100853623158880649,
                                                          +0.00100788962464550165,
                                                          +0.00100724282374948897,
                                                          +0.00100659787305245230,
                                                          +0.00100595272914749245,
                                                          +0.00100530942834129908,
                                                          +0.00100466593506815991,
                                                          +0.00100402427782986031,
                                                          +0.00100338242886180776,
                                                          +0.00100274240890072062,
                                                          +0.00100210219794331316,
                                                          +0.00100146380900081877,
                                                          +0.00100082522979170032,
                                                          +0.00100018846564103761,
                                                          +0.00099955151194973477,
                                                          +0.00099891636639579768,
                                                          +0.00099828103202351595,
                                                          +0.00099764749890265351,
                                                          +0.00099701377768207601,
                                                          +0.00099638185086189327,
                                                          +0.00099574973665698301,
                                                          +0.00099511941003614115,
                                                          +0.00099448889674194066,
                                                          +0.00099386016424996286,
                                                          +0.00099323124579239882,
                                                          +0.00099260410138947517,
                                                          +0.00099197677172516456,
                                                          +0.00099135120940195597,
                                                          +0.00099072546251800916,
                                                          +0.00099010147629545985,
                                                          +0.00098947730620929110,
                                                          +0.00098885489013843470,
                                                          +0.00098823229089757179,
                                                          +0.00098761143905934292,
                                                          +0.00098699040474123649,
                                                          +0.00098637111124628431,
                                                          +0.00098575163595812014,
                                                          +0.00098513389494662095,
                                                          +0.00098451597282513494,
                                                          +0.00098389977846660768,
                                                          +0.00098328340367789921,
                                                          +0.00098266875017102222,
                                                          +0.00098205391691037216,
                                                          +0.00098144079848280124,
                                                          +0.00098082750097448768,
                                                          +0.00098021591188267483,
                                                          +0.00097960414437979438,
                                                          +0.00097899407890880913,
                                                          +0.00097838383569309631,
                                                          +0.00097777528815644679,
                                                          +0.00097716656353809619,
                                                          +0.00097655952827755310,
                                                          +0.00097595231659504183,
                                                          +0.00097534678798046374,
                                                          +0.00097474108360037651,
                                                          +0.00097413705602953445,
                                                          +0.00097353285334638969,
                                                          +0.00097293032124479509,
                                                          +0.00097232761468087168,
                                                          +0.00097172657250160451,
                                                          +0.00097112535650676983,
                                                          +0.00097052579873030918,
                                                          +0.00096992606778184901,
                                                          +0.00096932798891590302,
                                                          +0.00096872973751835258,
                                                          +0.00096813313209769176,
                                                          +0.00096753635478266761,
                                                          +0.00096694121736895771,
                                                          +0.00096634590869499140,
                                                          +0.00096575223387662830,
                                                          +0.00096515838842900058,
                                                          +0.00096456617082094662,
                                                          +0.00096397378321152398,
                                                          +0.00096338301745514448,
                                                          +0.00096279208232221509,
                                                          +0.00096220276308511745,
                                                          +0.00096161327509323047,
                                                          +0.00096102539706910308,
                                                          +0.00096043735090890747,
                                                          +0.00095985090881736115,
                                                          +0.00095926429920544691,
                                                          +0.00095867928779185560,
                                                          +0.00095809410947059464,
                                                          +0.00095751052350594016,
                                                          +0.00095692677124332905,
                                                          +0.00095634460552404435,
                                                          +0.00095576227411354987,
                                                          +0.00095518152346136425,
                                                          +0.00095460060772176788,
                                                          +0.00095402126698355288,
                                                          +0.00095344176175879663,
                                                          +0.00095286382580641530,
                                                          +0.00095228572596544919,
                                                          +0.00095170918969560375,
                                                          +0.00095113249013223519,
                                                          +0.00095055734846631666,
                                                          +0.00094998204409905941,
                                                          +0.00094940829198299898,
                                                          +0.00094883437775492409,
                                                          +0.00094826201015904457,
                                                          +0.00094768948103763251,
                                                          +0.00094711849295650151,
                                                          +0.00094654734393349448,
                                                          +0.00094597773038577939,
                                                          +0.00094540795647703548,
                                                          +0.00094483971250535728,
                                                          +0.00094427130875070577,
                                                          +0.00094370442942149572,
                                                          +0.00094313739088459350,
                                                          +0.00094257187128795044,
                                                          +0.00094200619305613760,
                                                          +0.00094144202830568659,
                                                          +0.00094087770548984646,
                                                          +0.00094031489072259717,
                                                          +0.00093975191845701374,
                                                          +0.00093919044883322147,
                                                          +0.00093862882227544126,
                                                          +0.00093806869297846749,
                                                          +0.00093750840730915893,
                                                          +0.00093694961354533440,
                                                          +0.00093639066396815145,
                                                          +0.00093583320096663912,
                                                          +0.00093527558270808323,
                                                          +0.00093471944572074230,
                                                          +0.00093416315403002778,
                                                          +0.00093360833833127873,
                                                          +0.00093305336848019632,
                                                          +0.00093249986936688799,
                                                          +0.00093194621664967239,
                                                          +0.00093139402944094761,
                                                          +0.00093084168917414382,
                                                          +0.00093029080921130881,
                                                          +0.00092973977673363971,
                                                          +0.00092919019938003222,
                                                          +0.00092864047005226805,
                                                          +0.00092809219069312778,
                                                          +0.00092754375989795562,
                                                          +0.00092699677394029480,
                                                          +0.00092644963708218885,
                                                          +0.00092590393995466451,
                                                          +0.00092535809245975875,
                                                          +0.00092481367961254432,
                                                          +0.00092426911692850510,
                                                          +0.00092372598383316437,
                                                          +0.00092318270142906222,
                                                          +0.00092264084357842442,
                                                          +0.00092209883694461029,
                                                          +0.00092155824985264481,
                                                          +0.00092101751450062385,
                                                          +0.00092047819370231685,
                                                          +0.00091993872516462575,
                                                          +0.00091940066621585626,
                                                          +0.00091886246004593906,
                                                          +0.00091832565852335846,
                                                          +0.00091778871029544518,
                                                          +0.00091725316179635467,
                                                          +0.00091671746710533981,
                                                          +0.00091618316724757039,
                                                          +0.00091564872170889233,
                                                          +0.00091511566613068575,
                                                          +0.00091458246538020707,
                                                          +0.00091405064974009649,
                                                          +0.00091351868943398548,
                                                          +0.00091298810941067832,
                                                          +0.00091245738522529067,
                                                          +0.00091192803651755070,
                                                          +0.00091139854414931274,
                                                          +0.00091087042247584428,
                                                          +0.00091034215764113611,
                                                          +0.00090981525874046928,
                                                          +0.00090928821717551006,
                                                          +0.00090876253680588471,
                                                          +0.00090823671426661751,
                                                          +0.00090771224820587022,
                                                          +0.00090718764046784732,
                                                          +0.00090666438451329941,
                                                          +0.00090614098737157062,
                                                          +0.00090561893733991355,
                                                          +0.00090509674660891259,
                                                          +0.00090457589833609867,
                                                          +0.00090405490984953312,
                                                          +0.00090353525919066276,
                                                          +0.00090301546880140130,
                                                          +0.00090249701163061546,
                                                          +0.00090197841521057979,
                                                          +0.00090146114742094827,
                                                          +0.00090094374086100196,
                                                          +0.00090042765836441710,
                                                          +0.00089991143757425849,
                                                          +0.00089939653630132627,
                                                          +0.00089888149720937927,
                                                          +0.00089836777310931318,
                                                          +0.00089785391166262299,
                                                          +0.00089734136070313572,
                                                          +0.00089682867286725921,
                                                          +0.00089631729103445998,
                                                          +0.00089580577279336170,
                                                          +0.00089529555609165010,
                                                          +0.00089478520344759768,
                                                          +0.00089427614789955908,
                                                          +0.00089376695687301815,
                                                          +0.00089325905851932164,
                                                          +0.00089275102514885371,
                                                          +0.00089224428004814824,
                                                          +0.00089173740039030596,
                                                          +0.00089123180461911963,
                                                          +0.00089072607474834705,
                                                          +0.00089022162440098422,
                                                          +0.00088971704040951400,
                                                          +0.00088921373159795607,
                                                          +0.00088871028959570930,
                                                          +0.00088820811844951444,
                                                          +0.00088770581456400004,
                                                          +0.00088720477723020419,
                                                          +0.00088670360760642099,
                                                          +0.00088620370024943842,
                                                          +0.00088570366104977509,
                                                          +0.00088520487985130141,
                                                          +0.00088470596725543933,
                                                          +0.00088420830841435432,
                                                          +0.00088371051861917008,
                                                          +0.00088321397835144032,
                                                          +0.00088271730757090883,
                                                          +0.00088222188210949318,
                                                          +0.00088172632657459234,
                                                          +0.00088123201216934466,
                                                          +0.00088073756812795921,
                                                          +0.00088024436104553578,
                                                          +0.00087975102476236351,
                                                          +0.00087925892128612752,
                                                          +0.00087876668904258428,
                                                          +0.00087827568547251369,
                                                          +0.00087778455356664017,
                                                          +0.00087729464621923397,
                                                          +0.00087680461096560397,
                                                          +0.00087631579617379054,
                                                          +0.00087582685390341728,
                                                          +0.00087533912801646282,
                                                          +0.00087485127507670767,
                                                          +0.00087436463446012580,
                                                          +0.00087387786721460851,
                                                          +0.00087339230825006877,
                                                          +0.00087290662307857594,
                                                          +0.00087242214216381498,
                                                          +0.00087193753546221074,
                                                          +0.00087145412901094355,
                                                          +0.00087097059719107989,
                                                          +0.00087048826163291086,
                                                          +0.00087000580112253959,
                                                          +0.00086952453290287449,
                                                          +0.00086904314014555966,
                                                          +0.00086856293572551847,
                                                          +0.00086808260718054797,
                                                          +0.00086760346303687782,
                                                          +0.00086712419517917777,
                                                          +0.00086664610780416717,
                                                          +0.00086616789712421426,
                                                          +0.00086569086302560802,
                                                          +0.00086521370602934478,
                                                          +0.00086473772173025799,
                                                          +0.00086426161493900820,
                                                          +0.00086378667697784137,
                                                          +0.00086331161692822407,
                                                          +0.00086283772185858124,
                                                          +0.00086236370510242761,
                                                          +0.00086189084949303118,
                                                          +0.00086141787259730123,
                                                          +0.00086094605303190940,
                                                          +0.00086047411257860937,
                                                          +0.00086000332565593432,
                                                          +0.00085953241824203279,
                                                          +0.00085906266057565934,
                                                          +0.00085859278281300716,
                                                          +0.00085812405103131029,
                                                          +0.00085765519954655834,
                                                          +0.00085718749029262391,
                                                          +0.00085671966172714306,
                                                          +0.00085625297165868662,
                                                          +0.00085578616266848728,
                                                          +0.00085532048845777446,
                                                          +0.00085485469571342637,
                                                          +0.00085439003404719509,
                                                          +0.00085392525423374837,
                                                          +0.00085346160181312897,
                                                          +0.00085299783163003662,
                                                          +0.00085253518517047376,
                                                          +0.00085207242133151161,
                                                          +0.00085161077756268789,
                                                          +0.00085114901679587864,
                                                          +0.00085068837246163624,
                                                          +0.00085022761150917159,
                                                          +0.00084976796336743603,
                                                          +0.00084930819898560025,
                                                          +0.00084884954380830507,
                                                          +0.00084839077276739790,
                                                          +0.00084793310734040896,
                                                          +0.00084747532642467233,
                                                          +0.00084701864754771104,
                                                          +0.00084656185355525145,
                                                          +0.00084610615804182247,
                                                          +0.00084565034778453770,
                                                          +0.00084519563246185370,
                                                          +0.00084474080276535828,
                                                          +0.00084428706447426608,
                                                          +0.00084383321217781730,
                                                          +0.00084338044777272593,
                                                          +0.00084292756972915170,
                                                          +0.00084247577607795751,
                                                          +0.00084202386915358223,
                                                          +0.00084157304313759926,
                                                          +0.00084112210421217245,
                                                          +0.00084067224272605854,
                                                          +0.00084022226869268275,
                                                          +0.00083977336864436973,
                                                          +0.00083932435640942900,
                                                          +0.00083887641472005085,
                                                          +0.00083842836120314010,
                                                          +0.00083798137480696315,
                                                          +0.00083753427694081785,
                                                          +0.00083708824278517038,
                                                          +0.00083664209751559636,
                                                          +0.00083619701256079952,
                                                          +0.00083575181684660286,
                                                          +0.00083530767806590158,
                                                          +0.00083486342887882073,
                                                          +0.00083442023325831468,
                                                          +0.00083397692758295082,
                                                          +0.00083353467212152676,
                                                          +0.00083309230695527588,
                                                          +0.00083265098866453966,
                                                          +0.00083220956101752474,
                                                          +0.00083176917692173410,
                                                          +0.00083132868381673782,
                                                          +0.00083088923095273547,
                                                          +0.00083044966942513309,
                                                          +0.00083001114484228017,
                                                          +0.00082957251193997391,
                                                          +0.00082913491270008341,
                                                          +0.00082869720548343461,
                                                          +0.00082826052866070748,
                                                          +0.00082782374420247264,
                                                          +0.00082738798688343040,
                                                          +0.00082695212226869467,
                                                          +0.00082651728155211639,
                                                          +0.00082608233387823009,
                                                          +0.00082564840687508680,
                                                          +0.00082521437325159923,
                                                          +0.00082478135708499061,
                                                          +0.00082434823463358787,
                                                          +0.00082391612643867816,
                                                          +0.00082348391229311786,
                                                          +0.00082305270921707386,
                                                          +0.00082262140052312318,
                                                          +0.00082219109972504940,
                                                          +0.00082176069364042141,
                                                          +0.00082133129229130001,
                                                          +0.00082090178598559157,
                                                          +0.00082047328126821852,
                                                          +0.00082004467192284887,
                                                          +0.00081961706103177297,
                                                          +0.00081918934583992062,
                                                          +0.00081876262598138262,
                                                          +0.00081833580214792545,
                                                          +0.00081790997053979709,
                                                          +0.00081748403528125041,
                                                          +0.00081705908915297371,
                                                          +0.00081663403969743089,
                                                          +0.00081620997628995776,
                                                          +0.00081578580987702880,
                                                          +0.00081536262644276239,
                                                          +0.00081493934032351534,
                                                          +0.00081451703412624952,
                                                          +0.00081409462556315012,
                                                          +0.00081367319387801141,
                                                          +0.00081325166014486496,
                                                          +0.00081283110025825338,
                                                          +0.00081241043864014565,
                                                          +0.00081199074784967613,
                                                          +0.00081157095564291469,
                                                          +0.00081115213125736076,
                                                          +0.00081073320576941811,
                                                          +0.00081031524510865216,
                                                          +0.00080989718365810659,
                                                          +0.00080948008405304481,
                                                          +0.00080906288396952490,
                                                          +0.00080864664276206904,
                                                          +0.00080823030138619601,
                                                          +0.00080781491592917714,
                                                          +0.00080739943061250788,
                                                          +0.00080698489826963094,
                                                          +0.00080657026637460233,
                                                          +0.00080615658452039024,
                                                          +0.00080574280342026350,
                                                          +0.00080532996944000074,
                                                          +0.00080491703651880625,
                                                          +0.00080450504780848434,
                                                          +0.00080409296046096549,
                                                          +0.00080368181442722910,
                                                          +0.00080327057005878829,
                                                          +0.00080286026411887940,
                                                          +0.00080244986014552295,
                                                          +0.00080204039172722798,
                                                          +0.00080163082557551280,
                                                          +0.00080122219211710812,
                                                          +0.00080081346122408794,
                                                          +0.00080040566017428624,
                                                          +0.00079999776198745674,
                                                          +0.00079959079080535499,
                                                          +0.00079918372278260251,
                                                          +0.00079877757893762817,
                                                          +0.00079837133854717584,
                                                          +0.00079796601951903482,
                                                          +0.00079756060423938986,
                                                          +0.00079715610751801460,
                                                          +0.00079675151483791725,
                                                          +0.00079634783792341445,
                                                          +0.00079594406534178508,
                                                          +0.00079554120574438418,
                                                          +0.00079513825077027220,
                                                          +0.00079473620601027480,
                                                          +0.00079433406616280717,
                                                          +0.00079393283377053581,
                                                          +0.00079353150657886695,
                                                          +0.00079313108409461385,
                                                          +0.00079273056709787361,
                                                          +0.00079233095207185175,
                                                          +0.00079193124281909670,
                                                          +0.00079153243281138879,
                                                          +0.00079113352886155076,
                                                          +0.00079073552144206030,
                                                          +0.00079033742036389723,
                                                          +0.00078994021311229879,
                                                          +0.00078954291248434618,
                                                          +0.00078914650299003592,
                                                          +0.00078875000040055671,
                                                          +0.00078835438626260437,
                                                          +0.00078795867930954043,
                                                          +0.00078756385813664044,
                                                          +0.00078716894442756381,
                                                          +0.00078677491383798745,
                                                          +0.00078638079099005323,
                                                          +0.00078598754861159987,
                                                          +0.00078559421425149613,
                                                          +0.00078520175772144724,
                                                          +0.00078480920948534956,
                                                          +0.00078441753645041993,
                                                          +0.00078402577198394329,
                                                          +0.00078363488010023429,
                                                          +0.00078324389705838578,
                                                          +0.00078285378399133867,
                                                          +0.00078246358003847087,
                                                          +0.00078207424346282066,
                                                          +0.00078168481627258494,
                                                          +0.00078129625387231402,
                                                          +0.00078090760112761422,
                                                          +0.00078051981059590661,
                                                          +0.00078013192998885393,
                                                          +0.00077974490902804887,
                                                          +0.00077935779825991515,
                                                          +0.00077897154458146265,
                                                          +0.00077858520136263559,
                                                          +0.00077819971268705047,
                                                          +0.00077781413473698843,
                                                          +0.00077742940879380659,
                                                          +0.00077704459384099283,
                                                          +0.00077666062836872574,
                                                          +0.00077627657415062527,
                                                          +0.00077589336689671564,
                                                          +0.00077551007115973011,
                                                          +0.00077512761988050871,
                                                          +0.00077474508037993250,
                                                          +0.00077436338284057296,
                                                          +0.00077398159734054874,
                                                          +0.00077360065131502551,
                                                          +0.00077321961758850162,
                                                          +0.00077283942085954610,
                                                          +0.00077245913668823264,
                                                          +0.00077207968704728978,
                                                          +0.00077170015022161558,
                                                          +0.00077132144546880155,
                                                          +0.00077094265378787219,
                                                          +0.00077056469173193167,
                                                          +0.00077018664300348527,
                                                          +0.00076980942146174971,
                                                          +0.00076943211350211690,
                                                          +0.00076905563030046106,
                                                          +0.00076867906093452032,
                                                          +0.00076830331390732279,
                                                          +0.00076792748096846108,
                                                          +0.00076755246795856069,
                                                          +0.00076717736928863013,
                                                          +0.00076680308814728573,
                                                          +0.00076642872159656352,
                                                          +0.00076605517018341247,
                                                          +0.00076568153361055899,
                                                          +0.00076530870979357670,
                                                          +0.00076493580106559537,
                                                          +0.00076456370272105420,
                                                          +0.00076419151971325009,
                                                          +0.00076382014472568013,
                                                          +0.00076344868532162000,
                                                          +0.00076307803158376776,
                                                          +0.00076270729367524021,
                                                          +0.00076233735908802958,
                                                          +0.00076196734057500516,
                                                          +0.00076159812304749705,
                                                          +0.00076122882183808768,
                                                          +0.00076086031928744146,
                                                          +0.00076049173329786239,
                                                          +0.00076012394364929621,
                                                          +0.00075975607080382569,
                                                          +0.00075938899199057763,
                                                          +0.00075902183022151851,
                                                          +0.00075865546018480769,
                                                          +0.00075828900743244896,
                                                          +0.00075792334412143759,
                                                          +0.00075755759833401489,
                                                          +0.00075719263970576932,
                                                          +0.00075682759883942755,
                                                          +0.00075646334285888080,
                                                          +0.00075609900487763561,
                                                          +0.00075573544951754938,
                                                          +0.00075537181239324926,
                                                          +0.00075500895563417611,
                                                          +0.00075464601734646550,
                                                          +0.00075428385717671146,
                                                          +0.00075392161571299226,
                                                          +0.00075356015012857994,
                                                          +0.00075319860348397488,
                                                          +0.00075283783048860661,
                                                          +0.00075247697666592237,
                                                          +0.00075211689427094250,
                                                          +0.00075175673128063275,
                                                          +0.00075139733750499249,
                                                          +0.00075103786336512123,
                                                          +0.00075067915623534126,
                                                          +0.00075032036897154653,
                                                          +0.00074996234652168164,
                                                          +0.00074960424416713959,
                                                          +0.00074924690443874185,
                                                          +0.00074888948503413057,
                                                          +0.00074853282607621437,
                                                          +0.00074817608766967797,
                                                          +0.00074782010753868461,
                                                          +0.00074746404818579811,
                                                          +0.00074710874494555990,
                                                          +0.00074675336270929398,
                                                          +0.00074639873443099931,
                                                          +0.00074604402738168455,
                                                          +0.00074569007214384312,
                                                          +0.00074533603835913562,
                                                          +0.00074498275424754402,
                                                          +0.00074462939181239049,
                                                          +0.00074427677692009693,
                                                          +0.00074392408392670009,
                                                          +0.00074357213635397104,
                                                          +0.00074322011090175561,
                                                          +0.00074286882875604062,
                                                          +0.00074251746895161847,
                                                          +0.00074216685034751749,
                                                          +0.00074181615430465463,
                                                          +0.00074146619736388303,
                                                          +0.00074111616320346550,
                                                          +0.00074076686605482108,
                                                          +0.00074041749190482064,
                                                          +0.00074006885268415069,
                                                          +0.00073972013667959258,
                                                          +0.00073937215352976038,
                                                          +0.00073902409381268941,
                                                          +0.00073867676488354069,
                                                          +0.00073832935960298890,
                                                          +0.00073798268305131991,
                                                          +0.00073763593036327258,
                                                          +0.00073728990435279710,
                                                          +0.00073694380242016167,
                                                          +0.00073659842512147859,
                                                          +0.00073625297211405100,
                                                          +0.00073590824170461202,
                                                          +0.00073556343579904491,
                                                          +0.00073521935046312278,
                                                          +0.00073487518984289332,
                                                          +0.00073453174777154972,
                                                          +0.00073418823062692780,
                                                          +0.00073384543001798162,
                                                          +0.00073350255454599834,
                                                          +0.00073316039360399397,
                                                          +0.00073281815800840962,
                                                          +0.00073247663494458641,
                                                          +0.00073213503743585918,
                                                          +0.00073179415046811921,
                                                          +0.00073145318926337479,
                                                          +0.00073111293661625224,
                                                          +0.00073077260993925110,
                                                          +0.00073043298984388247,
                                                          +0.00073009329592499124,
                                                          +0.00072975430661908276,
                                                          +0.00072941524369524125,
                                                          +0.00072907688342304007,
                                                          +0.00072873844973773321,
                                                          +0.00072840071674999587,
                                                          +0.00072806291055322095,
                                                          +0.00072772580310718441,
                                                          +0.00072738862265542248,
                                                          +0.00072705213901477338,
                                                          +0.00072671558257095920,
                                                          +0.00072637972100580354,
                                                          +0.00072604378683929446,
                                                          +0.00072570854562612993,
                                                          +0.00072537323201267792,
                                                          +0.00072503860943436192,
                                                          +0.00072470391465608334,
                                                          +0.00072436990900180524,
                                                          +0.00072403583134715067,
                                                          +0.00072370244091240262,
                                                          +0.00072336897867612924,
                                                          +0.00072303620176267626,
                                                          +0.00072270335324581779,
                                                          +0.00072237118816166981,
                                                          +0.00072203895167150770,
                                                          +0.00072170739673089069,
                                                          +0.00072137577058092612,
                                                          +0.00072104482410425316,
                                                          +0.00072071380661417757,
                                                          +0.00072038346692802110,
                                                          +0.00072005305642368865,
                                                          +0.00071972332186075140,
                                                          +0.00071939351667415022,
                                                          +0.00071906438557323832,
                                                          +0.00071873518404246223,
                                                          +0.00071840665474845662,
                                                          +0.00071807805521767844,
                                                          +0.00071775012608150664,
                                                          +0.00071742212690094895,
                                                          +0.00071709479627955850,
                                                          +0.00071676739580546673,
                                                          +0.00071644066206179725,
                                                          +0.00071611385865641224,
                                                          +0.00071578772015936808,
                                                          +0.00071546151219089844,
                                                          +0.00071513596731532169,
                                                          +0.00071481035315791727,
                                                          +0.00071448540028456076,
                                                          +0.00071416037831828472,
                                                          +0.00071383601583378505,
                                                          +0.00071351158444458800,
                                                          +0.00071318781074143894,
                                                          +0.00071286396832113150,
                                                          +0.00071254078179765722,
                                                          +0.00071221752674388388,
                                                          +0.00071189492580421272,
                                                          +0.00071157225652042496,
                                                          +0.00071125023957446343,
                                                          +0.00071092815446989390,
                                                          +0.00071060671993330017,
                                                          +0.00071028521742293627,
                                                          +0.00070996436371709420,
                                                          +0.00070964344222165180,
                                                          +0.00070932316777364555,
                                                          +0.00070900282571954398,
                                                          +0.00070868312896213165,
                                                          +0.00070836336478146756,
                                                          +0.00070804424415305553,
                                                          +0.00070772505628357684,
                                                          +0.00070740651022819557,
                                                          +0.00070708789711327648,
                                                          +0.00070676992408055393,
                                                          +0.00070645188416917004,
                                                          +0.00070613448261430689,
                                                          +0.00070581701436100844,
                                                          +0.00070550018274475386,
                                                          +0.00070518328460964278,
                                                          +0.00070486702139826811,
                                                          +0.00070455069184697237,
                                                          +0.00070423499551224705,
                                                          +0.00070391923301589489,
                                                          +0.00070360410203506240,
                                                          +0.00070328890507025891,
                                                          +0.00070297433792601159,
                                                          +0.00070265970497481345,
                                                          +0.00070234570015526856,
                                                          +0.00070203162970516007,
                                                          +0.00070171818570383528,
                                                          +0.00070140467624770441,
                                                          +0.00070109179156349350,
                                                          +0.00070077884159960738,
                                                          +0.00070046651473675676,
                                                          +0.00070015412276873748,
                                                          +0.00069984235223682217,
                                                          +0.00069953051677362321,
                                                          +0.00069921930108752324,
                                                          +0.00069890802064340618,
                                                          +0.00069859735832328274,
                                                          +0.00069828663141779262,
                                                          +0.00069797652098906493,
                                                          +0.00069766634614700754,
                                                          +0.00069735678614032926,
                                                          +0.00069704716189174806,
                                                          +0.00069673815084298407,
                                                          +0.00069642907572313635,
                                                          +0.00069612061217333974,
                                                          +0.00069581208472267329,
                                                          +0.00069550416721806275,
                                                          +0.00069519618598219376,
                                                          +0.00069488881307413022,
                                                          +0.00069458137660382007,
                                                          +0.00069427454684878396,
                                                          +0.00069396765369991612,
                                                          +0.00069366136565948548,
                                                          +0.00069335501439304317,
                                                          +0.00069304926663387020,
                                                          +0.00069274345581591410,
                                                          +0.00069243824690970392,
                                                          +0.00069213297511134945,
                                                          +0.00069182830363483637,
                                                          +0.00069152356943223193,
                                                          +0.00069121943396715862,
                                                          +0.00069091523594146272,
                                                          +0.00069061163507455788,
                                                          +0.00069030797181191754,
                                                          +0.00069000490413487410,
                                                          +0.00068970177422640296,
                                                          +0.00068939923833585565,
                                                          +0.00068909664037761220,
                                                          +0.00068879463487511641,
                                                          +0.00068849256746808237,
                                                          +0.00068819109096009275,
                                                          +0.00068788955271015142,
                                                          +0.00068758860380799974,
                                                          +0.00068728759332591408,
                                                          +0.00068698717064578918,
                                                          +0.00068668668654718146,
                                                          +0.00068638678871010654,
                                                          +0.00068608682961543628,
                                                          +0.00068578745524724906,
                                                          +0.00068548801978179235,
                                                          +0.00068518916751312306,
                                                          +0.00068489025430695100,
                                                          +0.00068459192277320249,
                                                          +0.00068429353046116153,
                                                          +0.00068399571830248744,
                                                          +0.00068369784552417729,
                                                          +0.00068340055138546216,
                                                          +0.00068310319678521563,
                                                          +0.00068280641931605431,
                                                          +0.00068250958154291678,
                                                          +0.00068221331939759375,
                                                          +0.00068191699710530248,
                                                          +0.00068162124894277144,
                                                          +0.00068132544078973586,
                                                          +0.00068103020527359945,
                                                          +0.00068073490992287978,
                                                          +0.00068044018572137018,
                                                          +0.00068014540184065878,
                                                          +0.00067985118762661659,
                                                          +0.00067955691388821651,
                                                          +0.00067926320833907130,
                                                          +0.00067896944341987707,
                                                          +0.00067867624521762823,
                                                          +0.00067838298779910602,
                                                          +0.00067809029563030182,
                                                          +0.00067779754439846950,
                                                          +0.00067750535695418833,
                                                          +0.00067721311059959607,
                                                          +0.00067692142657542660,
                                                          +0.00067662968379313701,
                                                          +0.00067633850188915921,
                                                          +0.00067604726137872838,
                                                          +0.00067575658029949326,
                                                          +0.00067546584076495092,
                                                          +0.00067517565921946263,
                                                          +0.00067488541936929294,
                                                          +0.00067459573607098872,
                                                          +0.00067430599461811231,
                                                          +0.00067401680828484318,
                                                          +0.00067372756394659629,
                                                          +0.00067343887330060940,
                                                          +0.00067315012479872644,
                                                          +0.00067286192856664509,
                                                          +0.00067257367462723893,
                                                          +0.00067228597154004449,
                                                          +0.00067199821089358811,
                                                          +0.00067171099968660103,
                                                          +0.00067142373106790899,
                                                          +0.00067113701048077037,
                                                          +0.00067085023262898006,
                                                          +0.00067056400140563333,
                                                          +0.00067027771306418620,
                                                          +0.00066999196995285891,
                                                          +0.00066970616986948258,
                                                          +0.00066942091362266786,
                                                          +0.00066913560054935814,
                                                          +0.00066885082992379637,
                                                          +0.00066856600261679827,
                                                          +0.00066828171637345964,
                                                          +0.00066799737359324990,
                                                          +0.00066771357049731596,
                                                          +0.00066742971100858500,
                                                          +0.00066714638982943064,
                                                          +0.00066686301240106462,
                                                          +0.00066658017191224080,
                                                          +0.00066629727531730329,
                                                          +0.00066601491429651939,
                                                          +0.00066573249731223462,
                                                          +0.00066545061454134006,
                                                          +0.00066516867594907487,
                                                          +0.00066488727021404186,
                                                          +0.00066460580879928771,
                                                          +0.00066432487889019499,
                                                          +0.00066404389344255107,
                                                          +0.00066376343815356491,
                                                          +0.00066348292746672101,
                                                          +0.00066320294559607854,
                                                          +0.00066292290846779723,
                                                          +0.00066264339881778964,
                                                          +0.00066236383404988906,
                                                          +0.00066208479542684447,
                                                          +0.00066180570182518196,
                                                          +0.00066152713303944770,
                                                          +0.00066124850941390182,
                                                          +0.00066097040927982865,
                                                          +0.00066069225444428318,
                                                          +0.00066041462178020741,
                                                          +0.00066013693455253375,
                                                          +0.00065985976818076113,
                                                          +0.00065958254738280220,
                                                          +0.00065930584612959120,
                                                          +0.00065902909058714423,
                                                          +0.00065875285328268948,
                                                          +0.00065847656182549006,
                                                          +0.00065820078730390507,
                                                          +0.00065792495876560982,
                                                          +0.00065764964586491218,
                                                          +0.00065737427908308310,
                                                          +0.00065709942664517673,
                                                          +0.00065682452046126477,
                                                          +0.00065655012733192397,
                                                          +0.00065627568059125176,
                                                          +0.00065600174562010615,
                                                          +0.00065572775717185358,
                                                          +0.00065545427921237032,
                                                          +0.00065518074790955688,
                                                          +0.00065490772581902590,
                                                          +0.00065463465051849592,
                                                          +0.00065436208315801337,
                                                          +0.00065408946272041901,
                                                          +0.00065381734895487184,
                                                          +0.00065354518224465790,
                                                          +0.00065327352094270814,
                                                          +0.00065300180682809615,
                                                          +0.00065273059686216442,
                                                          +0.00065245933421513678,
                                                          +0.00065218857446138802,
                                                          +0.00065191776215767271,
                                                          +0.00065164745149599943,
                                                          +0.00065137708841505422,
                                                          +0.00065110722572906169,
                                                          +0.00065083731075405913,
                                                          +0.00065056789493104974,
                                                          +0.00065029842694886060,
                                                          +0.00065002945687981913,
                                                          +0.00064976043478099873,
                                                          +0.00064949190936057608,
                                                          +0.00064922333203934783,
                                                          +0.00064895525016584702,
                                                          +0.00064868711652008796,
                                                          +0.00064841947709544812,
                                                          +0.00064815178602667357,
                                                          +0.00064788458795645570,
                                                          +0.00064761733836980371,
                                                          +0.00064735058056317554,
                                                          +0.00064708377136739326,
                                                          +0.00064681745273711369,
                                                          +0.00064655108284454165,
                                                          +0.00064628520230694715,
                                                          +0.00064601927063350374,
                                                          +0.00064575382710849324,
                                                          +0.00064548833257366224,
                                                          +0.00064522332498468163,
                                                          +0.00064495826651149533,
                                                          +0.00064469369378552415,
                                                          +0.00064442907030054964,
                                                          +0.00064416493136808597,
                                                          +0.00064390074180141130,
                                                          +0.00064363703559645749,
                                                          +0.00064337327888167582,
                                                          +0.00064311000434172415,
                                                          +0.00064284667941592107,
                                                          +0.00064258383548193901,
                                                          +0.00064232094128567662,
                                                          +0.00064205852690209383,
                                                          +0.00064179606237939758,
                                                          +0.00064153407649409060,
                                                          +0.00064127204059243525,
                                                          +0.00064101048215671366,
                                                          +0.00064074887382700838,
                                                          +0.00064048774179560131,
                                                          +0.00064022655999217625,
                                                          +0.00063996585332321812,
                                                          +0.00063970509700381054,
                                                          +0.00063944481465882691,
                                                          +0.00063918448278456699,
                                                          +0.00063892462372846148,
                                                          +0.00063866471526385861,
                                                          +0.00063840527846489880,
                                                          +0.00063814579237782829,
                                                          +0.00063788677680763173,
                                                          +0.00063762771206932016,
                                                          +0.00063736911670284144,
                                                          +0.00063711047228785442,
                                                          +0.00063685229610337092,
                                                          +0.00063659407098959841,
                                                          +0.00063633631296869755,
                                                          +0.00063607850613734047,
                                                          +0.00063582116526490621,
                                                          +0.00063556377570046427,
                                                          +0.00063530685096466297,
                                                          +0.00063504987765491960,
                                                          +0.00063479336804718803,
                                                          +0.00063453680998319824,
                                                          +0.00063428071449822933,
                                                          +0.00063402457067430649,
                                                          +0.00063376888831003658,
                                                          +0.00063351315772373874,
                                                          +0.00063325788748133476,
                                                          +0.00063300256913345187,
                                                          +0.00063274771001729780,
                                                          +0.00063249280291183870,
                                                          +0.00063223835392952341,
                                                          +0.00063198385707370305,
                                                          +0.00063172981723600612,
                                                          +0.00063147572964023207,
                                                          +0.00063122209796111261,
                                                          +0.00063096841863897286,
                                                          +0.00063071519413555540,
                                                          +0.00063046192210380566,
                                                          +0.00063020910379636816,
                                                          +0.00062995623807491827,
                                                          +0.00062970382498687939,
                                                          +0.00062945136459878167,
                                                          +0.00062919935575668810,
                                                          +0.00062894729972812430,
                                                          +0.00062869569416163786,
                                                          +0.00062844404152190660,
                                                          +0.00062819283826379287,
                                                          +0.00062794158804529748,
                                                          +0.00062769078613141200,
                                                          +0.00062743993736964776,
                                                          +0.00062718953583892510,
                                                          +0.00062693908757246715,
                                                          +0.00062668908546690736,
                                                          +0.00062643903673739840,
                                                          +0.00062618943310205561,
                                                          +0.00062593978295419361,
                                                          +0.00062569057683716357,
                                                          +0.00062544132431868922,
                                                          +0.00062519251477109771,
                                                          +0.00062494365893278354,
                                                          +0.00062469524500877291,
                                                          +0.00062444678490440955,
                                                          +0.00062419876566112864,
                                                          +0.00062395070034751410,
                                                          +0.00062370307484510502,
                                                          +0.00062345540338203218,
                                                          +0.00062320817068361881,
                                                          +0.00062296089213386384,
                                                          +0.00062271405130553991,
                                                          +0.00062246716473485003,
                                                          +0.00062222071484566784,
                                                          +0.00062197421932275014,
                                                          +0.00062172815944470808,
                                                          +0.00062148205404121716,
                                                          +0.00062123638324924858,
                                                          +0.00062099066703977593,
                                                          +0.00062074538441173705,
                                                          +0.00062050005647379771,
                                                          +0.00062025516109045738,
                                                          +0.00062001022050448007,
                                                          +0.00061976571144950661,
                                                          +0.00061952115729882180,
                                                          +0.00061927703365877217,
                                                          +0.00061903286502960019,
                                                          +0.00061878912589390924,
                                                          +0.00061854534187534885,
                                                          +0.00061830198633631745,
                                                          +0.00061805858602033534,
                                                          +0.00061781561317311912,
                                                          +0.00061757259565453778,
                                                          +0.00061733000459713585,
                                                          +0.00061708736897362281,
                                                          +0.00061684515880686716,
                                                          +0.00061660290417892374,
                                                          +0.00061636107400646692,
                                                          +0.00061611919947741694,
                                                          +0.00061587774840572262,
                                                          +0.00061563625308170167,
                                                          +0.00061539518022003239,
                                                          +0.00061515406320997645,
                                                          +0.00061491336767038299,
                                                          +0.00061467262808601768,
                                                          +0.00061443230898332855,
                                                          +0.00061419194593915821,
                                                          +0.00061395200239096825,
                                                          +0.00061371201500426506,
                                                          +0.00061347244613092495,
                                                          +0.00061323283352171835,
                                                          +0.00061299363844632352,
                                                          +0.00061275439973738856,
                                                          +0.00061251557758576947,
                                                          +0.00061227671190261762,
                                                          +0.00061203826180332747,
                                                          +0.00061179976827419459,
                                                          +0.00061156168935850056,
                                                          +0.00061132356711433692,
                                                          +0.00061108585851620803,
                                                          +0.00061084810669066758,
                                                          +0.00061061076754676533,
                                                          +0.00061037338527619549,
                                                          +0.00061013641472586254,
                                                          +0.00060989940114929337,
                                                          +0.00060966279833454384,
                                                          +0.00060942615259367765,
                                                          +0.00060918991665918660,
                                                          +0.00060895363789838816,
                                                          +0.00060871776799148068,
                                                          +0.00060848185535776586,
                                                          +0.00060824635062840782,
                                                          +0.00060801080327143426,
                                                          +0.00060777566287222137,
                                                          +0.00060754047994427795,
                                                          +0.00060730570303042558,
                                                          +0.00060707088368642224,
                                                          +0.00060683646941575575,
                                                          +0.00060660201281321330,
                                                          +0.00060636796034615765,
                                                          +0.00060613386564519812,
                                                          +0.00060590017414476776,
                                                          +0.00060566644050810323,
                                                          +0.00060543310913989260,
                                                          +0.00060519973573281696,
                                                          +0.00060496676366499015,
                                                          +0.00060473374965536717,
                                                          +0.00060450113605864809,
                                                          +0.00060426848061690341,
                                                          +0.00060403622466456610,
                                                          +0.00060380392696367604,
                                                          +0.00060357202783153463,
                                                          +0.00060334008704701733,
                                                          +0.00060310854391341657,
                                                          +0.00060287695922332124,
                                                          +0.00060264577126912643,
                                                          +0.00060241454185402477,
                                                          +0.00060218370826261268,
                                                          +0.00060195283330558794,
                                                          +0.00060172235326283680,
                                                          +0.00060149183194947568,
                                                          +0.00060126170464375531,
                                                          +0.00060103153616213696,
                                                          +0.00060080176078429980,
                                                          +0.00060057194432498707,
                                                          +0.00060034252006835852,
                                                          +0.00060011305482438829,
                                                          +0.00059988398088475737,
                                                          +0.00059965486605163116,
                                                          +0.00059942614162724066,
                                                          +0.00059919737640291534,
                                                          +0.00059896900069445294,
                                                          +0.00059874058427933060,
                                                          +0.00059851255648991951,
                                                          +0.00059828448808683943,
                                                          +0.00059805680742202878,
                                                          +0.00059782908623625720,
                                                          +0.00059760175190401324,
                                                          +0.00059737437714323430,
                                                          +0.00059714738835393096,
                                                          +0.00059692035922823757,
                                                          +0.00059669371519464756,
                                                          +0.00059646703091653284,
                                                          +0.00059624073085381795,
                                                          +0.00059601439063816490,
                                                          +0.00059578843376386769,
                                                          +0.00059556243682794112,
                                                          +0.00059533682236197552,
                                                          +0.00059511116792541289,
                                                          +0.00059488589509005521,
                                                          +0.00059466058237485711,
                                                          +0.00059443565039473734,
                                                          +0.00059421067862525914,
                                                          +0.00059398608672735155,
                                                          +0.00059376145513029418,
                                                          +0.00059353720254390926,
                                                          +0.00059331291034831043,
                                                          +0.00059308899630508528,
                                                          +0.00059286504274231092,
                                                          +0.00059264146647620064,
                                                          +0.00059241785077993546,
                                                          +0.00059219461152720517,
                                                          +0.00059197133293344465,
                                                          +0.00059174842993265969,
                                                          +0.00059152548767970106,
                                                          +0.00059130292017171913,
                                                          +0.00059108031350015228,
                                                          +0.00059085808072811450,
                                                          +0.00059063580888081427,
                                                          +0.00059041391009013653,
                                                          +0.00059019197231225361,
                                                          +0.00058997040675061799,
                                                          +0.00058974880228957014,
                                                          +0.00058952756920691702,
                                                          +0.00058930629731238104,
                                                          +0.00058908539596089985,
                                                          +0.00058886445588480281,
                                                          +0.00058864388551892394,
                                                          +0.00058842327651543460,
                                                          +0.00058820303639182145,
                                                          +0.00058798275771734299,
                                                          +0.00058776284709488190,
                                                          +0.00058754289800804155,
                                                          +0.00058732331614783648,
                                                          +0.00058710369590947897,
                                                          +0.00058688444207484036,
                                                          +0.00058666514994801895,
                                                          +0.00058644622340445684,
                                                          +0.00058622725865442517,
                                                          +0.00058600865866964091,
                                                          +0.00058579002056384445,
                                                          +0.00058557174640772253,
                                                          +0.00058535343421579119,
                                                          +0.00058513548516039068,
                                                          +0.00058491749815412961,
                                                          +0.00058469987347367668,
                                                          +0.00058448221092705945,
                                                          +0.00058426490989793860,
                                                          +0.00058404757108709794,
                                                          +0.00058383059298784483,
                                                          +0.00058361357719106531,
                                                          +0.00058339692130235810,
                                                          +0.00058318022780006807,
                                                          +0.00058296389340471981,
                                                          +0.00058274752147948307,
                                                          +0.00058253150786243349,
                                                          +0.00058231545679894201,
                                                          +0.00058209976324725000,
                                                          +0.00058188403233231528,
                                                          +0.00058166865813515101,
                                                          +0.00058145324665769688,
                                                          +0.00058123819110633343,
                                                          +0.00058102309835738795,
                                                          +0.00058080836074519439,
                                                          +0.00058059358601788159,
                                                          +0.00058037916564031484,
                                                          +0.00058016470822984823,
                                                          +0.00057995060438444514,
                                                          +0.00057973646358811892,
                                                          +0.00057952267557448856,
                                                          +0.00057930885069167016,
                                                          +0.00057909537781148691,
                                                          +0.00057888186814360990,
                                                          +0.00057866870970060499,
                                                          +0.00057845551455116011,
                                                          +0.00057824266985111487,
                                                          +0.00057802978852564450,
                                                          +0.00057781725687638171,
                                                          +0.00057760468868247054,
                                                          +0.00057739246939384804,
                                                          +0.00057718021364111690,
                                                          +0.00057696830602501910,
                                                          +0.00057675636202511571,
                                                          +0.00057654476539544788,
                                                          +0.00057633313246204150,
                                                          +0.00057612184613471987,
                                                          +0.00057591052358349243,
                                                          +0.00057569954687643904,
                                                          +0.00057548853402507837,
                                                          +0.00057527786625821253,
                                                          +0.00057506716242640519,
                                                          +0.00057485680292163640,
                                                          +0.00057464640743105932,
                                                          +0.00057443635551228042,
                                                          +0.00057422626768659517,
                                                          +0.00057401652267967417,
                                                          +0.00057380674184451849,
                                                          +0.00057359730307729245,
                                                          +0.00057338782856027374,
                                                          +0.00057317869536254091,
                                                          +0.00057296952649322824,
                                                          +0.00057276069819674116,
                                                          +0.00057255183430665927,
                                                          +0.00057234331024511736,
                                                          +0.00057213475066773866,
                                                          +0.00057192653017678144,
                                                          +0.00057171827424751915,
                                                          +0.00057151035666471914,
                                                          +0.00057130240372092029,
                                                          +0.00057109478838577601,
                                                          +0.00057088713776671412,
                                                          +0.00057067982402064300,
                                                          +0.00057047247506751195,
                                                          +0.00057026546225384298,
                                                          +0.00057005841430974886,
                                                          +0.00056985170177371674,
                                                          +0.00056964495418367143,
                                                          +0.00056943854127240875,
                                                          +0.00056923209338332378,
                                                          +0.00056902597944585398,
                                                          +0.00056881983060653204,
                                                          +0.00056861401499376366,
                                                          +0.00056840816455489332,
                                                          +0.00056820264661961199,
                                                          +0.00056799709393375996,
                                                          +0.00056779187303062208,
                                                          +0.00056758661745222663,
                                                          +0.00056738169293775323,
                                                          +0.00056717673382311758,
                                                          +0.00056697210505568659,
                                                          +0.00056676744176297205,
                                                          +0.00056656310810281193,
                                                          +0.00056635873999203140,
                                                          +0.00056615470080121494,
                                                          +0.00056595062723422567,
                                                          +0.00056574688187666273,
                                                          +0.00056554310221716057,
                                                          +0.00056533965005859169,
                                                          +0.00056513616367210358,
                                                          +0.00056493300408009389,
                                                          +0.00056472981033397194,
                                                          +0.00056452694267790360,
                                                          +0.00056432404094131810,
                                                          +0.00056412146459238475,
                                                          +0.00056391885423631770,
                                                          +0.00056371656856751752,
                                                          +0.00056351424896475667,
                                                          +0.00056331225335088558,
                                                          +0.00056311022387601736,
                                                          +0.00056290851769366310,
                                                          +0.00056270677772306623,
                                                          +0.00056250536035060204,
                                                          +0.00056230390926244116,
                                                          +0.00056210278008001890,
                                                          +0.00056190161725423867,
                                                          +0.00056170077564378254,
                                                          +0.00056149990046210042,
                                                          +0.00056129934580730133,
                                                          +0.00056109875765320179,
                                                          +0.00056089848933950999,
                                                          +0.00056069818759823815,
                                                          +0.00056049820501285791,
                                                          +0.00056029818907141335,
                                                          +0.00056009849160329570,
                                                          +0.00055989876085042571,
                                                          +0.00055969934789026356,
                                                          +0.00055949990171645777,
                                                          +0.00055930077265667821,
                                                          +0.00055910161045416139,
                                                          +0.00055890276468892081,
                                                          +0.00055870388585164759,
                                                          +0.00055850532277682463,
                                                          +0.00055830672670047286,
                                                          +0.00055810844571366246,
                                                          +0.00055791013179562704,
                                                          +0.00055771213229613523,
                                                          +0.00055751409993552237,
                                                          +0.00055731638132435900,
                                                          +0.00055711862992197938,
                                                          +0.00055692119160185292,
                                                          +0.00055672372056021692,
                                                          +0.00055652656193552763,
                                                          +0.00055632937065883841,
                                                          +0.00055613249113567324,
                                                          +0.00055593557902982097,
                                                          +0.00055573897801594683,
                                                          +0.00055554234448850241,
                                                          +0.00055534602139336111,
                                                          +0.00055514966585357064,
                                                          +0.00055495362008827203,
                                                          +0.00055475754194705105,
                                                          +0.00055456177292436759,
                                                          +0.00055436597159429452,
                                                          +0.00055417047872865556,
                                                          +0.00055397495362396641,
                                                          +0.00055377973633145220,
                                                          +0.00055358448686803480,
                                                          +0.00055338954456637024,
                                                          +0.00055319457016175767,
                                                          +0.00055299990227030739,
                                                          +0.00055280520234367270,
                                                          +0.00055261080828343501,
                                                          +0.00055241638225558571,
                                                          +0.00055222226144918608,
                                                          +0.00055202810874255823,
                                                          +0.00055183426061424456,
                                                          +0.00055164038065289652,
                                                          +0.00055144680462853311,
                                                          +0.00055125319683814043,
                                                          +0.00055105989234520230,
                                                          +0.00055086655615305211,
                                                          +0.00055067352262061888,
                                                          +0.00055048045745560378,
                                                          +0.00055028769431435481,
                                                          +0.00055009489960696780,
                                                          +0.00054990240628917625,
                                                          +0.00054970988147150452,
                                                          +0.00054951765741103199,
                                                          +0.00054932540191675143,
                                                          +0.00054913344654904204,
                                                          +0.00054894145981341212,
                                                          +0.00054874977257548738,
                                                          +0.00054855805403534574,
                                                          +0.00054836663436579839,
                                                          +0.00054817518345955454,
                                                          +0.00054798403079854389,
                                                          +0.00054779284696617424,
                                                          +0.00054760196075542031,
                                                          +0.00054741104343846271,
                                                          +0.00054722042312124080,
                                                          +0.00054702977176278929,
                                                          +0.00054683941678392435,
                                                          +0.00054664903082862293,
                                                          +0.00054645894063448479,
                                                          +0.00054626881952852320,
                                                          +0.00054607899356702051,
                                                          +0.00054588913675812753,
                                                          +0.00054569957447870307,
                                                          +0.00054550998141614281,
                                                          +0.00054532068226976745,
                                                          +0.00054513135240433230,
                                                          +0.00054494231584350020,
                                                          +0.00054475324862750685,
                                                          +0.00054456447410623028,
                                                          +0.00054437566899351352,
                                                          +0.00054418715596731698,
                                                          +0.00054399861241322513,
                                                          +0.00054381036033914105,
                                                          +0.00054362207780053046,
                                                          +0.00054343408613709288,
                                                          +0.00054324606407232237,
                                                          +0.00054305833227956271,
                                                          +0.00054287057014848928,
                                                          +0.00054268309768793063,
                                                          +0.00054249559495190300,
                                                          +0.00054230838128655498,
                                                          +0.00054212113740840975,
                                                          +0.00054193418200276387,
                                                          +0.00054174719644681929,
                                                          +0.00054156049876684332,
                                                          +0.00054137377099889488,
                                                          +0.00054118733051202806,
                                                          +0.00054100085999934347,
                                                          +0.00054081467617449089,
                                                          +0.00054062846238580448,
                                                          +0.00054044253469333293,
                                                          +0.00054025657709884085,
                                                          +0.00054007090501057323,
                                                          +0.00053988520308192816,
                                                          +0.00053969978607113910,
                                                          +0.00053951433928144587,
                                                          +0.00053932917682285536,
                                                          +0.00053914398464666603,
                                                          +0.00053895907621643549,
                                                          +0.00053877413812974332,
                                                          +0.00053858948320547099,
                                                          +0.00053840479868570636,
                                                          +0.00053822039674642169,
                                                          +0.00053803596527244723,
                                                          +0.00053785181579860575,
                                                          +0.00053766763685071065,
                                                          +0.00053748373932419069,
                                                          +0.00053729981238408696,
                                                          +0.00053711616628818236,
                                                          +0.00053693249083899884,
                                                          +0.00053674909565841597,
                                                          +0.00053656567118469433,
                                                          +0.00053638252640554704,
                                                          +0.00053619935239323656,
                                                          +0.00053601645750304028,
                                                          +0.00053583353343949309,
                                                          +0.00053565088792716120,
                                                          +0.00053546821330112773,
                                                          +0.00053528581665696637,
                                                          +0.00053510339095859014,
                                                          +0.00053492124267429347,
                                                          +0.00053473906539510665,
                                                          +0.00053455716496375214,
                                                          +0.00053437523559667099,
                                                          +0.00053419358251271479,
                                                          +0.00053401190055203424,
                                                          +0.00053383049431130641,
                                                          +0.00053364905925269647,
                                                          +0.00053346789935239630,
                                                          +0.00053328671069289635,
                                                          +0.00053310579663158793,
                                                          +0.00053292485386960289,
                                                          +0.00053274418514720999,
                                                          +0.00053256348778250504,
                                                          +0.00053238306390030702,
                                                          +0.00053220261143400343,
                                                          +0.00053202243189463046,
                                                          +0.00053184222382920065,
                                                          +0.00053166228813662916,
                                                          +0.00053148232397589256,
                                                          +0.00053130263163544056,
                                                          +0.00053112291088455862,
                                                          +0.00053094346140288137,
                                                          +0.00053076398356835335,
                                                          +0.00053058477645343872,
                                                          +0.00053040554104309690,
                                                          +0.00053022657580426090,
                                                          +0.00053004758232726634,
                                                          +0.00052986885847514857,
                                                          +0.00052969010644198639,
                                                          +0.00052951162348854582,
                                                          +0.00052933311241102113,
                                                          +0.00052915486986953122,
                                                          +0.00052897659926076407,
                                                          +0.00052879859664580896,
                                                          +0.00052862056602023041,
                                                          +0.00052844280284769984,
                                                          +0.00052826501172104751,
                                                          +0.00052808748750813282,
                                                          +0.00052790993539744637,
                                                          +0.00052773264966263613,
                                                          +0.00052755533608625255,
                                                          +0.00052737828834932814,
                                                          +0.00052720121282687812,
                                                          +0.00052702440260890939,
                                                          +0.00052684756466131249,
                                                          +0.00052667099148465352,
                                                          +0.00052649439063411356,
                                                          +0.00052631805402239844,
                                                          +0.00052614168979240043,
                                                          +0.00052596558927053827,
                                                          +0.00052578946118584258,
                                                          +0.00052561359628001427,
                                                          +0.00052543770386665335,
                                                          +0.00052526207410430676,
                                                          +0.00052508641688958049,
                                                          +0.00052491102179942639,
                                                          +0.00052473559931189831,
                                                          +0.00052456043842390556,
                                                          +0.00052438525019339783,
                                                          +0.00052421032303879048,
                                                          +0.00052403536859638037,
                                                          +0.00052386067470763223,
                                                          +0.00052368595358564813,
                                                          +0.00052351149249647898,
                                                          +0.00052333700422849529,
                                                          +0.00052316277547386685,
                                                          +0.00052298851959470033,
                                                          +0.00052281452271081270,
                                                          +0.00052264049875651861,
                                                          +0.00052246673328080497,
                                                          +0.00052229294078867292,
                                                          +0.00052211940625979602,
                                                          +0.00052194584476834523,
                                                          +0.00052177254072619341,
                                                          +0.00052159920977516933,
                                                          +0.00052142613576085261,
                                                          +0.00052125303489122278,
                                                          +0.00052108019044706788,
                                                          +0.00052090731920101678,
                                                          +0.00052073470387056483,
                                                          +0.00052056206179149187,
                                                          +0.00052038967511949233,
                                                          +0.00052021726175200615,
                                                          +0.00052004510328441413,
                                                          +0.00051987291817432910,
                                                          +0.00051970098745830111,
                                                          +0.00051952903015263356,
                                                          +0.00051935732673652338,
                                                          +0.00051918559678348767,
                                                          +0.00051901412021684238,
                                                          +0.00051884261716584614,
                                                          +0.00051867136699940272,
                                                          +0.00051850009040104372,
                                                          +0.00051832906618672439,
                                                          +0.00051815801559278684,
                                                          +0.00051798721688369527,
                                                          +0.00051781639184714495,
                                                          +0.00051764581819756293,
                                                          +0.00051747521827254412,
                                                          +0.00051730486923792665,
                                                          +0.00051713449397975807,
                                                          +0.00051696436911673085,
                                                          +0.00051679421808190095,
                                                          +0.00051662431694825587,
                                                          +0.00051645438969442046,
                                                          +0.00051628471184911112,
                                                          +0.00051611500793508850,
                                                          +0.00051594555293822729,
                                                          +0.00051577607192399488,
                                                          +0.00051560683933684899,
                                                          +0.00051543758078353866,
                                                          +0.00051526857016852658,
                                                          +0.00051509953363842260,
                                                          +0.00051493074455910895,
                                                          +0.00051476192961564258,
                                                          +0.00051459336163673611,
                                                          +0.00051442476784448231,
                                                          +0.00051425642053183152,
                                                          +0.00051408804745650574,
                                                          +0.00051391992037709458,
                                                          +0.00051375176758554852,
                                                          +0.00051358386030749329,
                                                          +0.00051341592736771097,
                                                          +0.00051324823946025664,
                                                          +0.00051308052594135152,
                                                          +0.00051291305697486779,
                                                          +0.00051274556244707806,
                                                          +0.00051257831199305593,
                                                          +0.00051241103602774171,
                                                          +0.00051224400365878944,
                                                          +0.00051207694582842858,
                                                          +0.00051191013111826845,
                                                          +0.00051174329099645376,
                                                          +0.00051157669351991806,
                                                          +0.00051141007068135195,
                                                          +0.00051124369001437993,
                                                          +0.00051107728403487242,
                                                          +0.00051091111975450629,
                                                          +0.00051074493021097090,
                                                          +0.00051057898189535195,
                                                          +0.00051041300836580201,
                                                          +0.00051024727559416774,
                                                          +0.00051008151765771271,
                                                          +0.00050991600001039296,
                                                          +0.00050975045724723520,
                                                          +0.00050958515430564834,
                                                          +0.00050941982629707926,
                                                          +0.00050925473764372919,
                                                          +0.00050908962397212582,
                                                          +0.00050892474919059813,
                                                          +0.00050875984943942007,
                                                          +0.00050859518811437806,
                                                          +0.00050843050186816284,
                                                          +0.00050826605358534553,
                                                          +0.00050810158042970631,
                                                          +0.00050793734477592306,
                                                          +0.00050777308429754474,
                                                          +0.00050760906086067307,
                                                          +0.00050744501264730847,
                                                          +0.00050728120101629036,
                                                          +0.00050711736465675712,
                                                          +0.00050695376442159556,
                                                          +0.00050679013950577273,
                                                          +0.00050662675025752780,
                                                          +0.00050646333637635241,
                                                          +0.00050630015770713858,
                                                          +0.00050613695445260172,
                                                          +0.00050597398595558427,
                                                          +0.00050581099292072880,
                                                          +0.00050564823419012002,
                                                          +0.00050548545096903616,
                                                          +0.00050532290160009243,
                                                          +0.00050516032778791470,
                                                          +0.00050499798737693312,
                                                          +0.00050483562256983707,
                                                          +0.00050467349071415179,
                                                          +0.00050451133450935069,
                                                          +0.00050434941080732996,
                                                          +0.00050418746280307127,
                                                          +0.00050402574685411389,
                                                          +0.00050386400664967628,
                                                          +0.00050370249805420830,
                                                          +0.00050354096524989783,
                                                          +0.00050337966360936937,
                                                          +0.00050321833780651685,
                                                          +0.00050305724272339889,
                                                          +0.00050289612352435575,
                                                          +0.00050273523460213658,
                                                          +0.00050257432161027269,
                                                          +0.00050241363845345477,
                                                          +0.00050225293127315442,
                                                          +0.00050209245348725121,
                                                          +0.00050193195172390992,
                                                          +0.00050177167891544278,
                                                          +0.00050161138217546402,
                                                          +0.00050145131395195892,
                                                          +0.00050129122184275126,
                                                          +0.00050113135781273542,
                                                          +0.00050097146994270954,
                                                          +0.00050081180971570777,
                                                          +0.00050065212569427213,
                                                          +0.00050049266888080508,
                                                          +0.00050033318831836421,
                                                          +0.00050017393452994369,
                                                          +0.00050001465703789329,
                                                          +0.00049985560588702025,
                                                          +0.00049969653107774623,
                                                          +0.00049953768217790660,
                                                          +0.00049937880966477997,
                                                          +0.00049922016263044219,
                                                          +0.00049906149202781679,
                                                          +0.00049890304647442888,
                                                          +0.00049874457739763835,
                                                          +0.00049858633294162400,
                                                          +0.00049842806500697819,
                                                          +0.00049827002126573471,
                                                          +0.00049811195409051696,
                                                          +0.00049795411068241115,
                                                          +0.00049779624388487540,
                                                          +0.00049763860042924095,
                                                          +0.00049748093362860803,
                                                          +0.00049732348974574291,
                                                          +0.00049716602256219804,
                                                          +0.00049700877787336030,
                                                          +0.00049685150992805007,
                                                          +0.00049669446405545584,
                                                          +0.00049653739497048451,
                                                          +0.00049638054753730471,
                                                          +0.00049622367693573160,
                                                          +0.00049606702756608905,
                                                          +0.00049591035507192565,
                                                          +0.00049575390339089164,
                                                          +0.00049559742862909854,
                                                          +0.00049544117426269000,
                                                          +0.00049528489685917419,
                                                          +0.00049512883943435054,
                                                          +0.00049497275901596151,
                                                          +0.00049481689816062240,
                                                          +0.00049466101435514996,
                                                          +0.00049450534969813179,
                                                          +0.00049434966213430284,
                                                          +0.00049419419330537582,
                                                          +0.00049403870161285140,
                                                          +0.00049388342824271665,
                                                          +0.00049372813205208956,
                                                          +0.00049357305377237585,
                                                          +0.00049341795271516664,
                                                          +0.00049326306915842809,
                                                          +0.00049310816286708312,
                                                          +0.00049295347366679629,
                                                          +0.00049279876177468404,
                                                          +0.00049264426656524458,
                                                          +0.00049248974870665368,
                                                          +0.00049233544712337337,
                                                          +0.00049218112293350860,
                                                          +0.00049202701461261323,
                                                          +0.00049187288372759389,
                                                          +0.00049171896830621964,
                                                          +0.00049156503036307535,
                                                          +0.00049141130747926642,
                                                          +0.00049125756211593566,
                                                          +0.00049110403140864090,
                                                          +0.00049095047826396704,
                                                          +0.00049079713937303781,
                                                          +0.00049064377808676681,
                                                          +0.00049049063065295380,
                                                          +0.00049033746086573151,
                                                          +0.00049018450453068162,
                                                          +0.00049003152588405032,
                                                          +0.00048987876029030490,
                                                          +0.00048972597242670133,
                                                          +0.00048957339721769188,
                                                          +0.00048942079978044398,
                                                          +0.00048926841460049072,
                                                          +0.00048911600723381484,
                                                          +0.00048896381172812344,
                                                          +0.00048881159407712183,
                                                          +0.00048865958789178019,
                                                          +0.00048850755960243807,
                                                          +0.00048835574238441457,
                                                          +0.00048820390310359726,
                                                          +0.00048805227450073702,
                                                          +0.00048790062387618768,
                                                          +0.00048774918353721060,
                                                          +0.00048759772121754672,
                                                          +0.00048744646879204446,
                                                          +0.00048729519442675616,
                                                          +0.00048714412956518910,
                                                          +0.00048699304280463545,
                                                          +0.00048684216515833083,
                                                          +0.00048669126565373776,
                                                          +0.00048654057487488618,
                                                          +0.00048638986227834361,
                                                          +0.00048623935801999675,
                                                          +0.00048608883198445586,
                                                          +0.00048593851390052380,
                                                          +0.00048578817407979460,
                                                          +0.00048563804182504315,
                                                          +0.00048548788787379130,
                                                          +0.00048533794110383935,
                                                          +0.00048518797267758441,
                                                          +0.00048503821104890098,
                                                          +0.00048488842780401275,
                                                          +0.00048473885097391511,
                                                          +0.00048458925256761185,
                                                          +0.00048443986019426193,
                                                          +0.00048429044628460721,
                                                          +0.00048414123802701010,
                                                          +0.00048399200827291039,
                                                          +0.00048384298379091072,
                                                          +0.00048369393785211264,
                                                          +0.00048354509680639281,
                                                          +0.00048339623434348096,
                                                          +0.00048324757639555787,
                                                          +0.00048309889706995201,
                                                          +0.00048295042188217481,
                                                          +0.00048280192535612700,
                                                          +0.00048265363259167478,
                                                          +0.00048250531852826717,
                                                          +0.00048235720785114622,
                                                          +0.00048220907591428841,
                                                          +0.00048206114698932939,
                                                          +0.00048191319684375597,
                                                          +0.00048176544933661175,
                                                          +0.00048161768064787974,
                                                          +0.00048147011422502268,
                                                          +0.00048132252665950877,
                                                          +0.00048117514098822855,
                                                          +0.00048102773421312685,
                                                          +0.00048088052896152756,
                                                          +0.00048073330264484718,
                                                          +0.00048058627748184476,
                                                          +0.00048043923129240703,
                                                          +0.00048029238588772737,
                                                          +0.00048014551949516354,
                                                          +0.00047999885351933950,
                                                          +0.00047985216659408836,
                                                          +0.00047970567971845733,
                                                          +0.00047955917193176229,
                                                          +0.00047941286382846408,
                                                          +0.00047926653485237138,
                                                          +0.00047912040519434540,
                                                          +0.00047897425470170099,
                                                          +0.00047882830316268397,
                                                          +0.00047868233082713177,
                                                          +0.00047853655708165549,
                                                          +0.00047839076257763437,
                                                          +0.00047824516630102262,
                                                          +0.00047809954930376415,
                                                          +0.00047795413017213148,
                                                          +0.00047780869035765779,
                                                          +0.00047766344804790577,
                                                          +0.00047751818509302659,
                                                          +0.00047737311928284257,
                                                          +0.00047722803286515348,
                                                          +0.00047708314323300716,
                                                          +0.00047693823303088663,
                                                          +0.00047679351925602879,
                                                          +0.00047664878494863645,
                                                          +0.00047650424671109534,
                                                          +0.00047635968797836876,
                                                          +0.00047621532495894912,
                                                          +0.00047607094148160230,
                                                          +0.00047592675336188177,
                                                          +0.00047578254482140225,
                                                          +0.00047563853128372975,
                                                          +0.00047549449736237641,
                                                          +0.00047535065808986974,
                                                          +0.00047520679847067054,
                                                          +0.00047506313314721359,
                                                          +0.00047491944751396325,
                                                          +0.00047477595582420427,
                                                          +0.00047463244386146170,
                                                          +0.00047448912549081081,
                                                          +0.00047434578688389722,
                                                          +0.00047420264151852392,
                                                          +0.00047405947595352028,
                                                          +0.00047391650328035110,
                                                          +0.00047377351044409555,
                                                          +0.00047363071015081240,
                                                          +0.00047348788973089874,
                                                          +0.00047334526150593579,
                                                          +0.00047320261319071038,
                                                          +0.00047306015672325246,
                                                          +0.00047291768020181263,
                                                          +0.00047277539518179244,
                                                          +0.00047263309014398350,
                                                          +0.00047249097626208013,
                                                          +0.00047234884239849383,
                                                          +0.00047220689934612944,
                                                          +0.00047206493634810150,
                                                          +0.00047192316381743983,
                                                          +0.00047178137137704755,
                                                          +0.00047163976906099180,
                                                          +0.00047149814687105177,
                                                          +0.00047135671446324180,
                                                          +0.00047121526221730823,
                                                          +0.00047107399941211862,
                                                          +0.00047093271680448022,
                                                          +0.00047079162329701852,
                                                          +0.00047065051002269739,
                                                          +0.00047050958550880093,
                                                          +0.00047036864126354924,
                                                          +0.00047022788543978403,
                                                          +0.00047008710992008280,
                                                          +0.00046994652248374063,
                                                          +0.00046980591538679675,
                                                          +0.00046966549603589354,
                                                          +0.00046952505705963825,
                                                          +0.00046938480549291102,
                                                          +0.00046924453433599760,
                                                          +0.00046910445025290324,
                                                          +0.00046896434661470454,
                                                          +0.00046882442971541728,
                                                          +0.00046868449329602358,
                                                          +0.00046854474328143298,
                                                          +0.00046840497378165049,
                                                          +0.00046826539035335895,
                                                          +0.00046812578747470676,
                                                          +0.00046798637033502804,
                                                          +0.00046784693377973692,
                                                          +0.00046770768263169313,
                                                          +0.00046756841210270262,
                                                          +0.00046742932665002295,
                                                          +0.00046729022185097984,
                                                          +0.00046715130179809792,
                                                          +0.00046701236243335361,
                                                          +0.00046687360748540592,
                                                          +0.00046673483326001481,
                                                          +0.00046659624312283816,
                                                          +0.00046645763374255522,
                                                          +0.00046631920812268475,
                                                          +0.00046618076329396369,
                                                          +0.00046604250189863095,
                                                          +0.00046590422132862194,
                                                          +0.00046576612386575256,
                                                          +0.00046562800726230056,
                                                          +0.00046549007344051261,
                                                          +0.00046535212051215449,
                                                          +0.00046521435004075607,
                                                          +0.00046507656049671954,
                                                          +0.00046493895308570690,
                                                          +0.00046480132663590776,
                                                          +0.00046466388199596295,
                                                          +0.00046452641835100379,
                                                          +0.00046438913619349312,
                                                          +0.00046425183506466010,
                                                          +0.00046411471510163205,
                                                          +0.00046397757620089393,
                                                          +0.00046384061814507688,
                                                          +0.00046370364118508266,
                                                          +0.00046356684474988295,
                                                          +0.00046343002944395978,
                                                          +0.00046329339434345973,
                                                          +0.00046315674040561144,
                                                          +0.00046302026635456733,
                                                          +0.00046288377349947127,
                                                          +0.00046274746021331148,
                                                          +0.00046261112815631779,
                                                          +0.00046247497535114083,
                                                          +0.00046233880380826964,
                                                          +0.00046220281120084227,
                                                          +0.00046206679988878224,
                                                          +0.00046193096719653683,
                                                          +0.00046179511583264266,
                                                          +0.00046165944277367607,
                                                          +0.00046152375107596735,
                                                          +0.00046138823736903851,
                                                          +0.00046125270505619674,
                                                          +0.00046111735042072503,
                                                          +0.00046098197721209256,
                                                          +0.00046084678136815551,
                                                          +0.00046071156698373295,
                                                          +0.00046057652965206464,
                                                          +0.00046044147381250948,
                                                          +0.00046030659471449830,
                                                          +0.00046017169714112260,
                                                          +0.00046003697599880959,
                                                          +0.00045990223641357821,
                                                          +0.00045976767294965507,
                                                          +0.00045963309107518378,
                                                          +0.00045949868501299091,
                                                          +0.00045936426057254447,
                                                          +0.00045923001163606904,
                                                          +0.00045909574435355912,
                                                          +0.00045896165226743350,
                                                          +0.00045882754186741707,
                                                          +0.00045869360635691659,
                                                          +0.00045855965256459409,
                                                          +0.00045842587335563509,
                                                          +0.00045829207589684806,
                                                          +0.00045815845271598651,
                                                          +0.00045802481131721628,
                                                          +0.00045789134389164545,
                                                          +0.00045775785828001074,
                                                          +0.00045762454633755967,
                                                          +0.00045749121624081532,
                                                          +0.00045735805950994661,
                                                          +0.00045722488465648148,
                                                          +0.00045709188286628964,
                                                          +0.00045695886298512453,
                                                          +0.00045682601586533446,
                                                          +0.00045669315068612063,
                                                          +0.00045656045796708500,
                                                          +0.00045642774722010179,
                                                          +0.00045629520863280028,
                                                          +0.00045616265204895408,
                                                          +0.00045603026732499060,
                                                          +0.00045589786463581258,
                                                          +0.00045576563350741386,
                                                          +0.00045563338444505808,
                                                          +0.00045550130664507167,
                                                          +0.00045536921094231359,
                                                          +0.00045523728620420665,
                                                          +0.00045510534359444080,
                                                          +0.00045497357165229772,
                                                          +0.00045484178186953639,
                                                          +0.00045471016245805709,
                                                          +0.00045457852523692841,
                                                          +0.00045444705809142676,
                                                          +0.00045431557316717296,
                                                          +0.00045418425802357484,
                                                          +0.00045405292513205021,
                                                          +0.00045392176172689177,
                                                          +0.00045379058060456153,
                                                          +0.00045365956867498728,
                                                          +0.00045352853905892486,
                                                          +0.00045339767834268652,
                                                          +0.00045326679997057266,
                                                          +0.00045313609020602660,
                                                          +0.00045300536281614717,
                                                          +0.00045287480374225307,
                                                          +0.00045274422707349752,
                                                          +0.00045261381842981663,
                                                          +0.00045248339222167576,
                                                          +0.00045235313374836954,
                                                          +0.00045222285774093471,
                                                          +0.00045209274917876207,
                                                          +0.00045196262311272259,
                                                          +0.00045183266420303913,
                                                          +0.00045170268781968088,
                                                          +0.00045157287830443696,
                                                          +0.00045144305134564111,
                                                          +0.00045131339096737952,
                                                          +0.00045118371317561960,
                                                          +0.00045105420167747512,
                                                          +0.00045092467279581668,
                                                          +0.00045079530992151302,
                                                          +0.00045066592969361132,
                                                          +0.00045053671518746022,
                                                          +0.00045040748335755835,
                                                          +0.00045027841696445794,
                                                          +0.00045014933327738524,
                                                          +0.00045002041474281797,
                                                          +0.00044989147894398908,
                                                          +0.00044976270801401995,
                                                          +0.00044963391984943182,
                                                          +0.00044950529627070751,
                                                          +0.00044937665548693879,
                                                          +0.00044924817900668509,
                                                          +0.00044911968535089409,
                                                          +0.00044899135571691478,
                                                          +0.00044886300893683779,
                                                          +0.00044873482589751271,
                                                          +0.00044860662574146233,
                                                          +0.00044847858904574598,
                                                          +0.00044835053526260982,
                                                          +0.00044822264466002948,
                                                          +0.00044809473699926788,
                                                          +0.00044796699223992228,
                                                          +0.00044783923045156691,
                                                          +0.00044771163128612439,
                                                          +0.00044758401512077746,
                                                          +0.00044745656130047408,
                                                          +0.00044732909050930540,
                                                          +0.00044720178178594381,
                                                          +0.00044707445612068999,
                                                          +0.00044694729224663781,
                                                          +0.00044682011145960024,
                                                          +0.00044669309218778818,
                                                          +0.00044656605603183186,
                                                          +0.00044643918111575193,
                                                          +0.00044631228934430365,
                                                          +0.00044618555853800809,
                                                          +0.00044605881090505463,
                                                          +0.00044593222396315402,
                                                          +0.00044580562022324084,
                                                          +0.00044567917690090266,
                                                          +0.00044555271680913195,
                                                          +0.00044542641686207905,
                                                          +0.00044530010017410906,
                                                          +0.00044517394335861734,
                                                          +0.00044504776983065946,
                                                          +0.00044492175590355772,
                                                          +0.00044479572529237612,
                                                          +0.00044466985401104306,
                                                          +0.00044454396607395243,
                                                          +0.00044441823719631595,
                                                          +0.00044429249169118006,
                                                          +0.00044416690497571575,
                                                          +0.00044404130166094595,
                                                          +0.00044391585686667491,
                                                          +0.00044379039550122899,
                                                          +0.00044366509238771661,
                                                          +0.00044353977273109641,
                                                          +0.00044341461105845128,
                                                          +0.00044328943287070183,
                                                          +0.00044316441239957349,
                                                          +0.00044303937544128114,
                                                          +0.00044291449593285905,
                                                          +0.00044278959996515055,
                                                          +0.00044266486118116197,
                                                          +0.00044254010596570176,
                                                          +0.00044241550766841131,
                                                          +0.00044229089296740116,
                                                          +0.00044216643491960827,
                                                          +0.00044204196049578523,
                                                          +0.00044191764246082304,
                                                          +0.00044179330807745792,
                                                          +0.00044166912981919180,
                                                          +0.00044154493524008789,
                                                          +0.00044142089652291411,
                                                          +0.00044129684151240553,
                                                          +0.00044117294210124921,
                                                          +0.00044104902642419958,
                                                          +0.00044092526608451369,
                                                          +0.00044080148950631439,
                                                          +0.00044067786800407811,
                                                          +0.00044055423029064708,
                                                          +0.00044043074739236448,
                                                          +0.00044030724831014434,
                                                          +0.00044018390378284281,
                                                          +0.00044006054309879970,
                                                          +0.00043993733671002840,
                                                          +0.00043981411419165083,
                                                          +0.00043969104570947904,
                                                          +0.00043956796112477577,
                                                          +0.00043944503031779196,
                                                          +0.00043932208343529060,
                                                          +0.00043919929007260074,
                                                          +0.00043907648066134693,
                                                          +0.00043895382451257308,
                                                          +0.00043883115234212837,
                                                          +0.00043870863317740688,
                                                          +0.00043858609801784778,
                                                          +0.00043846371560782830,
                                                          +0.00043834131722974467,
                                                          +0.00043821907134558851,
                                                          +0.00043809680952008184,
                                                          +0.00043797469993346063,
                                                          +0.00043785257443214299,
                                                          +0.00043773060091523722,
                                                          +0.00043760861151022985,
                                                          +0.00043748677383572738,
                                                          +0.00043736492029965884,
                                                          +0.00043724321824075378,
                                                          +0.00043712150034675898,
                                                          +0.00043699993367714990,
                                                          +0.00043687835119886868,
                                                          +0.00043675691969275734,
                                                          +0.00043663547240433250,
                                                          +0.00043651417583642272,
                                                          +0.00043639286351249964,
                                                          +0.00043627170165799528,
                                                          +0.00043615052407371965,
                                                          +0.00043602949670832383,
                                                          +0.00043590845363934038,
                                                          +0.00043578756053925410,
                                                          +0.00043566665176170550,
                                                          +0.00043554589270362572,
                                                          +0.00043542511799415126,
                                                          +0.00043530449275526993,
                                                          +0.00043518385189100371,
                                                          +0.00043506336024900643,
                                                          +0.00043494285300757659,
                                                          +0.00043482249474064078,
                                                          +0.00043470212090016735,
                                                          +0.00043458189578696172,
                                                          +0.00043446165512605602,
                                                          +0.00043434156294573851,
                                                          +0.00043422145524350114,
                                                          +0.00043410149577571784,
                                                          +0.00043398152081173814,
                                                          +0.00043386169383662174,
                                                          +0.00043374185139097564,
                                                          +0.00043362215668914429,
                                                          +0.00043350244654239323,
                                                          +0.00043338288389494943,
                                                          +0.00043326330582813926,
                                                          +0.00043314387501666783,
                                                          +0.00043302442881132727,
                                                          +0.00043290512961789468,
                                                          +0.00043278581505603383,
                                                          +0.00043266664726318663,
                                                          +0.00043254746412729593,
                                                          +0.00043242842751805942,
                                                          +0.00043230937559110848,
                                                          +0.00043219046994898521,
                                                          +0.00043207154901442086,
                                                          +0.00043195277412338973,
                                                          +0.00043183398396513535,
                                                          +0.00043171533960964991,
                                                          +0.00043159668001210345,
                                                          +0.00043147816597709107,
                                                          +0.00043135963672512475,
                                                          +0.00043124125279598463,
                                                          +0.00043112285367494262,
                                                          +0.00043100459963754504,
                                                          +0.00043088633043324268,
                                                          +0.00043076820607392766,
                                                          +0.00043065006657265009,
                                                          +0.00043053207167822598,
                                                          +0.00043041406166672680,
                                                          +0.00043029619602446883,
                                                          +0.00043017831528996880,
                                                          +0.00043006057868761824,
                                                          +0.00042994282701780423,
                                                          +0.00042982521924356666,
                                                          +0.00042970759642659012,
                                                          +0.00042959011726913428,
                                                          +0.00042947262309360999,
                                                          +0.00042935527234206695,
                                                          +0.00042923790659707173,
                                                          +0.00042912068404103292,
                                                          +0.00042900344651610486,
                                                          +0.00042888635194562126,
                                                          +0.00042876924243075756,
                                                          +0.00042865227563633848,
                                                          +0.00042853529392199472,
                                                          +0.00042841845469460647,
                                                          +0.00042830160057169580,
                                                          +0.00042818488870276000,
                                                          +0.00042806816196265085,
                                                          +0.00042795157724404410,
                                                          +0.00042783497767855989,
                                                          +0.00042771851990261168,
                                                          +0.00042760204730402915,
                                                          +0.00042748571626352095,
                                                          +0.00042736937042456887,
                                                          +0.00042725316591273304,
                                                          +0.00042713694662659124,
                                                          +0.00042702086843710967,
                                                          +0.00042690477549740737,
                                                          +0.00042678882342441036,
                                                          +0.00042667285662522541,
                                                          +0.00042655703046329030,
                                                          +0.00042644118959914807,
                                                          +0.00042632548914329777,
                                                          +0.00042620977400916940,
                                                          +0.00042609419905487194,
                                                          +0.00042597860944617303,
                                                          +0.00042586315978933992,
                                                          +0.00042574769550193038,
                                                          +0.00042563237093891512,
                                                          +0.00042551703176909705,
                                                          +0.00042540183209669423,
                                                          +0.00042528661784121046,
                                                          +0.00042517154285665488,
                                                          +0.00042505645331268919,
                                                          +0.00042494150281365380,
                                                          +0.00042482653777882766,
                                                          +0.00042471171156342372,
                                                          +0.00042459687083579742,
                                                          +0.00042448216870257150,
                                                          +0.00042436745208064103,
                                                          +0.00042425287382857565,
                                                          +0.00042413828111127224,
                                                          +0.00042402382653978387,
                                                          +0.00042390935752647342,
                                                          +0.00042379502643541080,
                                                          +0.00042368068092589153,
                                                          +0.00042356647311553552,
                                                          +0.00042345225091003808,
                                                          +0.00042333816618109969,
                                                          +0.00042322406708028467,
                                                          +0.00042311010523390447,
                                                          +0.00042299612903886252,
                                                          +0.00042288228987660890,
                                                          +0.00042276843638885855,
                                                          +0.00042265471971272736,
                                                          +0.00042254098873421441,
                                                          +0.00042242739434662703,
                                                          +0.00042231378567972327,
                                                          +0.00042220031338352596,
                                                          +0.00042208682683102803,
                                                          +0.00042197347642949055,
                                                          +0.00042186011179461862,
                                                          +0.00042174688309143339,
                                                          +0.00042163364017783049,
                                                          +0.00042152053297711079,
                                                          +0.00042140741158884140,
                                                          +0.00042129442569512093,
                                                          +0.00042118142563666970,
                                                          +0.00042106856085490118,
                                                          +0.00042095568193117173,
                                                          +0.00042084293806672607,
                                                          +0.00042073018008304035,
                                                          +0.00042061755694170486,
                                                          +0.00042050491970380176,
                                                          +0.00042039241709177969,
                                                          +0.00042027990040581400,
                                                          +0.00042016751812972287,
                                                          +0.00042005512180226382,
                                                          +0.00041994285966913511,
                                                          +0.00041983058350716560,
                                                          +0.00041971844132444283,
                                                          +0.00041960628513535841,
                                                          +0.00041949426271089646,
                                                          +0.00041938222630250396,
                                                          +0.00041927032344456780,
                                                          +0.00041915840662508439,
                                                          +0.00041904662314234823,
                                                          +0.00041893482572040030,
                                                          +0.00041882316142194620,
                                                          +0.00041871148320656807,
                                                          +0.00041859993790188503,
                                                          +0.00041848837870251846,
                                                          +0.00041837695220150111,
                                                          +0.00041826551182799342,
                                                          +0.00041815420394094145,
                                                          +0.00041804288220354500,
                                                          +0.00041793169274116160,
                                                          +0.00041782048945053247,
                                                          +0.00041770941822392337,
                                                          +0.00041759833319112035,
                                                          +0.00041748738001179289,
                                                          +0.00041737641304827626,
                                                          +0.00041726557772813828,
                                                          +0.00041715472864576936,
                                                          +0.00041704401099712778,
                                                          +0.00041693327960816678,
                                                          +0.00041682267944372748,
                                                          +0.00041671206556083342,
                                                          +0.00041660158269369885,
                                                          +0.00041649108612992818,
                                                          +0.00041638072037359741,
                                                          +0.00041627034094240294,
                                                          +0.00041616009211076986,
                                                          +0.00041604982962599967,
                                                          +0.00041593969753335257,
                                                          +0.00041582955180924873,
                                                          +0.00041571953627026913,
                                                          +0.00041560950712146716,
                                                          +0.00041549960795122826,
                                                          +0.00041538969519275563,
                                                          +0.00041527991220672211,
                                                          +0.00041517011565399774,
                                                          +0.00041506044866802355,
                                                          +0.00041495076813685613,
                                                          +0.00041484121696718488,
                                                          +0.00041473165227377226,
                                                          +0.00041462221673703513,
                                                          +0.00041451276769796354,
                                                          +0.00041440344761117830,
                                                          +0.00041429411404342049,
                                                          +0.00041418490922399140,
                                                          +0.00041407569094490629,
                                                          +0.00041396660121062213,
                                                          +0.00041385749803795355,
                                                          +0.00041374852320698698,
                                                          +0.00041363953495886311,
                                                          +0.00041353067484976961,
                                                          +0.00041342180134470085,
                                                          +0.00041331305577641783,
                                                          +0.00041320429683329728,
                                                          +0.00041309566562514283,
                                                          +0.00041298702106324394,
                                                          +0.00041287850403491611,
                                                          +0.00041276997367389266,
                                                          +0.00041266157064546815,
                                                          +0.00041255315430535280,
                                                          +0.00041244486509728616,
                                                          +0.00041233656259848941,
                                                          +0.00041222838703161221,
                                                          +0.00041212019819492144,
                                                          +0.00041201213609044101,
                                                          +0.00041190406073701973,
                                                          +0.00041179611191651813,
                                                          +0.00041168814986790471,
                                                          +0.00041158031415333826,
                                                          +0.00041147246523144513,
                                                          +0.00041136474244514253,
                                                          +0.00041125700647225529,
                                                          +0.00041114939643691745,
                                                          +0.00041104177323569348,
                                                          +0.00041093427577439237,
                                                          +0.00041082676516786005,
                                                          +0.00041071938010403758,
                                                          +0.00041061198191559572,
                                                          +0.00041050470907306259,
                                                          +0.00041039742312647901,
                                                          +0.00041029026232941416,
                                                          +0.00041018308844882465,
                                                          +0.00041007603952177419,
                                                          +0.00040996897753168204,
                                                          +0.00040986204029955800,
                                                          +0.00040975509002483257,
                                                          +0.00040964826431291241,
                                                          +0.00040954142557878855,
                                                          +0.00040943471121271348,
                                                          +0.00040932798384479005,
                                                          +0.00040922138065056535,
                                                          +0.00040911476447480488,
                                                          +0.00040900827227879737,
                                                          +0.00040890176712152476,
                                                          +0.00040879538575046319,
                                                          +0.00040868899143836477,
                                                          +0.00040858272071933821,
                                                          +0.00040847643707946085,
                                                          +0.00040837027683991782,
                                                          +0.00040826410369966830,
                                                          +0.00040815805376741586,
                                                          +0.00040805199095455939,
                                                          +0.00040794605115776223,
                                                          +0.00040784009850042177,
                                                          +0.00040773426866760166,
                                                          +0.00040762842599425717,
                                                          +0.00040752270595429112,
                                                          +0.00040741697309377848,
                                                          +0.00040731136267589874,
                                                          +0.00040720573945740861,
                                                          +0.00040710023849120143,
                                                          +0.00040699472474427852,
                                                          +0.00040688933305968304,
                                                          +0.00040678392861422547,
                                                          +0.00040667864604153304,
                                                          +0.00040657335072779086,
                                                          +0.00040646817709764406,
                                                          +0.00040636299074621885,
                                                          +0.00040625792588961041,
                                                          +0.00040615284833145430,
                                                          +0.00040604789207972637,
                                                          +0.00040594292314614070,
                                                          +0.00040583807533098418,
                                                          +0.00040573321485361886,
                                                          +0.00040562847530707208,
                                                          +0.00040552372311792500,
                                                          +0.00040541909167237305,
                                                          +0.00040531444760378882,
                                                          +0.00040520992409196247,
                                                          +0.00040510538797663146,
                                                          +0.00040500097223160651,
                                                          +0.00040489654390256439,
                                                          +0.00040479223575776055,
                                                          +0.00040468791504838703,
                                                          +0.00040458371433756726,
                                                          +0.00040447950108158504,
                                                          +0.00040437540763885459,
                                                          +0.00040427130167032901,
                                                          +0.00040416731533013454,
                                                          +0.00040406331648347237,
                                                          +0.00040395943708060084,
                                                          +0.00040385554519054939,
                                                          +0.00040375177256012768,
                                                          +0.00040364798746177417,
                                                          +0.00040354432143926744,
                                                          +0.00040344064296803762,
                                                          +0.00040333708338924958,
                                                          +0.00040323351138090731,
                                                          +0.00040313005808197827,
                                                          +0.00040302659237262492,
                                                          +0.00040292324519003122,
                                                          +0.00040281988561610379,
                                                          +0.00040271664438665749,
                                                          +0.00040261339078492885,
                                                          +0.00040251025534577601,
                                                          +0.00040240710755335313,
                                                          +0.00040230407774197397,
                                                          +0.00040220103559629802,
                                                          +0.00040209811125050467,
                                                          +0.00040199517458934931,
                                                          +0.00040189235554728643,
                                                          +0.00040178952420875729,
                                                          +0.00040168681030890033,
                                                          +0.00040158408413143448,
                                                          +0.00040148147521258893,
                                                          +0.00040137885403495336,
                                                          +0.00040127634993625429,
                                                          +0.00040117383359754543,
                                                          +0.00040107143415845633,
                                                          +0.00040096902249809946,
                                                          +0.00040086672755841151,
                                                          +0.00040076442041615926,
                                                          +0.00040066222981599055,
                                                          +0.00040056002703192292,
                                                          +0.00040045794061171721,
                                                          +0.00040035584202624005,
                                                          +0.00040025385962676652,
                                                          +0.00040015186508061111,
                                                          +0.00040004998654296302,
                                                          +0.00039994809587718479,
                                                          +0.00039984632104277931,
                                                          +0.00039974453409875746,
                                                          +0.00039964286280933414,
                                                          +0.00039954117942877089,
                                                          +0.00039943961152639085,
                                                          +0.00039933803155130975,
                                                          +0.00039923656687835618,
                                                          +0.00039913509015110246,
                                                          +0.00039903372855027764,
                                                          +0.00039893235491351655,
                                                          +0.00039883109622784296,
                                                          +0.00039872982552455947,
                                                          +0.00039862866959737775,
                                                          +0.00039852750167087555,
                                                          +0.00039842644834584418,
                                                          +0.00039832538303974461,
                                                          +0.00039822443216083926,
                                                          +0.00039812346931908100,
                                                          +0.00039802262073059344,
                                                          +0.00039792176019743124,
                                                          +0.00039782101374396866,
                                                          +0.00039772025536397292,
                                                          +0.00039761961089045697,
                                                          +0.00039751895450851273,
                                                          +0.00039741841186017891,
                                                          +0.00039731785732148508,
                                                          +0.00039721741634388195,
                                                          +0.00039711696349395051,
                                                          +0.00039701662403293886,
                                                          +0.00039691627271759415,
                                                          +0.00039681603461934610,
                                                          +0.00039671578468472379,
                                                          +0.00039661564779572229,
                                                          +0.00039651549908826909,
                                                          +0.00039641546325530668,
                                                          +0.00039631541562177902,
                                                          +0.00039621548069195738,
                                                          +0.00039611553397942085,
                                                          +0.00039601569980015017,
                                                          +0.00039591585385597897,
                                                          +0.00039581612027497662,
                                                          +0.00039571637494685216,
                                                          +0.00039561674181214258,
                                                          +0.00039551709694805371,
                                                          +0.00039541756410796684,
                                                          +0.00039531801955620775,
                                                          +0.00039521858685937937,
                                                          +0.00039511914246854991,
                                                          +0.00039501980976391985,
                                                          +0.00039492046538292452,
                                                          +0.00039482123251973611,
                                                          +0.00039472198799778293,
                                                          +0.00039462285482558284,
                                                          +0.00039452371001218287,
                                                          +0.00039442467638081972,
                                                          +0.00039432563112578629,
                                                          +0.00039422669688540996,
                                                          +0.00039412775103885798,
                                                          +0.00039402891603991904,
                                                          +0.00039393006945226394,
                                                          +0.00039383133354551296,
                                                          +0.00039373258606747048,
                                                          +0.00039363394910395689,
                                                          +0.00039353530058654131,
                                                          +0.00039343676241761341,
                                                          +0.00039333821271213818,
                                                          +0.00039323977318944133,
                                                          +0.00039314132214751728,
                                                          +0.00039304298112299423,
                                                          +0.00039294462859652916,
                                                          +0.00039284638592241829,
                                                          +0.00039274813176361640,
                                                          +0.00039264998729245196,
                                                          +0.00039255183135381271,
                                                          +0.00039245378493842336,
                                                          +0.00039235572707274120,
                                                          +0.00039225777856624947,
                                                          +0.00039215981862661262,
                                                          +0.00039206196788243454,
                                                          +0.00039196410572222472,
                                                          +0.00039186635259406857,
                                                          +0.00039176858806695979,
                                                          +0.00039167093240882560,
                                                          +0.00039157326536878395,
                                                          +0.00039147570703496276,
                                                          +0.00039137813733624541,
                                                          +0.00039128067618131855,
                                                          +0.00039118320367847293,
                                                          +0.00039108583955731126,
                                                          +0.00039098846410517426,
                                                          +0.00039089119687293775,
                                                          +0.00039079391832663567,
                                                          +0.00039069674783877192,
                                                          +0.00039059956605371886,
                                                          +0.00039050249216596324,
                                                          +0.00039040540699786106,
                                                          +0.00039030842956623546,
                                                          +0.00039021144087107251,
                                                          +0.00039011455975188503,
                                                          +0.00039001766738593584,
                                                          +0.00038992088243577976,
                                                          +0.00038982408625560432,
                                                          +0.00038972739733135730,
                                                          +0.00038963069719380027,
                                                          +0.00038953410415262389,
                                                          +0.00038943749991481361,
                                                          +0.00038934100261415284,
                                                          +0.00038924449413350138,
                                                          +0.00038914809243108315,
                                                          +0.00038905167956528443,
                                                          +0.00038895537331911810,
                                                          +0.00038885905592614839,
                                                          +0.00038876284499452386,
                                                          +0.00038866662293264050,
                                                          +0.00038857050717412814,
                                                          +0.00038847438030186875,
                                                          +0.00038837835957531883,
                                                          +0.00038828232775150085,
                                                          +0.00038818640191604228,
                                                          +0.00038809046499976223,
                                                          +0.00038799463391480250,
                                                          +0.00038789879176543530,
                                                          +0.00038780305529065935,
                                                          +0.00038770730776785747,
                                                          +0.00038761166576322710,
                                                          +0.00038751601272692029,
                                                          +0.00038742046505267370,
                                                          +0.00038732490636306764,
                                                          +0.00038722945287971854,
                                                          +0.00038713398839729479,
                                                          +0.00038703862896563172,
                                                          +0.00038694325855114680,
                                                          +0.00038684799303223246,
                                                          +0.00038675271654671695,
                                                          +0.00038665754480188773,
                                                          +0.00038656236210664628,
                                                          +0.00038646728399751109,
                                                          +0.00038637219495412072,
                                                          +0.00038627721034256111,
                                                          +0.00038618221481287153,
                                                          +0.00038608732356104012,
                                                          +0.00038599242140717238,
                                                          +0.00038589762337749292,
                                                          +0.00038580281446183909,
                                                          +0.00038570810951700540,
                                                          +0.00038561339370222781,
                                                          +0.00038551878170520323,
                                                          +0.00038542415885423379,
                                                          +0.00038532963966825051,
                                                          +0.00038523510964428984,
                                                          +0.00038514068313284848,
                                                          +0.00038504624579936599,
                                                          +0.00038495191182623433,
                                                          +0.00038485756704696657,
                                                          +0.00038476332547617947,
                                                          +0.00038466907311513019,
                                                          +0.00038457492381098869,
                                                          +0.00038448076373242786,
                                                          +0.00038438670655949870,
                                                          +0.00038429263862796198,
                                                          +0.00038419867345107668,
                                                          +0.00038410469753136477,
                                                          +0.00038401082421561921,
                                                          +0.00038391694017279702,
                                                          +0.00038382315858355074,
                                                          +0.00038372936628294703,
                                                          +0.00038363567628582260,
                                                          +0.00038354197559302937,
                                                          +0.00038344837705391125,
                                                          +0.00038335476783478260,
                                                          +0.00038326126061981769,
                                                          +0.00038316774274046951,
                                                          +0.00038307432671606546,
                                                          +0.00038298090004287497,
                                                          +0.00038288757507569984,
                                                          +0.00038279423947530476,
                                                          +0.00038270100543228634,
                                                          +0.00038260776077158411,
                                                          +0.00038251461751990959,
                                                          +0.00038242146366605724,
                                                          +0.00038232841107317206,
                                                          +0.00038223534789358485,
                                                          +0.00038214238582719254,
                                                          +0.00038204941318954375,
                                                          +0.00038195654151760525,
                                                          +0.00038186365928982583,
                                                          +0.00038177087788055842,
                                                          +0.00038167808593083589,
                                                          +0.00038158539465271300,
                                                          +0.00038149269284949087,
                                                          +0.00038140009157124148,
                                                          +0.00038130747978321885,
                                                          +0.00038121496837382671,
                                                          +0.00038112244646995748,
                                                          +0.00038103002479866039,
                                                          +0.00038093759264815280,
                                                          +0.00038084526058444231,
                                                          +0.00038075291805675815,
                                                          +0.00038066067547037858,
                                                          +0.00038056842243523276,
                                                          +0.00038047626919618079,
                                                          +0.00038038410552354048,
                                                          +0.00038029204150206460,
                                                          +0.00038019996706214901,
                                                          +0.00038010799212874870,
                                                          +0.00038001600679202799,
                                                          +0.00037992412081745328,
                                                          +0.00037983222445464831,
                                                          +0.00037974042730989926,
                                                          +0.00037964861979198088,
                                                          +0.00037955691134830658,
                                                          +0.00037946519254649475,
                                                          +0.00037937357267539340,
                                                          +0.00037928194246115732,
                                                          +0.00037919041103437473,
                                                          +0.00037909886927943123,
                                                          +0.00037900742616896118,
                                                          +0.00037891597274527486,
                                                          +0.00037882461782335778,
                                                          +0.00037873325260314063,
                                                          +0.00037864198574226313,
                                                          +0.00037855070859797279,
                                                          +0.00037845952967086743,
                                                          +0.00037836834047520821,
                                                          +0.00037827724935485236,
                                                          +0.00037818614798077263,
                                                          +0.00037809514454038883,
                                                          +0.00037800413086108242,
                                                          +0.00037791321497413649,
                                                          +0.00037782228886304106,
                                                          +0.00037773146040324246,
                                                          +0.00037764062173403903,
                                                          +0.00037754988057533983,
                                                          +0.00037745912922195211,
                                                          +0.00037736847523854697,
                                                          +0.00037727781107514132,
                                                          +0.00037718724414146591,
                                                          +0.00037709666704244981,
                                                          +0.00037700618703318147,
                                                          +0.00037691569687320402,
                                                          +0.00037682530366325991,
                                                          +0.00037673490031721047,
                                                          +0.00037664459378174814,
                                                          +0.00037655427712475617,
                                                          +0.00037646405713917210,
                                                          +0.00037637382704660606,
                                                          +0.00037628369348653582,
                                                          +0.00037619354983400329,
                                                          +0.00037610350257532036,
                                                          +0.00037601344523866708,
                                                          +0.00037592348415748246,
                                                          +0.00037583351301279170,
                                                          +0.00037574363798545374,
                                                          +0.00037565375290904624,
                                                          +0.00037556396381213913,
                                                          +0.00037547416468057158,
                                                          +0.00037538446139091618,
                                                          +0.00037529474808098168,
                                                          +0.00037520513047563343,
                                                          +0.00037511550286436021,
                                                          +0.00037502597082060980,
                                                          +0.00037493642878526102,
                                                          +0.00037484698218063318,
                                                          +0.00037475752559870619,
                                                          +0.00037466816431095935,
                                                          +0.00037457879306018532,
                                                          +0.00037448951696731080,
                                                          +0.00037440023092565397,
                                                          +0.00037431103990587591,
                                                          +0.00037422183895153302,
                                                          +0.00037413273288330727,
                                                          +0.00037404361689470742,
                                                          +0.00037395459565672145,
                                                          +0.00037386556451252491,
                                                          +0.00037377662798369687,
                                                          +0.00037368768156279491,
                                                          +0.00037359882962227349,
                                                          +0.00037350996780378785,
                                                          +0.00037342120033095145,
                                                          +0.00037333242299423364,
                                                          +0.00037324373986868974,
                                                          +0.00037315504689332062,
                                                          +0.00037306644799490557,
                                                          +0.00037297783926069483,
                                                          +0.00037288932446947294,
                                                          +0.00037280079985645826,
                                                          +0.00037271236905272168,
                                                          +0.00037262392844116844,
                                                          +0.00037253558150543626,
                                                          +0.00037244722477583708,
                                                          +0.00037235896158885503,
                                                          +0.00037227068862192913,
                                                          +0.00037218250906466855,
                                                          +0.00037209431974136111,
                                                          +0.00037200622369501924,
                                                          +0.00037191811789650099,
                                                          +0.00037183010524249974,
                                                          +0.00037174208285016661,
                                                          +0.00037165415347015218,
                                                          +0.00037156621436562409,
                                                          +0.00037147836814146692,
                                                          +0.00037139051220658810,
                                                          +0.00037130274902038158,
                                                          +0.00037121497613721942,
                                                          +0.00037112729587128002,
                                                          +0.00037103960592212490,
                                                          +0.00037095200845899113,
                                                          +0.00037086440132635571,
                                                          +0.00037077688654878801,
                                                          +0.00037068936211540662,
                                                          +0.00037060192990638673,
                                                          +0.00037051448805521534,
                                                          +0.00037042713829794536,
                                                          +0.00037033977891216048,
                                                          +0.00037025251149006287,
                                                          +0.00037016523445306134,
                                                          +0.00037007804924977822,
                                                          +0.00036999085444517645,
                                                          +0.00036990375134456934,
                                                          +0.00036981663865620314,
                                                          +0.00036972961754235174,
                                                          +0.00036964258685427524,
                                                          +0.00036955564761147793,
                                                          +0.00036946869880796410,
                                                          +0.00036938184132073623,
                                                          +0.00036929497428627508,
                                                          +0.00036920819843934958,
                                                          +0.00036912141305864857,
                                                          +0.00036903471873697474,
                                                          +0.00036894801489495806,
                                                          +0.00036886140198370116,
                                                          +0.00036877477956550898,
                                                          +0.00036868824795005013,
                                                          +0.00036860170684103810,
                                                          +0.00036851525640697348,
                                                          +0.00036842879649271268,
                                                          +0.00036834242712585277,
                                                          +0.00036825604829212867,
                                                          +0.00036816975987849813,
                                                          +0.00036808346201131067,
                                                          +0.00036799725443714768,
                                                          +0.00036791103742271001,
                                                          +0.00036782491057446587,
                                                          +0.00036773877429920434,
                                                          +0.00036765272806354288,
                                                          +0.00036756667241409646,
                                                          +0.00036748070667789383,
                                                          +0.00036739473154111371,
                                                          +0.00036730884619145718,
                                                          +0.00036722295145440615,
                                                          +0.00036713714637859432,
                                                          +0.00036705133192854630,
                                                          +0.00036696560701408816,
                                                          +0.00036687987273852779,
                                                          +0.00036679422787314254,
                                                          +0.00036670857365976443,
                                                          +0.00036662300873138091,
                                                          +0.00036653743446808933,
                                                          +0.00036645194936484542,
                                                          +0.00036636645493975381,
                                                          +0.00036628104954999614,
                                                          +0.00036619563485142684,
                                                          +0.00036611030906370990,
                                                          +0.00036602497398019305,
                                                          +0.00036593972768327927,
                                                          +0.00036585447210355307,
                                                          +0.00036576930518641190,
                                                          +0.00036568412899942162,
                                                          +0.00036559904135122917,
                                                          +0.00036551394444612676,
                                                          +0.00036542893595626540,
                                                          +0.00036534391822240926,
                                                          +0.00036525898878046704,
                                                          +0.00036517405010742126,
                                                          +0.00036508919960319142,
                                                          +0.00036500433988072517,
                                                          +0.00036491956820420597,
                                                          +0.00036483478732229359,
                                                          +0.00036475009436368722,
                                                          +0.00036466539221250717,
                                                          +0.00036458077786222004,
                                                          +0.00036449615433215466,
                                                          +0.00036441161848079629,
                                                          +0.00036432707346243151,
                                                          +0.00036424261600081413,
                                                          +0.00036415814938493867,
                                                          +0.00036407377020407719,
                                                          +0.00036398938188168226,
                                                          +0.00036390508087279347,
                                                          +0.00036382077073507211,
                                                          +0.00036373654778957431,
                                                          +0.00036365231572792130,
                                                          +0.00036356817073743367,
                                                          +0.00036348401664344489,
                                                          +0.00036339994949978710,
                                                          +0.00036331587326525884,
                                                          +0.00036323188386045070,
                                                          +0.00036314788537737946,
                                                          +0.00036306397360364058,
                                                          +0.00036298005276422241,
                                                          +0.00036289621851397132,
                                                          +0.00036281237521060180,
                                                          +0.00036272861837645571,
                                                          +0.00036264485250172879,
                                                          +0.00036256117297650331,
                                                          +0.00036247748442321128,
                                                          +0.00036239388209991972,
                                                          +0.00036231027076105292,
                                                          +0.00036222674553290592,
                                                          +0.00036214321130165173,
                                                          +0.00036205976306205682,
                                                          +0.00036197630583179995,
                                                          +0.00036189293447436101,
                                                          +0.00036180955413868236,
                                                          +0.00036172625955719907,
                                                          +0.00036164295600987572,
                                                          +0.00036155973809834347,
                                                          +0.00036147651123334793,
                                                          +0.00036139336988595720,
                                                          +0.00036131021959745702,
                                                          +0.00036122715470859277,
                                                          +0.00036114408089095039,
                                                          +0.00036106109235519189,
                                                          +0.00036097809490296371,
                                                          +0.00036089518261508377,
                                                          +0.00036081226142302002,
                                                          +0.00036072942527798504,
                                                          +0.00036064658024102954,
                                                          +0.00036056382013399825,
                                                          +0.00036048105114728720,
                                                          +0.00036039836697361128,
                                                          +0.00036031567393247379,
                                                          +0.00036023306558769636,
                                                          +0.00036015044838765334,
                                                          +0.00036006791576750929,
                                                          +0.00035998537430427329,
                                                          +0.00035990291730468848,
                                                          +0.00035982045147416302,
                                                          +0.00035973806999125402,
                                                          +0.00035965567968953345,
                                                          +0.00035957337361960695,
                                                          +0.00035949105874297574,
                                                          +0.00035940882798252821,
                                                          +0.00035932658842746066,
                                                          +0.00035924443287317805,
                                                          +0.00035916226853633792,
                                                          +0.00035908018808509476,
                                                          +0.00035899809886333475,
                                                          +0.00035891609341219414,
                                                          +0.00035883407920255536,
                                                          +0.00035875214864876864,
                                                          +0.00035867020934848029,
                                                          +0.00035858835358948616,
                                                          +0.00035850648909596498,
                                                          +0.00035842470802938960,
                                                          +0.00035834291824023991,
                                                          +0.00035826121176389572,
                                                          +0.00035817949657690842,
                                                          +0.00035809786458879453,
                                                          +0.00035801622390194674,
                                                          +0.00035793466630024812,
                                                          +0.00035785310001170317,
                                                          +0.00035777161669479018,
                                                          +0.00035769012470289674,
                                                          +0.00035760871556932475,
                                                          +0.00035752729777261674,
                                                          +0.00035744596272112582,
                                                          +0.00035736461901832172,
                                                          +0.00035728335794783615,
                                                          +0.00035720208823783868,
                                                          +0.00035712090104746662,
                                                          +0.00035703970522936240,
                                                          +0.00035695859181839523,
                                                          +0.00035687746979145424,
                                                          +0.00035679643005936631,
                                                          +0.00035671538172304176,
                                                          +0.00035663441556948999,
                                                          +0.00035655344082341739,
                                                          +0.00035647254814824087,
                                                          +0.00035639164689223797,
                                                          +0.00035631082759545753,
                                                          +0.00035622999972952394,
                                                          +0.00035614925371134159,
                                                          +0.00035606849913565828,
                                                          +0.00035598782629645692,
                                                          +0.00035590714491138566,
                                                          +0.00035582654515172889,
                                                          +0.00035574593685781201,
                                                          +0.00035566541007844328,
                                                          +0.00035558487477640338,
                                                          +0.00035550442087824581,
                                                          +0.00035542395846898506,
                                                          +0.00035534357735314120,
                                                          +0.00035526318773774118,
                                                          +0.00035518287930549219,
                                                          +0.00035510256238521315,
                                                          +0.00035502232653801893,
                                                          +0.00035494208221429989,
                                                          +0.00035486191885379816,
                                                          +0.00035478174702825607,
                                                          +0.00035470165605626234,
                                                          +0.00035462155663069212,
                                                          +0.00035454153794919885,
                                                          +0.00035446151082557225,
                                                          +0.00035438156433674917,
                                                          +0.00035430160941721521,
                                                          +0.00035422173502340818,
                                                          +0.00035414185221029212,
                                                          +0.00035406204981402323,
                                                          +0.00035398223900982655,
                                                          +0.00035390250851379353,
                                                          +0.00035382276962119338,
                                                          +0.00035374311092826902,
                                                          +0.00035366344385011806,
                                                          +0.00035358385686335016,
                                                          +0.00035350426150267562,
                                                          +0.00035342474612528642,
                                                          +0.00035334522238529011,
                                                          +0.00035326577852067605,
                                                          +0.00035318632630473412,
                                                          +0.00035310695385646512,
                                                          +0.00035302757306812714,
                                                          +0.00035294827193994654,
                                                          +0.00035286896248293580,
                                                          +0.00035278973257875981,
                                                          +0.00035271049435697219,
                                                          +0.00035263133558088947,
                                                          +0.00035255216849839363,
                                                          +0.00035247308075466512,
                                                          +0.00035239398471570172,
                                                          +0.00035231496790876009,
                                                          +0.00035223594281774180,
                                                          +0.00035215699685219086,
                                                          +0.00035207804261370166,
                                                          +0.00035199916739431637,
                                                          +0.00035192028391311110,
                                                          +0.00035184147934483708,
                                                          +0.00035176266652584156,
                                                          +0.00035168393251379447,
                                                          +0.00035160519026210425,
                                                          +0.00035152652671156977,
                                                          +0.00035144785493245085,
                                                          +0.00035136926174888399,
                                                          +0.00035129066034777145,
                                                          +0.00035121213743679626,
                                                          +0.00035113360631929463,
                                                          +0.00035105515358670381,
                                                          +0.00035097669265858611,
                                                          +0.00035089831001034084,
                                                          +0.00035081991917754857,
                                                          +0.00035074160651977782,
                                                          +0.00035066328568842032,
                                                          +0.00035058504292742083,
                                                          +0.00035050679200377522,
                                                          +0.00035042861904601070,
                                                          +0.00035035043793652117,
                                                          +0.00035027233468862225,
                                                          +0.00035019422329990007,
                                                          +0.00035011618966866378,
                                                          +0.00035003814790748655,
                                                          +0.00034996018379987611,
                                                          +0.00034988221157318745,
                                                          +0.00034980431689633182,
                                                          +0.00034972641411124147,
                                                          +0.00034964858877243454,
                                                          +0.00034957075533621729,
                                                          +0.00034949299924291846,
                                                          +0.00034941523506301414,
                                                          +0.00034933754812284710,
                                                          +0.00034925985310686031,
                                                          +0.00034918223522761318,
                                                          +0.00034910460928331280,
                                                          +0.00034902706037293724,
                                                          +0.00034894950340825614,
                                                          +0.00034887202337486754,
                                                          +0.00034879453529790174,
                                                          +0.00034871712404977887,
                                                          +0.00034863970476878791,
                                                          +0.00034856236221437206,
                                                          +0.00034848501163777841,
                                                          +0.00034840773768567319,
                                                          +0.00034833045572206154,
                                                          +0.00034825325028103274,
                                                          +0.00034817603683915007,
                                                          +0.00034809889981812526,
                                                          +0.00034802175480687998,
                                                          +0.00034794468611494816,
                                                          +0.00034786760944341053,
                                                          +0.00034779060898982130,
                                                          +0.00034771360056722240,
                                                          +0.00034763666826138629,
                                                          +0.00034755972799711761,
                                                          +0.00034748286374860555,
                                                          +0.00034740599155221936,
                                                          +0.00034732919527076186,
                                                          +0.00034725239105197004,
                                                          +0.00034717566264745744,
                                                          +0.00034709892631613166,
                                                          +0.00034702226569861345,
                                                          +0.00034694559716478456,
                                                          +0.00034686900424446901,
                                                          +0.00034679240341832691,
                                                          +0.00034671587810558086,
                                                          +0.00034663934489747396,
                                                          +0.00034656288710282237,
                                                          +0.00034648642142325693,
                                                          +0.00034641003105738291,
                                                          +0.00034633363281702355,
                                                          +0.00034625730979076719,
                                                          +0.00034618097890043589,
                                                          +0.00034610472312479460,
                                                          +0.00034602845949547028,
                                                          +0.00034595227088159841,
                                                          +0.00034587607442441705,
                                                          +0.00034579995288362513,
                                                          +0.00034572382350987909,
                                                          +0.00034564776895363387,
                                                          +0.00034557170657477165,
                                                          +0.00034549571891469570,
                                                          +0.00034541972344232164,
                                                          +0.00034534380259019267,
                                                          +0.00034526787393606643,
                                                          +0.00034519201980381752,
                                                          +0.00034511615787985401,
                                                          +0.00034504037037957273,
                                                          +0.00034496457509784143,
                                                          +0.00034488885414176985,
                                                          +0.00034481312541449489,
                                                          +0.00034473747091502905,
                                                          +0.00034466180865458851,
                                                          +0.00034458622052427812,
                                                          +0.00034451062464320388,
                                                          +0.00034443510279475218,
                                                          +0.00034435957320572938,
                                                          +0.00034428411755199249,
                                                          +0.00034420865416785951,
                                                          +0.00034413326462184635,
                                                          +0.00034405786735559424,
                                                          +0.00034398254383046601,
                                                          +0.00034390721259523814,
                                                          +0.00034383195500430812,
                                                          +0.00034375668971340007,
                                                          +0.00034368149797013331,
                                                          +0.00034360629853699226,
                                                          +0.00034353117255500501,
                                                          +0.00034345603889322969,
                                                          +0.00034338097858628927,
                                                          +0.00034330591060962933,
                                                          +0.00034323091589165389,
                                                          +0.00034315591351400982,
                                                          +0.00034308098429906771,
                                                          +0.00034300604743449045,
                                                          +0.00034293118363680020,
                                                          +0.00034285631219949060,
                                                          +0.00034278151373342040,
                                                          +0.00034270670763772925,
                                                          +0.00034263197441779665,
                                                          +0.00034255723357822404,
                                                          +0.00034248256551909582,
                                                          +0.00034240788985029119,
                                                          +0.00034233328686678245,
                                                          +0.00034225867628354361,
                                                          +0.00034218413829061847,
                                                          +0.00034210959270789210,
                                                          +0.00034203511962066225,
                                                          +0.00034196063895354284,
                                                          +0.00034188623068726807,
                                                          +0.00034181181485099801,
                                                          +0.00034173747132108551,
                                                          +0.00034166312023105490,
                                                          +0.00034158884135305882,
                                                          +0.00034151455492480458,
                                                          +0.00034144034061442602,
                                                          +0.00034136611876363208,
                                                          +0.00034129196893671864,
                                                          +0.00034121781157921561,
                                                          +0.00034114372615176098,
                                                          +0.00034106963320352553,
                                                          +0.00034099561209166920,
                                                          +0.00034092158346882366,
                                                          +0.00034084762658885109,
                                                          +0.00034077366220766370,
                                                          +0.00034069976947600507,
                                                          +0.00034062586925288934,
                                                          +0.00034055204058611985,
                                                          +0.00034047820443763410,
                                                          +0.00034040443975247357,
                                                          +0.00034033066759532073,
                                                          +0.00034025696680863347,
                                                          +0.00034018325855966096,
                                                          +0.00034010962158845492,
                                                          +0.00034003597716465386,
                                                          +0.00033996240392608103,
                                                          +0.00033988882324458648,
                                                          +0.00033981531365594181,
                                                          +0.00033974179663403230,
                                                          +0.00033966835061275395,
                                                          +0.00033959489716785078,
                                                          +0.00033952151463151971,
                                                          +0.00033944812468118720,
                                                          +0.00033937480554752667,
                                                          +0.00033930147900947149,
                                                          +0.00033922822319634694,
                                                          +0.00033915495998841785,
                                                          +0.00033908176741383650,
                                                          +0.00033900856745402422,
                                                          +0.00033893543803613482,
                                                          +0.00033886230124257163,
                                                          +0.00033878923489966405,
                                                          +0.00033871616119062332,
                                                          +0.00033864315784112842,
                                                          +0.00033857014713502444,
                                                          +0.00033849720669751370,
                                                          +0.00033842425891290152,
                                                          +0.00033835138130608656,
                                                          +0.00033827849636166159,
                                                          +0.00033820568150439394,
                                                          +0.00033813285931899150,
                                                          +0.00033806010713026261,
                                                          +0.00033798734762285762,
                                                          +0.00033791465802179832,
                                                          +0.00033784196111150529,
                                                          +0.00033776933401738528,
                                                          +0.00033769669962345776,
                                                          +0.00033762413495568575,
                                                          +0.00033755156299751613,
                                                          +0.00033747906067563911,
                                                          +0.00033740655107275844,
                                                          +0.00033733411101646165,
                                                          +0.00033726166368853891,
                                                          +0.00033718928581764559,
                                                          +0.00033711690068448797,
                                                          +0.00033704458491895886,
                                                          +0.00033697226190051093,
                                                          +0.00033690000816044400,
                                                          +0.00033682774717678783,
                                                          +0.00033675555538241827,
                                                          +0.00033668335635377304,
                                                          +0.00033661122642547251,
                                                          +0.00033653908927219376,
                                                          +0.00033646702113047054,
                                                          +0.00033639494577305086,
                                                          +0.00033632293933854917,
                                                          +0.00033625092569761677,
                                                          +0.00033617898089111683,
                                                          +0.00033610702888743605,
                                                          +0.00033603514562985354,
                                                          +0.00033596325518432424,
                                                          +0.00033589143339671018,
                                                          +0.00033581960443036752,
                                                          +0.00033574784403390782,
                                                          +0.00033567607646792198,
                                                          +0.00033560437738393718,
                                                          +0.00033553267113961311,
                                                          +0.00033546103328955813,
                                                          +0.00033538938828833520,
                                                          +0.00033531781159379918,
                                                          +0.00033524622775725054,
                                                          +0.00033517471213995653,
                                                          +0.00033510318938978967,
                                                          +0.00033503173477159400,
                                                          +0.00033496027302964970,
                                                          +0.00033488887933254215,
                                                          +0.00033481747852079466,
                                                          +0.00033474614566689768,
                                                          +0.00033467480570745407,
                                                          +0.00033460353361902320,
                                                          +0.00033453225443412331,
                                                          +0.00033446104303354622,
                                                          +0.00033438982454556229,
                                                          +0.00033431867375535886,
                                                          +0.00033424751588679533,
                                                          +0.00033417642562961721,
                                                          +0.00033410532830311057,
                                                          +0.00033403429850174088,
                                                          +0.00033396326164005879,
                                                          +0.00033389229221741229,
                                                          +0.00033382131574345405,
                                                          +0.00033375040662257611,
                                                          +0.00033367949045937182,
                                                          +0.00033360864156343882,
                                                          +0.00033353778563414949,
                                                          +0.00033346699688646794,
                                                          +0.00033339620111438515,
                                                          +0.00033332547243839199,
                                                          +0.00033325473674693733,
                                                          +0.00033318406806619920,
                                                          +0.00033311339237892417,
                                                          +0.00033304278361713751,
                                                          +0.00033297216785772339,
                                                          +0.00033290161893871384,
                                                          +0.00033283106303097119,
                                                          +0.00033276057387869346,
                                                          +0.00033269007774656186,
                                                          +0.00033261964828509963,
                                                          +0.00033254921185264786,
                                                          +0.00033247884200621295,
                                                          +0.00033240846519763786,
                                                          +0.00033233815489057069,
                                                          +0.00033226783763019754,
                                                          +0.00033219758678696651,
                                                          +0.00033212732899924884,
                                                          +0.00033205713754444976,
                                                          +0.00033198693915396850,
                                                          +0.00033191680701232494,
                                                          +0.00033184666794378867,
                                                          +0.00033177659504015115,
                                                          +0.00033170651521839553,
                                                          +0.00033163650147774161,
                                                          +0.00033156648082772951,
                                                          +0.00033149652617516309,
                                                          +0.00033142656462198378,
                                                          +0.00033135666898273552,
                                                          +0.00033128676645160463,
                                                          +0.00033121692975103106,
                                                          +0.00033114708616729065,
                                                          +0.00033107730833087411,
                                                          +0.00033100752361999163,
                                                          +0.00033093780457334030,
                                                          +0.00033086807866090917,
                                                          +0.00033079841832975618,
                                                          +0.00033072875114149505,
                                                          +0.00033065914945169876,
                                                          +0.00033058954091345140,
                                                          +0.00033051999779099483,
                                                          +0.00033045044782872981,
                                                          +0.00033038096319972058,
                                                          +0.00033031147173953081,
                                                          +0.00033024204553020088,
                                                          +0.00033017261249830375,
                                                          +0.00033010324463500891,
                                                          +0.00033003386995774595,
                                                          +0.00032996456036696564,
                                                          +0.00032989524397080176,
                                                          +0.00032982599257913922,
                                                          +0.00032975673439066309,
                                                          +0.00032968754112484459,
                                                          +0.00032961834107076820,
                                                          +0.00032954920585764267,
                                                          +0.00032948006386480087,
                                                          +0.00032941098663134041,
                                                          +0.00032934190262669065,
                                                          +0.00032927288329998966,
                                                          +0.00032920385721061209,
                                                          +0.00032913489571788698,
                                                          +0.00032906592747098377,
                                                          +0.00032899702373957301,
                                                          +0.00032892811326246845,
                                                          +0.00032885926721983218,
                                                          +0.00032879041443997212,
                                                          +0.00032872162601369181,
                                                          +0.00032865283085864368,
                                                          +0.00032858409997642209,
                                                          +0.00032851536237387432,
                                                          +0.00032844668896353502,
                                                          +0.00032837800884129732,
                                                          +0.00032830939283078449,
                                                          +0.00032824077011678674,
                                                          +0.00032817221143416521,
                                                          +0.00032810364605645811,
                                                          +0.00032803514462991266,
                                                          +0.00032796663651666708,
                                                          +0.00032789819227450228,
                                                          +0.00032782974135400886,
                                                          +0.00032776135422464916,
                                                          +0.00032769296042531836,
                                                          +0.00032762463033730746,
                                                          +0.00032755629358766905,
                                                          +0.00032748802046967003,
                                                          +0.00032741974069837315,
                                                          +0.00032735152447916754,
                                                          +0.00032728330161497994,
                                                          +0.00032721514222346852,
                                                          +0.00032714697619527690,
                                                          +0.00032707887356047847,
                                                          +0.00032701076429728787,
                                                          +0.00032694271834833946,
                                                          +0.00032687466577927324,
                                                          +0.00032680667644542974,
                                                          +0.00032673868049972886,
                                                          +0.00032667074771036318,
                                                          +0.00032660280831738682,
                                                          +0.00032653493200198873,
                                                          +0.00032646704909121323,
                                                          +0.00032639922917938999,
                                                          +0.00032633140268040884,
                                                          +0.00032626363910188483,
                                                          +0.00032619586894440860,
                                                          +0.00032612816162902458,
                                                          +0.00032606044774288062,
                                                          +0.00032599279662059386,
                                                          +0.00032592513893572568,
                                                          +0.00032585754393661009,
                                                          +0.00032578994238307793,
                                                          +0.00032572240343732274,
                                                          +0.00032565485794530246,
                                                          +0.00032558737498321306,
                                                          +0.00032551988548299652,
                                                          +0.00032545245843499358,
                                                          +0.00032538502485698803,
                                                          +0.00032531765365360768,
                                                          +0.00032525027592833554,
                                                          +0.00032518296050022886,
                                                          +0.00032511563855832780,
                                                          +0.00032504837883626052,
                                                          +0.00032498111260848304,
                                                          +0.00032491390852333553,
                                                          +0.00032484669794054856,
                                                          +0.00032477954942331541,
                                                          +0.00032471239441650022,
                                                          +0.00032464530139829031,
                                                          +0.00032457820189854244,
                                                          +0.00032451116431057809,
                                                          +0.00032444412024910680,
                                                          +0.00032437713802272426,
                                                          +0.00032431014933085233,
                                                          +0.00032424322239750118,
                                                          +0.00032417628900666487,
                                                          +0.00032410941729790776,
                                                          +0.00032404253913965656,
                                                          +0.00032397572258716892,
                                                          +0.00032390889959316518,
                                                          +0.00032384213812873530,
                                                          +0.00032377537023075420,
                                                          +0.00032370866378628264,
                                                          +0.00032364195091621144,
                                                          +0.00032357529942371113,
                                                          +0.00032350864151354971,
                                                          +0.00032344204490514542,
                                                          +0.00032337544188700549,
                                                          +0.00032330890009493361,
                                                          +0.00032324235190103870,
                                                          +0.00032317586485764745,
                                                          +0.00032310937142033245,
                                                          +0.00032304293905808108,
                                                          +0.00032297650030979239,
                                                          +0.00032291012256125133,
                                                          +0.00032284373843454632,
                                                          +0.00032277741523239676,
                                                          +0.00032271108565994381,
                                                          +0.00032264481693697740,
                                                          +0.00032257854185155534,
                                                          +0.00032251232754067428,
                                                          +0.00032244610687517234,
                                                          +0.00032237994690938903,
                                                          +0.00032231378059680666,
                                                          +0.00032224767490924314,
                                                          +0.00032218156288268978,
                                                          +0.00032211551140657804,
                                                          +0.00032204945359927260,
                                                          +0.00032198345626795416,
                                                          +0.00032191745261322534,
                                                          +0.00032185150936015058,
                                                          +0.00032178555979143624,
                                                          +0.00032171967055016486,
                                                          +0.00032165377500101191,
                                                          +0.00032158793970521227,
                                                          +0.00032152209810927637,
                                                          +0.00032145631669272556,
                                                          +0.00032139052898377097,
                                                          +0.00032132480138035428,
                                                          +0.00032125906749225382,
                                                          +0.00032119339363596464,
                                                          +0.00032112771350269903,
                                                          +0.00032106209332763875,
                                                          +0.00032099646688329682,
                                                          +0.00032093090032367453,
                                                          +0.00032086532750245282,
                                                          +0.00032079981449258495,
                                                          +0.00032073429522878731,
                                                          +0.00032066883570309773,
                                                          +0.00032060336993113545,
                                                          +0.00032053796382415495,
                                                          +0.00032047255147854629,
                                                          +0.00032040719872491249,
                                                          +0.00032034183974028255,
                                                          +0.00032027654027473959,
                                                          +0.00032021123458582016,
                                                          +0.00032014598834321861,
                                                          +0.00032008073588484794,
                                                          +0.00032001554280014450,
                                                          +0.00031995034350726685,
                                                          +0.00031988520351552421,
                                                          +0.00031982005732318993,
                                                          +0.00031975497035957638,
                                                          +0.00031968987720294142,
                                                          +0.00031962484320273107,
                                                          +0.00031955980301705711,
                                                          +0.00031949482191562901,
                                                          +0.00031942983463628286,
                                                          +0.00031936490636912140,
                                                          +0.00031929997193157504,
                                                          +0.00031923509643426939,
                                                          +0.00031917021477409989,
                                                          +0.00031910539198234375,
                                                          +0.00031904056303523257,
                                                          +0.00031897579288482430,
                                                          +0.00031891101658655755,
                                                          +0.00031884629901339951,
                                                          +0.00031878157529986761,
                                                          +0.00031871691023996638,
                                                          +0.00031865223904716363,
                                                          +0.00031858762643662941,
                                                          +0.00031852300770065394,
                                                          +0.00031845844747570079,
                                                          +0.00031839388113275436,
                                                          +0.00031832937322969965,
                                                          +0.00031826485921608767,
                                                          +0.00031820040357135172,
                                                          +0.00031813594182348253,
                                                          +0.00031807153837358878,
                                                          +0.00031800712882797382,
                                                          +0.00031794277750954859,
                                                          +0.00031787842010280211,
                                                          +0.00031781412085257413,
                                                          +0.00031774981552141281,
                                                          +0.00031768556827621326,
                                                          +0.00031762131495745641,
                                                          +0.00031755711965421844,
                                                          +0.00031749291828478735,
                                                          +0.00031742877486054614,
                                                          +0.00031736462537746416,
                                                          +0.00031730053376935664,
                                                          +0.00031723643610974872,
                                                          +0.00031717239625501345,
                                                          +0.00031710835035610619,
                                                          +0.00031704436219208285,
                                                          +0.00031698036799120410,
                                                          +0.00031691643145533363,
                                                          +0.00031685248888991261,
                                                          +0.00031678860391973662,
                                                          +0.00031672471292730329,
                                                          +0.00031666087946046444,
                                                          +0.00031659703997864957,
                                                          +0.00031653325795289074,
                                                          +0.00031646946991942547,
                                                          +0.00031640573927259005,
                                                          +0.00031634200262530612,
                                                          +0.00031627832329533741,
                                                          +0.00031621463797216635,
                                                          +0.00031615100989710774,
                                                          +0.00031608737583608128,
                                                          +0.00031602379895407566,
                                                          +0.00031596021609332515,
                                                          +0.00031589669034261508,
                                                          +0.00031583315862037135,
                                                          +0.00031576968393929857,
                                                          +0.00031570620329389177,
                                                          +0.00031564277962089718,
                                                          +0.00031557934999075667,
                                                          +0.00031551597726438003,
                                                          +0.00031545259858803415,
                                                          +0.00031538927674691381,
                                                          +0.00031532594896298933,
                                                          +0.00031526267794586250,
                                                          +0.00031519940099308488,
                                                          +0.00031513618073878672,
                                                          +0.00031507295455597984,
                                                          +0.00031500978500344375,
                                                          +0.00031494660952952964,
                                                          +0.00031488349061778681,
                                                          +0.00031482036579178515,
                                                          +0.00031475729745996474,
                                                          +0.00031469422322099330,
                                                          +0.00031463120540832170,
                                                          +0.00031456818169559546,
                                                          +0.00031450521434139660,
                                                          +0.00031444224109422814,
                                                          +0.00031437932413792304,
                                                          +0.00031431640129572194,
                                                          +0.00031425353467682843,
                                                          +0.00031419066217910132,
                                                          +0.00031412784583723411,
                                                          +0.00031406502362358430,
                                                          +0.00031400225749845455,
                                                          +0.00031393948550858196,
                                                          +0.00031387676953999725,
                                                          +0.00031381404771369849,
                                                          +0.00031375138184156230,
                                                          +0.00031368871011872945,
                                                          +0.00031362609428304178,
                                                          +0.00031356347260366344,
                                                          +0.00031350090674451967,
                                                          +0.00031343833504868018,
                                                          +0.00031337581910627121,
                                                          +0.00031331329733415042,
                                                          +0.00031325083124876291,
                                                          +0.00031318835934063613,
                                                          +0.00031312594305265152,
                                                          +0.00031306352094888926,
                                                          +0.00031300115439878437,
                                                          +0.00031293878203985227,
                                                          +0.00031287646516819847,
                                                          +0.00031281414249465682,
                                                          +0.00031275187524212043,
                                                          +0.00031268960219462441,
                                                          +0.00031262738450196586,
                                                          +0.00031256516102126506,
                                                          +0.00031250299282933912,
                                                          +0.00031244081885627726,
                                                          +0.00031237870010603308,
                                                          +0.00031231657558154817,
                                                          +0.00031225450621402841,
                                                          +0.00031219243107915223,
                                                          +0.00031213041103549344,
                                                          +0.00031206838523135163,
                                                          +0.00031200641445278381,
                                                          +0.00031194443792059560,
                                                          +0.00031188251634844201,
                                                          +0.00031182058902951969,
                                                          +0.00031175871660519693,
                                                          +0.00031169683844094610,
                                                          +0.00031163501510596367,
                                                          +0.00031157318603788295,
                                                          +0.00031151141173384314,
                                                          +0.00031144963170352396,
                                                          +0.00031138790637212139,
                                                          +0.00031132617532124787,
                                                          +0.00031126449890426985,
                                                          +0.00031120281677461832,
                                                          +0.00031114118921394434,
                                                          +0.00031107955594738353,
                                                          +0.00031101797718498508,
                                                          +0.00031095639272347568,
                                                          +0.00031089486270141620,
                                                          +0.00031083332698701087,
                                                          +0.00031077184564744529,
                                                          +0.00031071035862228840,
                                                          +0.00031064892590746329,
                                                          +0.00031058748751379070,
                                                          +0.00031052610336604396,
                                                          +0.00031046471354618290,
                                                          +0.00031040337790794339,
                                                          +0.00031034203660431216,
                                                          +0.00031028074941809997,
                                                          +0.00031021945657320782,
                                                          +0.00031015821778163365,
                                                          +0.00031009697333808086,
                                                          +0.00031003578288384592,
                                                          +0.00030997458678432325,
                                                          +0.00030991344461021934,
                                                          +0.00030985229679750774,
                                                          +0.00030979120284641698,
                                                          +0.00030973010326338805,
                                                          +0.00030966905747828238,
                                                          +0.00030960800606789766,
                                                          +0.00030954700839183895,
                                                          +0.00030948600509714979,
                                                          +0.00030942505547328981,
                                                          +0.00030936410023743756,
                                                          +0.00030930319860901728,
                                                          +0.00030924229137523247,
                                                          +0.00030918143768558261,
                                                          +0.00030912057839718542,
                                                          +0.00030905977258972563,
                                                          +0.00030899896119012527,
                                                          +0.00030893820320836423,
                                                          +0.00030887743964105917,
                                                          +0.00030881672942859435,
                                                          +0.00030875601363717162,
                                                          +0.00030869535113768934,
                                                          +0.00030863468306582478,
                                                          +0.00030857406822309974,
                                                          +0.00030851344781455779,
                                                          +0.00030845288057245281,
                                                          +0.00030839230777108612,
                                                          +0.00030833178807355239,
                                                          +0.00030827126282330174,
                                                          +0.00030821079061437839,
                                                          +0.00030815031285927265,
                                                          +0.00030808988808308645,
                                                          +0.00030802945776724216,
                                                          +0.00030796908036800766,
                                                          +0.00030790869743562907,
                                                          +0.00030784836735764820,
                                                          +0.00030778803175302705,
                                                          +0.00030772774894068902,
                                                          +0.00030766746060820443,
                                                          +0.00030760722500598541,
                                                          +0.00030754698389010341,
                                                          +0.00030748679544256673,
                                                          +0.00030742660148784035,
                                                          +0.00030736646013963608,
                                                          +0.00030730631329070531,
                                                          +0.00030724621898656988,
                                                          +0.00030718611918816111,
                                                          +0.00030712607187291768,
                                                          +0.00030706601906984384,
                                                          +0.00030700601868840160,
                                                          +0.00030694601282556179,
                                                          +0.00030688605932291613,
                                                          +0.00030682610034529588,
                                                          +0.00030676619366652795,
                                                          +0.00030670628151919838,
                                                          +0.00030664642160947535,
                                                          +0.00030658655623759344,
                                                          +0.00030652674304216778,
                                                          +0.00030646692439097618,
                                                          +0.00030640715785518598,
                                                          +0.00030634738587001275,
                                                          +0.00030628766593928129,
                                                          +0.00030622794056553969,
                                                          +0.00030616826718537522,
                                                          +0.00030610858836856375,
                                                          +0.00030604896148455962,
                                                          +0.00030598932917026147,
                                                          +0.00030592974872809572,
                                                          +0.00030587016286197914,
                                                          +0.00030581062880741434,
                                                          +0.00030575108933523202,
                                                          +0.00030569160161411521,
                                                          +0.00030563210848170438,
                                                          +0.00030557266703996682,
                                                          +0.00030551322019324888,
                                                          +0.00030545382497690593,
                                                          +0.00030539442436188635,
                                                          +0.00030533507531703739,
                                                          +0.00030527572087980586,
                                                          +0.00030521641795263395,
                                                          +0.00030515710963936363,
                                                          +0.00030509785277613546,
                                                          +0.00030503859053308325,
                                                          +0.00030497937968014896,
                                                          +0.00030492016345365537,
                                                          +0.00030486099855744832,
                                                          +0.00030480182829393691,
                                                          +0.00030474270930097374,
                                                          +0.00030468358494695148,
                                                          +0.00030462451180383144,
                                                          +0.00030456543330588815,
                                                          +0.00030450640595929365,
                                                          +0.00030444737326410176,
                                                          +0.00030438839166079782,
                                                          +0.00030432940471511271,
                                                          +0.00030427046880194658,
                                                          +0.00030421152755260602,
                                                          +0.00030415263727650752,
                                                          +0.00030409374167043158,
                                                          +0.00030403489697841251,
                                                          +0.00030397604696260353,
                                                          +0.00030391724780175780,
                                                          +0.00030385844332330006,
                                                          +0.00030379968964080344,
                                                          +0.00030374093064686306,
                                                          +0.00030368222238997286,
                                                          +0.00030362350882779770,
                                                          +0.00030356484594385287,
                                                          +0.00030350617776077235,
                                                          +0.00030344756019719320,
                                                          +0.00030338893734061813,
                                                          +0.00030333036504490613,
                                                          +0.00030327178746232856,
                                                          +0.00030321326038206621,
                                                          +0.00030315472802105916,
                                                          +0.00030309624610391002,
                                                          +0.00030303775891212751,
                                                          +0.00030297932210583575,
                                                          +0.00030292088003101251,
                                                          +0.00030286248828340285,
                                                          +0.00030280409127335427,
                                                          +0.00030274574453233199,
                                                          +0.00030268739253495397,
                                                          +0.00030262909074850447,
                                                          +0.00030257078371177305,
                                                          +0.00030251252682796190,
                                                          +0.00030245426469993337,
                                                          +0.00030239605266690623,
                                                          +0.00030233783539571681,
                                                          +0.00030227966816169901,
                                                          +0.00030222149569556465,
                                                          +0.00030216337320886132,
                                                          +0.00030210524549607783,
                                                          +0.00030204716770507341,
                                                          +0.00030198908469401619,
                                                          +0.00030193105154717454,
                                                          +0.00030187301318629800,
                                                          +0.00030181502463216245,
                                                          +0.00030175703087000061,
                                                          +0.00030169908685719304,
                                                          +0.00030164113764235873,
                                                          +0.00030158323811958033,
                                                          +0.00030152533340076541,
                                                          +0.00030146747831679598,
                                                          +0.00030140961804277103,
                                                          +0.00030135180734646892,
                                                          +0.00030129399146608323,
                                                          +0.00030123622510638517,
                                                          +0.00030117845356856637,
                                                          +0.00030112073149448764,
                                                          +0.00030106300424824164,
                                                          +0.00030100532640887543,
                                                          +0.00030094764340328635,
                                                          +0.00030089000974780402,
                                                          +0.00030083237093203412,
                                                          +0.00030077478140968464,
                                                          +0.00030071718673297391,
                                                          +0.00030065964129308412,
                                                          +0.00030060209070475032,
                                                          +0.00030054458929672456,
                                                          +0.00030048708274616289,
                                                          +0.00030042962531948300,
                                                          +0.00030037216275616621,
                                                          +0.00030031474926039113,
                                                          +0.00030025733063386925,
                                                          +0.00030019996101863502,
                                                          +0.00030014258627853509,
                                                          +0.00030008526049355474,
                                                          +0.00030002792958958084,
                                                          +0.00029997064758464437,
                                                          +0.00029991336046657739,
                                                          +0.00029985612219155129,
                                                          +0.00029979887880924883,
                                                          +0.00029974168421407604,
                                                          +0.00029968448451747224,
                                                          +0.00029962733355217222,
                                                          +0.00029957017749127751,
                                                          +0.00029951307010594595,
                                                          +0.00029945595763084703,
                                                          +0.00029939889377565563,
                                                          +0.00029934182483651545,
                                                          +0.00029928480446171185,
                                                          +0.00029922777900876922,
                                                          +0.00029917080206467674,
                                                          +0.00029911382004824605,
                                                          +0.00029905688648526396,
                                                          +0.00029899994785573567,
                                                          +0.00029894305762433849,
                                                          +0.00029888616233217824,
                                                          +0.00029882931538291588,
                                                          +0.00029877246337866491,
                                                          +0.00029871565966216253,
                                                          +0.00029865885089643717,
                                                          +0.00029860209036339507,
                                                          +0.00029854532478688678,
                                                          +0.00029848860738808006,
                                                          +0.00029843188495155522,
                                                          +0.00029837521063783392,
                                                          +0.00029831853129213388,
                                                          +0.00029826190001442239,
                                                          +0.00029820526371046287,
                                                          +0.00029814867541976046,
                                                          +0.00029809208210853191,
                                                          +0.00029803553675591206,
                                                          +0.00029797898638847932,
                                                          +0.00029792248392508956,
                                                          +0.00029786597645259155,
                                                          +0.00029780951682965373,
                                                          +0.00029775305220330358,
                                                          +0.00029769663537211345,
                                                          +0.00029764021354319821,
                                                          +0.00029758383945512518,
                                                          +0.00029752746037500582,
                                                          +0.00029747112898149307,
                                                          +0.00029741479260160406,
                                                          +0.00029735850385416830,
                                                          +0.00029730221012601776,
                                                          +0.00029724596397624899,
                                                          +0.00029718971285141841,
                                                          +0.00029713350925097997,
                                                          +0.00029707730068112413,
                                                          +0.00029702113958175242,
                                                          +0.00029696497351859921,
                                                          +0.00029690885487210352,
                                                          +0.00029685273126745377,
                                                          +0.00029679665502571644,
                                                          +0.00029674057383144382,
                                                          +0.00029668453994641963,
                                                          +0.00029662850111447051,
                                                          +0.00029657250953818703,
                                                          +0.00029651651302058048,
                                                          +0.00029646056370513744,
                                                          +0.00029640460945396473,
                                                          +0.00029634870235153445,
                                                          +0.00029629279031895950,
                                                          +0.00029623692538178602,
                                                          +0.00029618105552004447,
                                                          +0.00029612523270044431,
                                                          +0.00029606940496184410,
                                                          +0.00029601362421220528,
                                                          +0.00029595783854912636,
                                                          +0.00029590209982190878,
                                                          +0.00029584635618680249,
                                                          +0.00029579065943453772,
                                                          +0.00029573495777992723,
                                                          +0.00029567930295521828,
                                                          +0.00029562364323369825,
                                                          +0.00029556803028921933,
                                                          +0.00029551241245345572,
                                                          +0.00029545684134195242,
                                                          +0.00029540126534468236,
                                                          +0.00029534573601897122,
                                                          +0.00029529020181300302,
                                                          +0.00029523471422597151,
                                                          +0.00029517922176418456,
                                                          +0.00029512377586879087,
                                                          +0.00029506832510413502,
                                                          +0.00029501292085340827,
                                                          +0.00029495751173890417,
                                                          +0.00029490214908594383,
                                                          +0.00029484678157468288,
                                                          +0.00029479146047265873,
                                                          +0.00029473613451780258,
                                                          +0.00029468085491995489,
                                                          +0.00029462557047473540,
                                                          +0.00029457033233437450,
                                                          +0.00029451508935209383,
                                                          +0.00029445989262259992,
                                                          +0.00029440469105663033,
                                                          +0.00029434953569145355,
                                                          +0.00029429437549523686,
                                                          +0.00029423926144789720,
                                                          +0.00029418414257494525,
                                                          +0.00029412906979903224,
                                                          +0.00029407399220292648,
                                                          +0.00029401896065209916,
                                                          +0.00029396392428649039,
                                                          +0.00029390893391447701,
                                                          +0.00029385393873308558,
                                                          +0.00029379898949368383,
                                                          +0.00029374403545029919,
                                                          +0.00029368912729737568,
                                                          +0.00029363421434585647,
                                                          +0.00029357934723334692,
                                                          +0.00029352447532762072,
                                                          +0.00029346964920952954,
                                                          +0.00029341481830359273,
                                                          +0.00029336003313399310,
                                                          +0.00029330524318191097,
                                                          +0.00029325049891494462,
                                                          +0.00029319574987085080,
                                                          +0.00029314104646072802,
                                                          +0.00029308633827882478,
                                                          +0.00029303167567982407,
                                                          +0.00029297700831438194,
                                                          +0.00029292238648085002,
                                                          +0.00029286775988620771,
                                                          +0.00029281317877255942,
                                                          +0.00029275859290312379,
                                                          +0.00029270405246384183,
                                                          +0.00029264950727408788,
                                                          +0.00029259500746372271,
                                                          +0.00029254050290819289,
                                                          +0.00029248604368136284,
                                                          +0.00029243157971466744,
                                                          +0.00029237716102605834,
                                                          +0.00029232273760287540,
                                                          +0.00029226835940724039,
                                                          +0.00029221397648231519,
                                                          +0.00029215963873447493,
                                                          +0.00029210529626262025,
                                                          +0.00029205099891746233,
                                                          +0.00029199669685355797,
                                                          +0.00029194243986603732,
                                                          +0.00029188817816503017,
                                                          +0.00029183396149016848,
                                                          +0.00029177974010707267,
                                                          +0.00029172556369995846,
                                                          +0.00029167138258985467,
                                                          +0.00029161724640564316,
                                                          +0.00029156310552367872,
                                                          +0.00029150900951759170,
                                                          +0.00029145490881898076,
                                                          +0.00029140085294630650,
                                                          +0.00029134679238632946,
                                                          +0.00029129277660242258,
                                                          +0.00029123875613642618,
                                                          +0.00029118478039670738,
                                                          +0.00029113079998010485,
                                                          +0.00029107686424006089,
                                                          +0.00029102292382833126,
                                                          +0.00029096902804351491,
                                                          +0.00029091512759220323,
                                                          +0.00029086127171823315,
                                                          +0.00029080741118295023,
                                                          +0.00029075359517551086,
                                                          +0.00029069977451193343,
                                                          +0.00029064599832677451,
                                                          +0.00029059221749064490,
                                                          +0.00029053848108358173,
                                                          +0.00029048474003070759,
                                                          +0.00029043104335762089,
                                                          +0.00029037734204387519,
                                                          +0.00029032368506071106,
                                                          +0.00029027002344203214,
                                                          +0.00029021640610480147,
                                                          +0.00029016278413719290,
                                                          +0.00029010920640197161,
                                                          +0.00029005562404150169,
                                                          +0.00029000208586443072,
                                                          +0.00028994854306723278,
                                                          +0.00028989504440451774,
                                                          +0.00028984154112678974,
                                                          +0.00028978808193470091,
                                                          +0.00028973461813270549,
                                                          +0.00028968119836757759,
                                                          +0.00028962777399764217,
                                                          +0.00028957439361587423,
                                                          +0.00028952100863439024,
                                                          +0.00028946766759244564,
                                                          +0.00028941432195586902,
                                                          +0.00028936102021027537,
                                                          +0.00028930771387512610,
                                                          +0.00028925445138247487,
                                                          +0.00028920118430533710,
                                                          +0.00028914796102228379,
                                                          +0.00028909473315980543,
                                                          +0.00028904154904306934,
                                                          +0.00028898836035196225,
                                                          +0.00028893521535832632,
                                                          +0.00028888206579536603,
                                                          +0.00028882895988167656,
                                                          +0.00028877584940370212,
                                                          +0.00028872278252686930,
                                                          +0.00028866971109078323,
                                                          +0.00028861668320778017,
                                                          +0.00028856365077054832,
                                                          +0.00028851066183841156,
                                                          +0.00028845766835706305,
                                                          +0.00028840471833289211,
                                                          +0.00028835176376451919,
                                                          +0.00028829885260547655,
                                                          +0.00028824593690723427,
                                                          +0.00028819306457054548,
                                                          +0.00028814018769965203,
                                                          +0.00028808735414260507,
                                                          +0.00028803451605634127,
                                                          +0.00028798172123628709,
                                                          +0.00028792892189199624,
                                                          +0.00028787616576634808,
                                                          +0.00028782340512143638,
                                                          +0.00028777068764766989,
                                                          +0.00028771796565960566,
                                                          +0.00028766528679525886,
                                                          +0.00028761260342157277,
                                                          +0.00028755996312424594,
                                                          +0.00028750731832253093,
                                                          +0.00028745471654988613,
                                                          +0.00028740211027779723,
                                                          +0.00028734954698755871,
                                                          +0.00028729697920281295,
                                                          +0.00028724445435276659,
                                                          +0.00028719192501314259,
                                                          +0.00028713943856113625,
                                                          +0.00028708694762447448,
                                                          +0.00028703449952841752,
                                                          +0.00028698204695262026,
                                                          +0.00028692963717048343,
                                                          +0.00028687722291351425,
                                                          +0.00028682485140332976,
                                                          +0.00028677247542321373,
                                                          +0.00028672014214307511,
                                                          +0.00028666780439789849,
                                                          +0.00028661550930596041,
                                                          +0.00028656320975387089,
                                                          +0.00028651095280834884,
                                                          +0.00028645869140755498,
                                                          +0.00028640647256672567,
                                                          +0.00028635424927549679,
                                                          +0.00028630206849769785,
                                                          +0.00028624988327436427,
                                                          +0.00028619774051799374,
                                                          +0.00028614559332094670,
                                                          +0.00028609348854446336,
                                                          +0.00028604137933215451,
                                                          +0.00028598931249407760,
                                                          +0.00028593724122501912,
                                                          +0.00028588521228392833,
                                                          +0.00028583317891669281,
                                                          +0.00028578118783122804,
                                                          +0.00028572919232444844,
                                                          +0.00028567723905330972,
                                                          +0.00028562528136567904,
                                                          +0.00028557336586762663,
                                                          +0.00028552144595789789,
                                                          +0.00028546956819175184,
                                                          +0.00028541768601873834,
                                                          +0.00028536584594337849,
                                                          +0.00028531400146595301,
                                                          +0.00028526219904031911,
                                                          +0.00028521039221741441,
                                                          +0.00028515862740050566,
                                                          +0.00028510685819111402,
                                                          +0.00028505513094198928,
                                                          +0.00028500339930516254,
                                                          +0.00028495170958293991,
                                                          +0.00028490001547778934,
                                                          +0.00028484836324164638,
                                                          +0.00028479670662734250,
                                                          +0.00028474509183651581,
                                                          +0.00028469347267228844,
                                                          +0.00028464189528607387,
                                                          +0.00028459031353121183,
                                                          +0.00028453877350896394,
                                                          +0.00028448722912281511,
                                                          +0.00028443572642394767,
                                                          +0.00028438421936591865,
                                                          +0.00028433275394990402,
                                                          +0.00028428128417946034,
                                                          +0.00028422985600582947,
                                                          +0.00028417842348249538,
                                                          +0.00028412703251083778,
                                                          +0.00028407563719419607,
                                                          +0.00028402428338415969,
                                                          +0.00028397292523385130,
                                                          +0.00028392160854514274,
                                                          +0.00028387028752086734,
                                                          +0.00028381900791325093,
                                                          +0.00028376772397476646,
                                                          +0.00028371648140806493,
                                                          +0.00028366523451518706,
                                                          +0.00028361402894928119,
                                                          +0.00028356281906188381,
                                                          +0.00028351165045671233,
                                                          +0.00028346047753472761,
                                                          +0.00028340934585028671,
                                                          +0.00028335820985370407,
                                                          +0.00028330711505004816,
                                                          +0.00028325601593891513,
                                                          +0.00028320495797615588,
                                                          +0.00028315389571057756,
                                                          +0.00028310287454888419,
                                                          +0.00028305184908902303,
                                                          +0.00028300086468862218,
                                                          +0.00028294987599469833,
                                                          +0.00028289892831587395,
                                                          +0.00028284797634816445,
                                                          +0.00028279706535125782,
                                                          +0.00028274615007009714,
                                                          +0.00028269527571550649,
                                                          +0.00028264439708128628,
                                                          +0.00028259355932946668,
                                                          +0.00028254271730263552,
                                                          +0.00028249191611409913,
                                                          +0.00028244111065516242,
                                                          +0.00028239034599047811,
                                                          +0.00028233957705999820,
                                                          +0.00028228884887979155,
                                                          +0.00028223811643838728,
                                                          +0.00028218742470334041,
                                                          +0.00028213672871168724,
                                                          +0.00028208607338253889,
                                                          +0.00028203541380136896,
                                                          +0.00028198479483891397,
                                                          +0.00028193417162901565,
                                                          +0.00028188358899410520,
                                                          +0.00028183300211632317,
                                                          +0.00028178245576986484,
                                                          +0.00028173190518509996,
                                                          +0.00028168139508805720,
                                                          +0.00028163088075726643,
                                                          +0.00028158040687065866,
                                                          +0.00028152992875485496,
                                                          +0.00028147949103975758,
                                                          +0.00028142904910000963,
                                                          +0.00028137864751755378,
                                                          +0.00028132824171498631,
                                                          +0.00028127787622635864,
                                                          +0.00028122750652215196,
                                                          +0.00028117717708859479,
                                                          +0.00028112684344398461,
                                                          +0.00028107655002679598,
                                                          +0.00028102625240307373,
                                                          +0.00028097599496360666,
                                                          +0.00028092573332211908,
                                                          +0.00028087551182178204,
                                                          +0.00028082528612393109,
                                                          +0.00028077510052418775,
                                                          +0.00028072491073143069,
                                                          +0.00028067476099379968,
                                                          +0.00028062460706764869,
                                                          +0.00028057449315370392,
                                                          +0.00028052437505572626,
                                                          +0.00028047429692709610,
                                                          +0.00028042421461891399,
                                                          +0.00028037417223728185,
                                                          +0.00028032412568057232,
                                                          +0.00028027411900767612,
                                                          +0.00028022410816417069,
                                                          +0.00028017413716180324,
                                                          +0.00028012416199328823,
                                                          +0.00028007422662329647,
                                                          +0.00028002428709161261,
                                                          +0.00027997438731589810,
                                                          +0.00027992448338294058,
                                                          +0.00027987461916345911,
                                                          +0.00027982475079117724,
                                                          +0.00027977492208993880,
                                                          +0.00027972508924033635,
                                                          +0.00027967529601940501,
                                                          +0.00027962549865453973,
                                                          +0.00027957574087603360,
                                                          +0.00027952597895801728,
                                                          +0.00027947625658410842,
                                                          +0.00027942653007510663,
                                                          +0.00027937684306802085,
                                                          +0.00027932715193025343,
                                                          +0.00027927750025227009,
                                                          +0.00027922784444801005,
                                                          +0.00027917822806146250,
                                                          +0.00027912860755303687,
                                                          +0.00027907902642031163,
                                                          +0.00027902944117010075,
                                                          +0.00027897989525363815,
                                                          +0.00027893034522407593,
                                                          +0.00027888083448636923,
                                                          +0.00027883131963994282,
                                                          +0.00027878184404353886,
                                                          +0.00027873236434278881,
                                                          +0.00027868292385028738,
                                                          +0.00027863347925780735,
                                                          +0.00027858407383186132,
                                                          +0.00027853466431029795,
                                                          +0.00027848529391361315,
                                                          +0.00027843591942566614,
                                                          +0.00027838658402100131,
                                                          +0.00027833724452942327,
                                                          +0.00027828794407958990,
                                                          +0.00027823863954718614,
                                                          +0.00027818937401504829,
                                                          +0.00027814010440467676,
                                                          +0.00027809087375315142,
                                                          +0.00027804163902772275,
                                                          +0.00027799244321977898,
                                                          +0.00027794324334225629,
                                                          +0.00027789408234091582,
                                                          +0.00027784491727431474,
                                                          +0.00027779579104265147,
                                                          +0.00027774666075003964,
                                                          +0.00027769756925117981,
                                                          +0.00027764847369567738,
                                                          +0.00027759941689279932,
                                                          +0.00027755035603757859,
                                                          +0.00027750133389391250,
                                                          +0.00027745230770219762,
                                                          +0.00027740332018102580,
                                                          +0.00027735432861609300,
                                                          +0.00027730537568074952,
                                                          +0.00027725641870592684,
                                                          +0.00027720750031979765,
                                                          +0.00027715857789846504,
                                                          +0.00027710969402498743,
                                                          +0.00027706080612057614,
                                                          +0.00027701195672323955,
                                                          +0.00027696310329923285,
                                                          +0.00027691428834157751,
                                                          +0.00027686546936150984,
                                                          +0.00027681668880712802,
                                                          +0.00027676790423458535,
                                                          +0.00027671915804712018,
                                                          +0.00027667040784573984,
                                                          +0.00027662169598888583,
                                                          +0.00027657298012235637,
                                                          +0.00027652430255985900,
                                                          +0.00027647562099192003,
                                                          +0.00027642697768757594,
                                                          +0.00027637833038201805,
                                                          +0.00027632972129967501,
                                                          +0.00027628110822033983,
                                                          +0.00027623253332389612,
                                                          +0.00027618395443467620,
                                                          +0.00027613541368808097,
                                                          +0.00027608686895291953,
                                                          +0.00027603836232017271,
                                                          +0.00027598985170306368,
                                                          +0.00027594137914821568,
                                                          +0.00027589290261320360,
                                                          +0.00027584446410035534,
                                                          +0.00027579602161153518,
                                                          +0.00027574761710483800,
                                                          +0.00027569920862635525,
                                                          +0.00027565083809001086,
                                                          +0.00027560246358606131,
                                                          +0.00027555412698432151,
                                                          +0.00027550578641915121,
                                                          +0.00027545748371631809,
                                                          +0.00027540917705422307,
                                                          +0.00027536090821464866,
                                                          +0.00027531263541997537,
                                                          +0.00027526440040806169,
                                                          +0.00027521616144520616,
                                                          +0.00027516796022540520,
                                                          +0.00027511975505881364,
                                                          +0.00027507158759562704,
                                                          +0.00027502341618979530,
                                                          +0.00027497528244777448,
                                                          +0.00027492714476724811,
                                                          +0.00027487904471099419,
                                                          +0.00027483094072036852,
                                                          +0.00027478287431453196,
                                                          +0.00027473480397845174,
                                                          +0.00027468677118773255,
                                                          +0.00027463873447089199,
                                                          +0.00027459073526003955,
                                                          +0.00027454273212718230,
                                                          +0.00027449476646099522,
                                                          +0.00027444679687691403,
                                                          +0.00027439886472024021,
                                                          +0.00027435092864977733,
                                                          +0.00027430302996751354,
                                                          +0.00027425512737556002,
                                                          +0.00027420726213265233,
                                                          +0.00027415939298414837,
                                                          +0.00027411156114559153,
                                                          +0.00027406372540552629,
                                                          +0.00027401592693636410,
                                                          +0.00027396812456977559,
                                                          +0.00027392035943510042,
                                                          +0.00027387259040707537,
                                                          +0.00027382485857202840,
                                                          +0.00027377712284770245,
                                                          +0.00027372942427747313,
                                                          +0.00027368172182203000,
                                                          +0.00027363405648185698,
                                                          +0.00027358638726052940,
                                                          +0.00027353875511569901,
                                                          +0.00027349111909376789,
                                                          +0.00027344352010961521,
                                                          +0.00027339591725240988,
                                                          +0.00027334835139431807,
                                                          +0.00027330078166721617,
                                                          +0.00027325324890061664,
                                                          +0.00027320571226904382,
                                                          +0.00027315821255941605,
                                                          +0.00027311070898884633,
                                                          +0.00027306324230171768,
                                                          +0.00027301577175767278,
                                                          +0.00027296833805861870,
                                                          +0.00027292090050666843,
                                                          +0.00027287349976131211,
                                                          +0.00027282609516707414,
                                                          +0.00027277872734108650,
                                                          +0.00027273135567022614,
                                                          +0.00027268402072932590,
                                                          +0.00027263668194755619,
                                                          +0.00027258937985750954,
                                                          +0.00027254207393059122,
                                                          +0.00027249480465721176,
                                                          +0.00027244753155095287,
                                                          +0.00027240029506010185,
                                                          +0.00027235305474035805,
                                                          +0.00027230585099794386,
                                                          +0.00027225864343061813,
                                                          +0.00027221147240259644,
                                                          +0.00027216429755363888,
                                                          +0.00027211715920601268,
                                                          +0.00027207001704142070,
                                                          +0.00027202291134023995,
                                                          +0.00027197580182605806,
                                                          +0.00027192872873741964,
                                                          +0.00027188165183973932,
                                                          +0.00027183461132978723,
                                                          +0.00027178756701474679,
                                                          +0.00027174055904967187,
                                                          +0.00027169354728345682,
                                                          +0.00027164657182949636,
                                                          +0.00027159959257833864,
                                                          +0.00027155264960177704,
                                                          +0.00027150570283195541,
                                                          +0.00027145879229912342,
                                                          +0.00027141187797696321,
                                                          +0.00027136499985423826,
                                                          +0.00027131811794611153,
                                                          +0.00027127127219991719,
                                                          +0.00027122442267224224,
                                                          +0.00027117760926904878,
                                                          +0.00027113079208829015,
                                                          +0.00027108401099461412,
                                                          +0.00027103722612728309,
                                                          +0.00027099047730968687,
                                                          +0.00027094372472234057,
                                                          +0.00027089700814743306,
                                                          +0.00027085028780667487,
                                                          +0.00027080360344111084,
                                                          +0.00027075691531359009,
                                                          +0.00027071026312407040,
                                                          +0.00027066360717648257,
                                                          +0.00027061698712975370,
                                                          +0.00027057036332884008,
                                                          +0.00027052377539169459,
                                                          +0.00027047718370424228,
                                                          +0.00027043062784351833,
                                                          +0.00027038406823636028,
                                                          +0.00027033754441894157,
                                                          +0.00027029101685895605,
                                                          +0.00027024452505177224,
                                                          +0.00027019802950588339,
                                                          +0.00027015156967590925,
                                                          +0.00027010510611108674,
                                                          +0.00027005867822534258,
                                                          +0.00027001224660860141,
                                                          +0.00026996585063415274,
                                                          +0.00026991945093255332,
                                                          +0.00026987308683651119,
                                                          +0.00026982671901715899,
                                                          +0.00026978038676667940,
                                                          +0.00026973405079672527,
                                                          +0.00026968775035900953,
                                                          +0.00026964144620564946,
                                                          +0.00026959517754794360,
                                                          +0.00026954890517841848,
                                                          +0.00026950266826801381,
                                                          +0.00026945642764960947,
                                                          +0.00026941022245384206,
                                                          +0.00026936401355388937,
                                                          +0.00026931784004013992,
                                                          +0.00026927166282601452,
                                                          +0.00026922552096170856,
                                                          +0.00026917937540083079,
                                                          +0.00026913326515343844,
                                                          +0.00026908715121327318,
                                                          +0.00026904107255030932,
                                                          +0.00026899499019836617,
                                                          +0.00026894894308739000,
                                                          +0.00026890289229122296,
                                                          +0.00026885687669983815,
                                                          +0.00026881085742704562,
                                                          +0.00026876487332290011,
                                                          +0.00026871888554112500,
                                                          +0.00026867293289191104,
                                                          +0.00026862697656884039,
                                                          +0.00026858105534229444,
                                                          +0.00026853513044565940,
                                                          +0.00026848924060956202,
                                                          +0.00026844334710713811,
                                                          +0.00026839748862931379,
                                                          +0.00026835162648892048,
                                                          +0.00026830579933723766,
                                                          +0.00026825996852673821,
                                                          +0.00026821417266910950,
                                                          +0.00026816837315641137,
                                                          +0.00026812260856079283,
                                                          +0.00026807684031384695,
                                                          +0.00026803110694823863,
                                                          +0.00026798536993503989,
                                                          +0.00026793966776748532,
                                                          +0.00026789396195607231,
                                                          +0.00026784829095465875,
                                                          +0.00026780261631311343,
                                                          +0.00026775697644597162,
                                                          +0.00026771133294241969,
                                                          +0.00026766572417772369,
                                                          +0.00026762011178033427,
                                                          +0.00026757453408630163,
                                                          +0.00026752895276328711,
                                                          +0.00026748340610817861,
                                                          +0.00026743785582779479,
                                                          +0.00026739234017991438,
                                                          +0.00026734682091046012,
                                                          +0.00026730133623815500,
                                                          +0.00026725584794797251,
                                                          +0.00026721039421963294,
                                                          +0.00026716493687710727,
                                                          +0.00026711951406116648,
                                                          +0.00026707408763472598,
                                                          +0.00026702869569965998,
                                                          +0.00026698330015777568,
                                                          +0.00026693793907210352,
                                                          +0.00026689257438328938,
                                                          +0.00026684724411557288,
                                                          +0.00026680191024838578,
                                                          +0.00026675661076722926,
                                                          +0.00026671130769026860,
                                                          +0.00026666603896431928,
                                                          +0.00026662076664622715,
                                                          +0.00026657552864417462,
                                                          +0.00026653028705363570,
                                                          +0.00026648507974421205,
                                                          +0.00026643986884995361,
                                                          +0.00026639469220193351,
                                                          +0.00026634951197272499,
                                                          +0.00026630436595492530,
                                                          +0.00026625921635957877,
                                                          +0.00026621410094085856,
                                                          +0.00026616898194822824,
                                                          +0.00026612389709748896,
                                                          +0.00026607880867647121,
                                                          +0.00026603375436265637,
                                                          +0.00026598869648218989,
                                                          +0.00026594367267428491,
                                                          +0.00026589864530335029,
                                                          +0.00026585365197038271,
                                                          +0.00026580865507800256,
                                                          +0.00026576369218904195,
                                                          +0.00026571872574428072,
                                                          +0.00026567379326843837,
                                                          +0.00026562885724040254,
                                                          +0.00026558395514683154,
                                                          +0.00026553904950466932,
                                                          +0.00026549417776256423,
                                                          +0.00026544930247546559,
                                                          +0.00026540446105406288,
                                                          +0.00026535961609125923,
                                                          +0.00026531480495983696,
                                                          +0.00026526999029060147,
                                                          +0.00026522520941847893,
                                                          +0.00026518042501212619,
                                                          +0.00026513567436866443,
                                                          +0.00026509092019455033,
                                                          +0.00026504619974915153,
                                                          +0.00026500147577667354,
                                                          +0.00026495678549878120,
                                                          +0.00026491209169737817,
                                                          +0.00026486743155647682,
                                                          +0.00026482276789562838,
                                                          +0.00026477813786124414,
                                                          +0.00026473350431047154,
                                                          +0.00026468890435217112,
                                                          +0.00026464430088103647,
                                                          +0.00026459973096842784,
                                                          +0.00026455515754653423,
                                                          +0.00026451061764926634,
                                                          +0.00026446607424625784,
                                                          +0.00026442156433402041,
                                                          +0.00026437705091958191,
                                                          +0.00026433257096210544,
                                                          +0.00026428808750596293,
                                                          +0.00026424363747301865,
                                                          +0.00026419918394493850,
                                                          +0.00026415476380633828,
                                                          +0.00026411034017612763,
                                                          +0.00026406594990172408,
                                                          +0.00026402155613923072,
                                                          +0.00026397719569891685,
                                                          +0.00026393283177402914,
                                                          +0.00026388850113773839,
                                                          +0.00026384416702038494,
                                                          +0.00026379986615809127,
                                                          +0.00026375556181824135,
                                                          +0.00026371129069995887,
                                                          +0.00026366701610762194,
                                                          +0.00026362277470340514,
                                                          +0.00026357852982863104,
                                                          +0.00026353431810857448,
                                                          +0.00026349010292145300,
                                                          +0.00026344592085569151,
                                                          +0.00026340173532635273,
                                                          +0.00026335758288506121,
                                                          +0.00026331342698367535,
                                                          +0.00026326930413706829,
                                                          +0.00026322517783384537,
                                                          +0.00026318108455217768,
                                                          +0.00026313698781736786,
                                                          +0.00026309292407093401,
                                                          +0.00026304885687482700,
                                                          +0.00026300482263396133,
                                                          +0.00026296078494688691,
                                                          +0.00026291678018196355,
                                                          +0.00026287277197429117,
                                                          +0.00026282879665572359,
                                                          +0.00026278481789786219,
                                                          +0.00026274087199610393,
                                                          +0.00026269692265850226,
                                                          +0.00026265300614404595,
                                                          +0.00026260908619719214,
                                                          +0.00026256519904057003,
                                                          +0.00026252130845499174,
                                                          +0.00026247745062677553,
                                                          +0.00026243358937303981,
                                                          +0.00026238976084384032,
                                                          +0.00026234592889255346,
                                                          +0.00026230212963302102,
                                                          +0.00026225832695482868,
                                                          +0.00026221455693565263,
                                                          +0.00026217078350123957,
                                                          +0.00026212704269314841,
                                                          +0.00026208329847323862,
                                                          +0.00026203958684699991,
                                                          +0.00026199587181235647,
                                                          +0.00026195218933877673,
                                                          +0.00026190850346020158,
                                                          +0.00026186485011012628,
                                                          +0.00026182119335846029,
                                                          +0.00026177756910277395,
                                                          +0.00026173394144889692,
                                                          +0.00026169034625852253,
                                                          +0.00026164674767335317,
                                                          +0.00026160318151925271,
                                                          +0.00026155961197374834,
                                                          +0.00026151607482692221,
                                                          +0.00026147253429207896,
                                                          +0.00026142902612356634,
                                                          +0.00026138551457041866,
                                                          +0.00026134203535129738,
                                                          +0.00026129855275091858,
                                                          +0.00026125510245230473,
                                                          +0.00026121164877580637,
                                                          +0.00026116822736885453,
                                                          +0.00026112480258738685,
                                                          +0.00026108141004328993,
                                                          +0.00026103801412804135,
                                                          +0.00026099465041803049,
                                                          +0.00026095128334022763,
                                                          +0.00026090794843557243,
                                                          +0.00026086461016648030,
                                                          +0.00026082130403848824,
                                                          +0.00026077799454941013,
                                                          +0.00026073471716942686,
                                                          +0.00026069143643170389,
                                                          +0.00026064818777111315,
                                                          +0.00026060493575612455,
                                                          +0.00026056171578634803,
                                                          +0.00026051849246551119,
                                                          +0.00026047530115800845,
                                                          +0.00026043210650277833,
                                                          +0.00026038894382904683,
                                                          +0.00026034577781091657,
                                                          +0.00026030264374249151,
                                                          +0.00026025950633299180,
                                                          +0.00026021640084144621,
                                                          +0.00026017329201214563,
                                                          +0.00026013021506908994,
                                                          +0.00026008713479159471,
                                                          +0.00026004408636867704,
                                                          +0.00026000103461463103,
                                                          +0.00025995801468353712,
                                                          +0.00025991499142462147,
                                                          +0.00025987199995707451,
                                                          +0.00025982900516500801,
                                                          +0.00025978604213276871,
                                                          +0.00025974307577930758,
                                                          +0.00025970014115417366,
                                                          +0.00025965720321111151,
                                                          +0.00025961429696491828,
                                                          +0.00025957138740408597,
                                                          +0.00025952850950870576,
                                                          +0.00025948562830197132,
                                                          +0.00025944277872931375,
                                                          +0.00025939992584858243,
                                                          +0.00025935710457059414,
                                                          +0.00025931427998780829,
                                                          +0.00025927148697647308,
                                                          +0.00025922869066361199,
                                                          +0.00025918592589095049,
                                                          +0.00025914315782003057,
                                                          +0.00025910042125810041,
                                                          +0.00025905768140117500,
                                                          +0.00025901497302207059,
                                                          +0.00025897226135122983,
                                                          +0.00025892958112708246,
                                                          +0.00025888689761445325,
                                                          +0.00025884424551743087,
                                                          +0.00025880159013517696,
                                                          +0.00025875896613748424,
                                                          +0.00025871633885780600,
                                                          +0.00025867374293168412,
                                                          +0.00025863114372681853,
                                                          +0.00025858857584454536,
                                                          +0.00025854600468676591,
                                                          +0.00025850346482065580,
                                                          +0.00025846092168227257,
                                                          +0.00025841840980467615,
                                                          +0.00025837589465803562,
                                                          +0.00025833341074134010,
                                                          +0.00025829092355882512,
                                                          +0.00025824846757545387,
                                                          +0.00025820600832948362,
                                                          +0.00025816358025189628,
                                                          +0.00025812114891492628,
                                                          +0.00025807874871561871,
                                                          +0.00025803634526014036,
                                                          +0.00025799397291164467,
                                                          +0.00025795159731018598,
                                                          +0.00025790925278507007,
                                                          +0.00025786690501019481,
                                                          +0.00025782458828106284,
                                                          +0.00025778226830537083,
                                                          +0.00025773997934486285,
                                                          +0.00025769768714098998,
                                                          +0.00025765542592178186,
                                                          +0.00025761316146239994,
                                                          +0.00025757092795720347,
                                                          +0.00025752869121501999,
                                                          +0.00025748648539658270,
                                                          +0.00025744427634434103,
                                                          +0.00025740209818544621,
                                                          +0.00025735991679592550,
                                                          +0.00025731776626939203,
                                                          +0.00025727561251540700,
                                                          +0.00025723348959408955,
                                                          +0.00025719136344849042,
                                                          +0.00025714926810527900,
                                                          +0.00025710716954095193,
                                                          +0.00025706510174877196,
                                                          +0.00025702303073863830,
                                                          +0.00025698099047045091,
                                                          +0.00025693894698746751,
                                                          +0.00025689693421626907,
                                                          +0.00025685491823342809,
                                                          +0.00025681293293225020,
                                                          +0.00025677094442257934,
                                                          +0.00025672898656448886,
                                                          +0.00025668702550105072,
                                                          +0.00025664509505914996,
                                                          +0.00025660316141504262,
                                                          +0.00025656125836246883,
                                                          +0.00025651935211082550,
                                                          +0.00025647747642075085,
                                                          +0.00025643559753473979,
                                                          +0.00025639374918037182,
                                                          +0.00025635189763319644,
                                                          +0.00025631007658777723,
                                                          +0.00025626825235267552,
                                                          +0.00025622645858948265,
                                                          +0.00025618466163972799,
                                                          +0.00025614289513207328,
                                                          +0.00025610112544097359,
                                                          +0.00025605938616220415,
                                                          +0.00025601764370310236,
                                                          +0.00025597593162659968,
                                                          +0.00025593421637287321,
                                                          +0.00025589253147205378,
                                                          +0.00025585084339711507,
                                                          +0.00025580918564542976,
                                                          +0.00025576752472272556,
                                                          +0.00025572589409365994,
                                                          +0.00025568426029667191,
                                                          +0.00025564265676374597,
                                                          +0.00025560105006598998,
                                                          +0.00025555947360275823,
                                                          +0.00025551789397778484,
                                                          +0.00025547634455783618,
                                                          +0.00025543479197923032,
                                                          +0.00025539326957618795,
                                                          +0.00025535174401756882,
                                                          +0.00025531024860509031,
                                                          +0.00025526875004011135,
                                                          +0.00025522728159188855,
                                                          +0.00025518580999423754,
                                                          +0.00025514436848399647,
                                                          +0.00025510292382739544,
                                                          +0.00025506150922889598,
                                                          +0.00025502009148710111,
                                                          +0.00025497870377413743,
                                                          +0.00025493731292093873,
                                                          +0.00025489595206733913,
                                                          +0.00025485458807656086,
                                                          +0.00025481325405618733,
                                                          +0.00025477191690168776,
                                                          +0.00025473060968843634,
                                                          +0.00025468929934410734,
                                                          +0.00025464801891190796,
                                                          +0.00025460673535167554,
                                                          +0.00025456548167449185,
                                                          +0.00025452422487231576,
                                                          +0.00025448299792414511,
                                                          +0.00025444176785401870,
                                                          +0.00025440056760889235,
                                                          +0.00025435936424484296,
                                                          +0.00025431819067682536,
                                                          +0.00025427701399291362,
                                                          +0.00025423586707610325,
                                                          +0.00025419471704642350,
                                                          +0.00025415359675495217,
                                                          +0.00025411247335363252,
                                                          +0.00025407137966166542,
                                                          +0.00025403028286286711,
                                                          +0.00025398921574460294,
                                                          +0.00025394814552252083,
                                                          +0.00025390710495219162,
                                                          +0.00025386606128105369,
                                                          +0.00025382504723292504,
                                                          +0.00025378403008699291,
                                                          +0.00025374304253536316,
                                                          +0.00025370205188893154,
                                                          +0.00025366109080813258,
                                                          +0.00025362012663552940,
                                                          +0.00025357919199992619,
                                                          +0.00025353825427551247,
                                                          +0.00025349734605950298,
                                                          +0.00025345643475767305,
                                                          +0.00025341555293568839,
                                                          +0.00025337466803086934,
                                                          +0.00025333381257737361,
                                                          +0.00025329295404402565,
                                                          +0.00025325212493351582,
                                                          +0.00025321129274613206,
                                                          +0.00025317048995313806,
                                                          +0.00025312968408624451,
                                                          +0.00025308890758532905,
                                                          +0.00025304812801348460,
                                                          +0.00025300737777924303,
                                                          +0.00025296662447703945,
                                                          +0.00025292590048409997,
                                                          +0.00025288517342616151,
                                                          +0.00025284447564918492,
                                                          +0.00025280377481016862,
                                                          +0.00025276310322384850,
                                                          +0.00025272242857844398,
                                                          +0.00025268178315750628,
                                                          +0.00025264113468043581,
                                                          +0.00025260051539963911,
                                                          +0.00025255989306565752,
                                                          +0.00025251929989979270,
                                                          +0.00025247870368368705,
                                                          +0.00025243813660757758,
                                                          +0.00025239756648416748,
                                                          +0.00025235702547266915,
                                                          +0.00025231648141680653,
                                                          +0.00025227596644480737,
                                                          +0.00025223544843137663,
                                                          +0.00025219495947379682,
                                                          +0.00025215446747771440,
                                                          +0.00025211400450950660,
                                                          +0.00025207353850572125,
                                                          +0.00025203310150186998,
                                                          +0.00025199266146536255,
                                                          +0.00025195225040088460,
                                                          +0.00025191183630666808,
                                                          +0.00025187145115661221,
                                                          +0.00025183106297973172,
                                                          +0.00025179070371917858,
                                                          +0.00025175034143471104,
                                                          +0.00025171000803877346,
                                                          +0.00025166967162182797,
                                                          +0.00025162936406565036,
                                                          +0.00025158905349136773,
                                                          +0.00025154877175012673,
                                                          +0.00025150848699367955,
                                                          +0.00025146823104258326,
                                                          +0.00025142797207917619,
                                                          +0.00025138774189346457,
                                                          +0.00025134750869833375,
                                                          +0.00025130730425327848,
                                                          +0.00025126709680169178,
                                                          +0.00025122691807259620,
                                                          +0.00025118673633985338,
                                                          +0.00025114658330205216,
                                                          +0.00025110642726348432,
                                                          +0.00025106629989234385,
                                                          +0.00025102616952331358,
                                                          +0.00025098606779423190,
                                                          +0.00025094596307013338,
                                                          +0.00025090588695853971,
                                                          +0.00025086580785479875,
                                                          +0.00025082575733615368,
                                                          +0.00025078570382822720,
                                                          +0.00025074567887802283,
                                                          +0.00025070565094139914,
                                                          +0.00025066565153515874,
                                                          +0.00025062564914535756,
                                                          +0.00025058567525863563,
                                                          +0.00025054569839120779,
                                                          +0.00025050574999959009,
                                                          +0.00025046579863011761,
                                                          +0.00025042587570922099,
                                                          +0.00025038594981331729,
                                                          +0.00025034605233878948,
                                                          +0.00025030615189209862,
                                                          +0.00025026627983961878,
                                                          +0.00025022640481781614,
                                                          +0.00025018655816309426,
                                                          +0.00025014670854188621,
                                                          +0.00025010688726066305,
                                                          +0.00025006706301578685,
                                                          +0.00025002726708383416,
                                                          +0.00024998746819105799,
                                                          +0.00024994769758417834,
                                                          +0.00024990792401930118,
                                                          +0.00024986817871332809,
                                                          +0.00024982843045217966,
                                                          +0.00024978871042297720,
                                                          +0.00024974898744141807,
                                                          +0.00024970929266488102,
                                                          +0.00024966959493880243,
                                                          +0.00024962992539085630,
                                                          +0.00024959025289618013,
                                                          +0.00024955060855278094,
                                                          +0.00024951096126545969,
                                                          +0.00024947134210259394,
                                                          +0.00024943171999861069,
                                                          +0.00024939212599229551,
                                                          +0.00024935252904766364,
                                                          +0.00024931296017394666,
                                                          +0.00024927338836471029,
                                                          +0.00024923384459966935,
                                                          +0.00024919429790190274,
                                                          +0.00024915477922164610,
                                                          +0.00024911525761145404,
                                                          +0.00024907576399212040,
                                                          +0.00024903626744563791,
                                                          +0.00024899679886339603,
                                                          +0.00024895732735678833,
                                                          +0.00024891788378783732,
                                                          +0.00024887843729730011,
                                                          +0.00024883901871786905,
                                                          +0.00024879959721962788,
                                                          +0.00024876020360597621,
                                                          +0.00024872080707628695,
                                                          +0.00024868143840470387,
                                                          +0.00024864206681985237,
                                                          +0.00024860272306665744,
                                                          +0.00024856337640295951,
                                                          +0.00024852405754450206,
                                                          +0.00024848473577830362,
                                                          +0.00024844544179096289,
                                                          +0.00024840614489863970,
                                                          +0.00024836687575882481,
                                                          +0.00024832760371678263,
                                                          +0.00024828835940093256,
                                                          +0.00024824911218560681,
                                                          +0.00024820989267019026,
                                                          +0.00024817067025804625,
                                                          +0.00024813147551956184,
                                                          +0.00024809227788709452,
                                                          +0.00024805310790207034,
                                                          +0.00024801393502580445,
                                                          +0.00024797478977079838,
                                                          +0.00024793564162728821,
                                                          +0.00024789652107888767,
                                                          +0.00024785739764471727,
                                                          +0.00024781830177953909,
                                                          +0.00024777920303132184,
                                                          +0.00024774013182601255,
                                                          +0.00024770105774039148,
                                                          +0.00024766201117162699,
                                                          +0.00024762296172527463,
                                                          +0.00024758393976976017,
                                                          +0.00024754491493937837,
                                                          +0.00024750591757384858,
                                                          +0.00024746691733616857,
                                                          +0.00024742794453738747,
                                                          +0.00024738896886916976,
                                                          +0.00024735002061393056,
                                                          +0.00024731106949196498,
                                                          +0.00024727214575709019,
                                                          +0.00024723321915819579,
                                                          +0.00024719431992053701,
                                                          +0.00024715541782156200,
                                                          +0.00024711654305799996,
                                                          +0.00024707766543582174,
                                                          +0.00024703881512326634,
                                                          +0.00024699996195479129,
                                                          +0.00024696113607018140,
                                                          +0.00024692230733234513,
                                                          +0.00024688350585264855,
                                                          +0.00024684470152241543,
                                                          +0.00024680592442462903,
                                                          +0.00024676714447899247,
                                                          +0.00024672839174014205,
                                                          +0.00024668963615612449,
                                                          +0.00024665090775326449,
                                                          +0.00024661217650791706,
                                                          +0.00024657347241813091,
                                                          +0.00024653476548853369,
                                                          +0.00024649608568893351,
                                                          +0.00024645740305219525,
                                                          +0.00024641874751992198,
                                                          +0.00024638008915318015,
                                                          +0.00024634145786540327,
                                                          +0.00024630282374582421,
                                                          +0.00024626421667974180,
                                                          +0.00024622560678452018,
                                                          +0.00024618702391735917,
                                                          +0.00024614843822371856,
                                                          +0.00024610987953273402,
                                                          +0.00024607131801792617,
                                                          +0.00024603278348040217,
                                                          +0.00024599424612170768,
                                                          +0.00024595573571495625,
                                                          +0.00024591722248968402,
                                                          +0.00024587873619104583,
                                                          +0.00024584024707653316,
                                                          +0.00024580178486337728,
                                                          +0.00024576331983698987,
                                                          +0.00024572488168671355,
                                                          +0.00024568644072584542,
                                                          +0.00024564802661587433,
                                                          +0.00024560960969794771,
                                                          +0.00024557121960573571,
                                                          +0.00024553282670820116,
                                                          +0.00024549446061123017,
                                                          +0.00024545609171156642,
                                                          +0.00024541774958734656,
                                                          +0.00024537940466306049,
                                                          +0.00024534108648913009,
                                                          +0.00024530276551775664,
                                                          +0.00024526447127168192,
                                                          +0.00024522617423078423,
                                                          +0.00024518790389015945,
                                                          +0.00024514963075732837,
                                                          +0.00024511138429977598,
                                                          +0.00024507313505263054,
                                                          +0.00024503491245580063,
                                                          +0.00024499668707198783,
                                                          +0.00024495848831355846,
                                                          +0.00024492028677075317,
                                                          +0.00024488211182843017,
                                                          +0.00024484393410433491,
                                                          +0.00024480578295585213,
                                                          +0.00024476762902819743,
                                                          +0.00024472950165131614,
                                                          +0.00024469137149786020,
                                                          +0.00024465326787036972,
                                                          +0.00024461516146889837,
                                                          +0.00024457708156861547,
                                                          +0.00024453899889694236,
                                                          +0.00024450094270171143,
                                                          +0.00024446288373767774,
                                                          +0.00024442485122537077,
                                                          +0.00024438681594684533,
                                                          +0.00024434880709536182,
                                                          +0.00024431079548024100,
                                                          +0.00024427281026750800,
                                                          +0.00024423482229371551,
                                                          +0.00024419686069768751,
                                                          +0.00024415889634317461,
                                                          +0.00024412095834183353,
                                                          +0.00024408301758457895,
                                                          +0.00024404510315593396,
                                                          +0.00024400718597394376,
                                                          +0.00024396929509603143,
                                                          +0.00024393140146733891,
                                                          +0.00024389353411822314,
                                                          +0.00024385566402088894,
                                                          +0.00024381782017866076,
                                                          +0.00024377997359077282,
                                                          +0.00024374215323355047,
                                                          +0.00024370433013322392,
                                                          +0.00024366653323915281,
                                                          +0.00024362873360452974,
                                                          +0.00024359096015178245,
                                                          +0.00024355318396103237,
                                                          +0.00024351543392780843,
                                                          +0.00024347768115912775,
                                                          +0.00024343995452365369,
                                                          +0.00024340222515526578,
                                                          +0.00024336452189579536,
                                                          +0.00024332681590595067,
                                                          +0.00024328913600076441,
                                                          +0.00024325145336774040,
                                                          +0.00024321379679514556,
                                                          +0.00024317613749724645,
                                                          +0.00024313850423557750,
                                                          +0.00024310086825113448,
                                                          +0.00024306325827875243,
                                                          +0.00024302564558612347,
                                                          +0.00024298805888141621,
                                                          +0.00024295046945898606,
                                                          +0.00024291290600036823,
                                                          +0.00024287533982654837,
                                                          +0.00024283779959246126,
                                                          +0.00024280025664568991,
                                                          +0.00024276273961460143,
                                                          +0.00024272521987334339,
                                                          +0.00024268772602374815,
                                                          +0.00024265022946649487,
                                                          +0.00024261275877691401,
                                                          +0.00024257528538218354,
                                                          +0.00024253783783116469,
                                                          +0.00024250038757750160,
                                                          +0.00024246296314361888,
                                                          +0.00024242553600959416,
                                                          +0.00024238813467144824,
                                                          +0.00024235073063565943,
                                                          +0.00024231335237187725,
                                                          +0.00024227597141294828,
                                                          +0.00024223861620218317,
                                                          +0.00024220125829876432,
                                                          +0.00024216392611969595,
                                                          +0.00024212659125046380,
                                                          +0.00024208928208179818,
                                                          +0.00024205197022545560,
                                                          +0.00024201468404592492,
                                                          +0.00024197739518120109,
                                                          +0.00024194013196956373,
                                                          +0.00024190286607521396,
                                                          +0.00024186562581025449,
                                                          +0.00024182838286506038,
                                                          +0.00024179116552558948,
                                                          +0.00024175394550835850,
                                                          +0.00024171675107321298,
                                                          +0.00024167955396277898,
                                                          +0.00024164238241082165,
                                                          +0.00024160520818604435,
                                                          +0.00024156805949616397,
                                                          +0.00024153090813592917,
                                                          +0.00024149378228704050,
                                                          +0.00024145665377025991,
                                                          +0.00024141955074130368,
                                                          +0.00024138244504691496,
                                                          +0.00024134536481685775,
                                                          +0.00024130828192382448,
                                                          +0.00024127122447165862,
                                                          +0.00024123416435897017,
                                                          +0.00024119712966371397,
                                                          +0.00024116009231038552,
                                                          +0.00024112308035108293,
                                                          +0.00024108606573615543,
                                                          +0.00024104907649187614,
                                                          +0.00024101208459441635,
                                                          +0.00024097511804425570,
                                                          +0.00024093814884335599,
                                                          +0.00024090120496643504,
                                                          +0.00024086425844121342,
                                                          +0.00024082733721667878,
                                                          +0.00024079041334627889,
                                                          +0.00024075351475330274,
                                                          +0.00024071661351689376,
                                                          +0.00024067973753467387,
                                                          +0.00024064285891145059,
                                                          +0.00024060600551921010,
                                                          +0.00024056914948839269,
                                                          +0.00024053231866538033,
                                                          +0.00024049548520621450,
                                                          +0.00024045867693170425,
                                                          +0.00024042186602346114,
                                                          +0.00024038508027675245,
                                                          +0.00024034829189872845,
                                                          +0.00024031152865914613,
                                                          +0.00024027476279066306,
                                                          +0.00024023802203755715,
                                                          +0.00024020127865796213,
                                                          +0.00024016456037070788,
                                                          +0.00024012783945937322,
                                                          +0.00024009114361737125,
                                                          +0.00024005444515369466,
                                                          +0.00024001777173637056,
                                                          +0.00023998109569977463,
                                                          +0.00023994444468657933,
                                                          +0.00023990779105651210,
                                                          +0.00023987116242692142,
                                                          +0.00023983453118285581,
                                                          +0.00023979792491637088,
                                                          +0.00023976131603780499,
                                                          +0.00023972473211395176,
                                                          +0.00023968814558040865,
                                                          +0.00023965158397873821,
                                                          +0.00023961501976976603,
                                                          +0.00023957848046985423,
                                                          +0.00023954193856502598,
                                                          +0.00023950542154647369,
                                                          +0.00023946890192538738,
                                                          +0.00023943240716782035,
                                                          +0.00023939590981009875,
                                                          +0.00023935943729316760,
                                                          +0.00023932296217845839,
                                                          +0.00023928651188183842,
                                                          +0.00023925005898981411,
                                                          +0.00023921363089320538,
                                                          +0.00023917720020356318,
                                                          +0.00023914079428669064,
                                                          +0.00023910438577915246,
                                                          +0.00023906800202176556,
                                                          +0.00023903161567607800,
                                                          +0.00023899525405795101,
                                                          +0.00023895888985388551,
                                                          +0.00023892255035481706,
                                                          +0.00023888620827216937,
                                                          +0.00023884989087198289,
                                                          +0.00023881357089057355,
                                                          +0.00023877727556911692,
                                                          +0.00023874097766879088,
                                                          +0.00023870470440593645,
                                                          +0.00023866842856656333,
                                                          +0.00023863217734220786,
                                                          +0.00023859592354368147,
                                                          +0.00023855969433774635,
                                                          +0.00023852346255998522,
                                                          +0.00023848725535241595,
                                                          +0.00023845104557536283,
                                                          +0.00023841486034612946,
                                                          +0.00023837867254975146,
                                                          +0.00023834250927884821,
                                                          +0.00023830634344313684,
                                                          +0.00023827020211058232,
                                                          +0.00023823405821555321,
                                                          +0.00023819793880139025,
                                                          +0.00023816181682708340,
                                                          +0.00023812571931137896,
                                                          +0.00023808961923785853,
                                                          +0.00023805354360070379,
                                                          +0.00023801746540805818,
                                                          +0.00023798141162956836,
                                                          +0.00023794535529791006,
                                                          +0.00023790932335822449,
                                                          +0.00023787328886768999,
                                                          +0.00023783727874697222,
                                                          +0.00023780126607772211,
                                                          +0.00023776527775615954,
                                                          +0.00023772928688837856,
                                                          +0.00023769332034618255,
                                                          +0.00023765735126007916,
                                                          +0.00023762140647748525,
                                                          +0.00023758545915329214,
                                                          +0.00023754953611055947,
                                                          +0.00023751361052853310,
                                                          +0.00023747770920594483,
                                                          +0.00023744180534636562,
                                                          +0.00023740592572422871,
                                                          +0.00023737004356740081,
                                                          +0.00023733418562604607,
                                                          +0.00023729832515229741,
                                                          +0.00023726248887207945,
                                                          +0.00023722665006176190,
                                                          +0.00023719083542305891,
                                                          +0.00023715501825654790,
                                                          +0.00023711922523976186,
                                                          +0.00023708342969745665,
                                                          +0.00023704765828301322,
                                                          +0.00023701188434533670,
                                                          +0.00023697613451368501,
                                                          +0.00023694038216108356,
                                                          +0.00023690465389269657,
                                                          +0.00023686892310564034,
                                                          +0.00023683321638101436,
                                                          +0.00023679750713999690,
                                                          +0.00023676182193965186,
                                                          +0.00023672613422519041,
                                                          +0.00023669047052966966,
                                                          +0.00023665480432230476,
                                                          +0.00023661916211217509,
                                                          +0.00023658351739247092,
                                                          +0.00023654789664832255,
                                                          +0.00023651227339686651,
                                                          +0.00023647667409931302,
                                                          +0.00023644107229671600,
                                                          +0.00023640549442639435,
                                                          +0.00023636991405329050,
                                                          +0.00023633435759086099,
                                                          +0.00023629879862790794,
                                                          +0.00023626326355405389,
                                                          +0.00023622772598193230,
                                                          +0.00023619221227736065,
                                                          +0.00023615669607677460,
                                                          +0.00023612120372221515,
                                                          +0.00023608570887389188,
                                                          +0.00023605023785009768,
                                                          +0.00023601476433478751,
                                                          +0.00023597931462253493,
                                                          +0.00023594386242101138,
                                                          +0.00023590843400109963,
                                                          +0.00023587300309415937,
                                                          +0.00023583759594741086,
                                                          +0.00023580218631587353,
                                                          +0.00023576680042313358,
                                                          +0.00023573141204784193,
                                                          +0.00023569604738997901,
                                                          +0.00023566068025179866,
                                                          +0.00023562533680970409,
                                                          +0.00023558999088952380,
                                                          +0.00023555466864411175,
                                                          +0.00023551934392284301,
                                                          +0.00023548404285505075,
                                                          +0.00023544873931362809,
                                                          +0.00023541345940441551,
                                                          +0.00023537817702379624,
                                                          +0.00023534291825414621,
                                                          +0.00023530765701531051,
                                                          +0.00023527241936622856,
                                                          +0.00023523717925017927,
                                                          +0.00023520196270269386,
                                                          +0.00023516674369045672,
                                                          +0.00023513154822561887,
                                                          +0.00023509635029824230,
                                                          +0.00023506117589712570,
                                                          +0.00023502599903568077,
                                                          +0.00023499084567938185,
                                                          +0.00023495568986496232,
                                                          +0.00023492055753460009,
                                                          +0.00023488542274832230,
                                                          +0.00023485031142503831,
                                                          +0.00023481519764804125,
                                                          +0.00023478010731299963,
                                                          +0.00023474501452644468,
                                                          +0.00023470994516083213,
                                                          +0.00023467487334590347,
                                                          +0.00023463982493092899,
                                                          +0.00023460477406883300,
                                                          +0.00023456974658572823,
                                                          +0.00023453471665769383,
                                                          +0.00023449971008771274,
                                                          +0.00023446470107499142,
                                                          +0.00023442971539941031,
                                                          +0.00023439472728327572,
                                                          +0.00023435976248339330,
                                                          +0.00023432479524514147,
                                                          +0.00023428985130227883,
                                                          +0.00023425490492322824,
                                                          +0.00023421998181872856,
                                                          +0.00023418505628021994,
                                                          +0.00023415015399544877,
                                                          +0.00023411524927884495,
                                                          +0.00023408036779519006,
                                                          +0.00023404548388187622,
                                                          +0.00023401062318074753,
                                                          +0.00023397576005213102,
                                                          +0.00023394092011496057,
                                                          +0.00023390607775247078,
                                                          +0.00023387125856071282,
                                                          +0.00023383643694580144,
                                                          +0.00023380163848093217,
                                                          +0.00023376683759507286,
                                                          +0.00023373205983859058,
                                                          +0.00023369727966327899,
                                                          +0.00023366252259670407,
                                                          +0.00023362776311345809,
                                                          +0.00023359302671833269,
                                                          +0.00023355828790869201,
                                                          +0.00023352357216658049,
                                                          +0.00023348885401210673,
                                                          +0.00023345415890459526,
                                                          +0.00023341946138687205,
                                                          +0.00023338478689556867,
                                                          +0.00023335010999620147,
                                                          +0.00023331545610273609,
                                                          +0.00023328079980335231,
                                                          +0.00023324616648937659,
                                                          +0.00023321153077162545,
                                                          +0.00023317691801881288,
                                                          +0.00023314230286436519,
                                                          +0.00023310771065441111,
                                                          +0.00023307311604495970,
                                                          +0.00023303854435958101,
                                                          +0.00023300397027684031,
                                                          +0.00023296941909777569,
                                                          +0.00023293486552348179,
                                                          +0.00023290033483249159,
                                                          +0.00023286580174840227,
                                                          +0.00023283129152726847,
                                                          +0.00023279677891516318,
                                                          +0.00023276228914568926,
                                                          +0.00023272779698736901,
                                                          +0.00023269332765138017,
                                                          +0.00023265885592866755,
                                                          +0.00023262440700801032,
                                                          +0.00023258995570274953,
                                                          +0.00023255552717929201,
                                                          +0.00023252109627334858,
                                                          +0.00023248668812898048,
                                                          +0.00023245227760424150,
                                                          +0.00023241788982087378,
                                                          +0.00023238349965924780,
                                                          +0.00023234913221881288,
                                                          +0.00023231476240222982,
                                                          +0.00023228041528668153,
                                                          +0.00023224606579709268,
                                                          +0.00023221173898840621,
                                                          +0.00023217740980778427,
                                                          +0.00023214310328795594,
                                                          +0.00023210879439829481,
                                                          +0.00023207450814934246,
                                                          +0.00023204021953265739,
                                                          +0.00023200595353661994,
                                                          +0.00023197168517494744,
                                                          +0.00023193743941388504,
                                                          +0.00023190319128928285,
                                                          +0.00023186896574527686,
                                                          +0.00023183473783982376,
                                                          +0.00023180053249497677,
                                                          +0.00023176632479077295,
                                                          +0.00023173213962720846,
                                                          +0.00023169795210637494,
                                                          +0.00023166378710623778,
                                                          +0.00023162961975091691,
                                                          +0.00023159547489637282,
                                                          +0.00023156132768872788,
                                                          +0.00023152720296196365,
                                                          +0.00023149307588417889,
                                                          +0.00023145897126740245,
                                                          +0.00023142486430168335,
                                                          +0.00023139077977712326,
                                                          +0.00023135669290569614,
                                                          +0.00023132262845560218,
                                                          +0.00023128856166071421,
                                                          +0.00023125451726735698,
                                                          +0.00023122047053127629,
                                                          +0.00023118644617694728,
                                                          +0.00023115241948196304,
                                                          +0.00023111841514897446,
                                                          +0.00023108440847739646,
                                                          +0.00023105042414808152,
                                                          +0.00023101643748224041,
                                                          +0.00023098247313895300,
                                                          +0.00023094850646120031,
                                                          +0.00023091456208631508,
                                                          +0.00023088061537902295,
                                                          +0.00023084669095493527,
                                                          +0.00023081276420049680,
                                                          +0.00023077885970962264,
                                                          +0.00023074495289045137,
                                                          +0.00023071106831522751,
                                                          +0.00023067718141375773,
                                                          +0.00023064331673664152,
                                                          +0.00023060944973532822,
                                                          +0.00023057560493879752,
                                                          +0.00023054175782011611,
                                                          +0.00023050793288666960,
                                                          +0.00023047410563311626,
                                                          +0.00023044030054527286,
                                                          +0.00023040649313936423,
                                                          +0.00023037270787966348,
                                                          +0.00023033892030393678,
                                                          +0.00023030515485493867,
                                                          +0.00023027138709195142,
                                                          +0.00023023764143623651,
                                                          +0.00023020389346856693,
                                                          +0.00023017016758873597,
                                                          +0.00023013643939898240,
                                                          +0.00023010273327765684,
                                                          +0.00023006902484843826,
                                                          +0.00023003533846825962,
                                                          +0.00023000164978221523,
                                                          +0.00022996798312584555,
                                                          +0.00022993431416563499,
                                                          +0.00022990066721575642,
                                                          +0.00022986701796405957,
                                                          +0.00022983339070337469,
                                                          +0.00022979976114289170,
                                                          +0.00022976615355412323,
                                                          +0.00022973254366757452,
                                                          +0.00022969895573346545,
                                                          +0.00022966536550359158,
                                                          +0.00022963179720690509,
                                                          +0.00022959822661646690,
                                                          +0.00022956467793998624,
                                                          +0.00022953112697176461,
                                                          +0.00022949759789829327,
                                                          +0.00022946406653508928,
                                                          +0.00022943055704745073,
                                                          +0.00022939704527208553,
                                                          +0.00022936355535312333,
                                                          +0.00022933006314843828,
                                                          +0.00022929659278101599,
                                                          +0.00022926312012987220,
                                                          +0.00022922966929687347,
                                                          +0.00022919621618215232,
                                                          +0.00022916278486648069,
                                                          +0.00022912935127108330,
                                                          +0.00022909593945566239,
                                                          +0.00022906252536251002,
                                                          +0.00022902913303028319,
                                                          +0.00022899573842231698,
                                                          +0.00022896236555624753,
                                                          +0.00022892899041642841,
                                                          +0.00022889563699949960,
                                                          +0.00022886228131080850,
                                                          +0.00022882894732602328,
                                                          +0.00022879561107146083,
                                                          +0.00022876229650184204,
                                                          +0.00022872897966442880,
                                                          +0.00022869568449301897,
                                                          +0.00022866238705579521,
                                                          +0.00022862911126565668,
                                                          +0.00022859583321168238,
                                                          +0.00022856257678589720,
                                                          +0.00022852931809825213,
                                                          +0.00022849608101992202,
                                                          +0.00022846284168170553,
                                                          +0.00022842962393395186,
                                                          +0.00022839640392828314,
                                                          +0.00022836320549424687,
                                                          +0.00022833000480426455,
                                                          +0.00022829682566710637,
                                                          +0.00022826364427596874,
                                                          +0.00022823048441886879,
                                                          +0.00022819732230975373,
                                                          +0.00022816418171591179,
                                                          +0.00022813103887201681,
                                                          +0.00022809791752465195,
                                                          +0.00022806479392919400,
                                                          +0.00022803169181154505,
                                                          +0.00022799858744776045,
                                                          +0.00022796550454308559,
                                                          +0.00022793241939423034,
                                                          +0.00022789935568580708,
                                                          +0.00022786628973515654,
                                                          +0.00022783324520628191,
                                                          +0.00022780019843713075,
                                                          +0.00022776717307112113,
                                                          +0.00022773414546678347,
                                                          +0.00022770113924697462,
                                                          +0.00022766813079078393,
                                                          +0.00022763514370053079,
                                                          +0.00022760215437583984,
                                                          +0.00022756918639851680,
                                                          +0.00022753621618869769,
                                                          +0.00022750326730769827,
                                                          +0.00022747031619614228,
                                                          +0.00022743738639487937,
                                                          +0.00022740445436499710,
                                                          +0.00022737154362690272,
                                                          +0.00022733863066212387,
                                                          +0.00022730573897064921,
                                                          +0.00022727284505442288,
                                                          +0.00022723997239303814,
                                                          +0.00022720709750883244,
                                                          +0.00022717424386102723,
                                                          +0.00022714138799232920,
                                                          +0.00022710855334161210,
                                                          +0.00022707571647192820,
                                                          +0.00022704290080182677,
                                                          +0.00022701008291468240,
                                                          +0.00022697728620874334,
                                                          +0.00022694448728768292,
                                                          +0.00022691170952947190,
                                                          +0.00022687892955805885,
                                                          +0.00022684617073116055,
                                                          +0.00022681340969297743,
                                                          +0.00022678066978099542,
                                                          +0.00022674792765964351,
                                                          +0.00022671520664620034,
                                                          +0.00022668248342530006,
                                                          +0.00022664978129403722,
                                                          +0.00022661707695722781,
                                                          +0.00022658439369180555,
                                                          +0.00022655170822274506,
                                                          +0.00022651904380684265,
                                                          +0.00022648637718920806,
                                                          +0.00022645373160652343,
                                                          +0.00022642108382401059,
                                                          +0.00022638845705826052,
                                                          +0.00022635582809458398,
                                                          +0.00022632322012950399,
                                                          +0.00022629060996839717,
                                                          +0.00022625802078774158,
                                                          +0.00022622542941295646,
                                                          +0.00022619285900049834,
                                                          +0.00022616028639580577,
                                                          +0.00022612773473533674,
                                                          +0.00022609518088452627,
                                                          +0.00022606264795985663,
                                                          +0.00022603011284673638,
                                                          +0.00022599759864169519,
                                                          +0.00022596508225009197,
                                                          +0.00022593258674852678,
                                                          +0.00022590008906228602,
                                                          +0.00022586761224806294,
                                                          +0.00022583513325104859,
                                                          +0.00022580267510805238,
                                                          +0.00022577021478414696,
                                                          +0.00022573777529628084,
                                                          +0.00022570533362938555,
                                                          +0.00022567291278057116,
                                                          +0.00022564048975460540,
                                                          +0.00022560808752878310,
                                                          +0.00022557568312768503,
                                                          +0.00022554329950881335,
                                                          +0.00022551091371653942,
                                                          +0.00022547854868859547,
                                                          +0.00022544618148912052,
                                                          +0.00022541383503609975,
                                                          +0.00022538148641341725,
                                                          +0.00022534915851933342,
                                                          +0.00022531682845745491,
                                                          +0.00022528451910634026,
                                                          +0.00022525220758929586,
                                                          +0.00022521991676520078,
                                                          +0.00022518762377703879,
                                                          +0.00022515535146403205,
                                                          +0.00022512307698881897,
                                                          +0.00022509082317098773,
                                                          +0.00022505856719280860,
                                                          +0.00022502633185425794,
                                                          +0.00022499409435721573,
                                                          +0.00022496187748206927,
                                                          +0.00022492965845028549,
                                                          +0.00022489746002268474,
                                                          +0.00022486525944029882,
                                                          +0.00022483307944440358,
                                                          +0.00022480089729557325,
                                                          +0.00022476873571556147,
                                                          +0.00022473657198446244,
                                                          +0.00022470442880453021,
                                                          +0.00022467228347535653,
                                                          +0.00022464015867971790,
                                                          +0.00022460803173668148,
                                                          +0.00022457592530956863,
                                                          +0.00022454381673689947,
                                                          +0.00022451172866256265,
                                                          +0.00022447963844450896,
                                                          +0.00022444756870721626,
                                                          +0.00022441549682804404,
                                                          +0.00022438344541208171,
                                                          +0.00022435139185607512,
                                                          +0.00022431935874574717,
                                                          +0.00022428732349720818,
                                                          +0.00022425530867683672,
                                                          +0.00022422329172008517,
                                                          +0.00022419129517401023,
                                                          +0.00022415929649338408,
                                                          +0.00022412731820596331,
                                                          +0.00022409533778581825,
                                                          +0.00022406337774142742,
                                                          +0.00022403141556613711,
                                                          +0.00022399947374916957,
                                                          +0.00022396752980312530,
                                                          +0.00022393560619799241,
                                                          +0.00022390368046560342,
                                                          +0.00022387177505673423,
                                                          +0.00022383986752242752,
                                                          +0.00022380798029426877,
                                                          +0.00022377609094248897,
                                                          +0.00022374422187950525,
                                                          +0.00022371235069471490,
                                                          +0.00022368049978138834,
                                                          +0.00022364864674806747,
                                                          +0.00022361681396889803,
                                                          +0.00022358497907154452,
                                                          +0.00022355316441104967,
                                                          +0.00022352134763417903,
                                                          +0.00022348955107689386,
                                                          +0.00022345775240503914,
                                                          +0.00022342597393551643,
                                                          +0.00022339419335322823,
                                                          +0.00022336243295603838,
                                                          +0.00022333067044788500,
                                                          +0.00022329892810761575,
                                                          +0.00022326718365818306,
                                                          +0.00022323545935943977,
                                                          +0.00022320373295333103,
                                                          +0.00022317202668073661,
                                                          +0.00022314031830257265,
                                                          +0.00022310863004076743,
                                                          +0.00022307693967518660,
                                                          +0.00022304526940882834,
                                                          +0.00022301359704048621,
                                                          +0.00022298194475425017,
                                                          +0.00022295029036782012,
                                                          +0.00022291865604639880,
                                                          +0.00022288701962657132,
                                                          +0.00022285540325467474,
                                                          +0.00022282378478615767,
                                                          +0.00022279218634851318,
                                                          +0.00022276058581603196,
                                                          +0.00022272900529738413,
                                                          +0.00022269742268568130,
                                                          +0.00022266586007079208,
                                                          +0.00022263429536462754,
                                                          +0.00022260275063827614,
                                                          +0.00022257120382242703,
                                                          +0.00022253967696940995,
                                                          +0.00022250814802867084,
                                                          +0.00022247663903380169,
                                                          +0.00022244512795298413,
                                                          +0.00022241363680109387,
                                                          +0.00022238214356502678,
                                                          +0.00022235067024096335,
                                                          +0.00022231919483449284,
                                                          +0.00022228773932312149,
                                                          +0.00022225628173111060,
                                                          +0.00022222484401731373,
                                                          +0.00022219340422464291,
                                                          +0.00022216198429331985,
                                                          +0.00022213056228488653,
                                                          +0.00022209916012095384,
                                                          +0.00022206775588167252,
                                                          +0.00022203637147006375,
                                                          +0.00022200498498486601,
                                                          +0.00022197361831053172,
                                                          +0.00022194224956436612,
                                                          +0.00022191090061227399,
                                                          +0.00022187954959010622,
                                                          +0.00022184821834524075,
                                                          +0.00022181688503205329,
                                                          +0.00022178557147941615,
                                                          +0.00022175425586020873,
                                                          +0.00022172295998481819,
                                                          +0.00022169166204460717,
                                                          +0.00022166038383149877,
                                                          +0.00022162910355531758,
                                                          +0.00022159784298954357,
                                                          +0.00022156658036244248,
                                                          +0.00022153533742907201,
                                                          +0.00022150409243611827,
                                                          +0.00022147286712023725,
                                                          +0.00022144163974651483,
                                                          +0.00022141043203322604,
                                                          +0.00022137922226383579,
                                                          +0.00022134803213825881,
                                                          +0.00022131683995831833,
                                                          +0.00022128566740558953,
                                                          +0.00022125449280023318,
                                                          +0.00022122333780550567,
                                                          +0.00022119218075988451,
                                                          +0.00022116104330832816,
                                                          +0.00022112990380761011,
                                                          +0.00022109878388441135,
                                                          +0.00022106766191378106,
                                                          +0.00022103655950414300,
                                                          +0.00022100545504880167,
                                                          +0.00022097437013794420,
                                                          +0.00022094328318310959,
                                                          +0.00022091221575626922,
                                                          +0.00022088114628717598,
                                                          +0.00022085009632960567,
                                                          +0.00022081904433150483,
                                                          +0.00022078801182847429,
                                                          +0.00022075697728663363,
                                                          +0.00022072596222342904,
                                                          +0.00022069494512313279,
                                                          +0.00022066394748505684,
                                                          +0.00022063294781160582,
                                                          +0.00022060196758397777,
                                                          +0.00022057098532268928,
                                                          +0.00022054002249084485,
                                                          +0.00022050905762705257,
                                                          +0.00022047811217634403,
                                                          +0.00022044716469539838,
                                                          +0.00022041623661119429,
                                                          +0.00022038530649846194,
                                                          +0.00022035439576614731,
                                                          +0.00022032348300701139,
                                                          +0.00022029258961198773,
                                                          +0.00022026169419184777,
                                                          +0.00022023081811953284,
                                                          +0.00022019994002380470,
                                                          +0.00022016908125963272,
                                                          +0.00022013822047374867,
                                                          +0.00022010737900317011,
                                                          +0.00022007653551257874,
                                                          +0.00022004571132106039,
                                                          +0.00022001488511122665,
                                                          +0.00021998407818425150,
                                                          +0.00021995326924065646,
                                                          +0.00021992247956372396,
                                                          +0.00021989168787186510,
                                                          +0.00021986091543049074,
                                                          +0.00021983014097588173,
                                                          +0.00021979938575559729,
                                                          +0.00021976862852376798,
                                                          +0.00021973789051012147,
                                                          +0.00021970715048661796,
                                                          +0.00021967642966517345,
                                                          +0.00021964570683555802,
                                                          +0.00021961500319189577,
                                                          +0.00021958429754174666,
                                                          +0.00021955361106146322,
                                                          +0.00021952292257637512,
                                                          +0.00021949225324508279,
                                                          +0.00021946158191066617,
                                                          +0.00021943092971399365,
                                                          +0.00021940027551587530,
                                                          +0.00021936964043946713,
                                                          +0.00021933900336328974,
                                                          +0.00021930838539280665,
                                                          +0.00021927776542422908,
                                                          +0.00021924716454534764,
                                                          +0.00021921656167004464,
                                                          +0.00021918597786845754,
                                                          +0.00021915539207211987,
                                                          +0.00021912482533353579,
                                                          +0.00021909425660187023,
                                                          +0.00021906370691201365,
                                                          +0.00021903315523074288,
                                                          +0.00021900262257535429,
                                                          +0.00021897208793021690,
                                                          +0.00021894157229505274,
                                                          +0.00021891105467180332,
                                                          +0.00021888055604263575,
                                                          +0.00021885005542704466,
                                                          +0.00021881957378966184,
                                                          +0.00021878909016751530,
                                                          +0.00021875862550772119,
                                                          +0.00021872815886482132,
                                                          +0.00021869771116843562,
                                                          +0.00021866726149060033,
                                                          +0.00021863683074345856,
                                                          +0.00021860639801652156,
                                                          +0.00021857598420447501,
                                                          +0.00021854556841428569,
                                                          +0.00021851517152320145,
                                                          +0.00021848477265562507,
                                                          +0.00021845439267138589,
                                                          +0.00021842401071230330,
                                                          +0.00021839364762080764,
                                                          +0.00021836328255611559,
                                                          +0.00021833293634327755,
                                                          +0.00021830258815888829,
                                                          +0.00021827225881063768,
                                                          +0.00021824192749247918,
                                                          +0.00021821161499476144,
                                                          +0.00021818130052877728,
                                                          +0.00021815100486755352,
                                                          +0.00021812070723970292,
                                                          +0.00021809042840094972,
                                                          +0.00021806014759720752,
                                                          +0.00021802988556691706,
                                                          +0.00021799962157327367,
                                                          +0.00021796937633745373,
                                                          +0.00021793912913991499,
                                                          +0.00021790890068458887,
                                                          +0.00021787867026917636,
                                                          +0.00021784845858038279,
                                                          +0.00021781824493313342,
                                                          +0.00021778804999692672,
                                                          +0.00021775785310389301,
                                                          +0.00021772767490634281,
                                                          +0.00021769749475359257,
                                                          +0.00021766733328078418,
                                                          +0.00021763716985440085,
                                                          +0.00021760702509243478,
                                                          +0.00021757687837851717,
                                                          +0.00021754675031350934,
                                                          +0.00021751662029817156,
                                                          +0.00021748650891625348,
                                                          +0.00021745639558562517,
                                                          +0.00021742630087294342,
                                                          +0.00021739620421316940,
                                                          +0.00021736612615588617,
                                                          +0.00021733604615312683,
                                                          +0.00021730598473741931,
                                                          +0.00021727592137785016,
                                                          +0.00021724587658991112,
                                                          +0.00021721582985972303,
                                                          +0.00021718580168576037,
                                                          +0.00021715577157115947,
                                                          +0.00021712575999739635,
                                                          +0.00021709574648460400,
                                                          +0.00021706575149727886,
                                                          +0.00021703575457253173,
                                                          +0.00021700577615789814,
                                                          +0.00021697579580744805,
                                                          +0.00021694583395177481,
                                                          +0.00021691587016188881,
                                                          +0.00021688592485145987,
                                                          +0.00021685597760842012,
                                                          +0.00021682604882953461,
                                                          +0.00021679611811963842,
                                                          +0.00021676620585861053,
                                                          +0.00021673629166817044,
                                                          +0.00021670639591132949,
                                                          +0.00021667649822667295,
                                                          +0.00021664661896036340,
                                                          +0.00021661673776783311,
                                                          +0.00021658687497841446,
                                                          +0.00021655701026436817,
                                                          +0.00021652716393821481,
                                                          +0.00021649731568902525,
                                                          +0.00021646748581252679,
                                                          +0.00021643765401458179,
                                                          +0.00021640784057414268,
                                                          +0.00021637802521384485,
                                                          +0.00021634822819588476,
                                                          +0.00021631842925965206,
                                                          +0.00021628864865060529,
                                                          +0.00021625886612487027,
                                                          +0.00021622910191118632,
                                                          +0.00021619933578239669,
                                                          +0.00021616958795053987,
                                                          +0.00021613983820515823,
                                                          +0.00021611010674160770,
                                                          +0.00021608037336611152,
                                                          +0.00021605065825736140,
                                                          +0.00021602094123824304,
                                                          +0.00021599124247080221,
                                                          +0.00021596154179456885,
                                                          +0.00021593185935496118,
                                                          +0.00021590217500813485,
                                                          +0.00021587250888289891,
                                                          +0.00021584284085201654,
                                                          +0.00021581319102770562,
                                                          +0.00021578353929931884,
                                                          +0.00021575390576250117,
                                                          +0.00021572427032317634,
                                                          +0.00021569465306043486,
                                                          +0.00021566503389675325,
                                                          +0.00021563543289468553,
                                                          +0.00021560582999324298,
                                                          +0.00021557624523846147,
                                                          +0.00021554665858586870,
                                                          +0.00021551709006500036,
                                                          +0.00021548751964788266,
                                                          +0.00021545796734756924,
                                                          +0.00021542841315256669,
                                                          +0.00021539887705946447,
                                                          +0.00021536933907323165,
                                                          +0.00021533981917401177,
                                                          +0.00021531029738321796,
                                                          +0.00021528079366456602,
                                                          +0.00021525128805589521,
                                                          +0.00021522180050451129,
                                                          +0.00021519231106466189,
                                                          +0.00021516283966726096,
                                                          +0.00021513336638294604,
                                                          +0.00021510391112625730,
                                                          +0.00021507445398420461,
                                                          +0.00021504501485497196,
                                                          +0.00021501557384192354,
                                                          +0.00021498615082690537,
                                                          +0.00021495672592961791,
                                                          +0.00021492731901558710,
                                                          +0.00021489791022083187,
                                                          +0.00021486851939457569,
                                                          +0.00021483912668913827,
                                                          +0.00021480975193745856,
                                                          +0.00021478037530813909,
                                                          +0.00021475101661785204,
                                                          +0.00021472165605146499,
                                                          +0.00021469231340940132,
                                                          +0.00021466296889277576,
                                                          +0.00021463364228578038,
                                                          +0.00021460431380575953,
                                                          +0.00021457500322069197,
                                                          +0.00021454569076413365,
                                                          +0.00021451639618786758,
                                                          +0.00021448709974164389,
                                                          +0.00021445782116106743,
                                                          +0.00021442854071206471,
                                                          +0.00021439927811408029,
                                                          +0.00021437001364919933,
                                                          +0.00021434076702072361,
                                                          +0.00021431151852687944,
                                                          +0.00021428228785484342,
                                                          +0.00021425305531896528,
                                                          +0.00021422384059031430,
                                                          +0.00021419462399934581,
                                                          +0.00021416542520103922,
                                                          +0.00021413622454193832,
                                                          +0.00021410704166094978,
                                                          +0.00021407785692068834,
                                                          +0.00021404868994400584,
                                                          +0.00021401952110957015,
                                                          +0.00021399037002419572,
                                                          +0.00021396121708258626,
                                                          +0.00021393208187553609,
                                                          +0.00021390294481376732,
                                                          +0.00021387382547207183,
                                                          +0.00021384470427717250,
                                                          +0.00021381560078787622,
                                                          +0.00021378649544688912,
                                                          +0.00021375740779705062,
                                                          +0.00021372831829703280,
                                                          +0.00021369924647372476,
                                                          +0.00021367017280174720,
                                                          +0.00021364111679205631,
                                                          +0.00021361205893520412,
                                                          +0.00021358301872623120,
                                                          +0.00021355397667160357,
                                                          +0.00021352495225046333,
                                                          +0.00021349592598517335,
                                                          +0.00021346691733899477,
                                                          +0.00021343790685016971,
                                                          +0.00021340891396609546,
                                                          +0.00021337991924087640,
                                                          +0.00021335094210606334,
                                                          +0.00021332196313160543,
                                                          +0.00021329300173322427,
                                                          +0.00021326403849669667,
                                                          +0.00021323509282193199,
                                                          +0.00021320614531051768,
                                                          +0.00021317721534656813,
                                                          +0.00021314828354746402,
                                                          +0.00021311936928154201,
                                                          +0.00021309045318195895,
                                                          +0.00021306155460129082,
                                                          +0.00021303265418845353,
                                                          +0.00021300377128027952,
                                                          +0.00021297488654142656,
                                                          +0.00021294601929300059,
                                                          +0.00021291715021538442,
                                                          +0.00021288829861397434,
                                                          +0.00021285944518486117,
                                                          +0.00021283060921774866,
                                                          +0.00021280177142441839,
                                                          +0.00021277295107889889,
                                                          +0.00021274412890864549,
                                                          +0.00021271532417202810,
                                                          +0.00021268651761215904,
                                                          +0.00021265772847176676,
                                                          +0.00021262893750960334,
                                                          +0.00021260016395277282,
                                                          +0.00021257138857565019,
                                                          +0.00021254263058973166,
                                                          +0.00021251387078499855,
                                                          +0.00021248512835735613,
                                                          +0.00021245638411237494,
                                                          +0.00021242765723038633,
                                                          +0.00021239892853253315,
                                                          +0.00021237021718358972,
                                                          +0.00021234150402025430,
                                                          +0.00021231280819176108,
                                                          +0.00021228411055034690,
                                                          +0.00021225543022972241,
                                                          +0.00021222674809764645,
                                                          +0.00021219808327232288,
                                                          +0.00021216941663701572,
                                                          +0.00021214076729443891,
                                                          +0.00021211211614334473,
                                                          +0.00021208348227097401,
                                                          +0.00021205484659155059,
                                                          +0.00021202622817685876,
                                                          +0.00021199760795657740,
                                                          +0.00021196900498705095,
                                                          +0.00021194040021339634,
                                                          +0.00021191181267653515,
                                                          +0.00021188322333700579,
                                                          +0.00021185465122032314,
                                                          +0.00021182607730243071,
                                                          +0.00021179752059345358,
                                                          +0.00021176896208472342,
                                                          +0.00021174042077099203,
                                                          +0.00021171187765896282,
                                                          +0.00021168335172803092,
                                                          +0.00021165482400025494,
                                                          +0.00021162631343968959,
                                                          +0.00021159780108373228,
                                                          +0.00021156930588111413,
                                                          +0.00021154080888455447,
                                                          +0.00021151232902747740,
                                                          +0.00021148384737790785,
                                                          +0.00021145538285397908,
                                                          +0.00021142691653900522,
                                                          +0.00021139846733584543,
                                                          +0.00021137001634308634,
                                                          +0.00021134158244832946,
                                                          +0.00021131314676541761,
                                                          +0.00021128472816671077,
                                                          +0.00021125630778129174,
                                                          +0.00021122790446629562,
                                                          +0.00021119949936602842,
                                                          +0.00021117111132241671,
                                                          +0.00021114272149497362,
                                                          +0.00021111434871043337,
                                                          +0.00021108597414349988,
                                                          +0.00021105761660573138,
                                                          +0.00021102925728700618,
                                                          +0.00021100091498372294,
                                                          +0.00021097257090091801,
                                                          +0.00021094424381984671,
                                                          +0.00021091591496068713,
                                                          +0.00021088760308956767,
                                                          +0.00021085928944179189,
                                                          +0.00021083099276837728,
                                                          +0.00021080269431973669,
                                                          +0.00021077441283179311,
                                                          +0.00021074612957005238,
                                                          +0.00021071786325535915,
                                                          +0.00021068959516829609,
                                                          +0.00021066134401464564,
                                                          +0.00021063309109005114,
                                                          +0.00021060485508524890,
                                                          +0.00021057661731092687,
                                                          +0.00021054839644279154,
                                                          +0.00021052017380655911,
                                                          +0.00021049196806292218,
                                                          +0.00021046376055260945,
                                                          +0.00021043556992131568,
                                                          +0.00021040737752476578,
                                                          +0.00021037920199367286,
                                                          +0.00021035102469874200,
                                                          +0.00021032286425572066,
                                                          +0.00021029470205027800,
                                                          +0.00021026655668321187,
                                                          +0.00021023840955513955,
                                                          +0.00021021027925192536,
                                                          +0.00021018214718911861,
                                                          +0.00021015403193766591,
                                                          +0.00021012591492803280,
                                                          +0.00021009781471626420,
                                                          +0.00021006971274772571,
                                                          +0.00021004162756357663,
                                                          +0.00021001354062406678,
                                                          +0.00020998547045548562,
                                                          +0.00020995739853295117,
                                                          +0.00020992934336789922,
                                                          +0.00020990128645030003,
                                                          +0.00020987324627675131,
                                                          +0.00020984520435205997,
                                                          +0.00020981717915800148,
                                                          +0.00020978915221420339,
                                                          +0.00020976114198763496,
                                                          +0.00020973313001272848,
                                                          +0.00020970513474166273,
                                                          +0.00020967713772365897,
                                                          +0.00020964915739612121,
                                                          +0.00020962117532304409,
                                                          +0.00020959320992707255,
                                                          +0.00020956524278695877,
                                                          +0.00020953729231060444,
                                                          +0.00020950934009150338,
                                                          +0.00020948140452282997,
                                                          +0.00020945346721280397,
                                                          +0.00020942554653988779,
                                                          +0.00020939762412701174,
                                                          +0.00020936971833794202,
                                                          +0.00020934181081030354,
                                                          +0.00020931391989318211,
                                                          +0.00020928602723888158,
                                                          +0.00020925815118182298,
                                                          +0.00020923027338897341,
                                                          +0.00020920241218010477,
                                                          +0.00020917454923683181,
                                                          +0.00020914670286429301,
                                                          +0.00020911885475873512,
                                                          +0.00020909102321067850,
                                                          +0.00020906318993098663,
                                                          +0.00020903537319557733,
                                                          +0.00020900755472991498,
                                                          +0.00020897975279533066,
                                                          +0.00020895194913187409,
                                                          +0.00020892416198630494,
                                                          +0.00020889637311324288,
                                                          +0.00020886860074489171,
                                                          +0.00020884082665042543,
                                                          +0.00020881306904750764,
                                                          +0.00020878530971985110,
                                                          +0.00020875756687059444,
                                                          +0.00020872982229797407,
                                                          +0.00020870209419061894,
                                                          +0.00020867436436127355,
                                                          +0.00020864665098407285,
                                                          +0.00020861893588625390,
                                                          +0.00020859123722747299,
                                                          +0.00020856353684944439,
                                                          +0.00020853585289736103,
                                                          +0.00020850816722739907,
                                                          +0.00020848049797030359,
                                                          +0.00020845282699669712,
                                                          +0.00020842517242289211,
                                                          +0.00020839751613394237,
                                                          +0.00020836987623174295,
                                                          +0.00020834223461576348,
                                                          +0.00020831460937349719,
                                                          +0.00020828698241881415,
                                                          +0.00020825937182482082,
                                                          +0.00020823175951977256,
                                                          +0.00020820416356240440,
                                                          +0.00020817656589534173,
                                                          +0.00020814898456296332,
                                                          +0.00020812140152224934,
                                                          +0.00020809383480323762,
                                                          +0.00020806626637724781,
                                                          +0.00020803871425999202,
                                                          +0.00020801116043711428,
                                                          +0.00020798362291001571,
                                                          +0.00020795608367864997,
                                                          +0.00020792856073012264,
                                                          +0.00020790103607868133,
                                                          +0.00020787352769715121,
                                                          +0.00020784601761405895,
                                                          +0.00020781852378796436,
                                                          +0.00020779102826165796,
                                                          +0.00020776354897944949,
                                                          +0.00020773606799837815,
                                                          +0.00020770860324851846,
                                                          +0.00020768113680114347,
                                                          +0.00020765368657210762,
                                                          +0.00020762623464690245,
                                                          +0.00020759879892717751,
                                                          +0.00020757136151262800,
                                                          +0.00020754394029071327,
                                                          +0.00020751651737531716,
                                                          +0.00020748911063972415,
                                                          +0.00020746170221199153,
                                                          +0.00020743430995124384,
                                                          +0.00020740691599969695,
                                                          +0.00020737953820233019,
                                                          +0.00020735215871550328,
                                                          +0.00020732479537006535,
                                                          +0.00020729743033650477,
                                                          +0.00020727008143155555,
                                                          +0.00020724273083981990,
                                                          +0.00020721539636393136,
                                                          +0.00020718806020259109,
                                                          +0.00020716074014434725,
                                                          +0.00020713341840198504,
                                                          +0.00020710611274998202,
                                                          +0.00020707880541519249,
                                                          +0.00020705151415803836,
                                                          +0.00020702422121942824,
                                                          +0.00020699694434574305,
                                                          +0.00020696966579193102,
                                                          +0.00020694240329034694,
                                                          +0.00020691513910996371,
                                                          +0.00020688789096912473,
                                                          +0.00020686064115081300,
                                                          +0.00020683340735937514,
                                                          +0.00020680617189178946,
                                                          +0.00020677895243842077,
                                                          +0.00020675173131022773,
                                                          +0.00020672452618360811,
                                                          +0.00020669731938348628,
                                                          +0.00020667012857230748,
                                                          +0.00020664293608894723,
                                                          +0.00020661575958191302,
                                                          +0.00020658858140401674,
                                                          +0.00020656141918984263,
                                                          +0.00020653425530612449,
                                                          +0.00020650710737353802,
                                                          +0.00020647995777272411,
                                                          +0.00020645282411046456,
                                                          +0.00020642568878129279,
                                                          +0.00020639856937811131,
                                                          +0.00020637144830933140,
                                                          +0.00020634434315399102,
                                                          +0.00020631723633436459,
                                                          +0.00020629014541564004,
                                                          +0.00020626305283394050,
                                                          +0.00020623597614061835,
                                                          +0.00020620889778563090,
                                                          +0.00020618183530650945,
                                                          +0.00020615477116703101,
                                                          +0.00020612772289092037,
                                                          +0.00020610067295575963,
                                                          +0.00020607363887148165,
                                                          +0.00020604660312945923,
                                                          +0.00020601958322584741,
                                                          +0.00020599256166579534,
                                                          +0.00020596555593169503,
                                                          +0.00020593854854245727,
                                                          +0.00020591155696672537,
                                                          +0.00020588456373715752,
                                                          +0.00020585758630866274,
                                                          +0.00020583060722763211,
                                                          +0.00020580364393525476,
                                                          +0.00020577667899164022,
                                                          +0.00020574972982427228,
                                                          +0.00020572277900696455,
                                                          +0.00020569584395350950,
                                                          +0.00020566890725141079,
                                                          +0.00020564198630078392,
                                                          +0.00020561506370280803,
                                                          +0.00020558815684393621,
                                                          +0.00020556124833900864,
                                                          +0.00020553435556083032,
                                                          +0.00020550746113788806,
                                                          +0.00020548058242935322,
                                                          +0.00020545370207734504,
                                                          +0.00020542683742741510,
                                                          +0.00020539997113530115,
                                                          +0.00020537312053294930,
                                                          +0.00020534626828970135,
                                                          +0.00020531943172391216,
                                                          +0.00020529259351851351,
                                                          +0.00020526577097828308,
                                                          +0.00020523894679972846,
                                                          +0.00020521213827406451,
                                                          +0.00020518532811136014,
                                                          +0.00020515853358928185,
                                                          +0.00020513173743144558,
                                                          +0.00020510495690198341,
                                                          +0.00020507817473804454,
                                                          +0.00020505140819024055,
                                                          +0.00020502464000923977,
                                                          +0.00020499788743214741,
                                                          +0.00020497113322313676,
                                                          +0.00020494439460582106,
                                                          +0.00020491765435786422,
                                                          +0.00020489092968940142,
                                                          +0.00020486420339157330,
                                                          +0.00020483749266105106,
                                                          +0.00020481078030243810,
                                                          +0.00020478408349895559,
                                                          +0.00020475738506865545,
                                                          +0.00020473070218132307,
                                                          +0.00020470401766844502,
                                                          +0.00020467734868638454,
                                                          +0.00020465067808004891,
                                                          +0.00020462402299239354,
                                                          +0.00020459736628173223,
                                                          +0.00020457072507762639,
                                                          +0.00020454408225178248,
                                                          +0.00020451745492038194,
                                                          +0.00020449082596850997,
                                                          +0.00020446421249898175,
                                                          +0.00020443759741024727,
                                                          +0.00020441099779176978,
                                                          +0.00020438439655534997,
                                                          +0.00020435781077711272,
                                                          +0.00020433122338219572,
                                                          +0.00020430465143339964,
                                                          +0.00020427807786918515,
                                                          +0.00020425151973904211,
                                                          +0.00020422495999474077,
                                                          +0.00020419841567247422,
                                                          +0.00020417186973730798,
                                                          +0.00020414533921215240,
                                                          +0.00020411880707535448,
                                                          +0.00020409229033655553,
                                                          +0.00020406577198737029,
                                                          +0.00020403926902418486,
                                                          +0.00020401276445186785,
                                                          +0.00020398627525356386,
                                                          +0.00020395978444738178,
                                                          +0.00020393330900323847,
                                                          +0.00020390683195246919,
                                                          +0.00020388037025177684,
                                                          +0.00020385390694570933,
                                                          +0.00020382745897776924,
                                                          +0.00020380100940570374,
                                                          +0.00020377457515982842,
                                                          +0.00020374813931107593,
                                                          +0.00020372171877658905,
                                                          +0.00020369529664047197,
                                                          +0.00020366888980670824,
                                                          +0.00020364248137255995,
                                                          +0.00020361608822886497,
                                                          +0.00020358969348602981,
                                                          +0.00020356331402176042,
                                                          +0.00020353693295959395,
                                                          +0.00020351056716411795,
                                                          +0.00020348419977198672,
                                                          +0.00020345784763468281,
                                                          +0.00020343149390196423,
                                                          +0.00020340515541222238,
                                                          +0.00020337881532830502,
                                                          +0.00020335249047552593,
                                                          +0.00020332616402980936,
                                                          +0.00020329985280340482,
                                                          +0.00020327353998529947,
                                                          +0.00020324724237469224,
                                                          +0.00020322094317361952,
                                                          +0.00020319465916824331,
                                                          +0.00020316837357363572,
                                                          +0.00020314210316293502,
                                                          +0.00020311583116423577,
                                                          +0.00020308957433766626,
                                                          +0.00020306331592432976,
                                                          +0.00020303707267135769,
                                                          +0.00020301082783284901,
                                                          +0.00020298459814295175,
                                                          +0.00020295836686874688,
                                                          +0.00020293215073141266,
                                                          +0.00020290593301099859,
                                                          +0.00020287973041572640,
                                                          +0.00020285352623860094,
                                                          +0.00020282733717490064,
                                                          +0.00020280114653057222,
                                                          +0.00020277497098796475,
                                                          +0.00020274879386595300,
                                                          +0.00020272263183396968,
                                                          +0.00020269646822380482,
                                                          +0.00020267031969198809,
                                                          +0.00020264416958321120,
                                                          +0.00020261803454111417,
                                                          +0.00020259189792327719,
                                                          +0.00020256577636046372,
                                                          +0.00020253965322312924,
                                                          +0.00020251354512917409,
                                                          +0.00020248743546191555,
                                                          +0.00020246134082640407,
                                                          +0.00020243524461880565,
                                                          +0.00020240916343133402,
                                                          +0.00020238308067299055,
                                                          +0.00020235701292316570,
                                                          +0.00020233094360368282,
                                                          +0.00020230488928112234,
                                                          +0.00020227883339011643,
                                                          +0.00020225279248444849,
                                                          +0.00020222675001154656,
                                                          +0.00020220072251241018,
                                                          +0.00020217469344724999,
                                                          +0.00020214867934429472,
                                                          +0.00020212266367652454,
                                                          +0.00020209666295941084,
                                                          +0.00020207066067868978,
                                                          +0.00020204467333708834,
                                                          +0.00020201868443308607,
                                                          +0.00020199271045667850,
                                                          +0.00020196673491907529,
                                                          +0.00020194077429755379,
                                                          +0.00020191481211604052,
                                                          +0.00020188886483910777,
                                                          +0.00020186291600338602,
                                                          +0.00020183698206075539,
                                                          +0.00020181104656053715,
                                                          +0.00020178512594193256,
                                                          +0.00020175920376694053,
                                                          +0.00020173329646209638,
                                                          +0.00020170738760206383,
                                                          +0.00020168149360072512,
                                                          +0.00020165559804539578,
                                                          +0.00020162971733731806,
                                                          +0.00020160383507644621,
                                                          +0.00020157796765139549,
                                                          +0.00020155209867474589,
                                                          +0.00020152624452249881,
                                                          +0.00020150038881984692,
                                                          +0.00020147454793019039,
                                                          +0.00020144870549132195,
                                                          +0.00020142287785405346,
                                                          +0.00020139704866876473,
                                                          +0.00020137123427369234,
                                                          +0.00020134541833179004,
                                                          +0.00020131961716873220,
                                                          +0.00020129381446003361,
                                                          +0.00020126802651881901,
                                                          +0.00020124223703315173,
                                                          +0.00020121646230361978,
                                                          +0.00020119068603082190,
                                                          +0.00020116492450282221,
                                                          +0.00020113916143274215,
                                                          +0.00020111341309613485,
                                                          +0.00020108766321863151,
                                                          +0.00020106192806328707,
                                                          +0.00020103619136822956,
                                                          +0.00020101046938402880,
                                                          +0.00020098474586129693,
                                                          +0.00020095903703813098,
                                                          +0.00020093332667761466,
                                                          +0.00020090763100538508,
                                                          +0.00020088193379698454,
                                                          +0.00020085625126560323,
                                                          +0.00020083056719922925,
                                                          +0.00020080489779861828,
                                                          +0.00020077922686419171,
                                                          +0.00020075357058428367,
                                                          +0.00020072791277173584,
                                                          +0.00020070226960247343,
                                                          +0.00020067662490174591,
                                                          +0.00020065099483308215,
                                                          +0.00020062536323412668,
                                                          +0.00020059974625602497,
                                                          +0.00020057412774880380,
                                                          +0.00020054852385123755,
                                                          +0.00020052291842572291,
                                                          +0.00020049732759867600,
                                                          +0.00020047173524485060,
                                                          +0.00020044615747831697,
                                                          +0.00020042057818617359,
                                                          +0.00020039501347015747,
                                                          +0.00020036944722969912,
                                                          +0.00020034389555421498,
                                                          +0.00020031834235545486,
                                                          +0.00020029280371052730,
                                                          +0.00020026726354348888,
                                                          +0.00020024173791915263,
                                                          +0.00020021621077386949,
                                                          +0.00020019069816016945,
                                                          +0.00020016518402668528,
                                                          +0.00020013968441367670,
                                                          +0.00020011418328204532,
                                                          +0.00020008869665979337,
                                                          +0.00020006320852007902,
                                                          +0.00020003773487865890,
                                                          +0.00020001225972093560,
                                                          +0.00019998679905043281,
                                                          +0.00019996133686478484,
                                                          +0.00019993588915529498,
                                                          +0.00019991043993181675,
                                                          +0.00019988500517344532,
                                                          +0.00019985956890224125,
                                                          +0.00019983414708510400,
                                                          +0.00019980872375628851,
                                                          +0.00019978331487051124,
                                                          +0.00019975790447420899,
                                                          +0.00019973250850992738,
                                                          +0.00019970711103627295,
                                                          +0.00019968172798363288,
                                                          +0.00019965634342277087,
                                                          +0.00019963097327192812,
                                                          +0.00019960560161401330,
                                                          +0.00019958024435513371,
                                                          +0.00019955488559033065,
                                                          +0.00019952954121359010,
                                                          +0.00019950419533207349,
                                                          +0.00019947886382765774,
                                                          +0.00019945353081961227,
                                                          +0.00019942821217771708,
                                                          +0.00019940289203333731,
                                                          +0.00019937758624416843,
                                                          +0.00019935227895365889,
                                                          +0.00019932698600743200,
                                                          +0.00019930169156100728,
                                                          +0.00019927641144794797,
                                                          +0.00019925112983583241,
                                                          +0.00019922586254617625,
                                                          +0.00019920059375860426,
                                                          +0.00019917533928259660,
                                                          +0.00019915008330981243,
                                                          +0.00019912484163770860,
                                                          +0.00019909959846996648,
                                                          +0.00019907436959203158,
                                                          +0.00019904913921959540,
                                                          +0.00019902392312610463,
                                                          +0.00019899870553924837,
                                                          +0.00019897350222048660,
                                                          +0.00019894829740949407,
                                                          +0.00019892310685575593,
                                                          +0.00019889791481092066,
                                                          +0.00019887273701251081,
                                                          +0.00019884755772413629,
                                                          +0.00019882239267136914,
                                                          +0.00019879722612976849,
                                                          +0.00019877207381296824,
                                                          +0.00019874692000846462,
                                                          +0.00019872178041796523,
                                                          +0.00019869663934089145,
                                                          +0.00019867151246703670,
                                                          +0.00019864638410773531,
                                                          +0.00019862126994087877,
                                                          +0.00019859615428970229,
                                                          +0.00019857105282020717,
                                                          +0.00019854594986751773,
                                                          +0.00019852086108575705,
                                                          +0.00019849577082192653,
                                                          +0.00019847069471828305,
                                                          +0.00019844561713369303,
                                                          +0.00019842055369855935,
                                                          +0.00019839548878360113,
                                                          +0.00019837043800737938,
                                                          +0.00019834538575245404,
                                                          +0.00019832034762555607,
                                                          +0.00019829530802107449,
                                                          +0.00019827028253392175,
                                                          +0.00019824525557030420,
                                                          +0.00019822024271332804,
                                                          +0.00019819522838100461,
                                                          +0.00019817022814464592,
                                                          +0.00019814522643405648,
                                                          +0.00019812023880876568,
                                                          +0.00019809524971035945,
                                                          +0.00019807027468659685,
                                                          +0.00019804529819083302,
                                                          +0.00019802033575906823,
                                                          +0.00019799537185641539,
                                                          +0.00019797042200712790,
                                                          +0.00019794547068806428,
                                                          +0.00019792053341174305,
                                                          +0.00019789559466675653,
                                                          +0.00019787066995390011,
                                                          +0.00019784574377348813,
                                                          +0.00019782083161460460,
                                                          +0.00019779591798927429,
                                                          +0.00019777101837488136,
                                                          +0.00019774611729514913,
                                                          +0.00019772123021577408,
                                                          +0.00019769634167216609,
                                                          +0.00019767146711834567,
                                                          +0.00019764659110139767,
                                                          +0.00019762172906367815,
                                                          +0.00019759686556393525,
                                                          +0.00019757201603287248,
                                                          +0.00019754716504088944,
                                                          +0.00019752232800704866,
                                                          +0.00019749748951338955,
                                                          +0.00019747266496734564,
                                                          +0.00019744783896258423,
                                                          +0.00019742302689492147,
                                                          +0.00019739821336964088,
                                                          +0.00019737341377095295,
                                                          +0.00019734861271574589,
                                                          +0.00019732382557663601,
                                                          +0.00019729903698210448,
                                                          +0.00019727426229318524,
                                                          +0.00019724948614994081,
                                                          +0.00019722472390183422,
                                                          +0.00019719996020049790,
                                                          +0.00019717521038383547,
                                                          +0.00019715045911503754,
                                                          +0.00019712572172046015,
                                                          +0.00019710098287484032,
                                                          +0.00019707625789299832,
                                                          +0.00019705153146120584,
                                                          +0.00019702681888275880,
                                                          +0.00019700210485545222,
                                                          +0.00019697740467106912,
                                                          +0.00019695270303891641,
                                                          +0.00019692801523927566,
                                                          +0.00019690332599295399,
                                                          +0.00019687865056874335,
                                                          +0.00019685397369893938,
                                                          +0.00019682931064085588,
                                                          +0.00019680464613826558,
                                                          +0.00019677999543701551,
                                                          +0.00019675534329234424,
                                                          +0.00019673070493864337,
                                                          +0.00019670606514260566,
                                                          +0.00019668143912717894,
                                                          +0.00019665681167049866,
                                                          +0.00019663219798408035,
                                                          +0.00019660758285749077,
                                                          +0.00019658298149082443,
                                                          +0.00019655837868506792,
                                                          +0.00019653378962890638,
                                                          +0.00019650919913473476,
                                                          +0.00019648462237983999,
                                                          +0.00019646004418801417,
                                                          +0.00019643547972515756,
                                                          +0.00019641091382644778,
                                                          +0.00019638636164640987,
                                                          +0.00019636180803159556,
                                                          +0.00019633726812516607,
                                                          +0.00019631272678503591,
                                                          +0.00019628819914301379,
                                                          +0.00019626367006836574,
                                                          +0.00019623915468155911,
                                                          +0.00019621463786320011,
                                                          +0.00019619013472242636,
                                                          +0.00019616563015117285,
                                                          +0.00019614113924725841,
                                                          +0.00019611664691393567,
                                                          +0.00019609216823771627,
                                                          +0.00019606768813315890,
                                                          +0.00019604322167547940,
                                                          +0.00019601875379053114,
                                                          +0.00019599429954224540,
                                                          +0.00019596984386775925,
                                                          +0.00019594540181973034,
                                                          +0.00019592095834656821,
                                                          +0.00019589652848966838,
                                                          +0.00019587209720870144,
                                                          +0.00019584767953381192,
                                                          +0.00019582326043592044,
                                                          +0.00019579885493393163,
                                                          +0.00019577444801000476,
                                                          +0.00019575005467181624,
                                                          +0.00019572565991275251,
                                                          +0.00019570127872927271,
                                                          +0.00019567689612597962,
                                                          +0.00019565252708812612,
                                                          +0.00019562815663152019,
                                                          +0.00019560379973021966,
                                                          +0.00019557944141122645,
                                                          +0.00019555509663741455,
                                                          +0.00019553075044696864,
                                                          +0.00019550641779159014,
                                                          +0.00019548208372063521,
                                                          +0.00019545776317464377,
                                                          +0.00019543344121413262,
                                                          +0.00019540913276849082,
                                                          +0.00019538482290938495,
                                                          +0.00019536052655506469,
                                                          +0.00019533622878833477,
                                                          +0.00019531194451631665,
                                                          +0.00019528765883294235,
                                                          +0.00019526338663421606,
                                                          +0.00019523911302518596,
                                                          +0.00019521485289075014,
                                                          +0.00019519059134706179,
                                                          +0.00019516634326792389,
                                                          +0.00019514209378058387,
                                                          +0.00019511785774776046,
                                                          +0.00019509362030778411,
                                                          +0.00019506939631230060,
                                                          +0.00019504517091071239,
                                                          +0.00019502095894360309,
                                                          +0.00019499674557143623,
                                                          +0.00019497254562374436,
                                                          +0.00019494834427204110,
                                                          +0.00019492415633481875,
                                                          +0.00019489996699463021,
                                                          +0.00019487579105893839,
                                                          +0.00019485161372132439,
                                                          +0.00019482744977823306,
                                                          +0.00019480328443426261,
                                                          +0.00019477913247485030,
                                                          +0.00019475497911560093,
                                                          +0.00019473083913095542,
                                                          +0.00019470669774751371,
                                                          +0.00019468256972873129,
                                                          +0.00019465844031219264,
                                                          +0.00019463432425037862,
                                                          +0.00019461020679184719,
                                                          +0.00019458610267811559,
                                                          +0.00019456199716870446,
                                                          +0.00019453790499417804,
                                                          +0.00019451381142500894,
                                                          +0.00019448973118081942,
                                                          +0.00019446564954302308,
                                                          +0.00019444158122031084,
                                                          +0.00019441751150502663,
                                                          +0.00019439345509494082,
                                                          +0.00019436939729331682,
                                                          +0.00019434535278701551,
                                                          +0.00019432130689020868,
                                                          +0.00019429727427885851,
                                                          +0.00019427324027803452,
                                                          +0.00019424921955281093,
                                                          +0.00019422519743914429,
                                                          +0.00019420118859123138,
                                                          +0.00019417717835590510,
                                                          +0.00019415318137649582,
                                                          +0.00019412918301070186,
                                                          +0.00019410519789099778,
                                                          +0.00019408121138593659,
                                                          +0.00019405723811714799,
                                                          +0.00019403326346402894,
                                                          +0.00019400930203737480,
                                                          +0.00019398533922741578,
                                                          +0.00019396138963412369,
                                                          +0.00019393743865855136,
                                                          +0.00019391350088985762,
                                                          +0.00019388956173990723,
                                                          +0.00019386563578705686,
                                                          +0.00019384170845397240,
                                                          +0.00019381779430821883,
                                                          +0.00019379387878325284,
                                                          +0.00019376997643585846,
                                                          +0.00019374607271027217,
                                                          +0.00019372218215250772,
                                                          +0.00019369829021757094,
                                                          +0.00019367441144071588,
                                                          +0.00019365053128770707,
                                                          +0.00019362666428304954,
                                                          +0.00019360279590325573,
                                                          +0.00019357894066209230,
                                                          +0.00019355508404680910,
                                                          +0.00019353124056044490,
                                                          +0.00019350739570097661,
                                                          +0.00019348356396072552,
                                                          +0.00019345973084838484,
                                                          +0.00019343591084556916,
                                                          +0.00019341208947167747,
                                                          +0.00019338828119762807,
                                                          +0.00019336447155351519,
                                                          +0.00019334067499957153,
                                                          +0.00019331687707657602,
                                                          +0.00019329309223408594,
                                                          +0.00019326930602355460,
                                                          +0.00019324553288387465,
                                                          +0.00019322175837716292,
                                                          +0.00019319799693165809,
                                                          +0.00019317423412013003,
                                                          +0.00019315048436017369,
                                                          +0.00019312673323520170,
                                                          +0.00019310299515217584,
                                                          +0.00019307925570514093,
                                                          +0.00019305552929043586,
                                                          +0.00019303180151272739,
                                                          +0.00019300808675774210,
                                                          +0.00019298437064075796,
                                                          +0.00019296066753689972,
                                                          +0.00019293696307204621,
                                                          +0.00019291327161073080,
                                                          +0.00019288957878942282,
                                                          +0.00019286589896207434,
                                                          +0.00019284221777573493,
                                                          +0.00019281854957378612,
                                                          +0.00019279488001384701,
                                                          +0.00019277122342873881,
                                                          +0.00019274756548663999,
                                                          +0.00019272392050982188,
                                                          +0.00019270027417701182,
                                                          +0.00019267664079994159,
                                                          +0.00019265300606787714,
                                                          +0.00019262938428202091,
                                                          +0.00019260576114216732,
                                                          +0.00019258215093899967,
                                                          +0.00019255853938283037,
                                                          +0.00019253494075383433,
                                                          +0.00019251134077283145,
                                                          +0.00019248775370949814,
                                                          +0.00019246416529515191,
                                                          +0.00019244058978898097,
                                                          +0.00019241701293278994,
                                                          +0.00019239344897528941,
                                                          +0.00019236988366876055,
                                                          +0.00019234633125144663,
                                                          +0.00019232277748609523,
                                                          +0.00019229923660049248,
                                                          +0.00019227569436784217,
                                                          +0.00019225216500548341,
                                                          +0.00019222863429706603,
                                                          +0.00019220511644949244,
                                                          +0.00019218159725684823,
                                                          +0.00019215809091560921,
                                                          +0.00019213458323028663,
                                                          +0.00019211108838693982,
                                                          +0.00019208759220049553,
                                                          +0.00019206410884660689,
                                                          +0.00019204062415060597,
                                                          +0.00019201715227774971,
                                                          +0.00019199367906376524,
                                                          +0.00019197021866352380,
                                                          +0.00019194675692313736,
                                                          +0.00019192330798710141,
                                                          +0.00019189985771190260,
                                                          +0.00019187642023167100,
                                                          +0.00019185298141325777,
                                                          +0.00019182955538043763,
                                                          +0.00019180612801041614,
                                                          +0.00019178271341662261,
                                                          +0.00019175929748660721,
                                                          +0.00019173589432346382,
                                                          +0.00019171248982507690,
                                                          +0.00019168909808421532,
                                                          +0.00019166570500908766,
                                                          +0.00019164232468214768,
                                                          +0.00019161894302191805,
                                                          +0.00019159557410054762,
                                                          +0.00019157220384686304,
                                                          +0.00019154884632271833,
                                                          +0.00019152548746723398,
                                                          +0.00019150214133197913,
                                                          +0.00019147879386635839,
                                                          +0.00019145545911166580,
                                                          +0.00019143212302757991,
                                                          +0.00019140879964513009,
                                                          +0.00019138547493425872,
                                                          +0.00019136216291574032,
                                                          +0.00019133884956977105,
                                                          +0.00019131554890688068,
                                                          +0.00019129224691750930,
                                                          +0.00019126895760195178,
                                                          +0.00019124566696088213,
                                                          +0.00019122238898437027,
                                                          +0.00019119910968331424,
                                                          +0.00019117584303756898,
                                                          +0.00019115257506824659,
                                                          +0.00019112931974499695,
                                                          +0.00019110606309913626,
                                                          +0.00019108281909011919,
                                                          +0.00019105957375945619,
                                                          +0.00019103634105641686,
                                                          +0.00019101310703269577,
                                                          +0.00019098988562738721,
                                                          +0.00019096666290236008,
                                                          +0.00019094345278654353,
                                                          +0.00019092024135197063,
                                                          +0.00019089704251741506,
                                                          +0.00019087384236506457,
                                                          +0.00019085065480354719,
                                                          +0.00019082746592519528,
                                                          +0.00019080428962850118,
                                                          +0.00019078111201593207,
                                                          +0.00019075794697585432,
                                                          +0.00019073478062086013,
                                                          +0.00019071162682919986,
                                                          +0.00019068847172358074,
                                                          +0.00019066532917214689,
                                                          +0.00019064218530771094,
                                                          +0.00019061905398832054,
                                                          +0.00019059592135688383,
                                                          +0.00019057280126136178,
                                                          +0.00019054967985474824,
                                                          +0.00019052657097492736,
                                                          +0.00019050346078496898,
                                                          +0.00019048036311269010,
                                                          +0.00019045726413122673,
                                                          +0.00019043417765833844,
                                                          +0.00019041108987721779,
                                                          +0.00019038801459557675,
                                                          +0.00019036493800665450,
                                                          +0.00019034187390812516,
                                                          +0.00019031880850326483,
                                                          +0.00019029575557971963,
                                                          +0.00019027270135079272,
                                                          +0.00019024965959411178,
                                                          +0.00019022661653299758,
                                                          +0.00019020358593506905,
                                                          +0.00019018055403365470,
                                                          +0.00019015753458637459,
                                                          +0.00019013451383655518,
                                                          +0.00019011150553182725,
                                                          +0.00019008849592550567,
                                                          +0.00019006549875524154,
                                                          +0.00019004250028432847,
                                                          +0.00019001951424044763,
                                                          +0.00018999652689686170,
                                                          +0.00018997355197129134,
                                                          +0.00018995057574695883,
                                                          +0.00018992761193163416,
                                                          +0.00018990464681848925,
                                                          +0.00018988169410535317,
                                                          +0.00018985874009533785,
                                                          +0.00018983579847634097,
                                                          +0.00018981285556140501,
                                                          +0.00018978992502850580,
                                                          +0.00018976699320060681,
                                                          +0.00018974407374577147,
                                                          +0.00018972115299687473,
                                                          +0.00018969824461207723,
                                                          +0.00018967533493415576,
                                                          +0.00018965243761137791,
                                                          +0.00018962953899641258,
                                                          +0.00018960665272764384,
                                                          +0.00018958376516762325,
                                                          +0.00018956088994486077,
                                                          +0.00018953801343178127,
                                                          +0.00018951514924702998,
                                                          +0.00018949228377289552,
                                                          +0.00018946943061816813,
                                                          +0.00018944657617499044,
                                                          +0.00018942373404230733,
                                                          +0.00018940089062210595,
                                                          +0.00018937805950349507,
                                                          +0.00018935522709829711,
                                                          +0.00018933240698579425,
                                                          +0.00018930958558763445,
                                                          +0.00018928677647328309,
                                                          +0.00018926396607420412,
                                                          +0.00018924116795005522,
                                                          +0.00018921836854210718,
                                                          +0.00018919558140021952,
                                                          +0.00018917279297546026,
                                                          +0.00018915001680790025,
                                                          +0.00018912723935839531,
                                                          +0.00018910447415723694,
                                                          +0.00018908170767505936,
                                                          +0.00018905895343238432,
                                                          +0.00018903619790961492,
                                                          +0.00018901345461751247,
                                                          +0.00018899071004623971,
                                                          +0.00018896797769680666,
                                                          +0.00018894524406912641,
                                                          +0.00018892252265446743,
                                                          +0.00018889979996248332,
                                                          +0.00018887708947471038,
                                                          +0.00018885437771053369,
                                                          +0.00018883167814176649,
                                                          +0.00018880897729751594,
                                                          +0.00018878628863988167,
                                                          +0.00018876359870768369,
                                                          +0.00018874092095331723,
                                                          +0.00018871824192530565,
                                                          +0.00018869557506634934,
                                                          +0.00018867290693466577,
                                                          +0.00018865025096326954,
                                                          +0.00018862759372006288,
                                                          +0.00018860494862838422,
                                                          +0.00018858230226581117,
                                                          +0.00018855966804601503,
                                                          +0.00018853703255623962,
                                                          +0.00018851440920049852,
                                                          +0.00018849178457569241,
                                                          +0.00018846917207618642,
                                                          +0.00018844655830852879,
                                                          +0.00018842395665744535,
                                                          +0.00018840135373912286,
                                                          +0.00018837876292865702,
                                                          +0.00018835617085186373,
                                                          +0.00018833359087421807,
                                                          +0.00018831100963115566,
                                                          +0.00018828844047854009,
                                                          +0.00018826587006141770,
                                                          +0.00018824331172604970,
                                                          +0.00018822075212708385,
                                                          +0.00018819820460118833,
                                                          +0.00018817565581260304,
                                                          +0.00018815311908841238,
                                                          +0.00018813058110243926,
                                                          +0.00018810805517219318,
                                                          +0.00018808552798107103,
                                                          +0.00018806301283701686,
                                                          +0.00018804049643299215,
                                                          +0.00018801799206738445,
                                                          +0.00018799548644271089,
                                                          +0.00018797299284781179,
                                                          +0.00018795049799475059,
                                                          +0.00018792801516282956,
                                                          +0.00018790553107364942,
                                                          +0.00018788305899698326,
                                                          +0.00018786058566396004,
                                                          +0.00018783812433483311,
                                                          +0.00018781566175025028,
                                                          +0.00018779321116095418,
                                                          +0.00018777075931710252,
                                                          +0.00018774831945993621,
                                                          +0.00018772587834911382,
                                                          +0.00018770344921638374,
                                                          +0.00018768101883089624,
                                                          +0.00018765860041491600,
                                                          +0.00018763618074707614,
                                                          +0.00018761377304016689,
                                                          +0.00018759136408229491,
                                                          +0.00018756896707678496,
                                                          +0.00018754656882120839,
                                                          +0.00018752418250943357,
                                                          +0.00018750179494848727,
                                                          +0.00018747941932279057,
                                                          +0.00018745704244881674,
                                                          +0.00018743467750154848,
                                                          +0.00018741231130689663,
                                                          +0.00018738995703041449,
                                                          +0.00018736760150744135,
                                                          +0.00018734525789411029,
                                                          +0.00018732291303517995,
                                                          +0.00018730058007737218,
                                                          +0.00018727824587485612,
                                                          +0.00018725592356495105,
                                                          +0.00018723360001122777,
                                                          +0.00018721128834161226,
                                                          +0.00018718897542906773,
                                                          +0.00018716667439213575,
                                                          +0.00018714437211316318,
                                                          +0.00018712208170131600,
                                                          +0.00018709979004831578,
                                                          +0.00018707751025396187,
                                                          +0.00018705522921934160,
                                                          +0.00018703296003489678,
                                                          +0.00018701068961107144,
                                                          +0.00018698843102895853,
                                                          +0.00018696617120835018,
                                                          +0.00018694392322099942,
                                                          +0.00018692167399603736,
                                                          +0.00018689943659588615,
                                                          +0.00018687719795900694,
                                                          +0.00018685497113849980,
                                                          +0.00018683274308214717,
                                                          +0.00018681052683373585,
                                                          +0.00018678830935036069,
                                                          +0.00018676610366650414,
                                                          +0.00018674389674856452,
                                                          +0.00018672170162172891,
                                                          +0.00018669950526169015,
                                                          +0.00018667732068434865,
                                                          +0.00018665513487468321,
                                                          +0.00018663296083931620,
                                                          +0.00018661078557250359,
                                                          +0.00018658862207159872,
                                                          +0.00018656645734012572,
                                                          +0.00018654430436617768,
                                                          +0.00018652215016253809,
                                                          +0.00018650000770804870,
                                                          +0.00018647786402474354,
                                                          +0.00018645573208222175,
                                                          +0.00018643359891175920,
                                                          +0.00018641147747372106,
                                                          +0.00018638935480861621,
                                                          +0.00018636724386758493,
                                                          +0.00018634513170036023,
                                                          +0.00018632303124886597,
                                                          +0.00018630092957205079,
                                                          +0.00018627883960263097,
                                                          +0.00018625674840876184,
                                                          +0.00018623466891396084,
                                                          +0.00018621258819558138,
                                                          +0.00018619051916795066,
                                                          +0.00018616844891761156,
                                                          +0.00018614639034970966,
                                                          +0.00018612433055996855,
                                                          +0.00018610228244436111,
                                                          +0.00018608023310778279,
                                                          +0.00018605819543704249,
                                                          +0.00018603615654619877,
                                                          +0.00018601412931290522,
                                                          +0.00018599210086037499,
                                                          +0.00018597008405711496,
                                                          +0.00018594806603548412,
                                                          +0.00018592605965485123,
                                                          +0.00018590405205671267,
                                                          +0.00018588205609130771,
                                                          +0.00018586005890926130,
                                                          +0.00018583807335169206,
                                                          +0.00018581608657834478,
                                                          +0.00018579411142122590,
                                                          +0.00018577213504919159,
                                                          +0.00018575017028514489,
                                                          +0.00018572820430704448,
                                                          +0.00018570624992869859,
                                                          +0.00018568429433716007,
                                                          +0.00018566235033715061,
                                                          +0.00018564040512480866,
                                                          +0.00018561847149577832,
                                                          +0.00018559653665527486,
                                                          +0.00018557461338987319,
                                                          +0.00018555268891385694,
                                                          +0.00018553077600474047,
                                                          +0.00018550886188586717,
                                                          +0.00018548695932569934,
                                                          +0.00018546505555663162,
                                                          +0.00018544316333808287,
                                                          +0.00018542126991149022,
                                                          +0.00018539938802723783,
                                                          +0.00018537750493579692,
                                                          +0.00018535563337852503,
                                                          +0.00018533376061491918,
                                                          +0.00018531189937731900,
                                                          +0.00018529003693423847,
                                                          +0.00018526818600900802,
                                                          +0.00018524633387915007,
                                                          +0.00018522449325899422,
                                                          +0.00018520265143506299,
                                                          +0.00018518082111269353,
                                                          +0.00018515898958740000,
                                                          +0.00018513716955553553,
                                                          +0.00018511534832159750,
                                                          +0.00018509353857296362,
                                                          +0.00018507172762310579,
                                                          +0.00018504992815043486,
                                                          +0.00018502812747738884,
                                                          +0.00018500633827342006,
                                                          +0.00018498454786992430,
                                                          +0.00018496276892740368,
                                                          +0.00018494098878620336,
                                                          +0.00018491922009788387,
                                                          +0.00018489745021173112,
                                                          +0.00018487569177037244,
                                                          +0.00018485393213202617,
                                                          +0.00018483218393039485,
                                                          +0.00018481043453262076,
                                                          +0.00018478869656349012,
                                                          +0.00018476695739906085,
                                                          +0.00018474522965521097,
                                                          +0.00018472350071690576,
                                                          +0.00018470178319112361,
                                                          +0.00018468006447172863,
                                                          +0.00018465835715680791,
                                                          +0.00018463664864911601,
                                                          +0.00018461495153785718,
                                                          +0.00018459325323466822,
                                                          +0.00018457156631987852,
                                                          +0.00018454987821399875,
                                                          +0.00018452820148849222,
                                                          +0.00018450652357273500,
                                                          +0.00018448485702933235,
                                                          +0.00018446318929651751,
                                                          +0.00018444153292804634,
                                                          +0.00018441987537100069,
                                                          +0.00018439822917029513,
                                                          +0.00018437658178185214,
                                                          +0.00018435494574175316,
                                                          +0.00018433330851475290,
                                                          +0.00018431168262810827,
                                                          +0.00018429005555539772,
                                                          +0.00018426843981506175,
                                                          +0.00018424682288949453,
                                                          +0.00018422521728832827,
                                                          +0.00018420361050276463,
                                                          +0.00018418201503363595,
                                                          +0.00018416041838094298,
                                                          +0.00018413883303672630,
                                                          +0.00018411724650977762,
                                                          +0.00018409567128335419,
                                                          +0.00018407409487503017,
                                                          +0.00018405252975928777,
                                                          +0.00018403096346247550,
                                                          +0.00018400940845030860,
                                                          +0.00018398785225790180,
                                                          +0.00018396630734221161,
                                                          +0.00018394476124711064,
                                                          +0.00018392322642080491,
                                                          +0.00018390169041591681,
                                                          +0.00018388016567190998,
                                                          +0.00018385863975014843,
                                                          +0.00018383712508136159,
                                                          +0.00018381560923564681,
                                                          +0.00018379410463500764,
                                                          +0.00018377259885826668,
                                                          +0.00018375110431870952,
                                                          +0.00018372960860387581,
                                                          +0.00018370812411834159,
                                                          +0.00018368663845835535,
                                                          +0.00018366516401979160,
                                                          +0.00018364368840759955,
                                                          +0.00018362222400896037,
                                                          +0.00018360075843751591,
                                                          +0.00018357930407176203,
                                                          +0.00018355784853402504,
                                                          +0.00018353640419412374,
                                                          +0.00018351495868306077,
                                                          +0.00018349352436198597,
                                                          +0.00018347208887057009,
                                                          +0.00018345066456130216,
                                                          +0.00018342923908251305,
                                                          +0.00018340782477803898,
                                                          +0.00018338640930486288,
                                                          +0.00018336500499817617,
                                                          +0.00018334359952360582,
                                                          +0.00018332220520770660,
                                                          +0.00018330080972474132,
                                                          +0.00018327942539263611,
                                                          +0.00018325803989428170,
                                                          +0.00018323666553898371,
                                                          +0.00018321529001825258,
                                                          +0.00018319392563278139,
                                                          +0.00018317256008269246,
                                                          +0.00018315120566007425,
                                                          +0.00018312985007365286,
                                                          +0.00018310850560692029,
                                                          +0.00018308715997719834,
                                                          +0.00018306582545939056,
                                                          +0.00018304448977940650,
                                                          +0.00018302316520356918,
                                                          +0.00018300183946636775,
                                                          +0.00018298052482555301,
                                                          +0.00018295920902418574,
                                                          +0.00018293790431145215,
                                                          +0.00018291659843897682,
                                                          +0.00018289530364738943,
                                                          +0.00018287400769687032,
                                                          +0.00018285272281950071,
                                                          +0.00018283143678400854,
                                                          +0.00018281016181393467,
                                                          +0.00018278888568654675,
                                                          +0.00018276762061685296,
                                                          +0.00018274635439065291,
                                                          +0.00018272509921443008,
                                                          +0.00018270384288250803,
                                                          +0.00018268259759285341,
                                                          +0.00018266135114830583,
                                                          +0.00018264011573832311,
                                                          +0.00018261887917425291,
                                                          +0.00018259765363705231,
                                                          +0.00018257642694656894,
                                                          +0.00018255521127526678,
                                                          +0.00018253399445148594,
                                                          +0.00018251278863920525,
                                                          +0.00018249158167524910,
                                                          +0.00018247038571511914,
                                                          +0.00018244918860411629,
                                                          +0.00018242800248927273,
                                                          +0.00018240681522435805,
                                                          +0.00018238563894794294,
                                                          +0.00018236446152225777,
                                                          +0.00018234329507741960,
                                                          +0.00018232212748411160,
                                                          +0.00018230097086400506,
                                                          +0.00018227981309622821,
                                                          +0.00018225866629401457,
                                                          +0.00018223751834492939,
                                                          +0.00018221638135377594,
                                                          +0.00018219524321654910,
                                                          +0.00018217411602962984,
                                                          +0.00018215298769743438,
                                                          +0.00018213187030792941,
                                                          +0.00018211075177394480,
                                                          +0.00018208964417504054,
                                                          +0.00018206853543245252,
                                                          +0.00018204743761734182,
                                                          +0.00018202633865934247,
                                                          +0.00018200525062122438,
                                                          +0.00018198416144101203,
                                                          +0.00018196308317309194,
                                                          +0.00018194200376387123,
                                                          +0.00018192093525936092,
                                                          +0.00018189986561434286,
                                                          +0.00018187880686646027,
                                                          +0.00018185774697886207,
                                                          +0.00018183669798083144,
                                                          +0.00018181564784387660,
                                                          +0.00018179460858892856,
                                                          +0.00018177356819584704,
                                                          +0.00018175253867721821,
                                                          +0.00018173150802124592,
                                                          +0.00018171048823217950,
                                                          +0.00018168946730655883,
                                                          +0.00018166845724030414,
                                                          +0.00018164744603828365,
                                                          +0.00018162644568809621,
                                                          +0.00018160544420293070,
                                                          +0.00018158445356207236,
                                                          +0.00018156346178702294,
                                                          +0.00018154248084876161,
                                                          +0.00018152149877709561,
                                                          +0.00018150052753470556,
                                                          +0.00018147955515969639,
                                                          +0.00018145859360645819,
                                                          +0.00018143763092138567,
                                                          +0.00018141667905058583,
                                                          +0.00018139572604873591,
                                                          +0.00018137478385366737,
                                                          +0.00018135384052833209,
                                                          +0.00018133290800229391,
                                                          +0.00018131197434677178,
                                                          +0.00018129105148306915,
                                                          +0.00018127012749066451,
                                                          +0.00018124921428260896,
                                                          +0.00018122829994663260,
                                                          +0.00018120739638754165,
                                                          +0.00018118649170131046,
                                                          +0.00018116559778450788,
                                                          +0.00018114470274134482,
                                                          +0.00018112381846016059,
                                                          +0.00018110293305339495,
                                                          +0.00018108205840116506,
                                                          +0.00018106118262413215,
                                                          +0.00018104031759419884,
                                                          +0.00018101945144024018,
                                                          +0.00018099859602595180,
                                                          +0.00018097773948841500,
                                                          +0.00018095689368312603,
                                                          +0.00018093604675536483,
                                                          +0.00018091521055243592,
                                                          +0.00018089437322781028,
                                                          +0.00018087354662060806,
                                                          +0.00018085271889248391,
                                                          +0.00018083190187438128,
                                                          +0.00018081108373613070,
                                                          +0.00018079027630050660,
                                                          +0.00018076946774550794,
                                                          +0.00018074866988574727,
                                                          +0.00018072787090738472,
                                                          +0.00018070708261687873,
                                                          +0.00018068629320854276,
                                                          +0.00018066551448068849,
                                                          +0.00018064473463577554,
                                                          +0.00018062396546397638,
                                                          +0.00018060319517588903,
                                                          +0.00018058243555355426,
                                                          +0.00018056167481570110,
                                                          +0.00018054092473624608,
                                                          +0.00018052017354204181,
                                                          +0.00018049943299888803,
                                                          +0.00018047869134175338,
                                                          +0.00018045796032832825,
                                                          +0.00018043722820169000,
                                                          +0.00018041650671142711,
                                                          +0.00018039578410871806,
                                                          +0.00018037507213505697,
                                                          +0.00018035435904971596,
                                                          +0.00018033365658610222,
                                                          +0.00018031295301157416,
                                                          +0.00018029226005145934,
                                                          +0.00018027156598119512,
                                                          +0.00018025088251803688,
                                                          +0.00018023019794549336,
                                                          +0.00018020952397275532,
                                                          +0.00018018884889139543,
                                                          +0.00018016818440254716,
                                                          +0.00018014751880583984,
                                                          +0.00018012686379435697,
                                                          +0.00018010620767577707,
                                                          +0.00018008556213514114,
                                                          +0.00018006491548816958,
                                                          +0.00018004427941186818,
                                                          +0.00018002364222999184,
                                                          +0.00018000301561151846,
                                                          +0.00017998238788823011,
                                                          +0.00017996177072108425,
                                                          +0.00017994115244988272,
                                                          +0.00017992054472756988,
                                                          +0.00017989993590195996,
                                                          +0.00017987933761799143,
                                                          +0.00017985873823148375,
                                                          +0.00017983814937937696,
                                                          +0.00017981755942548817,
                                                          +0.00017979697999876638,
                                                          +0.00017977639947101907,
                                                          +0.00017975582946321147,
                                                          +0.00017973525835513415,
                                                          +0.00017971469775977588,
                                                          +0.00017969413606490299,
                                                          +0.00017967358487553503,
                                                          +0.00017965303258740692,
                                                          +0.00017963249079757626,
                                                          +0.00017961194790973919,
                                                          +0.00017959141551299865,
                                                          +0.00017957088201900470,
                                                          +0.00017955035900891309,
                                                          +0.00017952983490232048,
                                                          +0.00017950932127244230,
                                                          +0.00017948880654681488,
                                                          +0.00017946830229072073,
                                                          +0.00017944779693962825,
                                                          +0.00017942730205089454,
                                                          +0.00017940680606791279,
                                                          +0.00017938632054012178,
                                                          +0.00017936583391883225,
                                                          +0.00017934535774557202,
                                                          +0.00017932488047956225,
                                                          +0.00017930441365442677,
                                                          +0.00017928394573729002,
                                                          +0.00017926348825387907,
                                                          +0.00017924302967921440,
                                                          +0.00017922258153113370,
                                                          +0.00017920213229254616,
                                                          +0.00017918169347340716,
                                                          +0.00017916125356450755,
                                                          +0.00017914082406792761,
                                                          +0.00017912039348233255,
                                                          +0.00017909997330193482,
                                                          +0.00017907955203326679,
                                                          +0.00017905914116268015,
                                                          +0.00017903872920456740,
                                                          +0.00017901832763742671,
                                                          +0.00017899792498350336,
                                                          +0.00017897753271344910,
                                                          +0.00017895713935735507,
                                                          +0.00017893675637803365,
                                                          +0.00017891637231341455,
                                                          +0.00017889599861847813,
                                                          +0.00017887562383898546,
                                                          +0.00017885525942209194,
                                                          +0.00017883489392138296,
                                                          +0.00017881453877619609,
                                                          +0.00017879418254793382,
                                                          +0.00017877383666812304,
                                                          +0.00017875348970597632,
                                                          +0.00017873315308521688,
                                                          +0.00017871281538286022,
                                                          +0.00017869248801483310,
                                                          +0.00017867215956594683,
                                                          +0.00017865184144433879,
                                                          +0.00017863152224260897,
                                                          +0.00017861121336111248,
                                                          +0.00017859090340023102,
                                                          +0.00017857060375254424,
                                                          +0.00017855030302620854,
                                                          +0.00017853001260603553,
                                                          +0.00017850972110794900,
                                                          +0.00017848943990899933,
                                                          +0.00017846915763287090,
                                                          +0.00017844888564885999,
                                                          +0.00017842861258840443,
                                                          +0.00017840834981305338,
                                                          +0.00017838808596199100,
                                                          +0.00017836783238902665,
                                                          +0.00017834757774108377,
                                                          +0.00017832733336423849,
                                                          +0.00017830708791314679,
                                                          +0.00017828685272615888,
                                                          +0.00017826661646565587,
                                                          +0.00017824639046226922,
                                                          +0.00017822616338609813,
                                                          +0.00017820594656006228,
                                                          +0.00017818572866197204,
                                                          +0.00017816552100704213,
                                                          +0.00017814531228078735,
                                                          +0.00017812511379072422,
                                                          +0.00017810491423006503,
                                                          +0.00017808472489863532,
                                                          +0.00017806453449733769,
                                                          +0.00017804435431831352,
                                                          +0.00017802417307014883,
                                                          +0.00017800400203730815,
                                                          +0.00017798382993605366,
                                                          +0.00017796366804317986,
                                                          +0.00017794350508261835,
                                                          +0.00017792335232350059,
                                                          +0.00017790319849742058,
                                                          +0.00017788305486585355,
                                                          +0.00017786291016804907,
                                                          +0.00017784277565783312,
                                                          +0.00017782264008210392,
                                                          +0.00017780251468704501,
                                                          +0.00017778238822719629,
                                                          +0.00017776227194110606,
                                                          +0.00017774215459094884,
                                                          +0.00017772204740764444,
                                                          +0.00017770193916099525,
                                                          +0.00017768184107429935,
                                                          +0.00017766174192498024,
                                                          +0.00017764165292872132,
                                                          +0.00017762156287056002,
                                                          +0.00017760148295857195,
                                                          +0.00017758140198540173,
                                                          +0.00017756133115152408,
                                                          +0.00017754125925718381,
                                                          +0.00017752119749526166,
                                                          +0.00017750113467359581,
                                                          +0.00017748108197747971,
                                                          +0.00017746102822233824,
                                                          +0.00017744098458588454,
                                                          +0.00017742093989112296,
                                                          +0.00017740090530819329,
                                                          +0.00017738086966767280,
                                                          +0.00017736084413213452,
                                                          +0.00017734081753972169,
                                                          +0.00017732080104544763,
                                                          +0.00017730078349501472,
                                                          +0.00017728077603588319,
                                                          +0.00017726076752130784,
                                                          +0.00017724076909120278,
                                                          +0.00017722076960636834,
                                                          +0.00017720078019917913,
                                                          +0.00017718078973797424,
                                                          +0.00017716080934759585,
                                                          +0.00017714082790391486,
                                                          +0.00017712085652424770,
                                                          +0.00017710088409199042,
                                                          +0.00017708092171694035,
                                                          +0.00017706095829001203,
                                                          +0.00017704100491349055,
                                                          +0.00017702105048580198,
                                                          +0.00017700110610172596,
                                                          +0.00017698116066719335,
                                                          +0.00017696122526948530,
                                                          +0.00017694128882203049,
                                                          +0.00017692136240461814,
                                                          +0.00017690143493816825,
                                                          +0.00017688151749498503,
                                                          +0.00017686159900347285,
                                                          +0.00017684169052845748,
                                                          +0.00017682178100582121,
                                                          +0.00017680188149291797,
                                                          +0.00017678198093310115,
                                                          +0.00017676209037625977,
                                                          +0.00017674219877321151,
                                                          +0.00017672231716638710,
                                                          +0.00017670243451406187,
                                                          +0.00017668256185121513,
                                                          +0.00017666268814357295,
                                                          +0.00017664282441866977,
                                                          +0.00017662295964967603,
                                                          +0.00017660310485668795,
                                                          +0.00017658324902031339,
                                                          +0.00017656340315321727,
                                                          +0.00017654355624343820,
                                                          +0.00017652371929621630,
                                                          +0.00017650388130701435,
                                                          +0.00017648405327365439,
                                                          +0.00017646422419901659,
                                                          +0.00017644440507351167,
                                                          +0.00017642458490743057,
                                                          +0.00017640477468377913,
                                                          +0.00017638496342025256,
                                                          +0.00017636516209245851,
                                                          +0.00017634535972548972,
                                                          +0.00017632556728756232,
                                                          +0.00017630577381115992,
                                                          +0.00017628599025711387,
                                                          +0.00017626620566529188,
                                                          +0.00017624643098914714,
                                                          +0.00017622665527592497,
                                                          +0.00017620688947170694,
                                                          +0.00017618712263110935,
                                                          +0.00017616736569284877,
                                                          +0.00017614760771890583,
                                                          +0.00017612785964063883,
                                                          +0.00017610811052738601,
                                                          +0.00017608837130315399,
                                                          +0.00017606863104463222,
                                                          +0.00017604890066848193,
                                                          +0.00017602916925873725,
                                                          +0.00017600944772472091,
                                                          +0.00017598972515780488,
                                                          +0.00017597001245997986,
                                                          +0.00017595029872994925,
                                                          +0.00017593059486237838,
                                                          +0.00017591088996329546,
                                                          +0.00017589119492004675,
                                                          +0.00017587149884597887,
                                                          +0.00017585181262112583,
                                                          +0.00017583212536614583,
                                                          +0.00017581244795376711,
                                                          +0.00017579276951195305,
                                                          +0.00017577310090613268,
                                                          +0.00017575343127156797,
                                                          +0.00017573377146639526,
                                                          +0.00017571411063316861,
                                                          +0.00017569445962273812,
                                                          +0.00017567480758494351,
                                                          +0.00017565516536335513,
                                                          +0.00017563552211509182,
                                                          +0.00017561588867645068,
                                                          +0.00017559625421182317,
                                                          +0.00017557662955023975,
                                                          +0.00017555700386335786,
                                                          +0.00017553738797294782,
                                                          +0.00017551777105792663,
                                                          +0.00017549816393281091,
                                                          +0.00017547855578377072,
                                                          +0.00017545895741807554,
                                                          +0.00017543935802914198,
                                                          +0.00017541976841699874,
                                                          +0.00017540017778230264,
                                                          +0.00017538059691784807,
                                                          +0.00017536101503152544,
                                                          +0.00017534144290890150,
                                                          +0.00017532186976509375,
                                                          +0.00017530230637844752,
                                                          +0.00017528274197130115,
                                                          +0.00017526318731478499,
                                                          +0.00017524363163845175,
                                                          +0.00017522408570622332,
                                                          +0.00017520453875486023,
                                                          +0.00017518500154108227,
                                                          +0.00017516546330885146,
                                                          +0.00017514593480769209,
                                                          +0.00017512640528876097,
                                                          +0.00017510688549439335,
                                                          +0.00017508736468293458,
                                                          +0.00017506785358953708,
                                                          +0.00017504834147972850,
                                                          +0.00017502883908148469,
                                                          +0.00017500933566750915,
                                                          +0.00017498984195860791,
                                                          +0.00017497034723465364,
                                                          +0.00017495086220928891,
                                                          +0.00017493137616954931,
                                                          +0.00017491189982192016,
                                                          +0.00017489242246059372,
                                                          +0.00017487295478490445,
                                                          +0.00017485348609619489,
                                                          +0.00017483402708665498,
                                                          +0.00017481456706477109,
                                                          +0.00017479511671559514,
                                                          +0.00017477566535475104,
                                                          +0.00017475622366015873,
                                                          +0.00017473678095457349,
                                                          +0.00017471734790878979,
                                                          +0.00017469791385268772,
                                                          +0.00017467848944994270,
                                                          +0.00017465906403755317,
                                                          +0.00017463964827208196,
                                                          +0.00017462023149763967,
                                                          +0.00017460082436368256,
                                                          +0.00017458141622142710,
                                                          +0.00017456201771322947,
                                                          +0.00017454261819740570,
                                                          +0.00017452322830921818,
                                                          +0.00017450383741407600,
                                                          +0.00017448445614015416,
                                                          +0.00017446507385994868,
                                                          +0.00017444570119455315,
                                                          +0.00017442632752354443,
                                                          +0.00017440696346094121,
                                                          +0.00017438759839339456,
                                                          +0.00017436824292785453,
                                                          +0.00017434888645804025,
                                                          +0.00017432953958383937,
                                                          +0.00017431019170603281,
                                                          +0.00017429085341745228,
                                                          +0.00017427151412593400,
                                                          +0.00017425218441725993,
                                                          +0.00017423285370631557,
                                                          +0.00017421353257183913,
                                                          +0.00017419421043575920,
                                                          +0.00017417489786977685,
                                                          +0.00017415558430285719,
                                                          +0.00017413628029967014,
                                                          +0.00017411697529621148,
                                                          +0.00017409767985012613,
                                                          +0.00017407838340443427,
                                                          +0.00017405909650976219,
                                                          +0.00017403980861614796,
                                                          +0.00017402053026720562,
                                                          +0.00017400125091998495,
                                                          +0.00017398198111109390,
                                                          +0.00017396271030458780,
                                                          +0.00017394344903007449,
                                                          +0.00017392418675860885,
                                                          +0.00017390493401280504,
                                                          +0.00017388568027071089,
                                                          +0.00017386643604795304,
                                                          +0.00017384719082956640,
                                                          +0.00017382795512419625,
                                                          +0.00017380871842385815,
                                                          +0.00017378949123022226,
                                                          +0.00017377026304227881,
                                                          +0.00017375104435472872,
                                                          +0.00017373182467353093,
                                                          +0.00017371261448642341,
                                                          +0.00017369340330632732,
                                                          +0.00017367420161402389,
                                                          +0.00017365499892939049,
                                                          +0.00017363580572625782,
                                                          +0.00017361661153145316,
                                                          +0.00017359742681186281,
                                                          +0.00017357824110125791,
                                                          +0.00017355906485958645,
                                                          +0.00017353988762755721,
                                                          +0.00017352071985818619,
                                                          +0.00017350155109911364,
                                                          +0.00017348239179642948,
                                                          +0.00017346323150469953,
                                                          +0.00017344408066309369,
                                                          +0.00017342492883309717,
                                                          +0.00017340578644696607,
                                                          +0.00017338664307309885,
                                                          +0.00017336750913684383,
                                                          +0.00017334837421350660,
                                                          +0.00017332924872153402,
                                                          +0.00017331012224313259,
                                                          +0.00017329100518985352,
                                                          +0.00017327188715079842,
                                                          +0.00017325277853062919,
                                                          +0.00017323366892533614,
                                                          +0.00017321456873269764,
                                                          +0.00017319546755558711,
                                                          +0.00017317637578490545,
                                                          +0.00017315728303040275,
                                                          +0.00017313819967610893,
                                                          +0.00017311911533864446,
                                                          +0.00017310004039517425,
                                                          +0.00017308096446918333,
                                                          +0.00017306189793097736,
                                                          +0.00017304283041090008,
                                                          +0.00017302377227240408,
                                                          +0.00017300471315268552,
                                                          +0.00017298566340834997,
                                                          +0.00017296661268344004,
                                                          +0.00017294757132772039,
                                                          +0.00017292852899207400,
                                                          +0.00017290949601943043,
                                                          +0.00017289046206750723,
                                                          +0.00017287143747240501,
                                                          +0.00017285241189866964,
                                                          +0.00017283339567557879,
                                                          +0.00017281437847450066,
                                                          +0.00017279537061789607,
                                                          +0.00017277636178394964,
                                                          +0.00017275736228831100,
                                                          +0.00017273836181597535,
                                                          +0.00017271937067578740,
                                                          +0.00017270037855954664,
                                                          +0.00017268139576929878,
                                                          +0.00017266241200364173,
                                                          +0.00017264343755782832,
                                                          +0.00017262446213724879,
                                                          +0.00017260549603036895,
                                                          +0.00017258652894936558,
                                                          +0.00017256757117592328,
                                                          +0.00017254861242899940,
                                                          +0.00017252966298350348,
                                                          +0.00017251071256516739,
                                                          +0.00017249177144213149,
                                                          +0.00017247282934689626,
                                                          +0.00017245389654083881,
                                                          +0.00017243496276322232,
                                                          +0.00017241603826866666,
                                                          +0.00017239711280319164,
                                                          +0.00017237819661466576,
                                                          +0.00017235927945585968,
                                                          +0.00017234037156789653,
                                                          +0.00017232146271029176,
                                                          +0.00017230256311742895,
                                                          +0.00017228366255556262,
                                                          +0.00017226477125234266,
                                                          +0.00017224587898075660,
                                                          +0.00017222699596172679,
                                                          +0.00017220811197496768,
                                                          +0.00017218923723468001,
                                                          +0.00017217036152729947,
                                                          +0.00017215149506031070,
                                                          +0.00017213262762686489,
                                                          +0.00017211376942773666,
                                                          +0.00017209491026278664,
                                                          +0.00017207606032608526,
                                                          +0.00017205720942419674,
                                                          +0.00017203836774449345,
                                                          +0.00017201952510023709,
                                                          +0.00017200069167210759,
                                                          +0.00017198185728005861,
                                                          +0.00017196303209808354,
                                                          +0.00017194420595282212,
                                                          +0.00017192538901158688,
                                                          +0.00017190657110769769,
                                                          +0.00017188776240179238,
                                                          +0.00017186895273386499,
                                                          +0.00017185015225788447,
                                                          +0.00017183135082051319,
                                                          +0.00017181255856905698,
                                                          +0.00017179376535684081,
                                                          +0.00017177498132451321,
                                                          +0.00017175619633205588,
                                                          +0.00017173742051346589,
                                                          +0.00017171864373537592,
                                                          +0.00017169987612513719,
                                                          +0.00017168110755602761,
                                                          +0.00017166234814875868,
                                                          +0.00017164358778324745,
                                                          +0.00017162483657357140,
                                                          +0.00017160608440628113,
                                                          +0.00017158734138882577,
                                                          +0.00017156859741438368,
                                                          +0.00017154986258378156,
                                                          +0.00017153112679681965,
                                                          +0.00017151240014770793,
                                                          +0.00017149367254286284,
                                                          +0.00017147495406988345,
                                                          +0.00017145623464179656,
                                                          +0.00017143752433959608,
                                                          +0.00017141881308291339,
                                                          +0.00017140011094614303,
                                                          +0.00017138140785551522,
                                                          +0.00017136271387883093,
                                                          +0.00017134401894891341,
                                                          +0.00017132533312697572,
                                                          +0.00017130664635242845,
                                                          +0.00017128796867990262,
                                                          +0.00017126929005539034,
                                                          +0.00017125062052694622,
                                                          +0.00017123195004713825,
                                                          +0.00017121328865745038,
                                                          +0.00017119462631702071,
                                                          +0.00017117597306076827,
                                                          +0.00017115731885439557,
                                                          +0.00017113867372626235,
                                                          +0.00017112002764862978,
                                                          +0.00017110139064330427,
                                                          +0.00017108275268909985,
                                                          +0.00017106412380127502,
                                                          +0.00017104549396519122,
                                                          +0.00017102687318956483,
                                                          +0.00017100825146629877,
                                                          +0.00017098963879757313,
                                                          +0.00017097102518182658,
                                                          +0.00017095242061470860,
                                                          +0.00017093381510118797,
                                                          +0.00017091521863038916,
                                                          +0.00017089662121380546,
                                                          +0.00017087803283404192,
                                                          +0.00017085944350911072,
                                                          +0.00017084086321510322,
                                                          +0.00017082228197654465,
                                                          +0.00017080370976301851,
                                                          +0.00017078513660555739,
                                                          +0.00017076657246724249,
                                                          +0.00017074800738560821,
                                                          +0.00017072945131723903,
                                                          +0.00017071089430616550,
                                                          +0.00017069234630248111,
                                                          +0.00017067379735670685,
                                                          +0.00017065525741245086,
                                                          +0.00017063671652671906,
                                                          +0.00017061818463663970,
                                                          +0.00017059965180569789,
                                                          +0.00017058112796454792,
                                                          +0.00017056260318314842,
                                                          +0.00017054408738568515,
                                                          +0.00017052557064858471,
                                                          +0.00017050706288957002,
                                                          +0.00017048855419152999,
                                                          +0.00017047005446573025,
                                                          +0.00017045155380151657,
                                                          +0.00017043306210370277,
                                                          +0.00017041456946808575,
                                                          +0.00017039608579303344,
                                                          +0.00017037760118078811,
                                                          +0.00017035912552327729,
                                                          +0.00017034064892918320,
                                                          +0.00017032218128399838,
                                                          +0.00017030371270283952,
                                                          +0.00017028525306476980,
                                                          +0.00017026679249133468,
                                                          +0.00017024834085517366,
                                                          +0.00017022988828425544,
                                                          +0.00017021144464480123,
                                                          +0.00017019300007119740,
                                                          +0.00017017456442325262,
                                                          +0.00017015612784176536,
                                                          +0.00017013770018013715,
                                                          +0.00017011927158557298,
                                                          +0.00017010085190507295,
                                                          +0.00017008243129224297,
                                                          +0.00017006401958768726,
                                                          +0.00017004560695140709,
                                                          +0.00017002720321761624,
                                                          +0.00017000879855270592,
                                                          +0.00016999040278450507,
                                                          +0.00016997200608578919,
                                                          +0.00016995361827800790,
                                                          +0.00016993522954031554,
                                                          +0.00016991684968778779,
                                                          +0.00016989846890595237,
                                                          +0.00016988009700351673,
                                                          +0.00016986172417237630,
                                                          +0.00016984336021487570,
                                                          +0.00016982499532927270,
                                                          +0.00016980663931155459,
                                                          +0.00016978828236633591,
                                                          +0.00016976993428325214,
                                                          +0.00016975158527326919,
                                                          +0.00016973324511967614,
                                                          +0.00016971490403978469,
                                                          +0.00016969657181054311,
                                                          +0.00016967823865560340,
                                                          +0.00016965991434557854,
                                                          +0.00016964158911045532,
                                                          +0.00016962327271451681,
                                                          +0.00016960495539407915,
                                                          +0.00016958664690710110,
                                                          +0.00016956833749622269,
                                                          +0.00016955003691308354,
                                                          +0.00016953173540664227,
                                                          +0.00016951344272222502,
                                                          +0.00016949514911510332,
                                                          +0.00016947686432429532,
                                                          +0.00016945857861138007,
                                                          +0.00016944030170907304,
                                                          +0.00016942202388525545,
                                                          +0.00016940375486634553,
                                                          +0.00016938548492652125,
                                                          +0.00016936722378590909,
                                                          +0.00016934896172497821,
                                                          +0.00016933070845756869,
                                                          +0.00016931245427043553,
                                                          +0.00016929420887113811,
                                                          +0.00016927596255271167,
                                                          +0.00016925772501644003,
                                                          +0.00016923948656163350,
                                                          +0.00016922125688330572,
                                                          +0.00016920302628703669,
                                                          +0.00016918480446157538,
                                                          +0.00016916658171876585,
                                                          +0.00016914836774109784,
                                                          +0.00016913015284667415,
                                                          +0.00016911194671173071,
                                                          +0.00016909373966062370,
                                                          +0.00016907554136334036,
                                                          +0.00016905734215048507,
                                                          +0.00016903915168580189,
                                                          +0.00016902096030613773,
                                                          +0.00016900277766899904,
                                                          +0.00016898459411746985,
                                                          +0.00016896641930282434,
                                                          +0.00016894824357437834,
                                                          +0.00016893007657717895,
                                                          +0.00016891190866676862,
                                                          +0.00016889374948197278,
                                                          +0.00016887558938455499,
                                                          +0.00016885743800712440,
                                                          +0.00016883928571766033,
                                                          +0.00016882114214256097,
                                                          +0.00016880299765601616,
                                                          +0.00016878486187821845,
                                                          +0.00016876672518956277,
                                                          +0.00016874859720404132,
                                                          +0.00016873046830824889,
                                                          +0.00016871234810998277,
                                                          +0.00016869422700203213,
                                                          +0.00016867611458600459,
                                                          +0.00016865800126087858,
                                                          +0.00016863989662207723,
                                                          +0.00016862179107476289,
                                                          +0.00016860369420817971,
                                                          +0.00016858559643366847,
                                                          +0.00016856750733429968,
                                                          +0.00016854941732758730,
                                                          +0.00016853133599043334,
                                                          +0.00016851325374651983,
                                                          +0.00016849518016658554,
                                                          +0.00016847710568047519,
                                                          +0.00016845903985276969,
                                                          +0.00016844097311947105,
                                                          +0.00016842291503900772,
                                                          +0.00016840485605353368,
                                                          +0.00016838680571533014,
                                                          +0.00016836875447269790,
                                                          +0.00016835071187177604,
                                                          +0.00016833266836700694,
                                                          +0.00016831463349839304,
                                                          +0.00016829659772651283,
                                                          +0.00016827857058523721,
                                                          +0.00016826054254127581,
                                                          +0.00016824252312237325,
                                                          +0.00016822450280136488,
                                                          +0.00016820649109987432,
                                                          +0.00016818847849685748,
                                                          +0.00016817047450782210,
                                                          +0.00016815246961783941,
                                                          +0.00016813447333630671,
                                                          +0.00016811647615440519,
                                                          +0.00016809848757542683,
                                                          +0.00016808049809665762,
                                                          +0.00016806251721528958,
                                                          +0.00016804453543470811,
                                                          +0.00016802656224601048,
                                                          +0.00016800858815867650,
                                                          +0.00016799062265771366,
                                                          +0.00016797265625869101,
                                                          +0.00016795469844053156,
                                                          +0.00016793673972488834,
                                                          +0.00016791878958460514,
                                                          +0.00016790083854741365,
                                                          +0.00016788289608008373,
                                                          +0.00016786495271642058,
                                                          +0.00016784701791712513,
                                                          +0.00016782908222207098,
                                                          +0.00016781115508589551,
                                                          +0.00016779322705453534,
                                                          +0.00016777530757656946,
                                                          +0.00016775738720399249,
                                                          +0.00016773947537932998,
                                                          +0.00016772156266062945,
                                                          +0.00016770365848436846,
                                                          +0.00016768575341464197,
                                                          +0.00016766785688188463,
                                                          +0.00016764995945623386,
                                                          +0.00016763207056208658,
                                                          +0.00016761418077561755,
                                                          +0.00016759629951519081,
                                                          +0.00016757841736301344,
                                                          +0.00016756054373142217,
                                                          +0.00016754266920865078,
                                                          +0.00016752480320101373,
                                                          +0.00016750693630276681,
                                                          +0.00016748907791420706,
                                                          +0.00016747121863560711,
                                                          +0.00016745336786125198,
                                                          +0.00016743551619742577,
                                                          +0.00016741767303240658,
                                                          +0.00016739982897848502,
                                                          +0.00016738199341793732,
                                                          +0.00016736415696905549,
                                                          +0.00016734632900811889,
                                                          +0.00016732850015941593,
                                                          +0.00016731067979323434,
                                                          +0.00016729285853985356,
                                                          +0.00016727504576357496,
                                                          +0.00016725723210066391,
                                                          +0.00016723942690944032,
                                                          +0.00016722162083215056,
                                                          +0.00016720382322113825,
                                                          +0.00016718602472462550,
                                                          +0.00016716823468898478,
                                                          +0.00016715044376840895,
                                                          +0.00016713266130330428,
                                                          +0.00016711487795382929,
                                                          +0.00016709710305442928,
                                                          +0.00016707932727122327,
                                                          +0.00016706155993270055,
                                                          +0.00016704379171093571,
                                                          +0.00016702603192846711,
                                                          +0.00016700827126331981,
                                                          +0.00016699051903208613,
                                                          +0.00016697276591873668,
                                                          +0.00016695502123392306,
                                                          +0.00016693727566755603,
                                                          +0.00016691953852435146,
                                                          +0.00016690180050015540,
                                                          +0.00016688407089375309,
                                                          +0.00016686634040692078,
                                                          +0.00016684861833251790,
                                                          +0.00016683089537824607,
                                                          +0.00016681318083104407,
                                                          +0.00016679546540453353,
                                                          +0.00016677775837973779,
                                                          +0.00016676005047619357,
                                                          +0.00016674235096901351,
                                                          +0.00016672465058364456,
                                                          +0.00016670695858929377,
                                                          +0.00016668926571731328,
                                                          +0.00016667158123100925,
                                                          +0.00016665389586763418,
                                                          +0.00016663621888459880,
                                                          +0.00016661854102505036,
                                                          +0.00016660087154050926,
                                                          +0.00016658320118001279,
                                                          +0.00016656553918919571,
                                                          +0.00016654787632298048,
                                                          +0.00016653022182112122,
                                                          +0.00016651256644442054,
                                                          +0.00016649491942675705,
                                                          +0.00016647727153480834,
                                                          +0.00016645963199658244,
                                                          +0.00016644199158462715,
                                                          +0.00016642435952108478,
                                                          +0.00016640672658436829,
                                                          +0.00016638910199075942,
                                                          +0.00016637147652453130,
                                                          +0.00016635385939610988,
                                                          +0.00016633624139562357,
                                                          +0.00016631863172764764,
                                                          +0.00016630102118816070,
                                                          +0.00016628341897589232,
                                                          +0.00016626581589266628,
                                                          +0.00016624822113137135,
                                                          +0.00016623062549967179,
                                                          +0.00016621303818462041,
                                                          +0.00016619544999971691,
                                                          +0.00016617787012618309,
                                                          +0.00016616028938334920,
                                                          +0.00016614271694661100,
                                                          +0.00016612514364112426,
                                                          +0.00016610757863646372,
                                                          +0.00016609001276360571,
                                                          +0.00016607245518630886,
                                                          +0.00016605489674136512,
                                                          +0.00016603734658672190,
                                                          +0.00016601979556498200,
                                                          +0.00016600225282828644,
                                                          +0.00016598470922504389,
                                                          +0.00016596717390159394,
                                                          +0.00016594963771214625,
                                                          +0.00016593210979724382,
                                                          +0.00016591458101689247,
                                                          +0.00016589706050584350,
                                                          +0.00016587953912989385,
                                                          +0.00016586202601800822,
                                                          +0.00016584451204176975,
                                                          +0.00016582700632436129,
                                                          +0.00016580949974314736,
                                                          +0.00016579200141553379,
                                                          +0.00016577450222466167,
                                                          +0.00016575701128216486,
                                                          +0.00016573951947695592,
                                                          +0.00016572203591490141,
                                                          +0.00016570455149068078,
                                                          +0.00016568707530439832,
                                                          +0.00016566959825649528,
                                                          +0.00016565212944131833,
                                                          +0.00016563465976506591,
                                                          +0.00016561719831633206,
                                                          +0.00016559973600706733,
                                                          +0.00016558228192011795,
                                                          +0.00016556482697318195,
                                                          +0.00016554738024336245,
                                                          +0.00016552993265409995,
                                                          +0.00016551249327675962,
                                                          +0.00016549505304051950,
                                                          +0.00016547762101101160,
                                                          +0.00016546018812314665,
                                                          +0.00016544276343682821,
                                                          +0.00016542533789269502,
                                                          +0.00016540792054492716,
                                                          +0.00016539050233988639,
                                                          +0.00016537309232603396,
                                                          +0.00016535568145545010,
                                                          +0.00016533827877088196,
                                                          +0.00016532087523012332,
                                                          +0.00016530347987021222,
                                                          +0.00016528608365465116,
                                                          +0.00016526869561477378,
                                                          +0.00016525130671978636,
                                                          +0.00016523392599532321,
                                                          +0.00016521654441628965,
                                                          +0.00016519917100262508,
                                                          +0.00016518179673492931,
                                                          +0.00016516443062745159,
                                                          +0.00016514706366648136,
                                                          +0.00016512970486058282,
                                                          +0.00016511234520172989,
                                                          +0.00016509499369280647,
                                                          +0.00016507764133146641,
                                                          +0.00016506029711491803,
                                                          +0.00016504295204649040,
                                                          +0.00016502561511772082,
                                                          +0.00016500827733760892,
                                                          +0.00016499094769202574,
                                                          +0.00016497361719563667,
                                                          +0.00016495629482865158,
                                                          +0.00016493897161139648,
                                                          +0.00016492165651842461,
                                                          +0.00016490434057571838,
                                                          +0.00016488703275217904,
                                                          +0.00016486972407944031,
                                                          +0.00016485242352075661,
                                                          +0.00016483512211340812,
                                                          +0.00016481782881500682,
                                                          +0.00016480053466847493,
                                                          +0.00016478324862578687,
                                                          +0.00016476596173550196,
                                                          +0.00016474868294396160,
                                                          +0.00016473140330535769,
                                                          +0.00016471413176040350,
                                                          +0.00016469685936891851,
                                                          +0.00016467959506599272,
                                                          +0.00016466232991706845,
                                                          +0.00016464507285161707,
                                                          +0.00016462781494069919,
                                                          +0.00016461056510817202,
                                                          +0.00016459331443070984,
                                                          +0.00016457607182656059,
                                                          +0.00016455882837800742,
                                                          +0.00016454159299769352,
                                                          +0.00016452435677350634,
                                                          +0.00016450712861248916,
                                                          +0.00016448989960812877,
                                                          +0.00016447267866187337,
                                                          +0.00016445545687280449,
                                                          +0.00016443824313677971,
                                                          +0.00016442102855847074,
                                                          +0.00016440382202814928,
                                                          +0.00016438661465607244,
                                                          +0.00016436941532693076,
                                                          +0.00016435221515656210,
                                                          +0.00016433502302408046,
                                                          +0.00016431783005089979,
                                                          +0.00016430064511056223,
                                                          +0.00016428345933005315,
                                                          +0.00016426628157734739,
                                                          +0.00016424910298499723,
                                                          +0.00016423193241541496,
                                                          +0.00016421476100671482,
                                                          +0.00016419759761575141,
                                                          +0.00016418043338619633,
                                                          +0.00016416327716935078,
                                                          +0.00016414612011443933,
                                                          +0.00016412897106721463,
                                                          +0.00016411182118244931,
                                                          +0.00016409467930035207,
                                                          +0.00016407753658123896,
                                                          +0.00016406040185977964,
                                                          +0.00016404326630182887,
                                                          +0.00016402613873652143,
                                                          +0.00016400901033524658,
                                                          +0.00016399188992160908,
                                                          +0.00016397476867252770,
                                                          +0.00016395765540608165,
                                                          +0.00016394054130471490,
                                                          +0.00016392343518098571,
                                                          +0.00016390632822285844,
                                                          +0.00016388922923737530,
                                                          +0.00016387212941801625,
                                                          +0.00016385503756631190,
                                                          +0.00016383794488125346,
                                                          +0.00016382086015886450,
                                                          +0.00016380377460364286,
                                                          +0.00016378669700610948,
                                                          +0.00016376961857626437,
                                                          +0.00016375254809913070,
                                                          +0.00016373547679020571,
                                                          +0.00016371841342901946,
                                                          +0.00016370134923656196,
                                                          +0.00016368429298687444,
                                                          +0.00016366723590643535,
                                                          +0.00016365018676380182,
                                                          +0.00016363313679093594,
                                                          +0.00016361609475091515,
                                                          +0.00016359905188118075,
                                                          +0.00016358201693933530,
                                                          +0.00016356498116829458,
                                                          +0.00016354795332019064,
                                                          +0.00016353092464340939,
                                                          +0.00016351390388461695,
                                                          +0.00016349688229766462,
                                                          +0.00016347986862375732,
                                                          +0.00016346285412220709,
                                                          +0.00016344584752876220,
                                                          +0.00016342884010819096,
                                                          +0.00016341184059078946,
                                                          +0.00016339484024677781,
                                                          +0.00016337784780100429,
                                                          +0.00016336085452913640,
                                                          +0.00016334386915057926,
                                                          +0.00016332688294644308,
                                                          +0.00016330990463069429,
                                                          +0.00016329292548988134,
                                                          +0.00016327595423253657,
                                                          +0.00016325898215064212,
                                                          +0.00016324201794730070,
                                                          +0.00016322505291992357,
                                                          +0.00016320809576618849,
                                                          +0.00016319113778893131,
                                                          +0.00016317418768040920,
                                                          +0.00016315723674887819,
                                                          +0.00016314029368117930,
                                                          +0.00016312334979098428,
                                                          +0.00016310641375972258,
                                                          +0.00016308947690647703,
                                                          +0.00016307254790727012,
                                                          +0.00016305561808659119,
                                                          +0.00016303869611506024,
                                                          +0.00016302177332256878,
                                                          +0.00016300485837433861,
                                                          +0.00016298794260565904,
                                                          +0.00016297103467635815,
                                                          +0.00016295412592711852,
                                                          +0.00016293722501237900,
                                                          +0.00016292032327821089,
                                                          +0.00016290342937366853,
                                                          +0.00016288653465020735,
                                                          +0.00016286964775150145,
                                                          +0.00016285276003438607,
                                                          +0.00016283588013715964,
                                                          +0.00016281899942203259,
                                                          +0.00016280212652193218,
                                                          +0.00016278525280443969,
                                                          +0.00016276838689711543,
                                                          +0.00016275152017290726,
                                                          +0.00016273466125401294,
                                                          +0.00016271780151874246,
                                                          +0.00016270094958393549,
                                                          +0.00016268409683325958,
                                                          +0.00016266725187820101,
                                                          +0.00016265040610778032,
                                                          +0.00016263356812813468,
                                                          +0.00016261672933363328,
                                                          +0.00016259989832506880,
                                                          +0.00016258306650215450,
                                                          +0.00016256624246034287,
                                                          +0.00016254941760468703,
                                                          +0.00016253260052530357,
                                                          +0.00016251578263258115,
                                                          +0.00016249897251130479,
                                                          +0.00016248216157719420,
                                                          +0.00016246535840970750,
                                                          +0.00016244855442989082,
                                                          +0.00016243175821187977,
                                                          +0.00016241496118204270,
                                                          +0.00016239817190919699,
                                                          +0.00016238138182502872,
                                                          +0.00016236459949304149,
                                                          +0.00016234781635023478,
                                                          +0.00016233104095480285,
                                                          +0.00016231426474905405,
                                                          +0.00016229749628587768,
                                                          +0.00016228072701288668,
                                                          +0.00016226396547766979,
                                                          +0.00016224720313314008,
                                                          +0.00016223044852158998,
                                                          +0.00016221369310122851,
                                                          +0.00016219694540905627,
                                                          +0.00016218019690857362,
                                                          +0.00016216345613149371,
                                                          +0.00016214671454660394,
                                                          +0.00016212998068033442,
                                                          +0.00016211324600675519,
                                                          +0.00016209651904701753,
                                                          +0.00016207979128047000,
                                                          +0.00016206307122298942,
                                                          +0.00016204635035919822,
                                                          +0.00016202963719970332,
                                                          +0.00016201292323439671,
                                                          +0.00016199621696861966,
                                                          +0.00016197950989752947,
                                                          +0.00016196281052120588,
                                                          +0.00016194611034006727,
                                                          +0.00016192941784893639,
                                                          +0.00016191272455348818,
                                                          +0.00016189603894329268,
                                                          +0.00016187935252927717,
                                                          +0.00016186267379576330,
                                                          +0.00016184599425892625,
                                                          +0.00016182932239784374,
                                                          +0.00016181264973393452,
                                                          +0.00016179598474103655,
                                                          +0.00016177931894580794,
                                                          +0.00016176266081685127,
                                                          +0.00016174600188605961,
                                                          +0.00016172935061680438,
                                                          +0.00016171269854620946,
                                                          +0.00016169605413241949,
                                                          +0.00016167940891778453,
                                                          +0.00016166277135522698,
                                                          +0.00016164613299231892,
                                                          +0.00016162950227676439,
                                                          +0.00016161287076135340,
                                                          +0.00016159624688857611,
                                                          +0.00016157962221643595,
                                                          +0.00016156300518221358,
                                                          +0.00016154638734912149,
                                                          +0.00016152977714923507,
                                                          +0.00016151316615097176,
                                                          +0.00016149656278120587,
                                                          +0.00016147995861355552,
                                                          +0.00016146336206969823,
                                                          +0.00016144676472844844,
                                                          +0.00016143017500629128,
                                                          +0.00016141358448723325,
                                                          +0.00016139700158257112,
                                                          +0.00016138041788149926,
                                                          +0.00016136384179013069,
                                                          +0.00016134726490284310,
                                                          +0.00016133069562056986,
                                                          +0.00016131412554286806,
                                                          +0.00016129756306549546,
                                                          +0.00016128099979318428,
                                                          +0.00016126444411652117,
                                                          +0.00016124788764540905,
                                                          +0.00016123133876526756,
                                                          +0.00016121478909116623,
                                                          +0.00016119824700336206,
                                                          +0.00016118170412208680,
                                                          +0.00016116516882243900,
                                                          +0.00016114863272980850,
                                                          +0.00016113210421413957,
                                                          +0.00016111557490597594,
                                                          +0.00016109905317011181,
                                                          +0.00016108253064224052,
                                                          +0.00016106601568201064,
                                                          +0.00016104949993026073,
                                                          +0.00016103299174149776,
                                                          +0.00016101648276170154,
                                                          +0.00016099998134024181,
                                                          +0.00016098347912823512,
                                                          +0.00016096698446991809,
                                                          +0.00016095048902154020,
                                                          +0.00016093400112220893,
                                                          +0.00016091751243330234,
                                                          +0.00016090103128880333,
                                                          +0.00016088454935521415,
                                                          +0.00016086807496139716,
                                                          +0.00016085159977897480,
                                                          +0.00016083513213169308,
                                                          +0.00016081866369629035,
                                                          +0.00016080220279140054,
                                                          +0.00016078574109887369,
                                                          +0.00016076928693223576,
                                                          +0.00016075283197844437,
                                                          +0.00016073638454592180,
                                                          +0.00016071993632672894,
                                                          +0.00016070349562418841,
                                                          +0.00016068705413546045,
                                                          +0.00016067062015877223,
                                                          +0.00016065418539637890,
                                                          +0.00016063775814141647,
                                                          +0.00016062133010123100,
                                                          +0.00016060490956387131,
                                                          +0.00016058848824177018,
                                                          +0.00016057207441789353,
                                                          +0.00016055565980975663,
                                                          +0.00016053925269524671,
                                                          +0.00016052284479695734,
                                                          +0.00016050644438770113,
                                                          +0.00016049004319514590,
                                                          +0.00016047364948703382,
                                                          +0.00016045725499610279,
                                                          +0.00016044086798502855,
                                                          +0.00016042448019161495,
                                                          +0.00016040809987347571,
                                                          +0.00016039171877347641,
                                                          +0.00016037534514417253,
                                                          +0.00016035897073348747,
                                                          +0.00016034260378892286,
                                                          +0.00016032623606345545,
                                                          +0.00016030987579953717,
                                                          +0.00016029351475519424,
                                                          +0.00016027716116783281,
                                                          +0.00016026080680052435,
                                                          +0.00016024445988563362,
                                                          +0.00016022811219127317,
                                                          +0.00016021177194477021,
                                                          +0.00016019543091927449,
                                                          +0.00016017909733707982,
                                                          +0.00016016276297636892,
                                                          +0.00016014643605440638,
                                                          +0.00016013010835440368,
                                                          +0.00016011378808860040,
                                                          +0.00016009746704523275,
                                                          +0.00016008115343151913,
                                                          +0.00016006483904071651,
                                                          +0.00016004853207502638,
                                                          +0.00016003222433272221,
                                                          +0.00016001592401099262,
                                                          +0.00015999962291312361,
                                                          +0.00015998332923129502,
                                                          +0.00015996703477380114,
                                                          +0.00015995074772781715,
                                                          +0.00015993445990664175,
                                                          +0.00015991817949244946,
                                                          +0.00015990189830353911,
                                                          +0.00015988562451708880,
                                                          +0.00015986934995639346,
                                                          +0.00015985308279363873,
                                                          +0.00015983681485711162,
                                                          +0.00015982055431400932,
                                                          +0.00015980429299760699,
                                                          +0.00015978803907011735,
                                                          +0.00015977178436979948,
                                                          +0.00015975553705388596,
                                                          +0.00015973928896561578,
                                                          +0.00015972304825724511,
                                                          +0.00015970680677698893,
                                                          +0.00015969057267213119,
                                                          +0.00015967433779585859,
                                                          +0.00015965811029048708,
                                                          +0.00015964188201417106,
                                                          +0.00015962566110426242,
                                                          +0.00015960943942387913,
                                                          +0.00015959322510541313,
                                                          +0.00015957701001694218,
                                                          +0.00015956080228590191,
                                                          +0.00015954459378532594,
                                                          +0.00015952839263769783,
                                                          +0.00015951219072100290,
                                                          +0.00015949599615277656,
                                                          +0.00015947980081595180,
                                                          +0.00015946361282312025,
                                                          +0.00015944742406215831,
                                                          +0.00015943124264071758,
                                                          +0.00015941506045161419,
                                                          +0.00015939888559756373,
                                                          +0.00015938270997631790,
                                                          +0.00015936654168566036,
                                                          +0.00015935037262827437,
                                                          +0.00015933421089701566,
                                                          +0.00015931804839949503,
                                                          +0.00015930189322364423,
                                                          +0.00015928573728199777,
                                                          +0.00015926958865756729,
                                                          +0.00015925343926780688,
                                                          +0.00015923729719081236,
                                                          +0.00015922115434895324,
                                                          +0.00015920501881541352,
                                                          +0.00015918888251747421,
                                                          +0.00015917275352341130,
                                                          +0.00015915662376541343,
                                                          +0.00015914050130685266,
                                                          +0.00015912437808482117,
                                                          +0.00015910826215779098,
                                                          +0.00015909214546775408,
                                                          +0.00015907603606828618,
                                                          +0.00015905992590627515,
                                                          +0.00015904382303040450,
                                                          +0.00015902771939245390,
                                                          +0.00015901162303621870,
                                                          +0.00015899552591836628,
                                                          +0.00015897943607780785,
                                                          +0.00015896334547609459,
                                                          +0.00015894726214725751,
                                                          +0.00015893117805772765,
                                                          +0.00015891510123665963,
                                                          +0.00015889902365536049,
                                                          +0.00015888295333811256,
                                                          +0.00015886688226109477,
                                                          +0.00015885081844372101,
                                                          +0.00015883475386703838,
                                                          +0.00015881869654559615,
                                                          +0.00015880263846530554,
                                                          +0.00015878658763585546,
                                                          +0.00015877053604801709,
                                                          +0.00015875449170662281,
                                                          +0.00015873844660730006,
                                                          +0.00015872240875002846,
                                                          +0.00015870637013528781,
                                                          +0.00015869033875820900,
                                                          +0.00015867430662412014,
                                                          +0.00015865828172330736,
                                                          +0.00015864225606594323,
                                                          +0.00015862623763747289,
                                                          +0.00015861021845290949,
                                                          +0.00015859420649286124,
                                                          +0.00015857819377717786,
                                                          +0.00015856218828163435,
                                                          +0.00015854618203091340,
                                                          +0.00015853018299596055,
                                                          +0.00015851418320628747,
                                                          +0.00015849819062801449,
                                                          +0.00015848219729547804,
                                                          +0.00015846621116997708,
                                                          +0.00015845022429066910,
                                                          +0.00015843424461403562,
                                                          +0.00015841826418405117,
                                                          +0.00015840229095238361,
                                                          +0.00015838631696782086,
                                                          +0.00015837035017722098,
                                                          +0.00015835438263418121,
                                                          +0.00015833842228075380,
                                                          +0.00015832246117534156,
                                                          +0.00015830650725519457,
                                                          +0.00015829055258351739,
                                                          +0.00015827460509276192,
                                                          +0.00015825865685093060,
                                                          +0.00015824271578568092,
                                                          +0.00015822677396980919,
                                                          +0.00015821083932618271,
                                                          +0.00015819490393238766,
                                                          +0.00015817897570650488,
                                                          +0.00015816304673090667,
                                                          +0.00015814712491889112,
                                                          +0.00015813120235761300,
                                                          +0.00015811528695559149,
                                                          +0.00015809937080475984,
                                                          +0.00015808346180886214,
                                                          +0.00015806755206460648,
                                                          +0.00015805164947096562,
                                                          +0.00015803574612941845,
                                                          +0.00015801984993417064,
                                                          +0.00015800395299146773,
                                                          +0.00015798806319075205,
                                                          +0.00015797217264303229,
                                                          +0.00015795628923299102,
                                                          +0.00015794040507639630,
                                                          +0.00015792452805317493,
                                                          +0.00015790865028385032,
                                                          +0.00015789277964359724,
                                                          +0.00015787690825769089,
                                                          +0.00015786104399655775,
                                                          +0.00015784517899022086,
                                                          +0.00015782932110436240,
                                                          +0.00015781346247374929,
                                                          +0.00015779761095932325,
                                                          +0.00015778175870059141,
                                                          +0.00015776591355375866,
                                                          +0.00015775006766306852,
                                                          +0.00015773422887999303,
                                                          +0.00015771838935350820,
                                                          +0.00015770255693035702,
                                                          +0.00015768672376424424,
                                                          +0.00015767089769718742,
                                                          +0.00015765507088761637,
                                                          +0.00015763925117282714,
                                                          +0.00015762343071597067,
                                                          +0.00015760761734962529,
                                                          +0.00015759180324165933,
                                                          +0.00015757599621993708,
                                                          +0.00015756018845704056,
                                                          +0.00015754438777612387,
                                                          +0.00015752858635447882,
                                                          +0.00015751279201055312,
                                                          +0.00015749699692634467,
                                                          +0.00015748120891559853,
                                                          +0.00015746542016501478,
                                                          +0.00015744963848363970,
                                                          +0.00015743385606287199,
                                                          +0.00015741808070706255,
                                                          +0.00015740230461230498,
                                                          +0.00015738653557825899,
                                                          +0.00015737076580570891,
                                                          +0.00015735500308962702,
                                                          +0.00015733923963548487,
                                                          +0.00015732348323357079,
                                                          +0.00015730772609403993,
                                                          +0.00015729197600250055,
                                                          +0.00015727622517378739,
                                                          +0.00015726048138883252,
                                                          +0.00015724473686714661,
                                                          +0.00015722899938498910,
                                                          +0.00015721326116654289,
                                                          +0.00015719752998339867,
                                                          +0.00015718179806440780,
                                                          +0.00015716607317649579,
                                                          +0.00015715034755317881,
                                                          +0.00015713462895672098,
                                                          +0.00015711890962529946,
                                                          +0.00015710319731652075,
                                                          +0.00015708748427321935 };
}

#endif
