"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FunctionTransformer = void 0;
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphql_mapping_template_1 = require("graphql-mapping-template");
const graphql_transformer_common_1 = require("graphql-transformer-common");
const cloudform_types_1 = require("cloudform-types");
const lambdaArns_1 = require("./lambdaArns");
const FUNCTION_DIRECTIVE_STACK = 'FunctionDirectiveStack';
class FunctionTransformer extends graphql_transformer_core_1.Transformer {
    constructor() {
        super('FunctionTransformer', graphql_transformer_core_1.gql `
        directive @function(name: String!, region: String) repeatable on FIELD_DEFINITION
      `);
        this.field = (parent, definition, directive, ctx) => {
            const { name, region } = graphql_transformer_core_1.getDirectiveArguments(directive);
            if (!name) {
                throw new graphql_transformer_core_1.TransformerContractError(`Must supply a 'name' to @function.`);
            }
            const iamRoleKey = graphql_transformer_common_1.FunctionResourceIDs.FunctionIAMRoleID(name, region);
            if (!ctx.getResource(iamRoleKey)) {
                ctx.setResource(iamRoleKey, this.role(name, region));
                ctx.mapResourceToStack(FUNCTION_DIRECTIVE_STACK, iamRoleKey);
            }
            const lambdaDataSourceKey = graphql_transformer_common_1.FunctionResourceIDs.FunctionDataSourceID(name, region);
            if (!ctx.getResource(lambdaDataSourceKey)) {
                ctx.setResource(lambdaDataSourceKey, this.datasource(name, region));
                ctx.mapResourceToStack(FUNCTION_DIRECTIVE_STACK, lambdaDataSourceKey);
            }
            const functionConfigurationKey = graphql_transformer_common_1.FunctionResourceIDs.FunctionAppSyncFunctionConfigurationID(name, region);
            if (!ctx.getResource(functionConfigurationKey)) {
                ctx.setResource(functionConfigurationKey, this.function(name, region));
                ctx.mapResourceToStack(FUNCTION_DIRECTIVE_STACK, functionConfigurationKey);
            }
            const typeName = parent.name.value;
            const fieldName = definition.name.value;
            const resolverKey = graphql_transformer_common_1.ResolverResourceIDs.ResolverResourceID(typeName, fieldName);
            const resolver = ctx.getResource(resolverKey);
            if (!resolver) {
                ctx.setResource(resolverKey, this.resolver(typeName, fieldName, name, region));
                ctx.mapResourceToStack(FUNCTION_DIRECTIVE_STACK, resolverKey);
            }
            else if (resolver.Properties.Kind === 'PIPELINE') {
                ctx.setResource(resolverKey, this.appendFunctionToResolver(resolver, graphql_transformer_common_1.FunctionResourceIDs.FunctionAppSyncFunctionConfigurationID(name, region)));
            }
        };
        this.role = (name, region) => {
            return new cloudform_types_1.IAM.Role({
                RoleName: cloudform_types_1.Fn.If(graphql_transformer_common_1.ResourceConstants.CONDITIONS.HasEnvironmentParameter, cloudform_types_1.Fn.Join('-', [
                    graphql_transformer_common_1.FunctionResourceIDs.FunctionIAMRoleName(name, true),
                    cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
                    cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.Env),
                ]), cloudform_types_1.Fn.Join('-', [
                    graphql_transformer_common_1.FunctionResourceIDs.FunctionIAMRoleName(name, false),
                    cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
                ])),
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [
                        {
                            Effect: 'Allow',
                            Principal: {
                                Service: 'appsync.amazonaws.com',
                            },
                            Action: 'sts:AssumeRole',
                        },
                    ],
                },
                Policies: [
                    {
                        PolicyName: 'InvokeLambdaFunction',
                        PolicyDocument: {
                            Version: '2012-10-17',
                            Statement: [
                                {
                                    Effect: 'Allow',
                                    Action: ['lambda:InvokeFunction'],
                                    Resource: lambdaArns_1.lambdaArnResource(name, region),
                                },
                            ],
                        },
                    },
                ],
            });
        };
        this.datasource = (name, region) => {
            return new cloudform_types_1.AppSync.DataSource({
                ApiId: cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.AppSyncApiId),
                Name: graphql_transformer_common_1.FunctionResourceIDs.FunctionDataSourceID(name, region),
                Type: 'AWS_LAMBDA',
                ServiceRoleArn: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.FunctionResourceIDs.FunctionIAMRoleID(name, region), 'Arn'),
                LambdaConfig: {
                    LambdaFunctionArn: lambdaArns_1.lambdaArnResource(name, region),
                },
            }).dependsOn(graphql_transformer_common_1.FunctionResourceIDs.FunctionIAMRoleID(name, region));
        };
        this.function = (name, region) => {
            return new cloudform_types_1.AppSync.FunctionConfiguration({
                ApiId: cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.AppSyncApiId),
                Name: graphql_transformer_common_1.FunctionResourceIDs.FunctionAppSyncFunctionConfigurationID(name, region),
                DataSourceName: graphql_transformer_common_1.FunctionResourceIDs.FunctionDataSourceID(name, region),
                FunctionVersion: '2018-05-29',
                RequestMappingTemplate: graphql_mapping_template_1.printBlock(`Invoke AWS Lambda data source: ${graphql_transformer_common_1.FunctionResourceIDs.FunctionDataSourceID(name, region)}`)(graphql_mapping_template_1.obj({
                    version: graphql_mapping_template_1.str('2018-05-29'),
                    operation: graphql_mapping_template_1.str('Invoke'),
                    payload: graphql_mapping_template_1.obj({
                        typeName: graphql_mapping_template_1.str('$ctx.stash.get("typeName")'),
                        fieldName: graphql_mapping_template_1.str('$ctx.stash.get("fieldName")'),
                        arguments: graphql_mapping_template_1.ref('util.toJson($ctx.arguments)'),
                        identity: graphql_mapping_template_1.ref('util.toJson($ctx.identity)'),
                        source: graphql_mapping_template_1.ref('util.toJson($ctx.source)'),
                        request: graphql_mapping_template_1.ref('util.toJson($ctx.request)'),
                        prev: graphql_mapping_template_1.ref('util.toJson($ctx.prev)'),
                    }),
                })),
                ResponseMappingTemplate: graphql_mapping_template_1.printBlock('Handle error or return result')(graphql_mapping_template_1.compoundExpression([
                    graphql_mapping_template_1.iff(graphql_mapping_template_1.ref('ctx.error'), graphql_mapping_template_1.raw('$util.error($ctx.error.message, $ctx.error.type)')),
                    graphql_mapping_template_1.raw('$util.toJson($ctx.result)'),
                ])),
            }).dependsOn(graphql_transformer_common_1.FunctionResourceIDs.FunctionDataSourceID(name, region));
        };
        this.resolver = (type, field, name, region) => {
            return new cloudform_types_1.AppSync.Resolver({
                ApiId: cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.AppSyncApiId),
                TypeName: type,
                FieldName: field,
                Kind: 'PIPELINE',
                PipelineConfig: {
                    Functions: [cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.FunctionResourceIDs.FunctionAppSyncFunctionConfigurationID(name, region), 'FunctionId')],
                },
                RequestMappingTemplate: graphql_mapping_template_1.printBlock('Stash resolver specific context.')(graphql_mapping_template_1.compoundExpression([graphql_mapping_template_1.qref(`$ctx.stash.put("typeName", "${type}")`), graphql_mapping_template_1.qref(`$ctx.stash.put("fieldName", "${field}")`), graphql_mapping_template_1.obj({})])),
                ResponseMappingTemplate: '$util.toJson($ctx.prev.result)',
            }).dependsOn(graphql_transformer_common_1.FunctionResourceIDs.FunctionAppSyncFunctionConfigurationID(name, region));
        };
    }
    appendFunctionToResolver(resolver, functionId) {
        if (resolver.Properties.PipelineConfig &&
            resolver.Properties.PipelineConfig.Functions &&
            Array.isArray(resolver.Properties.PipelineConfig.Functions)) {
            resolver.Properties.PipelineConfig.Functions.push(cloudform_types_1.Fn.GetAtt(functionId, 'FunctionId'));
        }
        return resolver;
    }
}
exports.FunctionTransformer = FunctionTransformer;
//# sourceMappingURL=FunctionTransformer.js.map