"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ModelConnectionTransformer = void 0;
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphql_1 = require("graphql");
const resources_1 = require("./resources");
const graphql_dynamodb_transformer_1 = require("graphql-dynamodb-transformer");
const graphql_transformer_common_1 = require("graphql-transformer-common");
const graphql_transformer_common_2 = require("graphql-transformer-common");
const definitions_1 = require("./definitions");
const CONNECTION_STACK_NAME = 'ConnectionStack';
function makeConnectionAttributeName(type, field) {
    return field ? graphql_transformer_common_1.toCamelCase([type, field, 'id']) : graphql_transformer_common_1.toCamelCase([type, 'id']);
}
function validateKeyField(field) {
    if (!field) {
        return;
    }
    const baseType = graphql_transformer_common_1.getBaseType(field.type);
    const isAList = graphql_transformer_common_1.isListType(field.type);
    if ((baseType === 'ID' || baseType === 'String') && !isAList) {
        return;
    }
    throw new graphql_transformer_core_1.InvalidDirectiveError(`If you define a field and specify it as a 'keyField', it must be of type 'ID' or 'String'.`);
}
function validateKeyFieldConnectionWithKey(field, ctx) {
    const isAList = graphql_transformer_common_1.isListType(field.type);
    const isAScalarOrEnum = graphql_transformer_common_1.isScalarOrEnum(field.type, ctx.getTypeDefinitionsOfKind(graphql_1.Kind.ENUM_TYPE_DEFINITION));
    if (!isAList && isAScalarOrEnum) {
        return;
    }
    throw new graphql_transformer_core_1.InvalidDirectiveError(`All fields provided to an @connection must be scalar or enum fields.`);
}
function getFieldType(relatedType, fieldName) {
    const foundField = relatedType.fields.find(f => f.name.value === fieldName);
    if (!foundField) {
        throw new graphql_transformer_core_1.InvalidDirectiveError(`${fieldName} is not defined in ${relatedType.name.value}.`);
    }
    return foundField.type;
}
function checkFieldsAgainstIndex(parent, relatedType, inputFieldNames, keySchema, field) {
    const hashAttributeName = keySchema[0].AttributeName;
    const tablePKType = getFieldType(relatedType, String(hashAttributeName));
    const queryPKType = getFieldType(parent, inputFieldNames[0]);
    const numFields = inputFieldNames.length;
    if (graphql_transformer_common_1.getBaseType(tablePKType) !== graphql_transformer_common_1.getBaseType(queryPKType)) {
        throw new graphql_transformer_core_1.InvalidDirectiveError(`${inputFieldNames[0]} field is not of type ${graphql_transformer_common_1.getBaseType(tablePKType)}`);
    }
    if (numFields > keySchema.length && keySchema.length !== 2) {
        throw new graphql_transformer_core_1.InvalidDirectiveError('Too many fields passed in to @connection directive.');
    }
    if (numFields > 1) {
        const querySortFields = inputFieldNames.slice(1);
        const tableSortFields = String(keySchema[1].AttributeName).split(graphql_transformer_common_2.ModelResourceIDs.ModelCompositeKeySeparator());
        if (querySortFields.length !== tableSortFields.length) {
            throw new graphql_transformer_core_1.InvalidDirectiveError(`Invalid @connection directive  ${field.name.value}. fields does not accept partial sort key`);
        }
    }
    if (numFields === 2) {
        const sortAttributeName = String(keySchema[1].AttributeName).split(graphql_transformer_common_2.ModelResourceIDs.ModelCompositeKeySeparator())[0];
        const tableSKType = getFieldType(relatedType, String(sortAttributeName));
        const querySKType = getFieldType(parent, inputFieldNames[1]);
        if (graphql_transformer_common_1.getBaseType(tableSKType) !== graphql_transformer_common_1.getBaseType(querySKType)) {
            throw new graphql_transformer_core_1.InvalidDirectiveError(`${inputFieldNames[1]} field is not of type ${graphql_transformer_common_1.getBaseType(tableSKType)}`);
        }
    }
    else if (numFields > 2) {
        const tableSortFields = String(keySchema[1].AttributeName).split(graphql_transformer_common_2.ModelResourceIDs.ModelCompositeKeySeparator());
        const tableSortKeyTypes = tableSortFields.map(name => getFieldType(relatedType, name));
        const querySortFields = inputFieldNames.slice(1);
        const querySortKeyTypes = querySortFields.map(name => getFieldType(parent, name));
        querySortKeyTypes.forEach((fieldType, index) => {
            if (graphql_transformer_common_1.getBaseType(fieldType) !== graphql_transformer_common_1.getBaseType(tableSortKeyTypes[index])) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`${querySortFields[index]} field is not of type ${graphql_transformer_common_1.getBaseType(tableSortKeyTypes[index])} arguments`);
            }
        });
    }
}
class ModelConnectionTransformer extends graphql_transformer_core_1.Transformer {
    constructor() {
        super('ModelConnectionTransformer', graphql_transformer_core_1.gql `
        directive @connection(
          name: String
          keyField: String
          sortField: String
          keyName: String
          limit: Int
          fields: [String!]
        ) on FIELD_DEFINITION
      `);
        this.before = (ctx) => {
            const template = this.resources.initTemplate();
            ctx.mergeResources(template.Resources);
            ctx.mergeParameters(template.Parameters);
            ctx.mergeOutputs(template.Outputs);
        };
        this.field = (parent, field, directive, ctx) => {
            const parentTypeName = parent.name.value;
            const fieldName = field.name.value;
            ctx.mapResourceToStack(CONNECTION_STACK_NAME, graphql_transformer_common_2.ResolverResourceIDs.ResolverResourceID(parentTypeName, fieldName));
            const parentModelDirective = parent.directives.find((dir) => dir.name.value === 'model');
            if (!parentModelDirective) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`@connection must be on an @model object type field.`);
            }
            const relatedTypeName = graphql_transformer_common_1.getBaseType(field.type);
            const relatedType = ctx.inputDocument.definitions.find(d => d.kind === graphql_1.Kind.OBJECT_TYPE_DEFINITION && d.name.value === relatedTypeName);
            if (!relatedType) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`Could not find an object type named ${relatedTypeName}.`);
            }
            const modelDirective = relatedType.directives.find((dir) => dir.name.value === 'model');
            if (!modelDirective) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`Object type ${relatedTypeName} must be annotated with @model.`);
            }
            if (graphql_transformer_common_1.getDirectiveArgument(directive, 'fields')) {
                this.connectionWithKey(parent, field, directive, ctx);
                return;
            }
            let connectionName = graphql_transformer_common_1.getDirectiveArgument(directive, 'name');
            let associatedSortFieldName = null;
            let sortType = null;
            const associatedConnectionField = relatedType.fields.find((f) => {
                if (f === field) {
                    return false;
                }
                const relatedDirective = f.directives.find((dir) => dir.name.value === 'connection');
                if (relatedDirective) {
                    const relatedDirectiveName = graphql_transformer_common_1.getDirectiveArgument(relatedDirective, 'name');
                    if (connectionName && relatedDirectiveName && relatedDirectiveName === connectionName) {
                        associatedSortFieldName = graphql_transformer_common_1.getDirectiveArgument(relatedDirective, 'sortField');
                        return true;
                    }
                }
                return false;
            });
            if (connectionName && !associatedConnectionField) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`Found one half of connection "${connectionName}" at ${parentTypeName}.${fieldName} but no related field on type ${relatedTypeName}`);
            }
            connectionName = connectionName || `${parentTypeName}.${fieldName}`;
            const leftConnectionIsList = graphql_transformer_common_1.isListType(field.type);
            const leftConnectionIsNonNull = graphql_transformer_common_1.isNonNullType(field.type);
            const rightConnectionIsList = associatedConnectionField ? graphql_transformer_common_1.isListType(associatedConnectionField.type) : undefined;
            const rightConnectionIsNonNull = associatedConnectionField ? graphql_transformer_common_1.isNonNullType(associatedConnectionField.type) : undefined;
            const limit = graphql_transformer_common_1.getDirectiveArgument(directive, 'limit');
            let connectionAttributeName = graphql_transformer_common_1.getDirectiveArgument(directive, 'keyField');
            const associatedSortField = associatedSortFieldName && parent.fields.find((f) => f.name.value === associatedSortFieldName);
            if (associatedSortField) {
                if (graphql_transformer_common_1.isListType(associatedSortField.type)) {
                    throw new graphql_transformer_core_1.InvalidDirectiveError(`sortField "${associatedSortFieldName}" is a list. It should be a scalar.`);
                }
                sortType = graphql_transformer_common_1.getBaseType(associatedSortField.type);
                if (!graphql_transformer_common_1.isScalar(associatedSortField.type) || sortType === graphql_transformer_common_1.STANDARD_SCALARS.Boolean) {
                    throw new graphql_transformer_core_1.InvalidDirectiveError(`sortField "${associatedSortFieldName}" is of type "${sortType}". ` +
                        `It should be a scalar that maps to a DynamoDB "String", "Number", or "Binary"`);
                }
            }
            const foreignAssociatedSortField = associatedSortFieldName && relatedType.fields.find((f) => f.name.value === associatedSortFieldName);
            const sortKeyInfo = foreignAssociatedSortField
                ? {
                    fieldName: foreignAssociatedSortField.name.value,
                    attributeType: graphql_transformer_common_1.attributeTypeFromScalar(foreignAssociatedSortField.type),
                    typeName: graphql_transformer_common_1.getBaseType(foreignAssociatedSortField.type),
                }
                : undefined;
            if (leftConnectionIsList && rightConnectionIsList) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`Invalid Connection (${connectionName}): Many to Many connections are not yet supported.`);
            }
            else if (leftConnectionIsList && rightConnectionIsList === false) {
                const primaryKeyField = this.getPrimaryKeyField(ctx, parent);
                const idFieldName = primaryKeyField ? primaryKeyField.name.value : 'id';
                if (!connectionAttributeName) {
                    connectionAttributeName = makeConnectionAttributeName(relatedTypeName, associatedConnectionField.name.value);
                }
                const existingKeyField = relatedType.fields.find(f => f.name.value === connectionAttributeName);
                validateKeyField(existingKeyField);
                const queryResolver = this.resources.makeQueryConnectionResolver(parentTypeName, fieldName, relatedTypeName, connectionAttributeName, connectionName, idFieldName, sortKeyInfo, limit);
                ctx.setResource(graphql_transformer_common_2.ResolverResourceIDs.ResolverResourceID(parentTypeName, fieldName), queryResolver);
                this.extendTypeWithConnection(ctx, parent, field, relatedType, sortKeyInfo);
            }
            else if (!leftConnectionIsList && rightConnectionIsList) {
                if (associatedSortFieldName && !associatedSortField) {
                    throw new graphql_transformer_core_1.InvalidDirectiveError(`sortField "${associatedSortFieldName}" not found on type "${parent.name.value}", other half of connection "${connectionName}".`);
                }
                const primaryKeyField = this.getPrimaryKeyField(ctx, relatedType);
                const idFieldName = primaryKeyField ? primaryKeyField.name.value : 'id';
                if (!connectionAttributeName) {
                    connectionAttributeName = makeConnectionAttributeName(parentTypeName, fieldName);
                }
                const existingKeyField = parent.fields.find(f => f.name.value === connectionAttributeName);
                validateKeyField(existingKeyField);
                const tableLogicalId = graphql_transformer_common_2.ModelResourceIDs.ModelTableResourceID(parentTypeName);
                const table = ctx.getResource(tableLogicalId);
                const sortField = associatedSortField ? { name: associatedSortFieldName, type: sortType } : null;
                const updated = this.resources.updateTableForConnection(table, connectionName, connectionAttributeName, sortField);
                ctx.setResource(tableLogicalId, updated);
                const getResolver = this.resources.makeGetItemConnectionResolver(parentTypeName, fieldName, relatedTypeName, connectionAttributeName, idFieldName);
                ctx.setResource(graphql_transformer_common_2.ResolverResourceIDs.ResolverResourceID(parentTypeName, fieldName), getResolver);
                const createInputName = graphql_transformer_common_2.ModelResourceIDs.ModelCreateInputObjectName(parentTypeName);
                const createInput = ctx.getType(createInputName);
                if (createInput) {
                    const updated = definitions_1.updateCreateInputWithConnectionField(createInput, connectionAttributeName, leftConnectionIsNonNull);
                    ctx.putType(updated);
                }
                const updateInputName = graphql_transformer_common_2.ModelResourceIDs.ModelUpdateInputObjectName(parentTypeName);
                const updateInput = ctx.getType(updateInputName);
                if (updateInput) {
                    const updated = definitions_1.updateUpdateInputWithConnectionField(updateInput, connectionAttributeName);
                    ctx.putType(updated);
                }
            }
            else if (leftConnectionIsList) {
                const primaryKeyField = this.getPrimaryKeyField(ctx, parent);
                const idFieldName = primaryKeyField ? primaryKeyField.name.value : 'id';
                if (!connectionAttributeName) {
                    connectionAttributeName = makeConnectionAttributeName(parentTypeName, fieldName);
                }
                const existingKeyField = relatedType.fields.find(f => f.name.value === connectionAttributeName);
                validateKeyField(existingKeyField);
                const tableLogicalId = graphql_transformer_common_2.ModelResourceIDs.ModelTableResourceID(relatedTypeName);
                const table = ctx.getResource(tableLogicalId);
                const updated = this.resources.updateTableForConnection(table, connectionName, connectionAttributeName);
                ctx.setResource(tableLogicalId, updated);
                const queryResolver = this.resources.makeQueryConnectionResolver(parentTypeName, fieldName, relatedTypeName, connectionAttributeName, connectionName, idFieldName, sortKeyInfo, limit);
                ctx.setResource(graphql_transformer_common_2.ResolverResourceIDs.ResolverResourceID(parentTypeName, fieldName), queryResolver);
                this.extendTypeWithConnection(ctx, parent, field, relatedType, sortKeyInfo);
                const createInputName = graphql_transformer_common_2.ModelResourceIDs.ModelCreateInputObjectName(relatedTypeName);
                const createInput = ctx.getType(createInputName);
                if (createInput) {
                    const updated = definitions_1.updateCreateInputWithConnectionField(createInput, connectionAttributeName);
                    ctx.putType(updated);
                }
                const updateInputName = graphql_transformer_common_2.ModelResourceIDs.ModelUpdateInputObjectName(relatedTypeName);
                const updateInput = ctx.getType(updateInputName);
                if (updateInput) {
                    const updated = definitions_1.updateUpdateInputWithConnectionField(updateInput, connectionAttributeName);
                    ctx.putType(updated);
                }
            }
            else {
                const primaryKeyField = this.getPrimaryKeyField(ctx, relatedType);
                const idFieldName = primaryKeyField ? primaryKeyField.name.value : 'id';
                if (!connectionAttributeName) {
                    connectionAttributeName = makeConnectionAttributeName(parentTypeName, fieldName);
                }
                let sortFieldInfo;
                const sortFieldName = graphql_transformer_common_1.getDirectiveArgument(directive, 'sortField');
                if (sortFieldName) {
                    const relatedSortField = this.getSortField(relatedType);
                    if (!relatedSortField) {
                        throw new graphql_transformer_core_1.InvalidDirectiveError(`sortField "${sortFieldName}" requires a primary @key on type "${relatedTypeName}" with a sort key that was not found.`);
                    }
                    const sortField = parent.fields.find(f => f.name.value === sortFieldName);
                    if (!sortField) {
                        throw new graphql_transformer_core_1.InvalidDirectiveError(`sortField with name "${sortFieldName} cannot be found on type: ${parent.name.value}`);
                    }
                    const relatedSortFieldType = graphql_transformer_common_1.getBaseType(relatedSortField.type);
                    const sortFieldType = graphql_transformer_common_1.getBaseType(sortField.type);
                    if (relatedSortFieldType !== sortFieldType) {
                        throw new graphql_transformer_core_1.InvalidDirectiveError(`sortField "${relatedSortField.name.value}" on type "${relatedTypeName}" is not matching the ` +
                            `type of field "${sortFieldName}" on type "${parentTypeName}"`);
                    }
                    let sortFieldIsStringLike = true;
                    if (sortFieldType === graphql_transformer_common_1.STANDARD_SCALARS.Int ||
                        sortFieldType === graphql_transformer_common_1.STANDARD_SCALARS.Float ||
                        sortFieldType === graphql_transformer_common_1.STANDARD_SCALARS.Bolean) {
                        sortFieldIsStringLike = false;
                    }
                    sortFieldInfo = {
                        primarySortFieldName: relatedSortField.name.value,
                        sortFieldName,
                        sortFieldIsStringLike,
                    };
                }
                const existingKeyField = parent.fields.find(f => f.name.value === connectionAttributeName);
                validateKeyField(existingKeyField);
                const getResolver = this.resources.makeGetItemConnectionResolver(parentTypeName, fieldName, relatedTypeName, connectionAttributeName, idFieldName, sortFieldInfo);
                ctx.setResource(graphql_transformer_common_2.ResolverResourceIDs.ResolverResourceID(parentTypeName, fieldName), getResolver);
                const createInputName = graphql_transformer_common_2.ModelResourceIDs.ModelCreateInputObjectName(parentTypeName);
                const createInput = ctx.getType(createInputName);
                if (createInput) {
                    const updated = definitions_1.updateCreateInputWithConnectionField(createInput, connectionAttributeName, leftConnectionIsNonNull);
                    ctx.putType(updated);
                }
                const updateInputName = graphql_transformer_common_2.ModelResourceIDs.ModelUpdateInputObjectName(parentTypeName);
                const updateInput = ctx.getType(updateInputName);
                if (updateInput) {
                    const updated = definitions_1.updateUpdateInputWithConnectionField(updateInput, connectionAttributeName);
                    ctx.putType(updated);
                }
            }
        };
        this.connectionWithKey = (parent, field, directive, ctx) => {
            const parentTypeName = parent.name.value;
            const fieldName = field.name.value;
            const args = graphql_transformer_core_1.getDirectiveArguments(directive);
            if (args.fields.length === 0) {
                throw new graphql_transformer_core_1.InvalidDirectiveError('No fields passed in to @connection directive.');
            }
            const relatedTypeName = graphql_transformer_common_1.getBaseType(field.type);
            const relatedType = ctx.inputDocument.definitions.find(d => d.kind === graphql_1.Kind.OBJECT_TYPE_DEFINITION && d.name.value === relatedTypeName);
            const tableLogicalID = graphql_transformer_common_2.ModelResourceIDs.ModelTableResourceID(relatedType.name.value);
            const tableResource = ctx.getResource(tableLogicalID);
            let inputFields = [];
            args.fields.forEach(item => {
                const fieldsArrayLength = inputFields.length;
                inputFields.push(parent.fields.find(f => f.name.value === item));
                if (!inputFields[fieldsArrayLength]) {
                    throw new graphql_transformer_core_1.InvalidDirectiveError(`${item} is not a field in ${parentTypeName}`);
                }
                validateKeyFieldConnectionWithKey(inputFields[fieldsArrayLength], ctx);
            });
            let index = undefined;
            if (!args.keyName) {
                checkFieldsAgainstIndex(parent, relatedType, args.fields, tableResource.Properties.KeySchema, field);
            }
            else {
                index =
                    (tableResource.Properties.GlobalSecondaryIndexes
                        ? tableResource.Properties.GlobalSecondaryIndexes.find(GSI => GSI.IndexName === args.keyName)
                        : null) ||
                        (tableResource.Properties.LocalSecondaryIndexes
                            ? tableResource.Properties.LocalSecondaryIndexes.find(LSI => LSI.IndexName === args.keyName)
                            : null);
                if (!index) {
                    throw new graphql_transformer_core_1.InvalidDirectiveError(`Key ${args.keyName} does not exist for model ${relatedTypeName}`);
                }
                checkFieldsAgainstIndex(parent, relatedType, args.fields, index.KeySchema, field);
            }
            if (!graphql_transformer_common_1.isListType(field.type)) {
                if (args.keyName) {
                    throw new graphql_transformer_core_1.InvalidDirectiveError(`Connection is to a single object but the keyName ${args.keyName} was provided which does not reference the default table.`);
                }
                const getResolver = this.resources.makeGetItemConnectionWithKeyResolver(parentTypeName, fieldName, relatedTypeName, args.fields, tableResource.Properties.KeySchema);
                ctx.setResource(graphql_transformer_common_2.ResolverResourceIDs.ResolverResourceID(parentTypeName, fieldName), getResolver);
            }
            else {
                const keySchema = index ? index.KeySchema : tableResource.Properties.KeySchema;
                const queryResolver = this.resources.makeQueryConnectionWithKeyResolver(parentTypeName, fieldName, relatedType, args.fields, keySchema, index ? String(index.IndexName) : undefined, args.limit);
                ctx.setResource(graphql_transformer_common_2.ResolverResourceIDs.ResolverResourceID(parentTypeName, fieldName), queryResolver);
                let sortKeyInfo = undefined;
                if (args.fields.length > 1) {
                    sortKeyInfo = undefined;
                }
                else {
                    const compositeSortKeyType = 'Composite';
                    const compositeSortKeyName = keySchema[1] ? this.resources.makeCompositeSortKeyName(String(keySchema[1].AttributeName)) : undefined;
                    const sortKeyField = keySchema[1] ? relatedType.fields.find(f => f.name.value === keySchema[1].AttributeName) : undefined;
                    if (sortKeyField) {
                        sortKeyInfo = keySchema[1]
                            ? {
                                fieldName: String(keySchema[1].AttributeName),
                                typeName: graphql_transformer_common_1.getBaseType(sortKeyField.type),
                                model: relatedTypeName,
                                keyName: index ? String(index.IndexName) : 'Primary',
                            }
                            : undefined;
                    }
                    else {
                        sortKeyInfo = keySchema[1]
                            ? {
                                fieldName: compositeSortKeyName,
                                typeName: compositeSortKeyType,
                                model: relatedTypeName,
                                keyName: index ? String(index.IndexName) : 'Primary',
                            }
                            : undefined;
                    }
                }
                this.extendTypeWithConnection(ctx, parent, field, relatedType, sortKeyInfo);
            }
        };
        this.resources = new resources_1.ResourceFactory();
    }
    typeExist(type, ctx) {
        return Boolean(type in ctx.nodeMap);
    }
    generateModelXConnectionType(ctx, typeDef) {
        const tableXConnectionName = graphql_transformer_common_2.ModelResourceIDs.ModelConnectionTypeName(typeDef.name.value);
        if (this.typeExist(tableXConnectionName, ctx)) {
            return;
        }
        const connectionType = graphql_transformer_common_1.blankObject(tableXConnectionName);
        ctx.addObject(connectionType);
        ctx.addObjectExtension(graphql_dynamodb_transformer_1.makeModelConnectionType(typeDef.name.value));
    }
    generateFilterAndKeyConditionInputs(ctx, field, sortKeyInfo) {
        const scalarFilters = graphql_dynamodb_transformer_1.makeScalarFilterInputs(this.supportsConditions(ctx));
        for (const filter of scalarFilters) {
            if (!this.typeExist(filter.name.value, ctx)) {
                ctx.addInput(filter);
            }
        }
        const enumFilters = graphql_dynamodb_transformer_1.makeEnumFilterInputObjects(field, ctx, this.supportsConditions(ctx));
        for (const filter of enumFilters) {
            if (!this.typeExist(filter.name.value, ctx)) {
                ctx.addInput(filter);
            }
        }
        const tableXQueryFilterInput = graphql_dynamodb_transformer_1.makeModelXFilterInputObject(field, ctx, this.supportsConditions(ctx));
        if (!this.typeExist(tableXQueryFilterInput.name.value, ctx)) {
            ctx.addInput(tableXQueryFilterInput);
        }
        if (this.supportsConditions(ctx)) {
            const attributeTypeEnum = graphql_dynamodb_transformer_1.makeAttributeTypeEnum();
            if (!this.typeExist(attributeTypeEnum.name.value, ctx)) {
                ctx.addType(attributeTypeEnum);
            }
        }
        if (sortKeyInfo && sortKeyInfo.typeName !== 'Composite') {
            const sortKeyConditionInput = graphql_transformer_common_1.makeScalarKeyConditionForType(graphql_transformer_common_1.makeNamedType(sortKeyInfo.typeName));
            if (!this.typeExist(sortKeyConditionInput.name.value, ctx)) {
                ctx.addInput(sortKeyConditionInput);
            }
        }
    }
    supportsConditions(context) {
        return context.getTransformerVersion() >= graphql_dynamodb_transformer_1.CONDITIONS_MINIMUM_VERSION;
    }
    extendTypeWithConnection(ctx, parent, field, returnType, sortKeyInfo) {
        this.generateModelXConnectionType(ctx, returnType);
        const type = ctx.getType(parent.name.value);
        if (type && (type.kind === graphql_1.Kind.OBJECT_TYPE_DEFINITION || type.kind === graphql_1.Kind.INTERFACE_TYPE_DEFINITION)) {
            const newFields = type.fields.map((f) => {
                if (f.name.value === field.name.value) {
                    const updated = graphql_dynamodb_transformer_1.makeModelConnectionField(field.name.value, returnType.name.value, sortKeyInfo, [...f.directives]);
                    return updated;
                }
                return f;
            });
            const updatedType = {
                ...type,
                fields: newFields,
            };
            ctx.putType(updatedType);
            if (!this.typeExist('ModelSortDirection', ctx)) {
                const modelSortDirection = graphql_dynamodb_transformer_1.makeModelSortDirectionEnumObject();
                ctx.addEnum(modelSortDirection);
            }
            this.generateFilterAndKeyConditionInputs(ctx, returnType, sortKeyInfo);
        }
        else {
            throw new graphql_transformer_core_1.InvalidDirectiveError(`Could not find a object or interface type named ${parent.name.value}.`);
        }
    }
    getPrimaryKeyField(ctx, type) {
        let field;
        for (const keyDirective of type.directives.filter(d => d.name.value === 'key')) {
            if (graphql_transformer_common_1.getDirectiveArgument(keyDirective, 'name') === undefined) {
                const fieldsArg = graphql_transformer_common_1.getDirectiveArgument(keyDirective, 'fields');
                if (fieldsArg && fieldsArg.length && fieldsArg.length >= 1 && fieldsArg.length <= 2) {
                    field = type.fields.find(f => f.name.value === fieldsArg[0]);
                }
                break;
            }
        }
        return field;
    }
    getSortField(type) {
        let field;
        for (const keyDirective of type.directives.filter(d => d.name.value === 'key')) {
            if (graphql_transformer_common_1.getDirectiveArgument(keyDirective, 'name') === undefined) {
                const fieldsArg = graphql_transformer_common_1.getDirectiveArgument(keyDirective, 'fields');
                if (fieldsArg && fieldsArg.length && fieldsArg.length === 2) {
                    field = type.fields.find(f => f.name.value === fieldsArg[1]);
                }
                break;
            }
        }
        return field;
    }
}
exports.ModelConnectionTransformer = ModelConnectionTransformer;
//# sourceMappingURL=ModelConnectionTransformer.js.map