"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomVTLTransformer = void 0;
const fs = require("fs");
const path = require("path");
const cloudform_types_1 = require("cloudform-types");
const resolver_1 = require("cloudform-types/types/appSync/resolver");
const graphql_1 = require("graphql");
const graphql_transformer_common_1 = require("graphql-transformer-common");
const graphql_transformer_core_1 = require("graphql-transformer-core");
const CUSTOM_DIRECTIVE_STACK_NAME = 'CustomDirectiveStack';
/**
 * Create a get item resolver for singular connections.
 * @param type The parent type name.
 * @param field The connection field name.
 */
/*eslint-disable @typescript-eslint/no-explicit-any */
function makeResolver(type, field, request, response, datasourceName = 'NONE') {
    return new resolver_1.default({
        ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
        DataSourceName: datasourceName,
        FieldName: field,
        TypeName: type,
        RequestMappingTemplate: request,
        ResponseMappingTemplate: response,
    }).dependsOn(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLSchemaLogicalID);
}
function noneDataSource() {
    return new cloudform_types_1.AppSync.DataSource({
        ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
        Name: 'NONE',
        Type: 'NONE',
    });
}
class CustomVTLTransformer extends graphql_transformer_core_1.Transformer {
    constructor(rootDirectory) {
        super('CustomVTLTransformer', graphql_transformer_core_1.gql `
          directive @custom(request: String, response: String) on FIELD_DEFINITION
        `);
        this.before = (acc) => {
            const directiveList = [];
            // gather all the http directives
            for (const def of acc.inputDocument.definitions) {
                if (def.kind === graphql_1.Kind.OBJECT_TYPE_DEFINITION && def.fields) {
                    for (const field of def.fields) {
                        if (field.directives) {
                            const customDirective = field.directives.find((dir) => dir.name.value === 'custom');
                            if (customDirective) {
                                directiveList.push(customDirective);
                            }
                        }
                    }
                }
            }
        };
        /*eslint-disable @typescript-eslint/no-explicit-any */
        this.field = (parent, field, directive, acc) => {
            const parentTypeName = parent.name.value;
            const fieldName = field.name.value;
            // add none ds if that does not exist
            const noneDS = acc.getResource(graphql_transformer_common_1.ResourceConstants.RESOURCES.NoneDataSource);
            if (!noneDS) {
                acc.setResource(graphql_transformer_common_1.ResourceConstants.RESOURCES.NoneDataSource, noneDataSource());
            }
            acc.mapResourceToStack(CUSTOM_DIRECTIVE_STACK_NAME, graphql_transformer_common_1.ResolverResourceIDs.ResolverResourceID(parentTypeName, fieldName));
            const requestFile = graphql_transformer_common_1.getDirectiveArgument(directive, 'request');
            const responseFile = graphql_transformer_common_1.getDirectiveArgument(directive, 'response');
            let datasourceName = 'NONE';
            if (!requestFile) {
                throw new Error(`The @custom directive on Type: ${parent.name.value} Field: ${field.name.value} is missing the request argument.`);
            }
            if (!responseFile) {
                throw new Error(`The @custom directive on Type: ${parent.name.value} Field: ${field.name.value} is missing the response argument.`);
            }
            let request, response;
            try {
                request = fs.readFileSync(path.join(this.rootDirectory, requestFile)).toString();
                response = fs.readFileSync(path.join(this.rootDirectory, responseFile)).toString();
            }
            catch (err) {
                throw new Error(`Couldn't load VTL files. ${err.message}`);
            }
            const fieldMappingResolver = makeResolver(parentTypeName, fieldName, request, response, datasourceName);
            acc.setResource(graphql_transformer_common_1.ResolverResourceIDs.ResolverResourceID(parentTypeName, fieldName), fieldMappingResolver);
            const templateResources = acc.template.Resources;
            if (!templateResources)
                return;
        };
        this.rootDirectory = rootDirectory;
    }
}
exports.CustomVTLTransformer = CustomVTLTransformer;
//# sourceMappingURL=data:application/json;base64,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