"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppSyncTransformer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_appsync_1 = require("@aws-cdk/aws-appsync");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const schema_transformer_1 = require("./transformer/schema-transformer");
const defaultAuthorizationConfig = {
    defaultAuthorization: {
        authorizationType: aws_appsync_1.AuthorizationType.API_KEY,
        apiKeyConfig: {
            description: 'Auto generated API Key from construct',
            name: 'dev',
        },
    },
};
/**
 * (experimental) AppSyncTransformer Construct.
 *
 * @experimental
 */
class AppSyncTransformer extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        this.props = props;
        this.tableMap = {};
        this.isSyncEnabled = props.syncEnabled ? props.syncEnabled : false;
        this.pointInTimeRecovery = (_b = props.enableDynamoPointInTimeRecovery) !== null && _b !== void 0 ? _b : false;
        const transformerConfiguration = {
            schemaPath: props.schemaPath,
            syncEnabled: (_c = props.syncEnabled) !== null && _c !== void 0 ? _c : false,
            customVtlTransformerRootDirectory: props.customVtlTransformerRootDirectory,
        };
        // Combine the arrays so we only loop once
        // Test each transformer to see if it implements ITransformer
        const allCustomTransformers = [...(_d = props.preCdkTransformers) !== null && _d !== void 0 ? _d : [], ...(_e = props.postCdkTransformers) !== null && _e !== void 0 ? _e : []];
        if (allCustomTransformers && allCustomTransformers.length > 0) {
            allCustomTransformers.forEach(transformer => {
                if (transformer && !this.implementsITransformer(transformer)) {
                    throw new Error(`Transformer does not implement ITransformer from graphql-transformer-core: ${transformer}`);
                }
            });
        }
        const transformer = new schema_transformer_1.SchemaTransformer(transformerConfiguration);
        this.outputs = transformer.transform(props.preCdkTransformers, props.postCdkTransformers);
        const resolvers = transformer.getResolvers();
        this.functionResolvers = (_f = this.outputs.functionResolvers) !== null && _f !== void 0 ? _f : {};
        // Remove any function resolvers from the total list of resolvers
        // Otherwise it will add them twice
        for (const [_, functionResolvers] of Object.entries(this.functionResolvers)) {
            functionResolvers.forEach((resolver) => {
                switch (resolver.typeName) {
                    case 'Query':
                    case 'Mutation':
                    case 'Subscription':
                        delete resolvers[resolver.fieldName];
                        break;
                }
            });
        }
        this.httpResolvers = (_g = this.outputs.httpResolvers) !== null && _g !== void 0 ? _g : {};
        // Remove any http resolvers from the total list of resolvers
        // Otherwise it will add them twice
        for (const [_, httpResolvers] of Object.entries(this.httpResolvers)) {
            httpResolvers.forEach((resolver) => {
                switch (resolver.typeName) {
                    case 'Query':
                    case 'Mutation':
                    case 'Subscription':
                        delete resolvers[resolver.fieldName];
                        break;
                }
            });
        }
        this.resolvers = resolvers;
        this.nestedAppsyncStack = new core_1.NestedStack(this, (_h = props.nestedStackName) !== null && _h !== void 0 ? _h : 'appsync-nested-stack');
        // AppSync
        this.appsyncAPI = new aws_appsync_1.GraphqlApi(this.nestedAppsyncStack, `${id}-api`, {
            name: props.apiName ? props.apiName : `${id}-api`,
            authorizationConfig: props.authorizationConfig
                ? props.authorizationConfig
                : defaultAuthorizationConfig,
            logConfig: {
                fieldLogLevel: props.fieldLogLevel
                    ? props.fieldLogLevel
                    : aws_appsync_1.FieldLogLevel.NONE,
            },
            schema: aws_appsync_1.Schema.fromAsset('./appsync/schema.graphql'),
            xrayEnabled: (_j = props.xrayEnabled) !== null && _j !== void 0 ? _j : false,
        });
        let tableData = (_k = this.outputs.cdkTables) !== null && _k !== void 0 ? _k : {};
        // Check to see if sync is enabled
        if (tableData.DataStore) {
            this.isSyncEnabled = true;
            this.syncTable = this.createSyncTable(tableData.DataStore);
            delete tableData.DataStore; // We don't want to create this again below so remove it from the tableData map
        }
        this.tableNameMap = this.createTablesAndResolvers(tableData, resolvers, props.tableNames);
        if (this.outputs.noneResolvers) {
            this.createNoneDataSourceAndResolvers(this.outputs.noneResolvers, resolvers);
        }
        this.createHttpResolvers();
        this.publicResourceArns = this.getResourcesFromGeneratedRolePolicy(transformer.unauthRolePolicy);
        this.privateResourceArns = this.getResourcesFromGeneratedRolePolicy(transformer.authRolePolicy);
        // Outputs so we can generate exports
        new core_1.CfnOutput(scope, 'appsyncGraphQLEndpointOutput', {
            value: this.appsyncAPI.graphqlUrl,
            description: 'Output for aws_appsync_graphqlEndpoint',
        });
    }
    /**
     * graphql-transformer-core needs to be jsii enabled to pull the ITransformer interface correctly.
     * Since it's not in peer dependencies it doesn't show up in the jsii deps list.
     * Since it's not jsii enabled it has to be bundled.
     * The package can't be in BOTH peer and bundled dependencies
     * So we do a fake test to make sure it implements these and hope for the best
     * @param transformer
     */
    implementsITransformer(transformer) {
        return 'name' in transformer
            && 'directive' in transformer
            && 'typeDefinitions' in transformer;
    }
    /**
     * Creates NONE data source and associated resolvers
     * @param noneResolvers The resolvers that belong to the none data source
     * @param resolvers The resolver map minus function resolvers
     */
    createNoneDataSourceAndResolvers(noneResolvers, resolvers) {
        const noneDataSource = this.appsyncAPI.addNoneDataSource('NONE');
        Object.keys(noneResolvers).forEach((resolverKey) => {
            const resolver = resolvers[resolverKey];
            new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                api: this.appsyncAPI,
                typeName: resolver.typeName,
                fieldName: resolver.fieldName,
                dataSource: noneDataSource,
                requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
            });
        });
    }
    /**
     * Creates each dynamodb table, gsis, dynamodb datasource, and associated resolvers
     * If sync is enabled then TTL configuration is added
     * Returns tableName: table map in case it is needed for lambda functions, etc
     * @param tableData The CdkTransformer table information
     * @param resolvers The resolver map minus function resolvers
     */
    createTablesAndResolvers(tableData, resolvers, tableNames = {}) {
        const tableNameMap = {};
        Object.keys(tableData).forEach((tableKey) => {
            var _b;
            const tableName = (_b = tableNames[tableKey]) !== null && _b !== void 0 ? _b : undefined;
            const table = this.createTable(tableData[tableKey], tableName);
            this.tableMap[tableKey] = table;
            const dataSource = this.appsyncAPI.addDynamoDbDataSource(tableKey, table);
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appsync-datasource-deltasyncconfig.html
            if (this.isSyncEnabled && this.syncTable) {
                //@ts-ignore - ds is the base CfnDataSource and the db config needs to be versioned - see CfnDataSource
                dataSource.ds.dynamoDbConfig.versioned = true;
                //@ts-ignore - ds is the base CfnDataSource - see CfnDataSource
                dataSource.ds.dynamoDbConfig.deltaSyncConfig = {
                    baseTableTtl: '43200',
                    deltaSyncTableName: this.syncTable.tableName,
                    deltaSyncTableTtl: '30',
                };
                // Need to add permission for our datasource service role to access the sync table
                dataSource.grantPrincipal.addToPolicy(new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: [
                        'dynamodb:*',
                    ],
                    resources: [this.syncTable.tableArn],
                }));
            }
            const dynamoDbConfig = dataSource.ds
                .dynamoDbConfig;
            tableNameMap[tableKey] = dynamoDbConfig.tableName;
            // Loop the basic resolvers
            tableData[tableKey].resolvers.forEach((resolverKey) => {
                let resolver = resolvers[resolverKey];
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
            // Loop the gsi resolvers
            tableData[tableKey].gsiResolvers.forEach((resolverKey) => {
                let resolver = resolvers.gsi[resolverKey];
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
        });
        return tableNameMap;
    }
    createTable(tableData, tableName) {
        var _b;
        // I do not want to force people to pass `TypeTable` - this way they are only passing the @model Type name
        const modelTypeName = tableData.tableName.replace('Table', '');
        const streamSpecification = this.props.dynamoDbStreamConfig && this.props.dynamoDbStreamConfig[modelTypeName];
        const tableProps = {
            tableName,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.partitionKey.name,
                type: this.convertAttributeType(tableData.partitionKey.type),
            },
            pointInTimeRecovery: this.pointInTimeRecovery,
            sortKey: tableData.sortKey && tableData.sortKey.name
                ? {
                    name: tableData.sortKey.name,
                    type: this.convertAttributeType(tableData.sortKey.type),
                } : undefined,
            timeToLiveAttribute: ((_b = tableData === null || tableData === void 0 ? void 0 : tableData.ttl) === null || _b === void 0 ? void 0 : _b.enabled) ? tableData.ttl.attributeName : undefined,
            stream: streamSpecification,
        };
        const table = new aws_dynamodb_1.Table(this.nestedAppsyncStack, tableData.tableName, tableProps);
        tableData.localSecondaryIndexes.forEach((lsi) => {
            table.addLocalSecondaryIndex({
                indexName: lsi.indexName,
                sortKey: {
                    name: lsi.sortKey.name,
                    type: this.convertAttributeType(lsi.sortKey.type),
                },
                projectionType: this.convertProjectionType(lsi.projection.ProjectionType),
            });
        });
        tableData.globalSecondaryIndexes.forEach((gsi) => {
            table.addGlobalSecondaryIndex({
                indexName: gsi.indexName,
                partitionKey: {
                    name: gsi.partitionKey.name,
                    type: this.convertAttributeType(gsi.partitionKey.type),
                },
                sortKey: gsi.sortKey && gsi.sortKey.name
                    ? {
                        name: gsi.sortKey.name,
                        type: this.convertAttributeType(gsi.sortKey.type),
                    } : undefined,
                projectionType: this.convertProjectionType(gsi.projection.ProjectionType),
            });
        });
        return table;
    }
    /**
     * Creates the sync table for Amplify DataStore
     * https://docs.aws.amazon.com/appsync/latest/devguide/conflict-detection-and-sync.html
     * @param tableData The CdkTransformer table information
     */
    createSyncTable(tableData) {
        var _b;
        return new aws_dynamodb_1.Table(this, 'appsync-api-sync-table', {
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.partitionKey.name,
                type: this.convertAttributeType(tableData.partitionKey.type),
            },
            sortKey: {
                name: tableData.sortKey.name,
                type: this.convertAttributeType(tableData.sortKey.type),
            },
            timeToLiveAttribute: ((_b = tableData.ttl) === null || _b === void 0 ? void 0 : _b.attributeName) || '_ttl',
        });
    }
    convertAttributeType(type) {
        switch (type) {
            case 'N':
                return aws_dynamodb_1.AttributeType.NUMBER;
            case 'B':
                return aws_dynamodb_1.AttributeType.BINARY;
            case 'S': // Same as default
            default:
                return aws_dynamodb_1.AttributeType.STRING;
        }
    }
    convertProjectionType(type) {
        switch (type) {
            case 'INCLUDE':
                return aws_dynamodb_1.ProjectionType.INCLUDE;
            case 'KEYS_ONLY':
                return aws_dynamodb_1.ProjectionType.KEYS_ONLY;
            case 'ALL': // Same as default
            default:
                return aws_dynamodb_1.ProjectionType.ALL;
        }
    }
    createHttpResolvers() {
        for (const [endpoint, httpResolvers] of Object.entries(this.httpResolvers)) {
            const strippedEndpoint = endpoint.replace(/[^_0-9A-Za-z]/g, '');
            const httpDataSource = this.appsyncAPI.addHttpDataSource(`${strippedEndpoint}`, endpoint);
            httpResolvers.forEach((resolver) => {
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: httpDataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultRequestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultResponseMappingTemplate),
                });
            });
        }
    }
    /**
     * This takes one of the autogenerated policies from AWS and builds the list of ARNs for granting GraphQL access later
     * @param policy The auto generated policy from the AppSync Transformers
     * @returns An array of resource arns for use with grants
     */
    getResourcesFromGeneratedRolePolicy(policy) {
        var _b, _c;
        if (!((_c = (_b = policy === null || policy === void 0 ? void 0 : policy.Properties) === null || _b === void 0 ? void 0 : _b.PolicyDocument) === null || _c === void 0 ? void 0 : _c.Statement))
            return [];
        const { region, account } = this.nestedAppsyncStack;
        const resolvedResources = [];
        for (const statement of policy.Properties.PolicyDocument.Statement) {
            const { Resource: resources = [] } = statement !== null && statement !== void 0 ? statement : {};
            for (const resource of resources) {
                const subs = resource['Fn::Sub'][1];
                const { typeName, fieldName } = subs !== null && subs !== void 0 ? subs : {};
                if (fieldName) {
                    resolvedResources.push(`arn:aws:appsync:${region}:${account}:apis/${this.appsyncAPI.apiId}/types/${typeName}/fields/${fieldName}`);
                }
                else {
                    resolvedResources.push(`arn:aws:appsync:${region}:${account}:apis/${this.appsyncAPI.apiId}/types/${typeName}/*`);
                }
            }
        }
        return resolvedResources;
    }
    /**
     * (experimental) Adds the function as a lambdaDataSource to the AppSync api Adds all of the functions resolvers to the AppSync api.
     *
     * @param functionName The function name specified in the.
     * @param id The id to give.
     * @param lambdaFunction The lambda function to attach.
     * @experimental
     * @function directive of the schema
     */
    addLambdaDataSourceAndResolvers(functionName, id, lambdaFunction, options) {
        const functionDataSource = this.appsyncAPI.addLambdaDataSource(id, lambdaFunction, options);
        for (const resolver of this.functionResolvers[functionName]) {
            new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                api: this.appsyncAPI,
                typeName: resolver.typeName,
                fieldName: resolver.fieldName,
                dataSource: functionDataSource,
                requestMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultRequestMappingTemplate),
                responseMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultResponseMappingTemplate),
            });
        }
        return functionDataSource;
    }
    /**
     * (experimental) Adds a stream to the dynamodb table associated with the type.
     *
     * @returns string - the stream arn token
     * @experimental
     */
    addDynamoDBStream(props) {
        const tableName = `${props.modelTypeName}Table`;
        const table = this.tableMap[tableName];
        if (!table)
            throw new Error(`Table with name '${tableName}' not found.`);
        const cfnTable = table.node.defaultChild;
        cfnTable.streamSpecification = {
            streamViewType: props.streamViewType,
        };
        return cfnTable.attrStreamArn;
    }
    /**
     * (experimental) Adds an IAM policy statement granting access to the public fields of the AppSync API.
     *
     * Policy is based off of the @auth transformer
     * https://docs.amplify.aws/cli/graphql-transformer/auth
     *
     * @param grantee The principal to grant access to.
     * @experimental
     */
    grantPublic(grantee) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['appsync:GraphQL'],
            resourceArns: this.publicResourceArns,
            scope: this,
        });
    }
    /**
     * (experimental) Adds an IAM policy statement granting access to the private fields of the AppSync API.
     *
     * Policy is based off of the @auth transformer
     * https://docs.amplify.aws/cli/graphql-transformer/auth
     *
     * @experimental
     */
    grantPrivate(grantee) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['appsync:GraphQL'],
            resourceArns: this.privateResourceArns,
        });
    }
}
exports.AppSyncTransformer = AppSyncTransformer;
_a = JSII_RTTI_SYMBOL_1;
AppSyncTransformer[_a] = { fqn: "cdk-appsync-transformer.AppSyncTransformer", version: "1.123.2" };
//# sourceMappingURL=data:application/json;base64,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