"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToDynamoDB = void 0;
const dynamodb = require("@aws-cdk/aws-dynamodb");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
class LambdaToDynamoDB extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the LambdaToDynamoDB class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        this.dynamoTable = defaults.buildDynamoDBTable(this, {
            dynamoTableProps: props.dynamoTableProps,
            existingTableObj: props.existingTableObj
        });
        this.lambdaFunction.addEnvironment('DDB_TABLE_NAME', this.dynamoTable.tableName);
        // Add the requested or default table permissions
        if (props.tablePermissions) {
            const _permissions = props.tablePermissions.toUpperCase();
            if (_permissions === 'ALL') {
                this.dynamoTable.grantFullAccess(this.lambdaFunction.grantPrincipal);
            }
            else if (_permissions === 'READ') {
                this.dynamoTable.grantReadData(this.lambdaFunction.grantPrincipal);
            }
            else if (_permissions === 'READWRITE') {
                this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
            }
            else if (_permissions === 'WRITE') {
                this.dynamoTable.grantWriteData(this.lambdaFunction.grantPrincipal);
            }
        }
        else {
            this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
        }
        // Conditional metadata for cfn_nag
        if (((_a = props.dynamoTableProps) === null || _a === void 0 ? void 0 : _a.billingMode) === dynamodb.BillingMode.PROVISIONED) {
            const cfnTable = this.dynamoTable.node.findChild('Resource');
            cfnTable.cfnOptions.metadata = {
                cfn_nag: {
                    rules_to_suppress: [{
                            id: 'W73',
                            reason: `PROVISIONED billing mode is a default and is not explicitly applied as a setting.`
                        }]
                }
            };
        }
    }
}
exports.LambdaToDynamoDB = LambdaToDynamoDB;
//# sourceMappingURL=data:application/json;base64,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