"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ensurePackage = exports.removeDependenciesFromPackageJson = exports.addDependenciesToPackageJson = void 0;
const tslib_1 = require("tslib");
const json_1 = require("nx/src/generators/utils/json");
const install_packages_task_1 = require("../tasks/install-packages-task");
const semver_1 = require("semver");
const package_manager_1 = require("nx/src/utils/package-manager");
const child_process_1 = require("child_process");
const package_json_1 = require("nx/src/utils/package-json");
const NON_SEMVER_TAGS = {
    '*': 2,
    next: 1,
    latest: 0,
    previous: -1,
    legacy: -2,
};
function filterExistingDependencies(dependencies, existingAltDependencies) {
    if (!existingAltDependencies) {
        return dependencies;
    }
    return Object.keys(dependencies !== null && dependencies !== void 0 ? dependencies : {})
        .filter((d) => !existingAltDependencies[d])
        .reduce((acc, d) => (Object.assign(Object.assign({}, acc), { [d]: dependencies[d] })), {});
}
function cleanSemver(version) {
    var _a;
    return (_a = (0, semver_1.clean)(version)) !== null && _a !== void 0 ? _a : (0, semver_1.coerce)(version);
}
function isIncomingVersionGreater(incomingVersion, existingVersion) {
    if (incomingVersion in NON_SEMVER_TAGS &&
        existingVersion in NON_SEMVER_TAGS) {
        return NON_SEMVER_TAGS[incomingVersion] > NON_SEMVER_TAGS[existingVersion];
    }
    if (incomingVersion in NON_SEMVER_TAGS ||
        existingVersion in NON_SEMVER_TAGS) {
        return true;
    }
    return (0, semver_1.gt)(cleanSemver(incomingVersion), cleanSemver(existingVersion));
}
function updateExistingDependenciesVersion(dependencies, existingAltDependencies) {
    return Object.keys(existingAltDependencies || {})
        .filter((d) => {
        if (!dependencies[d]) {
            return false;
        }
        const incomingVersion = dependencies[d];
        const existingVersion = existingAltDependencies[d];
        return isIncomingVersionGreater(incomingVersion, existingVersion);
    })
        .reduce((acc, d) => (Object.assign(Object.assign({}, acc), { [d]: dependencies[d] })), {});
}
/**
 * Add Dependencies and Dev Dependencies to package.json
 *
 * For example:
 * ```typescript
 * addDependenciesToPackageJson(tree, { react: 'latest' }, { jest: 'latest' })
 * ```
 * This will **add** `react` and `jest` to the dependencies and devDependencies sections of package.json respectively.
 *
 * @param tree Tree representing file system to modify
 * @param dependencies Dependencies to be added to the dependencies section of package.json
 * @param devDependencies Dependencies to be added to the devDependencies section of package.json
 * @param packageJsonPath Path to package.json
 * @returns Callback to install dependencies only if necessary, no-op otherwise
 */
function addDependenciesToPackageJson(tree, dependencies, devDependencies, packageJsonPath = 'package.json') {
    const currentPackageJson = (0, json_1.readJson)(tree, packageJsonPath);
    let filteredDependencies = filterExistingDependencies(dependencies, currentPackageJson.devDependencies);
    let filteredDevDependencies = filterExistingDependencies(devDependencies, currentPackageJson.dependencies);
    filteredDependencies = Object.assign(Object.assign({}, filteredDependencies), updateExistingDependenciesVersion(devDependencies, currentPackageJson.dependencies));
    filteredDevDependencies = Object.assign(Object.assign({}, filteredDevDependencies), updateExistingDependenciesVersion(dependencies, currentPackageJson.devDependencies));
    filteredDependencies = removeLowerVersions(filteredDependencies, currentPackageJson.dependencies);
    filteredDevDependencies = removeLowerVersions(filteredDevDependencies, currentPackageJson.devDependencies);
    if (requiresAddingOfPackages(currentPackageJson, filteredDependencies, filteredDevDependencies)) {
        (0, json_1.updateJson)(tree, packageJsonPath, (json) => {
            json.dependencies = Object.assign(Object.assign({}, (json.dependencies || {})), filteredDependencies);
            json.devDependencies = Object.assign(Object.assign({}, (json.devDependencies || {})), filteredDevDependencies);
            json.dependencies = sortObjectByKeys(json.dependencies);
            json.devDependencies = sortObjectByKeys(json.devDependencies);
            return json;
        });
        return () => {
            (0, install_packages_task_1.installPackagesTask)(tree);
        };
    }
    return () => { };
}
exports.addDependenciesToPackageJson = addDependenciesToPackageJson;
/**
 * @returns The the incoming dependencies that are higher than the existing verions
 **/
function removeLowerVersions(incomingDeps, existingDeps) {
    return Object.keys(incomingDeps).reduce((acc, d) => {
        if ((existingDeps === null || existingDeps === void 0 ? void 0 : existingDeps[d]) &&
            !isIncomingVersionGreater(incomingDeps[d], existingDeps[d])) {
            return acc;
        }
        return Object.assign(Object.assign({}, acc), { [d]: incomingDeps[d] });
    }, {});
}
/**
 * Remove Dependencies and Dev Dependencies from package.json
 *
 * For example:
 * ```typescript
 * removeDependenciesFromPackageJson(tree, ['react'], ['jest'])
 * ```
 * This will **remove** `react` and `jest` from the dependencies and devDependencies sections of package.json respectively.
 *
 * @param dependencies Dependencies to be removed from the dependencies section of package.json
 * @param devDependencies Dependencies to be removed from the devDependencies section of package.json
 * @returns Callback to uninstall dependencies only if necessary. undefined is returned if changes are not necessary.
 */
function removeDependenciesFromPackageJson(tree, dependencies, devDependencies, packageJsonPath = 'package.json') {
    const currentPackageJson = (0, json_1.readJson)(tree, packageJsonPath);
    if (requiresRemovingOfPackages(currentPackageJson, dependencies, devDependencies)) {
        (0, json_1.updateJson)(tree, packageJsonPath, (json) => {
            for (const dep of dependencies) {
                delete json.dependencies[dep];
            }
            for (const devDep of devDependencies) {
                delete json.devDependencies[devDep];
            }
            json.dependencies = sortObjectByKeys(json.dependencies);
            json.devDependencies = sortObjectByKeys(json.devDependencies);
            return json;
        });
    }
    return () => {
        (0, install_packages_task_1.installPackagesTask)(tree);
    };
}
exports.removeDependenciesFromPackageJson = removeDependenciesFromPackageJson;
function sortObjectByKeys(obj) {
    if (!obj || typeof obj !== 'object' || Array.isArray(obj)) {
        return obj;
    }
    return Object.keys(obj)
        .sort()
        .reduce((result, key) => {
        return Object.assign(Object.assign({}, result), { [key]: obj[key] });
    }, {});
}
/**
 * Verifies whether the given packageJson dependencies require an update
 * given the deps & devDeps passed in
 */
function requiresAddingOfPackages(packageJsonFile, deps, devDeps) {
    let needsDepsUpdate = false;
    let needsDevDepsUpdate = false;
    packageJsonFile.dependencies = packageJsonFile.dependencies || {};
    packageJsonFile.devDependencies = packageJsonFile.devDependencies || {};
    if (Object.keys(deps).length > 0) {
        needsDepsUpdate = Object.keys(deps).some((entry) => {
            const incomingVersion = deps[entry];
            if (packageJsonFile.dependencies[entry]) {
                const existingVersion = packageJsonFile.dependencies[entry];
                return isIncomingVersionGreater(incomingVersion, existingVersion);
            }
            if (packageJsonFile.devDependencies[entry]) {
                const existingVersion = packageJsonFile.devDependencies[entry];
                return isIncomingVersionGreater(incomingVersion, existingVersion);
            }
            return true;
        });
    }
    if (Object.keys(devDeps).length > 0) {
        needsDevDepsUpdate = Object.keys(devDeps).some((entry) => {
            const incomingVersion = devDeps[entry];
            if (packageJsonFile.devDependencies[entry]) {
                const existingVersion = packageJsonFile.devDependencies[entry];
                return isIncomingVersionGreater(incomingVersion, existingVersion);
            }
            if (packageJsonFile.dependencies[entry]) {
                const existingVersion = packageJsonFile.dependencies[entry];
                return isIncomingVersionGreater(incomingVersion, existingVersion);
            }
            return true;
        });
    }
    return needsDepsUpdate || needsDevDepsUpdate;
}
/**
 * Verifies whether the given packageJson dependencies require an update
 * given the deps & devDeps passed in
 */
function requiresRemovingOfPackages(packageJsonFile, deps, devDeps) {
    let needsDepsUpdate = false;
    let needsDevDepsUpdate = false;
    packageJsonFile.dependencies = packageJsonFile.dependencies || {};
    packageJsonFile.devDependencies = packageJsonFile.devDependencies || {};
    if (deps.length > 0) {
        needsDepsUpdate = deps.some((entry) => packageJsonFile.dependencies[entry]);
    }
    if (devDeps.length > 0) {
        needsDevDepsUpdate = devDeps.some((entry) => packageJsonFile.devDependencies[entry]);
    }
    return needsDepsUpdate || needsDevDepsUpdate;
}
/**
 * @typedef EnsurePackageOptions
 * @type {object}
 * @property {boolean} dev indicate if the package is a dev dependency
 * @property {throwOnMissing} boolean throws an error when the package is missing
 */
/**
 * Ensure that dependencies and devDependencies from package.json are installed at the required versions.
 *
 * For example:
 * ```typescript
 * ensurePackage(tree, {}, { '@nrwl/jest': nxVersion })
 * ```
 * This will check that @nrwl/jest@<nxVersion> exists in devDependencies.
 * If it exists then function returns, otherwise it will install the package before continuing.
 * When running with --dryRun, the function will throw when dependencies are missing.
 *
 * @param tree the file system tree
 * @param pkg the package to check (e.g. @nrwl/jest)
 * @param requiredVersion the version or semver range to check (e.g. ~1.0.0, >=1.0.0 <2.0.0)
 * @param {EnsurePackageOptions} options
 * @returns {Promise<void>}
 */
function ensurePackage(tree, pkg, requiredVersion, options = {}) {
    var _a, _b, _c;
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        let version;
        // Read package and version from root package.json file.
        const dev = (_a = options.dev) !== null && _a !== void 0 ? _a : true;
        const throwOnMissing = (_b = options.throwOnMissing) !== null && _b !== void 0 ? _b : !!process.env.NX_DRY_RUN; // NX_DRY_RUN is set in `packages/nx/src/command-line/generate.ts`
        const pmc = (0, package_manager_1.getPackageManagerCommand)();
        // Try to resolve the actual version from resolved module.
        try {
            version = (0, package_json_1.readModulePackageJson)(pkg).packageJson.version;
        }
        catch (_d) {
            // ignore
        }
        // Otherwise try to read in from package.json. This is needed for E2E tests to pass.
        if (!version) {
            const packageJson = (0, json_1.readJson)(tree, 'package.json');
            const field = dev ? 'devDependencies' : 'dependencies';
            version = (_c = packageJson[field]) === null || _c === void 0 ? void 0 : _c[pkg];
        }
        if (!(0, semver_1.satisfies)(version, requiredVersion)) {
            const installCmd = `${dev ? pmc.addDev : pmc.add} ${pkg}@${requiredVersion}`;
            if (throwOnMissing) {
                throw new Error(`Cannot install required package ${pkg} during a dry run. Run the generator without --dryRun, or install the package with "${installCmd}" and try again.`);
            }
            else {
                (0, child_process_1.execSync)(installCmd, {
                    cwd: tree.root,
                    stdio: [0, 1, 2],
                });
            }
        }
    });
}
exports.ensurePackage = ensurePackage;
//# sourceMappingURL=package-json.js.map