import { Project } from "projen";
import { TypeScriptProject, TypeScriptProjectOptions } from "projen/lib/typescript";
import { Nx } from "./nx-types";
import { SyncpackConfig } from "./syncpack-options";
/**
 * Workspace configurations.
 *
 * @see https://classic.yarnpkg.com/lang/en/docs/workspaces/
 */
export interface WorkspaceConfig {
    /**
     * List of package globs to exclude from hoisting in the workspace.
     *
     * @see https://classic.yarnpkg.com/blog/2018/02/15/nohoist/
     */
    readonly noHoist?: string[];
    /**
     * Disable automatically applying `noHoist` logic for all sub-project "bundledDependencies".
     *
     * @default false
     */
    readonly disableNoHoistBundled?: boolean;
    /**
     * List of additional package globs to include in the workspace.
     *
     * All packages which are parented by the monorepo are automatically added to the workspace, but you can use this
     * property to specify any additional paths to packages which may not be managed by projen.
     */
    readonly additionalPackages?: string[];
}
/**
 * Configuration for Monorepo Upgrade Deps task.
 */
export interface MonorepoUpgradeDepsOptions {
    /**
     * Name of the task to create.
     *
     * @default upgrade-deps
     */
    readonly taskName?: string;
    /**
     * Syncpack configuration.
     *
     * No merging is performed and as such a complete syncpackConfig is required if supplied.
     *
     * @default SyncpackConfig.DEFAULT_CONFIG
     */
    readonly syncpackConfig?: SyncpackConfig;
}
/**
 * Configuration options for the NxMonorepoProject.
 */
export interface NxMonorepoProjectOptions extends TypeScriptProjectOptions {
    /**
     * Configuration for NX.
     */
    readonly nxConfig?: Nx.WorkspaceConfig;
    /**
     * Configuration for workspace.
     */
    readonly workspaceConfig?: WorkspaceConfig;
    /**
     * Whether to include an upgrade-deps task at the root of the monorepo which will upgrade all dependencies.
     *
     * @default true
     */
    readonly monorepoUpgradeDeps?: boolean;
    /**
     * Monorepo Upgrade Deps options.
     *
     * This is only used if monorepoUpgradeDeps is true.
     *
     * @default undefined
     */
    readonly monorepoUpgradeDepsOptions?: MonorepoUpgradeDepsOptions;
}
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
export declare class NxMonorepoProject extends TypeScriptProject {
    private readonly implicitDependencies;
    private readonly nxConfig?;
    private readonly workspaceConfig?;
    private readonly workspacePackages;
    private readonly nxJson;
    constructor(options: NxMonorepoProjectOptions);
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent: Project, dependee: Project): void;
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs: string[]): void;
    private get instantiationOrderSubProjects();
    get subProjects(): Project[];
    /**
     * @inheritDoc
     */
    synth(): void;
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    private validateSubProjects;
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    private synthesizeNonNodePackageJson;
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    private updateWorkspace;
    /**
     * Updates the install task for python projects so that they are run serially and in dependency order such that python
     * projects within the monorepo can declare dependencies on one another.
     * @private
     */
    private wirePythonDependencies;
    /**
     * Copies the given step into the given task. Step and Task must be from the given Project
     * @private
     */
    private copyStepIntoTask;
}
