# Utility functions for the crystal module of py4DSTEM

import numpy as np
from dataclasses import dataclass
import copy
from py4DSTEM.utils.tqdmnd import tqdmnd
from scipy.ndimage import gaussian_filter


@dataclass
class Orientation:
    """
    A class for storing output orientations, generated by fitting a Crystal
    class orientation plan or Bloch wave pattern matching to a PointList.
    """

    num_matches: int

    def __post_init__(self):
        self.matrix = np.zeros((self.num_matches, 3, 3))
        self.family = np.zeros((self.num_matches, 3, 3))
        self.corr = np.zeros((self.num_matches))
        self.inds = np.zeros((self.num_matches, 2), dtype="int")
        self.mirror = np.zeros((self.num_matches), dtype="bool")
        self.angles = np.zeros((self.num_matches, 3))


@dataclass
class OrientationMap:
    """
    A class for storing output orientations, generated by fitting a Crystal class orientation plan or
    Bloch wave pattern matching to a PointListArray.

    """

    num_x: int
    num_y: int
    num_matches: int

    def __post_init__(self):
        # initialize empty arrays
        self.matrix = np.zeros((self.num_x, self.num_y, self.num_matches, 3, 3))
        self.family = np.zeros((self.num_x, self.num_y, self.num_matches, 3, 3))
        self.corr = np.zeros((self.num_x, self.num_y, self.num_matches))
        self.inds = np.zeros((self.num_x, self.num_y, self.num_matches, 2), dtype="int")
        self.mirror = np.zeros((self.num_x, self.num_y, self.num_matches), dtype="bool")
        self.angles = np.zeros((self.num_x, self.num_y, self.num_matches, 3))

    def set_orientation(self, orientation, ind_x, ind_y):
        # Add an orientation to the orientation map
        self.matrix[ind_x, ind_y] = orientation.matrix
        self.family[ind_x, ind_y] = orientation.family
        self.corr[ind_x, ind_y] = orientation.corr
        self.inds[ind_x, ind_y] = orientation.inds
        self.mirror[ind_x, ind_y] = orientation.mirror
        self.angles[ind_x, ind_y] = orientation.angles

    def get_orientation(self, ind_x, ind_y):
        # Return an orientation from the orientation map
        orientation = Orientation(num_matches=self.num_matches)
        orientation.matrix = self.matrix[ind_x, ind_y]
        orientation.family = self.family[ind_x, ind_y]
        orientation.corr = self.corr[ind_x, ind_y]
        orientation.inds = self.inds[ind_x, ind_y]
        orientation.mirror = self.mirror[ind_x, ind_y]
        orientation.angles = self.angles[ind_x, ind_y]
        return orientation

    # def __copy__(self):
    #     return OrientationMap(self.name)
    # def __deepcopy__(self, memo):
    #     return OrientationMap(copy.deepcopy(self.name, memo))


def sort_orientation_maps(
    orientation_map,
    sort = "intensity",
    cluster_thresh = 0.1,
    ):
    """
    Sort the orientation maps along the ind_match direction, either by intensity
    or by clustering similar angles (greedily, in order of intensity).

    Args:
        orientation_map             Initial OrientationMap
        sort (string):              "intensity" or "cluster" for sorting method.
        cluster_thresh (float):     similarity threshold for clustering method

    Returns:
        orientation_sort            Sorted OrientationMap
    """

    # make a deep copy
    orientation_sort = copy.deepcopy(orientation_map)

    if sort == "intensity":
        for rx, ry in tqdmnd(
            orientation_sort.num_x,
            orientation_sort.num_y,
            desc="Sorting orientations",
            unit=" probe positions",
            # disable=not progress_bar,
            ):
            inds = np.argsort(orientation_map.corr[rx,ry])[::-1]

            orientation_sort.matrix[rx,ry,:,:,:] = orientation_sort.matrix[rx,ry,inds,:,:]        
            orientation_sort.family[rx,ry,:,:,:] = orientation_sort.family[rx,ry,inds,:,:]        
            orientation_sort.corr[rx,ry,:] = orientation_sort.corr[rx,ry,inds]        
            orientation_sort.inds[rx,ry,:,:] = orientation_sort.inds[rx,ry,inds,:]        
            orientation_sort.mirror[rx,ry,:] = orientation_sort.mirror[rx,ry,inds]        
            orientation_sort.angles[rx,ry,:,:] = orientation_sort.angles[rx,ry,inds,:]        

        # elif sort == "cluster":
        #     mask = np.zeros_like(orientation_map.corr, dtype='bool')
        # TODO - implement clustering method for sorting

    else:
        err_msg = "Invalid sorting method: " + sort
        raise Exception(err_msg) 


    return orientation_sort




def calc_1D_profile(
    k,
    g_coords,
    g_int,
    remove_origin = True,
    k_broadening = 0.0,
    int_scale = None,
    normalize_intensity = True,
    ):
    """
    Utility function to calculate a 1D histogram from the diffraction vector lengths
    stored in a Crystal class. 

    Args:
            k (np.array):                       k coordinates.
            g_coords (np.array):                Scattering vector lengths g.
            bragg_intensity_power (np.array):   Scattering vector intensities.
            remove_origin (bool):               Remove the origin peak from the profile.
            k_broadening (float):               Broadening applied to full profile.
            int_scale (np.array):               Either a scalar value mulitiplied into all peak intensities,
                                                or a vector with 1 value per peak to scale peaks individually.
            normalize_intensity (bool):         Normalize maximum output value to 1.

        Returns:
        int_profile (np.array):                 Computed intensity profile
    """

    # init
    int_scale = np.atleast_1d(int_scale)
    k_num = k.shape[0]
    k_min = k[0]
    k_step = k[1] - k[0]
    k_max = k[-1]

    # get discrete plot from structure factor amplitudes
    int_profile = np.zeros_like(k)
    k_px = (g_coords - k_min) / k_step;
    kf = np.floor(k_px).astype('int')
    dk = k_px - kf;

    sub = np.logical_and(kf >= 0, kf < k_num)
    if int_scale.shape[0] > 1:
        int_profile = np.bincount(
            np.floor(k_px[sub]).astype('int'), 
            weights = (1-dk[sub])*g_int[sub]*int_scale[sub], 
            minlength = k_num)
    else:
        int_profile = np.bincount(
            np.floor(k_px[sub]).astype('int'), 
            weights = (1-dk[sub])*g_int[sub], 
            minlength = k_num)
    sub = np.logical_and(k_px >= -1, k_px < k_num-1)
    if int_scale.shape[0] > 1:
        int_profile += np.bincount(
            np.floor(k_px[sub] + 1).astype('int'), 
            weights = dk[sub]*g_int[sub]*int_scale[sub], 
            minlength = k_num)
    else:
        int_profile += np.bincount(
            np.floor(k_px[sub] + 1).astype('int'), 
            weights = dk[sub]*g_int[sub], 
            minlength = k_num)

    if remove_origin is True:
        int_profile[0:2] = 0

    # Apply broadening if needed
    if k_broadening > 0.0:
        int_profile = gaussian_filter(int_profile, k_broadening/k_step, mode='constant')

    if normalize_intensity:
        int_profile /= np.max(int_profile)
    if int_scale is not None:
        if int_scale.shape[0] == 1:
            int_profile *= int_scale

    return int_profile


def axisEqual3D(ax):
    extents = np.array([getattr(ax, "get_{}lim".format(dim))() for dim in "xyz"])
    sz = extents[:, 1] - extents[:, 0]
    centers = np.mean(extents, axis=1)
    maxsize = max(abs(sz))
    r = maxsize / 2
    for ctr, dim in zip(centers, "xyz"):
        getattr(ax, "set_{}lim".format(dim))(ctr - r, ctr + r)


# fmt: off

# a list of symbols for each element
element_symbols = ('H', 'He', 'Li', 'Be', 'B', 'C', 'N', 'O', 'F', 'Ne', 'Na', 'Mg', 'Al', 'Si',
       'P', 'S', 'Cl', 'Ar', 'K', 'Ca', 'Sc', 'Ti', 'V', 'Cr', 'Mn', 'Fe', 'Co', 'Ni',
       'Cu', 'Zn', 'Ga', 'Ge', 'As', 'Se', 'Br', 'Kr', 'Rb', 'Sr', 'Y', 'Zr', 'Nb',
       'Mo', 'Tc', 'Ru', 'Rh', 'Pd', 'Ag', 'Cd', 'In', 'Sn', 'Sb', 'Te', 'I', 'Xe',
       'Cs', 'Ba', 'La', 'Ce', 'Pr', 'Nd', 'Pm', 'Sm', 'Eu', 'Gd', 'Tb', 'Dy', 'Ho',
       'Er', 'Tm', 'Yb', 'Lu', 'Hf', 'Ta', 'W', 'Re', 'Os', 'Ir', 'Pt', 'Au', 'Hg', 'Tl',
       'Pb', 'Bi', 'Po', 'At', 'Rn', 'Fr', 'Ra', 'Ac', 'Th', 'Pa', 'U', 'Np', 'Pu', 'Am',
       'Cm', 'Bk', 'Cf', 'Es', 'Fm', 'Md', 'No', 'Lr', 'Rf', 'Db', 'Sg', 'Bh', 'Hs', 'Mt',
       'Ds', 'Rg', 'Cn', 'Nh', 'Fl', 'Mc', 'Lv', 'Ts', 'Og', 'Uue', 'Ubn', 'Ubn')

# a dictionary for converting element names into Z
elements = {element_symbols[i]: i + 1 for i in range(len(element_symbols))}
# fmt: on
