# Passphrase Dictionary
#
# Interfaces to the passphrase dictionary (the dictionary acts as the alphabet
# used to create pass phrases).

# License {{{1
# Copyright (C) 2013-2020 Kenneth S. Kundert and Kale Kundert
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/.

# Imports {{{1
from .config import get_setting
from .error import PasswordError
from .utilities import OSErrors
from inform import os_error
from .shlib import to_path


# Globals {{{1
known_dictionaries = {}


# Dictionary class {{{1
class Dictionary(object):
    """Read Dictionary"""
    def __init__(self, dictionary=None):
        self.dictionary = dictionary
        self.words = known_dictionaries.get(dictionary)

    def get_words(self):
        if not self.words:
            if self.dictionary in known_dictionaries:
                self.words = known_dictionaries[self.dictionary]
            else:
                if not self.dictionary:
                    self.dictionary = get_setting('dictionary_file')
                if not self.dictionary or self.dictionary in ['default']:
                    self.words = WORDS
                elif self.dictionary in ['bip39', 'mnemonic']:
                    self.words = BIP39_WORDS
                else:
                    try:
                        path = to_path(self.dictionary)
                        contents = path.read_text()
                    except OSErrors as e:
                        raise PasswordError(os_error(e))
                    self.words = contents.split()
                known_dictionaries[self.dictionary] = self.words
        return self.words


# Words {{{1
WORDS = [
    'abandon',
    'abase',
    'abate',
    'abbess',
    'abbey',
    'abbot',
    'abdicate',
    'abdomen',
    'abduct',
    'abhor',
    'abide',
    'ability',
    'abjure',
    'abode',
    'abolish',
    'abominate',
    'abort',
    'abortion',
    'abound',
    'abrasion',
    'abrasive',
    'abridge',
    'abrogate',
    'abscess',
    'abscond',
    'abseil',
    'absence',
    'absent',
    'absentee',
    'absolute',
    'absolve',
    'absorb',
    'abstain',
    'abstract',
    'abuse',
    'abyss',
    'acacia',
    'academic',
    'academy',
    'accede',
    'accent',
    'accept',
    'access',
    'accessory',
    'accident',
    'acclaim',
    'accolade',
    'accompany',
    'accord',
    'accordion',
    'accost',
    'account',
    'accredit',
    'accretion',
    'accrue',
    'accuse',
    'accuser',
    'accustom',
    'achieve',
    'acolyte',
    'acorn',
    'acoustic',
    'acquaint',
    'acquiesce',
    'acquire',
    'acquit',
    'acquittal',
    'acrobat',
    'acrobatic',
    'acronym',
    'acrostic',
    'acrylic',
    'action',
    'activate',
    'activist',
    'activity',
    'actor',
    'actress',
    'actuary',
    'actuate',
    'adapt',
    'adaptor',
    'addendum',
    'adder',
    'addict',
    'addiction',
    'addition',
    'additive',
    'addle',
    'address',
    'addressee',
    'adduce',
    'adept',
    'adhere',
    'adherent',
    'adhesive',
    'adieu',
    'adjective',
    'adjoin',
    'adjourn',
    'adjudge',
    'adjunct',
    'adjure',
    'adjust',
    'adjutant',
    'admiral',
    'admire',
    'admirer',
    'admission',
    'admit',
    'admonish',
    'adopt',
    'adore',
    'adorn',
    'adornment',
    'adult',
    'adulterer',
    'adultery',
    'advance',
    'advantage',
    'adventure',
    'adverb',
    'adversary',
    'adversity',
    'advert',
    'advertise',
    'advise',
    'adviser',
    'advocate',
    'aerate',
    'aerial',
    'aerodrome',
    'aerosol',
    'aesthete',
    'aesthetic',
    'affair',
    'affect',
    'affection',
    'affidavit',
    'affiliate',
    'affinity',
    'affirm',
    'affix',
    'afflict',
    'afford',
    'affray',
    'affront',
    'after',
    'afternoon',
    'agate',
    'agency',
    'agenda',
    'agent',
    'aggravate',
    'aggregate',
    'aggressor',
    'agitate',
    'agitator',
    'agnostic',
    'agonize',
    'agony',
    'agree',
    'agreement',
    'aileron',
    'ailment',
    'aircrew',
    'airfield',
    'airlift',
    'airline',
    'airliner',
    'airlock',
    'airman',
    'airplane',
    'airport',
    'airship',
    'airspeed',
    'airstrike',
    'airstrip',
    'airway',
    'aisle',
    'aitch',
    'alarm',
    'alarmist',
    'albatross',
    'albino',
    'album',
    'albumen',
    'alchemist',
    'alcoholic',
    'alcove',
    'alder',
    'alderman',
    'alert',
    'algorithm',
    'alias',
    'alibi',
    'alien',
    'alienate',
    'alight',
    'align',
    'alignment',
    'alkali',
    'allay',
    'allege',
    'allegory',
    'allergy',
    'alleviate',
    'alley',
    'alleyway',
    'alliance',
    'alligator',
    'allocate',
    'allot',
    'allotment',
    'allow',
    'allowance',
    'alloy',
    'allude',
    'allusion',
    'almanac',
    'almond',
    'almshouse',
    'alpha',
    'alphabet',
    'alpine',
    'altar',
    'alter',
    'alternate',
    'altimeter',
    'altitude',
    'alumna',
    'alumnus',
    'amalgam',
    'amass',
    'amateur',
    'amaze',
    'ambiguity',
    'ambition',
    'amble',
    'ambulance',
    'ambush',
    'amend',
    'amendment',
    'amenity',
    'amethyst',
    'amnesty',
    'amoeba',
    'amount',
    'amour',
    'ampere',
    'amphibian',
    'amplifier',
    'amplify',
    'amputate',
    'amulet',
    'amuse',
    'amusement',
    'anagram',
    'analgesic',
    'analogue',
    'analogy',
    'analyze',
    'analysis',
    'analyst',
    'analytic',
    'anarchist',
    'anatomist',
    'anatomy',
    'ancestor',
    'ancestry',
    'anchor',
    'anchorage',
    'anchovy',
    'ancient',
    'andante',
    'anecdote',
    'anemone',
    'angel',
    'anger',
    'angle',
    'angler',
    'anglicize',
    'angry',
    'animal',
    'animate',
    'animator',
    'animosity',
    'ankle',
    'annex',
    'annotate',
    'announce',
    'announcer',
    'annoy',
    'annoyance',
    'annual',
    'annuity',
    'annul',
    'anode',
    'anodyne',
    'anoint',
    'anomaly',
    'anorak',
    'answer',
    'antedate',
    'antelope',
    'antenatal',
    'antenna',
    'anteroom',
    'anthem',
    'anthology',
    'antibody',
    'antidote',
    'antigen',
    'antihero',
    'antipathy',
    'antiquary',
    'antique',
    'antiquity',
    'antitoxin',
    'antler',
    'antonym',
    'anvil',
    'anxiety',
    'aorta',
    'apartment',
    'aperitif',
    'aperture',
    'aphid',
    'aphorism',
    'apologia',
    'apologist',
    'apologize',
    'apology',
    'apostate',
    'apostle',
    'apparatus',
    'appeal',
    'appear',
    'appease',
    'append',
    'appendage',
    'appendix',
    'appendix',
    'appertain',
    'appetite',
    'appetizer',
    'applaud',
    'apple',
    'appliance',
    'applicant',
    'apply',
    'appoint',
    'appointee',
    'apportion',
    'appraisal',
    'appraise',
    'apprehend',
    'apprise',
    'approach',
    'approve',
    'apricot',
    'apron',
    'aptitude',
    'aqualung',
    'aquarium',
    'aqueduct',
    'arabesque',
    'arbiter',
    'arbitrate',
    'arcade',
    'archangel',
    'archer',
    'archetype',
    'architect',
    'archive',
    'archivist',
    'archway',
    'arena',
    'argot',
    'argue',
    'argument',
    'arise',
    'armada',
    'armadillo',
    'armament',
    'armband',
    'armchair',
    'armful',
    'armhole',
    'armistice',
    'armory',
    'armpit',
    'aroma',
    'arouse',
    'arraign',
    'arrange',
    'array',
    'arrest',
    'arrival',
    'arrive',
    'arrogate',
    'arrow',
    'arrowhead',
    'arsehole',
    'arsenal',
    'artifact',
    'artery',
    'arthritic',
    'artichoke',
    'article',
    'artifice',
    'artisan',
    'artist',
    'artiste',
    'ascend',
    'ascent',
    'ascertain',
    'ascetic',
    'ascribe',
    'ashcan',
    'ashtray',
    'aside',
    'aspect',
    'aspen',
    'asphalt',
    'aspirant',
    'aspirate',
    'aspire',
    'aspirin',
    'assail',
    'assailant',
    'assassin',
    'assault',
    'assay',
    'assemble',
    'assembly',
    'assent',
    'assert',
    'assess',
    'assessor',
    'asset',
    'assign',
    'assist',
    'assistant',
    'associate',
    'assuage',
    'assume',
    'assurance',
    'assure',
    'aster',
    'asterisk',
    'asteroid',
    'asthmatic',
    'astonish',
    'astound',
    'astronaut',
    'asylum',
    'atheist',
    'athlete',
    'athletic',
    'atlas',
    'atoll',
    'atomize',
    'atomizer',
    'atone',
    'atrocity',
    'atrophy',
    'attach',
    'attache',
    'attack',
    'attain',
    'attempt',
    'attend',
    'attendant',
    'attention',
    'attenuate',
    'attest',
    'attic',
    'attitude',
    'attorney',
    'attract',
    'attribute',
    'aubergine',
    'auction',
    'audience',
    'audit',
    'audition',
    'auditor',
    'augment',
    'augur',
    'augury',
    'auntie',
    'austerity',
    'autarchy',
    'author',
    'authoress',
    'authority',
    'authorize',
    'autobahn',
    'autocracy',
    'autocrat',
    'autocue',
    'autograph',
    'automate',
    'automatic',
    'automaton',
    'autopsy',
    'autumn',
    'auxiliary',
    'avail',
    'avalanche',
    'avenge',
    'avenue',
    'average',
    'aversion',
    'avert',
    'aviary',
    'aviator',
    'avocado',
    'avoid',
    'avowal',
    'await',
    'awake',
    'awaken',
    'awakening',
    'award',
    'awning',
    'axiom',
    'babble',
    'baboon',
    'bachelor',
    'backbench',
    'backbite',
    'backbone',
    'backcloth',
    'backdate',
    'backdrop',
    'backer',
    'backfire',
    'backhand',
    'backing',
    'backlog',
    'backpack',
    'backside',
    'backslide',
    'backtrack',
    'backwater',
    'badge',
    'badger',
    'baffle',
    'bagatelle',
    'baggy',
    'bailiff',
    'bairn',
    'baker',
    'bakery',
    'balaclava',
    'balalaika',
    'balance',
    'balcony',
    'ballad',
    'ballcock',
    'ballerina',
    'ballet',
    'ballistic',
    'balloon',
    'ballot',
    'ballpoint',
    'ballroom',
    'balmy',
    'bamboo',
    'bamboozle',
    'banana',
    'bandage',
    'bandanna',
    'bandit',
    'bandsman',
    'bandstand',
    'bandy',
    'banger',
    'bangle',
    'banish',
    'banister',
    'banjo',
    'banker',
    'banknote',
    'bankroll',
    'bankrupt',
    'banner',
    'banquet',
    'banshee',
    'bantam',
    'banter',
    'banyan',
    'baptism',
    'baptize',
    'barbarian',
    'barbarity',
    'barbecue',
    'barber',
    'bargain',
    'barge',
    'baritone',
    'barmaid',
    'barman',
    'barnacle',
    'barnyard',
    'barometer',
    'baron',
    'baroness',
    'baronet',
    'baronetcy',
    'barony',
    'barrack',
    'barracuda',
    'barrage',
    'barrel',
    'barricade',
    'barrier',
    'barrister',
    'barrow',
    'bartender',
    'barter',
    'baseline',
    'basement',
    'basic',
    'basilica',
    'basin',
    'basis',
    'basket',
    'bassoon',
    'bastard',
    'baste',
    'bastion',
    'batch',
    'bathe',
    'bather',
    'bathmat',
    'bathrobe',
    'bathroom',
    'bathtub',
    'batman',
    'baton',
    'batsman',
    'battalion',
    'batten',
    'batter',
    'battering',
    'battery',
    'battle',
    'batty',
    'bauble',
    'bawdy',
    'bayonet',
    'bazaar',
    'bazooka',
    'beach',
    'beachhead',
    'beacon',
    'beady',
    'beagle',
    'beaker',
    'beanpole',
    'beard',
    'bearer',
    'bearing',
    'bearskin',
    'beast',
    'beater',
    'beatify',
    'beating',
    'beatnik',
    'beaut',
    'beautify',
    'beauty',
    'beaver',
    'beckon',
    'become',
    'bedbug',
    'bedeck',
    'bedevil',
    'bedfellow',
    'bedhead',
    'bedpan',
    'bedpost',
    'bedroll',
    'bedroom',
    'bedsitter',
    'bedspread',
    'bedstead',
    'beech',
    'beefy',
    'beehive',
    'beetle',
    'beetroot',
    'befall',
    'befit',
    'befriend',
    'befuddle',
    'beget',
    'beggar',
    'begin',
    'beginner',
    'beginning',
    'begrudge',
    'beguile',
    'behave',
    'behead',
    'behest',
    'behind',
    'behold',
    'beholder',
    'behoove',
    'being',
    'belabor',
    'belch',
    'belfry',
    'belie',
    'belief',
    'believe',
    'believer',
    'belittle',
    'bellboy',
    'belle',
    'bellhop',
    'bellow',
    'belly',
    'bellyache',
    'belong',
    'bemoan',
    'bench',
    'benchmark',
    'bender',
    'benefit',
    'bequeath',
    'bequest',
    'berate',
    'bereave',
    'beret',
    'berry',
    'berth',
    'beseech',
    'beset',
    'besiege',
    'besmirch',
    'bespeak',
    'bestir',
    'bestow',
    'bestrew',
    'bestride',
    'betake',
    'bethink',
    'betoken',
    'betray',
    'betrayal',
    'betrothal',
    'better',
    'beverage',
    'bewail',
    'bewilder',
    'bewitch',
    'bible',
    'bicker',
    'bicycle',
    'bidder',
    'bidet',
    'bifocal',
    'bigamist',
    'bigot',
    'bigwig',
    'bikini',
    'bilberry',
    'billboard',
    'billet',
    'billfold',
    'billion',
    'billow',
    'billy',
    'binder',
    'binding',
    'binge',
    'biography',
    'biped',
    'biplane',
    'birch',
    'birdcage',
    'birdie',
    'birth',
    'birthday',
    'birthmark',
    'biscuit',
    'bisect',
    'bisexual',
    'bishop',
    'bishopric',
    'bison',
    'bistro',
    'bitch',
    'bitchy',
    'bitter',
    'bitty',
    'bivouac',
    'blabber',
    'black',
    'blackball',
    'blackbird',
    'blacken',
    'blackhead',
    'blackjack',
    'blackleg',
    'blacklist',
    'blackmail',
    'blackout',
    'bladder',
    'blade',
    'blame',
    'blanch',
    'bland',
    'blank',
    'blanket',
    'blare',
    'blaspheme',
    'blasphemy',
    'blast',
    'blaze',
    'blazer',
    'blazon',
    'bleach',
    'bleak',
    'bleary',
    'bleat',
    'bleed',
    'bleeder',
    'bleep',
    'bleeper',
    'blemish',
    'blench',
    'blend',
    'blender',
    'bless',
    'blessing',
    'blight',
    'blighter',
    'blind',
    'blindfold',
    'blink',
    'blister',
    'blitz',
    'blizzard',
    'bloater',
    'block',
    'blockade',
    'blockage',
    'blockhead',
    'bloke',
    'blonde',
    'bloodbath',
    'bloody',
    'bloom',
    'bloomer',
    'blossom',
    'blotch',
    'blotchy',
    'blotter',
    'blouse',
    'blowpipe',
    'blowtorch',
    'blubber',
    'bludgeon',
    'bluebell',
    'blueberry',
    'bluebird',
    'blueprint',
    'bluff',
    'blunder',
    'blunt',
    'blurry',
    'blurt',
    'blush',
    'bluster',
    'board',
    'boarder',
    'boarding',
    'boardroom',
    'boardwalk',
    'boast',
    'boater',
    'boathouse',
    'boatman',
    'boatswain',
    'bobbin',
    'bobble',
    'bobby',
    'bobsled',
    'bobsleigh',
    'bodice',
    'bodkin',
    'bodyguard',
    'boffin',
    'bogey',
    'boggle',
    'boggy',
    'bohemian',
    'boiler',
    'bolero',
    'bollard',
    'bolster',
    'bombard',
    'bomber',
    'bombshell',
    'bonanza',
    'bonfire',
    'bongo',
    'bonnet',
    'bonny',
    'bonsai',
    'bonus',
    'boogie',
    'boohoo',
    'bookcase',
    'bookend',
    'bookie',
    'booking',
    'booklet',
    'bookmaker',
    'bookmark',
    'bookshelf',
    'bookshop',
    'bookstall',
    'bookstore',
    'bookworm',
    'boomerang',
    'boost',
    'booster',
    'bootee',
    'booth',
    'bootlace',
    'booze',
    'boozer',
    'bordello',
    'border',
    'borehole',
    'borough',
    'borrow',
    'borrower',
    'borrowing',
    'borstal',
    'bosom',
    'bossy',
    'botanist',
    'botch',
    'bother',
    'bottle',
    'bottom',
    'boudoir',
    'bouffant',
    'bough',
    'bouillon',
    'boulder',
    'boulevard',
    'bounce',
    'bouncer',
    'bouncy',
    'bound',
    'boundary',
    'bounder',
    'bounty',
    'bouquet',
    'bourbon',
    'boutique',
    'bowel',
    'bower',
    'bowler',
    'boxcar',
    'boxer',
    'boycott',
    'boyfriend',
    'brace',
    'bracelet',
    'bracket',
    'braggart',
    'braid',
    'brain',
    'brainwash',
    'brainwave',
    'brainy',
    'braise',
    'brake',
    'bramble',
    'branch',
    'brand',
    'brandish',
    'brandy',
    'brash',
    'brass',
    'brasserie',
    'brassiere',
    'brassy',
    'brave',
    'brawl',
    'brazen',
    'brazier',
    'breach',
    'bread',
    'break',
    'breakage',
    'breakdown',
    'breaker',
    'breakfast',
    'breast',
    'breath',
    'breathe',
    'breather',
    'breech',
    'breed',
    'breeder',
    'breeze',
    'breezy',
    'brewer',
    'brewery',
    'briar',
    'bribe',
    'brick',
    'bride',
    'bridge',
    'bridle',
    'bridleway',
    'brief',
    'briefcase',
    'briefing',
    'brier',
    'brigade',
    'brigadier',
    'brigand',
    'bright',
    'brighten',
    'brimstone',
    'bring',
    'briquette',
    'brisk',
    'bristle',
    'bristly',
    'brittle',
    'broach',
    'broad',
    'broadcast',
    'broaden',
    'broadside',
    'brochure',
    'brogue',
    'broil',
    'broiler',
    'broker',
    'bromide',
    'bronco',
    'bronze',
    'brooch',
    'brood',
    'brooding',
    'brook',
    'broom',
    'broth',
    'brothel',
    'brother',
    'brougham',
    'browbeat',
    'brown',
    'brownie',
    'browse',
    'bruise',
    'bruiser',
    'brunch',
    'brunette',
    'brush',
    'brutality',
    'brutalize',
    'brute',
    'bubble',
    'bubbly',
    'buccaneer',
    'bucket',
    'bucketful',
    'buckle',
    'buckskin',
    'buddy',
    'budge',
    'budget',
    'budgie',
    'buffalo',
    'buffer',
    'buffet',
    'buffoon',
    'bugbear',
    'bugger',
    'buggy',
    'bugle',
    'build',
    'builder',
    'building',
    'bulge',
    'bulkhead',
    'bulky',
    'bulldog',
    'bulldoze',
    'bulldozer',
    'bullet',
    'bulletin',
    'bullfight',
    'bullfinch',
    'bullfrog',
    'bullock',
    'bullring',
    'bullshit',
    'bully',
    'bulrush',
    'bulwark',
    'bumble',
    'bumblebee',
    'bummer',
    'bumper',
    'bumpkin',
    'bumpy',
    'bunch',
    'bundle',
    'bungalow',
    'bungle',
    'bungler',
    'bunion',
    'bunker',
    'bunny',
    'burble',
    'burden',
    'bureau',
    'burgeon',
    'burgher',
    'burglar',
    'burglary',
    'burgle',
    'burial',
    'burlesque',
    'burly',
    'burner',
    'burnish',
    'burrow',
    'bursar',
    'bursary',
    'burst',
    'busby',
    'bushel',
    'bushy',
    'business',
    'busker',
    'bustle',
    'busty',
    'busybody',
    'butcher',
    'butchery',
    'butler',
    'butter',
    'buttercup',
    'butterfly',
    'buttery',
    'buttock',
    'button',
    'buttress',
    'buyer',
    'buzzard',
    'buzzer',
    'bylaw',
    'bygone',
    'bypass',
    'bystander',
    'byway',
    'byword',
    'cabal',
    'cabaret',
    'cabbage',
    'cabin',
    'cabinet',
    'cable',
    'cache',
    'cackle',
    'cactus',
    'cadaver',
    'caddie',
    'cadence',
    'cadenza',
    'cadet',
    'cadge',
    'cadger',
    'cadre',
    'cafeteria',
    'caftan',
    'cairn',
    'cajole',
    'calamity',
    'calculate',
    'calendar',
    'calibrate',
    'caliber',
    'caliph',
    'caller',
    'caliper',
    'callus',
    'calorie',
    'calumny',
    'calve',
    'calypso',
    'camber',
    'camel',
    'camellia',
    'cameo',
    'camera',
    'cameraman',
    'campaign',
    'camper',
    'campsite',
    'campus',
    'camshaft',
    'canal',
    'canalize',
    'canard',
    'canary',
    'cancel',
    'cancer',
    'candidate',
    'candle',
    'candy',
    'canine',
    'canister',
    'canker',
    'cannery',
    'cannibal',
    'cannon',
    'cannonade',
    'canny',
    'canoe',
    'canon',
    'canonize',
    'canoodle',
    'canopy',
    'cantata',
    'canteen',
    'canter',
    'canticle',
    'canto',
    'canton',
    'canvas',
    'canvass',
    'canyon',
    'capacity',
    'caper',
    'capillary',
    'capital',
    'capon',
    'caprice',
    'capsicum',
    'capsize',
    'capsule',
    'captain',
    'caption',
    'captivate',
    'captive',
    'captor',
    'capture',
    'carafe',
    'caramel',
    'carat',
    'caravan',
    'carbine',
    'carbon',
    'carbuncle',
    'carcass',
    'cardigan',
    'cardinal',
    'career',
    'careerist',
    'carer',
    'caress',
    'caretaker',
    'cargo',
    'caribou',
    'carillon',
    'carnation',
    'carnival',
    'carnivore',
    'carob',
    'carol',
    'carouse',
    'carousel',
    'carpenter',
    'carpet',
    'carriage',
    'carrier',
    'carrot',
    'carry',
    'cartel',
    'carthorse',
    'cartilage',
    'carton',
    'cartoon',
    'cartridge',
    'cartwheel',
    'carve',
    'carvery',
    'carving',
    'cascade',
    'casebook',
    'casement',
    'cashew',
    'cashier',
    'casing',
    'casino',
    'casket',
    'casserole',
    'cassette',
    'cassock',
    'castanet',
    'castaway',
    'caste',
    'castigate',
    'casting',
    'castle',
    'castrate',
    'casual',
    'casualty',
    'cataclysm',
    'catacomb',
    'catalog',
    'catalyst',
    'catamaran',
    'catapult',
    'cataract',
    'catcall',
    'catch',
    'catchy',
    'catechism',
    'category',
    'cater',
    'caterer',
    'caterwaul',
    'cathedral',
    'cathode',
    'catkin',
    'catnap',
    'cattleman',
    'catty',
    'catwalk',
    'caucus',
    'cauldron',
    'cause',
    'causeway',
    'cauterize',
    'caution',
    'cavalcade',
    'caveat',
    'caveman',
    'cavern',
    'cavil',
    'cavity',
    'cavort',
    'cease',
    'ceasefire',
    'cedar',
    'cedilla',
    'ceilidh',
    'ceiling',
    'celebrant',
    'celebrate',
    'celebrity',
    'celibate',
    'cellar',
    'cellist',
    'cello',
    'cement',
    'cemetery',
    'cenotaph',
    'censor',
    'censure',
    'census',
    'centaur',
    'centenary',
    'centipede',
    'center',
    'centrist',
    'centurion',
    'century',
    'ceramic',
    'cereal',
    'cerebrum',
    'ceremony',
    'certainty',
    'certify',
    'cervix',
    'cesspit',
    'chafe',
    'chagrin',
    'chain',
    'chair',
    'chairlift',
    'chairman',
    'chalet',
    'chalice',
    'chalk',
    'chalky',
    'challenge',
    'chamber',
    'chameleon',
    'champ',
    'champion',
    'chance',
    'chancel',
    'chancy',
    'change',
    'channel',
    'chant',
    'chapel',
    'chaperone',
    'chaplain',
    'chapter',
    'charabanc',
    'character',
    'charade',
    'charge',
    'charger',
    'chariot',
    'charity',
    'charlady',
    'charlatan',
    'charm',
    'charmer',
    'chart',
    'charter',
    'charwoman',
    'chase',
    'chaser',
    'chasm',
    'chasten',
    'chastise',
    'chateau',
    'chattel',
    'chatter',
    'chatterer',
    'chatty',
    'chauffeur',
    'cheap',
    'cheapen',
    'cheat',
    'check',
    'checker',
    'checkmate',
    'checkout',
    'cheddar',
    'cheek',
    'cheekbone',
    'cheeky',
    'cheer',
    'cheery',
    'cheese',
    'cheetah',
    'chemical',
    'chemist',
    'cherish',
    'cheroot',
    'cherry',
    'cherub',
    'chessman',
    'chest',
    'chestnut',
    'chevron',
    'chewy',
    'chicanery',
    'chick',
    'chicken',
    'chickpea',
    'chide',
    'chief',
    'chieftain',
    'chihuahua',
    'chilblain',
    'child',
    'childhood',
    'chill',
    'chili',
    'chilly',
    'chime',
    'chimera',
    'chimney',
    'chimp',
    'chink',
    'chipmunk',
    'chirp',
    'chirpy',
    'chirrup',
    'chisel',
    'chivvy',
    'chloride',
    'chocolate',
    'choice',
    'choir',
    'choirboy',
    'choke',
    'choker',
    'chomp',
    'choose',
    'choosy',
    'chopper',
    'choppy',
    'chopstick',
    'chord',
    'chore',
    'chorister',
    'chortle',
    'chorus',
    'christen',
    'chronicle',
    'chrysalis',
    'chubby',
    'chuck',
    'chuckle',
    'chummy',
    'chump',
    'chunk',
    'chunky',
    'church',
    'churchman',
    'churn',
    'chute',
    'chutney',
    'cicada',
    'cider',
    'cigar',
    'cigarette',
    'cinder',
    'cinema',
    'cipher',
    'circle',
    'circlet',
    'circuit',
    'circular',
    'circulate',
    'circus',
    'cistern',
    'citadel',
    'citation',
    'citizen',
    'civilian',
    'civility',
    'civilize',
    'clack',
    'claim',
    'claimant',
    'clamber',
    'clammy',
    'clamor',
    'clamp',
    'clampdown',
    'clang',
    'clanger',
    'clank',
    'clapboard',
    'clapper',
    'claret',
    'clarify',
    'clarinet',
    'clash',
    'clasp',
    'class',
    'classic',
    'classify',
    'classmate',
    'classroom',
    'classy',
    'clatter',
    'clause',
    'clean',
    'cleaner',
    'cleanse',
    'cleanser',
    'clear',
    'clearance',
    'clearing',
    'clearway',
    'cleavage',
    'cleave',
    'cleaver',
    'cleft',
    'clench',
    'clergyman',
    'cleric',
    'clerk',
    'clever',
    'cliche',
    'click',
    'client',
    'cliff',
    'climate',
    'climax',
    'climb',
    'climber',
    'clime',
    'clinch',
    'clincher',
    'cling',
    'clinic',
    'clink',
    'clipboard',
    'clipper',
    'clipping',
    'clique',
    'clitoris',
    'cloak',
    'cloakroom',
    'clobber',
    'cloche',
    'clock',
    'cloister',
    'clone',
    'clonk',
    'close',
    'closet',
    'closure',
    'cloth',
    'clothe',
    'cloud',
    'clout',
    'clove',
    'clown',
    'clubhouse',
    'cluck',
    'clump',
    'clumsy',
    'clunk',
    'cluster',
    'clutch',
    'clutter',
    'coach',
    'coachman',
    'coagulate',
    'coalesce',
    'coalface',
    'coalfield',
    'coalition',
    'coarse',
    'coarsen',
    'coast',
    'coaster',
    'coastline',
    'coating',
    'cobble',
    'cobbler',
    'cobra',
    'cobweb',
    'coccyx',
    'cockatoo',
    'cockerel',
    'cockfight',
    'cockle',
    'cockney',
    'cockpit',
    'cockroach',
    'cockscomb',
    'cocktail',
    'cocky',
    'coconut',
    'cocoon',
    'coddle',
    'codex',
    'codger',
    'codicil',
    'codify',
    'codpiece',
    'coerce',
    'coexist',
    'coffee',
    'coffer',
    'coffin',
    'cogitate',
    'cognac',
    'cohabit',
    'cohere',
    'cohort',
    'coiffure',
    'coinage',
    'coincide',
    'colander',
    'collage',
    'collapse',
    'collar',
    'collate',
    'collation',
    'colleague',
    'collect',
    'collector',
    'colleen',
    'college',
    'collide',
    'collie',
    'colliery',
    'collision',
    'collocate',
    'colloquy',
    'collude',
    'colon',
    'colonel',
    'colonial',
    'colonist',
    'colonize',
    'colonnade',
    'colony',
    'colossus',
    'color',
    'colored',
    'columbine',
    'column',
    'columnist',
    'combat',
    'combatant',
    'combine',
    'combo',
    'comeback',
    'comedian',
    'comedy',
    'comely',
    'comer',
    'comet',
    'comfort',
    'comforter',
    'comfy',
    'comic',
    'coming',
    'comma',
    'command',
    'commander',
    'commando',
    'commence',
    'commend',
    'comment',
    'commie',
    'commit',
    'committee',
    'commode',
    'commodity',
    'commodore',
    'common',
    'commoner',
    'commune',
    'communion',
    'communist',
    'community',
    'commute',
    'commuter',
    'compact',
    'companion',
    'company',
    'compare',
    'compass',
    'compel',
    'compete',
    'compile',
    'complain',
    'complaint',
    'complete',
    'complex',
    'comply',
    'component',
    'comport',
    'compose',
    'composer',
    'composite',
    'compost',
    'compote',
    'compound',
    'compress',
    'comprise',
    'compute',
    'computer',
    'comrade',
    'conceal',
    'concede',
    'conceit',
    'conceive',
    'concept',
    'concern',
    'concert',
    'concerto',
    'conch',
    'concierge',
    'conclave',
    'conclude',
    'concoct',
    'concourse',
    'concrete',
    'concubine',
    'concur',
    'concuss',
    'condemn',
    'condense',
    'condiment',
    'condition',
    'condole',
    'condom',
    'condone',
    'condor',
    'conduct',
    'conductor',
    'conduit',
    'confer',
    'confess',
    'confessor',
    'confidant',
    'confide',
    'confine',
    'confirm',
    'conflate',
    'conflict',
    'conform',
    'confound',
    'confront',
    'confuse',
    'congeal',
    'congress',
    'conifer',
    'conjure',
    'conjurer',
    'conker',
    'connect',
    'connive',
    'connote',
    'conquer',
    'conqueror',
    'conquest',
    'conscript',
    'consent',
    'conserve',
    'consider',
    'consign',
    'consist',
    'console',
    'consonant',
    'consort',
    'conspire',
    'constable',
    'constant',
    'constrain',
    'constrict',
    'construct',
    'construe',
    'consul',
    'consulate',
    'consult',
    'consume',
    'consumer',
    'contact',
    'contagion',
    'contain',
    'container',
    'contend',
    'contender',
    'content',
    'contest',
    'context',
    'continent',
    'continue',
    'contort',
    'contour',
    'contract',
    'contralto',
    'contrast',
    'contrive',
    'control',
    'contusion',
    'conundrum',
    'convector',
    'convene',
    'convener',
    'convent',
    'converge',
    'converse',
    'convert',
    'converter',
    'converter',
    'convey',
    'convict',
    'convince',
    'convoy',
    'convulse',
    'cookbook',
    'cooker',
    'cookie',
    'coolant',
    'cooler',
    'coolie',
    'copier',
    'coping',
    'copper',
    'coppice',
    'copse',
    'copula',
    'copulate',
    'copycat',
    'copyright',
    'coquette',
    'coracle',
    'coral',
    'corbel',
    'cordial',
    'cordon',
    'corduroy',
    'corkscrew',
    'cormorant',
    'cornea',
    'corner',
    'cornet',
    'cornfield',
    'cornflake',
    'cornice',
    'corny',
    'corollary',
    'corona',
    'coronary',
    'coroner',
    'coronet',
    'corporal',
    'corpse',
    'corpus',
    'corpuscle',
    'corral',
    'correct',
    'correlate',
    'corridor',
    'corrode',
    'corrupt',
    'corsage',
    'corset',
    'cortege',
    'cortex',
    'cosine',
    'cosmetic',
    'cosmology',
    'cosmonaut',
    'cosset',
    'costing',
    'costly',
    'costume',
    'coterie',
    'cottage',
    'cottager',
    'cotton',
    'couch',
    'cougar',
    'cough',
    'council',
    'councilor',
    'counsel',
    'counselor',
    'count',
    'countdown',
    'counter',
    'countess',
    'country',
    'county',
    'coupe',
    'couple',
    'couplet',
    'coupon',
    'courgette',
    'courier',
    'course',
    'court',
    'courtesan',
    'courtesy',
    'courtier',
    'courtly',
    'courtroom',
    'courtyard',
    'cousin',
    'couturier',
    'covenant',
    'cover',
    'covering',
    'coverlet',
    'covert',
    'covet',
    'coward',
    'cowboy',
    'cower',
    'cowman',
    'cowrie',
    'cowshed',
    'cowslip',
    'coyote',
    'crack',
    'crackdown',
    'cracker',
    'crackle',
    'crackpot',
    'cradle',
    'craft',
    'craftsman',
    'crafty',
    'cramp',
    'cranberry',
    'crane',
    'cranefly',
    'cranium',
    'crank',
    'cranky',
    'cranny',
    'crappy',
    'crash',
    'crass',
    'crate',
    'crater',
    'cravat',
    'crave',
    'craving',
    'crawl',
    'crawler',
    'crayfish',
    'crayon',
    'craze',
    'crazy',
    'creak',
    'creaky',
    'cream',
    'creamy',
    'crease',
    'create',
    'creation',
    'creator',
    'creature',
    'credit',
    'creditor',
    'credo',
    'creed',
    'creek',
    'creel',
    'creep',
    'creeper',
    'creepy',
    'cremate',
    'cremation',
    'creole',
    'creosote',
    'crescendo',
    'crescent',
    'crest',
    'cretin',
    'crevasse',
    'crevice',
    'crewman',
    'crick',
    'cricket',
    'cricketer',
    'crier',
    'crime',
    'criminal',
    'crimp',
    'crimson',
    'cringe',
    'crinkle',
    'crinoline',
    'cripple',
    'crisis',
    'crisp',
    'crispy',
    'criterion',
    'critic',
    'criticism',
    'criticize',
    'critique',
    'croak',
    'crochet',
    'crock',
    'crocodile',
    'crocus',
    'croft',
    'crofter',
    'croissant',
    'crone',
    'crony',
    'crook',
    'croon',
    'crooner',
    'croquette',
    'cross',
    'crossbar',
    'crossbow',
    'crossing',
    'crosswind',
    'crossword',
    'crotch',
    'crotchet',
    'crouch',
    'croupier',
    'crouton',
    'crowbar',
    'crowd',
    'crown',
    'crucible',
    'crucifix',
    'crucify',
    'crude',
    'cruel',
    'cruelty',
    'cruet',
    'cruise',
    'cruiser',
    'crumb',
    'crumble',
    'crumbly',
    'crummy',
    'crumpet',
    'crumple',
    'crunch',
    'crunchy',
    'crusade',
    'crusader',
    'crush',
    'crust',
    'crusty',
    'crutch',
    'crypt',
    'crystal',
    'cubicle',
    'cuckold',
    'cuckoo',
    'cucumber',
    'cuddle',
    'cuddly',
    'cudgel',
    'cufflink',
    'culminate',
    'culprit',
    'cultivate',
    'culture',
    'culvert',
    'cumulus',
    'cupboard',
    'cupful',
    'cupid',
    'cupola',
    'curacy',
    'curate',
    'curator',
    'curdle',
    'curfew',
    'curio',
    'curiosity',
    'curler',
    'curlew',
    'curly',
    'currant',
    'currency',
    'current',
    'curry',
    'curse',
    'curtail',
    'curtain',
    'curtsy',
    'curve',
    'curvy',
    'cushion',
    'cushy',
    'custodian',
    'custom',
    'customer',
    'customize',
    'cutback',
    'cutlass',
    'cutlet',
    'cutter',
    'cutting',
    'cyclamen',
    'cycle',
    'cyclist',
    'cyclone',
    'cygnet',
    'cylinder',
    'cymbal',
    'cynic',
    'cypher',
    'cypress',
    'dabble',
    'dabbler',
    'dachshund',
    'daddy',
    'daemon',
    'daffodil',
    'dagger',
    'dahlia',
    'daily',
    'dainty',
    'daiquiri',
    'dairy',
    'dairymaid',
    'dairyman',
    'daisy',
    'dally',
    'damage',
    'dampen',
    'damper',
    'damsel',
    'damson',
    'dance',
    'dancer',
    'dandelion',
    'dandy',
    'danger',
    'dangle',
    'daredevil',
    'darken',
    'darkroom',
    'darling',
    'dartboard',
    'dashboard',
    'database',
    'daughter',
    'daunt',
    'dawdle',
    'daydream',
    'daylight',
    'dazzle',
    'deacon',
    'deaconess',
    'deadbeat',
    'deaden',
    'deadline',
    'deadly',
    'deafen',
    'dealer',
    'deanery',
    'deary',
    'death',
    'deathbed',
    'debacle',
    'debar',
    'debase',
    'debate',
    'debater',
    'debauch',
    'debit',
    'debrief',
    'debtor',
    'debug',
    'debunk',
    'debut',
    'debutante',
    'decade',
    'decamp',
    'decant',
    'decanter',
    'decathlon',
    'decay',
    'deceive',
    'decency',
    'deception',
    'decibel',
    'decide',
    'decimal',
    'decimate',
    'decipher',
    'decision',
    'deckchair',
    'deckhand',
    'declaim',
    'declare',
    'decline',
    'decode',
    'decompose',
    'decorate',
    'decorator',
    'decoy',
    'decrease',
    'decree',
    'decry',
    'dedicate',
    'deduce',
    'deduct',
    'deduction',
    'deepen',
    'deface',
    'defame',
    'default',
    'defaulter',
    'defeat',
    'defeatist',
    'defecate',
    'defect',
    'defector',
    'defend',
    'defendant',
    'defender',
    'defer',
    'deferment',
    'deferral',
    'deficit',
    'defile',
    'define',
    'deflate',
    'deflect',
    'deflower',
    'defoliant',
    'defoliate',
    'deforest',
    'deform',
    'deformity',
    'defraud',
    'defray',
    'defrost',
    'defuse',
    'degrade',
    'degree',
    'dehydrate',
    'deify',
    'deign',
    'deity',
    'delay',
    'delegate',
    'delete',
    'delicacy',
    'delight',
    'delimit',
    'delineate',
    'deliver',
    'delivery',
    'delouse',
    'delta',
    'delude',
    'deluge',
    'delusion',
    'delve',
    'demagogue',
    'demand',
    'demarcate',
    'demean',
    'demerit',
    'demijohn',
    'democracy',
    'democrat',
    'demolish',
    'demon',
    'demote',
    'demur',
    'demystify',
    'denial',
    'denigrate',
    'denim',
    'denizen',
    'denote',
    'denounce',
    'dense',
    'density',
    'dentist',
    'denude',
    'deodorant',
    'deodorize',
    'depart',
    'departure',
    'depend',
    'dependant',
    'depict',
    'depiction',
    'deplete',
    'deplore',
    'deploy',
    'deport',
    'depose',
    'deposit',
    'depot',
    'deprave',
    'deprecate',
    'deprecate',
    'depress',
    'depress',
    'deprive',
    'depth',
    'derby',
    'derelict',
    'deride',
    'derive',
    'derrick',
    'dervish',
    'descant',
    'descend',
    'descent',
    'describe',
    'desecrate',
    'desert',
    'deserter',
    'deserve',
    'design',
    'designate',
    'designer',
    'desire',
    'desist',
    'desolate',
    'despair',
    'desperado',
    'despise',
    'despoil',
    'despot',
    'dessert',
    'destiny',
    'destroy',
    'destroyer',
    'detach',
    'detail',
    'detain',
    'detainee',
    'detect',
    'detective',
    'detector',
    'detention',
    'deter',
    'detergent',
    'determine',
    'deterrent',
    'detest',
    'dethrone',
    'detonate',
    'detonator',
    'detract',
    'detractor',
    'deuce',
    'devalue',
    'devastate',
    'develop',
    'developer',
    'deviant',
    'deviate',
    'deviation',
    'device',
    'devil',
    'devise',
    'devolve',
    'devote',
    'devotee',
    'devotion',
    'devour',
    'dewlap',
    'dhoti',
    'diabetic',
    'diadem',
    'diagnose',
    'diagnosis',
    'diagonal',
    'diagram',
    'dialect',
    'dialectic',
    'dialog',
    'diameter',
    'diamond',
    'diaper',
    'diaphragm',
    'diarist',
    'diary',
    'diatribe',
    'dicey',
    'dichotomy',
    'dictate',
    'dictation',
    'dictator',
    'dictum',
    'diddle',
    'diehard',
    'diesel',
    'dietitian',
    'differ',
    'diffuse',
    'digest',
    'digestion',
    'digestive',
    'digger',
    'digit',
    'dignify',
    'dignitary',
    'digress',
    'dilate',
    'dilemma',
    'dilute',
    'dimension',
    'diminish',
    'dimmer',
    'dimple',
    'dimwit',
    'diner',
    'dinghy',
    'dingo',
    'dingy',
    'dinner',
    'dinosaur',
    'diocese',
    'diphthong',
    'diploma',
    'diplomat',
    'dipper',
    'dipstick',
    'direct',
    'direction',
    'directive',
    'director',
    'directory',
    'dirge',
    'dirndl',
    'dirty',
    'disable',
    'disabuse',
    'disagree',
    'disallow',
    'disappear',
    'disarm',
    'disarmer',
    'disaster',
    'disavow',
    'disband',
    'disburse',
    'discard',
    'discern',
    'discharge',
    'disciple',
    'disclaim',
    'disclose',
    'disco',
    'discolor',
    'discomfit',
    'discord',
    'discount',
    'discourse',
    'discover',
    'discovery',
    'discredit',
    'discus',
    'discuss',
    'disdain',
    'disease',
    'disembark',
    'disengage',
    'disfavor',
    'disfigure',
    'disgorge',
    'disgrace',
    'disguise',
    'disgust',
    'dishcloth',
    'dishonor',
    'disinfect',
    'disinter',
    'dislike',
    'dislocate',
    'dislodge',
    'dismantle',
    'dismay',
    'dismember',
    'dismiss',
    'dismissal',
    'dismount',
    'disobey',
    'disorder',
    'disorient',
    'disown',
    'disparage',
    'disparity',
    'dispatch',
    'dispel',
    'dispense',
    'dispenser',
    'disperse',
    'displace',
    'display',
    'displease',
    'disport',
    'dispose',
    'disprove',
    'dispute',
    'disquiet',
    'disregard',
    'disrobe',
    'disrupt',
    'dissect',
    'dissemble',
    'dissent',
    'dissenter',
    'dissident',
    'dissipate',
    'dissolve',
    'dissuade',
    'distance',
    'distend',
    'distill',
    'distiller',
    'distort',
    'distract',
    'distress',
    'district',
    'distrust',
    'disturb',
    'disunite',
    'ditch',
    'dither',
    'ditty',
    'divan',
    'diver',
    'diverge',
    'diversify',
    'diversion',
    'diversity',
    'divert',
    'divest',
    'divide',
    'dividend',
    'divider',
    'divine',
    'divinity',
    'division',
    'divorce',
    'divorcee',
    'divot',
    'divulge',
    'dizzy',
    'docker',
    'docket',
    'dockland',
    'dockyard',
    'doctor',
    'doctorate',
    'doctrine',
    'document',
    'dodder',
    'dodge',
    'dodgem',
    'dodger',
    'dogcart',
    'dogfight',
    'doggie',
    'doggy',
    'dogleg',
    'dogma',
    'dogmatize',
    'dogsbody',
    'doily',
    'dollar',
    'dollop',
    'dolly',
    'dolphin',
    'domain',
    'domestic',
    'domicile',
    'dominate',
    'dominion',
    'domino',
    'donate',
    'donation',
    'donkey',
    'donor',
    'doodle',
    'doorbell',
    'doorknob',
    'doorman',
    'doormat',
    'doorstep',
    'doorstop',
    'doorway',
    'dormer',
    'dormitory',
    'dormouse',
    'dosage',
    'dossier',
    'dotty',
    'double',
    'doublet',
    'doubt',
    'doubter',
    'dough',
    'doughnut',
    'doughty',
    'doughy',
    'douse',
    'dovecote',
    'dovetail',
    'dowager',
    'dowdy',
    'dowel',
    'downer',
    'downgrade',
    'downpour',
    'downturn',
    'downy',
    'dowry',
    'dowse',
    'doyen',
    'doyenne',
    'dozen',
    'drachma',
    'draft',
    'draftee',
    'dragon',
    'dragonfly',
    'dragoon',
    'drain',
    'drainpipe',
    'drake',
    'drama',
    'dramatic',
    'dramatist',
    'dramatize',
    'drape',
    'draper',
    'drapery',
    'draught',
    'drawback',
    'drawer',
    'drawing',
    'drawl',
    'dread',
    'dream',
    'dreamer',
    'dreamy',
    'dreary',
    'dredge',
    'dredger',
    'drench',
    'dress',
    'dresser',
    'dressing',
    'dressy',
    'dribble',
    'drier',
    'drift',
    'drifter',
    'drill',
    'drink',
    'drinker',
    'drive',
    'drivel',
    'driver',
    'driveway',
    'drizzle',
    'dromedary',
    'drone',
    'drool',
    'droop',
    'droopy',
    'droplet',
    'dropper',
    'drought',
    'drove',
    'drown',
    'drowse',
    'drowsy',
    'drudge',
    'druggist',
    'drugstore',
    'druid',
    'drumbeat',
    'drummer',
    'drumstick',
    'drunk',
    'drunkard',
    'dryer',
    'duchess',
    'duchy',
    'ducking',
    'duckling',
    'duffel',
    'duffer',
    'dugout',
    'dukedom',
    'dullard',
    'dumbbell',
    'dummy',
    'dumpling',
    'dumpy',
    'dunce',
    'dungeon',
    'duodenum',
    'duplex',
    'duplicate',
    'dusky',
    'dustbin',
    'dustcart',
    'duster',
    'dustman',
    'dustpan',
    'dusty',
    'duvet',
    'dwarf',
    'dwell',
    'dweller',
    'dwelling',
    'dwindle',
    'dynamic',
    'dynamite',
    'dynamo',
    'dynasty',
    'eagle',
    'eardrum',
    'earldom',
    'earlobe',
    'early',
    'earmark',
    'earner',
    'earphone',
    'earpiece',
    'earplug',
    'earring',
    'earth',
    'earthwork',
    'earthworm',
    'earwig',
    'easel',
    'easterner',
    'eastward',
    'eater',
    'eavesdrop',
    'ebony',
    'echelon',
    'eclair',
    'eclipse',
    'ecologist',
    'economist',
    'economize',
    'economy',
    'ecosystem',
    'ecstasy',
    'edging',
    'edict',
    'edifice',
    'edify',
    'edition',
    'editor',
    'editorial',
    'educate',
    'educator',
    'eerie',
    'efface',
    'effect',
    'effigy',
    'effluent',
    'effort',
    'effusion',
    'eggcup',
    'egghead',
    'eggplant',
    'eggshell',
    'egoist',
    'egomaniac',
    'egotist',
    'eiderdown',
    'eight',
    'eighth',
    'eighty',
    'ejaculate',
    'eject',
    'elaborate',
    'elapse',
    'elbow',
    'elder',
    'elect',
    'election',
    'elector',
    'electrify',
    'electrode',
    'electron',
    'elegy',
    'element',
    'elephant',
    'elevate',
    'elevation',
    'elevator',
    'eleven',
    'eleventh',
    'elicit',
    'elide',
    'eliminate',
    'elite',
    'elitist',
    'elixir',
    'ellipse',
    'ellipsis',
    'elongate',
    'elope',
    'elopement',
    'elucidate',
    'elude',
    'emanate',
    'embalm',
    'embargo',
    'embark',
    'embarrass',
    'embassy',
    'embed',
    'embellish',
    'ember',
    'embezzle',
    'embitter',
    'emblazon',
    'emblem',
    'embody',
    'embolden',
    'embolism',
    'embrace',
    'embrasure',
    'embroider',
    'embroil',
    'embryo',
    'emend',
    'emerald',
    'emerge',
    'emergency',
    'emetic',
    'emigrant',
    'emigrate',
    'emirate',
    'emissary',
    'emission',
    'emollient',
    'emolument',
    'emotion',
    'empathize',
    'emperor',
    'emphasis',
    'emphasize',
    'empire',
    'employ',
    'employee',
    'employer',
    'emporium',
    'empower',
    'empress',
    'empty',
    'emulate',
    'emulsify',
    'emulsion',
    'enable',
    'enact',
    'enactment',
    'enamel',
    'encamp',
    'encase',
    'enchant',
    'enchanter',
    'encircle',
    'enclave',
    'enclose',
    'enclosure',
    'encode',
    'encompass',
    'encore',
    'encounter',
    'encourage',
    'encroach',
    'encumber',
    'endanger',
    'endear',
    'endeavor',
    'ending',
    'endive',
    'endorse',
    'endow',
    'endowment',
    'endure',
    'enema',
    'enemy',
    'energize',
    'energy',
    'enfold',
    'enforce',
    'engage',
    'engender',
    'engine',
    'engineer',
    'engrave',
    'engraver',
    'engraving',
    'engulf',
    'enhance',
    'enigma',
    'enjoin',
    'enjoy',
    'enjoyment',
    'enlarge',
    'enlighten',
    'enlist',
    'enliven',
    'enmity',
    'ennoble',
    'enormity',
    'enquire',
    'enquirer',
    'inquiry',
    'enrage',
    'enrich',
    'enroll',
    'ensconce',
    'ensemble',
    'enshrine',
    'enshroud',
    'ensign',
    'enslave',
    'ensnare',
    'ensue',
    'ensure',
    'entail',
    'entangle',
    'entente',
    'enter',
    'entertain',
    'enthrall',
    'enthrone',
    'enthuse',
    'entice',
    'entitle',
    'entity',
    'entomb',
    'entourage',
    'entrance',
    'entrant',
    'entrap',
    'entreat',
    'entreaty',
    'entree',
    'entrench',
    'entrust',
    'entry',
    'entwine',
    'enumerate',
    'enunciate',
    'envelop',
    'envelope',
    'envisage',
    'envision',
    'envoy',
    'enzyme',
    'epaulet',
    'epicenter',
    'epicure',
    'epidemic',
    'epigram',
    'epileptic',
    'epilogue',
    'episode',
    'epistle',
    'epitaph',
    'epithet',
    'epitomize',
    'epoch',
    'equal',
    'equalize',
    'equalizer',
    'equate',
    'equation',
    'equerry',
    'equinox',
    'equip',
    'eradicate',
    'erase',
    'eraser',
    'erasure',
    'erect',
    'erection',
    'erode',
    'errand',
    'error',
    'erupt',
    'escalate',
    'escalator',
    'escapade',
    'escape',
    'escapee',
    'escapist',
    'eschew',
    'escort',
    'esplanade',
    'espouse',
    'espresso',
    'essay',
    'essayist',
    'essence',
    'essential',
    'establish',
    'estate',
    'esteem',
    'estimate',
    'estuary',
    'etching',
    'eternity',
    'ethic',
    'etymology',
    'eulogize',
    'eulogy',
    'eunuch',
    'euphemism',
    'evacuate',
    'evacuee',
    'evade',
    'evaluate',
    'evaporate',
    'evasion',
    'evening',
    'event',
    'evergreen',
    'evict',
    'eviction',
    'evidence',
    'evildoer',
    'evince',
    'evocation',
    'evoke',
    'evolve',
    'exact',
    'exalt',
    'examine',
    'examiner',
    'example',
    'excavate',
    'excavator',
    'exceed',
    'excel',
    'except',
    'exception',
    'excerpt',
    'excess',
    'exchange',
    'excise',
    'excite',
    'exclaim',
    'exclude',
    'exclusive',
    'excrete',
    'exculpate',
    'excursion',
    'excuse',
    'execute',
    'executive',
    'executor',
    'exegesis',
    'exemplar',
    'exemplify',
    'exempt',
    'exercise',
    'exert',
    'exertion',
    'exhale',
    'exhaust',
    'exhibit',
    'exhibitor',
    'exhort',
    'exhume',
    'exigency',
    'exile',
    'exist',
    'existence',
    'exonerate',
    'exorcism',
    'exorcise',
    'expand',
    'expanse',
    'expatiate',
    'expect',
    'expedient',
    'expedite',
    'expel',
    'expend',
    'expense',
    'expert',
    'expiate',
    'expire',
    'explain',
    'expletive',
    'explicate',
    'explode',
    'exploit',
    'exploiter',
    'explore',
    'explorer',
    'explosion',
    'explosive',
    'exponent',
    'export',
    'exporter',
    'expose',
    'expose',
    'exposure',
    'expound',
    'express',
    'expulsion',
    'expunge',
    'expurgate',
    'extend',
    'extension',
    'extenuate',
    'exterior',
    'external',
    'extol',
    'extort',
    'extra',
    'extract',
    'extractor',
    'extradite',
    'extreme',
    'extremist',
    'extremity',
    'extricate',
    'extrovert',
    'extrude',
    'extrusion',
    'exude',
    'exult',
    'eyeball',
    'eyebrow',
    'eyeful',
    'eyelash',
    'eyelet',
    'eyelid',
    'eyepiece',
    'eyesore',
    'eyewash',
    'eyrie',
    'fable',
    'fabric',
    'fabricate',
    'facade',
    'facet',
    'facial',
    'facility',
    'facing',
    'facsimile',
    'faction',
    'factor',
    'factory',
    'factotum',
    'faculty',
    'failing',
    'failure',
    'faint',
    'fairway',
    'fairy',
    'fairyland',
    'faith',
    'faithful',
    'falcon',
    'falconer',
    'fallacy',
    'falsehood',
    'falsetto',
    'falsify',
    'falsity',
    'falter',
    'familiar',
    'family',
    'famine',
    'fanatic',
    'fancier',
    'fancy',
    'fandango',
    'fanfare',
    'fanlight',
    'fanny',
    'fantasize',
    'fantasy',
    'farce',
    'farewell',
    'farmer',
    'farmhouse',
    'farmland',
    'farmyard',
    'farrier',
    'farrow',
    'farthing',
    'fascia',
    'fascinate',
    'fascist',
    'fashion',
    'fasten',
    'fastener',
    'fastening',
    'fastness',
    'fatality',
    'fathead',
    'father',
    'fathom',
    'fatigue',
    'fatten',
    'fatty',
    'faucet',
    'fault',
    'favor',
    'favorite',
    'fealty',
    'feast',
    'feather',
    'feature',
    'federate',
    'feeble',
    'feeder',
    'feeler',
    'feeling',
    'feign',
    'feint',
    'felicity',
    'feline',
    'fellow',
    'felon',
    'felony',
    'female',
    'feminist',
    'femur',
    'fence',
    'fender',
    'ferment',
    'ferret',
    'ferrule',
    'ferry',
    'ferryboat',
    'fertilize',
    'fester',
    'festival',
    'festivity',
    'festoon',
    'fetch',
    'fetish',
    'fetlock',
    'fetter',
    'fever',
    'fiance',
    'fiancee',
    'fiasco',
    'fibber',
    'fiber',
    'fibula',
    'fiction',
    'fiddle',
    'fiddler',
    'fiddly',
    'fidget',
    'field',
    'fielder',
    'fiend',
    'fierce',
    'fiery',
    'fiesta',
    'fifteen',
    'fifth',
    'fifty',
    'fight',
    'fighter',
    'figment',
    'figure',
    'figurine',
    'filament',
    'filch',
    'filler',
    'fillet',
    'filling',
    'fillip',
    'filly',
    'filmy',
    'filter',
    'filthy',
    'final',
    'finale',
    'finalist',
    'finalize',
    'finance',
    'financier',
    'finch',
    'finder',
    'finding',
    'finger',
    'fingertip',
    'finish',
    'firearm',
    'fireball',
    'firebomb',
    'firebrand',
    'firebreak',
    'firebrick',
    'firefly',
    'fireguard',
    'fireman',
    'fireplace',
    'fireplug',
    'fireproof',
    'fireside',
    'firework',
    'first',
    'fishcake',
    'fisherman',
    'fishery',
    'fishwife',
    'fishy',
    'fissure',
    'fistful',
    'figment',
    'fitter',
    'fitting',
    'fiver',
    'fixation',
    'fixative',
    'fixture',
    'fizzle',
    'fizzy',
    'fjord',
    'flabby',
    'flagon',
    'flagpole',
    'flagship',
    'flagstaff',
    'flagstone',
    'flail',
    'flake',
    'flambe',
    'flame',
    'flamenco',
    'flamingo',
    'flange',
    'flank',
    'flannel',
    'flapjack',
    'flare',
    'flash',
    'flashback',
    'flashbulb',
    'flashcard',
    'flashcube',
    'flasher',
    'flashgun',
    'flashy',
    'flask',
    'flatfish',
    'flatiron',
    'flatmate',
    'flatten',
    'flatter',
    'flatterer',
    'flaunt',
    'flavor',
    'flavoring',
    'fleck',
    'fledgling',
    'fleece',
    'fleet',
    'flesh',
    'fleshpot',
    'fleshy',
    'flick',
    'flicker',
    'flight',
    'flighty',
    'flimsy',
    'flinch',
    'fling',
    'flint',
    'flintlock',
    'flipper',
    'flirt',
    'float',
    'flock',
    'flood',
    'floor',
    'floozy',
    'floppy',
    'florin',
    'florist',
    'flotation',
    'flotilla',
    'flounce',
    'flounder',
    'flour',
    'flourish',
    'flout',
    'flower',
    'flowerbed',
    'flowering',
    'flowerpot',
    'fluctuate',
    'fluff',
    'fluffy',
    'fluid',
    'fluke',
    'flummox',
    'flunk',
    'flurry',
    'flush',
    'fluster',
    'flute',
    'flutist',
    'flutter',
    'flyby',
    'flyer',
    'flyover',
    'flywheel',
    'foamy',
    'focus',
    'fetus',
    'foggy',
    'foghorn',
    'foible',
    'foist',
    'folder',
    'folio',
    'follicle',
    'follow',
    'follower',
    'following',
    'folly',
    'foment',
    'fondant',
    'fondle',
    'fondue',
    'foodstuff',
    'football',
    'footfall',
    'foothill',
    'foothold',
    'footman',
    'footnote',
    'footpath',
    'footplate',
    'footprint',
    'footstep',
    'footstool',
    'forage',
    'foray',
    'forbear',
    'forbid',
    'force',
    'forearm',
    'forebear',
    'forecast',
    'foreclose',
    'forecourt',
    'forefoot',
    'forgo',
    'forehand',
    'forehead',
    'foreigner',
    'foreknow',
    'foreleg',
    'forelock',
    'foreman',
    'forename',
    'forepart',
    'foresee',
    'foreshore',
    'foreskin',
    'forest',
    'forestall',
    'forester',
    'foretaste',
    'foretell',
    'forewarn',
    'foreword',
    'forfeit',
    'forge',
    'forger',
    'forgery',
    'forget',
    'forgive',
    'forgo',
    'formality',
    'formalize',
    'format',
    'formation',
    'formula',
    'formulate',
    'fornicate',
    'forsake',
    'forswear',
    'forte',
    'fortify',
    'fortnight',
    'fortress',
    'fortune',
    'forty',
    'forum',
    'forward',
    'fossil',
    'fossilize',
    'foster',
    'found',
    'founder',
    'foundling',
    'foundry',
    'fountain',
    'foursome',
    'fourth',
    'foxglove',
    'foxhole',
    'foxhound',
    'foyer',
    'fraction',
    'fracture',
    'fragment',
    'fragrance',
    'frail',
    'frailty',
    'frame',
    'framework',
    'franc',
    'franchise',
    'frank',
    'fraud',
    'freak',
    'freaky',
    'freckle',
    'freebie',
    'freedom',
    'freehold',
    'freelance',
    'freeman',
    'freeway',
    'freewheel',
    'freeze',
    'freezer',
    'freight',
    'freighter',
    'frenzy',
    'frequency',
    'frequent',
    'fresco',
    'fresh',
    'freshen',
    'fresher',
    'freshman',
    'friar',
    'fricassee',
    'friction',
    'fridge',
    'friend',
    'friendly',
    'frieze',
    'frigate',
    'fright',
    'frighten',
    'frill',
    'fringe',
    'frippery',
    'frisk',
    'frisky',
    'frisson',
    'fritter',
    'frivolity',
    'frizz',
    'frizzle',
    'frizzy',
    'frock',
    'frogging',
    'frogman',
    'frolic',
    'frond',
    'front',
    'frontage',
    'frontier',
    'frost',
    'frosty',
    'froth',
    'frothy',
    'frown',
    'fruit',
    'fruitcake',
    'fruity',
    'frump',
    'frustrate',
    'fuchsia',
    'fudge',
    'fugitive',
    'fugue',
    'fulcrum',
    'fulfill',
    'fulminate',
    'fumble',
    'fumigate',
    'function',
    'funeral',
    'funfair',
    'fungicide',
    'fungus',
    'funky',
    'funnel',
    'funny',
    'furbish',
    'furlong',
    'furnace',
    'furnish',
    'furrier',
    'furrow',
    'furry',
    'further',
    'fuselage',
    'fusion',
    'fusspot',
    'fussy',
    'fusty',
    'future',
    'fuzzy',
    'fuzzy',
    'gabardine',
    'gabble',
    'gable',
    'gadget',
    'gaffe',
    'gaffer',
    'gaggle',
    'gainsay',
    'gaiter',
    'galaxy',
    'galleon',
    'gallery',
    'galley',
    'gallivant',
    'gallon',
    'gallop',
    'gallstone',
    'galvanize',
    'gambit',
    'gamble',
    'gambler',
    'gambol',
    'gamma',
    'gander',
    'gangplank',
    'gangster',
    'gangway',
    'gantry',
    'gaoler',
    'garage',
    'garden',
    'gardener',
    'gardenia',
    'gargle',
    'gargoyle',
    'garland',
    'garment',
    'garner',
    'garnet',
    'garnish',
    'garret',
    'garrison',
    'garrote',
    'garter',
    'gasbag',
    'gasket',
    'gasometer',
    'gassy',
    'gateau',
    'gatecrash',
    'gatehouse',
    'gatepost',
    'gateway',
    'gather',
    'gathering',
    'gaudy',
    'gauge',
    'gauntlet',
    'gavel',
    'gawky',
    'gazelle',
    'gearbox',
    'gearshift',
    'geezer',
    'geisha',
    'gelatin',
    'gelding',
    'gemstone',
    'gendarme',
    'gender',
    'genealogy',
    'general',
    'generate',
    'generator',
    'genetic',
    'genie',
    'genital',
    'genius',
    'genre',
    'gentian',
    'gentile',
    'gentle',
    'gentleman',
    'genuflect',
    'genus',
    'geranium',
    'gerbil',
    'geriatric',
    'germinate',
    'gerund',
    'gesture',
    'getaway',
    'gewgaw',
    'geyser',
    'gherkin',
    'ghetto',
    'ghost',
    'ghoul',
    'giant',
    'gibber',
    'gibbet',
    'gibbon',
    'giddy',
    'giggle',
    'gigolo',
    'gimlet',
    'gimmick',
    'giraffe',
    'girder',
    'girdle',
    'girth',
    'glacier',
    'gladden',
    'glade',
    'gladiator',
    'glamorize',
    'glance',
    'gland',
    'glare',
    'glass',
    'glaze',
    'gleam',
    'glean',
    'glide',
    'glider',
    'glimmer',
    'glimpse',
    'glint',
    'glisten',
    'glitter',
    'gloat',
    'globe',
    'globule',
    'gloomy',
    'glorify',
    'glory',
    'gloss',
    'glossary',
    'glove',
    'glower',
    'glutton',
    'gnash',
    'gnome',
    'goalie',
    'goalpost',
    'goatee',
    'goatherd',
    'goatskin',
    'gobbet',
    'gobble',
    'gobbler',
    'goblet',
    'goblin',
    'godchild',
    'goddess',
    'godfather',
    'godmother',
    'godparent',
    'godson',
    'goggle',
    'goiter',
    'goldmine',
    'goldsmith',
    'golfer',
    'golly',
    'gondola',
    'gondolier',
    'goner',
    'goodbye',
    'goodnight',
    'goody',
    'gooey',
    'goofy',
    'goose',
    'gorge',
    'gorilla',
    'gosling',
    'gospel',
    'gossip',
    'gouge',
    'gourd',
    'gourmand',
    'gourmet',
    'govern',
    'governess',
    'governor',
    'grace',
    'gradation',
    'grade',
    'gradient',
    'graduate',
    'graft',
    'grain',
    'grammar',
    'granary',
    'grand',
    'grandad',
    'grandaddy',
    'grandee',
    'grandma',
    'grandpa',
    'grandson',
    'grange',
    'granny',
    'grant',
    'granule',
    'grape',
    'grapevine',
    'graph',
    'graphic',
    'grapnel',
    'grapple',
    'grasp',
    'grass',
    'grassland',
    'grassy',
    'grate',
    'grater',
    'gratify',
    'grating',
    'gratuity',
    'grave',
    'graveyard',
    'gravitate',
    'gravity',
    'gravy',
    'graze',
    'grease',
    'greasy',
    'great',
    'greatcoat',
    'greedy',
    'green',
    'greenback',
    'greenfly',
    'greengage',
    'greenhorn',
    'greenroom',
    'greet',
    'greeting',
    'gremlin',
    'grenade',
    'greyhound',
    'griddle',
    'grief',
    'grievance',
    'grieve',
    'grill',
    'grille',
    'grimace',
    'grind',
    'grinder',
    'gripe',
    'gritty',
    'grizzle',
    'grizzly',
    'groan',
    'grocer',
    'grocery',
    'groggy',
    'groin',
    'groom',
    'groove',
    'grope',
    'gross',
    'grotto',
    'grouch',
    'ground',
    'groundnut',
    'group',
    'groupie',
    'grouping',
    'grouse',
    'grove',
    'grovel',
    'grower',
    'growl',
    'growth',
    'grubby',
    'grudge',
    'grumble',
    'grumpy',
    'grunt',
    'guarantee',
    'guarantor',
    'guard',
    'guardian',
    'guardsman',
    'guava',
    'guerrilla',
    'guess',
    'guest',
    'guffaw',
    'guide',
    'guidebook',
    'guideline',
    'guild',
    'guilder',
    'guildhall',
    'guilty',
    'guinea',
    'guise',
    'guitar',
    'guitarist',
    'gulch',
    'gullet',
    'gully',
    'gumboot',
    'gumdrop',
    'gummy',
    'gunboat',
    'gunman',
    'gunner',
    'gunshot',
    'gunsmith',
    'guppy',
    'gurgle',
    'gusset',
    'gutter',
    'guzzle',
    'gymkhana',
    'gymnasium',
    'gymnast',
    'gymnastic',
    'gypsy',
    'gyrate',
    'gyroscope',
    'habit',
    'habitat',
    'habituate',
    'hacksaw',
    'haggis',
    'haggle',
    'hailstone',
    'hailstorm',
    'hairbrush',
    'haircut',
    'hairdo',
    'hairdryer',
    'hairline',
    'hairnet',
    'hairpiece',
    'hairpin',
    'hairstyle',
    'hairy',
    'halfpenny',
    'halftone',
    'halibut',
    'hallmark',
    'hallway',
    'halter',
    'halve',
    'hamburger',
    'hamlet',
    'hammer',
    'hammock',
    'hamper',
    'hamster',
    'hamstring',
    'handbag',
    'handbill',
    'handbook',
    'handbrake',
    'handcart',
    'handcuff',
    'handful',
    'handgun',
    'handicap',
    'handle',
    'handlebar',
    'handler',
    'handout',
    'handrail',
    'handshake',
    'handstand',
    'handy',
    'handyman',
    'hangar',
    'hanger',
    'hanging',
    'hangman',
    'hangout',
    'hangover',
    'hanker',
    'hankering',
    'hanky',
    'hansom',
    'happen',
    'happening',
    'happy',
    'harangue',
    'harass',
    'harbinger',
    'harbor',
    'hardback',
    'harden',
    'hardliner',
    'hardship',
    'hardwood',
    'hardy',
    'harelip',
    'harem',
    'haircut',
    'harlequin',
    'harlot',
    'harmonic',
    'harmonica',
    'harmonize',
    'harmony',
    'harness',
    'harpist',
    'harpoon',
    'harpy',
    'harrow',
    'harry',
    'harsh',
    'harvest',
    'harvester',
    'hassle',
    'hassock',
    'hasten',
    'hasty',
    'hatch',
    'hatchback',
    'hatchery',
    'hatchet',
    'hatchway',
    'haughty',
    'haunch',
    'haunt',
    'haven',
    'haversack',
    'hawker',
    'hawser',
    'hawthorn',
    'haycock',
    'haystack',
    'hazard',
    'hazel',
    'hazelnut',
    'headache',
    'headband',
    'headboard',
    'headdress',
    'header',
    'heading',
    'headlamp',
    'headland',
    'headlight',
    'headline',
    'headman',
    'headphone',
    'headrest',
    'headscarf',
    'headship',
    'headstand',
    'headstone',
    'headwind',
    'headword',
    'heady',
    'healer',
    'healthy',
    'hearer',
    'hearing',
    'hearse',
    'heart',
    'heartache',
    'heartbeat',
    'hearten',
    'hearth',
    'hearthrug',
    'heartland',
    'hearty',
    'heater',
    'heath',
    'heathen',
    'heatwave',
    'heave',
    'heaven',
    'heavy',
    'heckle',
    'heckler',
    'hectare',
    'hector',
    'hedge',
    'hedgehog',
    'hedgerow',
    'hefty',
    'heifer',
    'height',
    'heighten',
    'heiress',
    'heirloom',
    'heliport',
    'hello',
    'helmet',
    'helmsman',
    'helper',
    'helping',
    'hemline',
    'henchman',
    'henna',
    'heptagon',
    'herald',
    'herbalist',
    'herbicide',
    'herbivore',
    'herdsman',
    'heresy',
    'heretic',
    'hermit',
    'hernia',
    'heroic',
    'heroine',
    'heron',
    'herring',
    'hesitate',
    'hexagon',
    'hiatus',
    'hibernate',
    'hibiscus',
    'hiccup',
    'hideaway',
    'hideout',
    'hiding',
    'hierarchy',
    'highbrow',
    'highlight',
    'highroad',
    'highway',
    'hijack',
    'hijacker',
    'hiker',
    'hillbilly',
    'hillock',
    'hillside',
    'hilltop',
    'hilly',
    'hinder',
    'hindrance',
    'hinge',
    'hippie',
    'hippo',
    'hireling',
    'historian',
    'history',
    'hitch',
    'hoard',
    'hoarding',
    'hoarse',
    'hobble',
    'hobby',
    'hobnob',
    'hoist',
    'holdall',
    'holder',
    'holding',
    'holiday',
    'holler',
    'hollow',
    'holocaust',
    'hologram',
    'holograph',
    'holster',
    'homeland',
    'homeopath',
    'homeowner',
    'homestead',
    'homicide',
    'homily',
    'honey',
    'honeybee',
    'honeycomb',
    'honeymoon',
    'honorific',
    'honor',
    'hoodlum',
    'hoodwink',
    'hookah',
    'hooker',
    'hooligan',
    'hooter',
    'hopeful',
    'hopper',
    'horde',
    'horizon',
    'hormone',
    'hornet',
    'horny',
    'horoscope',
    'horrify',
    'horror',
    'horse',
    'horsefly',
    'horseman',
    'horseshoe',
    'horsewhip',
    'horsey',
    'hospice',
    'hospital',
    'hostage',
    'hostel',
    'hostess',
    'hostility',
    'hotbed',
    'hotel',
    'hotelier',
    'hothead',
    'hothouse',
    'hotpot',
    'hound',
    'hourglass',
    'house',
    'houseboat',
    'houseboy',
    'housecoat',
    'houseful',
    'household',
    'housemaid',
    'housetop',
    'housewife',
    'housing',
    'hovel',
    'hover',
    'howdah',
    'howitzer',
    'howler',
    'hubby',
    'hubcap',
    'huckster',
    'huddle',
    'huffy',
    'human',
    'humanity',
    'humanize',
    'humanoid',
    'humble',
    'humbug',
    'humdinger',
    'humerus',
    'humidify',
    'humiliate',
    'humorist',
    'humor',
    'humpback',
    'hunch',
    'hunchback',
    'hundred',
    'hundredth',
    'hunger',
    'hungry',
    'hunter',
    'huntsman',
    'hurdle',
    'hurricane',
    'hurry',
    'hurtle',
    'husband',
    'husky',
    'hussy',
    'hustle',
    'hustler',
    'hutch',
    'hyacinth',
    'hyena',
    'hybrid',
    'hydrant',
    'hydrate',
    'hydraulic',
    'hydrofoil',
    'hyena',
    'hymen',
    'hymnal',
    'hyperbola',
    'hyphen',
    'hypnotize',
    'hypocrisy',
    'hypocrite',
    'iceberg',
    'icicle',
    'ideal',
    'idealize',
    'identify',
    'identity',
    'ideology',
    'idiocy',
    'idiom',
    'idiot',
    'idler',
    'idolize',
    'idyll',
    'igloo',
    'ignite',
    'ignition',
    'ignoramus',
    'ignore',
    'illness',
    'illumine',
    'illusion',
    'image',
    'imagine',
    'imbalance',
    'imbecile',
    'imbibe',
    'imbroglio',
    'imbue',
    'imitate',
    'imitation',
    'imitator',
    'immerse',
    'immigrant',
    'immortal',
    'immunize',
    'impact',
    'impair',
    'impale',
    'impart',
    'impeach',
    'impede',
    'impel',
    'imperil',
    'impinge',
    'implant',
    'implement',
    'implicate',
    'implore',
    'imply',
    'import',
    'importer',
    'importune',
    'impose',
    'impostor',
    'impound',
    'impress',
    'imprint',
    'imprison',
    'improve',
    'improvise',
    'impugn',
    'impulse',
    'impulsion',
    'impurity',
    'impute',
    'incense',
    'incentive',
    'incident',
    'incise',
    'incision',
    'incisor',
    'incite',
    'incline',
    'include',
    'income',
    'increase',
    'increment',
    'incubate',
    'incubator',
    'inculcate',
    'incumbent',
    'incur',
    'incursion',
    'indemnity',
    'indent',
    'index',
    'indicate',
    'indicator',
    'indict',
    'indignity',
    'indoor',
    'induce',
    'induct',
    'induction',
    'indulge',
    'industry',
    'indwell',
    'inebriate',
    'inequity',
    'infant',
    'infect',
    'infection',
    'infer',
    'inference',
    'inferior',
    'inferno',
    'infest',
    'infidel',
    'infirmary',
    'inflame',
    'inflate',
    'inflect',
    'inflexion',
    'inflict',
    'influence',
    'inform',
    'informant',
    'informer',
    'infringe',
    'infuriate',
    'infuse',
    'infusion',
    'ingot',
    'inhabit',
    'inhale',
    'inherit',
    'inheritor',
    'inhibit',
    'iniquity',
    'initial',
    'initiate',
    'inject',
    'injection',
    'injure',
    'injury',
    'injustice',
    'inkling',
    'inkstand',
    'inkwell',
    'inlay',
    'inlet',
    'inmate',
    'inning',
    'innkeeper',
    'innocent',
    'innovate',
    'innovator',
    'innuendo',
    'inoculate',
    'input',
    'inquest',
    'inquire',
    'inquirer',
    'inquiry',
    'inscribe',
    'insect',
    'insert',
    'inset',
    'inside',
    'insider',
    'insight',
    'insinuate',
    'insist',
    'insole',
    'insomniac',
    'inspect',
    'inspector',
    'inspire',
    'install',
    'instance',
    'instant',
    'instep',
    'instigate',
    'instill',
    'instinct',
    'institute',
    'instruct',
    'insulate',
    'insulator',
    'insult',
    'insurance',
    'insure',
    'insurer',
    'insurgent',
    'intake',
    'integer',
    'integrate',
    'intellect',
    'intend',
    'intensify',
    'intent',
    'intention',
    'inter',
    'interact',
    'intercede',
    'intercept',
    'intercom',
    'interest',
    'interface',
    'interfere',
    'interior',
    'interject',
    'interlink',
    'interlock',
    'interlude',
    'intern',
    'internee',
    'interpose',
    'interpret',
    'interrupt',
    'intersect',
    'interval',
    'intervene',
    'interview',
    'intestine',
    'intimacy',
    'intimate',
    'intone',
    'intricacy',
    'intrigue',
    'introduce',
    'introvert',
    'intrude',
    'intruder',
    'intrusion',
    'intuit',
    'intuition',
    'inundate',
    'inure',
    'invade',
    'invader',
    'invalid',
    'invasion',
    'inveigh',
    'inveigle',
    'invent',
    'invention',
    'inventor',
    'inventory',
    'inversion',
    'invert',
    'invest',
    'investor',
    'invite',
    'invoice',
    'invoke',
    'involve',
    'irony',
    'irradiate',
    'irregular',
    'irrigate',
    'irritant',
    'irritate',
    'island',
    'islander',
    'islet',
    'isolate',
    'isometric',
    'isotope',
    'issue',
    'isthmus',
    'itemize',
    'itinerant',
    'itinerary',
    'ivory',
    'jabber',
    'jackal',
    'jackass',
    'jackboot',
    'jackdaw',
    'jacket',
    'jackpot',
    'gaffe',
    'jaguar',
    'jailbird',
    'jailbreak',
    'jailer',
    'jalopy',
    'jamboree',
    'jangle',
    'janitor',
    'jaunt',
    'javelin',
    'jawbone',
    'jaywalker',
    'jazzy',
    'jealousy',
    'jelly',
    'jemmy',
    'jerkin',
    'jerky',
    'jersey',
    'jester',
    'jettison',
    'jetty',
    'jewel',
    'jeweler',
    'jiggle',
    'jigsaw',
    'jimmy',
    'jingle',
    'jockey',
    'jockstrap',
    'jogger',
    'joggle',
    'joiner',
    'joint',
    'joist',
    'joker',
    'jolly',
    'jostle',
    'jotter',
    'jotting',
    'joule',
    'journal',
    'journey',
    'joust',
    'joyride',
    'joystick',
    'jubilee',
    'judge',
    'judgment',
    'juggle',
    'juggler',
    'jugular',
    'juice',
    'juicy',
    'jukebox',
    'jumble',
    'jumbo',
    'jumper',
    'jumpy',
    'junction',
    'juncture',
    'jungle',
    'junior',
    'juniper',
    'junket',
    'junkie',
    'junta',
    'jurist',
    'juror',
    'justice',
    'justify',
    'juvenile',
    'juxtapose',
    'kangaroo',
    'kayak',
    'kebab',
    'keeper',
    'keepsake',
    'kennel',
    'kerchief',
    'kerfuffle',
    'kernel',
    'kestrel',
    'ketch',
    'kettle',
    'keyboard',
    'keyhole',
    'keynote',
    'keyring',
    'keystone',
    'kibbutz',
    'kickback',
    'kiddie',
    'kidnap',
    'kidnapper',
    'kidney',
    'killer',
    'killing',
    'killjoy',
    'kilogram',
    'kilometer',
    'kilowatt',
    'kimono',
    'kindle',
    'kindness',
    'kinetic',
    'kingdom',
    'kinky',
    'kinsman',
    'kinswoman',
    'kiosk',
    'kipper',
    'kitchen',
    'kitten',
    'kitty',
    'klaxon',
    'knapsack',
    'knave',
    'knead',
    'kneecap',
    'kneel',
    'knife',
    'knight',
    'knitter',
    'knock',
    'knocker',
    'knockout',
    'knoll',
    'knotty',
    'knuckle',
    'koala',
    'kraal',
    'label',
    'labor',
    'laborer',
    'laburnum',
    'labyrinth',
    'lacerate',
    'lackey',
    'lacquer',
    'ladder',
    'laddie',
    'ladle',
    'ladybird',
    'ladyship',
    'lager',
    'laggard',
    'lagoon',
    'laird',
    'lambskin',
    'lament',
    'lampoon',
    'lampshade',
    'lance',
    'lancet',
    'landau',
    'landfall',
    'landing',
    'landlady',
    'landlord',
    'landmark',
    'landmine',
    'landowner',
    'landscape',
    'landslide',
    'landslip',
    'language',
    'languish',
    'lanky',
    'lantern',
    'lanyard',
    'lapel',
    'lapse',
    'lapwing',
    'larch',
    'larder',
    'large',
    'larva',
    'larynx',
    'laser',
    'lashing',
    'lassie',
    'lasso',
    'latch',
    'latchkey',
    'latecomer',
    'lathe',
    'lather',
    'latitude',
    'latrine',
    'lattice',
    'laugh',
    'launch',
    'launder',
    'laundry',
    'laurel',
    'lavatory',
    'lavish',
    'lawnmower',
    'lawsuit',
    'lawyer',
    'laxative',
    'layabout',
    'layer',
    'layette',
    'layman',
    'layoff',
    'layout',
    'leader',
    'leaflet',
    'league',
    'leakage',
    'leaky',
    'leaning',
    'leapfrog',
    'learn',
    'learner',
    'lease',
    'leash',
    'leave',
    'lecher',
    'lectern',
    'lecture',
    'lecturer',
    'ledge',
    'ledger',
    'leech',
    'leftie',
    'leftist',
    'leftover',
    'legacy',
    'legalize',
    'legate',
    'legation',
    'legend',
    'leggy',
    'legion',
    'legislate',
    'lemming',
    'lemon',
    'lemur',
    'lender',
    'length',
    'lengthen',
    'lengthy',
    'lentil',
    'leopard',
    'leotard',
    'leper',
    'lesbian',
    'lesion',
    'lessen',
    'lesson',
    'letdown',
    'letter',
    'letterbox',
    'lettuce',
    'level',
    'lever',
    'leviathan',
    'levitate',
    'lexicon',
    'liability',
    'liaise',
    'liaison',
    'libation',
    'libel',
    'liberal',
    'liberate',
    'liberator',
    'libertine',
    'liberty',
    'libido',
    'librarian',
    'library',
    'libretto',
    'license',
    'license',
    'licensee',
    'lichen',
    'licking',
    'lifebelt',
    'lifeboat',
    'lifebuoy',
    'lifeguard',
    'lifeline',
    'lifer',
    'lifespan',
    'lifetime',
    'ligament',
    'light',
    'lighten',
    'lighter',
    'lightship',
    'likely',
    'liken',
    'likeness',
    'lilac',
    'limber',
    'limerick',
    'limit',
    'limousine',
    'limpet',
    'linchpin',
    'linden',
    'lineage',
    'lineament',
    'linen',
    'liner',
    'linesman',
    'linger',
    'lingo',
    'linguist',
    'liniment',
    'lining',
    'linkage',
    'lintel',
    'lioness',
    'lipstick',
    'liquefy',
    'liqueur',
    'liquid',
    'liquidate',
    'liquidize',
    'liquor',
    'listen',
    'listener',
    'listing',
    'litany',
    'liter',
    'litigant',
    'liter',
    'litter',
    'litterbug',
    'liturgy',
    'lively',
    'liver',
    'livery',
    'living',
    'lizard',
    'llama',
    'loafer',
    'loathe',
    'lobby',
    'lobbyist',
    'lobotomy',
    'lobster',
    'local',
    'locale',
    'locality',
    'localize',
    'locate',
    'location',
    'locker',
    'locket',
    'lockout',
    'lockup',
    'locus',
    'locust',
    'locution',
    'lodge',
    'lodgment',
    'lodger',
    'lodging',
    'lodgment',
    'lofty',
    'logarithm',
    'loggia',
    'logician',
    'logistic',
    'loincloth',
    'loiter',
    'lollipop',
    'dollop',
    'lolly',
    'lonely',
    'loner',
    'longbow',
    'longing',
    'longitude',
    'loofah',
    'lookout',
    'loony',
    'loophole',
    'loose',
    'loosen',
    'looter',
    'lorgnette',
    'lorry',
    'loser',
    'lotion',
    'lottery',
    'lotus',
    'loudmouth',
    'lounge',
    'louse',
    'lousy',
    'lovely',
    'lover',
    'lower',
    'lowly',
    'loyalist',
    'loyalty',
    'lozenge',
    'lubricant',
    'lubricate',
    'lucky',
    'lullaby',
    'lumber',
    'luminary',
    'lumpy',
    'lunatic',
    'lunch',
    'luncheon',
    'lunchtime',
    'lunge',
    'lurch',
    'lusty',
    'luxuriate',
    'luxury',
    'lynch',
    'lyric',
    'lyricist',
    'macaroon',
    'machete',
    'machine',
    'machinist',
    'mackerel',
    'madden',
    'madhouse',
    'madman',
    'madrigal',
    'madwoman',
    'maestro',
    'magazine',
    'maggot',
    'magic',
    'magician',
    'magnate',
    'magnet',
    'magnetize',
    'magnify',
    'magnolia',
    'magnum',
    'magpie',
    'maiden',
    'mailbag',
    'mailbox',
    'mailman',
    'mainframe',
    'mainline',
    'mainstay',
    'maintain',
    'majesty',
    'major',
    'majorette',
    'majority',
    'maker',
    'making',
    'malady',
    'malign',
    'malinger',
    'mallard',
    'mallet',
    'maltreat',
    'mamba',
    'mammal',
    'mammoth',
    'manacle',
    'manage',
    'manager',
    'mandarin',
    'mandate',
    'mandible',
    'mandolin',
    'mandrake',
    'mandrill',
    'manger',
    'mangle',
    'mango',
    'mangrove',
    'mangy',
    'manhandle',
    'manhole',
    'manhunt',
    'mania',
    'maniac',
    'manicure',
    'manifest',
    'manifesto',
    'manly',
    'mannequin',
    'manner',
    'mannerism',
    'maneuver',
    'manor',
    'manse',
    'mansion',
    'mantel',
    'mantis',
    'mantle',
    'manual',
    'maple',
    'marathon',
    'marauder',
    'marble',
    'march',
    'marcher',
    'margin',
    'marginal',
    'marigold',
    'marina',
    'marinade',
    'marinate',
    'marine',
    'mariner',
    'marker',
    'market',
    'marking',
    'marksman',
    'marmoset',
    'maroon',
    'marquee',
    'marquis',
    'marriage',
    'marrow',
    'marry',
    'marsh',
    'marshal',
    'marshland',
    'marshy',
    'marsupial',
    'marten',
    'martin',
    'martinet',
    'martyr',
    'marvel',
    'mascot',
    'masochist',
    'mason',
    'massacre',
    'massage',
    'masseur',
    'masseuse',
    'massif',
    'master',
    'masthead',
    'masticate',
    'mastiff',
    'matador',
    'match',
    'matchbox',
    'material',
    'matinee',
    'matriarch',
    'matrix',
    'matron',
    'matter',
    'mattock',
    'mattress',
    'mature',
    'mausoleum',
    'mauve',
    'maverick',
    'maxim',
    'maximize',
    'mayday',
    'mayfly',
    'mayor',
    'mayoress',
    'meadow',
    'mealtime',
    'mealy',
    'meander',
    'meanie',
    'meaning',
    'measly',
    'measure',
    'meaty',
    'mecca',
    'mechanic',
    'mechanism',
    'mechanize',
    'medal',
    'medallion',
    'medalist',
    'meddle',
    'meddler',
    'mediate',
    'mediator',
    'medic',
    'medical',
    'medicine',
    'meditate',
    'medium',
    'medley',
    'meeting',
    'megalith',
    'megaphone',
    'megaton',
    'melee',
    'mellow',
    'melodrama',
    'melody',
    'melon',
    'member',
    'membrane',
    'memento',
    'memoir',
    'memorial',
    'memorize',
    'memory',
    'menace',
    'menage',
    'menagerie',
    'menial',
    'mentality',
    'mention',
    'mentor',
    'mercenary',
    'merchant',
    'mercy',
    'merge',
    'merger',
    'meridian',
    'meringue',
    'merit',
    'mermaid',
    'merry',
    'mesmerize',
    'message',
    'messenger',
    'messiah',
    'messy',
    'metal',
    'metaphor',
    'meteor',
    'meteorite',
    'meter',
    'methink',
    'method',
    'meter',
    'metro',
    'metronome',
    'mezzanine',
    'mezzo',
    'miasma',
    'micro',
    'microbe',
    'microchip',
    'microcosm',
    'microfilm',
    'microwave',
    'middle',
    'middleman',
    'midge',
    'midget',
    'midpoint',
    'midriff',
    'midwife',
    'mighty',
    'migraine',
    'migrant',
    'migrate',
    'mileage',
    'milepost',
    'milestone',
    'milieu',
    'militant',
    'militate',
    'militia',
    'milkmaid',
    'milkman',
    'milky',
    'miller',
    'milligram',
    'milliner',
    'million',
    'millionth',
    'millipede',
    'millstone',
    'mimic',
    'minaret',
    'mince',
    'mincer',
    'minder',
    'minefield',
    'miner',
    'mineral',
    'mingle',
    'miniature',
    'minibus',
    'minicab',
    'minim',
    'minimize',
    'minion',
    'minister',
    'ministry',
    'minnow',
    'minor',
    'minority',
    'minstrel',
    'minus',
    'minute',
    'miracle',
    'mirage',
    'mirror',
    'misapply',
    'misbehave',
    'miscarry',
    'miscast',
    'mischance',
    'misdeal',
    'misdeed',
    'misdirect',
    'misdo',
    'miser',
    'misery',
    'misfire',
    'misfit',
    'misgiving',
    'mishandle',
    'mishap',
    'mishear',
    'mishit',
    'misinform',
    'misjudge',
    'mislay',
    'mislead',
    'mismanage',
    'misnomer',
    'misprint',
    'misquote',
    'misread',
    'misrule',
    'missile',
    'mission',
    'missive',
    'misspell',
    'misspend',
    'mistake',
    'mistime',
    'mistreat',
    'mistress',
    'mistrust',
    'misty',
    'misuse',
    'mitigate',
    'mitten',
    'mixer',
    'mixture',
    'mnemonic',
    'moaner',
    'mobile',
    'mobilize',
    'mobster',
    'moccasin',
    'modal',
    'model',
    'moderate',
    'modernist',
    'modernize',
    'modifier',
    'modify',
    'modulate',
    'module',
    'mogul',
    'moist',
    'moisten',
    'molar',
    'molecule',
    'molehill',
    'molest',
    'molester',
    'mollify',
    'mollusk',
    'moment',
    'momma',
    'monarch',
    'monarchy',
    'monastery',
    'mongoose',
    'mongrel',
    'monitor',
    'monkey',
    'monocle',
    'monogram',
    'monograph',
    'monolith',
    'monologue',
    'monopoly',
    'monorail',
    'monotone',
    'monsoon',
    'monster',
    'montage',
    'month',
    'monthly',
    'monument',
    'mooch',
    'moody',
    'moonbeam',
    'moonlight',
    'mooring',
    'moorland',
    'moped',
    'moral',
    'moralist',
    'morality',
    'moralize',
    'morass',
    'morgue',
    'morning',
    'moron',
    'morsel',
    'mortal',
    'mortar',
    'mortgage',
    'mortician',
    'mortify',
    'mortuary',
    'mosaic',
    'mosque',
    'mosquito',
    'mossy',
    'motel',
    'mothball',
    'mother',
    'motif',
    'motion',
    'motivate',
    'motive',
    'motor',
    'motorbike',
    'motorboat',
    'motorcade',
    'motorist',
    'motorway',
    'motto',
    'moult',
    'mound',
    'mount',
    'mountain',
    'mourn',
    'mourner',
    'mouse',
    'mousetrap',
    'mustache',
    'mousy',
    'mouth',
    'mouthful',
    'mouthwash',
    'movement',
    'mover',
    'movie',
    'moviegoer',
    'mower',
    'muckraker',
    'mucky',
    'muddle',
    'muddy',
    'mudflat',
    'mudguard',
    'muezzin',
    'muffin',
    'muffle',
    'muffler',
    'mugger',
    'mulberry',
    'mulch',
    'mullah',
    'multiple',
    'multiply',
    'multitude',
    'mumble',
    'mummify',
    'mummy',
    'munch',
    'mural',
    'murder',
    'murderer',
    'murderess',
    'murky',
    'murmur',
    'muscle',
    'museum',
    'mushroom',
    'musical',
    'musician',
    'musket',
    'mussel',
    'muster',
    'musty',
    'mutant',
    'mutate',
    'mutation',
    'mutilate',
    'mutineer',
    'mutiny',
    'mutter',
    'muzzle',
    'myriad',
    'mystery',
    'mystic',
    'mystify',
    'namesake',
    'nanny',
    'napalm',
    'napkin',
    'nappy',
    'narcissus',
    'narcotic',
    'narrate',
    'narration',
    'narrative',
    'narrator',
    'narrow',
    'nasty',
    'nation',
    'national',
    'native',
    'natter',
    'natty',
    'natural',
    'nature',
    'naughty',
    'nauseate',
    'navel',
    'navigate',
    'navigator',
    'navvy',
    'necessity',
    'necklace',
    'neckline',
    'necktie',
    'nectarine',
    'needle',
    'needy',
    'negate',
    'negative',
    'neglect',
    'negligee',
    'negotiate',
    'neigh',
    'neighbor',
    'neologism',
    'neophyte',
    'nephew',
    'nerve',
    'nestle',
    'nestling',
    'nettle',
    'network',
    'neuron',
    'neurone',
    'neurosis',
    'neurotic',
    'neuter',
    'neutral',
    'neutron',
    'newcomer',
    'newel',
    'newsagent',
    'newsflash',
    'newsman',
    'newspaper',
    'newsreel',
    'newsroom',
    'nibble',
    'nicety',
    'niche',
    'nickel',
    'nickname',
    'niece',
    'nifty',
    'nigger',
    'niggle',
    'night',
    'nightcap',
    'nightclub',
    'nightgown',
    'nightie',
    'nightmare',
    'nimble',
    'nimbus',
    'ninety',
    'ninny',
    'ninth',
    'nipper',
    'nipple',
    'nirvana',
    'nitrate',
    'nitwit',
    'noble',
    'nobleman',
    'nobody',
    'nocturne',
    'noddle',
    'nodule',
    'noggin',
    'noise',
    'noisy',
    'nomad',
    'nominate',
    'nominee',
    'nonentity',
    'noodle',
    'noose',
    'normalize',
    'northward',
    'nosebag',
    'nosebleed',
    'nosedive',
    'nosegay',
    'nostril',
    'nostrum',
    'notable',
    'notary',
    'notation',
    'notch',
    'notebook',
    'notepad',
    'nothing',
    'notice',
    'notify',
    'notion',
    'naught',
    'nourish',
    'novel',
    'novelette',
    'novelist',
    'novelty',
    'novice',
    'nozzle',
    'nuance',
    'nucleus',
    'nudge',
    'nugget',
    'nuisance',
    'nullify',
    'number',
    'numeral',
    'numskull',
    'nunnery',
    'nuptial',
    'nurse',
    'nursemaid',
    'nursery',
    'nurture',
    'nutcase',
    'nuthouse',
    'nutrient',
    'nutriment',
    'nutty',
    'nuzzle',
    'nylon',
    'nymph',
    'oasis',
    'obeisance',
    'obelisk',
    'obfuscate',
    'obituary',
    'object',
    'objection',
    'objective',
    'objector',
    'oblige',
    'oblique',
    'oblong',
    'oboist',
    'obscenity',
    'obscure',
    'observe',
    'observer',
    'obsess',
    'obsession',
    'obstacle',
    'obstetric',
    'obstruct',
    'obtain',
    'obtrude',
    'obviate',
    'occasion',
    'occupant',
    'occupier',
    'occupy',
    'occur',
    'ocean',
    'octagon',
    'octave',
    'octopus',
    'oculist',
    'oddball',
    'oddity',
    'oddment',
    'odyssey',
    'esophagus',
    'offense',
    'offend',
    'offender',
    'offensive',
    'offer',
    'offering',
    'office',
    'officer',
    'official',
    'officiate',
    'offload',
    'offset',
    'offshoot',
    'oilcan',
    'oilfield',
    'oilman',
    'oilrig',
    'oilskin',
    'ointment',
    'oleander',
    'oligarchy',
    'olive',
    'ombudsman',
    'omelet',
    'omission',
    'omnibus',
    'onion',
    'onlooker',
    'onslaught',
    'onward',
    'opener',
    'opening',
    'opera',
    'operate',
    'operation',
    'operative',
    'operator',
    'operetta',
    'opiate',
    'opine',
    'opinion',
    'opossum',
    'opponent',
    'oppose',
    'opposite',
    'oppress',
    'oppressor',
    'optic',
    'optician',
    'option',
    'oracle',
    'orange',
    'orangery',
    'oration',
    'orator',
    'oratorio',
    'oratory',
    'orbit',
    'orchard',
    'orchestra',
    'orchid',
    'ordain',
    'ordeal',
    'order',
    'orderly',
    'ordinance',
    'organ',
    'organism',
    'organist',
    'organize',
    'organizer',
    'orgasm',
    'orient',
    'oriental',
    'orientate',
    'orifice',
    'origin',
    'original',
    'originate',
    'ornament',
    'orphan',
    'orphanage',
    'orthodoxy',
    'oscillate',
    'osier',
    'ossify',
    'osteopath',
    'ostracize',
    'ostrich',
    'other',
    'otter',
    'ounce',
    'outbid',
    'outbreak',
    'outburst',
    'outcast',
    'outclass',
    'outcome',
    'outcrop',
    'outcry',
    'outdo',
    'outdoor',
    'outfight',
    'outfit',
    'outfitter',
    'outflank',
    'outflow',
    'outfox',
    'outgoing',
    'outgrow',
    'outgrowth',
    'outhouse',
    'outing',
    'outlast',
    'outlaw',
    'outlay',
    'outlet',
    'outline',
    'outlive',
    'outlook',
    'outnumber',
    'outpost',
    'output',
    'outrage',
    'outride',
    'outrider',
    'outrun',
    'outsell',
    'outshine',
    'outside',
    'outsider',
    'outsmart',
    'outspread',
    'outstay',
    'outstrip',
    'outvote',
    'outward',
    'outwear',
    'outweigh',
    'outwit',
    'ovary',
    'ovation',
    'overact',
    'overall',
    'overawe',
    'overbear',
    'overbid',
    'overcoat',
    'overcome',
    'overdo',
    'overdose',
    'overdraft',
    'overdraw',
    'overeat',
    'overfeed',
    'overflow',
    'overfly',
    'overhang',
    'overhaul',
    'overhead',
    'overhear',
    'overheat',
    'overlap',
    'overlay',
    'overlie',
    'overload',
    'overlook',
    'overlord',
    'overpass',
    'overpay',
    'overplay',
    'overpower',
    'overrate',
    'overreach',
    'overreact',
    'override',
    'overrule',
    'overrun',
    'oversee',
    'overseer',
    'oversell',
    'overshoe',
    'overshoot',
    'oversight',
    'oversleep',
    'overspend',
    'overstate',
    'overstay',
    'overstep',
    'overtake',
    'overtax',
    'overthrow',
    'overtone',
    'overture',
    'overturn',
    'overvalue',
    'overview',
    'overwhelm',
    'overwork',
    'overwrite',
    'ovulate',
    'owner',
    'oxcart',
    'oxide',
    'oxidize',
    'oxygenate',
    'oyster',
    'pacemaker',
    'pachyderm',
    'pacifier',
    'pacifist',
    'pacify',
    'package',
    'packer',
    'packet',
    'paddle',
    'paddock',
    'paddy',
    'padlock',
    'padre',
    'paean',
    'pagan',
    'pageant',
    'pageboy',
    'pagoda',
    'paint',
    'paintbox',
    'painter',
    'painting',
    'palace',
    'palate',
    'palette',
    'palisade',
    'pallet',
    'palomino',
    'palpitate',
    'pamper',
    'pamphlet',
    'panacea',
    'pancake',
    'pancreas',
    'panda',
    'pandemic',
    'pander',
    'panegyric',
    'panel',
    'panelist',
    'panic',
    'pannier',
    'panorama',
    'pansy',
    'pantheon',
    'panther',
    'pantomime',
    'pantry',
    'papaya',
    'paper',
    'paperback',
    'paperboy',
    'paperclip',
    'papyrus',
    'parable',
    'parabola',
    'parachute',
    'parade',
    'paradigm',
    'paradox',
    'paragon',
    'paragraph',
    'parakeet',
    'parallel',
    'paralyze',
    'paralytic',
    'paramedic',
    'parameter',
    'paranoiac',
    'paranoid',
    'parapet',
    'parasite',
    'parasol',
    'parboil',
    'parcel',
    'parch',
    'parchment',
    'pardon',
    'parent',
    'pariah',
    'paring',
    'parish',
    'parka',
    'parkland',
    'parkway',
    'parley',
    'parlor',
    'parodist',
    'parody',
    'parole',
    'paroxysm',
    'parricide',
    'parrot',
    'parry',
    'parse',
    'parsnip',
    'parson',
    'parsonage',
    'partake',
    'particle',
    'parting',
    'partisan',
    'partition',
    'partitive',
    'partner',
    'partridge',
    'party',
    'parvenu',
    'passage',
    'passenger',
    'passion',
    'passport',
    'password',
    'pasta',
    'paste',
    'pastel',
    'pastiche',
    'pastille',
    'pastime',
    'pastor',
    'pastry',
    'pasture',
    'pasty',
    'patch',
    'patchwork',
    'patchy',
    'patella',
    'patent',
    'pathway',
    'patient',
    'patio',
    'patriarch',
    'patrician',
    'patricide',
    'patrimony',
    'patriot',
    'patrol',
    'patrolman',
    'patron',
    'patronize',
    'patsy',
    'patter',
    'pattern',
    'patty',
    'paunch',
    'paunchy',
    'pauper',
    'pauperize',
    'pause',
    'pavement',
    'pavilion',
    'payee',
    'payer',
    'payload',
    'paymaster',
    'payment',
    'payoff',
    'payroll',
    'payslip',
    'peach',
    'peacock',
    'peahen',
    'peaky',
    'peanut',
    'pearl',
    'peasant',
    'peaty',
    'pebble',
    'pebbly',
    'pecan',
    'pecker',
    'pectoral',
    'pedagogue',
    'pedal',
    'pedant',
    'peddle',
    'peddler',
    'pedestal',
    'pedicure',
    'pedigree',
    'pediment',
    'peeler',
    'peephole',
    'peepshow',
    'peerage',
    'peeress',
    'peeve',
    'pelican',
    'pellet',
    'pelvis',
    'penalize',
    'penalty',
    'penance',
    'pencil',
    'pendant',
    'pendulum',
    'penetrate',
    'penguin',
    'peninsula',
    'penis',
    'penknife',
    'pennant',
    'penny',
    'pension',
    'pensioner',
    'pentagon',
    'penthouse',
    'peony',
    'people',
    'pepper',
    'perceive',
    'perch',
    'percolate',
    'perennial',
    'perfect',
    'perforate',
    'perform',
    'performer',
    'perfume',
    'pergola',
    'peril',
    'perimeter',
    'period',
    'periphery',
    'periscope',
    'perish',
    'perisher',
    'perjure',
    'perjury',
    'perky',
    'permeate',
    'permit',
    'perplex',
    'persecute',
    'persevere',
    'persimmon',
    'persist',
    'person',
    'persona',
    'personage',
    'personify',
    'perspire',
    'persuade',
    'pertain',
    'perturb',
    'peruse',
    'pervade',
    'pervert',
    'peseta',
    'pesky',
    'pessimist',
    'pester',
    'pesticide',
    'pestle',
    'petal',
    'peter',
    'petition',
    'petrify',
    'petticoat',
    'petty',
    'petunia',
    'pfennig',
    'phalanx',
    'phallus',
    'phantom',
    'pharaoh',
    'pharmacy',
    'phase',
    'pheasant',
    'phial',
    'phobia',
    'phobic',
    'phoenix',
    'phone',
    'phonetic',
    'phosphate',
    'photo',
    'photocopy',
    'photostat',
    'phrase',
    'physical',
    'physician',
    'physicist',
    'physique',
    'pianist',
    'piano',
    'piazza',
    'piccolo',
    'pickax',
    'picker',
    'picket',
    'pickle',
    'picky',
    'picnic',
    'picnicker',
    'picture',
    'piddle',
    'pidgin',
    'piece',
    'pierce',
    'pigeon',
    'piggery',
    'piggy',
    'piggyback',
    'piggybank',
    'piglet',
    'pigment',
    'pigpen',
    'pigsty',
    'pigtail',
    'pilchard',
    'pilfer',
    'pilgrim',
    'pillage',
    'pillar',
    'pillbox',
    'pillory',
    'pillow',
    'pilot',
    'pimento',
    'pimple',
    'pinafore',
    'pincer',
    'pinch',
    'pineapple',
    'pinecone',
    'pinewood',
    'pinhead',
    'pinion',
    'pinkie',
    'pinnacle',
    'pinpoint',
    'pinprick',
    'pioneer',
    'pipeline',
    'piper',
    'pipette',
    'pipsqueak',
    'piranha',
    'pirate',
    'pirouette',
    'pistol',
    'piston',
    'pitch',
    'pitcher',
    'pitchfork',
    'pitfall',
    'pithy',
    'pituitary',
    'pivot',
    'pixie',
    'pizza',
    'placard',
    'placate',
    'place',
    'placebo',
    'placement',
    'placenta',
    'plague',
    'plaid',
    'plain',
    'plaint',
    'plaintiff',
    'plait',
    'plane',
    'planet',
    'plank',
    'planner',
    'plant',
    'plantain',
    'planter',
    'plaque',
    'plaster',
    'plasterer',
    'plastic',
    'plate',
    'plateau',
    'plateful',
    'platform',
    'platitude',
    'platoon',
    'platter',
    'playbill',
    'playboy',
    'player',
    'playgroup',
    'playhouse',
    'playmate',
    'playoff',
    'playpen',
    'playroom',
    'plaything',
    'plaza',
    'plead',
    'pleading',
    'pleasant',
    'please',
    'pleasure',
    'pleat',
    'plebeian',
    'plectrum',
    'pledge',
    'plenary',
    'plight',
    'plinth',
    'plodder',
    'plotter',
    'plough',
    'plover',
    'pluck',
    'plucky',
    'plughole',
    'plumb',
    'plumber',
    'plume',
    'plummet',
    'plummy',
    'plump',
    'plunder',
    'plunge',
    'plunger',
    'plunk',
    'plural',
    'plurality',
    'plush',
    'plushy',
    'plutocrat',
    'poach',
    'poacher',
    'pocket',
    'pocketful',
    'pockmark',
    'podium',
    'pogrom',
    'point',
    'pointer',
    'poison',
    'poisoner',
    'poker',
    'polarity',
    'polarize',
    'polecat',
    'polemic',
    'police',
    'policeman',
    'policy',
    'polish',
    'polisher',
    'polite',
    'politic',
    'politico',
    'polity',
    'polka',
    'pollinate',
    'pollutant',
    'pollute',
    'polyglot',
    'polygon',
    'polymath',
    'polymer',
    'polyp',
    'pommel',
    'ponce',
    'poncho',
    'ponder',
    'pontiff',
    'pontoon',
    'ponytail',
    'poodle',
    'poorhouse',
    'poplar',
    'popper',
    'poppet',
    'poppy',
    'populate',
    'porch',
    'porcupine',
    'porpoise',
    'portable',
    'portal',
    'portend',
    'portent',
    'porter',
    'portfolio',
    'porthole',
    'portico',
    'portion',
    'portly',
    'portrait',
    'portray',
    'portrayal',
    'poser',
    'poseur',
    'posit',
    'position',
    'posse',
    'possess',
    'possessor',
    'possible',
    'possum',
    'postbag',
    'postcard',
    'postcode',
    'postdate',
    'poster',
    'posterior',
    'posting',
    'postman',
    'postmark',
    'postpone',
    'postulate',
    'posture',
    'potato',
    'potentate',
    'pothole',
    'potion',
    'potpourri',
    'potter',
    'pottery',
    'potty',
    'pouch',
    'poultice',
    'pounce',
    'pound',
    'powder',
    'power',
    'powerboat',
    'practical',
    'practice',
    'pragmatic',
    'prairie',
    'praise',
    'prance',
    'prank',
    'prankster',
    'prattle',
    'prawn',
    'prayer',
    'preach',
    'preacher',
    'preamble',
    'precede',
    'precedent',
    'precept',
    'precinct',
    'precipice',
    'preclude',
    'precursor',
    'predate',
    'predator',
    'predicate',
    'predict',
    'preen',
    'prefab',
    'preface',
    'prefect',
    'prefer',
    'prefigure',
    'prefix',
    'pregnancy',
    'preheat',
    'prejudge',
    'prejudice',
    'prelate',
    'prelude',
    'premier',
    'premiere',
    'premise',
    'premium',
    'preoccupy',
    'prepare',
    'prepay',
    'presage',
    'prescribe',
    'presence',
    'present',
    'presenter',
    'preserve',
    'preserver',
    'preset',
    'preside',
    'president',
    'presidium',
    'press',
    'pressman',
    'pressure',
    'presume',
    'pretense',
    'pretend',
    'pretender',
    'pretext',
    'prettify',
    'pretty',
    'pretzel',
    'prevail',
    'prevent',
    'preview',
    'price',
    'pricey',
    'prick',
    'prickle',
    'pride',
    'priest',
    'priestess',
    'primary',
    'primate',
    'prime',
    'primer',
    'primrose',
    'prince',
    'princess',
    'principal',
    'principle',
    'print',
    'printer',
    'printout',
    'prior',
    'priority',
    'priory',
    'prise',
    'prism',
    'prison',
    'prisoner',
    'private',
    'privation',
    'privatize',
    'privilege',
    'privy',
    'prize',
    'probe',
    'problem',
    'proboscis',
    'procedure',
    'proceed',
    'process',
    'proclaim',
    'procreate',
    'procure',
    'procurer',
    'prodigal',
    'prodigy',
    'produce',
    'producer',
    'product',
    'profane',
    'profess',
    'professor',
    'proffer',
    'profile',
    'profit',
    'profiteer',
    'prognosis',
    'program',
    'progress',
    'prohibit',
    'project',
    'projector',
    'prologue',
    'prolong',
    'promenade',
    'promise',
    'promote',
    'promoter',
    'promotion',
    'prompt',
    'prompter',
    'prompting',
    'pronoun',
    'pronounce',
    'proof',
    'proofread',
    'propagate',
    'propel',
    'propeller',
    'property',
    'prophecy',
    'prophesy',
    'prophet',
    'proponent',
    'proposal',
    'propose',
    'proposer',
    'propound',
    'propriety',
    'proscribe',
    'prose',
    'prosecute',
    'prospect',
    'prosper',
    'prostate',
    'prostrate',
    'prosy',
    'protect',
    'protector',
    'protege',
    'protein',
    'protest',
    'protester',
    'proton',
    'prototype',
    'protrude',
    'proud',
    'prove',
    'proverb',
    'provide',
    'provider',
    'province',
    'provision',
    'proviso',
    'provoke',
    'provost',
    'prowl',
    'prowler',
    'proxy',
    'prude',
    'prune',
    'psalm',
    'pseudonym',
    'psych',
    'psyche',
    'psychic',
    'psycho',
    'psychosis',
    'psychotic',
    'publican',
    'publicist',
    'publicize',
    'publish',
    'publisher',
    'pucker',
    'pudding',
    'puddle',
    'puffball',
    'puffin',
    'puffy',
    'pullet',
    'pulley',
    'pullover',
    'pulpit',
    'pulsate',
    'pulse',
    'pulverize',
    'pummel',
    'pumpkin',
    'punch',
    'punchbag',
    'punchbowl',
    'punchy',
    'punctuate',
    'puncture',
    'pundit',
    'punish',
    'punter',
    'pupil',
    'puppet',
    'puppeteer',
    'puppy',
    'purchase',
    'purchaser',
    'puree',
    'purgative',
    'purge',
    'purify',
    'purist',
    'puritan',
    'purloin',
    'purple',
    'purport',
    'purpose',
    'purse',
    'purser',
    'pursue',
    'pursuer',
    'pursuit',
    'purvey',
    'purveyor',
    'pushcart',
    'pushchair',
    'pusher',
    'pushover',
    'pushy',
    'pussy',
    'pussyfoot',
    'pustule',
    'putrefy',
    'putsch',
    'putter',
    'puzzle',
    'pygmy',
    'pajama',
    'pylon',
    'pyramid',
    'python',
    'quack',
    'quadrille',
    'quadruped',
    'quadruple',
    'quaff',
    'quagmire',
    'quail',
    'quaint',
    'quake',
    'qualifier',
    'qualify',
    'quality',
    'qualm',
    'quandary',
    'quantify',
    'quantity',
    'quantum',
    'quarrel',
    'quarry',
    'quart',
    'quarter',
    'quarterly',
    'quartet',
    'quarto',
    'quasar',
    'quash',
    'quatrain',
    'quaver',
    'queasy',
    'queen',
    'queer',
    'queer',
    'quell',
    'quench',
    'query',
    'quest',
    'question',
    'queue',
    'quibble',
    'quiche',
    'quick',
    'quicken',
    'quickie',
    'quicksand',
    'quiet',
    'quieten',
    'quill',
    'quilt',
    'quince',
    'quintet',
    'quirk',
    'quirky',
    'quisling',
    'quitter',
    'quiver',
    'quoit',
    'quota',
    'quotation',
    'quote',
    'quotient',
    'rabbi',
    'rabbit',
    'raccoon',
    'racehorse',
    'racer',
    'racetrack',
    'racialist',
    'racist',
    'racket',
    'racketeer',
    'raconteur',
    'radial',
    'radiate',
    'radiator',
    'radical',
    'radio',
    'radiogram',
    'radish',
    'radius',
    'raffle',
    'rafter',
    'ragbag',
    'raider',
    'railcard',
    'railing',
    'railroad',
    'railway',
    'rainbow',
    'raincoat',
    'raindrop',
    'rainstorm',
    'rainy',
    'raise',
    'raisin',
    'rally',
    'ramble',
    'rambler',
    'rampage',
    'rampart',
    'ramrod',
    'ranch',
    'rancher',
    'randy',
    'range',
    'ranger',
    'ranking',
    'rankle',
    'ransack',
    'ransom',
    'rapid',
    'rapier',
    'rapist',
    'rapture',
    'rarity',
    'rascal',
    'rasher',
    'raspberry',
    'ratchet',
    'ratepayer',
    'ratify',
    'rating',
    'ratio',
    'ration',
    'ratter',
    'rattle',
    'rattler',
    'ratty',
    'raunchy',
    'ravage',
    'raven',
    'raver',
    'ravine',
    'raving',
    'ravish',
    'rawhide',
    'razor',
    'reach',
    'react',
    'reaction',
    'reactor',
    'reader',
    'reading',
    'readjust',
    'ready',
    'reaffirm',
    'realist',
    'reality',
    'realize',
    'realm',
    'realtor',
    'reaper',
    'reappear',
    'rearm',
    'rearrange',
    'reason',
    'reassert',
    'reassess',
    'reassure',
    'rebate',
    'rebel',
    'rebellion',
    'rebind',
    'rebound',
    'rebuff',
    'rebuild',
    'rebuke',
    'rebut',
    'rebuttal',
    'recall',
    'recant',
    'recap',
    'recapture',
    'recast',
    'recede',
    'receipt',
    'receive',
    'receiver',
    'reception',
    'recess',
    'recession',
    'recharge',
    'recipe',
    'recipient',
    'recital',
    'recite',
    'reckon',
    'reckoning',
    'reclaim',
    'recline',
    'recluse',
    'recognize',
    'recoil',
    'recollect',
    'recommend',
    'reconcile',
    'record',
    'recorder',
    'recording',
    'recount',
    'recoup',
    'recover',
    'recovery',
    'recreate',
    'recruit',
    'rectangle',
    'rectify',
    'rector',
    'rectory',
    'rectum',
    'recur',
    'recycle',
    'redden',
    'redeem',
    'redeemer',
    'redeploy',
    'redevelop',
    'redhead',
    'redirect',
    'redouble',
    'redoubt',
    'redress',
    'redskin',
    'reduce',
    'reduction',
    'redwood',
    'reedy',
    'reefer',
    'refectory',
    'refer',
    'referee',
    'reference',
    'refill',
    'refine',
    'refinery',
    'refit',
    'reflect',
    'reflector',
    'reflex',
    'reforest',
    'reform',
    'reformer',
    'reformist',
    'refract',
    'refrain',
    'refresh',
    'refuel',
    'refuge',
    'refugee',
    'refund',
    'refurbish',
    'refusal',
    'refuse',
    'refute',
    'regain',
    'regale',
    'regard',
    'regatta',
    'regency',
    'regent',
    'regicide',
    'regime',
    'regimen',
    'regiment',
    'region',
    'register',
    'registrar',
    'registry',
    'regress',
    'regret',
    'regroup',
    'regular',
    'regulate',
    'regulator',
    'rehash',
    'rehear',
    'rehearsal',
    'rehearse',
    'rehouse',
    'reign',
    'reimburse',
    'reinforce',
    'reinstate',
    'reissue',
    'reiterate',
    'reject',
    'rejoice',
    'rejoicing',
    'rejoin',
    'rejoinder',
    'rekindle',
    'relapse',
    'relate',
    'relation',
    'relative',
    'relax',
    'relay',
    'release',
    'relegate',
    'relent',
    'relic',
    'relief',
    'relieve',
    'religion',
    'reliquary',
    'relish',
    'relive',
    'reload',
    'relocate',
    'remain',
    'remainder',
    'remake',
    'remand',
    'remark',
    'remarry',
    'remedy',
    'remember',
    'remind',
    'reminder',
    'reminisce',
    'remit',
    'remnant',
    'remodel',
    'remote',
    'remount',
    'removal',
    'remove',
    'remover',
    'rename',
    'render',
    'rendering',
    'rendition',
    'renegade',
    'renege',
    'renew',
    'renewal',
    'renounce',
    'renovate',
    'rental',
    'reopen',
    'repair',
    'repartee',
    'repast',
    'repay',
    'repayment',
    'repeal',
    'repeat',
    'repeater',
    'repel',
    'repellent',
    'repent',
    'rephrase',
    'replace',
    'replay',
    'replenish',
    'replica',
    'replicate',
    'reply',
    'report',
    'reporter',
    'repose',
    'repossess',
    'represent',
    'repress',
    'reprieve',
    'reprimand',
    'reprint',
    'reprisal',
    'reproach',
    'reprobate',
    'reproduce',
    'reproof',
    'reprove',
    'reptile',
    'republic',
    'repudiate',
    'repulse',
    'request',
    'requiem',
    'require',
    'requisite',
    'requite',
    'reread',
    'rerun',
    'rescind',
    'rescue',
    'rescuer',
    'research',
    'resell',
    'resemble',
    'resent',
    'reserve',
    'reservist',
    'reservoir',
    'reset',
    'resettle',
    'reshuffle',
    'reside',
    'residence',
    'resident',
    'residue',
    'resign',
    'resin',
    'resist',
    'resistor',
    'resolve',
    'resonance',
    'resonate',
    'resort',
    'resound',
    'resource',
    'respect',
    'respecter',
    'respire',
    'respond',
    'response',
    'restate',
    'restock',
    'restore',
    'restorer',
    'restrain',
    'restraint',
    'restrict',
    'result',
    'resume',
    'resume',
    'resurface',
    'resurrect',
    'retail',
    'retailer',
    'retain',
    'retainer',
    'retake',
    'retaliate',
    'retard',
    'retch',
    'retell',
    'rethink',
    'reticule',
    'retina',
    'retinue',
    'retire',
    'retort',
    'retouch',
    'retrace',
    'retract',
    'retread',
    'retreat',
    'retrial',
    'retrieve',
    'retriever',
    'return',
    'reunion',
    'reunite',
    'revalue',
    'revamp',
    'reveal',
    'revel',
    'reveler',
    'revelry',
    'revenge',
    'revenue',
    'revere',
    'reverie',
    'reversal',
    'reverse',
    'reversion',
    'revert',
    'review',
    'reviewer',
    'revile',
    'revise',
    'revision',
    'revisit',
    'revival',
    'revive',
    'revivify',
    'revoke',
    'revolt',
    'revolve',
    'revolver',
    'revue',
    'reward',
    'rewind',
    'rewire',
    'rework',
    'rewrite',
    'rhapsody',
    'rheostat',
    'rhesus',
    'rheumatic',
    'rhino',
    'rhombus',
    'rhyme',
    'rhythm',
    'ribbon',
    'rickshaw',
    'ricochet',
    'riddle',
    'rider',
    'ridge',
    'ridicule',
    'riffle',
    'rifle',
    'rifleman',
    'right',
    'rightist',
    'rigidity',
    'rigmarole',
    'rigor',
    'ringer',
    'ringlet',
    'rinse',
    'rioter',
    'ripcord',
    'ripen',
    'riposte',
    'ripper',
    'ripple',
    'riser',
    'rising',
    'risky',
    'risotto',
    'rissole',
    'ritual',
    'rival',
    'rivalry',
    'river',
    'rivet',
    'rivulet',
    'roach',
    'roadblock',
    'roadhouse',
    'roadside',
    'roadster',
    'roadway',
    'roast',
    'roasting',
    'robber',
    'robbery',
    'robin',
    'robot',
    'robotic',
    'rocker',
    'rocket',
    'rocky',
    'rodent',
    'rodeo',
    'rogue',
    'roller',
    'romance',
    'romantic',
    'rondo',
    'rooftop',
    'rookery',
    'rookie',
    'roomful',
    'roommate',
    'roomy',
    'roost',
    'rooster',
    'rosary',
    'rosebud',
    'rosette',
    'roster',
    'rostrum',
    'rotate',
    'rotation',
    'rotor',
    'rotter',
    'rotunda',
    'rouge',
    'rough',
    'roughen',
    'round',
    'roundup',
    'rouse',
    'route',
    'routine',
    'rowboat',
    'rowdy',
    'rower',
    'royal',
    'royalist',
    'royalty',
    'rubber',
    'rubbing',
    'rubbish',
    'rubric',
    'rucksack',
    'ruction',
    'rudder',
    'ruddy',
    'ruffian',
    'ruffle',
    'ruler',
    'rumble',
    'rumbling',
    'ruminate',
    'rummage',
    'rumor',
    'rumple',
    'runabout',
    'runaway',
    'runner',
    'runny',
    'runway',
    'rupee',
    'rupture',
    'rustle',
    'rusty',
    'sable',
    'sabotage',
    'saboteur',
    'sachet',
    'sacking',
    'saber',
    'sacrament',
    'sacrifice',
    'sacristy',
    'sadden',
    'saddle',
    'saddlebag',
    'saddler',
    'safari',
    'safeguard',
    'sahib',
    'sailing',
    'sailor',
    'saint',
    'saintly',
    'salaam',
    'salad',
    'salami',
    'salary',
    'salesgirl',
    'salesman',
    'salient',
    'salivate',
    'sallow',
    'sally',
    'salon',
    'saloon',
    'salty',
    'salute',
    'salvage',
    'salve',
    'salver',
    'salvo',
    'samba',
    'samovar',
    'sampan',
    'sample',
    'sampler',
    'sanctify',
    'sanction',
    'sanctuary',
    'sandal',
    'sandbag',
    'sandbank',
    'sander',
    'sandpaper',
    'sandpit',
    'sandstorm',
    'sandwich',
    'sandy',
    'sapling',
    'sapphire',
    'sardine',
    'sarong',
    'satchel',
    'satellite',
    'satiate',
    'satire',
    'satirist',
    'satirize',
    'satisfy',
    'saturate',
    'satyr',
    'sauce',
    'saucepan',
    'saucer',
    'saucy',
    'sauna',
    'saunter',
    'sausage',
    'saute',
    'savage',
    'saver',
    'saving',
    'savior',
    'savor',
    'savory',
    'sawmill',
    'saxophone',
    'saying',
    'scabbard',
    'scabby',
    'scaffold',
    'scald',
    'scale',
    'scallop',
    'scallywag',
    'scalp',
    'scalpel',
    'scaly',
    'scamp',
    'scamper',
    'scandal',
    'scanner',
    'scant',
    'scanty',
    'scapegoat',
    'scapula',
    'scarce',
    'scare',
    'scarecrow',
    'scarf',
    'scarlet',
    'scary',
    'scatter',
    'scavenge',
    'scavenger',
    'scenario',
    'scene',
    'scent',
    'scepter',
    'skeptic',
    'scepter',
    'schedule',
    'schema',
    'scheme',
    'schemer',
    'schism',
    'scholar',
    'school',
    'schoolboy',
    'schooner',
    'science',
    'scientist',
    'scimitar',
    'scion',
    'scissor',
    'scoff',
    'scold',
    'scolding',
    'scone',
    'scoop',
    'scoopful',
    'scoot',
    'scooter',
    'scorch',
    'scorcher',
    'score',
    'scorecard',
    'scorer',
    'scorn',
    'scorpion',
    'scotch',
    'scoundrel',
    'scour',
    'scourer',
    'scourge',
    'scout',
    'scowl',
    'scrabble',
    'scraggy',
    'scram',
    'scramble',
    'scrambler',
    'scrap',
    'scrapbook',
    'scrape',
    'scrappy',
    'scratch',
    'scrawl',
    'scrawny',
    'scream',
    'scree',
    'screech',
    'screen',
    'screening',
    'screw',
    'screwy',
    'scribble',
    'scribe',
    'scrimmage',
    'scrimp',
    'script',
    'scripture',
    'scroll',
    'scrotum',
    'scrounge',
    'scrounger',
    'scrub',
    'scrubby',
    'scruff',
    'scruffy',
    'scrum',
    'scrunch',
    'scruple',
    'scuff',
    'scuffle',
    'scull',
    'scullery',
    'sculpt',
    'sculptor',
    'sculpture',
    'scupper',
    'scurry',
    'scuttle',
    'scythe',
    'seabird',
    'seaboard',
    'seafood',
    'seafront',
    'seagull',
    'seahorse',
    'seaman',
    'seamy',
    'seance',
    'seaplane',
    'seaport',
    'search',
    'seascape',
    'seashell',
    'seashore',
    'season',
    'seasoning',
    'seaward',
    'seaweed',
    'secede',
    'seclude',
    'second',
    'secondary',
    'seconder',
    'secret',
    'secretary',
    'secrete',
    'secretion',
    'section',
    'sector',
    'secure',
    'security',
    'sedan',
    'sedate',
    'sedative',
    'sedge',
    'sediment',
    'seduce',
    'seduction',
    'seedbed',
    'seedcake',
    'seedling',
    'seedy',
    'seesaw',
    'seethe',
    'segment',
    'segregate',
    'seize',
    'seizure',
    'select',
    'selection',
    'selector',
    'seller',
    'semantic',
    'semester',
    'seminar',
    'seminary',
    'semitone',
    'senator',
    'sender',
    'senior',
    'sensation',
    'sense',
    'sensitize',
    'sensor',
    'sentence',
    'sentiment',
    'sentinel',
    'sentry',
    'separate',
    'sepulcher',
    'sequel',
    'sequence',
    'sequester',
    'sequin',
    'seraph',
    'serenade',
    'sergeant',
    'serial',
    'serialize',
    'sermon',
    'sermonize',
    'serpent',
    'serum',
    'servant',
    'serve',
    'server',
    'service',
    'serviette',
    'serving',
    'session',
    'setback',
    'settee',
    'setter',
    'setting',
    'settle',
    'settler',
    'seventh',
    'seventy',
    'sever',
    'severe',
    'sewer',
    'sexist',
    'sextant',
    'sextet',
    'sexton',
    'shabby',
    'shack',
    'shackle',
    'shade',
    'shading',
    'shadow',
    'shady',
    'shaft',
    'shaggy',
    'shake',
    'shaker',
    'shaky',
    'shallot',
    'shallow',
    'shamble',
    'shame',
    'shampoo',
    'shamrock',
    'shandy',
    'shank',
    'shanty',
    'shape',
    'shapely',
    'shard',
    'share',
    'shark',
    'sharp',
    'sharpen',
    'sharpener',
    'shatter',
    'shave',
    'shaver',
    'shaving',
    'shawl',
    'sheaf',
    'shear',
    'sheath',
    'sheathe',
    'sheen',
    'sheepdog',
    'sheepfold',
    'sheepskin',
    'sheer',
    'sheet',
    'sheikh',
    'sheikhdom',
    'shelf',
    'shell',
    'shelter',
    'shelve',
    'shepherd',
    'sherbet',
    'sheriff',
    'sherry',
    'shield',
    'shift',
    'shifty',
    'shilling',
    'shimmer',
    'shindig',
    'shine',
    'shiner',
    'shingle',
    'shiny',
    'shipmate',
    'shipment',
    'shipwreck',
    'shipyard',
    'shire',
    'shirk',
    'shirt',
    'shirttail',
    'shiver',
    'shoal',
    'shock',
    'shocker',
    'shoddy',
    'shoehorn',
    'shoelace',
    'shoemaker',
    'shoetree',
    'shoot',
    'shooting',
    'shopper',
    'shore',
    'shoreline',
    'short',
    'shortage',
    'shorten',
    'shortfall',
    'shorty',
    'shotgun',
    'shoulder',
    'shout',
    'shove',
    'shovel',
    'shovelful',
    'showcase',
    'showdown',
    'shower',
    'showing',
    'showman',
    'showpiece',
    'showplace',
    'showroom',
    'showy',
    'shred',
    'shredder',
    'shrew',
    'shrewd',
    'shriek',
    'shrill',
    'shrimp',
    'shrine',
    'shrink',
    'shrivel',
    'shroud',
    'shrub',
    'shrubbery',
    'shrug',
    'shudder',
    'shuffle',
    'shunt',
    'shush',
    'shutdown',
    'shutter',
    'shuttle',
    'sibling',
    'sickbed',
    'sicken',
    'sickle',
    'sickly',
    'sickness',
    'sickroom',
    'sideboard',
    'sidecar',
    'sidekick',
    'sidelight',
    'sideline',
    'sideshow',
    'sidestep',
    'sideswipe',
    'sidetrack',
    'sidewalk',
    'siding',
    'sidle',
    'siege',
    'siesta',
    'sieve',
    'sight',
    'sighting',
    'sightseer',
    'signal',
    'signalman',
    'signatory',
    'signature',
    'signboard',
    'signify',
    'signpost',
    'silence',
    'silencer',
    'silkworm',
    'silky',
    'silly',
    'silver',
    'simile',
    'simmer',
    'simper',
    'simple',
    'simpleton',
    'simplify',
    'simulate',
    'simulator',
    'simulcast',
    'sinecure',
    'sinew',
    'singe',
    'singer',
    'single',
    'singlet',
    'sinker',
    'sinner',
    'sinus',
    'siphon',
    'siren',
    'sirloin',
    'sissy',
    'sister',
    'sitar',
    'sitcom',
    'sitter',
    'sitting',
    'situate',
    'situation',
    'sixpence',
    'sixteenth',
    'sixth',
    'sixty',
    'sizzle',
    'skate',
    'skater',
    'skein',
    'skeleton',
    'sketch',
    'sketchpad',
    'sketchy',
    'skewer',
    'skier',
    'skiff',
    'skill',
    'skillet',
    'skimp',
    'skimpy',
    'skinflint',
    'skinhead',
    'skinny',
    'skipper',
    'skirmish',
    'skirt',
    'skirting',
    'skitter',
    'skittle',
    'skulk',
    'skull',
    'skullcap',
    'skunk',
    'skydiver',
    'skylark',
    'skylight',
    'skyline',
    'slack',
    'slacken',
    'slacker',
    'slake',
    'slalom',
    'slander',
    'slangy',
    'slant',
    'slash',
    'slate',
    'slattern',
    'slaughter',
    'slave',
    'slaver',
    'sleazy',
    'sledge',
    'sleek',
    'sleep',
    'sleeper',
    'sleepwalk',
    'sleepy',
    'sleet',
    'sleeve',
    'sleigh',
    'sleuth',
    'slice',
    'slick',
    'slide',
    'slight',
    'slimmer',
    'slimy',
    'sling',
    'slink',
    'slinky',
    'slipcover',
    'slipknot',
    'slipper',
    'slipway',
    'slither',
    'sliver',
    'slobber',
    'slogan',
    'sloop',
    'slope',
    'sloppy',
    'slosh',
    'sloth',
    'slouch',
    'slough',
    'slowdown',
    'slowpoke',
    'sluice',
    'slumber',
    'slummy',
    'slump',
    'slurp',
    'slushy',
    'smack',
    'small',
    'smarmy',
    'smart',
    'smarten',
    'smash',
    'smasher',
    'smear',
    'smell',
    'smelly',
    'smelt',
    'smelter',
    'smile',
    'smirk',
    'smite',
    'smithy',
    'smock',
    'smoke',
    'smoker',
    'smoky',
    'smooch',
    'smooth',
    'smoothie',
    'smother',
    'smolder',
    'smudge',
    'smudgy',
    'smuggle',
    'smuggler',
    'smutty',
    'snack',
    'snail',
    'snake',
    'snaky',
    'snappy',
    'snapshot',
    'snare',
    'snarl',
    'snatch',
    'snazzy',
    'sneak',
    'sneaker',
    'sneaky',
    'sneer',
    'sneeze',
    'snick',
    'snicker',
    'sniff',
    'sniffle',
    'snifter',
    'snigger',
    'snipe',
    'sniper',
    'snippet',
    'snitch',
    'snivel',
    'snobby',
    'snooker',
    'snoop',
    'snooper',
    'snooty',
    'snooze',
    'snore',
    'snorkel',
    'snort',
    'snotty',
    'snout',
    'snowball',
    'snowdrift',
    'snowdrop',
    'snowfall',
    'snowfield',
    'snowflake',
    'snowman',
    'snowshoe',
    'snowstorm',
    'snowy',
    'snuff',
    'snuffle',
    'snuggle',
    'soapbox',
    'soapy',
    'sober',
    'social',
    'socialist',
    'socialite',
    'socialize',
    'society',
    'socket',
    'softball',
    'soften',
    'softener',
    'softie',
    'softwood',
    'softy',
    'soggy',
    'soiree',
    'sojourn',
    'solarium',
    'solder',
    'soldier',
    'solecism',
    'solicit',
    'solicitor',
    'solid',
    'solidify',
    'soliloquy',
    'solitaire',
    'soloist',
    'solstice',
    'solution',
    'solve',
    'solvent',
    'sombrero',
    'sonata',
    'songbird',
    'sonnet',
    'soothe',
    'sophistry',
    'soppy',
    'soprano',
    'sorbet',
    'sorcerer',
    'sorceress',
    'sorrow',
    'sorry',
    'sortie',
    'souffle',
    'sough',
    'sound',
    'sounding',
    'source',
    'southward',
    'souvenir',
    'sovereign',
    'soybean',
    'space',
    'spaceman',
    'spaceship',
    'spacesuit',
    'spade',
    'spangle',
    'spaniel',
    'spank',
    'spanking',
    'spanner',
    'spare',
    'spark',
    'sparkle',
    'sparkler',
    'sparrow',
    'sparse',
    'spasm',
    'spastic',
    'spate',
    'spatter',
    'spatula',
    'spawn',
    'speak',
    'speaker',
    'spear',
    'spearhead',
    'spearmint',
    'special',
    'specialty',
    'specialty',
    'specific',
    'specify',
    'specimen',
    'speck',
    'speckle',
    'spectacle',
    'spectator',
    'specter',
    'spectrum',
    'speculate',
    'speech',
    'speed',
    'speedboat',
    'speedwell',
    'speedy',
    'spell',
    'spellbind',
    'speller',
    'spelling',
    'spend',
    'spender',
    'sperm',
    'sphere',
    'sphinx',
    'spice',
    'spicy',
    'spider',
    'spike',
    'spiky',
    'spill',
    'spillage',
    'spindle',
    'spindly',
    'spine',
    'spinet',
    'spinner',
    'spinster',
    'spiral',
    'spire',
    'spirit',
    'spiritual',
    'spitfire',
    'spittoon',
    'splash',
    'splatter',
    'splay',
    'spleen',
    'splendor',
    'splice',
    'splint',
    'splinter',
    'split',
    'splodge',
    'splotch',
    'splurge',
    'splutter',
    'spoil',
    'spoke',
    'spokesman',
    'sponge',
    'sponger',
    'spongy',
    'sponsor',
    'spoof',
    'spook',
    'spooky',
    'spool',
    'spoon',
    'spoonful',
    'spore',
    'sporran',
    'sport',
    'sportsman',
    'spotlight',
    'spotter',
    'spotty',
    'spouse',
    'spout',
    'sprain',
    'sprat',
    'sprawl',
    'spray',
    'sprayer',
    'spread',
    'spree',
    'sprig',
    'sprightly',
    'spring',
    'springbok',
    'springy',
    'sprinkle',
    'sprinkler',
    'sprint',
    'sprinter',
    'sprite',
    'sprocket',
    'sprout',
    'spruce',
    'spunky',
    'spurn',
    'spurt',
    'sputter',
    'squabble',
    'squad',
    'squadron',
    'squall',
    'squander',
    'square',
    'squash',
    'squashy',
    'squat',
    'squatter',
    'squaw',
    'squawk',
    'squeak',
    'squeaky',
    'squeal',
    'squeeze',
    'squelch',
    'squib',
    'squid',
    'squiggle',
    'squint',
    'squire',
    'squirm',
    'squirrel',
    'squirt',
    'stabbing',
    'stabilize',
    'stable',
    'stack',
    'stadium',
    'staff',
    'stage',
    'stagehand',
    'stagger',
    'stagnate',
    'stagy',
    'stain',
    'stair',
    'staircase',
    'stairway',
    'stairwell',
    'stake',
    'stale',
    'stalemate',
    'stalk',
    'stall',
    'stallion',
    'stalwart',
    'stamen',
    'stammer',
    'stamp',
    'stampede',
    'stance',
    'stanchion',
    'stand',
    'standard',
    'standby',
    'standpipe',
    'stanza',
    'staple',
    'stapler',
    'starch',
    'starchy',
    'stare',
    'stark',
    'starlet',
    'starling',
    'start',
    'starter',
    'startle',
    'starve',
    'stash',
    'state',
    'stately',
    'statement',
    'stateroom',
    'statesman',
    'static',
    'station',
    'stationer',
    'statistic',
    'statue',
    'statuette',
    'statute',
    'staunch',
    'stave',
    'steady',
    'steak',
    'steal',
    'stealthy',
    'steam',
    'steamer',
    'steamy',
    'steed',
    'steel',
    'steep',
    'steeple',
    'steer',
    'stench',
    'stencil',
    'stepchild',
    'steppe',
    'stepson',
    'stereo',
    'sterilize',
    'stern',
    'sternum',
    'steroid',
    'stetson',
    'stevedore',
    'steward',
    'stick',
    'sticker',
    'stickler',
    'stickpin',
    'sticky',
    'stiff',
    'stiffen',
    'stiffener',
    'stifle',
    'stigma',
    'stile',
    'stiletto',
    'still',
    'stilt',
    'stimulant',
    'stimulate',
    'stimulus',
    'sting',
    'stingray',
    'stingy',
    'stink',
    'stint',
    'stipend',
    'stipulate',
    'stirrer',
    'stirring',
    'stirrup',
    'stitch',
    'stoat',
    'stock',
    'stockade',
    'stocking',
    'stockist',
    'stockpile',
    'stockroom',
    'stocky',
    'stodgy',
    'stoic',
    'stoic',
    'stoke',
    'stole',
    'stomach',
    'stomp',
    'stone',
    'stonewall',
    'stony',
    'stooge',
    'stool',
    'stoop',
    'stopcock',
    'stopgap',
    'stopover',
    'stoppage',
    'stopper',
    'stopwatch',
    'store',
    'storeroom',
    'stork',
    'storm',
    'stormy',
    'story',
    'storybook',
    'stout',
    'stove',
    'stowaway',
    'straddle',
    'strafe',
    'straggle',
    'straggly',
    'straight',
    'strain',
    'strainer',
    'strait',
    'strand',
    'strange',
    'stranger',
    'strangle',
    'strangler',
    'strap',
    'stratagem',
    'strategy',
    'stratum',
    'straw',
    'stray',
    'streak',
    'streaker',
    'streaky',
    'stream',
    'streamer',
    'street',
    'streetcar',
    'strength',
    'stress',
    'stretch',
    'stretcher',
    'stretchy',
    'strew',
    'strict',
    'stricture',
    'stride',
    'strike',
    'striker',
    'string',
    'stringy',
    'strip',
    'stripe',
    'stripling',
    'stripper',
    'stripy',
    'strive',
    'stroke',
    'stroll',
    'stroller',
    'strong',
    'structure',
    'struggle',
    'strum',
    'strumpet',
    'strut',
    'stubby',
    'student',
    'studio',
    'study',
    'stuff',
    'stuffy',
    'stultify',
    'stumble',
    'stump',
    'stumpy',
    'stunner',
    'stunt',
    'stupefy',
    'stupid',
    'stupidity',
    'stupor',
    'sturdy',
    'stutter',
    'style',
    'stylist',
    'stylistic',
    'stylus',
    'stymie',
    'suave',
    'subaltern',
    'subdivide',
    'subdue',
    'subgroup',
    'subject',
    'subjugate',
    'sublet',
    'sublimate',
    'submarine',
    'submerge',
    'submit',
    'subpoena',
    'subscribe',
    'subside',
    'subsidize',
    'subsidy',
    'subsist',
    'substance',
    'subsume',
    'subsystem',
    'subtle',
    'subtlety',
    'subtract',
    'suburb',
    'subvert',
    'subway',
    'succeed',
    'success',
    'successor',
    'succor',
    'succulent',
    'succumb',
    'sucker',
    'suckle',
    'suffer',
    'sufferer',
    'suffering',
    'suffice',
    'suffix',
    'suffocate',
    'suffuse',
    'sugar',
    'suggest',
    'suicide',
    'suitcase',
    'suite',
    'suitor',
    'sulky',
    'sully',
    'sultan',
    'sultana',
    'summarize',
    'summary',
    'summation',
    'summer',
    'summit',
    'summon',
    'summons',
    'sunbathe',
    'sunbather',
    'sunbeam',
    'sunbonnet',
    'sunburn',
    'sundae',
    'sundeck',
    'sundial',
    'sundry',
    'sunflower',
    'sunhat',
    'sunny',
    'sunrise',
    'sunroof',
    'sunset',
    'sunshade',
    'suntan',
    'superior',
    'superman',
    'supersede',
    'superstar',
    'supervise',
    'supper',
    'supplant',
    'supplier',
    'supply',
    'support',
    'supporter',
    'suppose',
    'suppress',
    'suppurate',
    'surcharge',
    'surety',
    'surface',
    'surfboard',
    'surfer',
    'surge',
    'surgeon',
    'surgery',
    'surly',
    'surmise',
    'surmount',
    'surname',
    'surpass',
    'surplice',
    'surplus',
    'surprise',
    'surrender',
    'surrogate',
    'surround',
    'survey',
    'surveyor',
    'survival',
    'survive',
    'survivor',
    'suspect',
    'suspend',
    'suspender',
    'suspicion',
    'sustain',
    'suture',
    'swaddle',
    'swagger',
    'swain',
    'swallow',
    'swamp',
    'swampy',
    'swank',
    'swarm',
    'swarthy',
    'swastika',
    'swathe',
    'swear',
    'sweat',
    'sweatband',
    'sweater',
    'sweatshop',
    'swede',
    'sweep',
    'sweeper',
    'sweeping',
    'sweet',
    'sweeten',
    'sweetener',
    'sweetmeat',
    'swell',
    'swelling',
    'swelter',
    'swerve',
    'swift',
    'swill',
    'swimmer',
    'swimsuit',
    'swindle',
    'swindler',
    'swine',
    'swing',
    'swinger',
    'swipe',
    'swirl',
    'swish',
    'switch',
    'swivel',
    'swoon',
    'swoop',
    'sword',
    'swordfish',
    'sycamore',
    'sycophant',
    'syllable',
    'syllabus',
    'symbol',
    'symbolize',
    'symmetry',
    'sympathy',
    'symphony',
    'symposium',
    'symptom',
    'synagogue',
    'syncopate',
    'syndicate',
    'syndrome',
    'synod',
    'synonym',
    'synopsis',
    'synthesis',
    'synthetic',
    'siphon',
    'syringe',
    'syrup',
    'system',
    'tabby',
    'table',
    'tableau',
    'tablet',
    'tabloid',
    'taboo',
    'tabulate',
    'tabulator',
    'tackle',
    'tacky',
    'tactic',
    'tactician',
    'tadpole',
    'tailback',
    'tailgate',
    'tailor',
    'tailwind',
    'taint',
    'takeaway',
    'takeoff',
    'takeover',
    'taker',
    'talent',
    'talisman',
    'talker',
    'talkie',
    'tally',
    'talon',
    'tamarind',
    'tamarack',
    'tamper',
    'tampon',
    'tandem',
    'tangent',
    'tangerine',
    'tangle',
    'tango',
    'tangy',
    'tankard',
    'tanker',
    'tanner',
    'tannery',
    'tantalize',
    'tantrum',
    'taper',
    'tapestry',
    'tapeworm',
    'tapir',
    'tappet',
    'tarantula',
    'tardy',
    'target',
    'tariff',
    'tarnish',
    'tarpaulin',
    'tarry',
    'tartan',
    'tartar',
    'tassel',
    'taste',
    'taster',
    'tasty',
    'tattoo',
    'tattooist',
    'tatty',
    'taunt',
    'tauten',
    'tautology',
    'tavern',
    'tawdry',
    'tawny',
    'taxicab',
    'taxonomy',
    'taxpayer',
    'teacake',
    'teach',
    'teacher',
    'teaching',
    'teacup',
    'teamster',
    'teapot',
    'tearaway',
    'teardrop',
    'tease',
    'teasel',
    'teaser',
    'teaspoon',
    'technique',
    'teddy',
    'teenager',
    'teeny',
    'teeter',
    'teethe',
    'telecast',
    'telegram',
    'telegraph',
    'telephone',
    'telescope',
    'televise',
    'telex',
    'teller',
    'telling',
    'telltale',
    'telly',
    'temper',
    'tempest',
    'template',
    'temple',
    'tempo',
    'temporize',
    'tempt',
    'tenancy',
    'tenant',
    'tendency',
    'tender',
    'tenderize',
    'tendon',
    'tendril',
    'tenement',
    'tenet',
    'tenner',
    'tenon',
    'tenor',
    'tenpin',
    'tense',
    'tension',
    'tentacle',
    'tenth',
    'tepee',
    'terminal',
    'terminate',
    'terminus',
    'termite',
    'terrace',
    'terrapin',
    'terrier',
    'terrify',
    'territory',
    'terror',
    'terrorist',
    'terrorize',
    'terse',
    'testament',
    'testicle',
    'testify',
    'testimony',
    'testy',
    'tether',
    'textbook',
    'textile',
    'texture',
    'thank',
    'thatch',
    'thatcher',
    'theater',
    'theft',
    'theme',
    'theocracy',
    'theology',
    'theorem',
    'theorist',
    'theorize',
    'theory',
    'therapist',
    'therm',
    'thermal',
    'thesaurus',
    'thesis',
    'thespian',
    'thick',
    'thicken',
    'thickener',
    'thicket',
    'thickness',
    'thief',
    'thigh',
    'thimble',
    'thing',
    'think',
    'thinker',
    'thinner',
    'third',
    'thirst',
    'thirsty',
    'thirty',
    'thistle',
    'thong',
    'thorax',
    'thorn',
    'thorny',
    'thought',
    'thousand',
    'thrash',
    'thread',
    'threat',
    'threaten',
    'three',
    'threesome',
    'thresh',
    'threshold',
    'thrifty',
    'thrill',
    'thriller',
    'thrive',
    'throat',
    'throaty',
    'throb',
    'throne',
    'throng',
    'throttle',
    'throw',
    'throwback',
    'thrum',
    'thrush',
    'thrust',
    'thumb',
    'thumbnail',
    'thumbtack',
    'thump',
    'thunder',
    'thwack',
    'thwart',
    'thyroid',
    'tiara',
    'tibia',
    'ticker',
    'ticket',
    'tickle',
    'tidbit',
    'tiger',
    'tight',
    'tighten',
    'tightrope',
    'tigress',
    'tilde',
    'tiller',
    'timber',
    'timepiece',
    'timer',
    'timetable',
    'timpanist',
    'tincture',
    'tinderbox',
    'tinge',
    'tingle',
    'tinker',
    'tinkle',
    'tinny',
    'tippet',
    'tipple',
    'tippler',
    'tipster',
    'tipsy',
    'tiptoe',
    'tirade',
    'tissue',
    'titan',
    'tithe',
    'titillate',
    'title',
    'titter',
    'toadstool',
    'toady',
    'toast',
    'toaster',
    'toboggan',
    'toccata',
    'toddle',
    'toddler',
    'toddy',
    'toecap',
    'toehold',
    'toenail',
    'toffee',
    'toggle',
    'toilet',
    'token',
    'tolerance',
    'tolerate',
    'tollhouse',
    'tomahawk',
    'tomato',
    'tombola',
    'tomboy',
    'tombstone',
    'tomcat',
    'tomtit',
    'tongue',
    'tonic',
    'tonnage',
    'tonne',
    'tonsil',
    'tonsure',
    'tooth',
    'toothpick',
    'toothy',
    'tootle',
    'tootsie',
    'topaz',
    'topcoat',
    'topic',
    'topper',
    'topping',
    'topple',
    'torch',
    'torment',
    'tormentor',
    'tornado',
    'torpedo',
    'torrent',
    'torso',
    'tortilla',
    'tortoise',
    'torture',
    'torturer',
    'total',
    'totem',
    'totter',
    'toucan',
    'touch',
    'touchdown',
    'touchy',
    'tough',
    'toughen',
    'toupee',
    'tourist',
    'towel',
    'tower',
    'townie',
    'township',
    'towpath',
    'towrope',
    'toxin',
    'trace',
    'trachea',
    'tracing',
    'track',
    'tracker',
    'tracksuit',
    'tract',
    'tractor',
    'trade',
    'trademark',
    'trader',
    'tradesman',
    'tradition',
    'traduce',
    'traffic',
    'tragedy',
    'trail',
    'trailer',
    'train',
    'trainee',
    'trainer',
    'traipse',
    'trait',
    'traitor',
    'trammel',
    'tramp',
    'trample',
    'tramway',
    'trance',
    'transact',
    'transcend',
    'transept',
    'transfer',
    'transfix',
    'transform',
    'translate',
    'transmit',
    'transmute',
    'transpire',
    'transport',
    'transpose',
    'trapeze',
    'trapper',
    'trashcan',
    'trashy',
    'trauma',
    'travel',
    'traveler',
    'traverse',
    'travesty',
    'trawl',
    'trawler',
    'treachery',
    'tread',
    'treadle',
    'treadmill',
    'treasure',
    'treasurer',
    'treasury',
    'treat',
    'treatise',
    'treatment',
    'treaty',
    'treble',
    'treetop',
    'trellis',
    'tremble',
    'tremolo',
    'tremor',
    'trench',
    'trend',
    'trendy',
    'trespass',
    'tress',
    'trestle',
    'triad',
    'trial',
    'triangle',
    'tribe',
    'tribesman',
    'tribunal',
    'tributary',
    'tribute',
    'trick',
    'trickle',
    'trickster',
    'tricky',
    'tricolor',
    'tricycle',
    'trier',
    'trifle',
    'trigger',
    'trike',
    'trilby',
    'trill',
    'trillion',
    'trilogy',
    'trimaran',
    'trimming',
    'trinket',
    'triple',
    'triplet',
    'tripod',
    'tripper',
    'triptych',
    'tripwire',
    'triumph',
    'troll',
    'trolley',
    'trollop',
    'trombone',
    'troop',
    'trooper',
    'trophy',
    'trotter',
    'trouble',
    'trough',
    'trounce',
    'troupe',
    'trouper',
    'trouser',
    'trousseau',
    'trout',
    'trowel',
    'truant',
    'truce',
    'truck',
    'trucker',
    'truckle',
    'truckload',
    'trudge',
    'truffle',
    'truism',
    'trump',
    'trumpet',
    'trumpeter',
    'truncate',
    'truncheon',
    'trundle',
    'trunk',
    'truss',
    'trust',
    'trustee',
    'trusty',
    'truth',
    'tryst',
    'tsarina',
    'tubby',
    'tuber',
    'tulip',
    'tumble',
    'tumbler',
    'tumbrel',
    'tummy',
    'tumor',
    'tumult',
    'tundra',
    'tuner',
    'tunic',
    'tunnel',
    'turban',
    'turbine',
    'turbo',
    'tureen',
    'turkey',
    'turnabout',
    'turncoat',
    'turning',
    'turnip',
    'turnout',
    'turnover',
    'turnpike',
    'turnround',
    'turnstile',
    'turntable',
    'turquoise',
    'turret',
    'turtle',
    'tussle',
    'tussock',
    'tutor',
    'tutorial',
    'tuxedo',
    'twang',
    'tweak',
    'tweed',
    'tweet',
    'twelfth',
    'twelve',
    'twentieth',
    'twenty',
    'twerp',
    'twiddle',
    'twine',
    'twinge',
    'twinkle',
    'twirl',
    'twist',
    'twister',
    'twitch',
    'twitter',
    'twosome',
    'tycoon',
    'typecast',
    'typewrite',
    'typhoon',
    'typify',
    'typist',
    'tyrannize',
    'tyranny',
    'tyrant',
    'udder',
    'ukelele',
    'ulcer',
    'ultimatum',
    'ululate',
    'umbrella',
    'umlaut',
    'umpire',
    'unbalance',
    'unbend',
    'unbind',
    'unbuckle',
    'unburden',
    'unbutton',
    'uncanny',
    'unchain',
    'unclasp',
    'uncle',
    'uncoil',
    'uncork',
    'uncover',
    'underbid',
    'undercut',
    'underdog',
    'undergo',
    'underlay',
    'underlie',
    'underline',
    'underling',
    'underlip',
    'undermine',
    'underpass',
    'underpay',
    'underpin',
    'underplay',
    'underrate',
    'undersell',
    'underside',
    'undertake',
    'undertone',
    'undertow',
    'undraw',
    'undress',
    'undulate',
    'unearth',
    'uneasy',
    'unfasten',
    'unfit',
    'unfold',
    'unfurl',
    'unhappy',
    'unhealthy',
    'unhinge',
    'unhook',
    'unicorn',
    'uniform',
    'unify',
    'union',
    'unionist',
    'unionize',
    'unite',
    'universal',
    'universe',
    'unkind',
    'unknown',
    'unlearn',
    'unleash',
    'unload',
    'unlock',
    'unmake',
    'unman',
    'unmask',
    'unnerve',
    'unpack',
    'unpick',
    'unplug',
    'unravel',
    'unroll',
    'unsaddle',
    'unsay',
    'unscrew',
    'unseat',
    'unsettle',
    'unstick',
    'unstrap',
    'unstring',
    'untangle',
    'untidy',
    'untie',
    'untruth',
    'unveil',
    'unwind',
    'unwrap',
    'unzip',
    'upbraid',
    'update',
    'upend',
    'upgrade',
    'upheaval',
    'uphold',
    'upholder',
    'upland',
    'uplift',
    'upper',
    'upright',
    'uprise',
    'uprising',
    'uproot',
    'upset',
    'upstage',
    'upstart',
    'upsweep',
    'upswing',
    'upturn',
    'urbanize',
    'urchin',
    'urinal',
    'urinate',
    'usage',
    'usher',
    'usherette',
    'usurer',
    'usurp',
    'utensil',
    'uterus',
    'utility',
    'utilize',
    'utopia',
    'utter',
    'utterance',
    'vacancy',
    'vacate',
    'vacation',
    'vaccinate',
    'vaccine',
    'vacillate',
    'vacuum',
    'vagabond',
    'vagary',
    'vagina',
    'vagrant',
    'vague',
    'valance',
    'valence',
    'valency',
    'valentine',
    'valet',
    'validate',
    'valise',
    'valley',
    'valuable',
    'valuation',
    'value',
    'valuer',
    'valve',
    'vampire',
    'vandal',
    'vandalize',
    'vanish',
    'vanity',
    'vanquish',
    'vaporize',
    'vapor',
    'variable',
    'variant',
    'variation',
    'variety',
    'varnish',
    'vasectomy',
    'vassal',
    'vault',
    'vaunt',
    'vector',
    'vegan',
    'vegetable',
    'vegetate',
    'vehicle',
    'velocity',
    'vendetta',
    'vendor',
    'venerate',
    'ventilate',
    'ventricle',
    'venture',
    'venue',
    'veranda',
    'verbalize',
    'verdict',
    'verge',
    'verger',
    'verify',
    'verity',
    'vermouth',
    'verruca',
    'verse',
    'version',
    'vertebra',
    'vessel',
    'vestibule',
    'vestige',
    'vestry',
    'vetch',
    'veteran',
    'vexation',
    'viaduct',
    'vibrate',
    'vibrato',
    'vibrator',
    'vicar',
    'vicarage',
    'victim',
    'victimize',
    'victor',
    'victory',
    'video',
    'viewer',
    'viewpoint',
    'vigil',
    'vigilante',
    'vignette',
    'vilify',
    'villa',
    'village',
    'villager',
    'villain',
    'vindicate',
    'vineyard',
    'vintage',
    'vintner',
    'viola',
    'violate',
    'violet',
    'violin',
    'violinist',
    'viper',
    'virago',
    'virgin',
    'virginal',
    'virtue',
    'virtuoso',
    'virus',
    'visage',
    'viscount',
    'vision',
    'visionary',
    'visit',
    'visitor',
    'visor',
    'vista',
    'visualize',
    'vitamin',
    'vitiate',
    'vixen',
    'vocal',
    'vocalist',
    'vocation',
    'vocative',
    'vodka',
    'voice',
    'volcano',
    'volley',
    'voltage',
    'volume',
    'voluntary',
    'volunteer',
    'vomit',
    'vortex',
    'voter',
    'vouch',
    'voucher',
    'vouchsafe',
    'vowel',
    'voyage',
    'voyager',
    'voyeur',
    'vulture',
    'vulva',
    'wacky',
    'waddle',
    'wader',
    'wafer',
    'waffle',
    'wager',
    'waggle',
    'wagon',
    'wagtail',
    'wainscot',
    'waist',
    'waistband',
    'waistcoat',
    'waistline',
    'waiter',
    'waitress',
    'waive',
    'waken',
    'walkabout',
    'walker',
    'walkout',
    'walkover',
    'walkway',
    'wallaby',
    'wallet',
    'wallop',
    'walloping',
    'wallow',
    'wallpaper',
    'wally',
    'walnut',
    'walrus',
    'waltz',
    'wander',
    'wanderer',
    'wandering',
    'wangle',
    'wanker',
    'warble',
    'warbler',
    'warden',
    'warder',
    'wardrobe',
    'warehouse',
    'warhead',
    'warmonger',
    'warning',
    'warrant',
    'warranty',
    'warren',
    'warrior',
    'warship',
    'warthog',
    'washbasin',
    'washbowl',
    'washcloth',
    'washday',
    'washer',
    'washout',
    'washroom',
    'washstand',
    'waste',
    'wasteland',
    'waster',
    'watch',
    'watchband',
    'watchdog',
    'watcher',
    'watchman',
    'watchword',
    'water',
    'waterbed',
    'waterfall',
    'waterhole',
    'waterlily',
    'waterline',
    'watermark',
    'watermill',
    'watershed',
    'waterway',
    'wattage',
    'wattle',
    'waveband',
    'waver',
    'waxwork',
    'wayfarer',
    'waylay',
    'wayside',
    'weaken',
    'weakling',
    'weakness',
    'wealthy',
    'weapon',
    'weary',
    'weasel',
    'weather',
    'weave',
    'weaver',
    'wedding',
    'wedge',
    'weedy',
    'weekday',
    'weekend',
    'weekly',
    'weeny',
    'weepy',
    'weevil',
    'weigh',
    'weight',
    'weighty',
    'weird',
    'weirdo',
    'welcome',
    'welder',
    'wench',
    'werewolf',
    'western',
    'westerner',
    'westward',
    'whack',
    'whacking',
    'whale',
    'whaler',
    'wharf',
    'wheat',
    'wheedle',
    'wheel',
    'wheelbase',
    'wheeze',
    'wheezy',
    'whelk',
    'whelp',
    'whetstone',
    'whiff',
    'while',
    'whimper',
    'whimsy',
    'whine',
    'whinny',
    'whiplash',
    'whippet',
    'whirl',
    'whirlpool',
    'whirlwind',
    'whisk',
    'whisker',
    'whiskey',
    'whisky',
    'whisper',
    'whistle',
    'white',
    'whiten',
    'whiteout',
    'whitewash',
    'whittle',
    'whodunit',
    'whole',
    'whoop',
    'whoosh',
    'whopper',
    'whore',
    'whorl',
    'wicket',
    'widen',
    'widow',
    'widower',
    'width',
    'wield',
    'wiggle',
    'wigwam',
    'wildcat',
    'willow',
    'wimple',
    'wince',
    'winch',
    'windbag',
    'windbreak',
    'windfall',
    'windlass',
    'windmill',
    'window',
    'windpipe',
    'windy',
    'winger',
    'wingspan',
    'winkle',
    'winner',
    'winning',
    'winnow',
    'winter',
    'wiper',
    'wireless',
    'wisdom',
    'wisecrack',
    'wishbone',
    'witch',
    'withdraw',
    'wither',
    'withhold',
    'withstand',
    'witness',
    'witter',
    'witticism',
    'witty',
    'wizard',
    'wobble',
    'wolfhound',
    'woman',
    'womanizer',
    'wombat',
    'wonder',
    'woodland',
    'woodlouse',
    'woodpile',
    'woodshed',
    'woodwind',
    'woodworm',
    'woody',
    'wooer',
    'woolly',
    'woozy',
    'wordy',
    'workbench',
    'workbook',
    'workday',
    'worker',
    'workforce',
    'workhorse',
    'workhouse',
    'working',
    'workload',
    'workman',
    'workmate',
    'workout',
    'workshop',
    'worktop',
    'world',
    'wormy',
    'worrier',
    'worry',
    'worsen',
    'worship',
    'worst',
    'worthy',
    'wound',
    'wraith',
    'wrangle',
    'wrapper',
    'wrapping',
    'wreak',
    'wreath',
    'wreck',
    'wrecker',
    'wrench',
    'wrest',
    'wrestle',
    'wrestler',
    'wretch',
    'wriggle',
    'wring',
    'wrinkle',
    'wrist',
    'write',
    'writer',
    'writhe',
    'writing',
    'wrong',
    'wrongdoer',
    'xylophone',
    'yacht',
    'yachtsman',
    'yammer',
    'yardstick',
    'yearbook',
    'yearn',
    'yearning',
    'yeast',
    'yellow',
    'yeoman',
    'yield',
    'yodel',
    'yogurt',
    'yokel',
    'young',
    'youngster',
    'yourself',
    'youth',
    'yuppie',
    'zealot',
    'zebra',
    'zigzag',
    'zipper',
    'zombie',
]

# Mnemonic Words {{{1
BIP39_WORDS = [
    'abandon',
    'ability',
    'able',
    'about',
    'above',
    'absent',
    'absorb',
    'abstract',
    'absurd',
    'abuse',
    'access',
    'accident',
    'account',
    'accuse',
    'achieve',
    'acid',
    'acoustic',
    'acquire',
    'across',
    'act',
    'action',
    'actor',
    'actress',
    'actual',
    'adapt',
    'add',
    'addict',
    'address',
    'adjust',
    'admit',
    'adult',
    'advance',
    'advice',
    'aerobic',
    'affair',
    'afford',
    'afraid',
    'again',
    'age',
    'agent',
    'agree',
    'ahead',
    'aim',
    'air',
    'airport',
    'aisle',
    'alarm',
    'album',
    'alcohol',
    'alert',
    'alien',
    'all',
    'alley',
    'allow',
    'almost',
    'alone',
    'alpha',
    'already',
    'also',
    'alter',
    'always',
    'amateur',
    'amazing',
    'among',
    'amount',
    'amused',
    'analyst',
    'anchor',
    'ancient',
    'anger',
    'angle',
    'angry',
    'animal',
    'ankle',
    'announce',
    'annual',
    'another',
    'answer',
    'antenna',
    'antique',
    'anxiety',
    'any',
    'apart',
    'apology',
    'appear',
    'apple',
    'approve',
    'april',
    'arch',
    'arctic',
    'area',
    'arena',
    'argue',
    'arm',
    'armed',
    'armor',
    'army',
    'around',
    'arrange',
    'arrest',
    'arrive',
    'arrow',
    'art',
    'artefact',
    'artist',
    'artwork',
    'ask',
    'aspect',
    'assault',
    'asset',
    'assist',
    'assume',
    'asthma',
    'athlete',
    'atom',
    'attack',
    'attend',
    'attitude',
    'attract',
    'auction',
    'audit',
    'august',
    'aunt',
    'author',
    'auto',
    'autumn',
    'average',
    'avocado',
    'avoid',
    'awake',
    'aware',
    'away',
    'awesome',
    'awful',
    'awkward',
    'axis',
    'baby',
    'bachelor',
    'bacon',
    'badge',
    'bag',
    'balance',
    'balcony',
    'ball',
    'bamboo',
    'banana',
    'banner',
    'bar',
    'barely',
    'bargain',
    'barrel',
    'base',
    'basic',
    'basket',
    'battle',
    'beach',
    'bean',
    'beauty',
    'because',
    'become',
    'beef',
    'before',
    'begin',
    'behave',
    'behind',
    'believe',
    'below',
    'belt',
    'bench',
    'benefit',
    'best',
    'betray',
    'better',
    'between',
    'beyond',
    'bicycle',
    'bid',
    'bike',
    'bind',
    'biology',
    'bird',
    'birth',
    'bitter',
    'black',
    'blade',
    'blame',
    'blanket',
    'blast',
    'bleak',
    'bless',
    'blind',
    'blood',
    'blossom',
    'blouse',
    'blue',
    'blur',
    'blush',
    'board',
    'boat',
    'body',
    'boil',
    'bomb',
    'bone',
    'bonus',
    'book',
    'boost',
    'border',
    'boring',
    'borrow',
    'boss',
    'bottom',
    'bounce',
    'box',
    'boy',
    'bracket',
    'brain',
    'brand',
    'brass',
    'brave',
    'bread',
    'breeze',
    'brick',
    'bridge',
    'brief',
    'bright',
    'bring',
    'brisk',
    'broccoli',
    'broken',
    'bronze',
    'broom',
    'brother',
    'brown',
    'brush',
    'bubble',
    'buddy',
    'budget',
    'buffalo',
    'build',
    'bulb',
    'bulk',
    'bullet',
    'bundle',
    'bunker',
    'burden',
    'burger',
    'burst',
    'bus',
    'business',
    'busy',
    'butter',
    'buyer',
    'buzz',
    'cabbage',
    'cabin',
    'cable',
    'cactus',
    'cage',
    'cake',
    'call',
    'calm',
    'camera',
    'camp',
    'can',
    'canal',
    'cancel',
    'candy',
    'cannon',
    'canoe',
    'canvas',
    'canyon',
    'capable',
    'capital',
    'captain',
    'car',
    'carbon',
    'card',
    'cargo',
    'carpet',
    'carry',
    'cart',
    'case',
    'cash',
    'casino',
    'castle',
    'casual',
    'cat',
    'catalog',
    'catch',
    'category',
    'cattle',
    'caught',
    'cause',
    'caution',
    'cave',
    'ceiling',
    'celery',
    'cement',
    'census',
    'century',
    'cereal',
    'certain',
    'chair',
    'chalk',
    'champion',
    'change',
    'chaos',
    'chapter',
    'charge',
    'chase',
    'chat',
    'cheap',
    'check',
    'cheese',
    'chef',
    'cherry',
    'chest',
    'chicken',
    'chief',
    'child',
    'chimney',
    'choice',
    'choose',
    'chronic',
    'chuckle',
    'chunk',
    'churn',
    'cigar',
    'cinnamon',
    'circle',
    'citizen',
    'city',
    'civil',
    'claim',
    'clap',
    'clarify',
    'claw',
    'clay',
    'clean',
    'clerk',
    'clever',
    'click',
    'client',
    'cliff',
    'climb',
    'clinic',
    'clip',
    'clock',
    'clog',
    'close',
    'cloth',
    'cloud',
    'clown',
    'club',
    'clump',
    'cluster',
    'clutch',
    'coach',
    'coast',
    'coconut',
    'code',
    'coffee',
    'coil',
    'coin',
    'collect',
    'color',
    'column',
    'combine',
    'come',
    'comfort',
    'comic',
    'common',
    'company',
    'concert',
    'conduct',
    'confirm',
    'congress',
    'connect',
    'consider',
    'control',
    'convince',
    'cook',
    'cool',
    'copper',
    'copy',
    'coral',
    'core',
    'corn',
    'correct',
    'cost',
    'cotton',
    'couch',
    'country',
    'couple',
    'course',
    'cousin',
    'cover',
    'coyote',
    'crack',
    'cradle',
    'craft',
    'cram',
    'crane',
    'crash',
    'crater',
    'crawl',
    'crazy',
    'cream',
    'credit',
    'creek',
    'crew',
    'cricket',
    'crime',
    'crisp',
    'critic',
    'crop',
    'cross',
    'crouch',
    'crowd',
    'crucial',
    'cruel',
    'cruise',
    'crumble',
    'crunch',
    'crush',
    'cry',
    'crystal',
    'cube',
    'culture',
    'cup',
    'cupboard',
    'curious',
    'current',
    'curtain',
    'curve',
    'cushion',
    'custom',
    'cute',
    'cycle',
    'dad',
    'damage',
    'damp',
    'dance',
    'danger',
    'daring',
    'dash',
    'daughter',
    'dawn',
    'day',
    'deal',
    'debate',
    'debris',
    'decade',
    'december',
    'decide',
    'decline',
    'decorate',
    'decrease',
    'deer',
    'defense',
    'define',
    'defy',
    'degree',
    'delay',
    'deliver',
    'demand',
    'demise',
    'denial',
    'dentist',
    'deny',
    'depart',
    'depend',
    'deposit',
    'depth',
    'deputy',
    'derive',
    'describe',
    'desert',
    'design',
    'desk',
    'despair',
    'destroy',
    'detail',
    'detect',
    'develop',
    'device',
    'devote',
    'diagram',
    'dial',
    'diamond',
    'diary',
    'dice',
    'diesel',
    'diet',
    'differ',
    'digital',
    'dignity',
    'dilemma',
    'dinner',
    'dinosaur',
    'direct',
    'dirt',
    'disagree',
    'discover',
    'disease',
    'dish',
    'dismiss',
    'disorder',
    'display',
    'distance',
    'divert',
    'divide',
    'divorce',
    'dizzy',
    'doctor',
    'document',
    'dog',
    'doll',
    'dolphin',
    'domain',
    'donate',
    'donkey',
    'donor',
    'door',
    'dose',
    'double',
    'dove',
    'draft',
    'dragon',
    'drama',
    'drastic',
    'draw',
    'dream',
    'dress',
    'drift',
    'drill',
    'drink',
    'drip',
    'drive',
    'drop',
    'drum',
    'dry',
    'duck',
    'dumb',
    'dune',
    'during',
    'dust',
    'dutch',
    'duty',
    'dwarf',
    'dynamic',
    'eager',
    'eagle',
    'early',
    'earn',
    'earth',
    'easily',
    'east',
    'easy',
    'echo',
    'ecology',
    'economy',
    'edge',
    'edit',
    'educate',
    'effort',
    'egg',
    'eight',
    'either',
    'elbow',
    'elder',
    'electric',
    'elegant',
    'element',
    'elephant',
    'elevator',
    'elite',
    'else',
    'embark',
    'embody',
    'embrace',
    'emerge',
    'emotion',
    'employ',
    'empower',
    'empty',
    'enable',
    'enact',
    'end',
    'endless',
    'endorse',
    'enemy',
    'energy',
    'enforce',
    'engage',
    'engine',
    'enhance',
    'enjoy',
    'enlist',
    'enough',
    'enrich',
    'enroll',
    'ensure',
    'enter',
    'entire',
    'entry',
    'envelope',
    'episode',
    'equal',
    'equip',
    'era',
    'erase',
    'erode',
    'erosion',
    'error',
    'erupt',
    'escape',
    'essay',
    'essence',
    'estate',
    'eternal',
    'ethics',
    'evidence',
    'evil',
    'evoke',
    'evolve',
    'exact',
    'example',
    'excess',
    'exchange',
    'excite',
    'exclude',
    'excuse',
    'execute',
    'exercise',
    'exhaust',
    'exhibit',
    'exile',
    'exist',
    'exit',
    'exotic',
    'expand',
    'expect',
    'expire',
    'explain',
    'expose',
    'express',
    'extend',
    'extra',
    'eye',
    'eyebrow',
    'fabric',
    'face',
    'faculty',
    'fade',
    'faint',
    'faith',
    'fall',
    'false',
    'fame',
    'family',
    'famous',
    'fan',
    'fancy',
    'fantasy',
    'farm',
    'fashion',
    'fat',
    'fatal',
    'father',
    'fatigue',
    'fault',
    'favorite',
    'feature',
    'february',
    'federal',
    'fee',
    'feed',
    'feel',
    'female',
    'fence',
    'festival',
    'fetch',
    'fever',
    'few',
    'fiber',
    'fiction',
    'field',
    'figure',
    'file',
    'film',
    'filter',
    'final',
    'find',
    'fine',
    'finger',
    'finish',
    'fire',
    'firm',
    'first',
    'fiscal',
    'fish',
    'fit',
    'fitness',
    'fix',
    'flag',
    'flame',
    'flash',
    'flat',
    'flavor',
    'flee',
    'flight',
    'flip',
    'float',
    'flock',
    'floor',
    'flower',
    'fluid',
    'flush',
    'fly',
    'foam',
    'focus',
    'fog',
    'foil',
    'fold',
    'follow',
    'food',
    'foot',
    'force',
    'forest',
    'forget',
    'fork',
    'fortune',
    'forum',
    'forward',
    'fossil',
    'foster',
    'found',
    'fox',
    'fragile',
    'frame',
    'frequent',
    'fresh',
    'friend',
    'fringe',
    'frog',
    'front',
    'frost',
    'frown',
    'frozen',
    'fruit',
    'fuel',
    'fun',
    'funny',
    'furnace',
    'fury',
    'future',
    'gadget',
    'gain',
    'galaxy',
    'gallery',
    'game',
    'gap',
    'garage',
    'garbage',
    'garden',
    'garlic',
    'garment',
    'gas',
    'gasp',
    'gate',
    'gather',
    'gauge',
    'gaze',
    'general',
    'genius',
    'genre',
    'gentle',
    'genuine',
    'gesture',
    'ghost',
    'giant',
    'gift',
    'giggle',
    'ginger',
    'giraffe',
    'girl',
    'give',
    'glad',
    'glance',
    'glare',
    'glass',
    'glide',
    'glimpse',
    'globe',
    'gloom',
    'glory',
    'glove',
    'glow',
    'glue',
    'goat',
    'goddess',
    'gold',
    'good',
    'goose',
    'gorilla',
    'gospel',
    'gossip',
    'govern',
    'gown',
    'grab',
    'grace',
    'grain',
    'grant',
    'grape',
    'grass',
    'gravity',
    'great',
    'green',
    'grid',
    'grief',
    'grit',
    'grocery',
    'group',
    'grow',
    'grunt',
    'guard',
    'guess',
    'guide',
    'guilt',
    'guitar',
    'gun',
    'gym',
    'habit',
    'hair',
    'half',
    'hammer',
    'hamster',
    'hand',
    'happy',
    'harbor',
    'hard',
    'harsh',
    'harvest',
    'hat',
    'have',
    'hawk',
    'hazard',
    'head',
    'health',
    'heart',
    'heavy',
    'hedgehog',
    'height',
    'hello',
    'helmet',
    'help',
    'hen',
    'hero',
    'hidden',
    'high',
    'hill',
    'hint',
    'hip',
    'hire',
    'history',
    'hobby',
    'hockey',
    'hold',
    'hole',
    'holiday',
    'hollow',
    'home',
    'honey',
    'hood',
    'hope',
    'horn',
    'horror',
    'horse',
    'hospital',
    'host',
    'hotel',
    'hour',
    'hover',
    'hub',
    'huge',
    'human',
    'humble',
    'humor',
    'hundred',
    'hungry',
    'hunt',
    'hurdle',
    'hurry',
    'hurt',
    'husband',
    'hybrid',
    'ice',
    'icon',
    'idea',
    'identify',
    'idle',
    'ignore',
    'ill',
    'illegal',
    'illness',
    'image',
    'imitate',
    'immense',
    'immune',
    'impact',
    'impose',
    'improve',
    'impulse',
    'inch',
    'include',
    'income',
    'increase',
    'index',
    'indicate',
    'indoor',
    'industry',
    'infant',
    'inflict',
    'inform',
    'inhale',
    'inherit',
    'initial',
    'inject',
    'injury',
    'inmate',
    'inner',
    'innocent',
    'input',
    'inquiry',
    'insane',
    'insect',
    'inside',
    'inspire',
    'install',
    'intact',
    'interest',
    'into',
    'invest',
    'invite',
    'involve',
    'iron',
    'island',
    'isolate',
    'issue',
    'item',
    'ivory',
    'jacket',
    'jaguar',
    'jar',
    'jazz',
    'jealous',
    'jeans',
    'jelly',
    'jewel',
    'job',
    'join',
    'joke',
    'journey',
    'joy',
    'judge',
    'juice',
    'jump',
    'jungle',
    'junior',
    'junk',
    'just',
    'kangaroo',
    'keen',
    'keep',
    'ketchup',
    'key',
    'kick',
    'kid',
    'kidney',
    'kind',
    'kingdom',
    'kiss',
    'kit',
    'kitchen',
    'kite',
    'kitten',
    'kiwi',
    'knee',
    'knife',
    'knock',
    'know',
    'lab',
    'label',
    'labor',
    'ladder',
    'lady',
    'lake',
    'lamp',
    'language',
    'laptop',
    'large',
    'later',
    'latin',
    'laugh',
    'laundry',
    'lava',
    'law',
    'lawn',
    'lawsuit',
    'layer',
    'lazy',
    'leader',
    'leaf',
    'learn',
    'leave',
    'lecture',
    'left',
    'leg',
    'legal',
    'legend',
    'leisure',
    'lemon',
    'lend',
    'length',
    'lens',
    'leopard',
    'lesson',
    'letter',
    'level',
    'liar',
    'liberty',
    'library',
    'license',
    'life',
    'lift',
    'light',
    'like',
    'limb',
    'limit',
    'link',
    'lion',
    'liquid',
    'list',
    'little',
    'live',
    'lizard',
    'load',
    'loan',
    'lobster',
    'local',
    'lock',
    'logic',
    'lonely',
    'long',
    'loop',
    'lottery',
    'loud',
    'lounge',
    'love',
    'loyal',
    'lucky',
    'luggage',
    'lumber',
    'lunar',
    'lunch',
    'luxury',
    'lyrics',
    'machine',
    'mad',
    'magic',
    'magnet',
    'maid',
    'mail',
    'main',
    'major',
    'make',
    'mammal',
    'man',
    'manage',
    'mandate',
    'mango',
    'mansion',
    'manual',
    'maple',
    'marble',
    'march',
    'margin',
    'marine',
    'market',
    'marriage',
    'mask',
    'mass',
    'master',
    'match',
    'material',
    'math',
    'matrix',
    'matter',
    'maximum',
    'maze',
    'meadow',
    'mean',
    'measure',
    'meat',
    'mechanic',
    'medal',
    'media',
    'melody',
    'melt',
    'member',
    'memory',
    'mention',
    'menu',
    'mercy',
    'merge',
    'merit',
    'merry',
    'mesh',
    'message',
    'metal',
    'method',
    'middle',
    'midnight',
    'milk',
    'million',
    'mimic',
    'mind',
    'minimum',
    'minor',
    'minute',
    'miracle',
    'mirror',
    'misery',
    'miss',
    'mistake',
    'mix',
    'mixed',
    'mixture',
    'mobile',
    'model',
    'modify',
    'mom',
    'moment',
    'monitor',
    'monkey',
    'monster',
    'month',
    'moon',
    'moral',
    'more',
    'morning',
    'mosquito',
    'mother',
    'motion',
    'motor',
    'mountain',
    'mouse',
    'move',
    'movie',
    'much',
    'muffin',
    'mule',
    'multiply',
    'muscle',
    'museum',
    'mushroom',
    'music',
    'must',
    'mutual',
    'myself',
    'mystery',
    'myth',
    'naive',
    'name',
    'napkin',
    'narrow',
    'nasty',
    'nation',
    'nature',
    'near',
    'neck',
    'need',
    'negative',
    'neglect',
    'neither',
    'nephew',
    'nerve',
    'nest',
    'net',
    'network',
    'neutral',
    'never',
    'news',
    'next',
    'nice',
    'night',
    'noble',
    'noise',
    'nominee',
    'noodle',
    'normal',
    'north',
    'nose',
    'notable',
    'note',
    'nothing',
    'notice',
    'novel',
    'now',
    'nuclear',
    'number',
    'nurse',
    'nut',
    'oak',
    'obey',
    'object',
    'oblige',
    'obscure',
    'observe',
    'obtain',
    'obvious',
    'occur',
    'ocean',
    'october',
    'odor',
    'off',
    'offer',
    'office',
    'often',
    'oil',
    'okay',
    'old',
    'olive',
    'olympic',
    'omit',
    'once',
    'one',
    'onion',
    'online',
    'only',
    'open',
    'opera',
    'opinion',
    'oppose',
    'option',
    'orange',
    'orbit',
    'orchard',
    'order',
    'ordinary',
    'organ',
    'orient',
    'original',
    'orphan',
    'ostrich',
    'other',
    'outdoor',
    'outer',
    'output',
    'outside',
    'oval',
    'oven',
    'over',
    'own',
    'owner',
    'oxygen',
    'oyster',
    'ozone',
    'pact',
    'paddle',
    'page',
    'pair',
    'palace',
    'palm',
    'panda',
    'panel',
    'panic',
    'panther',
    'paper',
    'parade',
    'parent',
    'park',
    'parrot',
    'party',
    'pass',
    'patch',
    'path',
    'patient',
    'patrol',
    'pattern',
    'pause',
    'pave',
    'payment',
    'peace',
    'peanut',
    'pear',
    'peasant',
    'pelican',
    'pen',
    'penalty',
    'pencil',
    'people',
    'pepper',
    'perfect',
    'permit',
    'person',
    'pet',
    'phone',
    'photo',
    'phrase',
    'physical',
    'piano',
    'picnic',
    'picture',
    'piece',
    'pig',
    'pigeon',
    'pill',
    'pilot',
    'pink',
    'pioneer',
    'pipe',
    'pistol',
    'pitch',
    'pizza',
    'place',
    'planet',
    'plastic',
    'plate',
    'play',
    'please',
    'pledge',
    'pluck',
    'plug',
    'plunge',
    'poem',
    'poet',
    'point',
    'polar',
    'pole',
    'police',
    'pond',
    'pony',
    'pool',
    'popular',
    'portion',
    'position',
    'possible',
    'post',
    'potato',
    'pottery',
    'poverty',
    'powder',
    'power',
    'practice',
    'praise',
    'predict',
    'prefer',
    'prepare',
    'present',
    'pretty',
    'prevent',
    'price',
    'pride',
    'primary',
    'print',
    'priority',
    'prison',
    'private',
    'prize',
    'problem',
    'process',
    'produce',
    'profit',
    'program',
    'project',
    'promote',
    'proof',
    'property',
    'prosper',
    'protect',
    'proud',
    'provide',
    'public',
    'pudding',
    'pull',
    'pulp',
    'pulse',
    'pumpkin',
    'punch',
    'pupil',
    'puppy',
    'purchase',
    'purity',
    'purpose',
    'purse',
    'push',
    'put',
    'puzzle',
    'pyramid',
    'quality',
    'quantum',
    'quarter',
    'question',
    'quick',
    'quit',
    'quiz',
    'quote',
    'rabbit',
    'raccoon',
    'race',
    'rack',
    'radar',
    'radio',
    'rail',
    'rain',
    'raise',
    'rally',
    'ramp',
    'ranch',
    'random',
    'range',
    'rapid',
    'rare',
    'rate',
    'rather',
    'raven',
    'raw',
    'razor',
    'ready',
    'real',
    'reason',
    'rebel',
    'rebuild',
    'recall',
    'receive',
    'recipe',
    'record',
    'recycle',
    'reduce',
    'reflect',
    'reform',
    'refuse',
    'region',
    'regret',
    'regular',
    'reject',
    'relax',
    'release',
    'relief',
    'rely',
    'remain',
    'remember',
    'remind',
    'remove',
    'render',
    'renew',
    'rent',
    'reopen',
    'repair',
    'repeat',
    'replace',
    'report',
    'require',
    'rescue',
    'resemble',
    'resist',
    'resource',
    'response',
    'result',
    'retire',
    'retreat',
    'return',
    'reunion',
    'reveal',
    'review',
    'reward',
    'rhythm',
    'rib',
    'ribbon',
    'rice',
    'rich',
    'ride',
    'ridge',
    'rifle',
    'right',
    'rigid',
    'ring',
    'riot',
    'ripple',
    'risk',
    'ritual',
    'rival',
    'river',
    'road',
    'roast',
    'robot',
    'robust',
    'rocket',
    'romance',
    'roof',
    'rookie',
    'room',
    'rose',
    'rotate',
    'rough',
    'round',
    'route',
    'royal',
    'rubber',
    'rude',
    'rug',
    'rule',
    'run',
    'runway',
    'rural',
    'sad',
    'saddle',
    'sadness',
    'safe',
    'sail',
    'salad',
    'salmon',
    'salon',
    'salt',
    'salute',
    'same',
    'sample',
    'sand',
    'satisfy',
    'satoshi',
    'sauce',
    'sausage',
    'save',
    'say',
    'scale',
    'scan',
    'scare',
    'scatter',
    'scene',
    'scheme',
    'school',
    'science',
    'scissors',
    'scorpion',
    'scout',
    'scrap',
    'screen',
    'script',
    'scrub',
    'sea',
    'search',
    'season',
    'seat',
    'second',
    'secret',
    'section',
    'security',
    'seed',
    'seek',
    'segment',
    'select',
    'sell',
    'seminar',
    'senior',
    'sense',
    'sentence',
    'series',
    'service',
    'session',
    'settle',
    'setup',
    'seven',
    'shadow',
    'shaft',
    'shallow',
    'share',
    'shed',
    'shell',
    'sheriff',
    'shield',
    'shift',
    'shine',
    'ship',
    'shiver',
    'shock',
    'shoe',
    'shoot',
    'shop',
    'short',
    'shoulder',
    'shove',
    'shrimp',
    'shrug',
    'shuffle',
    'shy',
    'sibling',
    'sick',
    'side',
    'siege',
    'sight',
    'sign',
    'silent',
    'silk',
    'silly',
    'silver',
    'similar',
    'simple',
    'since',
    'sing',
    'siren',
    'sister',
    'situate',
    'six',
    'size',
    'skate',
    'sketch',
    'ski',
    'skill',
    'skin',
    'skirt',
    'skull',
    'slab',
    'slam',
    'sleep',
    'slender',
    'slice',
    'slide',
    'slight',
    'slim',
    'slogan',
    'slot',
    'slow',
    'slush',
    'small',
    'smart',
    'smile',
    'smoke',
    'smooth',
    'snack',
    'snake',
    'snap',
    'sniff',
    'snow',
    'soap',
    'soccer',
    'social',
    'sock',
    'soda',
    'soft',
    'solar',
    'soldier',
    'solid',
    'solution',
    'solve',
    'someone',
    'song',
    'soon',
    'sorry',
    'sort',
    'soul',
    'sound',
    'soup',
    'source',
    'south',
    'space',
    'spare',
    'spatial',
    'spawn',
    'speak',
    'special',
    'speed',
    'spell',
    'spend',
    'sphere',
    'spice',
    'spider',
    'spike',
    'spin',
    'spirit',
    'split',
    'spoil',
    'sponsor',
    'spoon',
    'sport',
    'spot',
    'spray',
    'spread',
    'spring',
    'spy',
    'square',
    'squeeze',
    'squirrel',
    'stable',
    'stadium',
    'staff',
    'stage',
    'stairs',
    'stamp',
    'stand',
    'start',
    'state',
    'stay',
    'steak',
    'steel',
    'stem',
    'step',
    'stereo',
    'stick',
    'still',
    'sting',
    'stock',
    'stomach',
    'stone',
    'stool',
    'story',
    'stove',
    'strategy',
    'street',
    'strike',
    'strong',
    'struggle',
    'student',
    'stuff',
    'stumble',
    'style',
    'subject',
    'submit',
    'subway',
    'success',
    'such',
    'sudden',
    'suffer',
    'sugar',
    'suggest',
    'suit',
    'summer',
    'sun',
    'sunny',
    'sunset',
    'super',
    'supply',
    'supreme',
    'sure',
    'surface',
    'surge',
    'surprise',
    'surround',
    'survey',
    'suspect',
    'sustain',
    'swallow',
    'swamp',
    'swap',
    'swarm',
    'swear',
    'sweet',
    'swift',
    'swim',
    'swing',
    'switch',
    'sword',
    'symbol',
    'symptom',
    'syrup',
    'system',
    'table',
    'tackle',
    'tag',
    'tail',
    'talent',
    'talk',
    'tank',
    'tape',
    'target',
    'task',
    'taste',
    'tattoo',
    'taxi',
    'teach',
    'team',
    'tell',
    'ten',
    'tenant',
    'tennis',
    'tent',
    'term',
    'test',
    'text',
    'thank',
    'that',
    'theme',
    'then',
    'theory',
    'there',
    'they',
    'thing',
    'this',
    'thought',
    'three',
    'thrive',
    'throw',
    'thumb',
    'thunder',
    'ticket',
    'tide',
    'tiger',
    'tilt',
    'timber',
    'time',
    'tiny',
    'tip',
    'tired',
    'tissue',
    'title',
    'toast',
    'tobacco',
    'today',
    'toddler',
    'toe',
    'together',
    'toilet',
    'token',
    'tomato',
    'tomorrow',
    'tone',
    'tongue',
    'tonight',
    'tool',
    'tooth',
    'top',
    'topic',
    'topple',
    'torch',
    'tornado',
    'tortoise',
    'toss',
    'total',
    'tourist',
    'toward',
    'tower',
    'town',
    'toy',
    'track',
    'trade',
    'traffic',
    'tragic',
    'train',
    'transfer',
    'trap',
    'trash',
    'travel',
    'tray',
    'treat',
    'tree',
    'trend',
    'trial',
    'tribe',
    'trick',
    'trigger',
    'trim',
    'trip',
    'trophy',
    'trouble',
    'truck',
    'true',
    'truly',
    'trumpet',
    'trust',
    'truth',
    'try',
    'tube',
    'tuition',
    'tumble',
    'tuna',
    'tunnel',
    'turkey',
    'turn',
    'turtle',
    'twelve',
    'twenty',
    'twice',
    'twin',
    'twist',
    'two',
    'type',
    'typical',
    'ugly',
    'umbrella',
    'unable',
    'unaware',
    'uncle',
    'uncover',
    'under',
    'undo',
    'unfair',
    'unfold',
    'unhappy',
    'uniform',
    'unique',
    'unit',
    'universe',
    'unknown',
    'unlock',
    'until',
    'unusual',
    'unveil',
    'update',
    'upgrade',
    'uphold',
    'upon',
    'upper',
    'upset',
    'urban',
    'urge',
    'usage',
    'use',
    'used',
    'useful',
    'useless',
    'usual',
    'utility',
    'vacant',
    'vacuum',
    'vague',
    'valid',
    'valley',
    'valve',
    'van',
    'vanish',
    'vapor',
    'various',
    'vast',
    'vault',
    'vehicle',
    'velvet',
    'vendor',
    'venture',
    'venue',
    'verb',
    'verify',
    'version',
    'very',
    'vessel',
    'veteran',
    'viable',
    'vibrant',
    'vicious',
    'victory',
    'video',
    'view',
    'village',
    'vintage',
    'violin',
    'virtual',
    'virus',
    'visa',
    'visit',
    'visual',
    'vital',
    'vivid',
    'vocal',
    'voice',
    'void',
    'volcano',
    'volume',
    'vote',
    'voyage',
    'wage',
    'wagon',
    'wait',
    'walk',
    'wall',
    'walnut',
    'want',
    'warfare',
    'warm',
    'warrior',
    'wash',
    'wasp',
    'waste',
    'water',
    'wave',
    'way',
    'wealth',
    'weapon',
    'wear',
    'weasel',
    'weather',
    'web',
    'wedding',
    'weekend',
    'weird',
    'welcome',
    'west',
    'wet',
    'whale',
    'what',
    'wheat',
    'wheel',
    'when',
    'where',
    'whip',
    'whisper',
    'wide',
    'width',
    'wife',
    'wild',
    'will',
    'win',
    'window',
    'wine',
    'wing',
    'wink',
    'winner',
    'winter',
    'wire',
    'wisdom',
    'wise',
    'wish',
    'witness',
    'wolf',
    'woman',
    'wonder',
    'wood',
    'wool',
    'word',
    'work',
    'world',
    'worry',
    'worth',
    'wrap',
    'wreck',
    'wrestle',
    'wrist',
    'write',
    'wrong',
    'yard',
    'year',
    'yellow',
    'you',
    'young',
    'youth',
    'zebra',
    'zero',
    'zone',
    'zoo',
]
