"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJsonPath = exports.renderList = exports.State = void 0;
const cdk = require("@aws-cdk/core");
const fields_1 = require("../fields");
const types_1 = require("../types");
/**
 * Base class for all other state classes
 */
class State extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.branches = [];
        this.retries = [];
        this.catches = [];
        this.choices = [];
        this.prefixes = [];
        /**
         * States with references to this state.
         *
         * Used for finding complete connected graph that a state is part of.
         */
        this.incomingStates = [];
        this.startState = this;
        this.comment = props.comment;
        this.inputPath = props.inputPath;
        this.parameters = props.parameters;
        this.outputPath = props.outputPath;
        this.resultPath = props.resultPath;
    }
    /**
     * Add a prefix to the stateId of all States found in a construct tree
     */
    static prefixStates(root, prefix) {
        const queue = [root];
        while (queue.length > 0) {
            const el = queue.splice(0, 1)[0];
            if (isPrefixable(el)) {
                el.addPrefix(prefix);
            }
            queue.push(...el.node.children);
        }
    }
    /**
     * Find the set of states reachable through transitions from the given start state.
     * This does not retrieve states from within sub-graphs, such as states within a Parallel state's branch.
     */
    static findReachableStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            queue.push(...outgoing);
            ret.add(state);
        }
        return Array.from(ret);
    }
    /**
     * Find the set of end states states reachable through transitions from the given start state
     */
    static findReachableEndStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            if (outgoing.length > 0) {
                // We can continue
                queue.push(...outgoing);
            }
            else {
                // Terminal state
                ret.add(state);
            }
        }
        return Array.from(ret);
    }
    /**
     * Return only the states that allow chaining from an array of states
     */
    static filterNextables(states) {
        return states.filter(isNextable);
    }
    get id() {
        return this.node.id;
    }
    /**
     * Tokenized string that evaluates to the state's ID
     */
    get stateId() {
        return this.prefixes.concat(this.id).join('');
    }
    /**
     * Add a prefix to the stateId of this state
     */
    addPrefix(x) {
        if (x !== '') {
            this.prefixes.splice(0, 0, x);
        }
    }
    /**
     * Register this state as part of the given graph
     *
     * Don't call this. It will be called automatically when you work
     * with states normally.
     */
    bindToGraph(graph) {
        if (this.containingGraph === graph) {
            return;
        }
        if (this.containingGraph) {
            // eslint-disable-next-line max-len
            throw new Error(`Trying to use state '${this.stateId}' in ${graph}, but is already in ${this.containingGraph}. Every state can only be used in one graph.`);
        }
        this.containingGraph = graph;
        this.whenBoundToGraph(graph);
        for (const incoming of this.incomingStates) {
            incoming.bindToGraph(graph);
        }
        for (const outgoing of this.outgoingTransitions({ includeErrorHandlers: true })) {
            outgoing.bindToGraph(graph);
        }
        for (const branch of this.branches) {
            branch.registerSuperGraph(this.containingGraph);
        }
        if (!!this.iteration) {
            this.iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a retrier to the retry list of this state
     * @internal
     */
    _addRetry(props = {}) {
        this.retries.push({
            ...props,
            errors: props.errors ? props.errors : [types_1.Errors.ALL],
        });
    }
    /**
     * Add an error handler to the catch list of this state
     * @internal
     */
    _addCatch(handler, props = {}) {
        this.catches.push({
            next: handler,
            props: {
                errors: props.errors ? props.errors : [types_1.Errors.ALL],
                resultPath: props.resultPath,
            },
        });
        handler.addIncoming(this);
        if (this.containingGraph) {
            handler.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default transition of this state
     */
    makeNext(next) {
        // Can't be called 'setNext' because of JSII
        if (this._next) {
            throw new Error(`State '${this.id}' already has a next state`);
        }
        this._next = next;
        next.addIncoming(this);
        if (this.containingGraph) {
            next.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a choice branch to this state
     */
    addChoice(condition, next) {
        this.choices.push({ condition, next });
        next.startState.addIncoming(this);
        if (this.containingGraph) {
            next.startState.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a paralle branch to this state
     */
    addBranch(branch) {
        this.branches.push(branch);
        if (this.containingGraph) {
            branch.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a map iterator to this state
     */
    addIterator(iteration) {
        this.iteration = iteration;
        if (this.containingGraph) {
            iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default choice transition of this state
     */
    makeDefault(def) {
        // Can't be called 'setDefault' because of JSII
        if (this.defaultChoice) {
            throw new Error(`Choice '${this.id}' already has a default next state`);
        }
        this.defaultChoice = def;
    }
    /**
     * Render the default next state in ASL JSON format
     */
    renderNextEnd() {
        if (this._next) {
            return { Next: this._next.stateId };
        }
        else {
            return { End: true };
        }
    }
    /**
     * Render the choices in ASL JSON format
     */
    renderChoices() {
        return {
            Choices: renderList(this.choices, renderChoice),
            Default: this.defaultChoice ? this.defaultChoice.stateId : undefined,
        };
    }
    /**
     * Render InputPath/Parameters/OutputPath in ASL JSON format
     */
    renderInputOutput() {
        return {
            InputPath: renderJsonPath(this.inputPath),
            Parameters: this.parameters,
            OutputPath: renderJsonPath(this.outputPath),
        };
    }
    /**
     * Render parallel branches in ASL JSON format
     */
    renderBranches() {
        return {
            Branches: this.branches.map(b => b.toGraphJson()),
        };
    }
    /**
     * Render map iterator in ASL JSON format
     */
    renderIterator() {
        if (!this.iteration) {
            throw new Error('Iterator must not be undefined !');
        }
        return {
            Iterator: this.iteration.toGraphJson(),
        };
    }
    /**
     * Render error recovery options in ASL JSON format
     */
    renderRetryCatch() {
        return {
            Retry: renderList(this.retries, renderRetry),
            Catch: renderList(this.catches, renderCatch),
        };
    }
    /**
     * Called whenever this state is bound to a graph
     *
     * Can be overridden by subclasses.
     */
    whenBoundToGraph(graph) {
        graph.registerState(this);
    }
    /**
     * Add a state to the incoming list
     */
    addIncoming(source) {
        this.incomingStates.push(source);
    }
    /**
     * Return all states this state can transition to
     */
    outgoingTransitions(options) {
        const ret = new Array();
        if (this._next) {
            ret.push(this._next);
        }
        if (this.defaultChoice) {
            ret.push(this.defaultChoice);
        }
        for (const c of this.choices) {
            ret.push(c.next);
        }
        if (options.includeErrorHandlers) {
            for (const c of this.catches) {
                ret.push(c.next);
            }
        }
        return ret;
    }
}
exports.State = State;
/**
 * Render a choice transition
 */
function renderChoice(c) {
    return {
        ...c.condition.renderCondition(),
        Next: c.next.stateId,
    };
}
/**
 * Render a Retry object to ASL
 */
function renderRetry(retry) {
    return {
        ErrorEquals: retry.errors,
        IntervalSeconds: retry.interval && retry.interval.toSeconds(),
        MaxAttempts: retry.maxAttempts,
        BackoffRate: retry.backoffRate,
    };
}
/**
 * Render a Catch object to ASL
 */
function renderCatch(c) {
    return {
        ErrorEquals: c.props.errors,
        ResultPath: renderJsonPath(c.props.resultPath),
        Next: c.next.stateId,
    };
}
/**
 * Render a list or return undefined for an empty list
 */
function renderList(xs, fn) {
    if (xs.length === 0) {
        return undefined;
    }
    return xs.map(fn);
}
exports.renderList = renderList;
/**
 * Render JSON path, respecting the special value DISCARD
 */
function renderJsonPath(jsonPath) {
    if (jsonPath === undefined) {
        return undefined;
    }
    if (jsonPath === fields_1.JsonPath.DISCARD) {
        return null;
    }
    if (!jsonPath.startsWith('$')) {
        throw new Error(`Expected JSON path to start with '$', got: ${jsonPath}`);
    }
    return jsonPath;
}
exports.renderJsonPath = renderJsonPath;
/**
 * Whether an object is a Prefixable
 */
function isPrefixable(x) {
    return typeof (x) === 'object' && x.addPrefix;
}
/**
 * Whether an object is INextable
 */
function isNextable(x) {
    return typeof (x) === 'object' && x.next;
}
//# sourceMappingURL=data:application/json;base64,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