"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.jsonPathString = exports.recurseObject = exports.findReferencedPaths = exports.renderObject = exports.JsonPathToken = void 0;
const core_1 = require("@aws-cdk/core");
const JSON_PATH_TOKEN_SYMBOL = Symbol.for('@aws-cdk/aws-stepfunctions.JsonPathToken');
class JsonPathToken {
    constructor(path) {
        this.path = path;
        this.creationStack = core_1.captureStackTrace();
        this.displayHint = path.replace(/^[^a-zA-Z]+/, '');
        Object.defineProperty(this, JSON_PATH_TOKEN_SYMBOL, { value: true });
    }
    static isJsonPathToken(x) {
        return x[JSON_PATH_TOKEN_SYMBOL] === true;
    }
    resolve(_ctx) {
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.JsonPathToken = JsonPathToken;
/**
 * Deep render a JSON object to expand JSON path fields, updating the key to end in '.$'
 */
function renderObject(obj) {
    return recurseObject(obj, {
        handleString: renderString,
        handleList: renderStringList,
        handleNumber: renderNumber,
        handleBoolean: renderBoolean,
    });
}
exports.renderObject = renderObject;
/**
 * Return all JSON paths that are used in the given structure
 */
function findReferencedPaths(obj) {
    const found = new Set();
    recurseObject(obj, {
        handleString(_key, x) {
            const path = jsonPathString(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleList(_key, x) {
            const path = jsonPathStringList(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleNumber(_key, x) {
            const path = jsonPathNumber(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleBoolean(_key, _x) {
            return {};
        },
    });
    return found;
}
exports.findReferencedPaths = findReferencedPaths;
function recurseObject(obj, handlers) {
    if (obj === undefined) {
        return undefined;
    }
    const ret = {};
    for (const [key, value] of Object.entries(obj)) {
        if (typeof value === 'string') {
            Object.assign(ret, handlers.handleString(key, value));
        }
        else if (typeof value === 'number') {
            Object.assign(ret, handlers.handleNumber(key, value));
        }
        else if (Array.isArray(value)) {
            Object.assign(ret, recurseArray(key, value, handlers));
        }
        else if (typeof value === 'boolean') {
            Object.assign(ret, handlers.handleBoolean(key, value));
        }
        else if (value === null || value === undefined) {
            // Nothing
        }
        else if (typeof value === 'object') {
            ret[key] = recurseObject(value, handlers);
        }
    }
    return ret;
}
exports.recurseObject = recurseObject;
/**
 * Render an array that may or may not contain a string list token
 */
function recurseArray(key, arr, handlers) {
    if (isStringArray(arr)) {
        const path = jsonPathStringList(arr);
        if (path !== undefined) {
            return handlers.handleList(key, arr);
        }
        // Fall through to correctly reject encoded strings inside an array.
        // They cannot be represented because there is no key to append a '.$' to.
    }
    return {
        [key]: arr.map(value => {
            if ((typeof value === 'string' && jsonPathString(value) !== undefined)
                || (typeof value === 'number' && jsonPathNumber(value) !== undefined)
                || (isStringArray(value) && jsonPathStringList(value) !== undefined)) {
                throw new Error('Cannot use JsonPath fields in an array, they must be used in objects');
            }
            if (typeof value === 'object' && value !== null) {
                return recurseObject(value, handlers);
            }
            return value;
        }),
    };
}
function isStringArray(x) {
    return Array.isArray(x) && x.every(el => typeof el === 'string');
}
/**
 * Render a parameter string
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderString(key, value) {
    const path = jsonPathString(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter string list
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderStringList(key, value) {
    const path = jsonPathStringList(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter number
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderNumber(key, value) {
    const path = jsonPathNumber(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter boolean
 */
function renderBoolean(key, value) {
    return { [key]: value };
}
/**
 * If the indicated string is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathString(x) {
    const fragments = core_1.Tokenization.reverseString(x);
    const jsonPathTokens = fragments.tokens.filter(JsonPathToken.isJsonPathToken);
    if (jsonPathTokens.length > 0 && fragments.length > 1) {
        throw new Error(`Field references must be the entire string, cannot concatenate them (found '${x}')`);
    }
    if (jsonPathTokens.length > 0) {
        return jsonPathTokens[0].path;
    }
    return undefined;
}
exports.jsonPathString = jsonPathString;
/**
 * If the indicated string list is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathStringList(x) {
    return pathFromToken(core_1.Tokenization.reverseList(x));
}
/**
 * If the indicated number is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathNumber(x) {
    return pathFromToken(core_1.Tokenization.reverseNumber(x));
}
function pathFromToken(token) {
    return token && (JsonPathToken.isJsonPathToken(token) ? token.path : undefined);
}
//# sourceMappingURL=data:application/json;base64,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