"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FieldUtils = exports.Context = exports.Data = exports.JsonPath = void 0;
const core_1 = require("@aws-cdk/core");
const json_path_1 = require("./json-path");
/**
 * Extract a field from the State Machine data or context
 * that gets passed around between states
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-paths.html
 */
class JsonPath {
    constructor() { }
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateJsonPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal string list, get the value from a JSON path
     */
    static listAt(path) {
        // does not apply to task context
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateJsonPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Use the entire data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entirePayload() {
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * Determines if the indicated string is an encoded JSON path
     *
     * @param value string to be evaluated
     */
    static isEncodedJsonPath(value) {
        return !!json_path_1.jsonPathString(value);
    }
    /**
     * Return the Task Token field
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     */
    static get taskToken() {
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * Use the entire context data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entireContext() {
        return new json_path_1.JsonPathToken('$$').toString();
    }
}
exports.JsonPath = JsonPath;
/**
 * Special string value to discard state input, output or result
 */
JsonPath.DISCARD = 'DISCARD';
/**
 * Extract a field from the State Machine data that gets passed around between states
 *
 * @deprecated replaced by `JsonPath`
 */
class Data {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateDataPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal string list, get the value from a JSON path
     */
    static listAt(path) {
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateDataPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Use the entire data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entirePayload() {
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * Determines if the indicated string is an encoded JSON path
     *
     * @param value string to be evaluated
     */
    static isJsonPathString(value) {
        return !!json_path_1.jsonPathString(value);
    }
    constructor() { }
}
exports.Data = Data;
/**
 * Extract a field from the State Machine Context data
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#wait-token-contextobject
 *
 * @deprecated replaced by `JsonPath`
 */
class Context {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateContextPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateContextPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Return the Task Token field
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     */
    static get taskToken() {
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * Use the entire context data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entireContext() {
        return new json_path_1.JsonPathToken('$$').toString();
    }
    constructor() { }
}
exports.Context = Context;
/**
 * Helper functions to work with structures containing fields
 */
class FieldUtils {
    /**
     * Render a JSON structure containing fields to the right StepFunctions structure
     */
    static renderObject(obj) {
        return json_path_1.renderObject(obj);
    }
    /**
     * Return all JSON paths used in the given structure
     */
    static findReferencedPaths(obj) {
        return Array.from(json_path_1.findReferencedPaths(obj)).sort();
    }
    /**
     * Returns whether the given task structure contains the TaskToken field anywhere
     *
     * The field is considered included if the field itself or one of its containing
     * fields occurs anywhere in the payload.
     */
    static containsTaskToken(obj) {
        const paths = json_path_1.findReferencedPaths(obj);
        return paths.has('$$.Task.Token') || paths.has('$$.Task') || paths.has('$$');
    }
    constructor() { }
}
exports.FieldUtils = FieldUtils;
function validateJsonPath(path) {
    if (path !== '$' && !path.startsWith('$.') && path !== '$$' && !path.startsWith('$$.')) {
        throw new Error(`JSON path values must be exactly '$', '$$', start with '$.' or start with '$$.' Received: ${path}`);
    }
}
function validateDataPath(path) {
    if (path !== '$' && !path.startsWith('$.')) {
        throw new Error("Data JSON path values must either be exactly equal to '$' or start with '$.'");
    }
}
function validateContextPath(path) {
    if (path !== '$$' && !path.startsWith('$$.')) {
        throw new Error("Context JSON path values must either be exactly equal to '$$' or start with '$$.'");
    }
}
//# sourceMappingURL=data:application/json;base64,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