"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_applicationautoscaling_1 = require("monocdk/aws-applicationautoscaling");
const aws_dynamodb_1 = require("monocdk/aws-dynamodb");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * Provisioned capacity DynamoDB tables have auto-scaling enabled on their indexes
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_dynamodb_1.CfnTable) {
        if (nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.billingMode) !==
            aws_dynamodb_1.BillingMode.PAY_PER_REQUEST) {
            const indexWriteMatches = [''];
            const indexReadMatches = [''];
            const tableLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
            const tableName = monocdk_1.Stack.of(node).resolve(node.tableName);
            const globalSecondaryIndexes = monocdk_1.Stack.of(node).resolve(node.globalSecondaryIndexes);
            if (Array.isArray(globalSecondaryIndexes)) {
                globalSecondaryIndexes.forEach((gsi) => {
                    const resolvedGsi = monocdk_1.Stack.of(node).resolve(gsi);
                    indexWriteMatches.push(resolvedGsi.indexName);
                    indexReadMatches.push(resolvedGsi.indexName);
                });
            }
            for (const child of monocdk_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_applicationautoscaling_1.CfnScalableTarget) {
                    const writeMatchIndex = isMatchingScalableTarget(child, 'WriteCapacityUnits', tableLogicalId, tableName, indexWriteMatches);
                    if (writeMatchIndex !== -1) {
                        indexWriteMatches.splice(writeMatchIndex, 1);
                        continue;
                    }
                    const readMatchIndex = isMatchingScalableTarget(child, 'ReadCapacityUnits', tableLogicalId, tableName, indexReadMatches);
                    if (readMatchIndex !== -1) {
                        indexReadMatches.splice(readMatchIndex, 1);
                    }
                }
            }
            if (indexWriteMatches.length != 0 || indexReadMatches.length != 0) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the CfnScalableTarget is related to the given Table index
 * @param node the CfnScalableTarget to check
 * @param dimension the dimension of the CfnScalableTarget to check
 * @param tableLogicalId the Cfn Logical ID of the table
 * @param tableName the name of the table
 * @param indexNames the names of the indexes to check against
 * returns the location in the indexNames array of the matching index or -1 if no match is found
 */
function isMatchingScalableTarget(node, dimension, tableLogicalId, tableName, indexNames) {
    if (node.serviceNamespace === 'dynamodb') {
        let scalableDimension = '';
        const resourceId = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceId));
        for (let i = 0; i < indexNames.length; i++) {
            const regexes = Array();
            const indexName = indexNames[i];
            if (indexName === '') {
                regexes.push(`${tableLogicalId}.*`);
                if (tableName !== undefined) {
                    regexes.push(`${tableName}.*`);
                }
                scalableDimension = `dynamodb:table:${dimension}`;
            }
            else {
                regexes.push(`${tableLogicalId}.*index\/${indexName}(?![\\w\\-_\\.])`);
                if (tableName !== undefined) {
                    regexes.push(`${tableName}.*index\/${indexName}(?![\\w\\-_\\.])`);
                }
                scalableDimension = `dynamodb:index:${dimension}`;
            }
            const regex = new RegExp(regexes.join('|'), 'gm');
            if (node.scalableDimension === scalableDimension &&
                regex.test(resourceId)) {
                return i;
            }
        }
    }
    return -1;
}
//# sourceMappingURL=data:application/json;base64,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