"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cloudfront_1 = require("monocdk/aws-cloudfront");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * CloudFront distributions use an origin access identity for S3 origins
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_cloudfront_1.CfnDistribution) {
        const distributionConfig = monocdk_1.Stack.of(node).resolve(node.distributionConfig);
        if (distributionConfig.origins != undefined) {
            const origins = monocdk_1.Stack.of(node).resolve(distributionConfig.origins);
            for (const origin of origins) {
                const resolvedOrigin = monocdk_1.Stack.of(node).resolve(origin);
                const resolvedDomainName = monocdk_1.Stack.of(node).resolve(resolvedOrigin.domainName);
                const s3Regex = /^.+\.s3(?:-website)?(?:\..+)?(?:(?:\.amazonaws\.com(?:\.cn)?)|(?:\.c2s\.ic\.gov)|(?:\.sc2s\.sgov\.gov))$/;
                if (s3Regex.test(resolvedDomainName)) {
                    if (resolvedOrigin.s3OriginConfig == undefined) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                    const resolvedConfig = monocdk_1.Stack.of(node).resolve(resolvedOrigin.s3OriginConfig);
                    if (resolvedConfig.originAccessIdentity == undefined ||
                        resolvedConfig.originAccessIdentity.replace(/\s/g, '').length == 0) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_cloudfront_1.CfnStreamingDistribution) {
        const distributionConfig = monocdk_1.Stack.of(node).resolve(node.streamingDistributionConfig);
        const resolvedOrigin = monocdk_1.Stack.of(node).resolve(distributionConfig.s3Origin);
        if (resolvedOrigin.originAccessIdentity.replace(/\s/g, '').length == 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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