import { IAspect, IConstruct, CfnResource } from 'monocdk';
import { NagPackSuppression } from './models/nag-suppression';
import { NagRuleCompliance, NagRuleResult } from './nag-rules';
/**
 * Interface for creating a Nag rule pack.
 */
export interface NagPackProps {
    /**
     * Whether or not to enable extended explanatory descriptions on warning, error, and logged ignore messages (default: false).
     */
    readonly verbose?: boolean;
    /**
     * Whether or not to log triggered rules that have been suppressed as informational messages (default: false).
     */
    readonly logIgnores?: boolean;
    /**
     * Whether or not to generate CSV compliance reports for applied Stacks in the App's output directory (default: true).
     */
    readonly reports?: boolean;
}
/**
 * Interface for JSII interoperability for passing parameters and the Rule Callback to @applyRule method.
 */
export interface IApplyRule {
    /**
     * Override for the suffix of the Rule ID for this rule
     */
    ruleSuffixOverride?: string;
    /**
     * Why the rule was triggered.
     */
    info: string;
    /**
     * Why the rule exists.
     */
    explanation: string;
    /**
     * The annotations message level to apply to the rule if triggered.
     */
    level: NagMessageLevel;
    /**
     * Ignores listed in cdk-nag metadata.
     */
    node: CfnResource;
    /**
     * The callback to the rule.
     * @param node The CfnResource to check.
     */
    rule(node: CfnResource): NagRuleResult;
}
/**
 * The level of the message that the rule applies.
 */
export declare enum NagMessageLevel {
    WARN = "Warning",
    ERROR = "Error"
}
/**
 * Base class for all rule packs.
 */
export declare abstract class NagPack implements IAspect {
    protected verbose: boolean;
    protected logIgnores: boolean;
    protected reports: boolean;
    protected reportStacks: string[];
    protected packName: string;
    constructor(props?: NagPackProps);
    get readPackName(): string;
    get readReportStacks(): string[];
    /**
     * All aspects can visit an IConstruct.
     */
    abstract visit(node: IConstruct): void;
    /**
     * Create a rule to be used in the NagPack.
     * @param params The @IApplyRule interface with rule details.
     */
    protected applyRule(params: IApplyRule): void;
    /**
     * Check whether a specific rule should be ignored.
     * @param ignores The ignores listed in cdk-nag metadata.
     * @param ruleId The id of the rule to ignore.
     * @param findingId The id of the finding that is being checked.
     * @returns The reason the rule was ignored, or an empty string.
     */
    protected ignoreRule(ignores: NagPackSuppression[], ruleId: string, findingId: string): string;
    /**
     * The message to output to the console when a rule is triggered.
     * @param ruleId The id of the rule.
     * @param findingId The id of the finding.
     * @param info Why the rule was triggered.
     * @param explanation Why the rule exists.
     * @returns The formatted message string.
     */
    protected createMessage(ruleId: string, findingId: string, info: string, explanation: string): string;
    /**
     * Write a line to the rule pack's compliance report for the resource's Stack
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    protected writeToStackComplianceReport(params: IApplyRule, ruleId: string, compliance: NagRuleCompliance.COMPLIANT | NagRuleCompliance.NON_COMPLIANT | 'UNKNOWN', explanation?: string): void;
    /**
     * Initialize the report for the rule pack's compliance report for the resource's Stack if it doesn't exist
     * @param params
     */
    protected initializeStackReport(params: IApplyRule): void;
    /**
     * Helper function to create a line for the compliance report
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    protected createComplianceReportLine(params: IApplyRule, ruleId: string, compliance: NagRuleCompliance.COMPLIANT | NagRuleCompliance.NON_COMPLIANT | 'UNKNOWN', explanation?: string): string;
    private isNonCompliant;
    private asFindings;
}
