# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['tfimm', 'tfimm.architectures', 'tfimm.layers', 'tfimm.models', 'tfimm.utils']

package_data = \
{'': ['*']}

install_requires = \
['markdown==3.3.4', 'numpy', 'tensorflow>=2.4,<3.0']

extras_require = \
{'timm': ['pyyaml', 'timm'], 'train': ['tensorflow-datasets', 'tqdm']}

setup_kwargs = {
    'name': 'tfimm',
    'version': '0.1.2',
    'description': 'TensorFlow port of PyTorch Image Models (timm) - image models with pretrained weights',
    'long_description': '# TensorFlow-Image-Models\n\n- [Introduction](#introduction)\n- [Usage](#usage)\n- [Models](#models)\n- [Profiling](#profiling)\n- [License](#license)\n\n## Introduction\n\nTensorfFlow-Image-Models (`tfimm`) is a collection of image models with pretrained\nweights, obtained by porting architectures from \n[timm](https://github.com/rwightman/pytorch-image-models) to TensorFlow. The hope is\nthat the number of available architectures will grow over time. For now, it contains\nvision transformers (ViT, DeiT and Swin Transformers) and ResNets.\n\nThis work would not have been possible wihout Ross Wightman\'s `timm` library and the\nwork on PyTorch/TensorFlow interoperability in HuggingFace\'s `transformer` repository.\nI tried to make sure all source material is acknowledged. Please let me know if I have\nmissed something.\n\n## Usage\n\n### Installation \n\nThe package can be installed via `pip`,\n\n```shell\npip install tfimm\n```\n\nTo load pretrained weights, `timm` needs to be installed. This is an optional \ndependency and can be installed via\n\n```shell\npip install tfimm[timm]\n```\n\n### Creating models\n\nTo load pretrained models use\n\n```python\nimport tfimm\n\nmodel = tfimm.create_model("vit_tiny_patch16_224", pretrained="timm")\n```\n\nWe can list available models with pretrained weights via\n\n```python\nimport tfimm\n\nprint(tfimm.list_models(pretrained="timm"))\n```\n\nMost models are pretrained on ImageNet or ImageNet-21k. If we want to use them for other\ntasks we need to change the number of classes in the classifier or remove the \nclassifier altogether. We can do this by setting the `nb_classes` parameter in \n`create_model`. If `nb_classes=0`, the model will have no classification layer. If\n`nb_classes` is set to a value different from the default model config, the \nclassification layer will be randomly initialized, while all other weights will be\ncopied from the pretrained model.\n\nThe preprocessing function for each model can be created via\n```python\nimport tensorflow as tf\nimport tfimm\n\npreprocess = tfimm.create_preprocessing("vit_tiny_patch16_224", dtype="float32")\nimg = tf.ones((1, 224, 224, 3), dtype="uint8")\nimg_preprocessed = preprocess(img)\n```\n\n### Saving and loading models\n\nAll models are subclassed from `tf.keras.Model` (they are _not_ functional models).\nThey can still be saved and loaded using the `SavedModel` format.\n\n```\n>>> import tesnorflow as tf\n>>> import tfimm\n>>> model = tfimm.create_model("vit_tiny_patch16_224")\n>>> type(model)\n<class \'tfimm.architectures.vit.ViT\'>\n>>> model.save("/tmp/my_model")\n>>> loaded_model = tf.keras.models.load_model("/tmp/my_model")\n>>> type(loaded_model)\n<class \'tfimm.architectures.vit.ViT\'>\n```\n\nFor this to work, the `tfimm` library needs to be imported before the model is loaded,\nsince during the import process, `tfimm` is registering custom models with Keras.\nOtherwise, we obtain the following output\n\n```\n>>> import tensorflow as tf\n>>> loaded_model = tf.keras.models.load_model("/tmp/my_model")\n>>> type(loaded_model)\n<class \'keras.saving.saved_model.load.Custom>ViT\'>\n```\n\n## Models\n\nThe following architectures are currently available:\n\n- DeiT (vision transformer) \n  [\\[github\\]](https://github.com/facebookresearch/deit)\n  - Training data-efficient image transformers & distillation through attention. \n    [\\[arXiv:2012.12877\\]](https://arxiv.org/abs/2012.12877) \n- ViT (vision transformer) \n  [\\[github\\]](https://github.com/google-research/vision_transformer)\n  - An Image is Worth 16x16 Words: Transformers for Image Recognition at Scale.\n    [\\[arXiv:2010.11929\\]](https://arxiv.org/abs/2010.11929)\n  - How to train your ViT? Data, Augmentation, and Regularization in Vision \n    Transformers. [\\[arXiv:2106.10270\\]](https://arxiv.org/abs/2106.10270)\n  - Includes models trained with the SAM optimizer: Sharpness-Aware Minimization for \n    Efficiently Improving Generalization. \n    [\\[arXiv:2010.01412\\]](https://arxiv.org/abs/2010.01412)\n  - Includes models from: ImageNet-21K Pretraining for the Masses\n    [\\[arXiv:2104.10972\\]](https://arxiv.org/abs/2104.10972) \n    [\\[github\\]](https://github.com/Alibaba-MIIL/ImageNet21K)\n- Swin Transformer \n  [\\[github\\]](https://github.com/microsoft/Swin-Transformer)\n  - Swin Transformer: Hierarchical Vision Transformer using Shifted Windows. \n    [\\[arXiv:2103.14030\\]](https://arxiv.org/abs/2103.14030)\n  - Tensorflow code adapted from \n    [Swin-Transformer-TF](https://github.com/rishigami/Swin-Transformer-TF)\n- ConvMixer \n  [\\[github\\]](https://github.com/tmp-iclr/convmixer)\n  - Patches Are All You Need? \n    [\\[ICLR 2022 submission\\]](https://openreview.net/forum?id=TVHS5Y4dNvM)\n- ResNet (work in progress, most available weights are from `timm`)\n  - Deep Residual Learning for Image Recognition. \n    [\\[arXiv:1512.03385\\]](https://arxiv.org/abs/1512.03385)\n\n## Profiling\n\nTo understand how big each of the models is, I have done some profiling to measure\n- maximum batch size that fits in GPU memory and\n- throughput in images/second\nfor both inference and backpropagation on K80 and V100 GPUs. For V100, measurements \nwere done for both `float32` and mixed precision.\n\nThe results can be found in the `results/profiling_{k80, v100}.csv` files.\n\nFor backpropagation, we use as loss the mean of model outputs\n\n```python\ndef backprop():\n    with tf.GradientTape() as tape:\n        output = model(x, training=True)\n        loss = tf.reduce_mean(output)\n        grads = tape.gradient(loss, model.trainable_variables)\n    optimizer.apply_gradients(zip(grads, model.trainable_variables))\n```\n\n## License\n\nThis repository is released under the Apache 2.0 license as found in the \n[LICENSE](LICENSE) file.',
    'author': 'Martins Bruveris',
    'author_email': 'martins.bruveris@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/martinsbruveris/tensorflow-image-models',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<3.10',
}


setup(**setup_kwargs)
