import numpy as np
import qetpy as qp


__all__ = [
    'FeatureExtractors',
]



class FeatureExtractors:
    """
    A class that contains all of the possible feature extractors
    for a given trace, assuming processing on a single channel.
    Each feature extraction function is a staticmethod for processing
    convenience.

    """

    @staticmethod
    def of1x1_nodelay(of_base=None,  template_tag='default',
                      trace=None, template=None, psd=None,
                      fs=None, nb_samples_pretrigger=None,
                      coupling='AC', integralnorm=False,
                      feature_base_name='of1x1_nodelay',
                      **kwargs):
        """
        Feature extraction for the no delay Optimum Filter. 
        
        The arguments "trace", "template", "psd"
        (and associated parameters "fs", "nb_samples_pretrigger") 
        should only be used if  not already added in OF base object. 
        Otherwise keep as None


        Parameters
        ----------
        of_base : OFBase object, optional
           OFBase  if it has been instantiated independently


        template_tag : str, option
           tag of the template to be used for OF calculation,
           Default: 'default'
        
       
        trace : ndarray, optional
            An ndarray containing the raw data to extract the feature
            from. It is required if trace not already added in OFbase,
            otherwise keep it as None

        template : ndarray, optional
            The template to use for the optimum filter. It is required 
            if template not already added in OFbase,
            otherwise keep it as None

        psd : ndarray, optional
            The PSD to use for the optimum filter.It is required 
            if psd not already added in OFbase,
            otherwise keep it as None

        fs : float, optional
            The digitization rate of the data in trace, required
            if  "of_base" argument  is None, otherwise set to None

        nb_samples_pretrigger : int, optional
            Number of pre-trigger samples, required
            if "of_base" argument is None, otherwise set to None
        
        coupling : str, optional
            Only used if "psd" argument is not None. 
            "coupling" string etermines if the zero 
            frequency bin of the psd should be ignored (i.e. set to infinity) 
            when calculating the optimum amplitude. If set to 'AC', then ths zero
            frequency bin is ignored. If set to anything else, then the
            zero frequency bin is kept. O

        integralnorm : bool, optional
            Only used if "template" argument is not None. 
            If set to True, then  template will be normalized 
            to an integral of 1, and any optimum filters will
            instead return the optimum integral in units of Coulombs.
            If set to False, then the usual optimum filter amplitudes
            will be returned (in units of Amps).

        feature_base_name : str, option
            output feature base name 



        Returns
        -------
        retdict : dict
            Dictionary containing the various extracted features.

        """


        # instantiate OF 1x1
        OF = qp.OF1x1(
            of_base=of_base,
            template_tag=template_tag,
            template=template,
            psd=psd,
            sample_rate=fs,
            pretrigger_samples=nb_samples_pretrigger,
            coupling=coupling,
            integralnorm=integralnorm,
        )
            

        # calc (signal needs to be None if set already)
        OF.calc_nodelay(signal=trace)
        
        # get results
        amp, t0, chisq = OF.get_result_nodelay()
        
        # store features
        retdict = {
            ('amp_' + feature_base_name): amp,
            ('chi2_' + feature_base_name): chisq
        }

        return retdict


    
    @staticmethod
    def of1x1_unconstrained(of_base=None, template_tag='default',
                            interpolate=False,
                            trace=None, template=None, psd=None,
                            fs=None, nb_samples_pretrigger=None,
                            coupling='AC', integralnorm=False,
                            feature_base_name='of1x1_unconstrained',
                            **kwargs):
        """
        Feature extraction for the unconstrained Optimum Filter.
 
        The arguments "trace", "template", "psd"
        (and associated parameters "fs", "nb_samples_pretrigger") 
        should only be used if  not already added in OF base object. 
        Otherwise keep as None

        
        Parameters
        ----------
        of_base : OFBase object, optional
           OFBase  if it has been instantiated independently


        template_tag : str, option
           tag of the template to be used for OF calculation,
           Default: 'default'
        
       interpolate : bool, optional
           if True, do delay interpolation
           default: False
        
        trace : ndarray, optional
            An ndarray containing the raw data to extract the feature
            from. It is required if trace not already added in OFbase,
            otherwise keep it as None

        template : ndarray, optional
            The template to use for the optimum filter. It is required 
            if template not already added in OFbase,
            otherwise keep it as None

        psd : ndarray, optional
            The PSD to use for the optimum filter.It is required 
            if psd not already added in OFbase,
            otherwise keep it as None

        fs : float, optional
            The digitization rate of the data in trace, required
            if  "of_base" argument  is None, otherwise set to None

        nb_samples_pretrigger : int, optional
            Number of pre-trigger samples, required
            if "of_base" argument is None, otherwise set to None
        
        coupling : str, optional
            Only used if "psd" argument is not None. 
            "coupling" string etermines if the zero 
            frequency bin of the psd should be ignored (i.e. set to infinity) 
            when calculating the optimum amplitude. If set to 'AC', then ths zero
            frequency bin is ignored. If set to anything else, then the
            zero frequency bin is kept. O

        integralnorm : bool, optional
            Only used if "template" argument is not None. 
            If set to True, then  template will be normalized 
            to an integral of 1, and any optimum filters will
            instead return the optimum integral in units of Coulombs.
            If set to False, then the usual optimum filter amplitudes
            will be returned (in units of Amps).

        feature_base_name : str, option
            output feature base name 



        Returns
        -------
        retdict : dict
            Dictionary containing the various extracted features.

        """


        # instantiate OF1x1
        OF = qp.OF1x1(
            of_base=of_base,
            template_tag=template_tag,
            template=template,
            psd=psd,
            sample_rate=fs,
            pretrigger_samples=nb_samples_pretrigger,
            coupling=coupling,
            integralnorm=integralnorm,
        )
            
                    

        # calc (signal needs to be None if set already)
        OF.calc(signal=trace,
                interpolate_t0=interpolate,
                lgc_fit_nodelay=False,
                lgc_plot=False)
        
        # get results
        amp, t0, chisq = OF.get_result_withdelay()
        

        # store features
        retdict = {
            ('amp_' + feature_base_name): amp,
            ('t0_' + feature_base_name): t0,
            ('chi2_' + feature_base_name): chisq
        }

        return retdict


    @staticmethod
    def of1x1_constrained(of_base=None, template_tag='default',
                          window_min_from_trig_usec=None,
                          window_max_from_trig_usec=None,
                          window_min_index=None,
                          window_max_index=None,
                          lgc_outside_window=False,
                          interpolate=False,
                          trace=None, template=None, psd=None,
                          fs=None, nb_samples_pretrigger=None,
                          coupling='AC', integralnorm=False,
                          feature_base_name='of1x1_constrained',
                          **kwargs):
        """
        Feature extraction for the constrained Optimum Filter.

        The arguments "trace", "template", "psd"
        (and associated parameters "fs", "nb_samples_pretrigger") 
        should only be used if  not already added in OF base object. 
        Otherwise keep as None

        
        Parameters
        ----------
        of_base : OFBase object, optional
           OFBase  if it has been instantiated independently
        
        
        template_tag : str, optional
           tag of the template to be used for OF calculation,
           Default: 'default'
        
        window_min_from_trig_usec : float, optional
           OF filter window start in micro seconds from
           pre-trigger (can be negative if prior pre-trigger)
           Default: use "window_min_index"  or 
                    or set to 0 if both parameters are None

        window_max_from_trig_usec : float, optional
           OF filter window end in micro seconds from
           pre-trigger (can be negative if prior pre-trigger)
           Default: use "window_max_index"  or set to end trace
                    if  "window_max_index" also None

        
        window_min_index : int, optional
           ADC index OF filter window start (alternative 
           to "window_min_from_trig_usec")

           Default: use "window_min_from_trig_usec"  or 
                    set to 0 if both parameters are None

        window_max_index : int, optional
           ADC index OF filter window end (alternative 
           to "window_min_from_trig_usec")

           Default: use "window_max_from_trig_usec"  or 
                    set to end of trace if both parameters 
                     are None
       
        lgc_outside_window : bool, optional
           If True, define window to be outside [min:max]
           Default: False

        interpolate : bool, optional
           if True, do delay interpolation
           default: False
        
        trace : ndarray, optional
            An ndarray containing the raw data to extract the feature
            from. It is required if trace not already added in OFbase,
            otherwise keep it as None

        template : ndarray, optional
            The template to use for the optimum filter. It is required 
            if template not already added in OFbase,
            otherwise keep it as None

        psd : ndarray, optional
            The PSD to use for the optimum filter.It is required 
            if psd not already added in OFbase,
            otherwise keep it as None

        fs : float, optional
            The digitization rate of the data in trace, required
            if  "of_base" argument  is None, otherwise set to None

        nb_samples_pretrigger : int, optional
            Number of pre-trigger samples, required
            if "of_base" argument is None, otherwise set to None
        
        coupling : str, optional
            Only used if "psd" argument is not None. 
            "coupling" string etermines if the zero 
            frequency bin of the psd should be ignored (i.e. set to infinity) 
            when calculating the optimum amplitude. If set to 'AC', then ths zero
            frequency bin is ignored. If set to anything else, then the
            zero frequency bin is kept. O

        integralnorm : bool, optional
            Only used if "template" argument is not None. 
            If set to True, then  template will be normalized 
            to an integral of 1, and any optimum filters will
            instead return the optimum integral in units of Coulombs.
            If set to False, then the usual optimum filter amplitudes
            will be returned (in units of Amps).

        feature_base_name : str, optional
            output feature base name 



        Returns
        -------
        retdict : dict
            Dictionary containing the various extracted features.

        """

        
        # instantiate OF1x1
        OF = qp.OF1x1(
            of_base=of_base,
            template_tag=template_tag,
            template=template,
            psd=psd,
            sample_rate=fs,
            pretrigger_samples=nb_samples_pretrigger,
            coupling=coupling,
            integralnorm=integralnorm,
        )
        
            
            
        # calc (signal needs to be None if set already)
        OF.calc(signal=trace,
                window_min_from_trig_usec=window_min_from_trig_usec,
                window_max_from_trig_usec=window_max_from_trig_usec,
                window_min_index=window_min_index,
                window_max_index=window_max_index,
                interpolate_t0=interpolate,
                lgc_outside_window=lgc_outside_window,
                lgc_fit_nodelay=False,
                lgc_plot=False)
        
      
        # get results
        amp, t0, chisq = OF.get_result_withdelay()
        
        

        retdict = {
            ('amp_' + feature_base_name): amp,
            ('t0_' + feature_base_name): t0,
            ('chi2_' + feature_base_name): chisq
        }

        return retdict


    @staticmethod
    def baseline(trace,
                 window_min_index=None, window_max_index=None,
                 feature_base_name='baseline',
                 **kwargs):
        """
        Feature extraction for the trace baseline.

        Parameters
        ----------
        trace : ndarray
            An ndarray containing the raw data to extract the feature
            from.

        window_min_index : int, optional
            The minium index of the window used to average the trace.
            Default: 0

        window_max_index : int, optional
            The maximum index of the window used to average the trace.
            Default: end of trace

        feature_base_name : str, optional
            output feature base name 

        Returns
        -------
        retdict : dict
            Dictionary containing the various extracted features.

        """

        if window_min_index is None:
            window_min_index = 0

        if window_max_index is None:
            window_max_index = trace.shape[-1] - 1

        
        baseline = np.mean(trace[window_min_index:window_max_index])

        retdict = {
            feature_base_name: baseline,
        }
        
        return retdict



    @staticmethod
    def integral(trace, fs,
                 window_min_index=None, window_max_index=None,
                 feature_base_name='integral',
                 **kwargs):
        """
        Feature extraction for the pulse integral.

        Parameters
        ----------
        trace : ndarray
            An ndarray containing the raw data to extract the feature
            from.

        fs : float
            The digitization rate of the data in trace.

        window_min_index : int, optional
            The minium index of the window used to integrate the trace.
            Default: 0

        window_max_index : int, optional
            The maximum index of the window used to integrate the trace.
            Default: end of trace

        feature_base_name : str, optional
            output feature base name 

        Returns
        -------
        retdict : dict
            Dictionary containing the various extracted features.

        """
        
        if window_min_index is None:
            window_min_index = 0
            
        if window_max_index is None:
            window_max_index = trace.shape[-1] - 1
        
                
        integral = np.trapz(trace[window_min_index:window_max_index]) / fs

        retdict = {
            feature_base_name: integral,
        }

        return retdict


    @staticmethod
    def maximum(trace,
                window_min_index=None, window_max_index=None,
                feature_base_name='maximum',
                **kwargs):
        """
        Feature extraction for the maximum pulse value.

        Parameters
        ----------
        trace : ndarray
            An ndarray containing the raw data to extract the feature
            from.
   
        window_min_index : int, optional
            The minium index of the window used to find  maximum
            Default: 0

        window_max_index : int, optional
            The maximum index of the window used to find maximum
            Default: end of trace

        feature_base_name : str, optional
            output feature base name 


        Returns
        -------
        retdict : dict
            Dictionary containing the various extracted features.

        """

        if window_min_index is None:
            window_min_index = 0
            
        if window_max_index is None:
            window_max_index = trace.shape[-1] - 1

        max_trace = np.amax(trace[window_min_index:window_max_index])

        retdict = {
            feature_base_name: max_trace,
        }

        return retdict


    @staticmethod
    def minimum(trace,
                window_min_index=None, window_max_index=None,
                feature_base_name='minimum',
                **kwargs):
        """
        Feature extraction for the minimum pulse value.

        Parameters
        ----------
        trace : ndarray
            An ndarray containing the raw data to extract the feature
            from.


        window_min_index : int, optional
            The minium index of the window used to find  minimum
            Default: 0

        window_max_index : int, optional
            The maximum index of the window used to find minimum
            Default: end of trace

        feature_base_name : str, optional
            output feature base name 


        Returns
        -------
        retdict : dict
            Dictionary containing the various extracted features.

        """
        if window_min_index is None:
            window_min_index = 0
            
        if window_max_index is None:
            window_max_index = trace.shape[-1] - 1

            
        min_trace = np.amin(trace[window_min_index:window_max_index])

        retdict = {
            feature_base_name: min_trace,
        }

        return retdict

    @staticmethod
    def energyabsorbed(trace,
                       fs, vb, i0, rl,
                       window_min_index=None, window_max_index=None,
                       feature_base_name='energyabsorbed',
                       **kwargs):
        """
        Feature extraction for the minimum pulse value.

        Parameters
        ----------
        trace : ndarray
            An ndarray containing the raw data to extract the feature
            from.
        fs : float
            The digitization rate of the data in trace.
        vb : float
            Bias voltage applied to the TES.
        i0 : float
            Quiescent operating current of the TES.
        rl : float
            Load resistance in the TES circuit.

        window_min_index : int, optional
            The index of the trace to start the integration.
             Default: 0
        window_max_index : int, optional
            The index of the trace to end the integration.
            Default: end of trace
 
        feature_base_name : str, optional
            output feature base name 

        Returns
        -------
        retdict : dict
            Dictionary containing the various extracted features.

        """

        baseline = trace[:window_min_index].mean()
        i_trace = trace[window_min_index:window_max_index] - baseline

        p0 = i_trace * (vb - 2*i0*rl) - i_trace**2 * rl

        en_abs = np.trapz(p0, dx=1/fs, axis=-1)

        retdict = {
            feature_base_name: en_abs,
        }
            
        return retdict

