import logging
from enum import Enum

from typing_extensions import TypedDict

from lineapy.plugins.base_pipeline_writer import BasePipelineWriter
from lineapy.plugins.utils import load_plugin_template
from lineapy.utils.logging_config import configure_logging

logger = logging.getLogger(__name__)
configure_logging()


class DVCPipelineWriter(BasePipelineWriter):
    def _write_dag(self) -> None:
        dag_flavor = self.dag_config.get(
            "dag_flavor", "SingleStageAllSessions"
        )

        # Check if the given DAG flavor is a supported/valid one
        if dag_flavor not in DVCDagFlavor.__members__:
            raise ValueError(f'"{dag_flavor}" is an invalid dvc dag flavor.')

        # Construct DAG text for the given flavor
        if DVCDagFlavor[dag_flavor] == DVCDagFlavor.SingleStageAllSessions:
            full_code = self._write_operator_run_all_sessions()

        # Write out file
        file = self.output_dir / "dvc.yaml"
        file.write_text(full_code)
        logger.info(f"Generated DAG file: {file}")

    def _write_operator_run_all_sessions(self) -> str:
        """
        This hidden method implements DVC DAG code generation corresponding
        to the `SingleStageAllSessions` flavor. This DAG only has one stage and
        calls `run_all_sessions` generated by the module file.
        """

        DAG_TEMPLATE = load_plugin_template(
            "dvc_dag_SingleStageAllSessions.jinja"
        )

        full_code = DAG_TEMPLATE.render(
            MODULE_COMMAND=f"python {self.pipeline_name}_module.py",
        )

        return full_code

    @property
    def docker_template_name(self) -> str:
        return "dvc_dockerfile.jinja"


class DVCDagFlavor(Enum):
    SingleStageAllSessions = 1
    # TODO: StagePerSession
    # TODO: StagePerArtifact


DVCDagConfig = TypedDict(
    "DVCDagConfig",
    {
        "dag_flavor": str,  # Not native to DVC config
    },
    total=False,
)
