"""
Some auxiliary entities not falling into the other categories
"""

import collections.abc
import copy
import json
import pathlib
import subprocess
import sys
import time
from typing import Any, List, Mapping

from stm32pio.core.settings import pio_boards_cache_lifetime, config_default


def _get_version_from_scm() -> str:
    try:
        import setuptools_scm  # setuptools_scm is the dev-only dependency
    except ImportError:
        return "Portable (not-installed). See git tag"
    else:
        # Calculate the version in real-time from the Git repo state
        return setuptools_scm.get_version(root='../..', relative_to=__file__)

def get_version() -> str:
    """Retrieve the app version as string"""
    if sys.version_info >= (3, 8):
        import importlib.metadata
        try:
            # For modern Python use the package metadata (if we are installed). For this to be available the wheel build
            # should be done with setuptools_scm
            return importlib.metadata.version('stm32pio')
        except importlib.metadata.PackageNotFoundError:
            # stm32pio is not installed (i.e. running from sources)
            return _get_version_from_scm()
    else:
        try:
            # Version is stored in the stm32pio/core/version.py file auto-generated by setuptools_scm tool
            import stm32pio.core.version
        except ImportError:
            # Version file is not available, most likely we are not installed (i.e. running from sources)
            return _get_version_from_scm()
        else:
            return stm32pio.core.version.version


_pio_boards_cache: List[str] = []
_pio_boards_cache_fetched_at: float = 0

def get_platformio_boards(platformio_cmd: str = config_default['app']['platformio_cmd']) -> List[str]:
    """
    Obtain the PlatformIO boards list (string identifiers only). As we interested only in STM32 ones, cut off all of the others. Additionally,
    establish a short-time "cache" to prevent the over-flooding with requests to subprocess.

    IMPORTANT NOTE: PlatformIO can go to the Internet from time to time when it decides that its own cache is out of
    date. So it may take a long time to execute.
    """

    global _pio_boards_cache_fetched_at, _pio_boards_cache
    cache_is_empty = len(_pio_boards_cache) == 0
    current_time = time.time()
    cache_is_outdated = current_time - _pio_boards_cache_fetched_at >= pio_boards_cache_lifetime

    if cache_is_empty or cache_is_outdated:
        # Windows 7, as usual, correctly works only with shell=True...
        completed_process = subprocess.run(
            f"{platformio_cmd} boards --json-output stm32cube",
            encoding='utf-8', shell=True, stdout=subprocess.PIPE, check=True)
        _pio_boards_cache = [board['id'] for board in json.loads(completed_process.stdout)]
        _pio_boards_cache_fetched_at = current_time

    # Caller can mutate the array and damage our cache so we give it a copy (as the values are strings it is equivalent
    # to the deep copy of this list)
    return copy.copy(_pio_boards_cache)


def cleanup_mapping(mapping: Mapping[str, Any]) -> dict:
    """Return a deep copy of the given mapping excluding None and empty string values"""

    cleaned = {}

    for key, value in mapping.items():
        if isinstance(value, collections.abc.Mapping):
            cleaned[key] = cleanup_mapping(value)
        elif value is not None and value != '':
            cleaned[key] = value

    return cleaned


def get_folder_contents(path: pathlib.Path, pattern: str = '*',
                        ignore_list: List[pathlib.Path] = None) -> List[pathlib.Path]:
    """
    Return all endpoints inside the given directory (recursively). If specified, paths from the ignore_list will be
    excluded. The resulting array is fully "unfolded" meaning every folder will be expanded, so both it and its children
    will be included into the list. Conversely, the ignore_list is treated in the opposite way so for every folder met
    both it and its children will be ignored completely.

    Note: this is a "naive", straightforward and non-efficient solution (probably, both for time and memory
    consumption). The algorithm behind can (but not necessarily should) definitely be improved.

    Args:
        path: root directory
        pattern: optional glob-style pattern string to use. Default one will pass all
        ignore_list: optional list of pathlib Paths to ignore (see the full description)

    Returns:
        list of pathlib Paths
    """

    folder_contents = []

    if ignore_list is not None:
        ignore_list = sorted(ignore_list)
    else:
        ignore_list = []

    for child in sorted(path.rglob(pattern)):  # all files and folders, recursively

        # Check such cases:
        #   1) current child:        a/b/
        #      ignore list entry:    a/b/c/d.txt
        #
        #   2) current child:        a/b/c/d.txt
        #      ignore list entry:    a/b/
        is_root_of_another = next(
            (True for entry in ignore_list if (child in entry.parents) or (entry in child.parents)), False)

        if (child not in ignore_list) and (not is_root_of_another):
            folder_contents.append(child)

    return folder_contents
