import logging

import tensorflow as tf

from tensorflow.python.saved_model.loader_impl import parse_saved_model

DOCUMENTATION_TEMPLATE = """# {model_name}

_This README.md is autogenerated by Model Zoo_

{input_output_table}

### Predictions

- Install the Python client library `modelzoo-client`

```bash
pip install modelzoo-client
```

- Make a prediction

```python
import modelzoo.tensorflow

input = ...
prediction = modelzoo.tensorflow.predict("{model_name}",input)
```
"""


def generate_default_tensorflow_docs(model_name: str, saved_model_dir: str) -> str:
    return DOCUMENTATION_TEMPLATE.format(
        model_name=model_name,
        input_output_table=generate_input_output_table_from_saved_model(
            saved_model_dir,
        ),
    )


def generate_input_output_table_from_saved_model(saved_model_dir: str) -> str:
    """
    Given a saved model directory for a TensorFlow Serving model, attempt to
    generate markdown table representing the input and outputs. This function
    should not raise any exceptions, just return an empty string if it doesn't
    get to a quick representation of the metadata.

    Assumptions:

        "serving_default" is the signature definition to use,
        which seems to be the tf.keras default.
    """
    if not tf.saved_model.contains_saved_model(saved_model_dir):
        return ""

    try:
        saved_model_proto = parse_saved_model(saved_model_dir)
    except Exception as e:
        logging.warning(
            "Could not generate parse the TensorFlow saved model: {}".format(e)
        )
        return ""

    serving_default_def = None
    for meta_graph in saved_model_proto.meta_graphs:
        for name, signature_def in meta_graph.signature_def.items():
            if name == "serving_default":
                serving_default_def = signature_def

    if not serving_default_def:
        return ""

    output = [
        "| | Name | Shape | Type |",
        "|-| ---- | -----:| ----:|",
    ]

    def _format_placeholder(label, name, placeholder):
        if placeholder.HasField("tensor_shape"):
            shape = "`{}`".format([d.size for d in placeholder.tensor_shape.dim])
        else:
            shape = ""
        type_name = "`{}`".format(tf.dtypes.DType(placeholder.dtype).name)
        input_name = "`{}`".format(name)

        return "| {} | {} | {} | {} |".format(label, input_name, shape, type_name)

    for name, placeholder in serving_default_def.inputs.items():
        output += [_format_placeholder("input", name, placeholder)]
    for name, placeholder in serving_default_def.outputs.items():
        output += [_format_placeholder("output", name, placeholder)]

    return "\n".join(output)
