# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['arguebuf',
 'arguebuf.cli',
 'arguebuf.dump',
 'arguebuf.load',
 'arguebuf.model',
 'arguebuf.render',
 'arguebuf.schemas']

package_data = \
{'': ['*']}

install_requires = \
['arg-services>=1.1.0,<2.0.0',
 'graphviz>=0.20.1,<0.21.0',
 'lxml>=4.9.2,<5.0.0',
 'networkx>=2.8.8,<4.0',
 'pendulum>=2.1.2,<3.0.0']

extras_require = \
{'cli': ['typer[all]>=0.7.0,<0.8.0',
         'deepl>=1.13.0,<2.0.0',
         'multimethod>=1.9.1,<2.0.0'],
 'graphviz': ['pygraphviz>=1.10,<2.0']}

entry_points = \
{'console_scripts': ['arguebuf = arguebuf.cli.app:cli']}

setup_kwargs = {
    'name': 'arguebuf',
    'version': '2.0.0b14',
    'description': 'A library for loading argument graphs in various formats (e.g., AIF).',
    'long_description': '# Arguebuf\n\nArguebuf is a format for serializing argument graphs and specified using Protobuf.\nThe complete specification and documentation is available at the [Buf Schema Registry](https://buf.build/recap/arg-services/docs/main:arg_services.graph.v1).\nWhile Protobuf automatically generates native code for all major programming languages (including Python), we created a custom implementation that provides some additional benefits, including:\n\n- The ability to import existing formats like [AIF](http://www.argumentinterchange.org), [SADFace](https://github.com/Open-Argumentation/SADFace), and a few others.\n- Export of Arguebuf graphs to AIF, [NetworkX](https://networkx.org), and [Graphviz](https://graphviz.org).\n- Integration with the popular NLP library [spaCy](http://spacy.io).\n- Various helper methods to programmatically manipulate/create argument graphs.\n- More pythonic interfaces than the regular code generated by `protoc`.\n\nYou can easily install the library from [PyPI](https://pypi.org/project/arguebuf/) using pip. The documentation is hosted on [ReadTheDocs](https://arguebuf.readthedocs.io/en/latest/)\n\n## Command Line Interface (CLI)\n\nWe also offer some tools to simplify dealing with structured argument graphs.\nAmong others, it is possible to convert graphs between different formats, translate them, and render images using graphviz.\nTo use it, install the `cli` extras when installing the package.\nWhen using `pip`, this can be accomplished with\n\n`pip install arguebuf[cli]`\n\nAfterwards, you can execute it by calling `arguebuf`, for example:\n\n`arguebuf --help`\n\n## Theoretical Foundations\n\nAn argument graph is way to represent _structured_ argumentation.\nWhat sets it apart from _unstructured_ argumentation (e.g., in newspaper articles) is that only the essential/argumentative parts of a text art part of this representation.\nThese _units_ of argumentation are also called ADUs (argumentative discourse units).\nThe length of ADUs can differ dramatically (depending on various factors like the context), meaning they might contain only a few words, a sentence, or even a whole paragraph.\nThe structure of an argument is then represented through _relations_ between these units.\nFor this purpose, they can be further subdivided into _claims_ and _premises_:\nA claim is a statement that is supported or attacked by one or multiple premises.\nAt the same time, a claim may also function as a premise for another claim, making it possible to construct even complex _argument graphs_.\nOne of the claims is called _major claim_ and represents the overall claim of the whole argument.\nIn many cases (but not all), this major claim is located right at the top of the graph\nHere is a rather simple example.\n\nTODO: Example\n\nClaims, premises and the major claim are represented as _atom nodes_ while relations between them are represented by _scheme nodes_.\nThe set of nodes $V = A \\cup S$ is composed of the set of atom nodes $A$ and the set of scheme nodes $S$.\nThe supporting or attacking relations are encoded in a set of edges $E \\subseteq V \\times V \\setminus A \\times A$.\nEdges can be drawn between any type of nodes except for two atom nodes.\nFor instance, it is possible to connect two scheme nodes to support or attack the _inference_ between two ADUs.\nBased on this, we define an argument graph $G$ as the triple $G = ( V , E , M )$.\n\n## User Guide\n\nWhen importing this library, we recommend using an abbreviation such as `ag` (for _argument graph_).\n\n```python\nimport arguebuf as ag\n```\n\nIn the following, we will introduce the most important features of `arguebuf`.\nFor more details including examples, check out our API documentation.\n\n### Importing an Existing Graph\n\nWe support multiple established formats to represent argument graphs: `AIF`, `OVA`, `SADFace`, `ArgDown`, `BRAT`, and `Kialo`.\nGiven an input file, the library can automatically determine the correct format and convert it to a representation in the `arguebuf` format.\nOne can either pass a string pointing to the file or a `pathlib.Path` object.\n\n```python\ngraph = ag.load.file("graph.json")\n```\n\nIt is also possible to load multiple graphs within a folder.\nHere, you need to pass the folder along with a [glob pattern](https://docs.python.org/3/library/fnmatch.html#module-fnmatch) for selecting the argument graphs.\nThis also enables to recursively load all argument graphs from a common parent directory.\n\n```python\ngraphs = ag.load.folder("./data", "**/*.json")\n```\n\nSince atom nodes contain textual information that may need to be analyzed using NLP techniques, we support passing a custom `nlp` function to these loader methods.\nThis also makes it really easy to integrate the popular [`spacy` library](http://spacy.io) with `arguebuf` as all texts of atom nodes are automatically converted to a spacy `Doc`.\n\n```python\nimport spacy\nnlp = spacy.load("en_core_web_lg")\ngraph = ag.load.file("graph.json", nlp=nlp)\n```\n\n### Programmatically Create a New Graph\n\nInstead of importing an existing graph, you can also create a new one using an object-oriented API using our library.\nTo illustrate this, we generate a graph with two premises that are connected to a major claim.\n**Please note:** In case edges with nodes not yet contained in the graph are added, the respective nodes are added automatically.\n\n```python\ngraph = ag.Graph("TITLE")\n\npremise1 = ag.AtomNode("Text of premise 1")\npremise2 = ag.AtomNode("Text of premise 2")\nclaim = ag.AtomNode("Text of claim")\n\nscheme1 = ag.SchemeNode(ag.Support.DEFAULT)\nscheme2 = ag.SchemeNode(ag.Attack.DEFAULT)\n\ngraph.add_edge(ag.Edge(premise1, scheme1))\ngraph.add_edge(ag.Edge(scheme1, claim))\ngraph.add_edge(ag.Edge(premise2, scheme2))\ngraph.add_edge(ag.Edge(scheme2, claim))\n\ngraph.major_claim = claim\n```\n\nWith this code, we get the following output\n\n![Output of programmatic graph creation](./static/programmatic.png)\n\n### Exporting Argument Graphs\n\nWe support different output formats and integration with other libraries to ease the use of argument graphs.\nHave a look at the following code snippet to get an overview of the possibilities\n\n```python\n# Export to graphviz DOT format\ndot = ag.dump.graphviz(graph)\n\n# Export an image of this dot source to a file\nag.render.graphviz(dot, "./graph.pdf")\n\n# Convert to NetworkX graph\nnx = ag.dump.networkx(graph)\n\n# Save the graph as Arguebuf\nag.dump.file(graph, "./graph.json")\n\n# Save the graph as AIF\nag.dump.file(graph, "./graph.json", ag.GraphFormat.AIF)\n```\n',
    'author': 'Mirko Lenz',
    'author_email': 'info@mirko-lenz.de',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://recap.uni-trier.de',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
