"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClientAuthentication = exports.ClientBrokerEncryption = exports.ClusterMonitoringLevel = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const core = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cr = require("aws-cdk-lib/custom-resources");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const uniqueid_1 = require("constructs/lib/private/uniqueid");
const aws_msk_1 = require("aws-cdk-lib/aws-msk");
/**
 * A new or imported MSK Cluster.
 */
class ClusterBase extends core.Resource {
    /** Manages connections for the cluster */
    get connections() {
        if (!this._connections) {
            throw new Error('An imported Cluster cannot manage its security groups');
        }
        return this._connections;
    }
}
/**
 * The level of monitoring for the MSK cluster
 *
 * @see https://docs.aws.amazon.com/msk/latest/developerguide/monitoring.html#metrics-details
 */
var ClusterMonitoringLevel;
(function (ClusterMonitoringLevel) {
    /**
     * Default metrics are the essential metrics to monitor.
     */
    ClusterMonitoringLevel["DEFAULT"] = "DEFAULT";
    /**
     * Per Broker metrics give you metrics at the broker level.
     */
    ClusterMonitoringLevel["PER_BROKER"] = "PER_BROKER";
    /**
     * Per Topic Per Broker metrics help you understand volume at the topic level.
     */
    ClusterMonitoringLevel["PER_TOPIC_PER_BROKER"] = "PER_TOPIC_PER_BROKER";
    /**
     * Per Topic Per Partition metrics help you understand consumer group lag at the topic partition level.
     */
    ClusterMonitoringLevel["PER_TOPIC_PER_PARTITION"] = "PER_TOPIC_PER_PARTITION";
})(ClusterMonitoringLevel = exports.ClusterMonitoringLevel || (exports.ClusterMonitoringLevel = {}));
/**
 * Indicates the encryption setting for data in transit between clients and brokers.
 */
var ClientBrokerEncryption;
(function (ClientBrokerEncryption) {
    /**
     * TLS means that client-broker communication is enabled with TLS only.
     */
    ClientBrokerEncryption["TLS"] = "TLS";
    /**
     * TLS_PLAINTEXT means that client-broker communication is enabled for both TLS-encrypted, as well as plaintext data.
     */
    ClientBrokerEncryption["TLS_PLAINTEXT"] = "TLS_PLAINTEXT";
    /**
     * PLAINTEXT means that client-broker communication is enabled in plaintext only.
     */
    ClientBrokerEncryption["PLAINTEXT"] = "PLAINTEXT";
})(ClientBrokerEncryption = exports.ClientBrokerEncryption || (exports.ClientBrokerEncryption = {}));
/**
 * Configuration properties for client authentication.
 */
class ClientAuthentication {
    /**
     * @param saslProps - properties for SASL authentication
     * @param tlsProps - properties for TLS authentication
     */
    constructor(saslProps, tlsProps) {
        this.saslProps = saslProps;
        this.tlsProps = tlsProps;
    }
    /**
     * SASL authentication
     */
    static sasl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_SaslAuthProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.sasl);
            }
            throw error;
        }
        return new ClientAuthentication(props, undefined);
    }
    /**
     * TLS authentication
     */
    static tls(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_TlsAuthProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.tls);
            }
            throw error;
        }
        return new ClientAuthentication(undefined, props);
    }
    /**
     * SASL + TLS authentication
     */
    static saslTls(saslTlsProps) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_SaslTlsAuthProps(saslTlsProps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.saslTls);
            }
            throw error;
        }
        return new ClientAuthentication(saslTlsProps, saslTlsProps);
    }
}
exports.ClientAuthentication = ClientAuthentication;
_a = JSII_RTTI_SYMBOL_1;
ClientAuthentication[_a] = { fqn: "@aws-cdk/aws-msk-alpha.ClientAuthentication", version: "2.69.0-alpha.0" };
/**
 * Create a MSK Cluster.
 *
 * @resource AWS::MSK::Cluster
 */
class Cluster extends ClusterBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_ClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Cluster);
            }
            throw error;
        }
        const subnetSelection = props.vpc.selectSubnets(props.vpcSubnets);
        this._connections = new ec2.Connections({
            securityGroups: props.securityGroups ?? [
                new ec2.SecurityGroup(this, 'SecurityGroup', {
                    description: 'MSK security group',
                    vpc: props.vpc,
                }),
            ],
        });
        if (subnetSelection.subnets.length < 2) {
            throw Error(`Cluster requires at least 2 subnets, got ${subnetSelection.subnets.length}`);
        }
        if (!core.Token.isUnresolved(props.clusterName) &&
            !/^[a-zA-Z0-9]+$/.test(props.clusterName) &&
            props.clusterName.length > 64) {
            throw Error('The cluster name must only contain alphanumeric characters and have a maximum length of 64 characters.' +
                `got: '${props.clusterName}. length: ${props.clusterName.length}'`);
        }
        if (props.clientAuthentication?.saslProps?.iam &&
            props.clientAuthentication?.saslProps?.scram) {
            throw Error('Only one client authentication method can be enabled.');
        }
        if (props.encryptionInTransit?.clientBroker ===
            ClientBrokerEncryption.PLAINTEXT &&
            props.clientAuthentication) {
            throw Error('To enable client authentication, you must enabled TLS-encrypted traffic between clients and brokers.');
        }
        else if (props.encryptionInTransit?.clientBroker ===
            ClientBrokerEncryption.TLS_PLAINTEXT &&
            (props.clientAuthentication?.saslProps?.scram ||
                props.clientAuthentication?.saslProps?.iam)) {
            throw Error('To enable SASL/SCRAM or IAM authentication, you must only allow TLS-encrypted traffic between clients and brokers.');
        }
        const volumeSize = props.ebsStorageInfo?.volumeSize ?? 1000;
        // Minimum: 1 GiB, maximum: 16384 GiB
        if (volumeSize < 1 || volumeSize > 16384) {
            throw Error('EBS volume size should be in the range 1-16384');
        }
        const instanceType = props.instanceType
            ? this.mskInstanceType(props.instanceType)
            : this.mskInstanceType(ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE));
        const encryptionAtRest = props.ebsStorageInfo?.encryptionKey
            ? {
                dataVolumeKmsKeyId: props.ebsStorageInfo.encryptionKey.keyId,
            }
            : undefined; // MSK will create the managed key
        const encryptionInTransit = {
            clientBroker: props.encryptionInTransit?.clientBroker ??
                ClientBrokerEncryption.TLS,
            inCluster: props.encryptionInTransit?.enableInCluster ?? true,
        };
        const openMonitoring = props.monitoring?.enablePrometheusJmxExporter ||
            props.monitoring?.enablePrometheusNodeExporter
            ? {
                prometheus: {
                    jmxExporter: props.monitoring?.enablePrometheusJmxExporter
                        ? { enabledInBroker: true }
                        : undefined,
                    nodeExporter: props.monitoring
                        ?.enablePrometheusNodeExporter
                        ? { enabledInBroker: true }
                        : undefined,
                },
            }
            : undefined;
        const loggingBucket = props.logging?.s3?.bucket;
        if (loggingBucket && aws_cdk_lib_1.FeatureFlags.of(this).isEnabled(cx_api_1.S3_CREATE_DEFAULT_LOGGING_POLICY)) {
            const stack = core.Stack.of(this);
            loggingBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                principals: [
                    new iam.ServicePrincipal('delivery.logs.amazonaws.com'),
                ],
                resources: [
                    loggingBucket.arnForObjects(`AWSLogs/${stack.account}/*`),
                ],
                actions: ['s3:PutObject'],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                        'aws:SourceAccount': stack.account,
                    },
                    ArnLike: {
                        'aws:SourceArn': stack.formatArn({
                            service: 'logs',
                            resource: '*',
                        }),
                    },
                },
            }));
            loggingBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                principals: [
                    new iam.ServicePrincipal('delivery.logs.amazonaws.com'),
                ],
                resources: [loggingBucket.bucketArn],
                actions: [
                    's3:GetBucketAcl',
                    's3:ListBucket',
                ],
                conditions: {
                    StringEquals: {
                        'aws:SourceAccount': stack.account,
                    },
                    ArnLike: {
                        'aws:SourceArn': stack.formatArn({
                            service: 'logs',
                            resource: '*',
                        }),
                    },
                },
            }));
        }
        const loggingInfo = {
            brokerLogs: {
                cloudWatchLogs: {
                    enabled: props.logging?.cloudwatchLogGroup !== undefined,
                    logGroup: props.logging?.cloudwatchLogGroup?.logGroupName,
                },
                firehose: {
                    enabled: props.logging?.firehoseDeliveryStreamName !==
                        undefined,
                    deliveryStream: props.logging?.firehoseDeliveryStreamName,
                },
                s3: {
                    enabled: loggingBucket !== undefined,
                    bucket: loggingBucket?.bucketName,
                    prefix: props.logging?.s3?.prefix,
                },
            },
        };
        if (props.clientAuthentication?.saslProps?.scram &&
            props.clientAuthentication?.saslProps?.key === undefined) {
            this.saslScramAuthenticationKey = new kms.Key(this, 'SASLKey', {
                description: 'Used for encrypting MSK secrets for SASL/SCRAM authentication.',
                alias: `msk/${props.clusterName}/sasl/scram`,
            });
            // https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-policies
            this.saslScramAuthenticationKey.addToResourcePolicy(new iam.PolicyStatement({
                sid: 'Allow access through AWS Secrets Manager for all principals in the account that are authorized to use AWS Secrets Manager',
                principals: [new iam.AnyPrincipal()],
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:CreateGrant',
                    'kms:DescribeKey',
                ],
                resources: ['*'],
                conditions: {
                    StringEquals: {
                        'kms:ViaService': `secretsmanager.${core.Stack.of(this).region}.amazonaws.com`,
                        'kms:CallerAccount': core.Stack.of(this).account,
                    },
                },
            }));
        }
        let clientAuthentication;
        if (props.clientAuthentication?.saslProps?.iam) {
            clientAuthentication = {
                sasl: { iam: { enabled: props.clientAuthentication.saslProps.iam } },
            };
            if (props.clientAuthentication?.tlsProps) {
                clientAuthentication = {
                    sasl: { iam: { enabled: props.clientAuthentication.saslProps.iam } },
                    tls: {
                        certificateAuthorityArnList: props.clientAuthentication?.tlsProps?.certificateAuthorities?.map((ca) => ca.certificateAuthorityArn),
                    },
                };
            }
        }
        else if (props.clientAuthentication?.saslProps?.scram) {
            clientAuthentication = {
                sasl: {
                    scram: {
                        enabled: props.clientAuthentication.saslProps.scram,
                    },
                },
            };
        }
        else if (props.clientAuthentication?.tlsProps?.certificateAuthorities !== undefined) {
            clientAuthentication = {
                tls: {
                    certificateAuthorityArnList: props.clientAuthentication?.tlsProps?.certificateAuthorities.map((ca) => ca.certificateAuthorityArn),
                },
            };
        }
        const resource = new aws_msk_1.CfnCluster(this, 'Resource', {
            clusterName: props.clusterName,
            kafkaVersion: props.kafkaVersion.version,
            numberOfBrokerNodes: props.numberOfBrokerNodes !== undefined ?
                subnetSelection.availabilityZones.length * props.numberOfBrokerNodes : subnetSelection.availabilityZones.length,
            brokerNodeGroupInfo: {
                instanceType,
                clientSubnets: subnetSelection.subnetIds,
                securityGroups: this.connections.securityGroups.map((group) => group.securityGroupId),
                storageInfo: {
                    ebsStorageInfo: {
                        volumeSize: volumeSize,
                    },
                },
            },
            encryptionInfo: {
                encryptionAtRest,
                encryptionInTransit,
            },
            configurationInfo: props.configurationInfo,
            enhancedMonitoring: props.monitoring?.clusterMonitoringLevel,
            openMonitoring: openMonitoring,
            loggingInfo: loggingInfo,
            clientAuthentication: clientAuthentication,
        });
        this.clusterName = this.getResourceNameAttribute(core.Fn.select(1, core.Fn.split('/', resource.ref)));
        this.clusterArn = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.RETAIN,
        });
    }
    /**
     * Reference an existing cluster, defined outside of the CDK code, by name.
     */
    static fromClusterArn(scope, id, clusterArn) {
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.clusterArn = clusterArn;
                this.clusterName = core.Fn.select(1, core.Fn.split('/', clusterArn)); // ['arn:partition:kafka:region:account-id', clusterName, clusterId]
            }
        }
        return new Import(scope, id);
    }
    mskInstanceType(instanceType) {
        return `kafka.${instanceType.toString()}`;
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. ZookeeperConnectString, ZookeeperConnectStringTls
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    _zookeeperConnectionString(responseField) {
        if (!this._clusterDescription) {
            this._clusterDescription = new cr.AwsCustomResource(this, 'ZookeeperConnect', {
                onUpdate: {
                    service: 'Kafka',
                    action: 'describeCluster',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('ZooKeeperConnectionString'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
                installLatestAwsSdk: false,
            });
        }
        return this._clusterDescription.getResponseField(`ClusterInfo.${responseField}`);
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    get zookeeperConnectionString() {
        return this._zookeeperConnectionString('ZookeeperConnectString');
    }
    /**
     * Get the ZooKeeper Connection string for a TLS enabled cluster
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to zookeeper cluster on TLS port.
     */
    get zookeeperConnectionStringTls() {
        return this._zookeeperConnectionString('ZookeeperConnectStringTls');
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. BootstrapBrokerStringSaslScram, BootstrapBrokerString
     * @returns - A string containing one or more hostname:port pairs.
     */
    _bootstrapBrokers(responseField) {
        if (!this._clusterBootstrapBrokers) {
            this._clusterBootstrapBrokers = new cr.AwsCustomResource(this, `BootstrapBrokers${responseField}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'getBootstrapBrokers',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('BootstrapBrokers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
                // APIs are available in 2.1055.0
                installLatestAwsSdk: false,
            });
        }
        return this._clusterBootstrapBrokers.getResponseField(responseField);
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more hostname:port pairs.
     */
    get bootstrapBrokers() {
        return this._bootstrapBrokers('BootstrapBrokerString');
    }
    /**
     * Get the list of brokers that a TLS authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more DNS names (or IP) and TLS port pairs.
     */
    get bootstrapBrokersTls() {
        return this._bootstrapBrokers('BootstrapBrokerStringTls');
    }
    /**
     * Get the list of brokers that a SASL/SCRAM authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more dns name (or IP) and SASL SCRAM port pairs.
     */
    get bootstrapBrokersSaslScram() {
        return this._bootstrapBrokers('BootstrapBrokerStringSaslScram');
    }
    /**
     * Get the list of brokers that a SASL/IAM authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more DNS names (or IP) and TLS port pairs.
     */
    get bootstrapBrokersSaslIam() {
        return this._bootstrapBrokers('BootstrapBrokerStringSaslIam');
    }
    /**
     * A list of usersnames to register with the cluster. The password will automatically be generated using Secrets
     * Manager and the { username, password } JSON object stored in Secrets Manager as `AmazonMSK_username`.
     *
     * Must be using the SASL/SCRAM authentication mechanism.
     *
     * @param usernames - username(s) to register with the cluster
     */
    addUser(...usernames) {
        if (this.saslScramAuthenticationKey) {
            const MSK_SECRET_PREFIX = 'AmazonMSK_'; // Required
            const secrets = usernames.map((username) => new secretsmanager.Secret(this, `KafkaUser${username}`, {
                secretName: `${MSK_SECRET_PREFIX}${this.clusterName}_${username}`,
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({ username }),
                    generateStringKey: 'password',
                },
                encryptionKey: this.saslScramAuthenticationKey,
            }));
            new cr.AwsCustomResource(this, `BatchAssociateScramSecrets${uniqueid_1.addressOf(usernames)}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'batchAssociateScramSecret',
                    parameters: {
                        ClusterArn: this.clusterArn,
                        SecretArnList: secrets.map((secret) => secret.secretArn),
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('CreateUsers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['kms:CreateGrant'],
                        resources: [this.saslScramAuthenticationKey?.keyArn],
                    }),
                    new iam.PolicyStatement({
                        actions: ['kafka:BatchAssociateScramSecret'],
                        resources: [this.clusterArn],
                    }),
                ]),
                installLatestAwsSdk: false,
            });
        }
        else {
            throw Error('Cannot create users if an authentication KMS key has not been created/provided.');
        }
    }
}
exports.Cluster = Cluster;
_b = JSII_RTTI_SYMBOL_1;
Cluster[_b] = { fqn: "@aws-cdk/aws-msk-alpha.Cluster", version: "2.69.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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