"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClientAuthentication = exports.ClientBrokerEncryption = exports.ClusterMonitoringLevel = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const core = require("aws-cdk-lib");
const cr = require("aws-cdk-lib/custom-resources");
const uniqueid_1 = require("constructs/lib/private/uniqueid");
const aws_msk_1 = require("aws-cdk-lib/aws-msk");
/**
 * A new or imported MSK Cluster.
 */
class ClusterBase extends core.Resource {
    /**
     * (experimental) Manages connections for the cluster.
     *
     * @experimental
     */
    get connections() {
        if (!this._connections) {
            throw new Error('An imported Cluster cannot manage its security groups');
        }
        return this._connections;
    }
}
/**
 * (experimental) The level of monitoring for the MSK cluster.
 *
 * @see https://docs.aws.amazon.com/msk/latest/developerguide/monitoring.html#metrics-details
 * @experimental
 */
var ClusterMonitoringLevel;
(function (ClusterMonitoringLevel) {
    ClusterMonitoringLevel["DEFAULT"] = "DEFAULT";
    ClusterMonitoringLevel["PER_BROKER"] = "PER_BROKER";
    ClusterMonitoringLevel["PER_TOPIC_PER_BROKER"] = "PER_TOPIC_PER_BROKER";
    ClusterMonitoringLevel["PER_TOPIC_PER_PARTITION"] = "PER_TOPIC_PER_PARTITION";
})(ClusterMonitoringLevel = exports.ClusterMonitoringLevel || (exports.ClusterMonitoringLevel = {}));
/**
 * (experimental) Indicates the encryption setting for data in transit between clients and brokers.
 *
 * @experimental
 */
var ClientBrokerEncryption;
(function (ClientBrokerEncryption) {
    ClientBrokerEncryption["TLS"] = "TLS";
    ClientBrokerEncryption["TLS_PLAINTEXT"] = "TLS_PLAINTEXT";
    ClientBrokerEncryption["PLAINTEXT"] = "PLAINTEXT";
})(ClientBrokerEncryption = exports.ClientBrokerEncryption || (exports.ClientBrokerEncryption = {}));
/**
 * (experimental) Configuration properties for client authentication.
 *
 * @experimental
 */
class ClientAuthentication {
    /**
     * @param saslProps - properties for SASL authentication
     * @param tlsProps - properties for TLS authentication
     */
    constructor(saslProps, tlsProps) {
        this.saslProps = saslProps;
        this.tlsProps = tlsProps;
    }
    /**
     * (experimental) SASL authentication.
     *
     * @experimental
     */
    static sasl(props) {
        jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_SaslAuthProps(props);
        return new ClientAuthentication(props, undefined);
    }
    /**
     * (experimental) TLS authentication.
     *
     * @experimental
     */
    static tls(props) {
        jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_TlsAuthProps(props);
        return new ClientAuthentication(undefined, props);
    }
}
exports.ClientAuthentication = ClientAuthentication;
_a = JSII_RTTI_SYMBOL_1;
ClientAuthentication[_a] = { fqn: "@aws-cdk/aws-msk-alpha.ClientAuthentication", version: "2.0.0-alpha.11" };
/**
 * (experimental) Create a MSK Cluster.
 *
 * @experimental
 * @resource AWS::MSK::Cluster
 */
class Cluster extends ClusterBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23;
        super(scope, id, {
            physicalName: props.clusterName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_ClusterProps(props);
        const subnetSelection = props.vpc.selectSubnets(props.vpcSubnets);
        this._connections = new ec2.Connections({
            securityGroups: (_c = props.securityGroups) !== null && _c !== void 0 ? _c : [
                new ec2.SecurityGroup(this, 'SecurityGroup', {
                    description: 'MSK security group',
                    vpc: props.vpc,
                }),
            ],
        });
        if (subnetSelection.subnets.length < 2) {
            throw Error(`Cluster requires at least 2 subnets, got ${subnetSelection.subnets.length}`);
        }
        if (!core.Token.isUnresolved(props.clusterName) &&
            !/^[a-zA-Z0-9]+$/.test(props.clusterName) &&
            props.clusterName.length > 64) {
            throw Error('The cluster name must only contain alphanumeric characters and have a maximum length of 64 characters.' +
                `got: '${props.clusterName}. length: ${props.clusterName.length}'`);
        }
        if (((_e = (_d = props.clientAuthentication) === null || _d === void 0 ? void 0 : _d.saslProps) === null || _e === void 0 ? void 0 : _e.iam) && ((_g = (_f = props.clientAuthentication) === null || _f === void 0 ? void 0 : _f.saslProps) === null || _g === void 0 ? void 0 : _g.scram)) {
            throw Error('Only one client authentication method can be enabled.');
        }
        if (((_h = props.encryptionInTransit) === null || _h === void 0 ? void 0 : _h.clientBroker) ===
            ClientBrokerEncryption.PLAINTEXT &&
            props.clientAuthentication) {
            throw Error('To enable client authentication, you must enabled TLS-encrypted traffic between clients and brokers.');
        }
        else if (((_j = props.encryptionInTransit) === null || _j === void 0 ? void 0 : _j.clientBroker) ===
            ClientBrokerEncryption.TLS_PLAINTEXT &&
            (((_l = (_k = props.clientAuthentication) === null || _k === void 0 ? void 0 : _k.saslProps) === null || _l === void 0 ? void 0 : _l.scram) || ((_o = (_m = props.clientAuthentication) === null || _m === void 0 ? void 0 : _m.saslProps) === null || _o === void 0 ? void 0 : _o.iam))) {
            throw Error('To enable SASL/SCRAM or IAM authentication, you must only allow TLS-encrypted traffic between clients and brokers.');
        }
        const volumeSize = (_q = (_p = props.ebsStorageInfo) === null || _p === void 0 ? void 0 : _p.volumeSize) !== null && _q !== void 0 ? _q : 1000;
        // Minimum: 1 GiB, maximum: 16384 GiB
        if (volumeSize < 1 || volumeSize > 16384) {
            throw Error('EBS volume size should be in the range 1-16384');
        }
        const instanceType = props.instanceType
            ? this.mskInstanceType(props.instanceType)
            : this.mskInstanceType(ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE));
        const encryptionAtRest = ((_r = props.ebsStorageInfo) === null || _r === void 0 ? void 0 : _r.encryptionKey) ? {
            dataVolumeKmsKeyId: props.ebsStorageInfo.encryptionKey.keyId,
        }
            : undefined; // MSK will create the managed key
        const encryptionInTransit = {
            clientBroker: (_t = (_s = props.encryptionInTransit) === null || _s === void 0 ? void 0 : _s.clientBroker) !== null && _t !== void 0 ? _t : ClientBrokerEncryption.TLS,
            inCluster: (_v = (_u = props.encryptionInTransit) === null || _u === void 0 ? void 0 : _u.enableInCluster) !== null && _v !== void 0 ? _v : true,
        };
        const openMonitoring = ((_w = props.monitoring) === null || _w === void 0 ? void 0 : _w.enablePrometheusJmxExporter) || ((_x = props.monitoring) === null || _x === void 0 ? void 0 : _x.enablePrometheusNodeExporter)
            ? {
                prometheus: {
                    jmxExporter: ((_y = props.monitoring) === null || _y === void 0 ? void 0 : _y.enablePrometheusJmxExporter) ? { enabledInBroker: true }
                        : undefined,
                    nodeExporter: ((_z = props.monitoring) === null || _z === void 0 ? void 0 : _z.enablePrometheusNodeExporter) ? { enabledInBroker: true }
                        : undefined,
                },
            }
            : undefined;
        const loggingInfo = {
            brokerLogs: {
                cloudWatchLogs: {
                    enabled: ((_0 = props.logging) === null || _0 === void 0 ? void 0 : _0.cloudwatchLogGroup) !== undefined,
                    logGroup: (_2 = (_1 = props.logging) === null || _1 === void 0 ? void 0 : _1.cloudwatchLogGroup) === null || _2 === void 0 ? void 0 : _2.logGroupName,
                },
                firehose: {
                    enabled: ((_3 = props.logging) === null || _3 === void 0 ? void 0 : _3.firehoseDeliveryStreamName) !==
                        undefined,
                    deliveryStream: (_4 = props.logging) === null || _4 === void 0 ? void 0 : _4.firehoseDeliveryStreamName,
                },
                s3: {
                    enabled: ((_6 = (_5 = props.logging) === null || _5 === void 0 ? void 0 : _5.s3) === null || _6 === void 0 ? void 0 : _6.bucket) !== undefined,
                    bucket: (_8 = (_7 = props.logging) === null || _7 === void 0 ? void 0 : _7.s3) === null || _8 === void 0 ? void 0 : _8.bucket.bucketName,
                    prefix: (_10 = (_9 = props.logging) === null || _9 === void 0 ? void 0 : _9.s3) === null || _10 === void 0 ? void 0 : _10.prefix,
                },
            },
        };
        if (((_12 = (_11 = props.clientAuthentication) === null || _11 === void 0 ? void 0 : _11.saslProps) === null || _12 === void 0 ? void 0 : _12.scram) &&
            ((_14 = (_13 = props.clientAuthentication) === null || _13 === void 0 ? void 0 : _13.saslProps) === null || _14 === void 0 ? void 0 : _14.key) === undefined) {
            this.saslScramAuthenticationKey = new kms.Key(this, 'SASLKey', {
                description: 'Used for encrypting MSK secrets for SASL/SCRAM authentication.',
                alias: `msk/${props.clusterName}/sasl/scram`,
            });
            // https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-policies
            this.saslScramAuthenticationKey.addToResourcePolicy(new iam.PolicyStatement({
                sid: 'Allow access through AWS Secrets Manager for all principals in the account that are authorized to use AWS Secrets Manager',
                principals: [new iam.AnyPrincipal()],
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:CreateGrant',
                    'kms:DescribeKey',
                ],
                resources: ['*'],
                conditions: {
                    StringEquals: {
                        'kms:ViaService': `secretsmanager.${core.Stack.of(this).region}.amazonaws.com`,
                        'kms:CallerAccount': core.Stack.of(this).account,
                    },
                },
            }));
        }
        let clientAuthentication;
        if ((_16 = (_15 = props.clientAuthentication) === null || _15 === void 0 ? void 0 : _15.saslProps) === null || _16 === void 0 ? void 0 : _16.iam) {
            clientAuthentication = {
                sasl: { iam: { enabled: props.clientAuthentication.saslProps.iam } },
            };
        }
        else if ((_18 = (_17 = props.clientAuthentication) === null || _17 === void 0 ? void 0 : _17.saslProps) === null || _18 === void 0 ? void 0 : _18.scram) {
            clientAuthentication = {
                sasl: {
                    scram: {
                        enabled: props.clientAuthentication.saslProps.scram,
                    },
                },
            };
        }
        else if (((_20 = (_19 = props.clientAuthentication) === null || _19 === void 0 ? void 0 : _19.tlsProps) === null || _20 === void 0 ? void 0 : _20.certificateAuthorities) !== undefined) {
            clientAuthentication = {
                tls: {
                    certificateAuthorityArnList: (_22 = (_21 = props.clientAuthentication) === null || _21 === void 0 ? void 0 : _21.tlsProps) === null || _22 === void 0 ? void 0 : _22.certificateAuthorities.map((ca) => ca.certificateAuthorityArn),
                },
            };
        }
        const resource = new aws_msk_1.CfnCluster(this, 'Resource', {
            clusterName: props.clusterName,
            kafkaVersion: props.kafkaVersion.version,
            numberOfBrokerNodes: props.numberOfBrokerNodes !== undefined ?
                subnetSelection.availabilityZones.length * props.numberOfBrokerNodes : subnetSelection.availabilityZones.length,
            brokerNodeGroupInfo: {
                instanceType,
                clientSubnets: subnetSelection.subnetIds,
                securityGroups: this.connections.securityGroups.map((group) => group.securityGroupId),
                storageInfo: {
                    ebsStorageInfo: {
                        volumeSize: volumeSize,
                    },
                },
            },
            encryptionInfo: {
                encryptionAtRest,
                encryptionInTransit,
            },
            configurationInfo: props.configurationInfo,
            enhancedMonitoring: (_23 = props.monitoring) === null || _23 === void 0 ? void 0 : _23.clusterMonitoringLevel,
            openMonitoring: openMonitoring,
            loggingInfo: loggingInfo,
            clientAuthentication: clientAuthentication,
        });
        this.clusterName = this.getResourceNameAttribute(core.Fn.select(1, core.Fn.split('/', resource.ref)));
        this.clusterArn = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.RETAIN,
        });
    }
    /**
     * (experimental) Reference an existing cluster, defined outside of the CDK code, by name.
     *
     * @experimental
     */
    static fromClusterArn(scope, id, clusterArn) {
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.clusterArn = clusterArn;
                this.clusterName = core.Fn.select(1, core.Fn.split('/', clusterArn)); // ['arn:partition:kafka:region:account-id', clusterName, clusterId]
            }
        }
        return new Import(scope, id);
    }
    mskInstanceType(instanceType) {
        return `kafka.${instanceType.toString()}`;
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. ZookeeperConnectString, ZookeeperConnectStringTls
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    _zookeeperConnectionString(responseField) {
        if (!this._clusterDescription) {
            this._clusterDescription = new cr.AwsCustomResource(this, 'ZookeeperConnect', {
                onUpdate: {
                    service: 'Kafka',
                    action: 'describeCluster',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('ZooKeeperConnectionString'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
            });
        }
        return this._clusterDescription.getResponseField(`ClusterInfo.${responseField}`);
    }
    /**
     * (experimental) Get the ZooKeeper Connection string.
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     * @experimental
     */
    get zookeeperConnectionString() {
        return this._zookeeperConnectionString('ZookeeperConnectString');
    }
    /**
     * (experimental) Get the ZooKeeper Connection string for a TLS enabled cluster.
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to zookeeper cluster on TLS port.
     * @experimental
     */
    get zookeeperConnectionStringTls() {
        return this._zookeeperConnectionString('ZookeeperConnectStringTls');
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. BootstrapBrokerStringSaslScram, BootstrapBrokerString
     * @returns - A string containing one or more hostname:port pairs.
     */
    _bootstrapBrokers(responseField) {
        if (!this._clusterBootstrapBrokers) {
            this._clusterBootstrapBrokers = new cr.AwsCustomResource(this, `BootstrapBrokers${responseField}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'getBootstrapBrokers',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('BootstrapBrokers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
            });
        }
        return this._clusterBootstrapBrokers.getResponseField(responseField);
    }
    /**
     * (experimental) Get the list of brokers that a client application can use to bootstrap.
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more hostname:port pairs.
     * @experimental
     */
    get bootstrapBrokers() {
        return this._bootstrapBrokers('BootstrapBrokerString');
    }
    /**
     * (experimental) Get the list of brokers that a TLS authenticated client application can use to bootstrap.
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more DNS names (or IP) and TLS port pairs.
     * @experimental
     */
    get bootstrapBrokersTls() {
        return this._bootstrapBrokers('BootstrapBrokerStringTls');
    }
    /**
     * (experimental) Get the list of brokers that a SASL/SCRAM authenticated client application can use to bootstrap.
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more dns name (or IP) and SASL SCRAM port pairs.
     * @experimental
     */
    get bootstrapBrokersSaslScram() {
        return this._bootstrapBrokers('BootstrapBrokerStringSaslScram');
    }
    /**
     * (experimental) A list of usersnames to register with the cluster.
     *
     * The password will automatically be generated using Secrets
     * Manager and the { username, password } JSON object stored in Secrets Manager as `AmazonMSK_username`.
     *
     * Must be using the SASL/SCRAM authentication mechanism.
     *
     * @param usernames - username(s) to register with the cluster.
     * @experimental
     */
    addUser(...usernames) {
        var _c;
        if (this.saslScramAuthenticationKey) {
            const MSK_SECRET_PREFIX = 'AmazonMSK_'; // Required
            const secrets = usernames.map((username) => new secretsmanager.Secret(this, `KafkaUser${username}`, {
                secretName: `${MSK_SECRET_PREFIX}${this.clusterName}_${username}`,
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({ username }),
                    generateStringKey: 'password',
                },
                encryptionKey: this.saslScramAuthenticationKey,
            }));
            new cr.AwsCustomResource(this, `BatchAssociateScramSecrets${uniqueid_1.addressOf(usernames)}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'batchAssociateScramSecret',
                    parameters: {
                        ClusterArn: this.clusterArn,
                        SecretArnList: secrets.map((secret) => secret.secretArn),
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('CreateUsers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['kms:CreateGrant'],
                        resources: [(_c = this.saslScramAuthenticationKey) === null || _c === void 0 ? void 0 : _c.keyArn],
                    }),
                    new iam.PolicyStatement({
                        actions: ['kafka:BatchAssociateScramSecret'],
                        resources: [this.clusterArn],
                    }),
                ]),
            });
        }
        else {
            throw Error('Cannot create users if an authentication KMS key has not been created/provided.');
        }
    }
}
exports.Cluster = Cluster;
_b = JSII_RTTI_SYMBOL_1;
Cluster[_b] = { fqn: "@aws-cdk/aws-msk-alpha.Cluster", version: "2.0.0-alpha.11" };
//# sourceMappingURL=data:application/json;base64,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