"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const targets = require("../../lib");
describe('CodePipeline event target', () => {
    let stack;
    let pipeline;
    let pipelineArn;
    beforeEach(() => {
        stack = new core_1.Stack();
        pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
        const srcArtifact = new codepipeline.Artifact('Src');
        const buildArtifact = new codepipeline.Artifact('Bld');
        pipeline.addStage({
            stageName: 'Source',
            actions: [new TestAction({
                    actionName: 'Hello',
                    category: codepipeline.ActionCategory.SOURCE,
                    provider: 'x',
                    artifactBounds: { minInputs: 0, maxInputs: 0, minOutputs: 1, maxOutputs: 1 },
                    outputs: [srcArtifact]
                })],
        });
        pipeline.addStage({
            stageName: 'Build',
            actions: [new TestAction({
                    actionName: 'Hello',
                    category: codepipeline.ActionCategory.BUILD,
                    provider: 'y',
                    inputs: [srcArtifact],
                    outputs: [buildArtifact],
                    artifactBounds: { minInputs: 1, maxInputs: 1, minOutputs: 1, maxOutputs: 1 }
                })],
        });
        pipelineArn = {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':codepipeline:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'PipelineC660917D' }
                ],
            ],
        };
    });
    describe('when added to an event rule as a target', () => {
        let rule;
        beforeEach(() => {
            rule = new events.Rule(stack, 'rule', {
                schedule: events.Schedule.expression('rate(1 minute)'),
            });
        });
        describe('with default settings', () => {
            beforeEach(() => {
                rule.addTarget(new targets.CodePipeline(pipeline));
            });
            test("adds the pipeline's ARN and role to the targets of the rule", () => {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
                    Targets: [
                        {
                            Arn: pipelineArn,
                            Id: 'Target0',
                            RoleArn: { 'Fn::GetAtt': ['PipelineEventsRole46BEEA7C', 'Arn'] },
                        },
                    ],
                }));
            });
            test("creates a policy that has StartPipeline permissions on the pipeline's ARN", () => {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: [
                            {
                                Action: 'codepipeline:StartPipelineExecution',
                                Effect: 'Allow',
                                Resource: pipelineArn,
                            },
                        ],
                        Version: '2012-10-17',
                    },
                }));
            });
        });
        describe('with an explicit event role', () => {
            beforeEach(() => {
                const role = new iam.Role(stack, 'MyExampleRole', {
                    assumedBy: new iam.AnyPrincipal(),
                });
                const roleResource = role.node.defaultChild;
                roleResource.overrideLogicalId('MyRole'); // to make it deterministic in the assertion below
                rule.addTarget(new targets.CodePipeline(pipeline, {
                    eventRole: role,
                }));
            });
            test("points at the given event role in the rule's targets", () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                    Targets: [
                        {
                            Arn: pipelineArn,
                            RoleArn: { 'Fn::GetAtt': ['MyRole', 'Arn'] },
                        },
                    ],
                }));
            });
        });
    });
});
class TestAction {
    constructor(actionProperties) {
        this.actionProperties = actionProperties;
        // nothing to do
    }
    bind(_scope, _stage, _options) {
        return {};
    }
    onStateChange(_name, _target, _options) {
        throw new Error('onStateChange() is not available on MockAction');
    }
}
//# sourceMappingURL=data:application/json;base64,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