"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const codebuild = require("@aws-cdk/aws-codebuild");
const events = require("@aws-cdk/aws-events");
const core_1 = require("@aws-cdk/core");
const targets = require("../../lib");
test('use codebuild project as an eventrule target', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const project = new codebuild.PipelineProject(stack, 'MyProject');
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(1 min)'),
    });
    // WHEN
    rule.addTarget(new targets.CodeBuildProject(project));
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
        Targets: [
            {
                Arn: {
                    'Fn::GetAtt': [
                        'MyProject39F7B0AE',
                        'Arn',
                    ],
                },
                Id: 'Target0',
                RoleArn: {
                    'Fn::GetAtt': [
                        'MyProjectEventsRole5B7D93F5',
                        'Arn',
                    ],
                },
            },
        ],
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                    Principal: { Service: 'events.amazonaws.com' },
                },
            ],
            Version: '2012-10-17',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'codebuild:StartBuild',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'MyProject39F7B0AE',
                            'Arn',
                        ],
                    },
                },
            ],
            Version: '2012-10-17',
        },
    }));
});
test('specifying event for codebuild project target', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const project = new codebuild.PipelineProject(stack, 'MyProject');
    const rule = new events.Rule(stack, 'Rule', {
        schedule: events.Schedule.expression('rate(1 hour)'),
    });
    // WHEN
    const eventInput = {
        buildspecOverride: 'buildspecs/hourly.yml',
    };
    rule.addTarget(new targets.CodeBuildProject(project, {
        event: events.RuleTargetInput.fromObject(eventInput),
    }));
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
        Targets: [
            {
                Arn: {
                    'Fn::GetAtt': ['MyProject39F7B0AE', 'Arn'],
                },
                Id: 'Target0',
                Input: JSON.stringify(eventInput),
                RoleArn: {
                    'Fn::GetAtt': ['MyProjectEventsRole5B7D93F5', 'Arn'],
                },
            },
        ],
    }));
});
//# sourceMappingURL=data:application/json;base64,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