"""Chrome setup service for installing wrapper scripts (cross-platform)."""

import os
import logging
from typing import Dict, Any

from .platform import get_platform_info, ensure_directories

logger = logging.getLogger(__name__)

# Wrapper script templates
LINUX_BINDFS_WRAPPER = """#!/bin/bash
# Chrome Debug Launcher for Linux (bindfs mode)
# Generated by webtap setup-chrome --bindfs
# Mounts your real Chrome profile via bindfs for debugging with full session state

DEBUG_DIR="$HOME/.config/google-chrome-debug"
REAL_DIR="$HOME/.config/google-chrome"
PORT=${{WEBTAP_PORT:-9222}}
CHROME_BIN="{chrome_path}"

# Check if bindfs is installed
if ! command -v bindfs &>/dev/null; then
    echo "Error: bindfs not installed. Install with:" >&2
    echo "  Ubuntu/Debian: sudo apt install bindfs" >&2
    echo "  Arch: yay -S bindfs" >&2
    echo "  Fedora: sudo dnf install bindfs" >&2
    exit 1
fi

# Check if real Chrome profile exists
if [ ! -d "$REAL_DIR" ]; then
    echo "Error: Chrome profile not found at $REAL_DIR" >&2
    echo "Please run Chrome normally first to create a profile" >&2
    exit 1
fi

# Mount real profile via bindfs if not already mounted
if ! mountpoint -q "$DEBUG_DIR" 2>/dev/null; then
    mkdir -p "$DEBUG_DIR"
    if ! bindfs --no-allow-other "$REAL_DIR" "$DEBUG_DIR"; then
        echo "Error: Failed to mount Chrome profile via bindfs" >&2
        exit 1
    fi
    echo "Chrome debug profile mounted. To unmount: fusermount -u $DEBUG_DIR" >&2
fi

# Launch Chrome with debugging on bindfs mount
exec "$CHROME_BIN" \\
    --remote-debugging-port="$PORT" \\
    --remote-allow-origins='*' \\
    --user-data-dir="$DEBUG_DIR" \\
    "$@"
"""

LINUX_STANDARD_WRAPPER = """#!/bin/bash
# Chrome Debug Launcher for Linux
# Generated by webtap setup-chrome

# Configuration
PORT=${{WEBTAP_PORT:-9222}}
PROFILE_BASE="{profile_dir}"
CHROME_BIN="{chrome_path}"

# Profile handling
if [ "$1" = "--temp" ]; then
    PROFILE_DIR="$(mktemp -d /tmp/webtap-chrome-XXXXXX)"
    shift
    echo "Using temporary profile: $PROFILE_DIR" >&2
else
    PROFILE_DIR="$PROFILE_BASE/default"
    mkdir -p "$PROFILE_DIR"
fi

# Launch Chrome with debugging
exec "$CHROME_BIN" \\
    --remote-debugging-port="$PORT" \\
    --remote-allow-origins='*' \\
    --user-data-dir="$PROFILE_DIR" \\
    --no-first-run \\
    --no-default-browser-check \\
    "$@"
"""

MACOS_WRAPPER = """#!/bin/bash
# Chrome Debug Launcher for macOS
# Generated by webtap setup-chrome

# Configuration
PORT=${{WEBTAP_PORT:-9222}}
PROFILE_BASE="{profile_dir}"
CHROME_APP="{chrome_path}"

# Profile handling
if [ "$1" = "--temp" ]; then
    PROFILE_DIR="$(mktemp -d /tmp/webtap-chrome-XXXXXX)"
    shift
    echo "Using temporary profile: $PROFILE_DIR" >&2
else
    PROFILE_DIR="$PROFILE_BASE/default"
    mkdir -p "$PROFILE_DIR"
fi

# Launch Chrome with debugging
exec "$CHROME_APP" \\
    --remote-debugging-port="$PORT" \\
    --remote-allow-origins='*' \\
    --user-data-dir="$PROFILE_DIR" \\
    --no-first-run \\
    --no-default-browser-check \\
    "$@"
"""


class ChromeSetupService:
    """Chrome wrapper installation service (cross-platform)."""

    def __init__(self):
        self.info = get_platform_info()
        self.paths = self.info["paths"]
        self.chrome = self.info["chrome"]

        # Unified wrapper location for both platforms
        self.wrapper_dir = self.paths["bin_dir"]  # ~/.local/bin
        self.wrapper_name = self.chrome["wrapper_name"]  # chrome-debug
        self.wrapper_path = self.wrapper_dir / self.wrapper_name

        # Profile locations
        self.profile_dir = self.paths["data_dir"] / "profiles"
        self.temp_profile_dir = self.paths["runtime_dir"] / "profiles"

    def install_wrapper(self, force: bool = False, bindfs: bool = False) -> Dict[str, Any]:
        """Install Chrome wrapper script appropriate for platform.

        Args:
            force: Overwrite existing wrapper
            bindfs: Use bindfs to mount real Chrome profile (Linux only)

        Returns:
            Installation result
        """
        if not self.chrome["found"]:
            return {
                "success": False,
                "message": "Chrome not found on system",
                "details": "Please install Google Chrome first",
            }

        if self.wrapper_path.exists() and not force:
            return {
                "success": False,
                "message": f"Wrapper already exists at {self.wrapper_path}",
                "details": "Use --force to overwrite",
                "path": str(self.wrapper_path),
            }

        # Ensure directories exist
        ensure_directories()
        self.wrapper_dir.mkdir(parents=True, exist_ok=True)

        # Generate platform-specific wrapper
        if self.info["is_macos"]:
            wrapper_content = self._generate_macos_wrapper()
        else:
            wrapper_content = self._generate_linux_wrapper(bindfs=bindfs)

        self.wrapper_path.write_text(wrapper_content)
        self.wrapper_path.chmod(0o755)

        # Check if wrapper dir is in PATH
        path_dirs = os.environ.get("PATH", "").split(os.pathsep)
        in_path = str(self.wrapper_dir) in path_dirs

        logger.info(f"Installed Chrome wrapper to {self.wrapper_path}")

        # Build detailed message with PATH setup instructions
        if in_path:
            details = "✅ Run 'chrome-debug' to launch Chrome with debugging"
        else:
            shell = os.environ.get("SHELL", "/bin/bash")
            if "zsh" in shell:
                rc_file = "~/.zshrc"
            elif "fish" in shell:
                rc_file = "~/.config/fish/config.fish"
            else:
                rc_file = "~/.bashrc"

            details = (
                f"⚠️  Add to PATH to use from terminal:\n"
                f"echo 'export PATH=\"$HOME/.local/bin:$PATH\"' >> {rc_file}\n"
                f"source {rc_file}\n\n"
                f"Or run directly: ~/.local/bin/chrome-debug\n"
                f"✅ GUI launcher will work regardless (uses full path)"
            )

        return {
            "success": True,
            "message": "Chrome wrapper 'chrome-debug' installed successfully",
            "path": str(self.wrapper_path),
            "details": details,
        }

    def _generate_linux_wrapper(self, bindfs: bool = False) -> str:
        """Generate Linux wrapper script with optional bindfs support.

        Args:
            bindfs: If True, use bindfs to mount real Chrome profile

        Returns:
            Wrapper script content
        """
        chrome_path = self.chrome["path"]
        profile_dir = self.profile_dir

        if bindfs:
            return LINUX_BINDFS_WRAPPER.format(chrome_path=chrome_path)
        else:
            return LINUX_STANDARD_WRAPPER.format(chrome_path=chrome_path, profile_dir=profile_dir)

    def _generate_macos_wrapper(self) -> str:
        """Generate macOS wrapper script.

        Returns:
            Wrapper script content
        """
        chrome_path = self.chrome["path"]
        profile_dir = self.profile_dir

        return MACOS_WRAPPER.format(chrome_path=chrome_path, profile_dir=profile_dir)
