"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstanceEngine = exports.SqlServerEngineVersion = exports.OracleEngineVersion = exports.OracleLegacyEngineVersion = exports.PostgresEngineVersion = exports.MysqlEngineVersion = exports.MariaDbEngineVersion = void 0;
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const option_group_1 = require("./option-group");
class InstanceEngineBase {
    constructor(props) {
        var _a;
        this.engineType = props.engineType;
        this.features = props.features;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.engineVersion = props.version;
        this.parameterGroupFamily = (_a = props.parameterGroupFamily) !== null && _a !== void 0 ? _a : (this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined);
        this.engineFamily = props.engineFamily;
    }
    bindToInstance(_scope, options) {
        if (options.timezone && !this.supportsTimezone) {
            throw new Error(`timezone property can not be configured for ${this.engineType}`);
        }
        return {
            features: this.features,
            optionGroup: options.optionGroup,
        };
    }
    /** Defines whether this Instance Engine can support timezone properties. */
    get supportsTimezone() { return false; }
}
/**
 * The versions for the MariaDB instance engines (those returned by {@link DatabaseInstanceEngine.mariaDb}).
 *
 * @stability stable
 */
class MariaDbEngineVersion {
    constructor(mariaDbFullVersion, mariaDbMajorVersion) {
        this.mariaDbFullVersion = mariaDbFullVersion;
        this.mariaDbMajorVersion = mariaDbMajorVersion;
    }
    /**
     * Create a new MariaDbEngineVersion with an arbitrary version.
     *
     * @param mariaDbFullVersion the full version string, for example "10.5.28".
     * @param mariaDbMajorVersion the major version of the engine, for example "10.5".
     * @stability stable
     */
    static of(mariaDbFullVersion, mariaDbMajorVersion) {
        return new MariaDbEngineVersion(mariaDbFullVersion, mariaDbMajorVersion);
    }
}
exports.MariaDbEngineVersion = MariaDbEngineVersion;
/**
 * Version "10.0" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_0 = MariaDbEngineVersion.of('10.0', '10.0');
/**
 * Version "10.0.17".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_0_17 = MariaDbEngineVersion.of('10.0.17', '10.0');
/**
 * Version "10.0.24".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_0_24 = MariaDbEngineVersion.of('10.0.24', '10.0');
/**
 * Version "10.0.28".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_0_28 = MariaDbEngineVersion.of('10.0.28', '10.0');
/**
 * Version "10.0.31".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_0_31 = MariaDbEngineVersion.of('10.0.31', '10.0');
/**
 * Version "10.0.32".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_0_32 = MariaDbEngineVersion.of('10.0.32', '10.0');
/**
 * Version "10.0.34".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_0_34 = MariaDbEngineVersion.of('10.0.34', '10.0');
/**
 * Version "10.0.35".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_0_35 = MariaDbEngineVersion.of('10.0.35', '10.0');
/**
 * Version "10.1" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_1 = MariaDbEngineVersion.of('10.1', '10.1');
/**
 * Version "10.1.14".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_1_14 = MariaDbEngineVersion.of('10.1.14', '10.1');
/**
 * Version "10.1.19".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_1_19 = MariaDbEngineVersion.of('10.1.19', '10.1');
/**
 * Version "10.1.23".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_1_23 = MariaDbEngineVersion.of('10.1.23', '10.1');
/**
 * Version "10.1.26".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_1_26 = MariaDbEngineVersion.of('10.1.26', '10.1');
/**
 * Version "10.1.31".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_1_31 = MariaDbEngineVersion.of('10.1.31', '10.1');
/**
 * Version "10.1.34".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_1_34 = MariaDbEngineVersion.of('10.1.34', '10.1');
/**
 * Version "10.2" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_2 = MariaDbEngineVersion.of('10.2', '10.2');
/**
 * Version "10.2.11".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_2_11 = MariaDbEngineVersion.of('10.2.11', '10.2');
/**
 * Version "10.2.12".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_2_12 = MariaDbEngineVersion.of('10.2.12', '10.2');
/**
 * Version "10.2.15".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_2_15 = MariaDbEngineVersion.of('10.2.15', '10.2');
/**
 * Version "10.2.21".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_2_21 = MariaDbEngineVersion.of('10.2.21', '10.2');
/**
 * Version "10.3" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_3 = MariaDbEngineVersion.of('10.3', '10.3');
/**
 * Version "10.3.8".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_3_8 = MariaDbEngineVersion.of('10.3.8', '10.3');
/**
 * Version "10.3.13".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_3_13 = MariaDbEngineVersion.of('10.3.13', '10.3');
/**
 * Version "10.3.20".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_3_20 = MariaDbEngineVersion.of('10.3.20', '10.3');
/**
 * Version "10.3.23".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_3_23 = MariaDbEngineVersion.of('10.3.23', '10.3');
/**
 * Version "10.4" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_4 = MariaDbEngineVersion.of('10.4', '10.4');
/**
 * Version "10.4.8".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_4_8 = MariaDbEngineVersion.of('10.4.8', '10.4');
/**
 * Version "10.4.13".
 *
 * @stability stable
 */
MariaDbEngineVersion.VER_10_4_13 = MariaDbEngineVersion.of('10.4.13', '10.4');
class MariaDbInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'mariadb',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MARIADB_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.mariaDbFullVersion,
                    majorVersion: version.mariaDbMajorVersion,
                }
                : undefined,
        });
    }
    bindToInstance(scope, options) {
        if (options.domain) {
            throw new Error(`domain property cannot be configured for ${this.engineType}`);
        }
        return super.bindToInstance(scope, options);
    }
}
/**
 * The versions for the MySQL instance engines (those returned by {@link DatabaseInstanceEngine.mysql}).
 *
 * @stability stable
 */
class MysqlEngineVersion {
    constructor(mysqlFullVersion, mysqlMajorVersion) {
        this.mysqlFullVersion = mysqlFullVersion;
        this.mysqlMajorVersion = mysqlMajorVersion;
    }
    /**
     * Create a new MysqlEngineVersion with an arbitrary version.
     *
     * @param mysqlFullVersion the full version string, for example "8.1.43".
     * @param mysqlMajorVersion the major version of the engine, for example "8.1".
     * @stability stable
     */
    static of(mysqlFullVersion, mysqlMajorVersion) {
        return new MysqlEngineVersion(mysqlFullVersion, mysqlMajorVersion);
    }
}
exports.MysqlEngineVersion = MysqlEngineVersion;
/**
 * Version "5.5" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_5 = MysqlEngineVersion.of('5.5', '5.5');
/**
 * Version "5.5.46".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_5_46 = MysqlEngineVersion.of('5.5.46', '5.5');
/**
 * Version "5.5.53".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_5_53 = MysqlEngineVersion.of('5.5.53', '5.5');
/**
 * Version "5.5.57".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_5_57 = MysqlEngineVersion.of('5.5.57', '5.5');
/**
 * Version "5.5.59".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_5_59 = MysqlEngineVersion.of('5.5.59', '5.5');
/**
 * Version "5.5.61".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_5_61 = MysqlEngineVersion.of('5.5.61', '5.5');
/**
 * Version "5.6" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_6 = MysqlEngineVersion.of('5.6', '5.6');
/**
 * Version "5.6.34".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_6_34 = MysqlEngineVersion.of('5.6.34', '5.6');
/**
 * Version "5.6.35".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_6_35 = MysqlEngineVersion.of('5.6.35', '5.6');
/**
 * Version "5.6.37".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_6_37 = MysqlEngineVersion.of('5.6.37', '5.6');
/**
 * Version "5.6.39".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_6_39 = MysqlEngineVersion.of('5.6.39', '5.6');
/**
 * Version "5.6.40".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_6_40 = MysqlEngineVersion.of('5.6.40', '5.6');
/**
 * Version "5.6.41".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_6_41 = MysqlEngineVersion.of('5.6.41', '5.6');
/**
 * Version "5.6.43".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_6_43 = MysqlEngineVersion.of('5.6.43', '5.6');
/**
 * Version "5.6.44".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_6_44 = MysqlEngineVersion.of('5.6.44', '5.6');
/**
 * Version "5.6.46".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_6_46 = MysqlEngineVersion.of('5.6.46', '5.6');
/**
 * Version "5.6.48".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_6_48 = MysqlEngineVersion.of('5.6.48', '5.6');
/**
 * Version "5.7" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_7 = MysqlEngineVersion.of('5.7', '5.7');
/**
 * Version "5.7.16".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_7_16 = MysqlEngineVersion.of('5.7.16', '5.7');
/**
 * Version "5.7.17".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_7_17 = MysqlEngineVersion.of('5.7.17', '5.7');
/**
 * Version "5.7.19".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_7_19 = MysqlEngineVersion.of('5.7.19', '5.7');
/**
 * Version "5.7.21".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_7_21 = MysqlEngineVersion.of('5.7.21', '5.7');
/**
 * Version "5.7.22".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_7_22 = MysqlEngineVersion.of('5.7.22', '5.7');
/**
 * Version "5.7.23".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_7_23 = MysqlEngineVersion.of('5.7.23', '5.7');
/**
 * Version "5.7.24".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_7_24 = MysqlEngineVersion.of('5.7.24', '5.7');
/**
 * Version "5.7.25".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_7_25 = MysqlEngineVersion.of('5.7.25', '5.7');
/**
 * Version "5.7.26".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_7_26 = MysqlEngineVersion.of('5.7.26', '5.7');
/**
 * Version "5.7.28".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_7_28 = MysqlEngineVersion.of('5.7.28', '5.7');
/**
 * Version "5.7.30".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_7_30 = MysqlEngineVersion.of('5.7.30', '5.7');
/**
 * Version "5.7.31".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_5_7_31 = MysqlEngineVersion.of('5.7.31', '5.7');
/**
 * Version "8.0" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
MysqlEngineVersion.VER_8_0 = MysqlEngineVersion.of('8.0', '8.0');
/**
 * Version "8.0.11".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_8_0_11 = MysqlEngineVersion.of('8.0.11', '8.0');
/**
 * Version "8.0.13".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_8_0_13 = MysqlEngineVersion.of('8.0.13', '8.0');
/**
 * Version "8.0.15".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_8_0_15 = MysqlEngineVersion.of('8.0.15', '8.0');
/**
 * Version "8.0.16".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_8_0_16 = MysqlEngineVersion.of('8.0.16', '8.0');
/**
 * Version "8.0.17".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_8_0_17 = MysqlEngineVersion.of('8.0.17', '8.0');
/**
 * Version "8.0.19".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_8_0_19 = MysqlEngineVersion.of('8.0.19', '8.0');
/**
 * Version "8.0.20 ".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_8_0_20 = MysqlEngineVersion.of('8.0.20', '8.0');
/**
 * Version "8.0.21 ".
 *
 * @stability stable
 */
MysqlEngineVersion.VER_8_0_21 = MysqlEngineVersion.of('8.0.21', '8.0');
class MySqlInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'mysql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.mysqlFullVersion,
                    majorVersion: version.mysqlMajorVersion,
                }
                : undefined,
            engineFamily: 'MYSQL',
        });
    }
}
/**
 * The versions for the PostgreSQL instance engines (those returned by {@link DatabaseInstanceEngine.postgres}).
 *
 * @stability stable
 */
class PostgresEngineVersion {
    constructor(postgresFullVersion, postgresMajorVersion, postgresFeatures) {
        this.postgresFullVersion = postgresFullVersion;
        this.postgresMajorVersion = postgresMajorVersion;
        this._features = {
            s3Import: (postgresFeatures === null || postgresFeatures === void 0 ? void 0 : postgresFeatures.s3Import) ? 's3Import' : undefined,
        };
    }
    /**
     * Create a new PostgresEngineVersion with an arbitrary version.
     *
     * @param postgresFullVersion the full version string, for example "13.11".
     * @param postgresMajorVersion the major version of the engine, for example "13".
     * @stability stable
     */
    static of(postgresFullVersion, postgresMajorVersion, postgresFeatures) {
        return new PostgresEngineVersion(postgresFullVersion, postgresMajorVersion, postgresFeatures);
    }
}
exports.PostgresEngineVersion = PostgresEngineVersion;
/**
 * Version "9.5" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5 = PostgresEngineVersion.of('9.5', '9.5');
/**
 * Version "9.5.2".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_2 = PostgresEngineVersion.of('9.5.2', '9.5');
/**
 * Version "9.5.4".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_4 = PostgresEngineVersion.of('9.5.4', '9.5');
/**
 * Version "9.5.6".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_6 = PostgresEngineVersion.of('9.5.6', '9.5');
/**
 * Version "9.5.7".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_7 = PostgresEngineVersion.of('9.5.7', '9.5');
/**
 * Version "9.5.9".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_9 = PostgresEngineVersion.of('9.5.9', '9.5');
/**
 * Version "9.5.10".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_10 = PostgresEngineVersion.of('9.5.10', '9.5');
/**
 * Version "9.5.12".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_12 = PostgresEngineVersion.of('9.5.12', '9.5');
/**
 * Version "9.5.13".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_13 = PostgresEngineVersion.of('9.5.13', '9.5');
/**
 * Version "9.5.14".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_14 = PostgresEngineVersion.of('9.5.14', '9.5');
/**
 * Version "9.5.15".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_15 = PostgresEngineVersion.of('9.5.15', '9.5');
/**
 * Version "9.5.16".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_16 = PostgresEngineVersion.of('9.5.16', '9.5');
/**
 * Version "9.5.18".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_18 = PostgresEngineVersion.of('9.5.18', '9.5');
/**
 * Version "9.5.19".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_19 = PostgresEngineVersion.of('9.5.19', '9.5');
/**
 * Version "9.5.20".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_20 = PostgresEngineVersion.of('9.5.20', '9.5');
/**
 * Version "9.5.21".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_21 = PostgresEngineVersion.of('9.5.21', '9.5');
/**
 * Version "9.5.22".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_22 = PostgresEngineVersion.of('9.5.22', '9.5');
/**
 * Version "9.5.23".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_5_23 = PostgresEngineVersion.of('9.5.23', '9.5');
/**
 * Version "9.6" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6 = PostgresEngineVersion.of('9.6', '9.6');
/**
 * Version "9.6.1".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_1 = PostgresEngineVersion.of('9.6.1', '9.6');
/**
 * Version "9.6.2".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_2 = PostgresEngineVersion.of('9.6.2', '9.6');
/**
 * Version "9.6.3".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_3 = PostgresEngineVersion.of('9.6.3', '9.6');
/**
 * Version "9.6.5".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_5 = PostgresEngineVersion.of('9.6.5', '9.6');
/**
 * Version "9.6.6".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_6 = PostgresEngineVersion.of('9.6.6', '9.6');
/**
 * Version "9.6.8".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_8 = PostgresEngineVersion.of('9.6.8', '9.6');
/**
 * Version "9.6.9".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_9 = PostgresEngineVersion.of('9.6.9', '9.6');
/**
 * Version "9.6.10".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_10 = PostgresEngineVersion.of('9.6.10', '9.6');
/**
 * Version "9.6.11".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_11 = PostgresEngineVersion.of('9.6.11', '9.6');
/**
 * Version "9.6.12".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_12 = PostgresEngineVersion.of('9.6.12', '9.6');
/**
 * Version "9.6.14".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_14 = PostgresEngineVersion.of('9.6.14', '9.6');
/**
 * Version "9.6.15".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_15 = PostgresEngineVersion.of('9.6.15', '9.6');
/**
 * Version "9.6.16".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_16 = PostgresEngineVersion.of('9.6.16', '9.6');
/**
 * Version "9.6.17".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_17 = PostgresEngineVersion.of('9.6.17', '9.6');
/**
 * Version "9.6.18".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_18 = PostgresEngineVersion.of('9.6.18', '9.6');
/**
 * Version "9.6.19".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_9_6_19 = PostgresEngineVersion.of('9.6.19', '9.6');
/**
 * Version "10" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
PostgresEngineVersion.VER_10 = PostgresEngineVersion.of('10', '10');
/**
 * Version "10.1".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_10_1 = PostgresEngineVersion.of('10.1', '10');
/**
 * Version "10.3".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_10_3 = PostgresEngineVersion.of('10.3', '10');
/**
 * Version "10.4".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_10_4 = PostgresEngineVersion.of('10.4', '10');
/**
 * Version "10.5".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_10_5 = PostgresEngineVersion.of('10.5', '10');
/**
 * Version "10.6".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_10_6 = PostgresEngineVersion.of('10.6', '10');
/**
 * Version "10.7".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_10_7 = PostgresEngineVersion.of('10.7', '10', { s3Import: true });
/**
 * Version "10.9".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_10_9 = PostgresEngineVersion.of('10.9', '10', { s3Import: true });
/**
 * Version "10.10".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_10_10 = PostgresEngineVersion.of('10.10', '10', { s3Import: true });
/**
 * Version "10.11".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_10_11 = PostgresEngineVersion.of('10.11', '10', { s3Import: true });
/**
 * Version "10.12".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_10_12 = PostgresEngineVersion.of('10.12', '10', { s3Import: true });
/**
 * Version "10.13".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_10_13 = PostgresEngineVersion.of('10.13', '10', { s3Import: true });
/**
 * Version "10.14".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_10_14 = PostgresEngineVersion.of('10.14', '10', { s3Import: true });
/**
 * Version "11" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
PostgresEngineVersion.VER_11 = PostgresEngineVersion.of('11', '11', { s3Import: true });
/**
 * Version "11.1".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_11_1 = PostgresEngineVersion.of('11.1', '11', { s3Import: true });
/**
 * Version "11.2".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_11_2 = PostgresEngineVersion.of('11.2', '11', { s3Import: true });
/**
 * Version "11.4".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_11_4 = PostgresEngineVersion.of('11.4', '11', { s3Import: true });
/**
 * Version "11.5".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_11_5 = PostgresEngineVersion.of('11.5', '11', { s3Import: true });
/**
 * Version "11.6".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_11_6 = PostgresEngineVersion.of('11.6', '11', { s3Import: true });
/**
 * Version "11.7".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_11_7 = PostgresEngineVersion.of('11.7', '11', { s3Import: true });
/**
 * Version "11.8".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_11_8 = PostgresEngineVersion.of('11.8', '11', { s3Import: true });
/**
 * Version "11.9".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_11_9 = PostgresEngineVersion.of('11.9', '11', { s3Import: true });
/**
 * Version "12" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
PostgresEngineVersion.VER_12 = PostgresEngineVersion.of('12', '12', { s3Import: true });
/**
 * Version "12.2".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_12_2 = PostgresEngineVersion.of('12.2', '12', { s3Import: true });
/**
 * Version "12.3".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_12_3 = PostgresEngineVersion.of('12.3', '12', { s3Import: true });
/**
 * Version "12.4".
 *
 * @stability stable
 */
PostgresEngineVersion.VER_12_4 = PostgresEngineVersion.of('12.4', '12', { s3Import: true });
/**
 * The instance engine for PostgreSQL.
 */
class PostgresInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'postgres',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.postgresFullVersion,
                    majorVersion: version.postgresMajorVersion,
                }
                : undefined,
            features: version ? version === null || version === void 0 ? void 0 : version._features : { s3Import: 's3Import' },
            engineFamily: 'POSTGRESQL',
        });
        this.defaultUsername = 'postgres';
    }
}
/**
 * (deprecated) The versions for the legacy Oracle instance engines (those returned by {@link DatabaseInstanceEngine.oracleSe} and {@link DatabaseInstanceEngine.oracleSe1}). Note: RDS will stop allowing creating new databases with this version in August 2020.
 *
 * @deprecated instances can no longer be created with these engine versions. See https://forums.aws.amazon.com/ann.jspa?annID=7341
 */
class OracleLegacyEngineVersion {
    constructor(oracleLegacyFullVersion, oracleLegacyMajorVersion) {
        this.oracleLegacyFullVersion = oracleLegacyFullVersion;
        this.oracleLegacyMajorVersion = oracleLegacyMajorVersion;
    }
    static of(oracleLegacyFullVersion, oracleLegacyMajorVersion) {
        return new OracleLegacyEngineVersion(oracleLegacyFullVersion, oracleLegacyMajorVersion);
    }
}
exports.OracleLegacyEngineVersion = OracleLegacyEngineVersion;
/**
 * (deprecated) Version "11.2" (only a major version, without a specific minor version).
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2 = OracleLegacyEngineVersion.of('11.2', '11.2');
/**
 * (deprecated) Version "11.2.0.2.v2".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_2_V2 = OracleLegacyEngineVersion.of('11.2.0.2.v2', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v1".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V1 = OracleLegacyEngineVersion.of('11.2.0.4.v1', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v3".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V3 = OracleLegacyEngineVersion.of('11.2.0.4.v3', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v4".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V4 = OracleLegacyEngineVersion.of('11.2.0.4.v4', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v5".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V5 = OracleLegacyEngineVersion.of('11.2.0.4.v5', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v6".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V6 = OracleLegacyEngineVersion.of('11.2.0.4.v6', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v7".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V7 = OracleLegacyEngineVersion.of('11.2.0.4.v7', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v8".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V8 = OracleLegacyEngineVersion.of('11.2.0.4.v8', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v9".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V9 = OracleLegacyEngineVersion.of('11.2.0.4.v9', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v10".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V10 = OracleLegacyEngineVersion.of('11.2.0.4.v10', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v11".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V11 = OracleLegacyEngineVersion.of('11.2.0.4.v11', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v12".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V12 = OracleLegacyEngineVersion.of('11.2.0.4.v12', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v13".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V13 = OracleLegacyEngineVersion.of('11.2.0.4.v13', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v14".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V14 = OracleLegacyEngineVersion.of('11.2.0.4.v14', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v15".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V15 = OracleLegacyEngineVersion.of('11.2.0.4.v15', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v16".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V16 = OracleLegacyEngineVersion.of('11.2.0.4.v16', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v17".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V17 = OracleLegacyEngineVersion.of('11.2.0.4.v17', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v18".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V18 = OracleLegacyEngineVersion.of('11.2.0.4.v18', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v19".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V19 = OracleLegacyEngineVersion.of('11.2.0.4.v19', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v20".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V20 = OracleLegacyEngineVersion.of('11.2.0.4.v20', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v21".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V21 = OracleLegacyEngineVersion.of('11.2.0.4.v21', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v22".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V22 = OracleLegacyEngineVersion.of('11.2.0.4.v22', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v23".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V23 = OracleLegacyEngineVersion.of('11.2.0.4.v23', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v24".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V24 = OracleLegacyEngineVersion.of('11.2.0.4.v24', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v25".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V25 = OracleLegacyEngineVersion.of('11.2.0.4.v25', '11.2');
/**
 * The versions for the Oracle instance engines (those returned by {@link DatabaseInstanceEngine.oracleSe2} and {@link DatabaseInstanceEngine.oracleEe}).
 *
 * @stability stable
 */
class OracleEngineVersion {
    constructor(oracleFullVersion, oracleMajorVersion) {
        this.oracleFullVersion = oracleFullVersion;
        this.oracleMajorVersion = oracleMajorVersion;
    }
    /**
     * Creates a new OracleEngineVersion with an arbitrary version.
     *
     * @param oracleFullVersion the full version string, for example "19.0.0.0.ru-2019-10.rur-2019-10.r1".
     * @param oracleMajorVersion the major version of the engine, for example "19".
     * @stability stable
     */
    static of(oracleFullVersion, oracleMajorVersion) {
        return new OracleEngineVersion(oracleFullVersion, oracleMajorVersion);
    }
}
exports.OracleEngineVersion = OracleEngineVersion;
/**
 * Version "12.1" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1 = OracleEngineVersion.of('12.1', '12.1');
/**
 * Version "12.1.0.2.v1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V1 = OracleEngineVersion.of('12.1.0.2.v1', '12.1');
/**
 * Version "12.1.0.2.v2".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V2 = OracleEngineVersion.of('12.1.0.2.v2', '12.1');
/**
 * Version "12.1.0.2.v3".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V3 = OracleEngineVersion.of('12.1.0.2.v3', '12.1');
/**
 * Version "12.1.0.2.v4".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V4 = OracleEngineVersion.of('12.1.0.2.v4', '12.1');
/**
 * Version "12.1.0.2.v5".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V5 = OracleEngineVersion.of('12.1.0.2.v5', '12.1');
/**
 * Version "12.1.0.2.v6".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V6 = OracleEngineVersion.of('12.1.0.2.v6', '12.1');
/**
 * Version "12.1.0.2.v7".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V7 = OracleEngineVersion.of('12.1.0.2.v7', '12.1');
/**
 * Version "12.1.0.2.v8".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V8 = OracleEngineVersion.of('12.1.0.2.v8', '12.1');
/**
 * Version "12.1.0.2.v9".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V9 = OracleEngineVersion.of('12.1.0.2.v9', '12.1');
/**
 * Version "12.1.0.2.v10".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V10 = OracleEngineVersion.of('12.1.0.2.v10', '12.1');
/**
 * Version "12.1.0.2.v11".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V11 = OracleEngineVersion.of('12.1.0.2.v11', '12.1');
/**
 * Version "12.1.0.2.v12".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V12 = OracleEngineVersion.of('12.1.0.2.v12', '12.1');
/**
 * Version "12.1.0.2.v13".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V13 = OracleEngineVersion.of('12.1.0.2.v13', '12.1');
/**
 * Version "12.1.0.2.v14".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V14 = OracleEngineVersion.of('12.1.0.2.v14', '12.1');
/**
 * Version "12.1.0.2.v15".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V15 = OracleEngineVersion.of('12.1.0.2.v15', '12.1');
/**
 * Version "12.1.0.2.v16".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V16 = OracleEngineVersion.of('12.1.0.2.v16', '12.1');
/**
 * Version "12.1.0.2.v17".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V17 = OracleEngineVersion.of('12.1.0.2.v17', '12.1');
/**
 * Version "12.1.0.2.v18".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V18 = OracleEngineVersion.of('12.1.0.2.v18', '12.1');
/**
 * Version "12.1.0.2.v19".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V19 = OracleEngineVersion.of('12.1.0.2.v19', '12.1');
/**
 * Version "12.1.0.2.v20".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V20 = OracleEngineVersion.of('12.1.0.2.v20', '12.1');
/**
 * Version "12.1.0.2.v21".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_1_0_2_V21 = OracleEngineVersion.of('12.1.0.2.v21', '12.1');
/**
 * Version "12.2" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_2 = OracleEngineVersion.of('12.2', '12.2');
/**
 * Version "12.2.0.1.ru-2018-10.rur-2018-10.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_2_0_1_2018_10_R1 = OracleEngineVersion.of('12.2.0.1.ru-2018-10.rur-2018-10.r1', '12.2');
/**
 * Version "12.2.0.1.ru-2019-01.rur-2019-01.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_2_0_1_2019_01_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-01.rur-2019-01.r1', '12.2');
/**
 * Version "12.2.0.1.ru-2019-04.rur-2019-04.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_2_0_1_2019_04_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-04.rur-2019-04.r1', '12.2');
/**
 * Version "12.2.0.1.ru-2019-07.rur-2019-07.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_2_0_1_2019_07_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-07.rur-2019-07.r1', '12.2');
/**
 * Version "12.2.0.1.ru-2019-10.rur-2019-10.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_2_0_1_2019_10_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-10.rur-2019-10.r1', '12.2');
/**
 * Version "12.2.0.1.ru-2020-01.rur-2020-01.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_2_0_1_2020_01_R1 = OracleEngineVersion.of('12.2.0.1.ru-2020-01.rur-2020-01.r1', '12.2');
/**
 * Version "12.2.0.1.ru-2020-04.rur-2020-04.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_2_0_1_2020_04_R1 = OracleEngineVersion.of('12.2.0.1.ru-2020-04.rur-2020-04.r1', '12.2');
/**
 * Version "12.2.0.1.ru-2020-07.rur-2020-07.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_12_2_0_1_2020_07_R1 = OracleEngineVersion.of('12.2.0.1.ru-2020-07.rur-2020-07.r1', '12.2');
/**
 * Version "18" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
OracleEngineVersion.VER_18 = OracleEngineVersion.of('18', '18');
/**
 * Version "18.0.0.0.ru-2019-07.rur-2019-07.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_18_0_0_0_2019_07_R1 = OracleEngineVersion.of('18.0.0.0.ru-2019-07.rur-2019-07.r1', '18');
/**
 * Version "18.0.0.0.ru-2019-10.rur-2019-10.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_18_0_0_0_2019_10_R1 = OracleEngineVersion.of('18.0.0.0.ru-2019-10.rur-2019-10.r1', '18');
/**
 * Version "18.0.0.0.ru-2020-01.rur-2020-01.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_18_0_0_0_2020_01_R1 = OracleEngineVersion.of('18.0.0.0.ru-2020-01.rur-2020-01.r1', '18');
/**
 * Version "18.0.0.0.ru-2020-04.rur-2020-04.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_18_0_0_0_2020_04_R1 = OracleEngineVersion.of('18.0.0.0.ru-2020-04.rur-2020-04.r1', '18');
/**
 * Version "18.0.0.0.ru-2020-07.rur-2020-07.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_18_0_0_0_2020_07_R1 = OracleEngineVersion.of('18.0.0.0.ru-2020-07.rur-2020-07.r1', '18');
/**
 * Version "19" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
OracleEngineVersion.VER_19 = OracleEngineVersion.of('19', '19');
/**
 * Version "19.0.0.0.ru-2019-07.rur-2019-07.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_19_0_0_0_2019_07_R1 = OracleEngineVersion.of('19.0.0.0.ru-2019-07.rur-2019-07.r1', '19');
/**
 * Version "19.0.0.0.ru-2019-10.rur-2019-10.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_19_0_0_0_2019_10_R1 = OracleEngineVersion.of('19.0.0.0.ru-2019-10.rur-2019-10.r1', '19');
/**
 * Version "19.0.0.0.ru-2020-01.rur-2020-01.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_19_0_0_0_2020_01_R1 = OracleEngineVersion.of('19.0.0.0.ru-2020-01.rur-2020-01.r1', '19');
/**
 * Version "19.0.0.0.ru-2020-04.rur-2020-04.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_19_0_0_0_2020_04_R1 = OracleEngineVersion.of('19.0.0.0.ru-2020-04.rur-2020-04.r1', '19');
/**
 * Version "19.0.0.0.ru-2020-07.rur-2020-07.r1".
 *
 * @stability stable
 */
OracleEngineVersion.VER_19_0_0_0_2020_07_R1 = OracleEngineVersion.of('19.0.0.0.ru-2020-07.rur-2020-07.r1', '19');
class OracleInstanceEngineBase extends InstanceEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER,
            parameterGroupFamily: props.version ? `${props.engineType}-${props.version.majorVersion}` : undefined,
            features: {
                s3Import: 'S3_INTEGRATION',
                s3Export: 'S3_INTEGRATION',
            },
        });
    }
    bindToInstance(scope, options) {
        const config = super.bindToInstance(scope, options);
        let optionGroup = options.optionGroup;
        if (options.s3ImportRole || options.s3ExportRole) {
            if (!optionGroup) {
                optionGroup = new option_group_1.OptionGroup(scope, 'InstanceOptionGroup', {
                    engine: this,
                    configurations: [],
                });
            }
            // https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html
            optionGroup.addConfiguration({
                name: 'S3_INTEGRATION',
                version: '1.0',
            });
        }
        return {
            ...config,
            optionGroup,
        };
    }
}
/** @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341 */
class OracleSeInstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se',
            version: version
                ? {
                    fullVersion: version.oracleLegacyFullVersion,
                    majorVersion: version.oracleLegacyMajorVersion,
                }
                : {
                    majorVersion: '11.2',
                },
        });
    }
}
/** @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341 */
class OracleSe1InstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se1',
            version: version
                ? {
                    fullVersion: version.oracleLegacyFullVersion,
                    majorVersion: version.oracleLegacyMajorVersion,
                }
                : {
                    majorVersion: '11.2',
                },
        });
    }
}
class OracleSe2InstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se2',
            version: version
                ? {
                    fullVersion: version.oracleFullVersion,
                    majorVersion: version.oracleMajorVersion,
                }
                : undefined,
        });
    }
}
class OracleEeInstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-ee',
            version: version
                ? {
                    fullVersion: version.oracleFullVersion,
                    majorVersion: version.oracleMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the SQL Server instance engines (those returned by {@link DatabaseInstanceEngine.sqlServerSe}, {@link DatabaseInstanceEngine.sqlServerEx}, {@link DatabaseInstanceEngine.sqlServerWeb} and {@link DatabaseInstanceEngine.sqlServerEe}).
 *
 * @stability stable
 */
class SqlServerEngineVersion {
    constructor(sqlServerFullVersion, sqlServerMajorVersion) {
        this.sqlServerFullVersion = sqlServerFullVersion;
        this.sqlServerMajorVersion = sqlServerMajorVersion;
    }
    /**
     * Create a new SqlServerEngineVersion with an arbitrary version.
     *
     * @param sqlServerFullVersion the full version string, for example "15.00.3049.1.v1".
     * @param sqlServerMajorVersion the major version of the engine, for example "15.00".
     * @stability stable
     */
    static of(sqlServerFullVersion, sqlServerMajorVersion) {
        return new SqlServerEngineVersion(sqlServerFullVersion, sqlServerMajorVersion);
    }
}
exports.SqlServerEngineVersion = SqlServerEngineVersion;
/**
 * Version "11.00" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_11 = SqlServerEngineVersion.of('11.00', '11.00');
/**
 * Version "11.00.5058.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_11_00_5058_0_V1 = SqlServerEngineVersion.of('11.00.5058.0.v1', '11.00');
/**
 * Version "11.00.6020.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_11_00_6020_0_V1 = SqlServerEngineVersion.of('11.00.6020.0.v1', '11.00');
/**
 * Version "11.00.6594.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_11_00_6594_0_V1 = SqlServerEngineVersion.of('11.00.6594.0.v1', '11.00');
/**
 * Version "11.00.7462.6.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_11_00_7462_6_V1 = SqlServerEngineVersion.of('11.00.7462.6.v1', '11.00');
/**
 * Version "11.00.7493.4.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_11_00_7493_4_V1 = SqlServerEngineVersion.of('11.00.7493.4.v1', '11.00');
/**
 * Version "12.00" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_12 = SqlServerEngineVersion.of('12.00', '12.00');
/**
 * Version "12.00.5000.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_12_00_5000_0_V1 = SqlServerEngineVersion.of('12.00.5000.0.v1', '12.00');
/**
 * Version "12.00.5546.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_12_00_5546_0_V1 = SqlServerEngineVersion.of('12.00.5546.0.v1', '12.00');
/**
 * Version "12.00.5571.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_12_00_5571_0_V1 = SqlServerEngineVersion.of('12.00.5571.0.v1', '12.00');
/**
 * Version "12.00.6293.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_12_00_6293_0_V1 = SqlServerEngineVersion.of('12.00.6293.0.v1', '12.00');
/**
 * Version "12.00.6329.1.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_12_00_6329_1_V1 = SqlServerEngineVersion.of('12.00.6329.1.v1', '12.00');
/**
 * Version "13.00" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_13 = SqlServerEngineVersion.of('13.00', '13.00');
/**
 * Version "13.00.2164.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_13_00_2164_0_V1 = SqlServerEngineVersion.of('13.00.2164.0.v1', '13.00');
/**
 * Version "13.00.4422.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_13_00_4422_0_V1 = SqlServerEngineVersion.of('13.00.4422.0.v1', '13.00');
/**
 * Version "13.00.4451.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_13_00_4451_0_V1 = SqlServerEngineVersion.of('13.00.4451.0.v1', '13.00');
/**
 * Version "13.00.4466.4.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_13_00_4466_4_V1 = SqlServerEngineVersion.of('13.00.4466.4.v1', '13.00');
/**
 * Version "13.00.4522.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_13_00_4522_0_V1 = SqlServerEngineVersion.of('13.00.4522.0.v1', '13.00');
/**
 * Version "13.00.5216.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_13_00_5216_0_V1 = SqlServerEngineVersion.of('13.00.5216.0.v1', '13.00');
/**
 * Version "13.00.5292.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_13_00_5292_0_V1 = SqlServerEngineVersion.of('13.00.5292.0.v1', '13.00');
/**
 * Version "13.00.5366.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_13_00_5366_0_V1 = SqlServerEngineVersion.of('13.00.5366.0.v1', '13.00');
/**
 * Version "13.00.5426.0.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_13_00_5426_0_V1 = SqlServerEngineVersion.of('13.00.5426.0.v1', '13.00');
/**
 * Version "13.00.5598.27.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_13_00_5598_27_V1 = SqlServerEngineVersion.of('13.00.5598.27.v1', '13.00');
/**
 * Version "13.00.5820.21.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_13_00_5820_21_V1 = SqlServerEngineVersion.of('13.00.5820.21.v1', '13.00');
/**
 * Version "14.00" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_14 = SqlServerEngineVersion.of('14.00', '14.00');
/**
 * Version "14.00.1000.169.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_14_00_1000_169_V1 = SqlServerEngineVersion.of('14.00.1000.169.v1', '14.00');
/**
 * Version "14.00.3015.40.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_14_00_3015_40_V1 = SqlServerEngineVersion.of('14.00.3015.40.v1', '14.00');
/**
 * Version "14.00.3035.2.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_14_00_3035_2_V1 = SqlServerEngineVersion.of('14.00.3035.2.v1', '14.00');
/**
 * Version "14.00.3049.1.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_14_00_3049_1_V1 = SqlServerEngineVersion.of('14.00.3049.1.v1', '14.00');
/**
 * Version "14.00.3192.2.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_14_00_3192_2_V1 = SqlServerEngineVersion.of('14.00.3192.2.v1', '14.00');
/**
 * Version "14.00.3223.3.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_14_00_3223_3_V1 = SqlServerEngineVersion.of('14.00.3223.3.v1', '14.00');
/**
 * Version "14.00.3281.6.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_14_00_3281_6_V1 = SqlServerEngineVersion.of('14.00.3281.6.v1', '14.00');
/**
 * Version "14.00.3294.2.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_14_00_3294_2_V1 = SqlServerEngineVersion.of('14.00.3294.2.v1', '14.00');
/**
 * Version "15.00" (only a major version, without a specific minor version).
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_15 = SqlServerEngineVersion.of('15.00', '15.00');
/**
 * Version "15.00.4043.16.v1".
 *
 * @stability stable
 */
SqlServerEngineVersion.VER_15_00_4043_16_V1 = SqlServerEngineVersion.of('15.00.4043.16.v1', '15.00');
class SqlServerInstanceEngineBase extends InstanceEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER,
            version: props.version
                ? {
                    fullVersion: props.version.sqlServerFullVersion,
                    majorVersion: props.version.sqlServerMajorVersion,
                }
                : undefined,
            parameterGroupFamily: props.version
                // for some reason, even though SQL Server major versions usually end in '.00',
                // the ParameterGroup family has to end in '.0'
                ? `${props.engineType}-${props.version.sqlServerMajorVersion.endsWith('.00')
                    ? props.version.sqlServerMajorVersion.slice(0, -1)
                    : props.version.sqlServerMajorVersion}`
                : undefined,
            features: {
                s3Import: 'S3_INTEGRATION',
                s3Export: 'S3_INTEGRATION',
            },
        });
    }
    bindToInstance(scope, options) {
        var _a;
        const config = super.bindToInstance(scope, options);
        let optionGroup = options.optionGroup;
        const s3Role = (_a = options.s3ImportRole) !== null && _a !== void 0 ? _a : options.s3ExportRole;
        if (s3Role) {
            if (options.s3ImportRole && options.s3ExportRole && options.s3ImportRole !== options.s3ExportRole) {
                throw new Error('S3 import and export roles must be the same for SQL Server engines');
            }
            if (!optionGroup) {
                optionGroup = new option_group_1.OptionGroup(scope, 'InstanceOptionGroup', {
                    engine: this,
                    configurations: [],
                });
            }
            // https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.SQLServer.Options.BackupRestore.html
            optionGroup.addConfiguration({
                name: 'SQLSERVER_BACKUP_RESTORE',
                settings: { IAM_ROLE_ARN: s3Role.roleArn },
            });
        }
        return {
            ...config,
            optionGroup,
        };
    }
    get supportsTimezone() { return true; }
}
class SqlServerSeInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-se',
            version,
        });
    }
}
class SqlServerExInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-ex',
            version,
        });
    }
}
class SqlServerWebInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-web',
            version,
        });
    }
}
class SqlServerEeInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-ee',
            version,
        });
    }
}
/**
 * A database instance engine.
 *
 * Provides mapping to DatabaseEngine used for
 * secret rotation.
 *
 * @stability stable
 */
class DatabaseInstanceEngine {
    /**
     * Creates a new MariaDB instance engine.
     *
     * @stability stable
     */
    static mariaDb(props) {
        return new MariaDbInstanceEngine(props.version);
    }
    /**
     * Creates a new MySQL instance engine.
     *
     * @stability stable
     */
    static mysql(props) {
        return new MySqlInstanceEngine(props.version);
    }
    /**
     * Creates a new PostgreSQL instance engine.
     *
     * @stability stable
     */
    static postgres(props) {
        return new PostgresInstanceEngine(props.version);
    }
    /**
     * (deprecated) Creates a new Oracle Standard Edition instance engine.
     *
     * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
     */
    static oracleSe(props) {
        return new OracleSeInstanceEngine(props.version);
    }
    /**
     * (deprecated) Creates a new Oracle Standard Edition 1 instance engine.
     *
     * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
     */
    static oracleSe1(props) {
        return new OracleSe1InstanceEngine(props.version);
    }
    /**
     * Creates a new Oracle Standard Edition 1 instance engine.
     *
     * @stability stable
     */
    static oracleSe2(props) {
        return new OracleSe2InstanceEngine(props.version);
    }
    /**
     * Creates a new Oracle Enterprise Edition instance engine.
     *
     * @stability stable
     */
    static oracleEe(props) {
        return new OracleEeInstanceEngine(props.version);
    }
    /**
     * Creates a new SQL Server Standard Edition instance engine.
     *
     * @stability stable
     */
    static sqlServerSe(props) {
        return new SqlServerSeInstanceEngine(props.version);
    }
    /**
     * Creates a new SQL Server Express Edition instance engine.
     *
     * @stability stable
     */
    static sqlServerEx(props) {
        return new SqlServerExInstanceEngine(props.version);
    }
    /**
     * Creates a new SQL Server Web Edition instance engine.
     *
     * @stability stable
     */
    static sqlServerWeb(props) {
        return new SqlServerWebInstanceEngine(props.version);
    }
    /**
     * Creates a new SQL Server Enterprise Edition instance engine.
     *
     * @stability stable
     */
    static sqlServerEe(props) {
        return new SqlServerEeInstanceEngine(props.version);
    }
}
exports.DatabaseInstanceEngine = DatabaseInstanceEngine;
/**
 * (deprecated) The unversioned 'mariadb' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link mariaDb()} method
 */
DatabaseInstanceEngine.MARIADB = new MariaDbInstanceEngine();
/**
 * (deprecated) The unversioned 'mysql' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link mysql()} method
 */
DatabaseInstanceEngine.MYSQL = new MySqlInstanceEngine();
/**
 * (deprecated) The unversioned 'oracle-ee' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link oracleEe()} method
 */
DatabaseInstanceEngine.ORACLE_EE = new OracleEeInstanceEngine();
/**
 * (deprecated) The unversioned 'oracle-se2' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link oracleSe2()} method
 */
DatabaseInstanceEngine.ORACLE_SE2 = new OracleSe2InstanceEngine();
/**
 * (deprecated) The unversioned 'oracle-se1' instance engine.
 *
 * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
 */
DatabaseInstanceEngine.ORACLE_SE1 = new OracleSe1InstanceEngine();
/**
 * (deprecated) The unversioned 'oracle-se' instance engine.
 *
 * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
 */
DatabaseInstanceEngine.ORACLE_SE = new OracleSeInstanceEngine();
/**
 * (deprecated) The unversioned 'postgres' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link postgres()} method
 */
DatabaseInstanceEngine.POSTGRES = new PostgresInstanceEngine();
/**
 * (deprecated) The unversioned 'sqlserver-ee' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link sqlServerEe()} method
 */
DatabaseInstanceEngine.SQL_SERVER_EE = new SqlServerEeInstanceEngine();
/**
 * (deprecated) The unversioned 'sqlserver-se' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link sqlServerSe()} method
 */
DatabaseInstanceEngine.SQL_SERVER_SE = new SqlServerSeInstanceEngine();
/**
 * (deprecated) The unversioned 'sqlserver-ex' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link sqlServerEx()} method
 */
DatabaseInstanceEngine.SQL_SERVER_EX = new SqlServerExInstanceEngine();
/**
 * (deprecated) The unversioned 'sqlserver-web' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link sqlServerWeb()} method
 */
DatabaseInstanceEngine.SQL_SERVER_WEB = new SqlServerWebInstanceEngine();
//# sourceMappingURL=data:application/json;base64,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