# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['invoke_databricks_wheel_tasks', 'invoke_databricks_wheel_tasks.utils']

package_data = \
{'': ['*']}

install_requires = \
['Jinja2>=3.0',
 'databricks-cli>=0.16.4,<0.17.0',
 'invoke>=1.7.0,<2.0.0',
 'poetry-core>=1.0.8,<2.0.0']

setup_kwargs = {
    'name': 'invoke-databricks-wheel-tasks',
    'version': '0.7.1',
    'description': 'Databricks Python Wheel dev tasks in a namespaced collection of tasks to enrich the Invoke CLI task runner.',
    'long_description': '# Invoke Databricks Wheel Tasks\n\nDatabricks Python Wheel dev tasks in a namespaced collection of tasks to enrich the Invoke CLI task runner.\n\n## Getting Started\n\n```sh\npip install invoke-databricks-wheel-tasks\n```\n\nThis will also install `invoke` and `databricks-cli`.\n\n### Databricks CLI Config\n\nIt is assumed you will follow the documentation provided to setup `databricks-cli`.\n\nhttps://docs.databricks.com/dev-tools/cli/index.html\n\nYou\'ll need to setup a Personal Access Token. Then run the following command:\n\n```sh\ndatabricks configure --profile yourprofilename --token\n\nDatabricks Host (should begin with https://): https://myorganisation.cloud.databricks.com/\nToken: \n```\n\nWhich will create a configuration file in your home directory at `~/.databrickscfg` like:\n\n```sh\ncat ~/.databrickscfg\n\n[yourprofilename]\nhost = https://myorganisation.cloud.databricks.com/\ntoken = dapi0123456789abcdef0123456789abcdef\njobs-api-version = 2.1\n```\n\n### Invoke Setup\n\n`tasks.py`\n\n```python\nfrom invoke import task\nfrom invoke_databricks_wheel_tasks import * # noqa\n\n@task\ndef format(c):\n    """Autoformat code for code style."""\n    c.run("black .")\n    c.run("isort .")\n\n@task\ndef build(c):\n    """Build wheel."""\n    c.run("rm -rfv dist/")\n    c.run("poetry build -f wheel")\n```\n\nOnce your `tasks.py` is setup like this `invoke` will have the extra commands:\n\n```sh\nλ invoke --list\nAvailable tasks:\n\n  build                  Build wheel.\n  clean                  Clean wheel artifact from DBFS.\n  dbfs-wheel-path        Generate the target path (including wheelname) this wheel should be uploaded to.\n  dbfs-wheel-path-root   Generate the target path (excluding wheelname) this wheel should be uploaded to.\n  define-job             Generate templated Job definition and upsert by Job Name in template.\n  format                 Autoformat code for code style.\n  poetry-wheel-name      Display the name of the wheel file poetry would build.\n  reinstall              Reinstall version of wheel on cluster with a restart.\n  runjob                 Trigger default job associated for this project.\n  upload                 Upload wheel artifact to DBFS.\n```\n\n## The Tasks\n\n### upload\n\nThis task will use `dbfs` to empty the upload path and then copy the built wheel from `dist/`.\nThis project assumes you\'re using `poetry` or your wheel build output is located in `dist/`.\n\nIf you have other requirements then _pull requests welcome_.\n\n### clean\n\nThis tasks will clean up all items on the target `--artifact-path`.\n\n### reinstall\n\nAfter some trial and error, creating a job which creates a job cluster everytime is roughly 7 minutes.\n\nHowever if you create an all purpose cluster that you:\n - Mark the old wheel for uninstall\n - restart cluster\n - install updated wheel from dbfs location\n \n This takes roughly 2 minutes which is a much tighter development loop. So these three steps are what `db.reinstall` performs.\n\n### runjob\n\nAssuming you have defined a job, that uses a pre-existing cluster, that has your latest wheel installed, this will create a manual trigger of your job with `job-id`.\n\nThe triggering returns a `run-id`, where this `run-id` gets polled until the state gets to an end state.\n\nThen a call to `databricks runs get-output --run-id` happens to retrieve and `error`, `error_trace` and/or `logs` to be emitted to console.\n\n### define-job\n\nYou may want to run `invoke --help define-job` to get the help documentation.\n\nThere are a few arguments that get abbreviated which we will explain before discussing how they work together.\n - `--jinja-template` or `-j`: This is a [Jinja2 Template](https://jinja.palletsprojects.com/en/3.1.x/) that must resolve to a valid [Databricks Jobs JSON payload](https://docs.databricks.com/dev-tools/api/latest/jobs.html#) spec.\n - `--config-file` or `-c`: This is either a JSON or YAML file to define the config, that gets used to parametrise the above `jinja-template`. This `config-file` can also be a Jinja template to inject values that can only be known at runtime like the git feature branch you are currently on. By default it is treated as a plain config file and not a Jinja Template unless `environment-variable` flags are specified (_see next_).\n - `--environment-variable` or `-e`: This flag can be repeated many times to specify multiple values. It takes a string in the `key=value` format. \n    - Eg `-e branch=$(git branch --show-current) -e main_wheel=$MAIN -e utils_wheel=$UTILS`\n\nSo an example command could look like:\n```sh\ninvoke define-job \\\n    -j jobs/base-job-template.json.j2 \\\n    -c jobs/customer360-etl-job.yaml \\\n    -e branch=$(git branch --show-current) \\\n    -e main_whl=$(invoke dbfs-wheel-path) \\\n    -e utils_whl=$UTILS_DBFS_WHEEL_PATH\n```\n\nThen the `-e` values can get templated into `customer360-etl-job.yml`. Then that YAML file gets parsed and injected into `base-job-template.json.j2`\n\nThis will then check the list of Jobs in your workspace, see if a job with the same name exists already and perform a _create or replace job_ operation. This expects the `config-file` to have a key `name` to be able to cross check the list of existing jobs.\n\n**The beauty is that the specifics of `config-file` and `jinja-template` is completely up to you.**\n\n`config-file` is the minimal datastructure you need to configure `jinja-template` and you just use the [Jinja Control Structures](https://jinja.palletsprojects.com/en/3.1.x/templates/#list-of-control-structures) (`if-else`, `for-loop`, etc) to traverse it and populate `jinja-template`.\n\n\n# Contributing\n\nAt all times, you have the power to fork this project, make changes as you see fit and then:\n\n```sh\npip install https://github.com/user/repository/archive/branch.zip\n```\n[Stackoverflow: pip install from github branch](https://stackoverflow.com/a/24811490/622276)\n\nThat way you can run from your own custom fork in the interim or even in-house your work and simply use this project as a starting point. That is totally ok.\n\nHowever if you would like to contribute your changes back, then open a Pull Request "across forks".\n\nOnce your changes are merged and published you can revert to the canonical version of `pip install`ing this package.\n\nIf you\'re not sure how to make changes or if you should sink the time and effort, then open an Issue instead and we can have a chat to triage the issue.\n\n\n# Resources\n\n - [`pyinvoke`](https://pyinvoke.org)\n - [`databricks-cli`](https://docs.databricks.com/dev-tools/cli/index.html)\n\n# Prior Art\n\n - https://github.com/Smile-SA/invoke-sphinx\n - https://github.com/Dashlane/dbt-invoke\n\n',
    'author': 'Josh Peak',
    'author_email': 'neozenith.dev@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/neozenith/invoke-databricks-wheel-tasks',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
