"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClientAuthentication = exports.ClientBrokerEncryption = exports.ClusterMonitoringLevel = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const core = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cr = require("aws-cdk-lib/custom-resources");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const uniqueid_1 = require("constructs/lib/private/uniqueid");
const aws_msk_1 = require("aws-cdk-lib/aws-msk");
/**
 * A new or imported MSK Cluster.
 */
class ClusterBase extends core.Resource {
    /** Manages connections for the cluster */
    get connections() {
        if (!this._connections) {
            throw new Error('An imported Cluster cannot manage its security groups');
        }
        return this._connections;
    }
}
/**
 * The level of monitoring for the MSK cluster
 *
 * @see https://docs.aws.amazon.com/msk/latest/developerguide/monitoring.html#metrics-details
 */
var ClusterMonitoringLevel;
(function (ClusterMonitoringLevel) {
    /**
     * Default metrics are the essential metrics to monitor.
     */
    ClusterMonitoringLevel["DEFAULT"] = "DEFAULT";
    /**
     * Per Broker metrics give you metrics at the broker level.
     */
    ClusterMonitoringLevel["PER_BROKER"] = "PER_BROKER";
    /**
     * Per Topic Per Broker metrics help you understand volume at the topic level.
     */
    ClusterMonitoringLevel["PER_TOPIC_PER_BROKER"] = "PER_TOPIC_PER_BROKER";
    /**
     * Per Topic Per Partition metrics help you understand consumer group lag at the topic partition level.
     */
    ClusterMonitoringLevel["PER_TOPIC_PER_PARTITION"] = "PER_TOPIC_PER_PARTITION";
})(ClusterMonitoringLevel = exports.ClusterMonitoringLevel || (exports.ClusterMonitoringLevel = {}));
/**
 * Indicates the encryption setting for data in transit between clients and brokers.
 */
var ClientBrokerEncryption;
(function (ClientBrokerEncryption) {
    /**
     * TLS means that client-broker communication is enabled with TLS only.
     */
    ClientBrokerEncryption["TLS"] = "TLS";
    /**
     * TLS_PLAINTEXT means that client-broker communication is enabled for both TLS-encrypted, as well as plaintext data.
     */
    ClientBrokerEncryption["TLS_PLAINTEXT"] = "TLS_PLAINTEXT";
    /**
     * PLAINTEXT means that client-broker communication is enabled in plaintext only.
     */
    ClientBrokerEncryption["PLAINTEXT"] = "PLAINTEXT";
})(ClientBrokerEncryption = exports.ClientBrokerEncryption || (exports.ClientBrokerEncryption = {}));
/**
 * Configuration properties for client authentication.
 */
class ClientAuthentication {
    /**
     * @param saslProps - properties for SASL authentication
     * @param tlsProps - properties for TLS authentication
     */
    constructor(saslProps, tlsProps) {
        this.saslProps = saslProps;
        this.tlsProps = tlsProps;
    }
    /**
     * SASL authentication
     */
    static sasl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_SaslAuthProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.sasl);
            }
            throw error;
        }
        return new ClientAuthentication(props, undefined);
    }
    /**
     * TLS authentication
     */
    static tls(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_TlsAuthProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.tls);
            }
            throw error;
        }
        return new ClientAuthentication(undefined, props);
    }
    /**
     * SASL + TLS authentication
     */
    static saslTls(saslTlsProps) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_SaslTlsAuthProps(saslTlsProps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.saslTls);
            }
            throw error;
        }
        return new ClientAuthentication(saslTlsProps, saslTlsProps);
    }
}
exports.ClientAuthentication = ClientAuthentication;
_a = JSII_RTTI_SYMBOL_1;
ClientAuthentication[_a] = { fqn: "@aws-cdk/aws-msk-alpha.ClientAuthentication", version: "2.49.1-alpha.0" };
/**
 * Create a MSK Cluster.
 *
 * @resource AWS::MSK::Cluster
 */
class Cluster extends ClusterBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_ClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Cluster);
            }
            throw error;
        }
        const subnetSelection = props.vpc.selectSubnets(props.vpcSubnets);
        this._connections = new ec2.Connections({
            securityGroups: props.securityGroups ?? [
                new ec2.SecurityGroup(this, 'SecurityGroup', {
                    description: 'MSK security group',
                    vpc: props.vpc,
                }),
            ],
        });
        if (subnetSelection.subnets.length < 2) {
            throw Error(`Cluster requires at least 2 subnets, got ${subnetSelection.subnets.length}`);
        }
        if (!core.Token.isUnresolved(props.clusterName) &&
            !/^[a-zA-Z0-9]+$/.test(props.clusterName) &&
            props.clusterName.length > 64) {
            throw Error('The cluster name must only contain alphanumeric characters and have a maximum length of 64 characters.' +
                `got: '${props.clusterName}. length: ${props.clusterName.length}'`);
        }
        if (props.clientAuthentication?.saslProps?.iam &&
            props.clientAuthentication?.saslProps?.scram) {
            throw Error('Only one client authentication method can be enabled.');
        }
        if (props.encryptionInTransit?.clientBroker ===
            ClientBrokerEncryption.PLAINTEXT &&
            props.clientAuthentication) {
            throw Error('To enable client authentication, you must enabled TLS-encrypted traffic between clients and brokers.');
        }
        else if (props.encryptionInTransit?.clientBroker ===
            ClientBrokerEncryption.TLS_PLAINTEXT &&
            (props.clientAuthentication?.saslProps?.scram ||
                props.clientAuthentication?.saslProps?.iam)) {
            throw Error('To enable SASL/SCRAM or IAM authentication, you must only allow TLS-encrypted traffic between clients and brokers.');
        }
        const volumeSize = props.ebsStorageInfo?.volumeSize ?? 1000;
        // Minimum: 1 GiB, maximum: 16384 GiB
        if (volumeSize < 1 || volumeSize > 16384) {
            throw Error('EBS volume size should be in the range 1-16384');
        }
        const instanceType = props.instanceType
            ? this.mskInstanceType(props.instanceType)
            : this.mskInstanceType(ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE));
        const encryptionAtRest = props.ebsStorageInfo?.encryptionKey
            ? {
                dataVolumeKmsKeyId: props.ebsStorageInfo.encryptionKey.keyId,
            }
            : undefined; // MSK will create the managed key
        const encryptionInTransit = {
            clientBroker: props.encryptionInTransit?.clientBroker ??
                ClientBrokerEncryption.TLS,
            inCluster: props.encryptionInTransit?.enableInCluster ?? true,
        };
        const openMonitoring = props.monitoring?.enablePrometheusJmxExporter ||
            props.monitoring?.enablePrometheusNodeExporter
            ? {
                prometheus: {
                    jmxExporter: props.monitoring?.enablePrometheusJmxExporter
                        ? { enabledInBroker: true }
                        : undefined,
                    nodeExporter: props.monitoring
                        ?.enablePrometheusNodeExporter
                        ? { enabledInBroker: true }
                        : undefined,
                },
            }
            : undefined;
        const loggingBucket = props.logging?.s3?.bucket;
        if (loggingBucket && aws_cdk_lib_1.FeatureFlags.of(this).isEnabled(cx_api_1.S3_CREATE_DEFAULT_LOGGING_POLICY)) {
            const stack = core.Stack.of(this);
            loggingBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                principals: [
                    new iam.ServicePrincipal('delivery.logs.amazonaws.com'),
                ],
                resources: [
                    loggingBucket.arnForObjects(`AWSLogs/${stack.account}/*`),
                ],
                actions: ['s3:PutObject'],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                        'aws:SourceAccount': stack.account,
                    },
                    ArnLike: {
                        'aws:SourceArn': stack.formatArn({
                            service: 'logs',
                            resource: '*',
                        }),
                    },
                },
            }));
            loggingBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                principals: [
                    new iam.ServicePrincipal('delivery.logs.amazonaws.com'),
                ],
                resources: [loggingBucket.bucketArn],
                actions: [
                    's3:GetBucketAcl',
                    's3:ListBucket',
                ],
                conditions: {
                    StringEquals: {
                        'aws:SourceAccount': stack.account,
                    },
                    ArnLike: {
                        'aws:SourceArn': stack.formatArn({
                            service: 'logs',
                            resource: '*',
                        }),
                    },
                },
            }));
        }
        const loggingInfo = {
            brokerLogs: {
                cloudWatchLogs: {
                    enabled: props.logging?.cloudwatchLogGroup !== undefined,
                    logGroup: props.logging?.cloudwatchLogGroup?.logGroupName,
                },
                firehose: {
                    enabled: props.logging?.firehoseDeliveryStreamName !==
                        undefined,
                    deliveryStream: props.logging?.firehoseDeliveryStreamName,
                },
                s3: {
                    enabled: loggingBucket !== undefined,
                    bucket: loggingBucket?.bucketName,
                    prefix: props.logging?.s3?.prefix,
                },
            },
        };
        if (props.clientAuthentication?.saslProps?.scram &&
            props.clientAuthentication?.saslProps?.key === undefined) {
            this.saslScramAuthenticationKey = new kms.Key(this, 'SASLKey', {
                description: 'Used for encrypting MSK secrets for SASL/SCRAM authentication.',
                alias: `msk/${props.clusterName}/sasl/scram`,
            });
            // https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-policies
            this.saslScramAuthenticationKey.addToResourcePolicy(new iam.PolicyStatement({
                sid: 'Allow access through AWS Secrets Manager for all principals in the account that are authorized to use AWS Secrets Manager',
                principals: [new iam.AnyPrincipal()],
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:CreateGrant',
                    'kms:DescribeKey',
                ],
                resources: ['*'],
                conditions: {
                    StringEquals: {
                        'kms:ViaService': `secretsmanager.${core.Stack.of(this).region}.amazonaws.com`,
                        'kms:CallerAccount': core.Stack.of(this).account,
                    },
                },
            }));
        }
        let clientAuthentication;
        if (props.clientAuthentication?.saslProps?.iam) {
            clientAuthentication = {
                sasl: { iam: { enabled: props.clientAuthentication.saslProps.iam } },
            };
            if (props.clientAuthentication?.tlsProps) {
                clientAuthentication = {
                    sasl: { iam: { enabled: props.clientAuthentication.saslProps.iam } },
                    tls: {
                        certificateAuthorityArnList: props.clientAuthentication?.tlsProps?.certificateAuthorities?.map((ca) => ca.certificateAuthorityArn),
                    },
                };
            }
        }
        else if (props.clientAuthentication?.saslProps?.scram) {
            clientAuthentication = {
                sasl: {
                    scram: {
                        enabled: props.clientAuthentication.saslProps.scram,
                    },
                },
            };
        }
        else if (props.clientAuthentication?.tlsProps?.certificateAuthorities !== undefined) {
            clientAuthentication = {
                tls: {
                    certificateAuthorityArnList: props.clientAuthentication?.tlsProps?.certificateAuthorities.map((ca) => ca.certificateAuthorityArn),
                },
            };
        }
        const resource = new aws_msk_1.CfnCluster(this, 'Resource', {
            clusterName: props.clusterName,
            kafkaVersion: props.kafkaVersion.version,
            numberOfBrokerNodes: props.numberOfBrokerNodes !== undefined ?
                subnetSelection.availabilityZones.length * props.numberOfBrokerNodes : subnetSelection.availabilityZones.length,
            brokerNodeGroupInfo: {
                instanceType,
                clientSubnets: subnetSelection.subnetIds,
                securityGroups: this.connections.securityGroups.map((group) => group.securityGroupId),
                storageInfo: {
                    ebsStorageInfo: {
                        volumeSize: volumeSize,
                    },
                },
            },
            encryptionInfo: {
                encryptionAtRest,
                encryptionInTransit,
            },
            configurationInfo: props.configurationInfo,
            enhancedMonitoring: props.monitoring?.clusterMonitoringLevel,
            openMonitoring: openMonitoring,
            loggingInfo: loggingInfo,
            clientAuthentication: clientAuthentication,
        });
        this.clusterName = this.getResourceNameAttribute(core.Fn.select(1, core.Fn.split('/', resource.ref)));
        this.clusterArn = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.RETAIN,
        });
    }
    /**
     * Reference an existing cluster, defined outside of the CDK code, by name.
     */
    static fromClusterArn(scope, id, clusterArn) {
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.clusterArn = clusterArn;
                this.clusterName = core.Fn.select(1, core.Fn.split('/', clusterArn)); // ['arn:partition:kafka:region:account-id', clusterName, clusterId]
            }
        }
        return new Import(scope, id);
    }
    mskInstanceType(instanceType) {
        return `kafka.${instanceType.toString()}`;
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. ZookeeperConnectString, ZookeeperConnectStringTls
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    _zookeeperConnectionString(responseField) {
        if (!this._clusterDescription) {
            this._clusterDescription = new cr.AwsCustomResource(this, 'ZookeeperConnect', {
                onUpdate: {
                    service: 'Kafka',
                    action: 'describeCluster',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('ZooKeeperConnectionString'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
            });
        }
        return this._clusterDescription.getResponseField(`ClusterInfo.${responseField}`);
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    get zookeeperConnectionString() {
        return this._zookeeperConnectionString('ZookeeperConnectString');
    }
    /**
     * Get the ZooKeeper Connection string for a TLS enabled cluster
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to zookeeper cluster on TLS port.
     */
    get zookeeperConnectionStringTls() {
        return this._zookeeperConnectionString('ZookeeperConnectStringTls');
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. BootstrapBrokerStringSaslScram, BootstrapBrokerString
     * @returns - A string containing one or more hostname:port pairs.
     */
    _bootstrapBrokers(responseField) {
        if (!this._clusterBootstrapBrokers) {
            this._clusterBootstrapBrokers = new cr.AwsCustomResource(this, `BootstrapBrokers${responseField}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'getBootstrapBrokers',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('BootstrapBrokers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
            });
        }
        return this._clusterBootstrapBrokers.getResponseField(responseField);
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more hostname:port pairs.
     */
    get bootstrapBrokers() {
        return this._bootstrapBrokers('BootstrapBrokerString');
    }
    /**
     * Get the list of brokers that a TLS authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more DNS names (or IP) and TLS port pairs.
     */
    get bootstrapBrokersTls() {
        return this._bootstrapBrokers('BootstrapBrokerStringTls');
    }
    /**
     * Get the list of brokers that a SASL/SCRAM authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more dns name (or IP) and SASL SCRAM port pairs.
     */
    get bootstrapBrokersSaslScram() {
        return this._bootstrapBrokers('BootstrapBrokerStringSaslScram');
    }
    /**
     * Get the list of brokers that a SASL/IAM authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more DNS names (or IP) and TLS port pairs.
     */
    get bootstrapBrokersSaslIam() {
        return this._bootstrapBrokers('BootstrapBrokerStringSaslIam');
    }
    /**
     * A list of usersnames to register with the cluster. The password will automatically be generated using Secrets
     * Manager and the { username, password } JSON object stored in Secrets Manager as `AmazonMSK_username`.
     *
     * Must be using the SASL/SCRAM authentication mechanism.
     *
     * @param usernames - username(s) to register with the cluster
     */
    addUser(...usernames) {
        if (this.saslScramAuthenticationKey) {
            const MSK_SECRET_PREFIX = 'AmazonMSK_'; // Required
            const secrets = usernames.map((username) => new secretsmanager.Secret(this, `KafkaUser${username}`, {
                secretName: `${MSK_SECRET_PREFIX}${this.clusterName}_${username}`,
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({ username }),
                    generateStringKey: 'password',
                },
                encryptionKey: this.saslScramAuthenticationKey,
            }));
            new cr.AwsCustomResource(this, `BatchAssociateScramSecrets${uniqueid_1.addressOf(usernames)}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'batchAssociateScramSecret',
                    parameters: {
                        ClusterArn: this.clusterArn,
                        SecretArnList: secrets.map((secret) => secret.secretArn),
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('CreateUsers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['kms:CreateGrant'],
                        resources: [this.saslScramAuthenticationKey?.keyArn],
                    }),
                    new iam.PolicyStatement({
                        actions: ['kafka:BatchAssociateScramSecret'],
                        resources: [this.clusterArn],
                    }),
                ]),
            });
        }
        else {
            throw Error('Cannot create users if an authentication KMS key has not been created/provided.');
        }
    }
}
exports.Cluster = Cluster;
_b = JSII_RTTI_SYMBOL_1;
Cluster[_b] = { fqn: "@aws-cdk/aws-msk-alpha.Cluster", version: "2.49.1-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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