"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_sns_1 = require("monocdk/aws-sns");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * SNS topics require SSL requests
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_sns_1.CfnTopic) {
        const topicKey = monocdk_1.Stack.of(node).resolve(node.kmsMasterKeyId);
        if (topicKey === undefined) {
            const topicLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
            const topicName = monocdk_1.Stack.of(node).resolve(node.topicName);
            let found = false;
            for (const child of monocdk_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_sns_1.CfnTopicPolicy) {
                    if (isMatchingCompliantPolicy(child, topicLogicalId, topicName)) {
                        found = true;
                        break;
                    }
                }
            }
            if (!found) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the topic Policy requires SSL on the given topic.
 * @param node The CfnTopicPolicy to check.
 * @param topicLogicalId The Cfn Logical ID of the topic.
 * @param topicName The name of the topic.
 * @returns Whether the CfnTopicPolicy requires SSL on the given topic.
 */
function isMatchingCompliantPolicy(node, topicLogicalId, topicName) {
    var _a, _b;
    let found = false;
    for (const topic of node.topics) {
        const resolvedTopic = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, topic);
        if (resolvedTopic === topicLogicalId ||
            (topicName !== undefined && resolvedTopic.endsWith(topicName))) {
            found = true;
            break;
        }
    }
    if (!found) {
        return false;
    }
    const resolvedPolicyDocument = monocdk_1.Stack.of(node).resolve(node.policyDocument);
    for (const statement of resolvedPolicyDocument.Statement) {
        const resolvedStatement = monocdk_1.Stack.of(node).resolve(statement);
        const secureTransport = (_b = (_a = resolvedStatement === null || resolvedStatement === void 0 ? void 0 : resolvedStatement.Condition) === null || _a === void 0 ? void 0 : _a.Bool) === null || _b === void 0 ? void 0 : _b['aws:SecureTransport'];
        if (resolvedStatement.Effect === 'Deny' &&
            checkMatchingAction(resolvedStatement.Action) === true &&
            checkMatchingPrincipal(resolvedStatement.Principal) === true &&
            (secureTransport === 'false' || secureTransport === false)) {
            return true;
        }
    }
    return false;
}
/**
 * Helper function to check whether the topic Policy applies to topic actions
 * @param node The CfnTopicPolicy to check
 * @param actions The action in the topic policy
 * @returns Whether the CfnTopicPolicy applies to topic actions
 */
function checkMatchingAction(actions) {
    if (Array.isArray(actions)) {
        for (const action of actions) {
            if (action === '*' || action.toLowerCase() === 'sns:*') {
                return true;
            }
        }
    }
    else if (actions === '*' || actions.toLowerCase() === 'sns:*') {
        return true;
    }
    return false;
}
/**
 * Helper function to check whether the topic Policy applies to all principals
 * @param node The CfnTopicPolicy to check
 * @param principal The principals in the topic policy
 * @returns Whether the CfnTopicPolicy applies to all principals
 */
function checkMatchingPrincipal(principals) {
    if (principals === '*') {
        return true;
    }
    const awsPrincipal = principals.AWS;
    if (Array.isArray(awsPrincipal)) {
        for (const account of awsPrincipal) {
            if (account === '*') {
                return true;
            }
        }
    }
    else if (awsPrincipal === '*') {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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