"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_glue_1 = require("monocdk/aws-glue");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * Glue job bookmarks have encryption at rest enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_glue_1.CfnJob) {
        const securityConfigurationId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.securityConfiguration);
        if (securityConfigurationId == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_glue_1.CfnSecurityConfiguration) {
                if (isMatchingSecurityConfig(child, securityConfigurationId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the referenced Security Configuration encrypts job bookmarks
 * @param node the CfnSecurityConfiguration to check
 * @param securityConfigurationId the Cfn Logical ID of the security configuration
 * returns whether the CfnSecurityConfiguration encrypts job bookmarks
 */
function isMatchingSecurityConfig(node, securityConfigurationId) {
    const resolvedSecurityConfigurationLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
    if (resolvedSecurityConfigurationLogicalId === securityConfigurationId ||
        node.name === securityConfigurationId) {
        const encryptionConfiguration = monocdk_1.Stack.of(node).resolve(node.encryptionConfiguration);
        const jobBookmarksEncryption = encryptionConfiguration.jobBookmarksEncryption;
        if (jobBookmarksEncryption !== undefined) {
            const jobBookmarksEncryptionMode = nag_rules_1.NagRules.resolveIfPrimitive(node, jobBookmarksEncryption.jobBookmarksEncryptionMode);
            if (jobBookmarksEncryptionMode === 'CSE-KMS') {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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