"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsSolutionsChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../nag-pack");
const apigw_1 = require("../rules/apigw");
const appsync_1 = require("../rules/appsync");
const athena_1 = require("../rules/athena");
const autoscaling_1 = require("../rules/autoscaling");
const cloud9_1 = require("../rules/cloud9");
const cloudfront_1 = require("../rules/cloudfront");
const codebuild_1 = require("../rules/codebuild");
const cognito_1 = require("../rules/cognito");
const documentdb_1 = require("../rules/documentdb");
const dynamodb_1 = require("../rules/dynamodb");
const ec2_1 = require("../rules/ec2");
const ecr_1 = require("../rules/ecr");
const ecs_1 = require("../rules/ecs");
const efs_1 = require("../rules/efs");
const eks_1 = require("../rules/eks");
const elasticache_1 = require("../rules/elasticache");
const elasticbeanstalk_1 = require("../rules/elasticbeanstalk");
const elb_1 = require("../rules/elb");
const emr_1 = require("../rules/emr");
const eventbridge_1 = require("../rules/eventbridge");
const glue_1 = require("../rules/glue");
const iam_1 = require("../rules/iam");
const kinesis_1 = require("../rules/kinesis");
const kms_1 = require("../rules/kms");
const mediastore_1 = require("../rules/mediastore");
const msk_1 = require("../rules/msk");
const neptune_1 = require("../rules/neptune");
const opensearch_1 = require("../rules/opensearch");
const quicksight_1 = require("../rules/quicksight");
const rds_1 = require("../rules/rds");
const redshift_1 = require("../rules/redshift");
const s3_1 = require("../rules/s3");
const sagemaker_1 = require("../rules/sagemaker");
const secretsmanager_1 = require("../rules/secretsmanager");
const sns_1 = require("../rules/sns");
const sqs_1 = require("../rules/sqs");
const stepfunctions_1 = require("../rules/stepfunctions");
const timestream_1 = require("../rules/timestream");
const vpc_1 = require("../rules/vpc");
/**
 * Check Best practices based on AWS Solutions Security Matrix
 *
 */
class AwsSolutionsChecks extends nag_pack_1.NagPack {
    constructor(props) {
        super(props);
        this.packName = 'AwsSolutions';
    }
    visit(node) {
        if (node instanceof monocdk_1.CfnResource) {
            this.checkCompute(node);
            this.checkStorage(node);
            this.checkDatabases(node);
            this.checkNetworkDelivery(node);
            this.checkManagementGovernance(node);
            this.checkMachineLearning(node);
            this.checkAnalytics(node);
            this.checkSecurityCompliance(node);
            this.checkServerless(node);
            this.checkApplicationIntegration(node);
            this.checkMediaServices(node);
            this.checkDeveloperTools(node);
        }
    }
    /**
     * Check Compute Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCompute(node) {
        this.applyRule({
            ruleSuffixOverride: 'EB1',
            info: 'The Elastic Beanstalk environment is not configured to use a specific VPC.',
            explanation: 'Use a non-default in order to seperate your environment from default resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkVPCSpecified,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EB3',
            info: 'The Elastic Beanstalk environment does not have managed updates enabled.',
            explanation: 'Enable managed platform updates for beanstalk environments in order to receive bug fixes, software updates and new features. Managed platform updates perform immutable environment updates.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkManagedUpdatesEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EB4',
            info: 'The Elastic Beanstalk environment does not upload EC2 Instance logs to S3.',
            explanation: 'Beanstalk environment logs should be retained and uploaded to Amazon S3 in order to keep the logging data for future audits, historical purposes or to track and analyze the EB application environment behavior for a long period of time.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: elasticbeanstalk_1.ElasticBeanstalkEC2InstanceLogsToS3,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC23',
            info: 'The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.',
            explanation: 'Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when  all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedInbound,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC26',
            info: 'The EBS volume has encryption disabled.',
            explanation: "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2EBSVolumeEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC27',
            info: 'The Security Group does not have a description.',
            explanation: 'Descriptions help simplify operations and remove any opportunities for operator errors.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2SecurityGroupDescription,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC28',
            info: 'The EC2 instance/AutoScaling launch configuration does not have detailed monitoring enabled.',
            explanation: 'Monitoring data helps make better decisions on architecting and managing compute resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceDetailedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC29',
            info: 'The EC2 instance is not part of an ASG and has Termination Protection disabled.',
            explanation: 'Termination Protection safety feature enabled in order to protect the instances from being accidentally terminated.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceTerminationProtection,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECR1',
            info: 'The ECR Repository allows open access.',
            explanation: 'Removing * principals in an ECR Repository helps protect against unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecr_1.ECROpenAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS2',
            info: 'The ECS Task Definition includes a container definition that directly specifies environment variables.',
            explanation: 'Use secrets to inject environment variables during container startup from AWS Systems Manager Parameter Store or Secrets Manager instead of directly specifying plaintext environment variables. Updates to direct environment variables require operators to change task definitions and perform new deployments.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionNoEnvironmentVariables,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS4',
            info: 'The ECS Cluster has CloudWatch Container Insights disabled.',
            explanation: 'CloudWatch Container Insights allow operators to gain a better perspective on how the cluster’s applications and microservices are performing.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSClusterCloudWatchContainerInsights,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS7',
            info: 'The ECS Task Definition does not have awslogs logging enabled at the minimum.',
            explanation: 'Container logging allows operators to view and aggregate the logs from the container.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionContainerLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EKS1',
            info: "The EKS cluster's Kubernetes API server endpoint has public access enabled.",
            explanation: "A cluster's Kubernetes API server endpoint should not be publicly accessible from the Internet in order to avoid exposing private data and minimizing security risks. The API server endpoints should only be accessible from within a AWS Virtual Private Cloud (VPC).",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: eks_1.EKSClusterNoEndpointPublicAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EKS2',
            info: "The EKS Cluster does not publish 'api', 'audit', 'authenticator, 'controllerManager', and 'scheduler' control plane logs.",
            explanation: 'EKS control plane logging provides audit and diagnostic logs directly from the Amazon EKS control plane to CloudWatch Logs in your account. These logs make it easy for you to secure and run your clusters.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: eks_1.EKSClusterControlPlaneLogs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB1',
            info: 'The CLB is used for incoming HTTP/HTTPS traffic. Use ALBs instead.',
            explanation: 'HTTP/HTTPS applications (monolithic or containerized) should use the ALB instead of the CLB for enhanced incoming traffic distribution, better performance and lower costs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.CLBNoInboundHttpHttps,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB2',
            info: 'The ELB does not have access logs enabled.',
            explanation: 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB3',
            info: 'The CLB does not have connection draining enabled.',
            explanation: 'With Connection Draining feature enabled, if an EC2 backend instance fails health checks The CLB will not send any new requests to the unhealthy instance. However, it will still allow existing (in-flight) requests to complete for the duration of the configured timeout.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.CLBConnectionDraining,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB4',
            info: 'The CLB does not use at least two AZs with the Cross-Zone Load Balancing feature enabled.',
            explanation: 'CLBs can distribute the traffic evenly across all backend instances. To use Cross-Zone Load Balancing at optimal level, the system should maintain an equal EC2 capacity distribution in each of the AZs registered with the load balancer.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBCrossZoneLoadBalancingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB5',
            info: 'The CLB listener is not configured for secure (HTTPs or SSL) protocols for client communication.',
            explanation: 'The HTTPs or SSL protocols enable secure communication by encrypting the communication between the client and the load balancer.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBTlsHttpsListenersOnly,
            node: node,
        });
    }
    /**
     * Check Storage Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkStorage(node) {
        this.applyRule({
            ruleSuffixOverride: 'S1',
            info: 'The S3 Bucket has server access logs disabled.',
            explanation: 'The bucket should have server access logging enabled to provide detailed records for the requests that are made to the bucket.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S2',
            info: 'The S3 Bucket does not have public access restricted and blocked.',
            explanation: 'The bucket should have public access restricted and blocked to prevent unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLevelPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S3',
            info: 'The S3 Bucket does not default encryption enabled.',
            explanation: 'The bucket should minimally have SSE enabled to help protect data-at-rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketServerSideEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S10',
            info: 'The S3 Bucket does not require requests to use SSL.',
            explanation: 'You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on Amazon S3 bucket policies.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EFS1',
            info: 'The EFS is not configured for encryption at rest.',
            explanation: 'By using an encrypted file system, data and metadata are automatically encrypted before being written to the file system. Similarly, as data and metadata are read, they are automatically decrypted before being presented to the application. These processes are handled transparently by EFS without requiring modification of applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check Database Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDatabases(node) {
        this.applyRule({
            ruleSuffixOverride: 'RDS2',
            info: 'The RDS instance or Aurora DB cluster does not have storage encryption enabled.',
            explanation: 'Storage encryption helps protect data-at-rest by encrypting the underlying storage, automated backups, read replicas, and snapshots for the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSStorageEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS3',
            info: 'The non-Aurora RDS DB instance does not have multi-AZ support enabled.',
            explanation: 'Use multi-AZ deployment configurations for high availability and automatic failover support fully managed by AWS.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSMultiAZSupport,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS6',
            info: 'The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLPostgresIAMAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS8',
            info: 'The RDS DB Security Group allows for 0.0.0.0/0 inbound access.',
            explanation: 'RDS DB security groups should not allow access from 0.0.0.0/0 (i.e. anywhere, every machine that has the ability to establish a connection) in order to reduce the risk of unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSRestrictedInbound,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS10',
            info: 'The RDS instance or Aurora DB cluster does not have deletion protection enabled.',
            explanation: 'The deletion protection feature helps protect the database from being accidentally deleted.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS11',
            info: 'The RDS instance or Aurora DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS13',
            info: 'The RDS instance is not configured for automated backups.',
            explanation: 'Automated backups allow for point-in-time recovery.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceBackupEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS14',
            info: 'The RDS Aurora MySQL cluster does not have Backtrack enabled.',
            explanation: 'Backtrack helps order to rewind cluster tables to a specific time, without using backups.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLBacktrack,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS15',
            info: 'The RDS DB instance or Aurora DB cluster does not have deletion protection enabled.',
            explanation: 'Enabling Deletion Protection at the cluster level for Amazon Aurora databases or instance level for non Aurora instances helps protect from accidental deletion.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS16',
            info: 'The RDS Aurora MySQL serverless cluster does not have audit, error, general, and slowquery Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DDB3',
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled.',
            explanation: 'DynamoDB continuous backups represent an additional layer of insurance against accidental loss of data on top of on-demand backups. The DynamoDB service can back up the data with per-second granularity and restore it to any single second from the time PITR was enabled up to the prior 35 days.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: dynamodb_1.DynamoDBPITREnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DDB4',
            info: 'The DAX cluster does not have server-side encryption enabled.',
            explanation: 'Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DAXEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC1',
            info: 'The ElastiCache cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheClusterInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC3',
            info: 'The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.',
            explanation: 'Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC4',
            info: 'The ElastiCache Redis cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterMultiAZ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC5',
            info: 'The ElastiCache cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC6',
            info: 'The ElastiCache Redis cluster does not use Redis AUTH for user authentication.',
            explanation: 'Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterRedisAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N1',
            info: 'The Neptune DB cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterMultiAZ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N2',
            info: 'The Neptune DB instance does have Auto Minor Version Upgrade enabled.',
            explanation: 'The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterAutomaticMinorVersionUpgrade,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N3',
            info: 'The Neptune DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterBackupRetentionPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N4',
            info: 'The Neptune DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N5',
            info: 'The Neptune DB cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterIAMAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS1',
            info: 'The Redshift cluster does not require TLS/SSL encryption.',
            explanation: 'Enabling the "require_ssl" parameter secures data-in-transit by encrypting the connection between the clients and the Redshift clusters.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftRequireTlsSSL,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS2',
            info: 'The Redshift cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS3',
            info: 'The Redshift cluster uses the default "awsuser" username.',
            explanation: 'Using a custom master user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterNonDefaultUsername,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS4',
            info: 'The Redshift cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS5',
            info: 'The Redshift cluster does not have audit logging enabled.',
            explanation: 'Audit logging helps operators troubleshoot issues and ensure security.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterAuditLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS6',
            info: 'The Redshift cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS8',
            info: 'The Redshift cluster is publicly accessible.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS9',
            info: 'The Redshift cluster does not have version upgrade enabled.',
            explanation: 'Version Upgrade must enabled to enable the cluster to automatically receive upgrades during the maintenance window.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterVersionUpgrade,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS10',
            info: 'The Redshift cluster does not have a retention period for automated snapshots configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A positive retention period should be set to configure this feature.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftBackupEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS11',
            info: 'The Redshift cluster does not have user activity logging enabled.',
            explanation: 'User activity logging logs each query before it is performed on the clusters databse. To enable this feature associate a Resdhsift Cluster Parameter Group with the "enable_user_activity_logging" parameter set to "true".',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterUserActivityLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC1',
            info: 'The Document DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC2',
            info: 'The Document DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC3',
            info: 'The Document DB cluster does not have the username and password stored in Secrets Manager.',
            explanation: "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBCredentialsInSecretsManager,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC4',
            info: 'The Document DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterBackupRetentionPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC5',
            info: 'The Document DB cluster does not have authenticate, createIndex, and dropCollection Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database. The events recorded by the AWS DocumentDB audit logs include successful and failed authentication attempts, creating indexes or dropping a collection in a database within the DocumentDB cluster.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterLogExports,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'TS3',
            info: 'The Timestream database does not use a Customer Managed KMS Key for at rest encryption.',
            explanation: 'All Timestream tables in a database are encrypted at rest by default using AWS Managed Key. These keys are rotated every three years. Data at rest must be encrypted using CMKs if you require more control over the permissions and lifecycle of your keys, including the ability to have them automatically rotated on an annual basis.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: timestream_1.TimestreamDatabaseCustomerManagedKey,
            node: node,
        });
    }
    /**
     * Check Network and Delivery Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkNetworkDelivery(node) {
        this.applyRule({
            ruleSuffixOverride: 'VPC3',
            info: 'A Network ACL or Network ACL entry has been implemented.',
            explanation: 'Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: vpc_1.VPCNoNACLs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'VPC7',
            info: 'The VPC does not have an associated Flow Log.',
            explanation: 'VPC Flow Logs capture network flow information for a VPC, subnet, or network interface and stores it in Amazon CloudWatch Logs. Flow log data can help customers troubleshoot network issues; for example, to diagnose why specific traffic is not reaching an instance, which might be a result of overly restrictive security group rules.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCFlowLogsEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR1',
            info: 'The CloudFront distribution may require Geo restrictions.',
            explanation: 'Geo restriction may need to be enabled for the distribution in order to allow or deny a country in order to allow or restrict users in specific locations from accessing content.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: cloudfront_1.CloudFrontDistributionGeoRestrictions,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR2',
            info: 'The CloudFront distribution may require integration with AWS WAF.',
            explanation: 'The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: cloudfront_1.CloudFrontDistributionWAFIntegration,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR3',
            info: 'The CloudFront distributions does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators track all viewer requests for the content delivered through the Content Delivery Network.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR5',
            info: 'The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.',
            explanation: 'Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionNoOutdatedSSL,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR6',
            info: 'The CloudFront distribution does not use an origin access identity an S3 origin.',
            explanation: 'Origin access identities help with security by restricting any direct access to objects through S3 URLs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionS3OriginAccessIdentity,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG1',
            info: 'The API does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators view who accessed an API and how the caller accessed the API.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG2',
            info: 'The REST API does not have request validation enabled.',
            explanation: 'The API should have basic request validation enabled. If the API is integrated with custom source (Lambda, ECS, etc..) in the backend, deeper input validation should be considered for implementation.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWRequestValidation,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG3',
            info: 'The REST API stage is not associated with AWS WAFv2 web ACL.',
            explanation: 'AWS WAFv2 is a web application firewall that helps protect web applications and APIs from attacks by allowing configured rules to allow, block, or monitor (count) web requests based on customizable rules and conditions that are defined.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: apigw_1.APIGWAssociatedWithWAF,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG4',
            info: 'The API does not implement authorization.',
            explanation: 'In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAuthorization,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG6',
            info: 'The REST API Stage does not have CloudWatch logging enabled for all methods.',
            explanation: 'Enabling CloudWatch logs at the stage level helps operators to track and analyze execution behavior at the API stage level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWExecutionLoggingEnabled,
            node: node,
        });
    }
    /**
     * Check Management and Governance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkManagementGovernance(node) {
        this.applyRule({
            ruleSuffixOverride: 'AS1',
            info: 'The Auto Scaling Group does not have a cooldown period.',
            explanation: 'A cooldown period temporarily suspends any scaling activities in order to allow the newly launched EC2 instance(s) some time to start handling the application traffic.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupCooldownPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AS2',
            info: 'The Auto Scaling Group does not have properly configured health checks.',
            explanation: 'The health check feature enables the service to detect whether its registered EC2 instances are healthy or not.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupHealthCheck,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AS3',
            info: 'The Auto Scaling Group does not have notifications configured for all scaling events.',
            explanation: 'Notifications on EC2 instance launch, launch error, termination, and termination errors allow operators to gain better insights into systems attributes such as activity and health.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupScalingNotifications,
            node: node,
        });
    }
    /**
     * Check Machine Learning Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMachineLearning(node) {
        this.applyRule({
            ruleSuffixOverride: 'SM1',
            info: 'The SageMaker notebook instance is not provisioned inside a VPC.',
            explanation: 'Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SM2',
            info: 'The SageMaker notebook instance does not have an encrypted storage volume.',
            explanation: 'Encrypting storage volumes helps protect SageMaker data-at-rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SM3',
            info: 'The SageMaker notebook instance has direct internet access enabled.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Analytics Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAnalytics(node) {
        this.applyRule({
            ruleSuffixOverride: 'ATH1',
            info: 'The Athena workgroup does not encrypt query results.',
            explanation: 'Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: athena_1.AthenaWorkgroupEncryptedQueryResults,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR2',
            info: 'The EMR cluster does not have S3 logging enabled.',
            explanation: 'Uploading logs to S3 enables the system to keep the logging data for historical purposes or to track and analyze the clusters behavior.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRS3AccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR4',
            info: 'The EMR cluster does not use a security configuration with local disk encryption enabled.',
            explanation: 'Local disk encryption uses a combination of open-source HDFS encryption and LUKS encryption to secure data at rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRLocalDiskEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR5',
            info: 'The EMR cluster does not use a security configuration with encryption in transit enabled and configured.',
            explanation: 'EMR Clusters should have a method for encrypting data in transit using Transport Layer Security (TLS).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMREncryptionInTransit,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR6',
            info: 'The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.',
            explanation: 'SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRAuthEC2KeyPairOrKerberos,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'GL1',
            info: 'The Glue crawler or job does not use a security configuration with CloudWatch Log encryption enabled.',
            explanation: 'Enabling encryption at rest helps prevent unauthorized users from getting access to the logging data published to CloudWatch Logs.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: glue_1.GlueEncryptedCloudWatchLogs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'GL3',
            info: 'The Glue job does not have use a security configuration with job bookmark encryption enabled.',
            explanation: 'Job bookmark encryption encrypts bookmark data before it is sent to Amazon S3 for storage.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: glue_1.GlueJobBookmarkEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDA3',
            info: 'The Kinesis Data Analytics Flink Application does not have checkpointing enabled.',
            explanation: 'Checkpoints are backups of application state that KDA automatically creates periodically and uses to restore from faults.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: kinesis_1.KinesisDataAnalyticsFlinkCheckpointing,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDF1',
            info: 'The Kinesis Data Firehose delivery stream does have server-side encryption enabled.',
            explanation: 'This allows the system to meet strict regulatory requirements and enhance the security of system data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kinesis_1.KinesisDataFirehoseSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDS1',
            info: 'The Kinesis Data Stream does not has server-side encryption enabled.',
            explanation: "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kinesis_1.KinesisDataStreamSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDS3',
            info: 'The Kinesis Data Stream specifies server-side encryption and does not use the "aws/kinesis" key.',
            explanation: 'Customer Managed Keys can incur additional costs that scale with the amount of consumers and producers. Ensure that Customer Managed Keys are required for compliance before using them (https://docs.aws.amazon.com/streams/latest/dev/costs-performance.html).',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: kinesis_1.KinesisDataStreamDefaultKeyWhenSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK2',
            info: 'The MSK cluster uses plaintext communication between clients and brokers.',
            explanation: 'TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKClientToBrokerTLS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK3',
            info: 'The MSK cluster uses plaintext communication between brokers.',
            explanation: 'TLS communication secures data-in-transit by encrypting the connection between brokers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKBrokerToBrokerTLS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK6',
            info: 'The MSK cluster does not send broker logs to a supported destination.',
            explanation: 'Broker logs enable operators to troubleshoot Apache Kafka applications and to analyze their communications with the MSK cluster. The cluster can deliver logs to the following resources: a CloudWatch log group, an S3 bucket, a Kinesis Data Firehose delivery stream.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKBrokerLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS1',
            info: 'The OpenSearch Service domain is not provisioned inside a VPC.',
            explanation: 'Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS2',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled.',
            explanation: 'Enabling the node-to-node encryption feature adds an extra layer of data protection on top of the existing ES security features such as HTTPS client to cluster encryption and data-at-rest encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNodeToNodeEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS3',
            info: 'The OpenSearch Service domain does not only grant access via allowlisted IP addresses.',
            explanation: 'Using allowlisted IP addresses helps protect the domain against unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchAllowlistedIPs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS4',
            info: 'The OpenSearch Service domain does not use dedicated master nodes.',
            explanation: 'Using dedicated master nodes helps improve environmental stability by offloading all the management tasks from the data nodes.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchDedicatedMasterNode,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS5',
            info: 'The OpenSearch Service domain does not allow for unsigned requests or anonymous access.',
            explanation: 'Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNoUnsignedOrAnonymousAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS7',
            info: 'The OpenSearch Service domain does not have Zone Awareness enabled.',
            explanation: 'Enabling cross-zone replication (Zone Awareness) increases the availability of the OpenSearch Service domain by allocating the nodes and replicate the data across two AZs in the same region in order to prevent data loss and minimize downtime in the event of node or AZ failure.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchZoneAwareness,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS8',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS9',
            info: 'The OpenSearch Service domain does not minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs.',
            explanation: 'These logs enable operators to gain full insight into the performance of these operations.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchSlowLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'QS1',
            info: 'The Quicksight data sources connection is not configured to use SSL.',
            explanation: 'SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: quicksight_1.QuicksightSSLConnections,
            node: node,
        });
    }
    /**
     * Check Security and Compliance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecurityCompliance(node) {
        this.applyRule({
            ruleSuffixOverride: 'IAM4',
            info: 'The IAM user, role, or group uses AWS managed policies.',
            explanation: 'An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoManagedPolicies,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'IAM5',
            info: 'The IAM entity contains wildcard permissions and does not have a cdk_nag rule suppression with evidence for those permission.',
            explanation: 'Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoWildcardPermissions,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG1',
            info: 'The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.',
            explanation: 'Strong password policies increase system security by encouraging users to create reliable and secure passwords.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolStrongPasswordPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG2',
            info: 'The Cognito user pool does not require MFA.',
            explanation: 'Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: cognito_1.CognitoUserPoolMFA,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG3',
            info: 'The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.',
            explanation: 'Advanced security features enable the system to detect and act upon malicious sign-in attempts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolAdvancedSecurityModeEnforced,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG4',
            info: 'The API GW method does not use a Cognito user pool authorizer.',
            explanation: 'API Gateway validates the tokens from a successful user pool authentication, and uses them to grant your users access to resources including Lambda functions, or your own API.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolAPIGWAuthorizer,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG7',
            info: 'The Cognito identity pool allows for unauthenticated logins and does not have a cdk_nag rule suppression with a reason.',
            explanation: 'In many cases applications do not warrant unauthenticated guest access applications. Metadata explaining the use case allows for transparency to operators.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolNoUnauthenticatedLogins,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KMS5',
            info: 'The KMS Symmetric key does not have automatic key rotation enabled.',
            explanation: 'KMS key rotation allow a system to set an yearly rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kms_1.KMSBackingKeyRotationEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SMG4',
            info: 'The secret does not have automatic rotation scheduled.',
            explanation: 'AWS Secrets Manager can be configured to automatically rotate the secret for a secured service or database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.SecretsManagerRotationEnabled,
            node: node,
        });
    }
    /**
     * Check Serverless Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkServerless(node) {
        this.applyRule({
            ruleSuffixOverride: 'ASC3',
            info: 'The GraphQL API does not have request leveling logging enabled.',
            explanation: 'It is important to use CloudWatch Logs to log metrics such as who has accessed the GraphQL API, how the caller accessed the API, and invalid requests.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: appsync_1.AppSyncGraphQLRequestLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SF1',
            info: 'The Step Function does not log "ALL" events to CloudWatch Logs.',
            explanation: 'Logging "ALL" events to CloudWatch logs help operators troubleshoot and audit systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: stepfunctions_1.StepFunctionStateMachineAllLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SF2',
            info: 'The Step Function does not have X-Ray tracing enabled.',
            explanation: 'X-ray provides an end-to-end view of how an application is performing. This helps operators to discover performance issues, detect permission problems, and track requests made to and from other AWS services.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: stepfunctions_1.StepFunctionStateMachineXray,
            node: node,
        });
    }
    /**
     * Check Application Integration Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkApplicationIntegration(node) {
        this.applyRule({
            ruleSuffixOverride: 'EVB1',
            info: 'The event bus policy allows for open access.',
            explanation: 'An open policy ("*" principal without a condition) grants anonymous access to an event bus. Use a condition to limit the permission to accounts that fulfill a certain requirement, such as being a member of a certain AWS organization.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: eventbridge_1.EventBusOpenAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SNS2',
            info: 'The SNS Topic does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSEncryptedKMS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SNS3',
            info: 'The SNS Topic does not require requests to use SSL.',
            explanation: 'Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition in the topic policy to force requests to use SSL. If SSE is already enabled then this control is auto enforced.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSTopicSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS2',
            info: 'The SQS Queue does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS3',
            info: 'The SQS queue does not have a dead-letter queue (DLQ) enabled or have a cdk_nag rule suppression indicating it is a DLQ.',
            explanation: 'Using a DLQ helps maintain the queue flow and avoid losing data by detecting and mitigating failures and service disruptions on time.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueDLQ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS4',
            info: 'The SQS queue does not require requests to use SSL.',
            explanation: 'Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition in the queue policy to force requests to use SSL.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueSSLRequestsOnly,
            node: node,
        });
    }
    /**
     * Check Media Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMediaServices(node) {
        this.applyRule({
            ruleSuffixOverride: 'MS1',
            info: 'The MediaStore container does not have container access logging enabled.',
            explanation: 'The container should have access logging enabled to provide detailed records for the requests that are made to the container.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: mediastore_1.MediaStoreContainerAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS3',
            info: 'The MediaStore container does not require requests to use SSL.',
            explanation: 'You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on MediaStore container policies.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: mediastore_1.MediaStoreContainerSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS4',
            info: 'The MediaStore container does not define a metric policy to send metrics to CloudWatch.',
            explanation: 'Using a combination of MediaStore metrics and CloudWatch alarms helps operators gain better insights into container operations.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreCloudWatchMetricPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS7',
            info: 'The MediaStore container does not define a container policy.',
            explanation: 'Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerHasContainerPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS8',
            info: 'The MediaStore container does not define a CORS policy.',
            explanation: 'Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerCORSPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS10',
            info: 'The MediaStore container does not define a lifecycle policy.',
            explanation: 'Many use cases warrant the usage of lifecycle configurations to manage container objects during their lifetime.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerLifecyclePolicy,
            node: node,
        });
    }
    /**
     * Check Developer Tools Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDeveloperTools(node) {
        this.applyRule({
            ruleSuffixOverride: 'CB3',
            info: 'The CodeBuild project has privileged mode enabled.',
            explanation: 'Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: codebuild_1.CodeBuildProjectPrivilegedModeDisabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CB4',
            info: 'The CodeBuild project does not use an AWS KMS key for encryption.',
            explanation: 'Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.CodeBuildProjectKMSEncryptedArtifacts,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CB5',
            info: 'The Codebuild project does not use images provided by the CodeBuild service or have a cdk_nag suppression rule explaining the need for a custom image.',
            explanation: 'Explaining differences/edits to Docker images helps operators better understand system dependencies.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: codebuild_1.CodeBuildProjectManagedImages,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'C91',
            info: 'The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.',
            explanation: 'SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloud9_1.Cloud9InstanceNoIngressSystemsManager,
            node: node,
        });
    }
}
exports.AwsSolutionsChecks = AwsSolutionsChecks;
_a = JSII_RTTI_SYMBOL_1;
AwsSolutionsChecks[_a] = { fqn: "monocdk-nag.AwsSolutionsChecks", version: "1.6.2" };
//# sourceMappingURL=data:application/json;base64,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