"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagPack = exports.NagMessageLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const fs_1 = require("fs");
const path_1 = require("path");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("./nag-rules");
const VALIDATION_FAILURE_ID = 'CdkNagValidationFailure';
const SUPPRESSION_ID = 'CdkNagSuppression';
/**
 * The level of the message that the rule applies.
 */
var NagMessageLevel;
(function (NagMessageLevel) {
    NagMessageLevel["WARN"] = "Warning";
    NagMessageLevel["ERROR"] = "Error";
})(NagMessageLevel = exports.NagMessageLevel || (exports.NagMessageLevel = {}));
/**
 * Base class for all rule packs.
 */
class NagPack {
    constructor(props) {
        this.reportStacks = new Array();
        this.packName = '';
        this.verbose =
            props == undefined || props.verbose == undefined ? false : props.verbose;
        this.logIgnores =
            props == undefined || props.logIgnores == undefined
                ? false
                : props.logIgnores;
        this.reports =
            props == undefined || props.reports == undefined ? true : props.reports;
    }
    get readPackName() {
        return this.packName;
    }
    get readReportStacks() {
        return this.reportStacks;
    }
    /**
     * Create a rule to be used in the NagPack.
     * @param params The @IApplyRule interface with rule details.
     */
    applyRule(params) {
        var _b, _c, _d;
        if (this.packName === '') {
            throw Error('The NagPack does not have a pack name, therefore the rule could not be applied. Set a packName in the NagPack constructor.');
        }
        let resourceIgnores = (_b = params.node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
        resourceIgnores = resourceIgnores ? resourceIgnores : [];
        let stackIgnores = (_d = (_c = monocdk_1.Stack.of(params.node).templateOptions.metadata) === null || _c === void 0 ? void 0 : _c.cdk_nag) === null || _d === void 0 ? void 0 : _d.rules_to_suppress;
        stackIgnores = stackIgnores ? stackIgnores : [];
        const allIgnores = resourceIgnores.concat(stackIgnores);
        const ruleSuffix = params.ruleSuffixOverride
            ? params.ruleSuffixOverride
            : params.rule.name;
        const ruleId = `${this.packName}-${ruleSuffix}`;
        try {
            const ruleCompliance = params.rule(params.node);
            if (this.reports === true &&
                ruleCompliance === nag_rules_1.NagRuleCompliance.COMPLIANT) {
                this.writeToStackComplianceReport(params, ruleId, ruleCompliance);
            }
            else if (ruleCompliance === nag_rules_1.NagRuleCompliance.NON_COMPLIANT) {
                const reason = this.ignoreRule(allIgnores, ruleId);
                if (this.reports === true) {
                    this.writeToStackComplianceReport(params, ruleId, ruleCompliance, reason);
                }
                if (reason) {
                    if (this.logIgnores === true) {
                        const message = this.createMessage(SUPPRESSION_ID, `${ruleId} was triggered but suppressed.`, `Provided reason: "${reason}"`);
                        monocdk_1.Annotations.of(params.node).addInfo(message);
                    }
                }
                else {
                    const message = this.createMessage(ruleId, params.info, params.explanation);
                    if (params.level == NagMessageLevel.ERROR) {
                        monocdk_1.Annotations.of(params.node).addError(message);
                    }
                    else if (params.level == NagMessageLevel.WARN) {
                        monocdk_1.Annotations.of(params.node).addWarning(message);
                    }
                }
            }
        }
        catch (error) {
            const reason = this.ignoreRule(allIgnores, VALIDATION_FAILURE_ID);
            if (this.reports === true) {
                this.writeToStackComplianceReport(params, ruleId, 'UNKNOWN', reason);
            }
            if (reason) {
                if (this.logIgnores === true) {
                    const message = this.createMessage(SUPPRESSION_ID, `${VALIDATION_FAILURE_ID} was triggered but suppressed.`, reason);
                    monocdk_1.Annotations.of(params.node).addInfo(message);
                }
            }
            else {
                const information = `'${ruleId}' threw an error during validation. This is generally caused by a parameter referencing an intrinsic function. For more details enable verbose logging.'`;
                const message = this.createMessage(VALIDATION_FAILURE_ID, information, error.message);
                monocdk_1.Annotations.of(params.node).addWarning(message);
            }
        }
    }
    /**
     * Check whether a specific rule should be ignored.
     * @param ignores The ignores listed in cdk-nag metadata.
     * @param ruleId The id of the rule to ignore.
     * @returns The reason the rule was ignored, or an empty string.
     */
    ignoreRule(ignores, ruleId) {
        for (let ignore of ignores) {
            if (ignore.id &&
                ignore.reason &&
                JSON.stringify(ignore.reason).length >= 10) {
                if (ignore.id == ruleId) {
                    return ignore.reason;
                }
            }
            else {
                throw Error(`Improperly formatted cdk_nag rule suppression detected: ${JSON.stringify(ignore)}. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
            }
        }
        return '';
    }
    /**
     * The message to output to the console when a rule is triggered.
     * @param ruleId The id of the rule.
     * @param info Why the rule was triggered.
     * @param explanation Why the rule exists.
     * @returns The formatted message string.
     */
    createMessage(ruleId, info, explanation) {
        let message = `${ruleId}: ${info}`;
        return this.verbose ? `${message} ${explanation}\n` : `${message}\n`;
    }
    /**
     * Write a line to the rule packs compliance report for the resource's Stack
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    writeToStackComplianceReport(params, ruleId, compliance, explanation = '') {
        var _b;
        const line = this.createComplianceReportLine(params, ruleId, compliance, explanation);
        let outDir = (_b = monocdk_1.App.of(params.node)) === null || _b === void 0 ? void 0 : _b.outdir;
        const fileName = `${this.packName}-${params.node.stack.stackName}-NagReport.csv`;
        const filePath = path_1.join(outDir ? outDir : '', fileName);
        if (!this.reportStacks.includes(fileName)) {
            this.reportStacks.push(fileName);
            fs_1.writeFileSync(filePath, 'Rule ID,Resource ID,Compliance,Exception Reason,Rule Level,Rule Info\n');
        }
        fs_1.appendFileSync(filePath, line);
    }
    /**
     * Helper function to create a line for the compliance report
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    createComplianceReportLine(params, ruleId, compliance, explanation = '') {
        //| Rule ID | Resource ID | Compliance | Exception Reason | Rule Level | Rule Info
        const line = Array();
        line.push(ruleId);
        line.push(params.node.node.path);
        if ((compliance === nag_rules_1.NagRuleCompliance.NON_COMPLIANT ||
            compliance === 'UNKNOWN') &&
            explanation !== '') {
            line.push('Suppressed');
            line.push(explanation);
        }
        else {
            line.push(compliance);
            line.push('N/A');
        }
        line.push(params.level);
        line.push(params.info);
        return line.map((i) => '"' + i.replace(/"/g, '""') + '"').join(',') + '\n';
    }
}
exports.NagPack = NagPack;
_a = JSII_RTTI_SYMBOL_1;
NagPack[_a] = { fqn: "monocdk-nag.NagPack", version: "1.6.2" };
//# sourceMappingURL=data:application/json;base64,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