# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['okama', 'okama.api', 'okama.common', 'okama.common.helpers', 'okama.frontier']

package_data = \
{'': ['*']}

install_requires = \
['importlib-metadata>=4.11.1,<5.0.0',
 'matplotlib>=3.5.1,<4.0.0',
 'pandas>=0.25.0',
 'requests>=2.27.1,<3.0.0',
 'scipy>=0.14.0']

extras_require = \
{'dev': ['nbmake>=1.2,<2.0'],
 'docs': ['sphinx>=3,<4'],
 'jupyter': ['jupyter>=1.0.0,<2.0.0']}

setup_kwargs = {
    'name': 'okama',
    'version': '1.1.4',
    'description': 'Investment portfolio analyzing & optimization tools',
    'long_description': '\n[![Documentation Status](https://img.shields.io/readthedocs/okama.svg?style=popout)](http://okama.readthedocs.io/)\n[![Python](https://img.shields.io/badge/python-v3-brightgreen.svg)](https://www.python.org/)\n[![PyPI Latest Release](https://img.shields.io/pypi/v/okama.svg)](https://pypi.org/project/okama/)\n[![Coverage](https://coveralls.io/repos/github/mbk-dev/okama/badge.svg?branch=master)](https://coveralls.io/github/mbk-dev/okama?branch=master)\n[![License](https://img.shields.io/pypi/l/okama.svg)](https://opensource.org/licenses/MIT)\n[![Open In Collab](https://colab.research.google.com/assets/colab-badge.svg)](https://colab.research.google.com/github/mbk-dev/okama/blob/master/examples/01%20howto.ipynb)\n\n# Okama\n\n_okama_ is a library with investment portfolio analyzing & optimization tools. CFA recommendations are used in quantitative finance.\n\n_okama_ goes with **free** «end of day» historical stock markets data and macroeconomic indicators through API.\n>...entities should not be multiplied without necessity\n>\n> -- <cite>William of Ockham (c. 1287–1347)</cite>\n\n## Table of contents\n\n- [Okama main features](#okama-main-features)\n- [Financial data and macroeconomic indicators](#financial-data-and-macroeconomic-indicators)\n  - [End of day historical data](#end-of-day-historical-data)\n  - [Macroeconomic indicators](#macroeconomic-indicators)\n  - [Other historical data](#other-historical-data)\n- [Installation](#installation)\n- [Getting started](#getting-started)\n- [Documentation](#documentation)\n- [RoadMap](#roadmap)\n- [Contributing to okama](#contributing-to-okama)\n- [Communication](#communication)\n\n## Okama main features\n\n- Investment portfolio constrained Markowitz Mean-Variance Analysis (MVA) and optimization\n- Rebalanced portfolio optimization with constraints (multi-period Efficient Frontier)\n- Monte Carlo Simulations for financial assets and investment portfolios\n- Popular risk metrics: VAR, CVaR, semi-deviation, variance and drawdowns\n- Different financial ratios: Sharpe ratio, Sortino ratio, Diversification ratio \n- Forecasting models according to normal and lognormal distribution\n- Testing distribution on historical data\n- Dividend yield and other dividend indicators for stocks\n- Backtesting and comparing historical performance of broad range of assets and indexes in multiple currencies\n- Methods to track the performance of index funds (ETF) and compare them with benchmarks\n- Main macroeconomic indicators: inflation, central banks rates\n- Matplotlib visualization scripts for the Efficient Frontier, Transition map and assets risk / return performance\n\n## Financial data and macroeconomic indicators\n\n### End of day historical data\n\n- Stocks and ETF for main world markets\n- Mutual funds\n- Commodities\n- Stock indexes\n\n### Currencies\n\n- FX currencies\n- Crypto currencies\n- Central bank exchange rates\n\n### Macroeconomic indicators\n\n- Inflation for many countries (USA, United Kingdom, European Union, Russia etc.)\n- Central bank rates\n\n### Other historical data\n\n- Real estate prices\n- Top bank rates\n\n## Installation\n\n`pip install okama`\n\nThe latest development version can be installed directly from GitHub:\n\n`pip install git+https://github.com/mbk-dev/okama@dev`\n\n\n## Getting started\n\n### 1. Compare several assets from different stock markets. Get USD-adjusted performance\n\n```python\nimport okama as ok\n\nx = ok.AssetList([\'SPY.US\', \'BND.US\', \'DBXD.XFRA\'], ccy=\'USD\')\nx  # all examples are for Jupyter Notebook/iPython. For raw Python interpreter use \'print(x)\' instead.\n\n```\n![](../images/images/readmi01.jpg?raw=true) \n\nGet the main parameters for the set:\n```python\nx.describe()\n```\n![](../images/images/readmi02.jpg?raw=true) \n\nGet the assets accumulated return, plot it and compare with the USD inflation:\n```python\nx.wealth_indexes.plot()\n```\n![](../images/images/readmi03.jpg?raw=true) \n\n### 2. Create a dividend stocks portfolio with base currency EUR\n\n```python\nweights = [0.3, 0.2, 0.2, 0.2, 0.1]\nassets = [\'T.US\', \'XOM.US\', \'FRE.XETR\', \'SNW.XETR\', \'LKOH.MOEX\']\npf = ok.Portfolio(assets, weights=weights, ccy=\'EUR\')\npf.table\n```\n![](../images/images/readmi04.jpg?raw=true) \n\nPlot the dividend yield of the portfolio (adjusted to the base currency).\n\n```python\npf.dividend_yield.plot()\n```\n![](../images/images/readmi05.png?raw=true) \n\n### 3. Draw an Efficient Frontier for 2 popular ETF: SPY and GLD\n```python\nls = [\'SPY.US\', \'GLD.US\']\ncurr = \'USD\'\nlast_date=\'2020-10\'\n# Rebalancing periods is one year (default value)\nfrontier = ok.EfficientFrontierReb(ls, last_date=last_date\', ccy=curr, reb_period=\'year\')\nfrontier.names\n```\n![](../images/images/readmi06.jpg?raw=true) \n\nGet the Efficient Frontier points for rebalanced portfolios and plot the chart with the assets risk/CAGR points:\n```python\nimport matplotlib.pyplot as plt\n\npoints = frontier.ef_points\n\nfig = plt.figure(figsize=(12,6))\nfig.subplots_adjust(bottom=0.2, top=1.5)\nfrontier.plot_assets(kind=\'cagr\')  # plots the assets points on the chart\nax = plt.gca()\nax.plot(points.Risk, points.CAGR) \n```\n![](../images/images/readmi07.jpg?raw=true)   \n<nowiki>*</nowiki> - *rebalancing period is one year*.\n\n### 4. Get a Transition Map for allocations\n```python\nls = [\'SPY.US\', \'GLD.US\', \'BND.US\']\nmap = ok.EfficientFrontier(ls, ccy=\'USD\').plot_transition_map(cagr=False)\n```\n![](../images/images/readmi08.jpg?v23-11-2020,raw=true "Transition map")  \n\nMore examples are available in form of [Jupyter Notebooks](https://github.com/mbk-dev/okama/tree/master/examples).\n\n## Documentation\n\nThe official documentation is hosted on readthedocs.org: [https://okama.readthedocs.io/](https://okama.readthedocs.io/)\n\n## RoadMap\n\nThe plan for _okama_ is to add more functions that will be useful to investors and asset managers.\n\n- Add Omega ratio to EfficientFrontier, EfficientFrontierReb and Portfolio classes.\n- Add withdrawals as an attribute of Portfolio class.\n- Add Black-Litterman asset allocation \n- Accelerate optimization for multi-period Efficient Frontier: minimize_risk and maximize_risk methods of EfficientFrontierReb class.\n- Make a single EfficientFrontier class for all optimizations: single-period or multu-period with rebalancing period as a parameter.\n- Add different utility functions for optimizers: semi-deviation, VaR, CVaR, drawdowns etc.\n- Add more functions based on suggestion of users.\n\n## Contributing to okama\n\nContributions are *most welcome*. Have a look at the [Contribution Guide](https://github.com/mbk-dev/okama/blob/master/CONTRIBUTING.md) for more.  \nFeel free to ask questions on [Discussuions](https://github.com/mbk-dev/okama/discussions).  \nAs contributors and maintainers to this project, you are expected to abide by okama\' code of conduct. More information can be found at: [Contributor Code of Conduct](https://github.com/mbk-dev/okama/blob/master/CODE_OF_CONDUCT.md)\n\n## Communication\n\nFor basic usage questions (e.g., "_Is XXX currency supported by okama?_") and for sharing ideas please use [GitHub Discussions](https://github.com/mbk-dev/okama/discussions/3).\nRussian language community is available at [okama.io forums](https://community.okama.io/c/python-okama).\n\n## License\n\nMIT\n',
    'author': 'Sergey Kikevich',
    'author_email': 'chilango74@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://okama.io',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<3.10',
}


setup(**setup_kwargs)
