"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const runner_configuration_1 = require("../runner-configuration");
const cache_1 = require("./cache");
const network_1 = require("./network");
/**
 * The Gitlab Runner autoscaling on EC2 by Docker Machine.
 *
 * @stability stable
 * @example
 *
 * <caption>Provisioning a basic Runner</caption>
 * const app = new cdk.App();
 * const stack = new cdk.Stack(app, "RunnerStack", {
 * env: {
 * account: "000000000000",
 * region: "us-east-1",
 * }
 * });
 *
 * new GitlabRunnerAutoscaling(scope, "GitlabRunner", {
 * gitlabToken: "xxxxxxxxxxxxxxxxxxxx",
 * });
 */
class GitlabRunnerAutoscaling extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id);
        const { manager, cache, runners, network } = props;
        /**
         * S3 Bucket for Runners' cache
         */
        this.cacheBucket = (cache === null || cache === void 0 ? void 0 : cache.bucket) || new cache_1.Cache(scope, "Cache", cache === null || cache === void 0 ? void 0 : cache.options).bucket;
        /**
         * Network
         */
        this.network = new network_1.Network(scope, "Network", network);
        /**
         * IAM
         */
        const ec2ServicePrincipal = new aws_iam_1.ServicePrincipal("ec2.amazonaws.com", {});
        const ec2ManagedPolicyForSSM = aws_iam_1.ManagedPolicy.fromManagedPolicyArn(scope, "AmazonEC2RoleforSSM", "arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM");
        /**
         * GitLab Runners
         */
        const runnersSecurityGroupName = `${scope.stackName}-RunnersSecurityGroup`;
        const runnersSecurityGroup = new aws_ec2_1.SecurityGroup(scope, "RunnersSecurityGroup", {
            securityGroupName: runnersSecurityGroupName,
            description: "Security group for GitLab Runners.",
            vpc: this.network.vpc,
        });
        const runnersRole = (runners === null || runners === void 0 ? void 0 : runners.role) ||
            new aws_iam_1.Role(scope, "RunnersRole", {
                assumedBy: ec2ServicePrincipal,
                managedPolicies: [ec2ManagedPolicyForSSM],
            });
        const runnersInstanceProfile = new aws_iam_1.CfnInstanceProfile(scope, "RunnersInstanceProfile", {
            roles: [runnersRole.roleName],
        });
        const runnersInstanceType = (runners === null || runners === void 0 ? void 0 : runners.instanceType) || aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO);
        const runnersLookupMachineImage = new aws_ec2_1.LookupMachineImage({
            name: "ubuntu/images/hvm-ssd/ubuntu-focal-20.04-amd64-server-*",
            owners: ["099720109477"],
            filters: {
                architecture: ["x86_64"],
                "image-type": ["machine"],
                state: ["available"],
                "root-device-type": ["ebs"],
                "virtualization-type": ["hvm"],
            },
        }).getImage(this);
        const runnersMachineImage = (runners === null || runners === void 0 ? void 0 : runners.machineImage) ||
            aws_ec2_1.MachineImage.genericLinux({
                [scope.region]: runnersLookupMachineImage.imageId,
            });
        /**
         * GitLab Manager
         */
        const managerSecurityGroup = new aws_ec2_1.SecurityGroup(scope, "ManagerSecurityGroup", {
            vpc: this.network.vpc,
            description: "Security group for GitLab Runners Manager.",
        });
        managerSecurityGroup.connections.allowTo(runnersSecurityGroup, aws_ec2_1.Port.tcp(22), "SSH traffic from Manager");
        managerSecurityGroup.connections.allowTo(runnersSecurityGroup, aws_ec2_1.Port.tcp(2376), "SSH traffic from Docker");
        const managerInstanceType = (runners === null || runners === void 0 ? void 0 : runners.instanceType) || aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.NANO);
        const managerMachineImage = (runners === null || runners === void 0 ? void 0 : runners.machineImage) ||
            aws_ec2_1.MachineImage.latestAmazonLinux({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
                edition: aws_ec2_1.AmazonLinuxEdition.STANDARD,
                virtualization: aws_ec2_1.AmazonLinuxVirt.HVM,
                storage: aws_ec2_1.AmazonLinuxStorage.EBS,
                cpuType: aws_ec2_1.AmazonLinuxCpuType.X86_64,
            });
        const managerRole = new aws_iam_1.Role(scope, "ManagerRole", {
            assumedBy: ec2ServicePrincipal,
            managedPolicies: [ec2ManagedPolicyForSSM],
            inlinePolicies: {
                Cache: aws_iam_1.PolicyDocument.fromJson({
                    Version: "2012-10-17",
                    Statement: [
                        {
                            Effect: "Allow",
                            Action: ["s3:ListObjects*", "s3:GetObject*", "s3:DeleteObject*", "s3:PutObject*"],
                            Resource: [`${this.cacheBucket.bucketArn}/*`],
                        },
                        {
                            Effect: "Allow",
                            Action: ["s3:ListBucket"],
                            Resource: [`${this.cacheBucket.bucketArn}`],
                        },
                    ],
                }),
                Runners: aws_iam_1.PolicyDocument.fromJson({
                    Version: "2012-10-17",
                    Statement: [
                        {
                            Effect: "Allow",
                            Action: ["ec2:CreateKeyPair", "ec2:DeleteKeyPair", "ec2:ImportKeyPair", "ec2:Describe*"],
                            Resource: ["*"],
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:CreateTags", "ssm:UpdateInstanceInformation"],
                            Resource: ["*"],
                            Condition: {
                                StringLike: {
                                    "aws:RequestTag/Name": "*gitlab-runner-*",
                                },
                                "ForAllValues:StringEquals": {
                                    "aws:TagKeys": ["Name"],
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:RequestSpotInstances", "ec2:CancelSpotInstanceRequests"],
                            Resource: ["*"],
                            Condition: {
                                StringEqualsIfExists: {
                                    "ec2:Region": `${scope.region}`,
                                },
                                ArnEqualsIfExists: {
                                    "ec2:Vpc": `${this.network.vpc.vpcArn}`,
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:RunInstances"],
                            Resource: ["*"],
                            Condition: {
                                StringEquals: {
                                    "ec2:InstanceType": [`${runnersInstanceType.toString()}`],
                                    "ec2:InstanceProfile": `${runnersInstanceProfile.ref}`,
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:TerminateInstances", "ec2:StopInstances", "ec2:StartInstances", "ec2:RebootInstances"],
                            Resource: ["*"],
                            Condition: {
                                StringLike: {
                                    "ec2:ResourceTag/Name": "*gitlab-runner-*",
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["iam:PassRole"],
                            Resource: [`${runnersRole.roleArn}`],
                        },
                    ],
                }),
            },
        });
        const userData = aws_ec2_1.UserData.forLinux({});
        userData.addCommands(`yum update -y aws-cfn-bootstrap` // !/bin/bash -xe
        );
        const gitlabRunnerConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        gitlabRunnerConfigRestartHandle._addFile("/etc/gitlab-runner/config.toml");
        const rsyslogConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        rsyslogConfigRestartHandle._addFile("/etc/rsyslog.d/25-gitlab-runner.conf");
        /**
         * Config set keys
         * @see {@link https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html#aws-resource-init-configsets}
         */
        const REPOSITORIES = "repositories";
        const PACKAGES = "packages";
        const CONFIG = "config";
        const RESTART = "restart";
        const initConfig = aws_ec2_1.CloudFormationInit.fromConfigSets({
            configSets: {
                default: [REPOSITORIES, PACKAGES, CONFIG, RESTART],
            },
            configs: {
                [REPOSITORIES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://packages.gitlab.com/install/repositories/runner/gitlab-runner/script.rpm.sh | bash", { key: "10-gitlab-runner" }),
                ]),
                [PACKAGES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitPackage.yum("docker"),
                    aws_ec2_1.InitPackage.yum("gitlab-runner"),
                    aws_ec2_1.InitPackage.yum("tzdata"),
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://gitlab-docker-machine-downloads.s3.amazonaws.com/v0.16.2-gitlab.12/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine", 
                    //"curl -L https://github.com/docker/machine/releases/download/v0.16.2/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine",
                    { key: "10-docker-machine" }),
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner start", {
                        key: "20-gitlab-runner-start",
                    }),
                ]),
                [CONFIG]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitFile.fromString("/etc/gitlab-runner/config.toml", runner_configuration_1.ConfigurationMapper.withDefaults({
                        globalConfiguration: {
                            concurrent: (_b = props === null || props === void 0 ? void 0 : props.concurrent) !== null && _b !== void 0 ? _b : 10,
                            checkInterval: (_c = props === null || props === void 0 ? void 0 : props.checkInterval) !== null && _c !== void 0 ? _c : 0,
                            logFormat: (props === null || props === void 0 ? void 0 : props.logFormat) || "runner",
                            logLevel: (props === null || props === void 0 ? void 0 : props.logLevel) || "info",
                        },
                        runnerConfiguration: {
                            token: props.gitlabToken,
                            url: props.gitlabUrl || "https://gitlab.com",
                            limit: (_d = runners === null || runners === void 0 ? void 0 : runners.limit) !== null && _d !== void 0 ? _d : 10,
                            outputLimit: (_e = runners === null || runners === void 0 ? void 0 : runners.outputLimit) !== null && _e !== void 0 ? _e : 52428800,
                            environment: (runners === null || runners === void 0 ? void 0 : runners.environment) || ["DOCKER_DRIVER=overlay2", "DOCKER_TLS_CERTDIR=/certs"],
                        },
                        dockerConfiguration: {
                            ...runners === null || runners === void 0 ? void 0 : runners.docker,
                        },
                        machineConfiguration: {
                            ...runners === null || runners === void 0 ? void 0 : runners.machine,
                            machineOptions: {
                                ...(_f = runners === null || runners === void 0 ? void 0 : runners.machine) === null || _f === void 0 ? void 0 : _f.machineOptions,
                                instanceType: (_g = runners === null || runners === void 0 ? void 0 : runners.instanceType) === null || _g === void 0 ? void 0 : _g.toString(),
                                ami: (_h = runners === null || runners === void 0 ? void 0 : runners.machineImage) === null || _h === void 0 ? void 0 : _h.getImage(scope).imageId,
                                region: scope.region,
                                vpcId: this.network.vpc.vpcId,
                                zone: this.network.availabilityZone.slice(-1),
                                subnetId: this.network.subnet.subnetId,
                                securityGroup: `${runnersSecurityGroupName}`,
                                usePrivateAddress: true,
                                iamInstanceProfile: `${runnersInstanceProfile.ref}`,
                            },
                        },
                        autoscalingConfigurations: (runners === null || runners === void 0 ? void 0 : runners.autoscaling) || [],
                        cacheConfiguration: {
                            s3: {
                                serverAddress: `s3.${scope.urlSuffix}`,
                                bucketName: `${this.cacheBucket.bucketName}`,
                                bucketLocation: `${scope.region}`,
                            },
                        },
                    }).toToml(), {
                        owner: "gitlab-runner",
                        group: "gitlab-runner",
                        mode: "000600",
                    }),
                    aws_ec2_1.InitFile.fromString("/etc/rsyslog.d/25-gitlab-runner.conf", `:programname, isequal, "gitlab-runner" /var/log/gitlab-runner.log`, {
                        owner: "root",
                        group: "root",
                        mode: "000644",
                    }),
                    aws_ec2_1.InitService.enable("gitlab-runner", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: gitlabRunnerConfigRestartHandle,
                    }),
                    aws_ec2_1.InitService.enable("rsyslog", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: rsyslogConfigRestartHandle,
                    }),
                ]),
                [RESTART]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner restart", {
                        key: "10-gitlab-runner-restart",
                    }),
                ]),
            },
        });
        const managerAutoScalingGroup = new aws_autoscaling_1.AutoScalingGroup(scope, "ManagerAutoscalingGroup", {
            vpc: this.network.vpc,
            vpcSubnets: {
                subnets: [this.network.subnet],
            },
            instanceType: managerInstanceType,
            machineImage: managerMachineImage,
            keyName: manager === null || manager === void 0 ? void 0 : manager.keyPairName,
            securityGroup: managerSecurityGroup,
            role: managerRole,
            userData: userData,
            init: initConfig,
            initOptions: {
                ignoreFailures: false,
            },
            maxCapacity: 1,
            minCapacity: 1,
            desiredCapacity: 1,
            signals: aws_autoscaling_1.Signals.waitForCount(1, { timeout: core_1.Duration.minutes(15) }),
        });
        this.runners = {
            securityGroupName: runnersSecurityGroupName,
            securityGroup: runnersSecurityGroup,
            role: runnersRole,
            instanceProfile: runnersInstanceProfile,
            instanceType: runnersInstanceType,
            machineImage: runnersMachineImage,
        };
        this.manager = {
            securityGroup: managerSecurityGroup,
            instanceType: managerInstanceType,
            machineImage: managerMachineImage,
            autoScalingGroup: managerAutoScalingGroup,
            role: managerRole,
        };
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscaling[_a] = { fqn: "@pepperize-testing/cdk-autoscaling-gitlab-runner.GitlabRunnerAutoscaling", version: "0.0.73" };
//# sourceMappingURL=data:application/json;base64,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